/* Keyboard tester
 *
 * Copyright (c) 2009 John Elliott <jce@seasip.demon.co.uk>
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <sys.h>
#include <time.h>
#include <conio.h>
#include <ctype.h>
#include <intrpt.h>

#include "kblib.h"

#asm
	.psect text
	.globl	_wait_pc_kb
	.globl	_wait_kb_pc

;
; Wait until the PC can successfully transmit to the keyboard.
;
_wait_pc_kb:
	push	cx
	push	ax
	xor	cx, cx
kw1:	in	al, 0x64
	test	al, #2
	loopnz	kw1
	pop	ax
	pop	cx
	retf
;
; Wait until the keyboard can successfully transmit to the PC.
;
_wait_kb_pc:
	push	cx
	push	ax
	xor	cx, cx
kw2:	in	al, 0x64
	test	al, #1
	loopnz	kw2
	pop	ax
	pop	cx
	retf

;
; Wait for an ACK, NAK or other result byte from the keyboard
;
_wait_ack:
	push	cx
	xor	cx, cx
kw3:	in	al, 0x64	;Wait until there is a byte returned
	and	al, #1
	jnz	kw4
	loop	kw3
	xor	ax,ax		;Nothing
	dec	ax		;-1
	jmp	kw5
		
kw4:	in	al, 0x60	;Result byte
	xor	ah, ah
kw5:	pop	cx
	retf
	
#endasm

static void send_byte(byte b)
{
	wait_pc_kb();
	outp(0x60, b);	
}
	
static unsigned getticks()
{
	union REGS rg;

	rg.x.ax = 0;
	int86(0x1A, &rg, &rg);
	return rg.x.dx;
}
	
/* Wait for <n> ticks of the system clock (18.2Hz) */
void wait(int n)
{
	union REGS rg;
	int t0;

	do
	{
		rg.x.ax = 0x0000;
		int86(0x1A, &rg, &rg);
		t0 = rg.x.dx;

		do
		{
			rg.x.ax = 0x0000;
			int86(0x1A, &rg, &rg);
		}
		while (t0 == rg.x.dx);
		--n;
	}
	while (n);
}


/* Send a command byte to the keyboard controller */
void send64(byte cmd)
{
	di();
	wait_pc_kb();
	outp(0x64, cmd);
	ei();
}

/* Send a command byte to the keyboard. */
void send60(byte cmd)
{
	send64(0xAD);	/* disable keyboard */
	di();
	send_byte(cmd);
	send64(0xae);	/* enable keyboard */
	ei();
}


/* Set or reset a bit in the keyboard controller's command register. 
 * Bit 7: Unused
 * Bit 6: Translate set 2 or 3 to set 1 
 * Bit 5: Disable mouse
 * Bit 4: Disable keyboard
 * Bit 3: Disable keylock
 * Bit 2: Cold reboot flag
 * Bit 1: Mouse interrupt enable
 * Bit 0: Keyboard interrupt enable */
void ctl_command(byte mask, int b)
{
	int cur, n;

//	printf("ctl_command %02x %02x\n", mask, b);
	di();
	outp(0x64, 0x20);		/* Read controller register 0 */
	cur = n = wait_ack();		/* Hopefully we got its value */
	if (n != -1)			/* If so... */
	{
		if (b) n |= mask;	/* Toggle the bit in question */
		else   n &= ~mask;

		outp(0x64, 0x60);	/* Write controller register 0 */
		send_byte(n);		/* Write new value */
	}
	else
	{
		fprintf(stderr,"Could not read control register\n");
	}
	wait_ack();	// More for the delay than the ACK
	ei();
//	printf("ctl_command end: cur=%04x n=%04x\n", cur, n);
}

/* Enable or disable translated mode */
void translate(int b) 
{ 
	ctl_command(0x40, b);
}

/* Enable or disable keyboard interrupts */
void kb_interrupt(int b)
{
	ctl_command(0x01, b);
}






void showcodes(int timeout)
{
	byte status;
	unsigned t, t2;
	int tdelta;
	
	kb_interrupt(0);

	t = getticks();
	tdelta = 1;
	do
	{
		status = inp(0x64);	/* Controller status */
		if (status & 1)
		{
			t = getticks();
			printf("%02x ", inp(0x60));
			fflush(stdout);
		}
		t2 = getticks();
		if (t2 > t)
		{
			tdelta = t2 - t;
		}
		else
		{
			tdelta = (long)(0x10000L+t2) - t;
		}
	} while (tdelta < timeout);
	kb_interrupt(1);
}









/*
byte readcmd(int b)
{
	int cur;

	di();
	wait_kb_pc();
	wait_pc_kb();
	outp(0x64, 0x20 + b);
	wait_pc_kb();
	cur = wait_ack();
	ei();
	return cur;
}
*/

int scancode_set(int verbose)
{
	int n[3];

	di();
//	wait_kb_pc();
//	wait_pc_kb();
	outp(0x64, 0xad);		/* Disable keyboard */
	send_byte(0xF0);		/* Command: Get/set scancode set */
	n[0] = wait_ack();		/* ACK for first command byte */
	send_byte(0x00);		/* Get current set */
	n[1] = wait_ack();
	n[2] = wait_ack();		/* Result byte(s) */
	outp(0x64, 0xae);		/* Enable keyboard */
	ei();
	if (verbose)
	{
		printf("Get scancode set: Sent F0 00, got %02x %02x ", n[0], n[1]);
		if (n[2] >= 0) printf("%02x", n[2]);
		putchar('\n');
	}
	return n[2];
}



void set_scancodes(int set, int verbose)
{
	int n[2];

//	wait_kb_pc();
//	wait_pc_kb();
	kb_interrupt(0);
	di();
	outp(0x64, 0xad);		/* Disable keyboard */
	send_byte(0xF0);		/* Command: Get/set scancode set */
	n[0] = wait_ack();
	send_byte(set);			/* Select the set */
	n[1] = wait_ack();	/* ACK for second command byte */
	outp(0x64, 0xae);	/* Enable keyboard */
	ei();
	kb_interrupt(1);
	if (verbose)
		printf("Set scancode set: Sent F0 %02x, got %02x %02x\n", set, n[0], n[1]);
}



void setleds(int l, int verbose)
{
	int n[2];
	
//	wait_kb_pc();
//	wait_pc_kb();
	kb_interrupt(0);
	di();
	outp(0x64, 0xad);	/* Disable keyboard */
	send_byte(0xED);	/* Command: Set LEDs */
	n[0] = wait_ack();	/* ACK for first command byte */
	send_byte(l);		/* LED values */
	n[1] = wait_ack();	/* ACK for second command byte */
	outp(0x64, 0xae);	/* Enable keyboard */
	ei();
	kb_interrupt(1);
	if (verbose)
		printf("Set LEDs: Sent ED %02x, got %02x %02x\n", l, n[0], n[1]);
}


void send_cmd(int c, int verbose)
{
	int n[2];
	
	di();
//	wait_kb_pc();
//	wait_pc_kb();
	outp(0x64, 0xad);	/* Disable keyboard */
	send_byte(c);		/* Command */
	n[0] = wait_ack();	/* ACK for first command byte */
	outp(0x64, 0xae);	/* Enable keyboard */
	ei();
	if (verbose)
		printf("Sent command %02x, got %02x\n", c, n[0]);
}




void get_id(char *buf, int verbose)
{
	int n[3];
	
	kb_interrupt(0);
	di();
//	wait_kb_pc();		/* Wait until KB has finished sending */
	outp(0x64, 0xad);	/* Disable keyboard */
	send_byte(0xF2);	/* Command: Identify */
	n[0] = wait_ack();	/* ACK for command */
	n[1] = wait_ack();	/* First ID byte */
	n[2] = wait_ack();	/* Second ID byte */
	outp(0x64, 0xae);	/* Enable keyboard */
//	wait_kb_pc();
	ei();
	kb_interrupt(1);
	if (n[1] == -1)
		sprintf(buf, "Nothing");
	else	sprintf(buf, "0x%02x 0x%02x", n[1], n[2]);

	if (verbose)
	{
		printf("Get identity: Sent F2, got %02x ", n[0]);
		if (n[1] >= 0) printf("%02x ", n[1]);
		if (n[2] >= 0) printf("%02x ", n[2]);
		putchar('\n');
	}
}

