WITH Ada.Calendar;
PACKAGE Dates IS
------------------------------------------------------------------
--|                                                              
--| specification for package to represent calendar dates
--|                                                              
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: September 1995                                     
--|                                                              
------------------------------------------------------------------

  SUBTYPE YearNumber  IS Ada.Calendar.Year_Number;  
  SUBTYPE MonthNumber IS Ada.Calendar.Month_Number; 
  SUBTYPE DayNumber   IS Ada.Calendar.Day_Number;   
  SUBTYPE JulianDay   IS Positive RANGE 1..366;
  SUBTYPE WeekDay     IS Positive RANGE 1..7;

  TYPE Date IS PRIVATE;

  -- exported exception
  Date_Error : EXCEPTION;

  -- constructors
  FUNCTION Today                       RETURN Date;
  -- Pre:  none
  -- Post: returns the current date

  FUNCTION MakeDate(Year  : YearNumber;
                    Month : MonthNumber;
                    Day   : DayNumber) RETURN Date;
  -- Pre:    Year, Month, and Day are defined
  -- Post:   returns a Date object
  -- Raises: Date_Error if Year, Month, and Day do not
  --   form a valid date (e.g. 6/31/93 or 2/29/93)

  -- selectors
  FUNCTION Year       (Right: Date) RETURN YearNumber;
  FUNCTION Month      (Right: Date) RETURN MonthNumber;
  FUNCTION DayOfMonth (Right: Date) RETURN DayNumber;
  FUNCTION DayOfYear  (Right: Date) RETURN JulianDay;
  FUNCTION DayOfWeek  (Right: Date) RETURN WeekDay;
  -- Pre:  Right is defined
  -- Post: these return the corresponding parts of the Date object

  -- comparison operators
  FUNCTION "<"  (Left, Right: Date) RETURN Boolean;
  FUNCTION "<=" (Left, Right: Date) RETURN Boolean;
  FUNCTION ">"  (Left, Right: Date) RETURN Boolean;
  FUNCTION ">=" (Left, Right: Date) RETURN Boolean;
  -- Pre:  Left and Right are defined
  -- Post: these return the result of the corresponding comparison

  -- arithmetic operators
  FUNCTION "+"  (Left: Date; Right: JulianDay) RETURN Date;
  FUNCTION "+"  (Left: JulianDay; Right: Date) RETURN Date;
  FUNCTION "-"  (Left: Date; Right: JulianDay) RETURN Date;
  -- Pre:  the arguments are defined
  -- Post: return a Date in the near future or recent past

PRIVATE
  TYPE Date IS RECORD
    Year:      YearNumber  := YearNumber'First;
    DayOfYear: JulianDay   := 1;
  END RECORD;
END Dates;
