PACKAGE Metric_System IS
------------------------------------------------------------------
--| Specification for Metric System Package
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: September 1995                                     
------------------------------------------------------------------
 
--  Type definition
 
  TYPE    Metric(Mass, Length, Time : Integer) IS PRIVATE;

  -- constrained subtypes

  SUBTYPE Scalar   IS Metric(0, 0, 0);
 
  SUBTYPE Accel    IS Metric(0, 1, -2);
  SUBTYPE Area     IS Metric(0, 2, 0);
  SUBTYPE Length   IS Metric(0, 1, 0);
  SUBTYPE Distance IS Metric(0, 1, 0);
  SUBTYPE Mass     IS Metric(1, 0, 0);
  SUBTYPE Time     IS Metric(0, 0, 1);
  SUBTYPE Velocity IS Metric(0, 1, -1);
  SUBTYPE Volume   IS Metric(0, 3, 0);

  -- exported exception

  Dimension_Error : EXCEPTION;

  -- exported unit constants; these will be defined in full below

  Gram       : CONSTANT Metric;
  METER      : CONSTANT Metric;
  SEC        : CONSTANT Metric;
  Square_M   : CONSTANT Metric;
  Cubic_M    : CONSTANT Metric;
  M_per_Sec  : CONSTANT Metric;
  M_per_Sec2 : CONSTANT Metric;
 
  FUNCTION "*" (Left : Float; Right : Metric) RETURN Metric;
  -- Pre:  Left and Right are defined
  -- Post: constructor: produces a metric quantity from a Float one

  FUNCTION Value(Left : Metric) RETURN Float;
  -- Pre:  Left is defined
  -- Post: selector: returns the Float (dimensionless) part
  --   of a metric quantity

  FUNCTION "<"  (Left, Right : Metric) RETURN Boolean;
  FUNCTION "<=" (Left, Right : Metric) RETURN Boolean;
  FUNCTION ">"  (Left, Right : Metric) RETURN Boolean;
  FUNCTION ">=" (Left, Right : Metric) RETURN Boolean;
  -- Pre:    Left and Right are defined
  -- Post:   the usual comparison operations
  -- Raises: Dimension_Error if Left and Right 
  --   have different dimensions

  FUNCTION "+" (Right : Metric) RETURN Metric;
  FUNCTION "-" (Right : Metric) RETURN Metric;
  FUNCTION "abs" (Right : Metric) RETURN Metric;
  -- Pre:  Right is defined
  -- Post: the usual monadic arithmetic operations;
  --   the dimensions of Right are, of course, preserved
 
  FUNCTION "+" (Left, Right : Metric) RETURN Metric;
  FUNCTION "-" (Left, Right : Metric) RETURN Metric;
  -- Pre:    Left and Right are defined
  -- Post:   the usual additive operations are performed on the
  --   numeric parts of Left and Right; the dimensions are preserved
  -- Raises: Dimension_Error if Left and Right
  --   have different dimensions

  FUNCTION "*" (Left, Right : Metric) RETURN Metric;
  FUNCTION "/" (Left, Right : Metric) RETURN Metric;
  -- Pre:    Left and Right are defined
  -- Post:   the usual multiplication and division operations 
  --   are performed on the numeric parts of Left and Right; 
  --   the dimensions are added pairwise (multiplication)
  --   or subtracted pairwise (division)
  --   Left and Right need not have the same dimensions.
  
PRIVATE

  -- A Metric quantity is a 3-discriminant variant record,
  -- with no default values. Each object of the type must
  -- therefore be constrained to a subtype, that is, to a
  -- fixed set of dimensions. This is physically realistic.

  TYPE Metric(Mass, Length, Time : Integer) IS RECORD
    Value : Float := 0.0;
  END RECORD;
 
  Gram       : CONSTANT Metric := (1, 0, 0, 1.0);
  Meter      : CONSTANT Metric := (0, 1, 0, 1.0);
  Sec        : CONSTANT Metric := (0, 0, 1, 1.0);
  Square_M   : CONSTANT Metric := (0, 2, 0, 1.0);
  Cubic_M    : CONSTANT Metric := (0, 3, 0, 1.0);
  M_per_Sec  : CONSTANT Metric := (0, 1, -1, 1.0);
  M_per_Sec2 : CONSTANT Metric := (0, 1, -2, 1.0);
 
END Metric_System;
