WITH Ada.Text_IO;
WITH Ada.Integer_Text_IO;
WITH Debugging_Support;
PACKAGE BODY Tables_Generic.Backup IS
------------------------------------------------------------------------
--| Body of generic backup/restore. Save simply copies the occupied
--| part of the array into the file, one record per line. The first
--| line of the file gives the number of records, T.CurrentSize.
--| Note that because this is a child package, it can "see" into the
--| private part of the parent and "knows" the structure of a table.
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: January 1996                                     
------------------------------------------------------------------------

  PROCEDURE Save (T: IN TableType; FileName: IN String) IS
    BackupFile: Ada.Text_IO.File_Type;
  BEGIN -- Save
    Debugging_Support.Enter(Subprogram => "Save");

    Ada.Text_IO.Create(File => BackupFile,
                       Mode => Ada.Text_IO.Out_File,
                       Name => FileName);
    Ada.Integer_Text_IO.Put(File => BackupFile, Item => T.CurrentSize);
    Ada.Text_IO.New_Line(File => BackupFile);

    FOR Count IN 1..T.CurrentSize LOOP
      Put(File => BackupFile, Item => T.Data(Count));
    END LOOP;

    Ada.Text_IO.Close(File => BackupFile);

    Debugging_Support.Leave(Subprogram => "Save");
  END Save;

  PROCEDURE Restore (T: OUT TableType; FileName: IN String) IS
    BackupFile: Ada.Text_IO.File_Type;
    Count: TableSize;
  BEGIN -- Restore
    Debugging_Support.Enter(Subprogram => "Restore");
    
    Ada.Text_IO.Open (File => BackupFile,
                       Mode => Ada.Text_IO.In_File,
                       Name => FileName);

    Ada.Integer_Text_IO.Get(File => BackupFile, Item => T.CurrentSize);
    Ada.Text_IO.Skip_Line(File => BackupFile);
    FOR Count IN 1..T.CurrentSize LOOP
      Get(File => BackupFile, Item => T.Data(Count));
    END LOOP;

    Ada.Text_IO.Close(File => BackupFile);

    Debugging_Support.Leave(Subprogram => "Restore");
  END Restore;

END Tables_Generic.Backup;
