WITH Reporter; USE Reporter;
WITH Types; USE Types;
WITH Database; USE Database;
WITH Ada.Numerics.Discrete_Random;
PACKAGE BODY Tellers IS
------------------------------------------------------------------------
--| Body of teller package. A teller object just waits a random
--| length of time (to simulate the time of a transaction), then
--| waits for a customer to ask for a transaction.
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: January 1996                                     
------------------------------------------------------------------------

  PACKAGE RandomTransactionTimes IS
    NEW Ada.Numerics.Discrete_Random (TransactionTimeRange);
  T: RandomTransactionTimes.Generator;

  TASK BODY Teller IS

    NewBalance      : Money;
    Del             : Integer;
    Stat            : Status;

    PROCEDURE SimulateWait IS
      WaitTime: TransactionTimeRange 
        := RandomTransactionTimes.Random(T);
    BEGIN
      ScreenManager.Put("  Teller " & TellerID & ": Transaction will take"
        & Integer'Image(WaitTime) & " sec");
      DELAY Duration(WaitTime);
    END SimulateWait;

    PROCEDURE ReportResult (Stat: Status; TellID: TellerRange;
                            ID: CustID; NewBalance: Money) IS
    BEGIN
      CASE Stat IS
        WHEN OK =>
          ScreenManager.Put("  Teller " & TellerID 
            & ": Acct"  & Integer'Image(ID) 
            & " - Balance is" & Integer'Image(NewBalance));
        WHEN BadCustID =>
          ScreenManager.Put("  Teller " & TellerID 
            & ": Acct"  & Integer'Image(ID) 
            & " - Invalid Account Number");
        WHEN InsufficientFunds =>
          ScreenManager.Put("  Teller " & TellerID 
            & ": Acct"  & Integer'Image(ID) 
            & " - InsufficientFunds");      
      END CASE;
    END ReportResult;

  BEGIN -- Teller
    RandomTransactionTimes.Reset(T); -- seed random sequence
    ScreenManager.Put("  Teller  " & TellerID  & " - at your service");
    LOOP
      SELECT -- Wait for any transaction request
        ACCEPT NewAcct(Id : OUT CustId; Stat: OUT Status) DO
          SimulateWait;
          Database.Manager.EnterCustID(Id, Stat);
          NewBalance := 0;
          ReportResult (Stat, TellerID, ID, NewBalance);
        END NewAcct;
      OR
        ACCEPT Deposit (Id : CustId; Val : IN Money; Stat: OUT Status) DO
          SimulateWait;
          Database.Manager.Deposit (Id, Val, NewBalance, Stat);
          ReportResult (Stat, TellerID, ID, NewBalance);
        END Deposit;
      OR
        ACCEPT Withdraw (Id : CustId; Val : IN Money; Stat: OUT Status) DO
          SimulateWait;
          Database.Manager.Withdraw (Id, Val, NewBalance, Stat);
          ReportResult (Stat, TellerID, ID, NewBalance);
        END Withdraw;
      OR
        ACCEPT Balance (Id : CustId; Stat: OUT Status) DO
          SimulateWait;
          Database.Manager.Balance (Id, NewBalance, Stat);
          ReportResult (Stat, TellerID, ID, NewBalance);
        END Balance;
      OR
        TERMINATE; -- if no more customers
      END SELECT;
    END LOOP; 

  END Teller;

END Tellers;
