/* functions to save the user-definable objects, referred to as "x" and "y".
 * this way, once defined, the objects can be quieried for position just like
 * the other bodies, with obj_cir(). 
 */

#include <stdio.h>
#include <math.h>
#ifdef VMS
#include <stdlib.h>
#endif
#include "astro.h"
#include "circum.h"
#include "screen.h"

extern char *strcat(), *strcpy(), *strncpy(), *getenv();

static char *dbfile;			/* !0 if set by -d option */
static char dbfdef[] = "ephem.db"; 	/* default database file name */

/* structures to describe objects of various types.
 */
#define	MAXNM		16	/* longest allowed object name, inc \0 */
typedef struct {
    double f_ra;	/* ra, rads, at given epoch */
    double f_dec;	/* dec, rads, at given epoch */
    double f_mag;	/* visual magnitude */
    double f_epoch;	/* the given epoch, as an mjd */
    char f_name[MAXNM];	/* name */
} ObjF;			/* fixed object */
typedef struct {
    double e_inc;	/* inclination, degrees */
    double e_Om;	/* longitude of ascending node, degrees */
    double e_om;	/* argument of perihelion, degress */
    double e_a;		/* mean distance, aka, semi-maj axis, in AU */
    double e_n;		/* daily motion, degrees/day */
    double e_e;		/* eccentricity */
    double e_M;		/* mean anomaly, ie, degrees from perihelion at... */
    double e_cepoch;	/* epoch date (M reference), as an mjd */
    double e_epoch;	/* equinox year (inc/Om/om reference), as an mjd */
    double e_m1, e_m2;	/* magnitude model coefficients: H/G or g/k */
    int e_whichm;	/* MAG_HG (default) or MAG_gk */
    char e_name[MAXNM];	/* name */
} ObjE;			/* object in heliocentric elliptical orbit */
typedef struct {
    double p_ep;	/* epoch of perihelion, as an mjd */
    double p_inc;	/* inclination, degs */
    double p_qp;	/* perihelion distance, AU */
    double p_ap;	/* argument of perihelion, degs. */
    double p_om;	/* longitude of ascending node, degs */
    double p_epoch;	/* reference epoch, as an mjd */
    double p_g, p_k;	/* magnitude model coefficients */
    char p_name[MAXNM];	/* name */
} ObjP;			/* object in heliocentric parabolic trajectory */

typedef struct {
    int  o_type;	/* current object type; see flags, below */
    int  o_on;		/* !=0 while current object is active */
    ObjF o_f;		/* the fixed object */
    ObjE o_e;		/* the elliptical orbit object */
    ObjP o_p;		/* the parabolic orbit object */
} Obj;
#define	FIXED		1
#define	ELLIPTICAL	2
#define	PARABOLIC	3
#define	MAG_HG		0	/* using 0 makes HG the initial default */
#define	MAG_gk		1

static Obj objx;
static Obj objy;

#define	DY	0		/* decimal year flag for set_year() */
#define	YMD	1		/* year/mon/day flag for set_year() */

/* run when Objx or y is picked from menu.
 * we tell which by the planet code.
 * let op define object and turn it on and off.
 */
obj_setup(p)
int p;
{
	static char *pr[5] = { /* leave a slot for "On"/"Off" */
	    "Fixed", "Elliptical", "Parabolic", "Lookup"
	};
	int f;
	Obj *op;

	op = (p == OBJX) ? &objx : &objy;

    rechk:
	/* map o_type to popup choice.
	 */
	switch (op->o_type) {
	case FIXED: f = 0; break;
	case ELLIPTICAL: f = 1; break;
	case PARABOLIC: f = 2; break;
	default: f = 3; break;
	}

    ask:
	pr[4] = op->o_on ? "On" : "Off";
	switch (f = popup (pr, f, 5)) {
	case 0: obj_dfixed(op, (char**)0); goto ask;
	case 1: obj_delliptical(op, (char**)0); goto ask;
	case 2: obj_dparabolic(op, (char**)0); goto ask;
	case 3: obj_filelookup (p, (char *)0); goto rechk;
	case 4: op->o_on ^= 1; break;
	}
}

/* turn "on" or "off" but don't forget facts about object the object.
 */
obj_on (p)
int p;
{
	if (p == OBJX)
	    objx.o_on = 1;
	else
	    objy.o_on = 1;
}
obj_off (p)
int p;
{
	if (p == OBJX)
	    objx.o_on = 0;
	else
	    objy.o_on = 0;
}

/* return true if object is now on, else 0.
 */
obj_ison(p)
int p;
{
	return ((p == OBJX) ? objx.o_on : objy.o_on);
}

/* set an alternate database file name.
 * N.B. we assume the storage pointed to by name is permanent.
 */
obj_setdbfilename (name)
char *name;
{
	dbfile = name;
}

/* fill in info about object x or y.
 * most arguments and conditions are the same as for plans().
 * only difference is that mag is already apparent, not absolute magnitude.
 * this is called by body_cir() for object x and y just like plans() is called
 * for the planets.
 */
obj_cir (jd, p, lpd0, psi0, rp0, rho0, lam, bet, mag)
double jd;	/* mjd now */
int p;		/* OBJX or OBJY */
double *lpd0;	/* heliocentric longitude, or NOHELIO  */
double *psi0;	/* heliocentric latitude, or 0 if *lpd0 set to NOHELIO */
double *rp0;	/* distance from the sun, or 0 */
double *rho0;	/* true distance from the Earth, or 0 */
double *lam;	/* apparent geocentric ecliptic longitude */
double *bet;	/* apparent geocentric ecliptic latitude */
double *mag;	/* APPARENT magnitude */
{
	Obj *op = (p == OBJX) ? &objx : &objy;

	switch (op->o_type) {
	case FIXED: {
	    double xr, xd;
	    xr = op->o_f.f_ra;
	    xd = op->o_f.f_dec;
	    if (op->o_f.f_epoch != jd)
		precess (op->o_f.f_epoch, jd, &xr, &xd);
	    eq_ecl (jd, xr, xd, bet, lam);

	    *lpd0 = NOHELIO;
	    *psi0 = *rp0 = *rho0 = 0.0;
	    *mag = op->o_f.f_mag;
	    }
	    break;
	case PARABOLIC: {
	    double inc, ap, om;
	    double lpd, psi, rp, rho;
	    double dt;
	    int pass;
	    /* two passes to correct lam and bet for light travel time. */
	    dt = 0.0;
	    for (pass = 0; pass < 2; pass++) {
		reduce_elements (op->o_p.p_epoch, jd-dt, degrad(op->o_p.p_inc),
		    degrad(op->o_p.p_ap), degrad(op->o_p.p_om), &inc, &ap, &om);
		comet (jd-dt, op->o_p.p_ep, inc, ap, op->o_p.p_qp, om,
					&lpd, &psi, &rp, &rho, lam, bet);
		if (pass == 0) {
		    *lpd0 = lpd;
		    *psi0 = psi;
		    *rp0 = rp;
		    *rho0 = rho;
		}
		dt = rho*5.775518e-3;	/* au to light-days */
	    }
	    *mag = op->o_p.p_g + 5*log10(*rho0) + 2.5*op->o_p.p_k*log10(*rp0);
	    }
	    break;
	case ELLIPTICAL: {
	    /* this is basically the same code as pelement() and plans()
	     * combined and simplified for the special case of osculating
	     * (unperturbed) elements.
	     * inputs have been changed to match the Astronomical Almanac.
	     * we have added reduction of elements using reduce_elements().
	     */
	    double dt, lg, lsn, rsn;
	    double nu, ea;
	    double ma, rp, lo, slo, clo;
	    double inc, psi, spsi, cpsi;
	    double y, lpd, rpd, ll, rho, sll, cll;
	    double om;		/* arg of perihelion */
	    double Om;		/* long of ascending node. */
	    double e;
	    int pass;

	    dt = 0;
	    sunpos (jd, &lsn, &rsn);
	    lg = lsn + PI;
	    e = op->o_e.e_e;

	    for (pass = 0; pass < 2; pass++) {

		reduce_elements (op->o_e.e_epoch, jd-dt, degrad(op->o_e.e_inc),
				degrad (op->o_e.e_om), degrad (op->o_e.e_Om),
				&inc, &om, &Om);

		ma = degrad (op->o_e.e_M
				+ (jd - op->o_e.e_cepoch - dt) * op->o_e.e_n);
		anomaly (ma, e, &nu, &ea);
		rp= op->o_e.e_a * (1-e*e) / (1+e*cos(nu));
		lo = nu + om;
		slo = sin(lo);
		clo = cos(lo);
		spsi = slo*sin(inc);
		y = slo*cos(inc);
		psi = asin(spsi);
		lpd = atan(y/clo)+Om;
		if (clo<0) lpd += PI;
		range (&lpd, 2*PI);
		cpsi = cos(psi);
		rpd = rp*cpsi;
		ll = lpd-lg;
		rho = sqrt(rsn*rsn+rp*rp-2*rsn*rp*cpsi*cos(ll));
		dt = rho*5.775518e-3;	/* light travel time, in days */
		if (pass == 0) {
		    *lpd0 = lpd;
		    *psi0 = psi;
		    *rp0 = rp;
		    *rho0 = rho;
		}
	    }

	    sll = sin(ll);
	    cll = cos(ll);
	    if (rpd < rsn)
		*lam = atan(-1*rpd*sll/(rsn-rpd*cll))+lg+PI;
	    else
		*lam = atan(rsn*sll/(rpd-rsn*cll))+lpd;
	    range (lam, 2*PI);
	    *bet = atan(rpd*spsi*sin(*lam-lpd)/(cpsi*rsn*sll));

	    if (op->o_e.e_whichm == MAG_HG) {
		/* this is for the H and G parameters from the Astro. Almanac.
		 */
		double psi_t, Psi_1, Psi_2, beta;
		beta = acos((rp*rp + rho*rho - rsn*rsn)/ (2*rp*rho));
		psi_t = exp(log(tan(beta/2.0))*0.63);
		Psi_1 = exp(-3.33*psi_t);
		psi_t = exp(log(tan(beta/2.0))*1.22);
		Psi_2 = exp(-1.87*psi_t);
		*mag = op->o_e.e_m1 + 5.0*log10(rp*rho)
		    - 2.5*log10((1-op->o_e.e_m2)*Psi_1 + op->o_e.e_m2*Psi_2);

	    } else {
		/* this uses the g/k model of comets */
		*mag =
		  op->o_e.e_m1 + 5*log10(*rho0) + 2.5*op->o_e.e_m2*log10(*rp0);
	    }
	    }
	    break;
	default:
	    f_msg ((p == OBJX) ? "Obj X is not defined"
			       : "Obj Y is not defined");
	    break;
	}
}

/* define obj based on the ephem.db line, s.
 * p is one of OBJX or OBJY.
 * format: name,type,[other fields, as per corresponding ObjX typedef]
 * N.B. we replace all ',' within s with '\0' IN PLACE.
 * return 0 if ok, else print reason why not with f_msg() and return -1.
 */
obj_define (p, s)
int p;	/* OBJX or OBJY */
char *s;
{
#define	MAXARGS	20
	char *av[MAXARGS];	/* point to each field for easy reference */
	char c;
	int ac;
	Obj *op = (p == OBJX) ? &objx : &objy;

	/* parse into comma separated fields */
	ac = 0;
	av[0] = s;
	do {
	    c = *s++;
	    if (c == ',' || c == '\0') {
		s[-1] = '\0';
		av[++ac] = s;
	    }
	} while (c);

	if (ac < 2) {
	    char buf[NC];
	    if (ac > 0)
		(void) sprintf (buf, "No type for Object %s", av[0]);
	    else
		(void) sprintf (buf, "No fields in %s", s);
	    f_msg (buf);
	    return (-1);
	}

	/* switch out on type of object - the second field */
	switch (av[1][0]) {
	case 'f':
	    if (ac != 6) {
		char buf[NC];
		(void) sprintf(buf,
		    "Need ra,dec,mag,epoch for fixed object %s", av[0]);
		f_msg (buf);
		return (-1);
	    }
	    obj_dfixed (op, av);
	    break;

	case 'e':
	    if (ac != 13) {
		char buf[NC];
		(void) sprintf (buf,
	"Need inc,lan,aop,md,dm,ecc,ma,cep,ep,H/g,G/k for elliptical object %s",
								    av[0]);
		f_msg (buf);
		return (-1);
	    }
	    obj_delliptical (op, av);
	    break;

	case 'p':
	    if (ac != 10) {
		char buf[NC];
		(void) sprintf (buf,
		    "Need ep,inc,ap,qp,om,epoch,g,k for parabolic object %s",
									av[0]);
		f_msg (buf);
		return (-1);
	    }
	    obj_dparabolic (op, av);
	    break;

	default: {
		char buf[NC];
		(void) sprintf (buf, "Unknown type for Object %s: %s",
								av[0], av[1]);
		f_msg (buf);
		return (-1);
	    }
	}

	return (0);
}

/* search through an ephem database file for an entry and use it to define
 *   either OBJX or OBJY (as set by p).
 * if a name, np, is not set then we ask for it.
 * if -d was used use it; else if EPHEMDB env set use it, else use default.
 * accept first partial match.
 */
obj_filelookup (p, np)
int p;			/* OBJX or OBJY */
char *np;
{
	FILE *fp;
	char *fn;
	int nl;
	char buf[160];
	char name[64];
	int found;

	/* open the database file */
	if (dbfile)
	    fn = dbfile;
	else {
	    fn = getenv ("EPHEMDB");
	    if (!fn)
		fn = dbfdef;
	}
	fp = fopen (fn, "r");
	if (!fp) {
	    (void) sprintf (buf, "Can not open database file %s", fn);
	    f_msg(buf);
	    return;
	}

	/* set up object name in name with a trailing ',' */
	if (np) {
	    (void) strncpy (name, np, sizeof(name)-2);
	    name[sizeof(name)-2] = '\0';	/* insure trailing '\0' */
	} else  {
	    f_prompt ("Object name: ");
	    if (read_line (name, sizeof(name)-2) <= 0)
		return;
	}
	nl = strlen (name);

	/* search for first line beginning with name.
	 * then rest of line is the info.
	 */
	found = 0;
	while (fgets (buf, sizeof(buf), fp))
	    if (strncmp (name, buf, nl) == 0) {
		found = 1;
		break;
	    }
	(void) fclose (fp);

	if (found)
	    (void) obj_define (p, buf);
	else {
	    (void) sprintf (buf, "Object %s not found", name);
	    f_msg (buf);
	}
}

/* define a fixed object.
 * args in av, in order, are name, type, ra, dec, magnitude and reference epoch.
 * if av then it is a list of strings to use for each parameter, else must
 * ask for each (but type). the av option is for cracking the ephem.db line.
 * if asking show current settings and leave unchanged if hit RETURN.
 * END aborts without making any more changes.
 * o_type is set to FIXED.
 * N.B. we don't error check av in any way, not even for length.
 */
static
obj_dfixed (op, av)
Obj *op;
char *av[];
{
	char buf[NC];
	char *bp;
	int sts;

	op->o_type = FIXED;

	if (set_name (av, op->o_f.f_name) < 0)
	    return;

	if (av) {
	    bp = av[2];
	    sts = 1;
	} else {
	    static char p[] = "RA (h:m:s): (";
	    f_prompt (p);
	    f_ra (R_PROMPT, C_PROMPT+sizeof(p)-1, op->o_f.f_ra);
	    (void) printf (") ");
	    sts = read_line (buf, 8+1);
	    if (sts < 0)
		return;
	    bp = buf;
	}
	if (sts > 0) {
	    int h, m, s;
	    f_dec_sexsign (radhr(op->o_f.f_ra), &h, &m, &s);
	    f_sscansex (bp, &h, &m, &s);
	    sex_dec (h, m, s, &op->o_f.f_ra);
	    op->o_f.f_ra = hrrad(op->o_f.f_ra);
	}

	if (av) {
	    bp = av[3];
	    sts = 1;
	} else {
	    static char p[] = "Dec (d:m:s): (";
	    f_prompt (p);
	    f_gangle (R_PROMPT, C_PROMPT+sizeof(p)-1, op->o_f.f_dec);
	    (void) printf (") ");
	    sts = read_line (buf, 9+1);
	    if (sts < 0)
		return;
	    bp = buf;
	}
	if (sts > 0) {
	    int dg, m, s;
	    f_dec_sexsign (raddeg(op->o_f.f_dec), &dg, &m, &s);
	    f_sscansex (bp, &dg, &m, &s);
	    sex_dec (dg, m, s, &op->o_f.f_dec);
	    op->o_f.f_dec = degrad(op->o_f.f_dec);
	}

	if (set_double (av, 4, "Magnitude: ", &op->o_f.f_mag) < 0)
	    return;

	(void) set_year (av, 5,"Reference epoch (UT Date, m/d.d/y or year.d): ",
						    DY, &op->o_f.f_epoch);

}

/* define an object in an elliptical heliocentric orbit.
 * 13 args in av, in order, are name, type, inclination, longitude of
 *   ascending node, argument of perihelion, mean distance (aka semi-major
 *   axis), daily motion, eccentricity, mean anomaly (ie, degrees from
 *   perihelion), epoch date (ie, time of the mean anomaly value), equinox year
 *   (ie, time of inc/lon/aop), and then two magnitude coefficients.
 * the mag may be H/G or g/k model, set by leading g or H (use H/G if none).
 * if av then it is a list of strings to use for each parameter, else must
 * ask for each. the av option is for cracking the ephem.db line.
 * if asking show current settings and leave unchanged if hit RETURN.
 * END aborts without making any more changes.
 * o_type is set to ELLIPTICAL.
 * N.B. we don't error check av in any way, not even for length.
 */
static
obj_delliptical(op, av)
Obj *op;
char *av[];
{
	op->o_type = ELLIPTICAL;

	if (set_name (av, op->o_e.e_name) < 0)
	    return;

	if (set_double (av, 2, "Inclination (degs):", &op->o_e.e_inc) < 0)
	    return;

	if (set_double (av, 3, "Longitude of ascending node (degs):",
				&op->o_e.e_Om) < 0)
	    return;

	if (set_double (av, 4, "Argument of Perihelion (degs):",
				&op->o_e.e_om) < 0)
	    return;

	if (set_double (av, 5, "Mean distance (AU):", &op->o_e.e_a) < 0)
	    return;

	if (set_double (av, 6, "Daily motion (degs/day):", &op->o_e.e_n) < 0)
	    return;

	if (set_double (av, 7, "Eccentricity:", &op->o_e.e_e) < 0)
	    return;

	if (set_double (av, 8, "Mean anomaly (degs):", &op->o_e.e_M) < 0)
	    return;

	if (set_year (av, 9, "Epoch date (UT Date, m/d.d/y or year.d): ",
						    YMD, &op->o_e.e_cepoch) < 0)
	    return;

	if (set_year (av, 10, "Equinox year (UT Date, m/d.d/y or year.d): ",
						    DY, &op->o_e.e_epoch) < 0)
	    return;

	if (av)
	    op->o_e.e_whichm = MAG_HG;	/* always the default for the db file */
	(void) set_elmag (av, 11, &op->o_e);

}

/* define an object in heliocentric parabolic orbit.
 * 10 args in av, in order, are name, type, epoch of perihelion, inclination,
 *   argument of perihelion, perihelion distance, longitude of ascending node,
 *   reference epoch, absolute magnitude and luminosity index.
 * if av then it is a list of strings to use for each parameter, else must
 * ask for each. the av option is for cracking the ephem.db line.
 * if asking show current settings and leave unchanged if hit RETURN.
 * END aborts without making any more changes.
 * o_type is set to PARABOLIC.
 * N.B. we don't error check av in any way, not even for length.
 */
static
obj_dparabolic(op, av)
Obj *op;
char *av[];
{
	op->o_type = PARABOLIC;

	if (set_name (av, op->o_p.p_name) < 0)
	    return;

	if (set_year(av,2,"Epoch of perihelion (UT Date, m/d.d/y or year.d): ",
						    YMD, &op->o_p.p_ep) < 0)
	    return;

	if (set_double (av, 3, "Inclination (degs):", &op->o_p.p_inc) < 0)
	    return;

	if (set_double(av,4,"Argument of perihelion (degs):", &op->o_p.p_ap) <0)
	    return;

	if (set_double (av, 5, "Perihelion distance (AU):", &op->o_p.p_qp) < 0)
	    return;

	if (set_double (av, 6,
		"Longitude of ascending node (degs):", &op->o_p.p_om) < 0)
	    return;

	if (set_year (av, 7, "Reference epoch (UT Date, m/d.d/y or year.d): ",
						    DY, &op->o_p.p_epoch) < 0)
	    return;

	if (set_double (av, 8, "g:", &op->o_p.p_g) < 0)
	    return;

	(void) set_double (av, 9, "k:", &op->o_p.p_k);
}


static
set_double (av, vn, pr, fp)
char *av[];	/* arg list */
int vn;		/* which arg */
char *pr;	/* prompt */
double *fp;	/* ptr to double to be set */
{
	int sts;
	char buf[NC];
	char *bp;

	if (av) {
	    bp = av[vn];
	    sts = 1;
	} else {
	    f_prompt (pr);
	    f_double (R_PROMPT, C_PROMPT+1+strlen(pr), "(%g) ", *fp);
	    sts = read_line (buf, 9);
	    if (sts < 0)
		return (-1);
	    bp = buf;
	}
	if (sts > 0)
	    *fp = atof (bp);
	return (0);
}

static
set_name (av, np)
char *av[];	/* arg list */
char *np;	/* name to be set */
{
	int sts;
	char buf[NC];
	char *bp;

	if (av) {
	    bp = av[0];
	    sts = 1;
	} else {
	    (void) sprintf (buf, "Name: (%s) ", np);
	    f_prompt (buf);
	    sts = read_line (buf, MAXNM-1);
	    if (sts < 0)
		return (-1);
	    bp = buf;
	}
	if (sts > 0)
	    (void) strcpy (np, bp);
	return (0);
}

static
set_year (av, vn, pr, type, yp)
char *av[];	/* arg list */
int vn;		/* which arg */
char *pr;	/* prompt */
int type;	/* display type: YMD or DY */
double *yp;	/* ptr to year to be set */
{
	int sts;
	char buf[NC];
	char *bp;

	if (av) {
	    bp = av[vn];
	    sts = 1;
	} else {
	    f_prompt (pr);
	    if (type == DY) {
		double y;
		mjd_year (*yp, &y);
		(void) printf ("(%g) ", y);
	    } else {
		int m, y;
		double d;
		mjd_cal (*yp, &m, &d, &y);
		(void) printf ("(%d/%g/%d) ", m, d, y);
	    }
	    sts = read_line (buf, 20);
	    if (sts < 0)
		return (-1);
	    bp = buf;
	}
	if (sts > 0)
	    crack_year (bp, yp);
	return (0);
}

/* given either a decimal year (xxxx. something) or a calendar (x/x/x)
 * convert it to an mjd and store it at *p;
 */
static
crack_year (bp, p)
char *bp;
double *p;
{
	if (decimal_year(bp)) {
	    double y = atof (bp);
	    year_mjd (y, p);
	} else {
	    int m, y;
	    double d;
	    mjd_cal (*p, &m, &d, &y);	/* init with current */
	    f_sscandate (bp, &m, &d, &y);
	    cal_mjd (m, d, y, p);
	}
}

/* read two args and load the magnitude members e_m1 and e_m2.
 * #,#     -> model is unchanged
 * g#,[k]# -> g/k
 * H#,[G]# -> H/G
 */
static
set_elmag (av, vn, ep)
char *av[];	/* arg list */
int vn;		/* which arg. we use av[vn] and av[vn+1] */
ObjE *ep;
{
	int sts;
	char buf[NC];
	char *bp;

	if (av) {
	    bp = av[vn];
	    sts = 1;
	} else {
	    /* show both the value and the type of the first mag param,
	     * as well as a hint as to how to set the type if desired.
	     */
	    (void) sprintf (buf, "%c: (%g) (g# H# or #) ",
				ep->e_whichm == MAG_HG ? 'H' : 'g', ep->e_m1);
	    f_prompt (buf);
	    sts = read_line (buf, 9);
	    if (sts < 0)
		return (-1);
	    bp = buf;
	}
	if (sts > 0) {
	    switch (bp[0]) {
	    case 'g':
		ep->e_whichm = MAG_gk;
		bp++;
		break;
	    case 'H':
		ep->e_whichm = MAG_HG;
		bp++;
	    default:
		/* leave type unchanged if no prefix */
		break;
	    }
	    ep->e_m1 = atof (bp);
	}

	if (av) {
	    bp = av[vn+1];
	    sts = 1;
	} else {
	    /* can't change the type in the second param */
	    (void) sprintf (buf, "%c: (%g) ",
				ep->e_whichm == MAG_HG ? 'G' : 'k', ep->e_m2);
	    f_prompt (buf);
	    sts = read_line (buf, 9);
	    if (sts < 0)
		return (-1);
	    bp = buf;
	}
	if (sts > 0) {
	    int ok = 0;
	    switch (bp[0]) {
	    case 'k':
		if (ep->e_whichm == MAG_gk) {
		    bp++;
		    ok = 1;
		}
		break;
	    case 'G':
		if (ep->e_whichm == MAG_HG) {
		    bp++;
		    ok = 1;
		}
		break;
	    default:
		ok = 1;
		break;
	    }
	    if (ok)
		ep->e_m2 = atof (bp);
	    else
		f_msg ("Can't switch magnitude models at second parameter.");
	}
	return (0);
}
