     REM
     REM   SATSAT2.BAS                     Dan Bruton, astro@tamu.edu
     REM
     REM     This BASIC program computes and displays the locations
     REM  of Saturn's Satellites for a given date and time.  See
     REM  "Practical Astronomy with your Calculator" by Peter
     REM  Duffett-Smith and the Astronomical Almanac for explanations
     REM  of some of the calculations here.  The code is included so
     REM  that users can make changes or convert to other languages.
     REM  This code was made using QBASIC (comes with DOS 5.0).
     REM
     REM ********************************************************************
     REM *                                                                  *
     REM *                        Constants                                 *
     REM *                                                                  *
     REM ********************************************************************
     REM # means double-precision, $ means string
     DIM SMA(10), e(10), U(10), U0(10), CS(10), N$(10), PD#(10), RSS(10)
     DIM SAT%(1000), X(2, 10), Y(2, 10), LASTX(2, 10), LASTY(2, 10)
     DIM DIR$(-1 TO 1)
     PI = 3.141592654#
     P = PI / 180
     NMF$ = "SATSAT2"
     REM Orbital Rate of Saturn in Radians per Days
     TP = 2 * PI / (29.45771 * 365.2422)
     REM Orbital Rate of Earth in Radians per Day
     TE = 2 * PI / (1.00004 * 365.2422)
     REM Eccentricity of Saturn's Orbit
     EP = .0556155#
     REM Eccentricity of Earth's Orbit
     EE = .016718#
     REM Semimajor axis of Earth's and Saturn's orbit in Astronomical Units
     RE0 = 1: RP0 = 9.554747
     REM Tilt of Earth's Axis - obliquity of the ecliptic
     OBL = 23.43928 * P
     REM Names of the Satellites
     N$(1) = "Mimas": N$(2) = "Enceladus": N$(3) = "Tethys": N$(4) = "Dione"
     N$(5) = "Rhea": N$(6) = "Titan": N$(7) = "Hyperion": N$(8) = "Iapetus"
     REM Semimajor Axis of the Satellites' Orbit in Kilometers
     SMA(1) = 185600: SMA(2) = 238100: SMA(3) = 294700: SMA(4) = 377500
     SMA(5) = 527200: SMA(6) = 1221600: SMA(7) = 1483000: SMA(8) = 3560100
     REM Visual Magnitude of the Satellites
     MAG(1) = 13: MAG(2) = 11.8: MAG(3) = 10.3: MAG(4) = 10.2
     MAG(5) = 9.8: MAG(6) = 8.4: MAG(7) = 14.3: MAG(8) = 11.2
     REM Pixel Size of Moons
     RSS(1) = 0: RSS(2) = 0: RSS(3) = 0: RSS(4) = 0
     RSS(5) = 1: RSS(6) = 1: RSS(7) = 0: RSS(8) = 0
     REM Semimajor Axis of the Satellites' Orbit in Kilometers
     SMA(1) = 185600: SMA(2) = 238100: SMA(3) = 294700: SMA(4) = 377500
     SMA(5) = 527200: SMA(6) = 1221600: SMA(7) = 1483000: SMA(8) = 3560100
     REM Eccentricity of Satellites' Orbit (Program uses 0)
     REM Satellites' Color
     CS(1) = 15: CS(2) = 15: CS(3) = 15: CS(4) = 15
     CS(5) = 15: CS(6) = 15: CS(7) = 15: CS(8) = 15
     REM Synodic Orbital Period of Moons in Days
     PD#(1) = .9425049#
     PD#(2) = 1.3703731#
     PD#(3) = 1.8880926#
     PD#(4) = 2.7375218#
     PD#(5) = 4.5191631#
     PD#(6) = 15.9669028#
     PD#(7) = 21.3174647#
     PD#(8) = 79.9208313#
     RS = 60330: REM Radius of planet in kilometers
     RS1 = 92000: REM Inner radius of inner ring in kilometers
     RS2 = 117500: REM Outer radius of inner ring in kilometers
     RS3 = 122500: REM Inner radius of outer ring in kilometers
     RS4 = 136500: REM Outer radius of outer ring in kilometers
     NM = 8: REM Number of moons displayed
     REM Directions
     DIR$(1) = "SOUTH": DIR$(-1) = "NORTH"
     REM ********************************************************************
     REM *                                                                  *
     REM *                      Epoch Information                           *
     REM *                                                                  *
     REM ********************************************************************
     JDE# = 2444238.5#: REM Epoch Jan 0.0 1980 = December 31,1979 0:0:0 UT
     LPE = 165.322242# * P: REM Longitude of Saturn at Epoch
     LPP = 92.6653974# * P: REM Longitude of Saturn`s Perihelion
     OHM = 113.4888341# * P: REM Longitude of Saturn`s Ascending Node
     INCS = 2.4893741# * P: REM Inclination of Saturn`s Orbit
     LEE = 98.83354# * P: REM Longitude of Earth at Epoch
     LEP = 102.596403# * P: REM Longitude of Earth's Perihelion
     REM U0(I) = Angle from inferior geocentric conjuction
     REM         measured westward along the orbit at epoch
     U0(1) = 18.2919 * P
     U0(2) = 174.2135 * P
     U0(3) = 172.8546 * P
     U0(4) = 76.8438 * P
     U0(5) = 37.2555 * P
     U0(6) = 57.7005 * P
     U0(7) = 266.6977 * P
     U0(8) = 197.7663 * P
     REM ********************************************************************
     REM *                                                                  *
     REM *    Program Control                                               *
     REM *                                                                  *
     REM ********************************************************************
     IJK = 5: REM Starting Zoom Size
     INVT = 1: INVTX = 1: INVTY = 1:  REM Starting Inverted Image
     O1 = 0: O2 = 1: REM Option to Draw Orbits
     CC1 = 1: CC2 = 0: REM Paint Moons Option
     CHI = 0: REM Change Zoom Factor?
     GOSUB 8700: REM Show Startup Screen
1    GOSUB 8800: REM Get Local Date and Time
     LASTH = LHOUR: LASTM = NMIN
     GOSUB 1000: REM Calculations
     GOSUB 2000: REM Display
5    GOSUB 8800: REM Get Local Date and Time
     IF NSEC = LSEC THEN GOTO 100
     LOCATE STRT, 1
     PRINT USING "LT: ##:##:##  ##/##/####"; LHOUR; NMIN; NSEC; LMONTH; LDAY; LYEAR;
     LOCATE STRT + 1, 1
     PRINT USING "UT: ##:##:##  ##/##/####"; NHOUR; NMIN; NSEC; NMONTH; NDAY; NYEAR
     IF INT(NMIN / 10) <> INT(LASTM / 10) THEN GOSUB 1000: GOSUB 2000
     LASTM = NMIN
99   T$ = RIGHT$(LEFT$(TIME$, 7), 1): NSC = (ASC(T$) - 48) * 10
     T$ = RIGHT$(LEFT$(TIME$, 8), 1): NSC = NSC + ASC(T$) - 48
     IF NSC = LSC THEN GOTO 100
     LSC = NSC
100  Z$ = INKEY$
101  IF Z$ = "1" THEN moon = 1: GOSUB 3000
     IF Z$ = "2" THEN moon = 2: GOSUB 3000
     IF Z$ = "3" THEN moon = 3: GOSUB 3000
     IF Z$ = "4" THEN moon = 4: GOSUB 3000
     IF Z$ = "5" THEN moon = 5: GOSUB 3000
     IF Z$ = "6" THEN moon = 6: GOSUB 3000
     IF Z$ = "7" THEN moon = 7: GOSUB 3000
     IF Z$ = "8" THEN moon = 8: GOSUB 3000
     IF Z$ = "0" THEN moon = 0: GOSUB 3010
     IF Z$ = "P" OR Z$ = "p" THEN GOSUB 5000: GOSUB 2000
     IF Z$ = "I" OR Z$ = "i" THEN GOSUB 6000
     IF Z$ = "F" OR Z$ = "f" THEN NMIN = NMIN + 10: NSEC = 0: CHG = 1: GOSUB 8900: GOSUB 1000: GOSUB 5100
     IF Z$ = "B" OR Z$ = "b" THEN NMIN = NMIN - 10: NSEC = 0: CHG = 1: GOSUB 8900: GOSUB 1000: GOSUB 5100
     IF Z$ = "A" OR Z$ = "a" THEN GOSUB 4000: CHG = 1: GOSUB 1000: GOSUB 2000
     IF Z$ = "D" OR Z$ = "d" THEN GOSUB 8600: CHG = 1: GOSUB 1000: GOSUB 2000
     IF Z$ = "X" OR Z$ = "x" THEN GOTO 8999
     IF Z$ = "Z" THEN IJK = IJK - 1: CHI = 1
     IF Z$ = "z" THEN IJK = IJK + 1: CHI = 1
     IF IJK < 1 THEN IJK = 1: CHI = 0
     IF IJK > NM THEN IJK = NM: CHI = 0
     IF CHI = 1 THEN CHI = 0: GOSUB 2000
     IF Z$ = "C" OR Z$ = "c" THEN CHG = 0: GOTO 1
     IF Z$ = "O" OR Z$ = "o" THEN SWAP O1, O2: GOSUB 2000
     IF Z$ = "h" OR Z$ = "H" THEN GOSUB 8500: GOSUB 2000
     IF Z$ = "T" OR Z$ = "t" THEN GOSUB 8400: GOSUB 2000
     IF CHG = 0 THEN GOTO 5 ELSE GOTO 99

1000 REM ********************************************************************
     REM *                                                                  *
     REM *   Orbit Calculations                                             *
     REM *                                                                  *
     REM ********************************************************************
     GGG = 1: IF NYEAR <= 1585 THEN GGG = 0
     JD# = -1 * INT(7 * (INT((NMONTH + 9) / 12) + NYEAR) / 4)
     S = SGN(NMONTH - 9): a = ABS(NMONTH - 9)
     J1 = INT(NYEAR + S * INT(a / 7))
     J1 = -1 * INT((INT(J1 / 100) + 1) * 3 / 4)
     JD# = JD# + INT(275 * NMONTH / 9) + NDAY + (GGG * J1)
     JD# = JD# + 1721027 + 2 * GGG + 367 * NYEAR - .5
     JD# = JD# + (NHOUR / 24) + (NMIN / (24 * 60)):  REM      Julian Date
     REM ****************** FIND MOON ORBITAL ANGLES ************************
     NN = JD# - JDE#: REM NN = Number of days since epoch
     ME = ((TE * NN) + LEE - LEP): REM Mean Anomoly of Earth
     MP = ((TP * NN) + LPE - LPP): REM Mean Anomoly of Saturn
     VE = ME: VP = MP: REM True Anomolies - Solve Kepler's Equation
     FOR I = 1 TO 3
     VE = VE - (VE - (EE * SIN(VE)) - ME) / (1 - (EE * COS(VE)))
     VP = VP - (VP - (EP * SIN(VP)) - MP) / (1 - (EP * COS(VP)))
     NEXT I
     VE = 2 * ATN(SQR((1 + EE) / (1 - EE)) * TAN(VE / 2))
     IF VE < 0 THEN VE = (2 * PI) + VE
     VP = 2 * ATN(SQR((1 + EP) / (1 - EP)) * TAN(VP / 2))
     IF VP < 0 THEN VP = (2 * PI) + VP
     REM  Heliocentric Longitudes of Earth and Saturn
     LE = VE + LEP: IF LE > (2 * PI) THEN LE = LE - (2 * PI)
     LP = VP + LPP: IF LP > (2 * PI) THEN LP = LP - (2 * PI)
     REM  Distances of Earth and Saturn from the Sun in AU's
     RE = RE0 * (1 - EE * EE) / (1 + EE * COS(VE))
     RP = RP0 * (1 - EP * EP) / (1 + EP * COS(VP))
     REM  DT = Distance from Saturn to Earth in AU's - Law of Cosines
     DT = SQR((RE ^ 2) + (RP ^ 2) - (2 * RE * RP * COS(LE - LP)))
     REM  II = Angle between Earth and Sun as seen from Saturn
     II = RE * SIN(LE - LP) / DT
     II = ATN(II / SQR(1 - II * II)): REM  ArcSIN and Law of Sines
     REM   F = NN - (Light Time to Earth in days)
     F = NN - (DT / 173.83)
     F1 = II + MP - VP
     REM U(I) = Angle from inferior geocentric conjuction measured westward
     FOR I = 1 TO NM
     U(I) = U0(I) + (F * 2 * PI / PD#(I)) + F1
     U(I) = ((U(I) / (2 * PI)) - INT(U(I) / (2 * PI))) * 2 * PI
     NEXT I
     REM ******************** FIND SATURN'S COORDINATES *********************
     REM PSI -> Heliocentric Latitude
     REM LAMB -> Geocentric Ecliptic Longitude
     REM BETA -> Geocentric Ecliptic Latitude
     SINPSI = SIN(LP - OHM) * SIN(INCS)
     PSI = ATN(SINPSI / SQR(1 - SINPSI * SINPSI)): REM ArcSIN
     NUM = SIN(LP - OHM) * COS(INCS)
     DEN = COS(LP - OHM)
     LPQ = ATN(NUM / DEN)
     IF NUM * DEN < 0 THEN LPQ = LPQ + PI
     IF NUM < 0 THEN LPQ = LPQ + PI
     LPQ = LPQ + OHM
     RPQ = RP * COS(PSI)
     NUM = RE * SIN(LPQ - LE)
     DEN = RPQ - RE * COS(LPQ - LE)
     LAMB = ATN(NUM / DEN)
     IF NUM * DEN < 0 THEN LAMB = LAMB + PI
     IF NUM < 0 THEN LAMB = LAMB + PI
     LAMB = LAMB + LPQ
     BETA = RPQ * TAN(PSI) * SIN(LAMB - LPQ)
     BETA = ATN(BETA / (RE * SIN(LPQ - LE)))
     NUM = (SIN(LAMB) * COS(OBL)) - (TAN(BETA) * SIN(OBL))
     DEN = COS(LAMB)
     RA = ATN(NUM / DEN)
     IF NUM * DEN < 0 THEN RA = RA + PI
     IF NUM < 0 THEN RA = RA + PI
     SINDEC = SIN(BETA) * COS(OBL)
     SINDEC = SINDEC + COS(BETA) * SIN(OBL) * SIN(LAMB)
     DEC = ATN(SINDEC / SQR(1 - SINDEC * SINDEC)): REM ArcSIN
     REM **************** FIND INCLINATION OF RINGS *************************
     REM Use dot product of Earth-Saturn vector and Saturn's rotation axis
     TVA = (90 - 83.51) * P: REM Theta coordinate of Saturn's axis
     PVA = 40.27 * P: REM Phi coordinate of Saturn's axis
     TVC = (PI / 2) - DEC
     PVC = RA
     DOT1 = SIN(TVA) * COS(PVA) * SIN(TVC) * COS(PVC)
     DOT1 = DOT1 + SIN(TVA) * SIN(PVA) * SIN(TVC) * SIN(PVC)
     DOT1 = DOT1 + COS(TVA) * COS(TVC)
     INC = ATN(SQR(1 - DOT1 ^ 2) / DOT1): REM   ArcCOS
     IF INC > 0 THEN INC = (PI / 2) - INC ELSE INC = -(PI / 2) - INC
     REM ************* FIND INCLINATION OF IAPETUS' ORBIT *******************
     REM Use dot product of Earth-Saturn vector and Iapetus' orbit axis
     REM Vector B
     TVB = (90 - 75.6) * P: REM Theta coordinate of Iapetus' orbit axis (estimate)
     PVB = 21.34 * 2 * PI / 24: REM Phi coordinate of Iapetus' orbit axis (estimate)
     DOT2 = SIN(TVB) * COS(PVB) * SIN(TVC) * COS(PVC)
     DOT2 = DOT2 + SIN(TVB) * SIN(PVB) * SIN(TVC) * SIN(PVC)
     DOT2 = DOT2 + COS(TVB) * COS(TVC)
     INCI = ATN(SQR(1 - DOT2 ^ 2) / DOT2): REM   ArcCOS
     IF INCI > 0 THEN INCI = (PI / 2) - INCI ELSE INCI = -(PI / 2) - INCI
     REM ************* FIND ROTATION ANGLE OF IAPETUS' ORBIT ****************
     REM Use inclination of Iapetus' orbit with respect to ring plane
     REM Triple Product
     TRIP = SIN(TVC) * COS(PVC) * SIN(TVA) * SIN(PVA) * COS(TVB)
     TRIP = TRIP - SIN(TVC) * COS(PVC) * SIN(TVB) * SIN(PVB) * COS(TVA)
     TRIP = TRIP + SIN(TVC) * SIN(PVC) * SIN(TVB) * COS(PVB) * COS(TVA)
     TRIP = TRIP - SIN(TVC) * SIN(PVC) * SIN(TVA) * COS(PVA) * COS(TVB)
     TRIP = TRIP + COS(TVC) * SIN(TVA) * COS(PVA) * SIN(TVB) * SIN(PVB)
     TRIP = TRIP - COS(TVC) * SIN(TVB) * COS(PVB) * SIN(TVA) * SIN(PVA)
     GAM = -1 * ATN(TRIP / SQR(1 - TRIP * TRIP)): REM ArcSIN
     REM ********************************************************************
     RETURN

2000 REM ********************************************************************
     REM *                                                                  *
     REM *   Draw Saturn - Polar View                                       *
     REM *                                                                  *
     REM ********************************************************************
     CLS
2001 SCALE = (3 * XMAX) / (17 * SMA(IJK))
     XC = INT(XMAX / 2): YC = 3 * YMAX / 4
     XS1 = XC + (1.1 * RS * SCALE * INVTX * COS(II))
     XS2 = XC - (1.1 * RS * SCALE * INVTX * COS(II))
     YS1 = YC + (RS * SCALE * SIN(II))
     YS2 = YC - (RS * SCALE * SIN(II))
     XS3 = YMAX * INVTX * TAN(II) / 4
     REM Draw rings
     CIRCLE (XC, YC), RS1 * SCALE, 8
     CIRCLE (XC, YC), RS2 * SCALE, 8
     CIRCLE (XC, YC), RS3 * SCALE, 8
     CIRCLE (XC, YC), RS4 * SCALE, 8
     PAINT (XC - (SCALE * (RS1 + RS2) / 2), YC), 7, 8
     PAINT (XC + (SCALE * (RS1 + RS2) / 2), YC), 7, 8
     PAINT (XC - (SCALE * (RS3 + RS4) / 2), YC), 7, 8
     PAINT (XC + (SCALE * (RS3 + RS4) / 2), YC), 7, 8
     CIRCLE (XC, YC), RS1 * SCALE + 1, 7
     CIRCLE (XC, YC), RS2 * SCALE - 1, 7
     CIRCLE (XC, YC), RS3 * SCALE + 1, 7
     CIRCLE (XC, YC), RS4 * SCALE - 1, 7
     REM **********************  Draw Shadows *************************
     CIRCLE (XC, YC), RS * SCALE, 7
     PAINT (XC, YC), 7, 7
     LINE (5 * XMAX / 16, YMAX / 2)-(11 * XMAX / 16, YMAX - 1), 15, B: REM   Bottom White Box
     LINE (XS1, YS1)-(XS2, YS2), 15
     LINE (XS1 + XS3, YMAX / 2)-(XS1, YS1), 15
     LINE (XS2 + XS3, YMAX / 2)-(XS2, YS2), 15
     PAINT (XC, YC - 2), 8, 15
     PAINT (XC, YC - 2), 0, 15
     LINE (XS1, YS1)-(XS2, YS2), 0
     LINE (XS1 + XS3, YMAX / 2)-(XS1, YS1), 0
     LINE (XS2 + XS3, YMAX / 2)-(XS2, YS2), 0
     REM ****************** DRAW MOONS (Polar View) *************************
     FOR I = 1 TO IJK
     REM   Orbital Paths
     IF O1 = 1 THEN CIRCLE (XC, YC), SMA(I) * SCALE, 8
     X(2, I) = XC - (SMA(I) * SIN(INVTX * U(I)) * SCALE)
     Y(2, I) = YC + INT((SMA(I) * COS(U(I)) * SCALE / RATYX))
     CIRCLE (X(2, I), Y(2, I)), RSS(I), CS(I)
     NEXT I
     REM ******************** DRAW SATURN (Earth View) **********************
     YC = YMAX / 4
     ASP = SIN(INVTY * INC) / RATYX: REM Aspect Ratio
     A1 = PI: A2 = 0: REM Start and End of Arcs
     IF ASP < 0 THEN SWAP A1, A2: ASP = -ASP
     REM Draw planet
     CIRCLE (XC, YC), RS * SCALE, 7, , , .9 / RATYX
     PAINT (XC, YC), 7, 7
     REM Draw rings
     CIRCLE (XC, YC), RS1 * SCALE - 1, 8, , , ASP
     CIRCLE (XC, YC), RS2 * SCALE + 1, 8, , , ASP
     CIRCLE (XC, YC), RS3 * SCALE - 1, 8, , , ASP
     CIRCLE (XC, YC), RS4 * SCALE + 1, 8, , , ASP
     PAINT (XC - (SCALE * (RS1 + RS2) / 2), YC), 7, 8
     PAINT (XC + (SCALE * (RS1 + RS2) / 2), YC), 7, 8
     PAINT (XC - (SCALE * (RS3 + RS4) / 2), YC), 7, 8
     PAINT (XC + (SCALE * (RS3 + RS4) / 2), YC), 7, 8
     REM ****************** DRAW MOONS (Earth View) *************************
     FOR I = 1 TO NM - 1
     REM   Orbital Paths
     IF SMA(I) * SCALE * ASP < YMAX / 4 AND O1 = 1 THEN CIRCLE (XC, YC), SMA(I) * SCALE, 8, , , ASP
     X(1, I) = XC - (SMA(I) * SIN(INVTX * U(I)) * SCALE)
     Y(1, I) = YC + INT((SMA(I) * COS(U(I)) * SIN(INVTY * INC) * SCALE / RATYX))
     IF Y(1, I) < (YMAX / 2 - 10) THEN CIRCLE (X(1, I), Y(1, I)), RSS(I), CS(I)
     NEXT I
     REM ******************** Iapetus' Orbit (Earth View) *******************
     TEMPX = -1 * (SMA(8) * SIN(U(8)) * SCALE)
     TEMPY = INT((SMA(8) * COS(U(8)) * SIN(INCI) * SCALE / RATYX))
     X(1, 8) = XC + INVTX * (TEMPX * COS(GAM) + TEMPY * SIN(GAM)): REM Rotation
     Y(1, 8) = YC + INVTY * (-TEMPX * SIN(GAM) + TEMPY * COS(GAM))
     IF X(1, 8) > XMAX OR X(1, 8) < 0 THEN GOTO 2910
     IF Y(1, 8) > (YMAX / 2 - 10) OR Y(1, 8) < 0 THEN GOTO 2910
     IF O1 = 0 THEN GOTO 2909
     REM   Orbital Path
     TEMPY = INT((SMA(8) * SIN(INCI) * SCALE / RATYX))
     XLAST = XC + INVTX * TEMPY * SIN(GAM): REM Rotation
     YLAST = YC + INVTY * TEMPY * COS(GAM)
     FOR I = 0 TO 360 STEP 2
     TEMPX = -1 * (SMA(8) * SIN(I * P) * SCALE)
     TEMPY = INT((SMA(8) * COS(I * P) * SIN(INCI) * SCALE / RATYX))
     XDRAW = XC + INVTX * (TEMPX * COS(GAM) + TEMPY * SIN(GAM)): REM Rotation
     YDRAW = YC + INVTY * (-TEMPX * SIN(GAM) + TEMPY * COS(GAM))
     IF YDRAW < (YMAX / 2 - 10) AND YLAST < (YMAX / 2 - 10) THEN LINE (XLAST, YLAST)-(XDRAW, YDRAW), 8
     XLAST = XDRAW: YLAST = YDRAW
     NEXT I
     REM Draw Iapetus
2909 CIRCLE (X(1, 8), Y(1, 8)), RSS(8), CS(8)
2910 REM ************** Redraw Saturn's exposed hemisphere ****************
     CIRCLE (XC, YC), RS * SCALE + 1, 0, A2, A1, .9 / RATYX
     LINE (XC - RS * SCALE - 1, YC)-(XC + RS * SCALE + 1, YC), 0
     PAINT (XC, YC - RS * SCALE * SGN(INVTY * INC) / 2), 1, 0
     PAINT (XC, YC - RS * SCALE * SGN(INVTY * INC) / 2), 7, 0
     CIRCLE (XC, YC), RS * SCALE + 1, 8, A2, A1, .9 / RATYX
     LINE (XC - RS * SCALE - 1, YC)-(XC + RS * SCALE + 1, YC), 7
     REM ********************* DRAW BACKGROUND *****************************
     LINE (3, 6)-(XMAX - 4, YMAX / 2 - 8), 15, B: REM          Top White Box
     LINE (5 * XMAX / 16, YMAX / 2)-(11 * XMAX / 16, YMAX - 1), 15, B: REM   Bottom White Box
     LOCATE STRT, 1
     PRINT USING "LT: ##:##:##  ##/##/####"; LHOUR; NMIN; NSEC; LMONTH; LDAY; LYEAR;
     LOCATE STRT + 1, 1
     PRINT USING "UT: ##:##:##  ##/##/####"; NHOUR; NMIN; NSEC; NMONTH; NDAY; NYEAR
     LOCATE STRT + 4, 1: PRINT "No. SATELLITE  MAGNITUDE"
    
     FOR I = 1 TO NM
     LOCATE STRT + 4 + I, 1
     PRINT USING " #  \         \   ##.#"; I; N$(I); MAG(I);
     LOCATE STRT + 4 + I, 5
     IF CC1 = 0 THEN COLOR CS(I): PRINT USING "\         \"; N$(I);
     COLOR 7
     NEXT I
    
     LOCATE STRT - 0, 58: PRINT "KEY  FUNCTION";
     LOCATE STRT + 1, 59: PRINT "h    help";
     LOCATE STRT + 2, 59: PRINT "x    exit";
     LOCATE STRT + 3, 59: PRINT "o    draw orbits";
     LOCATE STRT + 4, 59: PRINT "d    set date";
     LOCATE STRT + 5, 59: PRINT "c    current view";
     LOCATE STRT + 6, 59: PRINT "t    show table";
     LOCATE STRT + 7, 59: PRINT "a    animate";
     LOCATE STRT + 8, 59: PRINT "p    paint moons";
     LOCATE STRT + 9, 59: PRINT "i    invert display";
     LOCATE STRT + 10, 59: PRINT "f    forward 10 min";
     LOCATE STRT + 11, 59: PRINT "b    backward 10 min";
     LOCATE STRT + 12, 58: PRINT "Z/z   zoom in/out";
     LOCATE STRT + 13, 58: PRINT "1-8   identify";
     LOCATE 1, 38: PRINT DIR$(INVTY)
     LOCATE 6, 1: IF INVTX = 1 THEN PRINT "W" ELSE PRINT "E"
     LOCATE 7, 1: IF INVTX = 1 THEN PRINT "E" ELSE PRINT "A"
     LOCATE 8, 1: PRINT "S"
     LOCATE 9, 1: PRINT "T"
     LOCATE 6, 80: IF INVTX = 1 THEN PRINT "E" ELSE PRINT "W"
     LOCATE 7, 80: IF INVTX = 1 THEN PRINT "A" ELSE PRINT "E"
     LOCATE 8, 80: PRINT "S";
     LOCATE 9, 80: PRINT "T";

     LOCATE 15, 38: PRINT DIR$(-1 * INVTY)
2999 RETURN

3000 REM ********************************************************************
     REM *                                                                  *
     REM *   Identify Moons                                                 *
     REM *                                                                  *
     REM ********************************************************************
     IF X(1, moon) - 10 > 0 AND X(1, moon) + 10 < XMAX AND Y(1, moon) < (YMAX / 2 - 10) AND Y(1, moon) > 10 THEN
      GET (X(1, moon) - 10, Y(1, moon) - 10)-(X(1, moon) + 10, Y(1, moon) + 10), SAT%
      FOR K = 1 TO 3
      FOR I = 5 TO 2 STEP -1
      CIRCLE (X(1, moon), Y(1, moon)), I, CS(moon)
      IF moon <= IJK THEN CIRCLE (X(2, moon), Y(2, moon)), I, CS(moon)
      FOR J = 1 TO 100: NEXT J
      CIRCLE (X(1, moon), Y(1, moon)), I, 0
      IF moon <= IJK THEN CIRCLE (X(2, moon), Y(2, moon)), I, 0
      NEXT I
      NEXT K
      PUT (X(1, moon) - 10, Y(1, moon) - 10), SAT%, PSET
     ELSE
      FOR I = 1 TO 3
      LOCATE 2, 3: PRINT "OUTSIDE WINDOW"
      FOR J = 1 TO 1000: NEXT J
      LOCATE 2, 3: PRINT "              "
      FOR J = 1 TO 200: NEXT J
      NEXT I
     END IF
3010 REM ********************* REDRAW MOONS ******************************
     FOR I = 1 TO IJK
     CIRCLE (X(2, I), Y(2, I)), RSS(I), CS(I)
     NEXT I
     FOR I = 1 TO NM
     IF Y(1, I) < (YMAX / 2 - 10) THEN CIRCLE (X(1, I), Y(1, I)), RSS(I), CS(I)
     NEXT I
     RETURN

4000 REM ********************************************************************
     REM *                                                                  *
     REM *   Animate                                                        *
     REM *                                                                  *
     REM ********************************************************************
     NSEC = 0
     COUNT = 0
     IF O1 = 1 THEN SWAP O1, O2
     CLS
     GOSUB 2000
     LINE (11 * XMAX / 16, YMAX / 2)-(XMAX - 1, YMAX - 1), 8, B
     PAINT (11 * XMAX / 16 + 3, YMAX / 2 + 3), 8, 8
     LINE (11 * XMAX / 16, YMAX / 2)-(XMAX - 1, YMAX - 1), 15, B
     PAINT (11 * XMAX / 16 + 3, YMAX / 2 + 3), 0, 15
     LOCATE STRT + 5, 58
     PRINT "Enter the time step"
     LOCATE STRT + 6, 58
     PRINT "size in minutes."
     LOCATE STRT + 7, 58
     INPUT "(Default -> 10): ", STP
     IF STP = 0 THEN STP = 10
     LOCATE STRT + 5, 58
     PRINT "                     "
     LOCATE STRT + 6, 58
     PRINT "Press any key to stop."
     LOCATE STRT + 7, 58
     PRINT "                      "
     WHILE INKEY$ = ""
     NMIN = NMIN + STP
     GOSUB 5100
     COUNT = COUNT + 1
     IF COUNT > 100 THEN GOSUB 2000: COUNT = 0
     IF ABS(INC / P) < 2 THEN LINE (XC - RS4 * SCALE, YMAX / 4)-(XC + RS4 * SCALE, YMAX / 4), 8
     WEND
     CLS
     RETURN

5000 REM ********************************************************************
     REM *                                                                  *
     REM *   Paint Moons                                                    *
     REM *                                                                  *
     REM ********************************************************************
     SWAP CC1, CC2
     IF CC1 = 0 THEN
       CS(1) = 4: CS(2) = 6: CS(3) = 14: CS(4) = 2
       CS(5) = 9: CS(6) = 5: CS(7) = 13: CS(8) = 15
       FOR I = 1 TO 8: RSS(I) = 1: NEXT I
     ELSE
       FOR I = 1 TO 9: CS(I) = 15: RSS(I) = 0: NEXT I
       RSS(5) = 1: RSS(6) = 1
     END IF
     RETURN

5100 REM ********************************************************************
     REM *                                                                  *
     REM *   Move Moons                                                     *
     REM *                                                                  *
     REM ********************************************************************
     FOR J = 1 TO 2
     FOR I = 1 TO 8
     LASTX(J, I) = X(J, I)
     LASTY(J, I) = Y(J, I)
     NEXT I
     NEXT J
     GOSUB 8900
     GOSUB 1000
     REM ********************************************************************
     SCALE = (3 * XMAX) / (17 * SMA(IJK))
     XC = INT(XMAX / 2): YC = 3 * YMAX / 4
     REM ****************** DRAW MOONS (Polar View) *************************
     FOR I = 1 TO IJK
     X(2, I) = XC - INT(SMA(I) * SIN(INVTX * U(I)) * SCALE)
     Y(2, I) = YC + INT((SMA(I) * COS(U(I)) * SCALE / RATYX))
     IF LASTX(2, I) <> X(2, I) OR LASTY(2, I) <> Y(2, I) THEN CIRCLE (LASTX(2, I), LASTY(2, I)), RSS(I), 0
     CIRCLE (X(2, I), Y(2, I)), RSS(I), CS(I)
     NEXT I
     REM ********************************************************************
     YC = YMAX / 4
     REM ****************** DRAW MOONS (Earth View) *************************
     FOR I = 1 TO NM - 1
     X(1, I) = XC - INT(SMA(I) * SIN(INVTX * U(I)) * SCALE)
     Y(1, I) = YC + INT((SMA(I) * COS(U(I)) * SIN(INVTY * INC) * SCALE / RATYX))
     IF LASTY(1, I) < (YMAX / 2 - 10) AND LASTY(1, I) > 10 AND LASTX(1, I) > 10 AND LASTX(1, I) < XMAX - 10 THEN
       IF SQR((LASTX(1, I) - XC) ^ 2 + ((LASTY(1, I) - YC) / .9) ^ 2) < RS * SCALE THEN COLE = 7 ELSE COLE = 0
       IF LASTX(1, I) <> X(1, I) OR LASTY(1, I) <> Y(1, I) THEN CIRCLE (LASTX(1, I), LASTY(1, I)), RSS(I), COLE
     END IF
     IF Y(1, I) < (YMAX / 2 - 10) AND Y(1, I) > 10 AND X(1, I) > 10 AND X(1, I) < XMAX - 10 THEN
       IF (SQR((X(1, I) - XC) ^ 2 + ((Y(1, I) - YC) / .9) ^ 2) > RS * SCALE) OR (COS(U(I)) > 0) THEN CIRCLE (X(1, I), Y(1, I)), RSS(I), CS(I)
     END IF
     NEXT I
     REM ******************** Iapetus' Orbit (Earth View) *******************
     TEMPX = -1 * (SMA(8) * SIN(U(8)) * SCALE)
     TEMPY = INT((SMA(8) * COS(U(8)) * SIN(INCI) * SCALE / RATYX))
     X(1, 8) = INT(XC + INVTX * (TEMPX * COS(GAM) + TEMPY * SIN(GAM))): REM Rotation
     Y(1, 8) = INT(YC + INVTY * (-TEMPX * SIN(GAM) + TEMPY * COS(GAM)))
     IF LASTY(1, 8) < (YMAX / 2 - 10) AND LASTY(1, 8) > 10 AND LASTX(1, 8) > 10 AND LASTX(1, 8) < XMAX - 10 THEN
       IF SQR((LASTX(1, 8) - XC) ^ 2 + ((LASTY(1, 8) - YC) / .9) ^ 2) < RS * SCALE THEN COLE = 7 ELSE COLE = 0
       IF LASTX(1, 8) <> X(1, 8) OR LASTY(1, 8) <> Y(1, 8) THEN CIRCLE (LASTX(1, 8), LASTY(1, 8)), RSS(8), COLE
     END IF
     IF Y(1, 8) < (YMAX / 2 - 10) AND Y(1, 8) > 10 AND X(1, 8) > 10 AND X(1, 8) < XMAX - 10 THEN
       IF (COLE = 0 OR COS(U(8)) > 0) THEN CIRCLE (X(1, 8), Y(1, 8)), RSS(8), CS(8)
     END IF
     REM ******************************************************************
     LOCATE STRT, 1
     PRINT USING "LT: ##:##:##  ##/##/####"; LHOUR; NMIN; NSEC; LMONTH; LDAY; LYEAR;
     LOCATE STRT + 1, 1
     PRINT USING "UT: ##:##:##  ##/##/####"; NHOUR; NMIN; NSEC; NMONTH; NDAY; NYEAR
     RETURN

6000 REM ********************************************************************
     REM *                                                                  *
     REM *   Invert Image (Only needed in subroutines 5100 and 2000)        *
     REM *                   U, II, INC, INCI, GAMMA                        *
     REM ********************************************************************
     INVT = INVT + 1
     IF INVT > 4 THEN INVT = 1
     IF INVT = 1 THEN INVTX = 1: INVTY = 1
     IF INVT = 2 THEN INVTX = 1: INVTY = -1
     IF INVT = 3 THEN INVTX = -1: INVTY = -1
     IF INVT = 4 THEN INVTX = -1: INVTY = 1
     GOSUB 2000
     RETURN

8400 REM ********************************************************************
     REM *                                                                  *
     REM *   Show Table                                                     *
     REM *                                                                  *
     REM ********************************************************************
     CLS
     PRINT
     PRINT USING "          Local Date: ##/##/####    Univ. Date: ##/##/####"; LMONTH; LDAY; LYEAR; NMONTH; NDAY; NYEAR
     PRINT USING "          Local Time:  ##:##:##     Univ. Time:  ##:##:##"; LHOUR; NMIN; NSEC; NHOUR; NMIN; NSEC
     PRINT
     PRINT USING "                           Julian Date : ########.###"; JD#
     PRINT USING "             Right Ascension of Saturn : ###.## Hours"; RA * 24 / (2 * PI)
     PRINT USING "                 Declination of Saturn : ###.##!"; DEC / P; CHR$(248)
     PRINT USING "   Ring Inclination as seen from Earth : ###.##!"; -1 * INC / P; CHR$(248)
     PRINT USING "      Heliocentric Longitude of Saturn : ###.##!"; LP / P; CHR$(248)
     PRINT USING "       Heliocentric Longitude of Earth : ###.##!"; LE / P; CHR$(248)
     PRINT USING "                Sun-Saturn-Earth Angle : ###.##!"; II / P; CHR$(248)
     PRINT USING "     Distance between Saturn and Earth : ###.## AU = #### million miles"; DT; (DT * 93)
     PRINT USING "       Light time from Saturn to Earth : ###.## minutes"; DT * 8.28
     TEMP = 2 * ATN(TAN(165.6 * P / (2 * 3600)) / DT) * 3600 / P
     PRINT USING "                Angular Size of Saturn : ###.## arcsec"; TEMP
     PRINT USING "  Major Angular Size of Saturn's Rings : ###.## arcsec"; RS4 * TEMP / RS
     PRINT USING "  Minor Angular Size of Saturn's Rings : ###.## arcsec"; ABS(RS4 * TEMP * SIN(INC) / RS)
     PRINT
     PRINT "     SATELLITE    MAGNITUDE    ORBITAL RADIUS    ORBITAL PERIOD "
     FOR I = 1 TO NM
     PRINT USING "    \          \     ##.#        ####### km       ###.## days"; N$(I); MAG(I); SMA(I); PD#(I)
     NEXT I
     PRINT : PRINT "                            Press any key."
     WHILE INKEY$ = "": WEND
     RETURN

8500 REM ********************************************************************
     REM *                                                                  *
     REM *   Help Screen                                                    *
     REM *                                                                  *
     REM ********************************************************************
     CLS : SCREEN 12: WIDTH 80, 30: COLOR 7
     LOCATE 1, 1
     PRINT : PRINT : PRINT
     PRINT "                             E A R T H   V I E W"
     PRINT
     PRINT "        Saturn and its moons are shown here as they appear though Earth"
     PRINT "   based telescopes.  The celestial directions (North, South, East, West)"
     PRINT "   are labeled at the sides of the box.  Titan is the easiest moon to see,"
     PRINT "   followed by wide-ranging Iapetus.  A 4-inch telescope often shows Rhea"
     PRINT "   and Dione and sometimes Tethys, but Enceladus usually requires at least"
     PRINT "   a 10-inch.  Mimas and Hyperion require an even larger telescope."
     PRINT : PRINT : PRINT : PRINT : PRINT : PRINT
     PRINT "                              P O L A R   V I E W       "
     PRINT
     PRINT "                              A view from above  "
     PRINT "                            Saturn's pole is shown      "
     PRINT "                            here with the Sun-Saturn-   "
     PRINT "                            Earth angle used to mark    "
     PRINT "                            the shadows on the planet   "
     PRINT "                            and rings.  Earth is    "
     PRINT "                            straight down in this"
     PRINT "                            view.  "
     LINE (3, 6)-(XMAX - 4, YMAX / 2 - 8), 15, B: REM          Top White Box
     LINE (5 * XMAX / 16, YMAX / 2)-(11 * XMAX / 16, YMAX - 1), 15, B: REM   Bottom White Box
     LOCATE STRT, 1
     PRINT "  Local time (LT) and"
     PRINT "  Universal time (UT) "
     PRINT "    are given here."
     PRINT : PRINT : PRINT
     PRINT " Satellites are listed"
     PRINT "   here in order of"
     PRINT "  increasing orbital"
     PRINT "        radii."
     LOCATE STRT + 1, 58: PRINT "  A list of commands";
     LOCATE STRT + 2, 58: PRINT "     is given here.";
     LOCATE STRT + 7, 58: PRINT "    Press any key. ";
     WHILE INKEY$ = "": WEND
     CLS
     PRINT : PRINT : PRINT
     PRINT "    KEY      FUNCTION"
     PRINT
     PRINT "     h - shows the help screens"
     PRINT
     PRINT "     x - exits the program  "
     PRINT
     PRINT "     o - draws the orbital paths of the moons (toggles on/off)"
     PRINT
     PRINT "     d - changes the date and time of observation"
     PRINT "         (does not change the computer clock date and time)"
     PRINT
     PRINT "     c - displays the current view of Saturn and its moons"
     PRINT "         (use this after changing the date and time or after animation)"
     PRINT
     PRINT "     t - displays a data table with Saturn's coordinates, ring"
     PRINT "         inclination and other information"
     PRINT
     PRINT "     a - animates the motion of Saturn's satellites"
     PRINT
     PRINT "     p - paints the moons (toggles on/off)"
     PRINT : PRINT : PRINT : PRINT
     PRINT "                                                               Press any key."
     WHILE INKEY$ = "": WEND
     CLS
     PRINT : PRINT : PRINT
     PRINT "    KEY      FUNCTION"
     PRINT
     PRINT "     i - inverts the images (4 possibilities)"
     PRINT
     PRINT "             To find North in your eyepiece nudge the open end of your "
     PRINT "          telescope toward the South horizon and the Saturn will appear to"
     PRINT "          move North in your eyepiece.  To find East in your eyepiece move"
     PRINT "          the end of your telescope toward the West horizon and the Saturn"
     PRINT "          will appear to move East in your eyepiece.  This technique will"
     PRINT "          work independent of the number of mirrors or lenses you have in"
     PRINT "          your telescope. "
     PRINT
     PRINT "     f - increments the time by 10 minutes"
     PRINT
     PRINT "     b - decrements the time by 10 minutes"
     PRINT
     PRINT "   Z/z - changes the zoom setting; includes 8 settings which allow"
     PRINT "         the orbits of each moon path to fit in the polar view box"
     PRINT
     PRINT "   1-8 - press any number between 1 and 8 and the corresponding satellite"
     PRINT "         will be identified; use the zoom command if the moon is not in"
     PRINT "         the field of view"
     PRINT : PRINT : PRINT
     PRINT "                                                               Press any key.";
     WHILE INKEY$ = "": WEND
     RETURN
    
8600 REM ********************************************************************
     REM *                                                                  *
     REM *   Change Date and Time Manually                                  *
     REM *                                                                  *
     REM ********************************************************************
     LINE (11 * XMAX / 16, YMAX / 2)-(XMAX - 1, YMAX - 1), 8, B
     PAINT (11 * XMAX / 16 + 3, YMAX / 2 + 3), 8, 8
     LINE (11 * XMAX / 16, YMAX / 2)-(XMAX - 1, YMAX - 1), 15, B
     PAINT (11 * XMAX / 16 + 3, YMAX / 2 + 3), 0, 15
     LOCATE STRT, 58
     PRINT "Enter Universal Date"
     LOCATE STRT + 2, 60
     INPUT "Month (MM) : ", NMONTH
     LOCATE STRT + 3, 60
     INPUT "Day (DD) : ", NDAY
     LOCATE STRT + 4, 60
     INPUT "Year (YYYY) : ", NYEAR
     LOCATE STRT + 6, 58
     PRINT "Enter Universal Time"
     LOCATE STRT + 8, 60
     INPUT "Hour : ", NHOUR
     LOCATE STRT + 9, 60
     INPUT "Minute : ", NMIN
     LHOUR = NHOUR - DHOUR
     LDAY = NDAY
     LYEAR = NYEAR
     LMONTH = NMONTH
     NSEC = 0
     GOSUB 8900
     RETURN
    
8700 REM ********************************************************************
     REM *                                                                  *
     REM *    Startup Screen                                                *
     REM *                                                                  *
     REM ********************************************************************
     CLS : COLOR 7
     TEMP$ = NMF$ + ".CFG"
     OPEN TEMP$ FOR INPUT AS #1
     INPUT #1, Z$, DHOUR
     CLOSE
     LOCATE 9, 23
     PRINT "      Satellites of Saturn "
     LOCATE 10, 23
     PRINT "           Version 2.0"
     PRINT : PRINT
     PRINT
     PRINT TAB(17); "Enter the number of hours that you add to your "
     PRINT TAB(17); "local time to get Universal Time (Default ->"; DHOUR; ") : ";
     INPUT "", DHOUR$
     IF DHOUR$ = "" THEN GOTO 8702
     LN = LEN(DHOUR$): DHOUR = 0
     FOR I = 1 TO LN
     DHOR$ = LEFT$(RIGHT$(DHOUR$, I), 1)
     DHOR = ASC(DHOR$) - 48
     IF DHOR = -3 THEN DHOUR = -1 * DHOUR: GOTO 8701
     DHOUR = DHOUR + (DHOR * (10 ^ (I - 1)))
8701 NEXT I
     OPEN TEMP$ FOR OUTPUT AS #1
     PRINT #1, Z$
     PRINT #1, DHOUR
     CLOSE
8702 XMAX = 639: YMAX = 479: RATYX = 1: SCREEN 12: WIDTH 80, 30: STRT = 17: COLOR 7
     RETURN

8800 REM ********************************************************************
     REM *                                                                  *
     REM *    Determine Time                                                *
     REM *                                                                  *
     REM ********************************************************************
     T$ = LEFT$(DATE$, 1): LMONTH = (ASC(T$) - 48) * 10
     T$ = RIGHT$(LEFT$(DATE$, 2), 1): LMONTH = LMONTH + ASC(T$) - 48
     T$ = RIGHT$(LEFT$(DATE$, 4), 1): LDAY = (ASC(T$) - 48) * 10
     T$ = RIGHT$(LEFT$(DATE$, 5), 1): LDAY = LDAY + ASC(T$) - 48
     T$ = RIGHT$(LEFT$(DATE$, 7), 1): LYEAR = (ASC(T$) - 48) * 1000
     T$ = RIGHT$(LEFT$(DATE$, 8), 1): LYEAR = LYEAR + (ASC(T$) - 48) * 100
     T$ = RIGHT$(LEFT$(DATE$, 9), 1): LYEAR = LYEAR + (ASC(T$) - 48) * 10
     T$ = RIGHT$(LEFT$(DATE$, 10), 1): LYEAR = LYEAR + ASC(T$) - 48
     T$ = (LEFT$(TIME$, 1)): LHOUR = (ASC(T$) - 48) * 10
     T$ = RIGHT$(LEFT$(TIME$, 2), 1): LHOUR = LHOUR + ASC(T$) - 48
     T$ = RIGHT$(LEFT$(TIME$, 4), 1): NMIN = (ASC(T$) - 48) * 10
     T$ = RIGHT$(LEFT$(TIME$, 5), 1): NMIN = NMIN + ASC(T$) - 48
     T$ = RIGHT$(LEFT$(TIME$, 7), 1): NSEC = (ASC(T$) - 48) * 10
     T$ = RIGHT$(LEFT$(TIME$, 8), 1): NSEC = NSEC + ASC(T$) - 48
     GOSUB 8900
     RETURN

8900 REM ********************************************************************
     REM *                                                                  *
     REM *  Correct Time after Time Increment                               *
     REM *                                                                  *
     REM ********************************************************************
     IF NSEC >= 60 THEN NSEC = NSEC - 60: NMIN = NMIN + 1
     IF NSEC < 0 THEN NSEC = NSEC + 60: NMIN = NMIN - 1
8901 IF NMIN >= 60 THEN NMIN = NMIN - 60: LHOUR = LHOUR + 1: GOTO 8901
8902 IF NMIN < 0 THEN NMIN = NMIN + 60: LHOUR = LHOUR - 1: GOTO 8902
     LastDAY = 31
     IF (LMONTH = 4) OR (LMONTH = 6) THEN LastDAY = 30
     IF (LMONTH = 9) OR (LMONTH = 11) THEN LastDAY = 30
     IF LMONTH = 2 THEN LastDAY = 28
     IF (INT(LYEAR / 4) = (LYEAR / 4)) AND (LMONTH = 2) THEN LastDAY = 29
     IF LHOUR >= 24 THEN LHOUR = LHOUR - 24: LDAY = LDAY + 1
     IF LHOUR < 0 THEN LHOUR = LHOUR + 24: LDAY = LDAY - 1
     IF LDAY > LastDAY THEN LDAY = LDAY - LastDAY: LMONTH = LMONTH + 1
     LastDAY = 31
     IF ((LMONTH - 1) = 4) OR ((LMONTH - 1) = 6) THEN LastDAY = 30
     IF ((LMONTH - 1) = 9) OR ((LMONTH - 1) = 11) THEN LastDAY = 30
     IF (LMONTH - 1) = 2 THEN LastDAY = 28
     IF (INT(LYEAR / 4) = (LYEAR / 4)) AND ((LMONTH - 1) = 2) THEN LastDAY = 29
     IF LDAY < 1 THEN LDAY = LDAY + LastDAY: LMONTH = LMONTH - 1
     IF LMONTH > 12 THEN LMONTH = LMONTH - 12: LYEAR = LYEAR + 1
     IF LMONTH < 1 THEN LMONTH = LMONTH + 12: LYEAR = LYEAR - 1
     NMONTH = LMONTH
     NYEAR = LYEAR
     NDAY = LDAY
     NHOUR = LHOUR + DHOUR
     LastDAY = 31
     IF (NMONTH = 4) OR (NMONTH = 6) THEN LastDAY = 30
     IF (NMONTH = 9) OR (NMONTH = 11) THEN LastDAY = 30
     IF NMONTH = 2 THEN LastDAY = 28
     IF (INT(NYEAR / 4) = (NYEAR / 4)) AND (NMONTH = 2) THEN LastDAY = 29
     IF NHOUR >= 24 THEN NHOUR = NHOUR - 24: NDAY = NDAY + 1
     IF NHOUR < 0 THEN NHOUR = NHOUR + 24: NDAY = NDAY - 1
     IF NDAY > LastDAY THEN NDAY = NDAY - LastDAY: NMONTH = NMONTH + 1
     LastDAY = 31
     IF ((NMONTH - 1) = 4) OR ((NMONTH - 1) = 6) THEN LastDAY = 30
     IF ((NMONTH - 1) = 9) OR ((NMONTH - 1) = 11) THEN LastDAY = 30
     IF (NMONTH - 1) = 2 THEN LastDAY = 28
     IF (INT(NYEAR / 4) = (NYEAR / 4)) AND ((NMONTH - 1) = 2) THEN LastDAY = 29
     IF NDAY < 1 THEN NDAY = NDAY + LastDAY: NMONTH = NMONTH - 1
     IF NMONTH > 12 THEN NMONTH = NMONTH - 12: NYEAR = NYEAR + 1
     IF NMONTH < 1 THEN NMONTH = NMONTH + 12: NYEAR = NYEAR - 1
     RETURN
    
8999 REM ********************************************************************
     REM *                                                                  *
     REM *  Print End of Program Information                                *
     REM *                                                                  *
     REM ********************************************************************
     SCREEN 0
     WIDTH 80, 25
     CLS : PRINT
     PRINT " ͻ"
     PRINT "                                                                             "
     PRINT "    The Earth will pass through Saturn's ring plane on the following days:   "
     PRINT "                                                                             "
     PRINT "                                May 22, 1995                                 "
     PRINT "                               August 11, 1995                               "
     PRINT "                              February 12, 1996                              "
     PRINT "                                                                             "
     PRINT "        The Sun will pass through the ring plane on November 19, 1995.       "
     PRINT "                                                                             "
     PRINT " ͼ"
     PRINT : PRINT : PRINT
     PRINT TAB(16); "                    Dan Bruton"
     PRINT TAB(16); "              Texas A & M University"
     PRINT TAB(16); "                  astro@tamu.edu"
     PRINT TAB(16); "       http://isc.tamu.edu/~astro/saturn.html"
     PRINT TAB(16); "                   December 1994"
     PRINT : PRINT
     END

     REM ************************ END OF PROGRAM ***************************

