/***************************************************************

	sfs_pe.c        Distant Perspective Display Routines for
			Space Flight Simulator

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

***************************************************************/

#include "stdio.h"
#include "ctype.h"
#include "bw.h"
#include "gr.h"
#include "kb.h"
#include "ui.h"
#include "as.h"
#include "sfs.h"

#define USEHIDDEN          /* use hidden buffer to draw perspective */
#define PE_VLAT         0.0
#define PE_VLON         0.0
#define PE_ROT          0.0

/***************************************************************

	pe_calc()

***************************************************************/

pe_calc( sorbit_array, n_orbits, poll )
   struct sfs_orbit **sorbit_array;
   int n_orbits;
   int (*poll)();
   {
   register int orbit, c;
   static double x_lat, x_lon;
   double save_lif;
   static long x_r, x_t, x_n;

   for ( orbit = 0; orbit < n_orbits; ++orbit )
      {

      if ( sorbit_array[ orbit ] != NULL )
	 {

	 sprintf( bw_ebuf, PE_CALC,
	    orbit + 1 );
	 bw_message( bw_ebuf );

	 save_lif = sorbit_array[ orbit ]->aorbit->lif;
	 sorbit_array[ orbit ]->aorbit->lif = 0.0;

	 sorbit_array[ orbit ]->pe_inc = sorbit_array[ orbit ]->aorbit->period / PE_POINTS;

	 x_t = 1;

	 for ( c = 0; c < PE_POINTS; ++c )
	    {

	    or_ssp( sorbit_array[ orbit ]->aorbit, x_t,
	       &x_lat, &x_lon, &x_r, &x_n );

	    sorbit_array[ orbit ]->pe_buffer[ c ].altitude  = x_r;
	    sorbit_array[ orbit ]->pe_buffer[ c ].latitude  = x_lat;
	    sorbit_array[ orbit ]->pe_buffer[ c ].longitude = x_lon;

	    x_t += sorbit_array[ orbit ]->pe_inc;

	    /* call poll function */

	    (*poll) ();

	    }                   /* end for point++ */

	 sorbit_array[ orbit ]->aorbit->lif = save_lif;

	 }                      /* end if orbit != NULL */

     }                         /* end while ++orbit */
   }                            /* end of function

/***************************************************************

	pe_draw()

***************************************************************/

pe_draw( uiwind, focus, sorbit_array, n_orbits, o_start, o_end, newtime, elements )
   struct uiwindow *uiwind;             /* ui  window to draw in */
   struct as_focus *focus;              /* orbital focus to display */
   struct sfs_orbit **sorbit_array;     /* array of orbits */
   int n_orbits;                        /* number of orbits in array */
   struct spj_pt *o_start, *o_end;      /* start, end orb display pattern */
   long newtime;                        /* current time */
   int elements;			/* elements to draw */
   {
   register int orbit;
   static double vlat, vlon, rotation;
   static double distance;
   static double vert_degrees;          /* calculate vertical degrees */
   double max_ap;
   int l, c, selected;
   int x_screen;

   /***  inform the viewer */

   bw_message( PE_DRAW );

   /***  initial screen setup */

#ifdef  USEHIDDEN
   if ( gr_screens > 1 )
      {
      x_screen = GR_HIDDEN;
      }
   else
      {
      x_screen = GR_PRIMARY;
      }
   ui_setscreen( x_screen );
#else
   x_screen = GR_PRIMARY;
#endif

   /* calculate vertical degrees */

   vert_degrees = HORDEGS * ( ( ( uiwind->u_y2 - (double) uiwind->u_y1 )
      * ( gr_pysize / (double) gr_pxsize ) )
      / ( uiwind->u_x2 - (double) uiwind->u_x1 ) );

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "Vert degrees: %.0lf", vert_degrees );
   bw_debug( bw_ebuf );
#endif

   /* Clear the display area */

   ui_fbox( uiwind->u_x1, uiwind->u_y1,
      uiwind->u_x2, uiwind->u_y2,
      BLACK, SOLID );

   /* Calculate viewer latitude, longitude, rotation, and distance of display */

   vlat = PE_VLAT;
   vlon = PE_VLON;
   rotation = PE_ROT;
   max_ap = 1.0;

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "pe_draw(): n_orbits %d", n_orbits );
   bw_debug( bw_ebuf );
#endif

   /* find the largest apoapsis among all selected orbits */

   selected = n_orbits + 1;
   for ( orbit = 0; orbit < n_orbits; ++orbit )
      {
      if ( ( sorbit_array[ orbit ] != NULL ) 
         && ( sorbit_array[ orbit ]->aorbit->focus == focus ))
	 {

         selected = orbit;		/* make sure we find at least
					   one selected orbit */
	 if ( max_ap < sorbit_array[ orbit ]->aorbit->apoapsis )
	    {
	    max_ap = sorbit_array[ orbit ]->aorbit->apoapsis;
#ifdef  OLD_DEBUG
	    sprintf( bw_ebuf, "pe_draw(): orbit %d, apoapsis = %.2g km",
	       orbit,
	       sorbit_array[ orbit ]->aorbit->apoapsis );
	    bw_debug( bw_ebuf );
#endif
	    }
	 }				/* if orbit selected */
      }					/* for loop */

#ifdef  DEBUG
   if ( selected == ( n_orbits + 1 ) )
      {
      bw_error( "[pr:] pe_draw(): no orbits initialized" );
      return BW_ERROR;
      }
#endif

   /* initial distance is max_ap plus focal radius */

   distance = max_ap + sorbit_array[ selected ]->aorbit->focus->radius;

   /* increase distance until the entire image will fit in the screen */

   while( spj_angrad( distance, max_ap + sorbit_array[ selected ]->aorbit->focus->radius )
      > ( vert_degrees / 2.0 ) )
      {

      distance *= 2.0;

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "pe_draw(): distance %.0lf km; ang. radius %.2lf deg",
      distance, spj_angrad( distance, sorbit_array[ selected ]->aorbit->focus->radius ));
   bw_debug( bw_ebuf );
#endif

      }

   /* Show the orbits */

   pe_plot( x_screen, uiwind, sorbit_array, n_orbits, vlat, vlon, rotation,
      distance, o_start, o_end, newtime, focus );

   /***  Show crosshairs */

#ifdef  BLOCKEDOUT
   pe_crosshair( uiwind, HORDEGS, x_screen );
#endif

   /* Show toggle status */

   el_show( x_screen, uiwind, elements );

   /***  blit and return screen to original state */

#ifdef  USEHIDDEN
   if ( gr_screens > 1 )
      {
      gr_blit( GR_HIDDEN, GR_PRIMARY, uiwind->u_x1, uiwind->u_y1,
	 uiwind->u_x2, uiwind->u_y2 );
      }
   ui_setscreen( GR_PRIMARY );
#endif

   /* Title the screen */

   sprintf( bw_ebuf, PE_WTITLE,
      focus->name,
      distance, sorbit_array[ selected ]->pe_inc );
   ui_wtitle( uiwind, bw_ebuf );

   }

/***************************************************************

	pe_plot()

	This routine draws all orbits as they appear in space.

***************************************************************/

pe_plot( screen, uiwind, sorbit_array, n_orbits, vlat, vlon, rotation, vdis,
   o_start, o_end, newtime, focus )
   int screen;
   struct uiwindow *uiwind;             /* ui  window to draw in */
   struct sfs_orbit **sorbit_array;
   int n_orbits;
   double vlat, vlon, rotation;
   double vdis;
   struct spj_pt *o_start, *o_end;      /* start, end orb display pattern */
   long newtime;
   struct as_focus *focus;
   {
   register int orbit, c;
   static double x, y;
   static int v;
   double prevx, prevy;
   int prevv;
   static struct uiwindow *oldwind = NULL; /* save window pointer to avoid recalculation */
   static int x_midx, x_midy;           /* x, y mid points of window */
   static double x_xfactor, x_yfactor;  /* factors for x, y expansion in window */
   static double vert_degrees;          /* calculate vertical degrees */
   int selected;
   long x_tio;                          /* time into orbit */

   /* set up the window if necessary */

   if ( uiwind != oldwind )
      {
      x_xfactor = ( uiwind->u_x2 - (double) uiwind->u_x1 ) / HORDEGS;
      vert_degrees = HORDEGS * ( ( ( uiwind->u_y2 - (double) uiwind->u_y1 )
	 * ( gr_pysize / (double) gr_pxsize ) )
	 / ( uiwind->u_x2 - (double) uiwind->u_x1 ) );
      x_yfactor = ( uiwind->u_y2 - (double) uiwind->u_y1 ) / vert_degrees;
      x_midx = uiwind->u_x1 + (( uiwind->u_x2 - (double) uiwind->u_x1 ) / 2.0 );
      x_midy = uiwind->u_y1 + (( uiwind->u_y2 - (double) uiwind->u_y1 ) / 2.0 );
      oldwind = uiwind;
      }

   /* Show the far side of each orbit */

   selected = n_orbits + 1;
   for ( orbit = 0; orbit < n_orbits; ++orbit )
      {
      if ( ( sorbit_array[ orbit ] != NULL )
         && ( sorbit_array[ orbit ]->aorbit->focus == focus ))
	 {
#ifdef  OLD_DEBUG
         sprintf( bw_ebuf, "Draw far side, orbit %d", orbit );
         bw_debug( bw_ebuf );
#endif
	 selected = orbit;
	 x_tio = newtime % (long) sorbit_array[ orbit ]->aorbit->period;
	 pe_show( screen, sorbit_array, orbit, x_tio,
	    vlat, vlon, vdis, rotation,
	    x_xfactor, x_yfactor, x_midx, x_midy,
	    SPJ_FARSIDE, cl_orbits[ orbit % 6 ], SOLID );
	 }
      }

   /* Show the orb of the planet */

   pe_orb( uiwind, screen,
      sorbit_array[ selected ]->aorbit->focus->radius,
      vdis, cl_grid, x_midx, x_midy, x_xfactor, x_yfactor );

   /* Now show the near side of each orbit */

   for ( orbit = 0; orbit < n_orbits; ++orbit )
      {
      if ( ( sorbit_array[ orbit ] != NULL )
         && ( sorbit_array[ orbit ]->aorbit->focus == focus ))
	 {
#ifdef  OLD_DEBUG
         sprintf( bw_ebuf, "Draw near side, orbit %d", orbit );
         bw_debug( bw_ebuf );
#endif
	 x_tio = newtime % (long) sorbit_array[ orbit ]->aorbit->period;
	 pe_show( screen, sorbit_array, orbit, x_tio,
	    vlat, vlon, vdis, rotation,
	    x_xfactor, x_yfactor, x_midx, x_midy,
	    SPJ_NEARSIDE, cl_orbits[ orbit % 6 ], SOLID );
	 }
      }

   }

/***************************************************************

	pe_show()

	This routine draws the front or back side of a specific
	orbit as it appears in space.

***************************************************************/

pe_show( screen, sorbit_array, orbit, tio, vlat, vlon, vdis, rotation,
   xfactor, yfactor, midx, midy,
   side, color, style )
   int screen;
   struct sfs_orbit **sorbit_array;
   int orbit;
   long tio;
   double vlat, vlon, vdis, rotation;
   double xfactor, yfactor;
   int midx, midy;
   int side;
   int color, style;
   {
   register int c;
   static double x, y;
   double prevx, prevy;
   static int s;
   int x_noline;

   spj_point( sorbit_array[ orbit ]->pe_buffer[ 0 ].latitude,
      sorbit_array[ orbit ]->pe_buffer[ 0 ].longitude,
      sorbit_array[ orbit ]->pe_buffer[ 0 ].altitude,
      vlat, vlon, vdis, rotation, side,
      &x, &y, &s );
   prevx = x;
   prevy = y;
   x_noline = TRUE;

   c = 0;
   while ( c < PE_POINTS )
      {

      s = 255;

      /* Get coordinates for the point */

      spj_point( sorbit_array[ orbit ]->pe_buffer[ c ].latitude,
	   sorbit_array[ orbit ]->pe_buffer[ c ].longitude,
	   sorbit_array[ orbit ]->pe_buffer[ c ].altitude,
	   vlat, vlon, vdis, rotation, side,
	   &x, &y, &s );

#ifdef  OLD_DEBUG
      sprintf( bw_ebuf, "sp_show(): x = %.2lf, prevx = %.2lf, y = %.2lf, prevy = %.2lf",
	 x, prevx, y, prevy );
      bw_debug( bw_ebuf );
#endif

      /* show the point */

      if ( s == side )
	 {
	 if ( ( ( c * sorbit_array[ orbit ]->pe_inc ) < tio )
	    && ( x_noline != TRUE ) )
	    {
	    gr_line( screen,
	       midx + ( (int) ( x     * xfactor )),
	       midy + ( (int) ( y     * yfactor )),
	       midx + ( (int) ( prevx * xfactor )),
	       midy + ( (int) ( prevy * yfactor )),
	       color, SOLID );
	    x_noline = FALSE;
	    }
	 else
	    {
	    gr_pixel( screen,
	       midx + ( (int) ( x * xfactor )),
	       midy + ( (int) ( y * yfactor )),
	       color );
	    x_noline = FALSE;
	    }
	 }

      /* point is not on the side selected; thus set x_noline */

      else
	 {
	 x_noline = TRUE;
	 }

      /* save x and y values and increment the counter */

      prevx = x;
      prevy = y;
      ++c;
      }

   }

/***************************************************************

	pe_orb()

	This routine draws the orb of the planet for the
	distant perspective display.

***************************************************************/

pe_orb( uiwind, screen, radius, distance, color, midx, midy,
   xfactor, yfactor )
   struct uiwindow *uiwind;             /* ui  window to draw in */
   int screen;
   double radius, distance;
   int color;
   int midx, midy;
   double xfactor, yfactor;
   {
   double x_angrad;                     /* angular radius */
   int x_yrad;                          /* radius on y axis */

   /*** 1. Calculate angular radius and y axis radius */

   x_angrad = spj_angrad( distance, radius );
   x_yrad   = (int) ( x_angrad * yfactor );

#ifdef  DEBUG
   if ( x_yrad < 1 )
      {
      sprintf( bw_ebuf, "pe_orb(): x_yrad = %d", x_yrad );
      bw_error( bw_ebuf );
      }
#endif

   /*** 1. Blank the orb area */

   gr_circle( screen, midx, midy, x_yrad, WHITE, SOLID );
   gr_circle( screen, midx, midy, x_yrad, BLACK, SOLID );

   /*** 2. Show the orb outline */

   gr_circle( screen, midx, midy, x_yrad, color, HOLLOW );

   }

pe_crosshair( uiwind, hor_deg, screen )
   struct uiwindow *uiwind;             /* ui window for display */
   double hor_deg;                      /* degrees of horizontal axis */
   int screen;                          /* screen to draw to */
   {
   static double vert_degrees;          /* calculate vertical degrees */
   int x_midx, x_midy;                  /* mid points of x, y axes */
   register int x_deg;                  /* counter for degrees */
   int x_xsize, x_ysize;                /* total x and y sizes of display */

   /* calculate vertical degrees */

   vert_degrees = hor_deg * ( ( ( uiwind->u_y2 - (double) uiwind->u_y1 )
      * ( gr_pysize / (double) gr_pxsize ) )
      / ( uiwind->u_x2 - (double) uiwind->u_x1 ) );

   /* calculate mid points */

   x_midx = uiwind->u_x1 + (( uiwind->u_x2 - (double) uiwind->u_x1 ) / 2.0 );
   x_midy = uiwind->u_y1 + (( uiwind->u_y2 - (double) uiwind->u_y1 ) / 2.0 );
   x_xsize = uiwind->u_x2 - uiwind->u_x1;
   x_ysize = uiwind->u_y2 - uiwind->u_y1;

   /* show vertical crosshairs */

   gr_line( screen, x_midx, uiwind->u_y1, x_midx, uiwind->u_y2,
      cl_marker, GRID );
   for ( x_deg = 10; x_deg < ( hor_deg / 2 ); x_deg += 10 )
      {
      gr_line( screen,
	 (int) ( x_midx + (( x_deg * x_xsize ) / hor_deg )),
	 uiwind->u_y1,
	 (int) ( x_midx + (( x_deg * x_xsize ) / hor_deg )),
	 uiwind->u_y2,
	 cl_marker, GRID );
      gr_line( screen,
	 (int) ( x_midx - (( x_deg * x_xsize ) / hor_deg )),
	 uiwind->u_y1,
	 (int) ( x_midx - (( x_deg * x_xsize ) / hor_deg )),
	 uiwind->u_y2,
	 cl_marker, GRID );
      }

   /* show horizontal crosshairs */

   gr_line( screen, uiwind->u_x1, x_midy, uiwind->u_x2, x_midy,
      cl_marker, GRID );
   for ( x_deg = 10; x_deg < ( vert_degrees / 2 ); x_deg += 10 )
      {
      gr_line( screen,
	 uiwind->u_x1,
	 (int) ( x_midy + (( x_deg * x_ysize ) / vert_degrees )),
	 uiwind->u_x2,
	 (int) ( x_midy + (( x_deg * x_ysize ) / vert_degrees )),
	 cl_marker, GRID );
      gr_line( screen,
	 uiwind->u_x1,
	 (int) ( x_midy - (( x_deg * x_ysize ) / vert_degrees )),
	 uiwind->u_x2,
	 (int) ( x_midy - (( x_deg * x_ysize ) / vert_degrees )),
	 cl_marker, GRID );
      }

   }


/***************************************************************

	el_toggle()             Toggle Simulation Elements

***************************************************************/

el_toggle( command, elements )
   int command;                         /* command (letter) entered */
   int *elements;              		/* pointer to simulation elements */
   {
   register int c;

   c = toupper( command );

   switch( c )
      {
      case    'O':
	 if ( ( *elements & VI_ORBOUTLINE ) > 0  )
	    {
	    *elements &= ~( VI_ORBOUTLINE );
	    }
	 else
	    {
	    *elements |= VI_ORBOUTLINE;
	    }
	 break;
      case    'C':
	 if ( ( *elements & VI_CROSSH ) > 0  )
	    {
	    *elements &= ~( VI_CROSSH );
	    }
	 else
	    {
	    *elements |= VI_CROSSH;
	    }
	 break;
      case    'G':
	 if ( ( *elements & VI_GRIDFRONT ) > 0 )
	    {
	    *elements = (*elements) & ~( VI_GRIDFRONT );
	    }
	 else
	    {
	    *elements = (*elements) | VI_GRIDFRONT;
	    }
	 break;
      case    'S':
	 if ( ( *elements & VI_SURFRONT ) > 0  )
	    {
	    *elements &= ~( VI_SURFRONT );
	    }
	 else
	    {
	    *elements |= VI_SURFRONT;
	    }
	 break;
      }
   }

/***************************************************************

	el_show()             Show Toggle Elements

***************************************************************/

el_show( screen, uiwind, elements )
   int screen;
   struct uiwindow *uiwind;
   unsigned int elements;
   {
   char x_buf[ 64 ];
   int x, y, l;

   /* calculate initial position of display streng */

   sprintf( x_buf, "%s:   MMMMM", SFS_TOGGLE );
   l = gr_strlen( x_buf );
   x = uiwind->u_x2 - ( l + 2 );
   y = uiwind->u_y2 - ( ui_grwind->fysize + 2 );

   /* blank the area */

   gr_rectangle( screen, x - 2, y - 1, x + l, y + ui_grwind->fysize + 1,
      cl_mback, SOLID );

   /* display string without toggles */

   sprintf( x_buf, "%s:  ", SFS_TOGGLE );
   gr_text( screen, x, y, x_buf, cl_mfore, cl_mback );

   /* increment x to first toggle position */

   l = gr_strlen( x_buf );
   x += l;

   /* show grid toggle status */

   if ( ( elements & VI_GRIDFRONT ) > 0 )
      {
      gr_text( screen, x, y, "G", cl_mfore, cl_mback );
      }
   else
      {
      gr_text( screen, x, y, "G", cl_mback, cl_mfore );
      }

   /* increment x to next character position */

   x += gr_strlen( "G" ) + 2;
      
   /* show surface toggle status */

   if ( ( elements & VI_SURFRONT ) > 0 )
      {
      gr_text( screen, x, y, "S", cl_mfore, cl_mback );
      }
   else
      {
      gr_text( screen, x, y, "S", cl_mback, cl_mfore );
      }

   /* increment x to next character position */

   x += gr_strlen( "S" ) + 2;
      
   /* show orb toggle status */

   if ( ( elements & VI_ORBOUTLINE ) > 0 )
      {
      gr_text( screen, x, y, "O", cl_mfore, cl_mback );
      }
   else
      {
      gr_text( screen, x, y, "O", cl_mback, cl_mfore );
      }

   /* increment x to next character position */

   x += gr_strlen( "O" ) + 2;
      
   /* show crosshair toggle status */

   if ( ( elements & VI_CROSSH ) > 0 )
      {
      gr_text( screen, x, y, "C", cl_mfore, cl_mback );
      }
   else
      {
      gr_text( screen, x, y, "C", cl_mback, cl_mfore );
      }

   }


