/*
   Module:  hpparms.c
   Date:    3/9/92
   Version: 1.0b
   Author:  Dave Lutz
   Email:   lutz@psych.rochester.edu
   Copyright: 1992 University of Rochester, Psychology Dept.

   Disclaimer:  This software is distributed free of charge.  As such, it
                comes with ABSOLUTELY NO WARRANTY.  The user of the software
                assumes ALL RISKS associated with its use.

                Your rights to modify and/or distribute this software are
                outlined in the file ADI7221.DOC.

   Purpose: This module provides the functions required to convert ADI
            code arguments to HP parameter codes.

   Functions provided:

        uinttosbn    (unsigned int to Single Byte Number)
        uiptombp     (unsigned int pair to Multiple Byte Pair)
*/

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

/*  Prototypes for private functions */
int calcbytes (unsigned larger);
void twobyte (unsigned num1, unsigned num2, char *mbpout);
void threebyte (unsigned num1, unsigned num2, char *mbpout);
void fourbyte (unsigned num1, unsigned num2, char *mbpout);
void fivebyte (unsigned num1, unsigned num2, char *mbpout);

/*
   Function: uinttosbn
   Purpose:  Convert an unsigned int to an HP Single Byte Number.

   Pre: numin is the number to convert.
        0 <= numin <= 63
        sbnout is a pointer to storage for the converted Single Byte Number.

   Post: An attempt is made to convert numin to SBN format and copy the
         retult to sbnout.
         If an error occurs (ie numin is out of range), FALSE is returned.
         Otherwise, TRUE is returned.
*/

int uinttosbn (numin, sbnout)
   unsigned numin;
   char *sbnout;
{
   if (numin > 63)
      return (FALSE);
   *sbnout = (char) (numin > 31) ? numin : numin+64;
   return (TRUE);
}



/*
   Function: uiptombp
   Purpose: Convert a pair of unsigned ints to an HP Multiple Byte Pair.

   Pre: num1 and num2 comprise the pair of numbers to be converted to an
        HP MBP.
        0 <= num1,num2 <= 16383
        mbpout is a pointer to storage for the converted code.
        mbpout points to enough storage to hold the maximum size of a mbp
        formatted code (MAXMBPLEN).
        mbpsize is a pointer to storage for the number of bytes actually
        placed in mbpout.

   Post: an attempt is made to convert num1 and num2 to MBP format and copy
         the result to mbpout.
         The number of bytes in mbpout is returned in mbpsize.
         If an error occurs (num1 or num2 are out of range), FALSE is
         returned.
         Otherwise, TRUE is returned.

*/

int uiptombp (num1, num2, mbpout, mbpsize)
   unsigned num1, num2;
   char *mbpout;
   int *mbpsize;
{
   unsigned larger;

   if ((num1 > 16383) || (num2 > 16383))
      return (FALSE);

   larger = (num1 > num2) ? num1 : num2;
   *mbpsize = calcbytes (larger);

   switch (*mbpsize) {
      case 1:
         /* the following formula is explained on p208 of the HP manual */
         *mbpout=num2 + 96 + 4 * num1;
         break;
      case 2:
         twobyte(num1,num2,mbpout);
         break;
      case 3:
         threebyte(num1,num2,mbpout);
         break;
      case 4:
         fourbyte(num1,num2,mbpout);
         break;
      case 5:
         fivebyte(num1,num2,mbpout);
         break;
   }
   return (TRUE);
}

/*
   Function: calcbytes
   Purpose:  This private procedure calculates the number of bytes needed to 
             store a Multiple Byte Pair.

             This procedure is based on the flow chart on page 207 of the
             "Hewlett-Packard 7221A Graphics Plotter Operating and
             Programming Manual."

   Pre: larger is the larger of the unsigned int pair.
        It has already been determined that 0 <= larger <= 16383.

   Post: The number of bytes needed to store the MBP is returned.
*/

int calcbytes (larger)
   unsigned larger;
{
   if (larger < 256)
      if (larger > 31)
         return (3);
      else if (larger > 3)
         return (2);
      else
         return (1);
   else if (larger < 2048)
      return (4);
   else if (larger < 16384)
      return (5);

   /* we should never reach this point */
   return (0);
}


/*
   Function: twobyte
   Purpose:  This private procedure calculates a Multiple Byte Pair when it 
             is known that unsigned int pair is valid and will require two 
             bytes for an MBP representation.

             The procedure is based on the flowchart at the bottom of page 
             208 in the "Hewlett-Packard 7221A Graphics Plotter Operating and
             Programming Manual."

   Pre: It has been determined that num1 and num2 are in the range that will
        produce a two byte MBP.
        mbpout is a pointer to storage for the two byte MBP.
        mbpout points to at least two bytes of storage.

   Post: num1 and num2 are converted to a MBP that is stored in mbpout.
*/

void twobyte (num1, num2, mbpout)
   unsigned num1, num2;
   char *mbpout;
{
   unsigned x1, x2;

   x1 = num1 / 2;
   x2 = num1 -2 * x1;
   *mbpout++ = (char) (x1 + 96);
   *mbpout = (char) (num2 + 32 * x2);
   if (*mbpout <= (char) 31)
      *mbpout += (char) 64;
}


/*
   Function: threebyte
   Purpose:  This private procedure calculates a Multiple Byte Pair when it 
             is known that unsigned int pair is valid and will require three
             bytes for an MBP representation.

             The following procedure is based on the flowchart on p209 of
             the "Hewlett-Packard 7221A Graphics Plotter Operating and
             Programming Manual."

   Pre: It has been determined that num1 and num2 are in the range that will
        produce a three byte MBP.
        mbpout is a pointer to storage for the three byte MBP.
        mbpout points to at leat three bytes of storage.

   Post: num1 and num2 are converted to a MBP that is stored in mbpout.
*/

void threebyte (num1, num2, mbpout)
   unsigned num1, num2;
   char *mbpout;
{
   unsigned x1, x2, y2, y3;

   x1 = num1 / 16;
   x2 = num1 - 16 * x1;
   y2 = num2 / 64;
   y3 = num2 - 64 * y2;
   *mbpout++ = (char) (x1 + 96);
   *mbpout = (char) (y2 + 4 * x2);
   if (*mbpout <= (char) 31)
      *mbpout += 64;
   mbpout++;
   *mbpout = (char) (y3);
   if (*mbpout <= (char) 31)
      *mbpout += 64;
}


/*
   Function: fourbyte
   Purpose:  This private procedure calculates a Multiple Byte Pair when it 
             is known that unsigned int pair is valid and will require four
             bytes for an MBP representation.

             This procedure is based on the flow chart on p210 of the
             "Hewlett-Packard 7221A Graphics Plotter Operating and
             Programming Manual."

   Pre: It has been determined that num1 and num2 are in the range that will
        produce a four byte MBP.
        mbpout is a pointer to storage for the four byte MBP.
        mbpout points to at leat four bytes of storage.

   Post: num1 and num2 are converted to a MBP that is stored in mbpout.
*/

void fourbyte (num1, num2, mbpout)
   unsigned num1, num2;
   char *mbpout;
{
   unsigned x1, xr, x2, x3, y3, y4;

   x1 = num1 / 128;
   xr = num1 - 128 * x1;
   x2 = xr / 2;
   x3 = xr - 2 * x2;
   y3 = num2 / 64;
   y4 = num2 - 64 * y3;
   *mbpout++ = (char) (x1 + 96);
   *mbpout = (char) (x2);
   if (*mbpout <= (char) 31)
      *mbpout += 64;
   mbpout++;
   *mbpout = (char) (y3 + 32 * x3);
   if (*mbpout <= (char) 31)
      *mbpout += 64;
   mbpout++;
   *mbpout = (char) (y4);
   if (*mbpout <= (char) 31)
      *mbpout += 64;
}


/*
   Function: fivebyte
   Purpose:  This private procedure calculates a Multiple Byte Pair when it 
             is known that unsigned int pair is valid and will require five
             bytes for an MBP representation.

             This procedure is based on the flow chart on p211 of the
             "Hewlett-Packard 7221A Graphics Plotter Operating and
             Programming Manual."

   Pre: It has been determined that num1 and num2 are in the range that will
        produce a five byte MBP.
        mbpout is a pointer to storage for the five byte MBP.
        mbpout points to at leat five bytes of storage.

   Post: num1 and num2 are converted to a MBP that is stored in mbpout.
*/

void fivebyte (num1, num2, mbpout)
   unsigned num1, num2;
   char *mbpout;
{
   unsigned x1, xr, x2, x3, y3, yr, y4, y5;

   x1 = num1 / 1024;
   xr = num1 - 1024 * x1;
   x2 = xr / 16;
   x3 = xr - 16 * x2;
   y3 = num2 / 4096;
   yr = num2 - 4096 * y3;
   y4 = yr / 64;
   y5 = yr - 64 * y4;
   *mbpout++ = (char) (x1 + 96);
   *mbpout = (char) (x2);
   if (*mbpout <= (char) 31)
      *mbpout += 64;
   mbpout++;
   *mbpout = (char) (y3 + 4 * x3);
   if (*mbpout <= (char) 31)
      *mbpout += 64;
   mbpout++;
   *mbpout = (char) (y4);
   if (*mbpout <= (char) 31)
      *mbpout += 64;
   mbpout++;
   *mbpout = (char) (y5);
   if (*mbpout <= (char) 31)
      *mbpout += 64;
}
