#include "copyleft.h"

/*
    GEPASI - a simulator of metabolic pathways and other dynamical systems
    Copyright (C) 1989, 1992  Pedro Mendes
*/

/*************************************/
/*                                   */
/*   databases and their functions   */
/*                                   */
/*        Zortech C/C++ 3.0 r4       */
/*          MICROSOFT C 6.00         */
/*          Visual C/C++ 1.0         */
/*           QuickC/WIN 1.0          */
/*             ULTRIX cc             */
/*              GNU gcc              */
/*                                   */
/*   (include here compilers that    */
/*   compiled GEPASI successfully)   */
/*                                   */
/*************************************/


#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#ifdef _ZTC
#define MEM_DEBUG 1
#include "mem.h"
#else
#define mem_malloc malloc
#define mem_free free
#define mem_realloc realloc
#endif

#include "globals.h"
#include "globvar.h"
#include "lsoda.h"
#include "strtbl.h"
#include "heapchk.h"

/* structure holding information relative to kinetic mechanisms */

struct kint {
				unsigned char nsub;             /* number ofsubstrates  */
				unsigned char npro;             /* number of products   */
				unsigned char nmodf;    /* number of modifiers  */
				unsigned char revers;   /* reversability                */
				unsigned char nconst;   /* number of kin. const.*/
				char *descr;                    /* title                                */
				char *constnam;                 /* kin. constant names  */
		    };

/* structures for user-defined rate equations                                   */

struct nodet{
	     char item;
	     unsigned char val;
	     unsigned char left;
	     unsigned char right;
	    } ;

struct treet{
	     struct nodet node[256];
	     char id[64][10];
	     float constant[32];
	     int nnode,
		 nnum,
		 nid,
		 nsub,
		 npro,
		 nmodf,
		 nconst,
		 revers;
	     char descr[64];
	    } ;

/* structure holding simulation options                                                 */

struct opt {
	    int		dyn;
	    long	pfo;
		double	endtime;
	    double	reltol;
	    double	abstol;
	    double	hrcz;
	    int		adams;
	    int		bdf;
	    int		ss;
	    int		debug;
	    int		txt;
	    int		structan;
	    int		staban;
	    int		stdela;
	    int		nonela;
	    int		stdcc;
	    int		noncc;
	    int		dat;
		char	datname[PWIDTH];
	    int		datsep;
	    int		datwidth;
	    int		dattit;
	    int		datmca;
	    int		datss;
	    int		append;
	    int		quotes;
		char	timeu[32];
		char	concu[32];
	    int		scan;
	    int		scandens;
	    int		scanlog;
	   };

struct sp{
	      PDBL var;
	      double low;
	      double high;
	      double ampl;
	      int dens;
	      int log;
	      PDBL linkedto;
	      double factor;
	      int operation;
	     };

PDBL    params[MAX_STEP];                                                       /* ptr to parameters for each rate eq.  */
int     *eff[MAX_STEP];                                                         /* ptr to parameters for each rate eq.  */
unsigned char	revers[MAX_STEP];                                       /* 1 if reaction is reversible                  */
struct  opt options;                                                            /* structure with simulation options    */
struct  kint *ktype;                                                            /* ptr array of kinetic types & proprt  */
struct  sp *sparam;                                                                     /* ptr array with parameters to scan    */
PDBL    *scanpar;                                                                       /* ptrs to all possbl.params 4 scanning */
PDBL    *outpel;                                                                        /* ptrs to all possbl. values for output*/
PDBL    *poutelem;                                                                        /* ptrs to values for actual dat output */
char    *outtit;                                                                        /* pointer to buffer with column titles */
char    *treestr;                                                                       /* pointer to buffer with constant names*/
char    *treeptr;                                                                       /* pointer to buffer with constant names*/
int     *scindex;                                                                       /* ptr to base of array with idx to scan*/
int     *lindex;                                                                        /* ptr to base of array with idx 2 links*/
int     nscanpar;                                   /* number of scanning elements                      */
int     noutpel;                                    /* number of output elements                        */
int     totscan;                                    /* number of selected scanning params       */
int     nlinks;                                                                         /* number of linked parameters                  */
int     nudf;                                                                           /* number of user-def functions                 */
int     nrateq;                                                                         /* number of total rate equations               */
int     totsel;                                     /* number of selected output elements       */
int     kfl[MAX_STEP];                                                          /* flags for kinetic types                              */
unsigned int sizespar;
unsigned int sizeparam;
unsigned int sizeeff;
unsigned int sizeoutp;
unsigned int sizetr;
struct  treet *tree;                                                            /* function tree for rate equations             */
struct  treet tr;                                                                       /* tree for the input                                   */

/* main point for memory block allocation               */
int MemAlloc( void )
{
 int i;


 sizeparam = sizeof( double );
 params[0] = (double *) mem_malloc( sizeparam );
 if( params[0] == NULL ) return -1;
 /* make the other params[i] point to NULL                                                                */
 for( i=1; i<MAX_STEP; i++ ) params[i] = NULL;

 sizeeff = 1;
 eff[0] = (int *) mem_malloc( sizeeff * sizeof( int ) );
 if( eff[0] == NULL )
 {
  mem_free( params[0] );
  return -1;
 }
 /* make the other eff[i] point to NULL                                                                   */
 for( i=1; i<MAX_STEP; i++ ) eff[i] = NULL;

 /* allocate space for the MAX_TYPE descriptors of kinetic types	*/
 ktype = (struct kint *) mem_malloc( MAX_TYP * sizeof( struct kint ) );
 if( ktype == NULL )
 {
  mem_free( params[0] );
  mem_free( eff[0] );
  return -1;
 }

 sizeoutp = 4 * sizeof( PDBL );
 outpel = (PDBL *) mem_malloc( sizeoutp );
 if( outpel == NULL )
 {
  mem_free( ktype );
  mem_free( params[0] );
  mem_free( eff[0] );
  return -1;
 }

 poutelem = (PDBL *) mem_malloc( sizeof(PDBL) );
 if( poutelem == NULL )
 {
  mem_free( outpel );
  mem_free( ktype );
  mem_free( params[0] );
  mem_free( eff[0] );
  return -1;
 }

 loop = ( unsigned char (*)[MAX_STEP][MAX_MET] ) mem_malloc( MAX_STEP * MAX_MET * sizeof( unsigned char ) );
 if( loop == NULL )
 {
  mem_free( poutelem );
  mem_free( outpel );
  mem_free( ktype );
  mem_free( params[0] );
  mem_free( eff[0] );
  return -1;
 }

 rstr = ( int (*)[MAX_STEP][MAX_MOL] ) mem_malloc( MAX_STEP * MAX_MOL * sizeof( int ) );
 if( rstr == NULL )
 {
  mem_free( loop );
  mem_free( poutelem );
  mem_free( outpel );
  mem_free( ktype );
  mem_free( params[0] );
  mem_free( eff[0] );
  return -1;
 }

 outtit = ( char * ) mem_malloc( 25 * sizeof( char ) );
 if( outtit == NULL )
 {
  mem_free( rstr );
  mem_free( loop );
  mem_free( poutelem );
  mem_free( outpel );
  mem_free( ktype );
  mem_free( params[0] );
  mem_free( eff[0] );
  return -1;
 }

 sparam = (struct sp *) mem_malloc( sizeof( struct sp ) );
 if( sparam == NULL )
 {
  mem_free( outtit );
  mem_free( rstr );
  mem_free( loop );
  mem_free( poutelem );
  mem_free( outpel );
  mem_free( ktype );
  mem_free( params[0] );
  mem_free( eff[0] );
  return -1;
 }

 sizespar = 4 * sizeof( PDBL );
 scanpar = (PDBL *) mem_malloc( sizespar );
 if( scanpar == NULL )
 {
  mem_free( sparam );
  mem_free( outtit );
  mem_free( rstr );
  mem_free( loop );
  mem_free( poutelem );
  mem_free( outpel );
  mem_free( ktype );
  mem_free( params[0] );
  mem_free( eff[0] );
  return -1;
 }

 scindex = (int *) mem_malloc( 4 * sizeof(int) );
 if( scindex == NULL )
 {
  mem_free( scanpar );
  mem_free( sparam );
  mem_free( outtit );
  mem_free( rstr );
  mem_free( loop );
  mem_free( poutelem );
  mem_free( outpel );
  mem_free( ktype );
  mem_free( params[0] );
  mem_free( eff[0] );
  return -1;
 }

 lindex = (int *) mem_malloc( sizeof(int) );
 if( lindex == NULL )
 {
  mem_free( scindex );
  mem_free( scanpar );
  mem_free( sparam );
  mem_free( outtit );
  mem_free( rstr );
  mem_free( loop );
  mem_free( poutelem );
  mem_free( outpel );
  mem_free( ktype );
  mem_free( params[0] );
  mem_free( eff[0] );
  return -1;
 }

 tree = (struct treet *) mem_malloc( sizeof( struct treet ) );
 if( tree == NULL )
 {
  mem_free( lindex );
  mem_free( scindex );
  mem_free( scanpar );
  mem_free( sparam );
  mem_free( outtit );
  mem_free( rstr );
  mem_free( loop );
  mem_free( poutelem );
  mem_free( outpel );
  mem_free( ktype );
  mem_free( params[0] );
  mem_free( eff[0] );
  return -1;
 }

 sizetr = 4;
 treestr = (char *) mem_malloc( sizetr * sizeof( char ) );
 if( treestr == NULL )
 {
  mem_free( tree );
  mem_free( lindex );
  mem_free( scindex );
  mem_free( scanpar );
  mem_free( sparam );
  mem_free( outtit );
  mem_free( rstr );
  mem_free( loop );
  mem_free( poutelem );
  mem_free( outpel );
  mem_free( ktype );
  mem_free( params[0] );
  mem_free( eff[0] );
  return -1;
 }

 return 0;
}

/* Initialization of databases */

void InitDataBase( void )
{
 ktype[NOT].nsub = 0;                                                           /* init info of kinetic types   */
 ktype[NOT].npro = 0;
 ktype[NOT].nconst = 0;
 ktype[NOT].nmodf = 0;
 ktype[NOT].constnam = "";
 ktype[NOT].descr = "<not defined>";
 ktype[I01].nsub = 0;
 ktype[I01].npro = 1;
 ktype[I01].nconst = 1;
 ktype[I01].nmodf = 0;
 ktype[I01].constnam = "k";
 ktype[I01].descr = "constant rate";
 ktype[I10].nsub = 1;
 ktype[I10].npro = 0;
 ktype[I10].nconst = 1;
 ktype[I10].nmodf = 0;
 ktype[I10].constnam = ktype[I01].constnam;
 ktype[I10].descr = ktype[I01].descr;
 ktype[I11].nsub = 1;
 ktype[I11].npro = 1;
 ktype[I11].revers = 0;
 ktype[I11].nconst = 1;
 ktype[I11].nmodf = 0;
 ktype[I11].constnam = ktype[I01].constnam;
 ktype[I11].descr = "mass action";
 ktype[R11].nsub = 1;
 ktype[R11].npro = 1;
 ktype[R11].nconst = 2;
 ktype[R11].nmodf = 0;
 ktype[R11].constnam = "k1\0k2";
 ktype[R11].descr = ktype[I11].descr;
 ktype[I21].nsub = 2;
 ktype[I21].npro = 1;
 ktype[I21].nconst = 1;
 ktype[I21].nmodf = 0;
 ktype[I21].constnam = ktype[I01].constnam;
 ktype[I21].descr = ktype[I11].descr;
 ktype[R21].nsub = 2;
 ktype[R21].npro = 1;
 ktype[R21].nconst = 2;
 ktype[R21].nmodf = 0;
 ktype[R21].constnam = ktype[R11].constnam;
 ktype[R21].descr = ktype[I11].descr;
 ktype[I12].nsub = 1;
 ktype[I12].npro = 2;
 ktype[I12].nconst = 1;
 ktype[I12].nmodf = 0;
 ktype[I12].constnam = ktype[I01].constnam;
 ktype[I12].descr = ktype[I11].descr;
 ktype[R12].nsub = 1;
 ktype[R12].npro = 2;
 ktype[R12].nconst = 2;
 ktype[R12].nmodf = 0;
 ktype[R12].constnam = ktype[R11].constnam;
 ktype[R12].descr = ktype[I11].descr;
 ktype[I31].nsub = 3;
 ktype[I31].npro = 1;
 ktype[I31].nconst = 1;
 ktype[I31].nmodf = 0;
 ktype[I31].constnam = ktype[I01].constnam;
 ktype[I31].descr = ktype[I11].descr;
 ktype[R31].nsub = 3;
 ktype[R31].npro = 1;
 ktype[R31].nconst = 2;
 ktype[R31].nmodf = 0;
 ktype[R31].constnam = ktype[R11].constnam;
 ktype[R31].descr = ktype[I11].descr;
 ktype[I13].nsub = 1;
 ktype[I13].npro = 3;
 ktype[I13].nconst = 1;
 ktype[I13].nmodf = 0;
 ktype[I13].constnam = ktype[I01].constnam;
 ktype[I13].descr = ktype[I11].descr;
 ktype[R13].nsub = 1;
 ktype[R13].npro = 3;
 ktype[R13].nconst = 2;
 ktype[R13].nmodf = 0;
 ktype[R13].constnam = ktype[R11].constnam;
 ktype[R13].descr = ktype[I11].descr;
 ktype[I22].nsub = 2;
 ktype[I22].npro = 2;
 ktype[I22].nconst = 1;
 ktype[I22].nmodf = 0;
 ktype[I22].constnam = ktype[I01].constnam;
 ktype[I22].descr = ktype[I11].descr;
 ktype[R22].nsub = 2;
 ktype[R22].npro = 2;
 ktype[R22].nconst = 2;
 ktype[R22].nmodf = 0;
 ktype[R22].constnam = ktype[R11].constnam;
 ktype[R22].descr = ktype[I11].descr;
 ktype[I32].nsub = 3;
 ktype[I32].npro = 2;
 ktype[I32].nconst = 1;
 ktype[I32].nmodf = 0;
 ktype[I32].constnam = ktype[I01].constnam;
 ktype[I32].descr = ktype[I11].descr;
 ktype[R32].nsub = 3;
 ktype[R32].npro = 2;
 ktype[R32].nconst = 2;
 ktype[R32].nmodf = 0;
 ktype[R32].constnam = ktype[R11].constnam;
 ktype[R32].descr = ktype[I11].descr;
 ktype[I23].nsub = 2;
 ktype[I23].npro = 3;
 ktype[I23].nconst = 1;
 ktype[I23].nmodf = 0;
 ktype[I23].constnam = ktype[I01].constnam;
 ktype[I23].descr = ktype[I11].descr;
 ktype[R23].nsub = 2;
 ktype[R23].npro = 3;
 ktype[R23].nconst = 2;
 ktype[R23].nmodf = 0;
 ktype[R23].constnam = ktype[R11].constnam;
 ktype[R23].descr = ktype[I11].descr;
 ktype[I33].nsub = 3;
 ktype[I33].npro = 3;
 ktype[I33].nconst = 1;
 ktype[I33].nmodf = 0;
 ktype[I33].constnam = ktype[I01].constnam;
 ktype[I33].descr = ktype[I11].descr;
 ktype[R33].nsub = 3;
 ktype[R33].npro = 3;
 ktype[R33].nconst = 2;
 ktype[R33].nmodf = 0;
 ktype[R33].constnam = ktype[R11].constnam;
 ktype[R33].descr = ktype[I11].descr;
 ktype[IMM].nsub = 1;
 ktype[IMM].npro = 1;
 ktype[IMM].nconst = 2;
 ktype[IMM].nmodf = 0;
 ktype[IMM].constnam = "Km\0V";
 ktype[IMM].descr = "Michaelis-Menten";
 ktype[RMM].nsub = 1;
 ktype[RMM].npro = 1;
 ktype[RMM].nconst = 4;
 ktype[RMM].nmodf = 0;
 ktype[RMM].constnam = "Kms\0Kmp\0Vf\0Vr";
 ktype[RMM].descr = "Reversible Michaelis-Menten";
 ktype[PSI].nsub = 1;
 ktype[PSI].npro = 1;
 ktype[PSI].nconst = 5;
 ktype[PSI].nmodf = 1;
 ktype[PSI].constnam = "Kms\0Kmp\0Vf\0Vr\0Ki";
 ktype[PSI].descr = "Specific inhibition";
 ktype[PCI].nsub = 1;
 ktype[PCI].npro = 1;
 ktype[PCI].nconst = 5;
 ktype[PCI].nmodf = 1;
 ktype[PCI].constnam = ktype[PSI].constnam;
 ktype[PCI].descr = "Catalytic inhibition";
 ktype[MXI].nsub = 1;
 ktype[MXI].npro = 1;
 ktype[MXI].nconst = 6;
 ktype[MXI].nmodf = 1;
 ktype[MXI].constnam = "Kms\0Kmp\0Vf\0Vr\0Ki\0Ki'";
 ktype[MXI].descr = "Mixed inhibition";
 ktype[PSA].nsub = 1;
 ktype[PSA].npro = 1;
 ktype[PSA].nconst = 5;
 ktype[PSA].nmodf = 1;
 ktype[PSA].constnam = "Kms\0Kmp\0Vf\0Vr\0Ka";
 ktype[PSA].descr = "Specific activation";
 ktype[PCA].nsub = 1;
 ktype[PCA].npro = 1;
 ktype[PCA].nconst = 5;
 ktype[PCA].nmodf = 1;
 ktype[PCA].constnam = ktype[PSA].constnam;
 ktype[PCA].descr = "Catalytic activation";
 ktype[MXA].nsub = 1;
 ktype[MXA].npro = 1;
 ktype[MXA].nconst = 6;
 ktype[MXA].nmodf = 1;
 ktype[MXA].constnam = "Kms\0Kmp\0Vf\0Vr\0Ka\0Ka'";
 ktype[MXA].descr = "Mixed activation";
 ktype[GOM].nsub = 1;
 ktype[GOM].npro = 1;
 ktype[GOM].nconst = 6;
 ktype[GOM].nmodf = 1;
 ktype[GOM].constnam = "Kms\0Kmp\0Vf\0Vr\0Ki\0Ki'";
 ktype[GOM].descr = "Botts-Morales";
 ktype[HIL].nsub = 1;
 ktype[HIL].npro = 1;
 ktype[HIL].nconst = 3;
 ktype[HIL].nmodf = 0;
 ktype[HIL].constnam = "Km\0V\0nh";
 ktype[HIL].descr = "Hill";
 ktype[UBS].nsub = 1;
 ktype[UBS].npro = 2;
 ktype[UBS].nconst = 6;
 ktype[UBS].nmodf = 0;
 ktype[UBS].constnam = "Ka\0Kp\0Kq\0Kpq\0Vf\0Vr";
 ktype[UBS].descr = "Ordered Uni Bi (A=P+Q)";
 ktype[UBM].nsub = 1;
 ktype[UBM].npro = 1;
 ktype[UBM].nconst = 6;
 ktype[UBM].nmodf = 0;
 ktype[UBM].constnam = "Ka\0Kp\0Vf\0Vr";
 ktype[UBM].descr = "Ordered Uni Bi (A=2*P)";
 ktype[ALI].nsub = 1;
 ktype[ALI].npro = 1;
 ktype[ALI].nconst = 6;
 ktype[ALI].nmodf = 1;
 ktype[ALI].constnam = "Kms\0Kmp\0Vf\0Vr\0Ki\0n";
 ktype[ALI].descr = "Allosteric inhibition";
}

/*
	set up the pointers to params
*/

int SetParams( void )
{
 int i, j;

 /* calculate the number of kinetic parameters                  */
 for( i=0, j=0; i<nsteps; i++ )
  j += ktype[kinetype[i]].nconst;
 /* and the space they take in memory                                   */
 sizeparam = j * sizeof( double );
 /* reallocate memory for parameters                                    */
 /* setting already the first (0) pointer                               */
 params[0] = (double *) mem_realloc( params[0], sizeparam );
 if( params[0] == NULL )
 {
  return -1;
 }
 /* set the other pointers                                                              */
 for( i=1; i<nsteps; i++ )
  params[i] = params[i-1] + ktype[kinetype[i-1]].nconst;
 /* return successfuly                                                                  */
 return 0;
}

/*
	set up the pointers to eff (indexes of effectors)
*/

int SetEff( void )
{
 int i, j;

 /* calculate the total number of effectors                     */
 for( i=0, sizeeff=0; i<nsteps; i++ )
 {             
  j = ktype[kinetype[i]].nsub +
      ktype[kinetype[i]].npro +
      ktype[kinetype[i]].nmodf;
  sizeeff += j;
 }

 /* mem_reallocate memory for effector indexes                          */
 /* setting already the first (0) pointer                               */
 eff[0] = (int *) mem_realloc( eff[0], sizeeff * sizeof( int ) );
 if( eff[0] == NULL )
 {
  return -1;
 }
 /* set the other pointers                                                              */
 for( i=1; i<nsteps; i++ )
  eff[i] = eff[i-1] + ( ktype[kinetype[i-1]].nsub +
						ktype[kinetype[i-1]].npro +
						ktype[kinetype[i-1]].nmodf );
 /* return successfuly                                                                  */
 return 0;
}


/* setup the output and scanning elements                               */

int SetOutpEl( void )
{
 int i, j, k, l;

 /* calculate the number of output elements                             */
 /* all metabolite concentrations and step fluxes               */
 noutpel = 2*totmet + nsteps;
 nscanpar = totmet;
 /* all kinetic constants...                                                     */
 for( i=0; i<nsteps; i++ )
 {
  noutpel += ktype[kinetu[i]].nconst;
  nscanpar += ktype[kinetu[i]].nconst;
 }
 /* all elasticities                                                                    */
 noutpel += nsteps * totmet;
 /* all concentration control coefficients                              */
 noutpel += totmet * nsteps;
 /* all flux control coefficients                                               */
 noutpel += nsteps * nsteps;
 /* endtime, rel & abs tol, nfunc, njac, nistep, smstep */
 noutpel += 8;
 nscanpar += 4;

 /* calculate the size of outpel and the string pool    */
 sizeoutp = noutpel * sizeof( PDBL );
 sizespar = nscanpar * sizeof( struct sp );

 /* reallocate memory for outpel                                                */
 outpel = (PDBL *) mem_realloc( outpel, sizeoutp );
 if( outpel == NULL )
  return -1;

 /* mem_reallocate memory for sparam                                            */
 sparam = (struct sp *) mem_realloc( sparam, sizespar );
 if( sparam == NULL )
  return -1;

 /* mem_reallocate memory for scindex                                           */
 scindex = (int *) mem_realloc( scindex, nscanpar * sizeof( int ) );
 if( scindex == NULL )
 {
  return -1;
 }

 /* mem_reallocate memory for lindex                                            */
 lindex = (int *) mem_realloc( lindex, nscanpar * sizeof( int ) );
 if( lindex == NULL )
 {
  return -1;
 }

 /* load the pointers                                                                   */

 /* initial concentrations                                                      */
 for( i=l=0; i<totmet; i++, l++ )
 {
  outpel[i] = &(x0[i]);
  sparam[l].var = &(x0[i]);
 }
 /* final concentrations                                                                */
 for( j=0; j<totmet; i++, j++ )
  outpel[i] = &(x[j]);

 /* fluxes                                                                              */
 for( j=0; j<nsteps; j++, i++ )
  outpel[i] = &(flux[j]);

 /* kinetic constants                                                                   */
 for( j=0; j<nsteps; j++ )
  for( k=0; k < (int) ktype[kinetype[j]].nconst; k++, i++, l++ )
   outpel[i] = sparam[l].var = params[j]+k;

 /* elasticities                                                                        */
  for( j=0; j<nsteps; j++ )
   for(k=0;k<totmet;k++,i++)
    outpel[i] = &Dxv[j][k];

 /* concentration control coefficients                                  */
 for( j=0; j<totmet; j++ )
   for(k=0;k<nsteps;k++,i++)
    outpel[i] = &Gamma[j][k];

 /* flux control coefficients                                                   */
 for( j=0; j<nsteps; j++ )
  for(k=0;k<nsteps;k++,i++)
   outpel[i] = &C[j][k];

 /* endtime, tolerance, nfunc, njacob, nistep, smstep   */
 outpel[i++] = sparam[l++].var = &endtime;
 outpel[i++] = &intst;
 outpel[i++] = &nfeval;
 outpel[i++] = &njeval;
 outpel[i++] = &hu;
 outpel[i++] = sparam[l++].var = &options.reltol;
 outpel[i++] = sparam[l++].var = &options.abstol;
 outpel[i]   = sparam[l].var = &options.hrcz;

 return 0;
}

/*
   create a new function tree object
*/

int new_tree( int idx )
{
 nudf++;
 tree = (struct treet *) mem_realloc( (void *) tree, nudf * sizeof( struct treet ) );
 if( tree == NULL )
  return IDS_ERR_NOEXEC;
 memcpy( (void *) &tree[idx], (void *) &tr, sizeof( struct treet ) );
 return 0;
}

/*
  add a user-defined rate equation to the database
*/

int new_rateq( int idx )
{
 unsigned int strsize;
 int i, pm, ef;
 long l;
 char *tmpptr;

 /* measure the required length for the constant names                          */
 for( i=0, strsize=0; i<tree[idx].nid; i++ )
  if( tree[idx].id[i][9] == (char) 0 )
   strsize += strlen(tree[idx].id[i]) + 1;
 sizetr += strsize * sizeof( char );
 l = (long) (treeptr - treestr);
 /* reallocate the buffer to hold new strings                                           */
 treestr = (char *) mem_realloc( (void *) treestr, sizetr );
 if( treestr == NULL )
  return IDS_ERR_NOEXEC;
 treeptr = treestr + l;
 tmpptr = treeptr;
 /* copy the strings to the buffer                                                                      */
 for( i=0; i<tree[idx].nid; i++ )
  if( tree[idx].id[i][9] == (char) 0 )
  {
   strcpy( treeptr, tree[idx].id[i] );
   treeptr += strlen( tree[idx].id[i] ) + 1;
  }
 /* reuse the space in tree.id[idx][0] to store indexes                         */
 for( i=0, ef=pm=0; i<tree[idx].nid; i++ )
  switch( (int) tree[idx].id[i][9] )
  {
   case 0: tree[idx].id[i][0] = (char) pm; pm++; break; /* constant  */
   case 1: tree[idx].id[i][0] = (char) ef; ef++;        /* substrate */
  }
 /* the first nsub are reserved for substrates even when not explicit*/
 for( i=0, ef=tree[idx].nsub; i<tree[idx].nid; i++ )
  if( (int) tree[idx].id[i][9] == 2 )                   /* product   */
  {
   tree[idx].id[i][0] = (char) ef; ef++;
  }
 /* nsub+npro are reserved for substs.& prods. even when not explicit*/
 for( i=0, ef=tree[idx].nsub+tree[idx].npro; i<tree[idx].nid; i++ )
  if( (int) tree[idx].id[i][9] == 3 )                   /* modifier  */
  {
   tree[idx].id[i][0] = (char) ef; ef++;
  }

 /* expand the memory block                                                                             */
 nrateq++;
 ktype = (struct kint *) mem_realloc( ktype, nrateq * sizeof( struct kint ) );
 if( ktype != NULL )
 {
  ktype[MAX_TYP+idx].nmodf    = (unsigned char) tree[idx].nmodf;
  ktype[MAX_TYP+idx].nconst   = (unsigned char) tree[idx].nconst;
  ktype[MAX_TYP+idx].nsub     = (unsigned char) tree[idx].nsub;
  ktype[MAX_TYP+idx].npro     = (unsigned char) tree[idx].npro;
  ktype[MAX_TYP+idx].descr    = tree[idx].descr;
  ktype[MAX_TYP+idx].constnam = tmpptr;
  return 0;
 }
 else
  return IDS_ERR_NOEXEC;
}

/*
  add a new function read from a .TOP file
  and renumber references to this type in the .TOP
*/

int addtr( int e )
{
 int i, j, k, nRc;

 j = nudf;
 k = kinetu[e];
 if( (nRc = new_tree( j )) != 0 ) return nRc;
 if( (nRc = new_rateq( j )) != 0 ) return nRc;
 for( i=e; i<nsteps; i++ )
  if( (kinetu[i]==k) && (kfl[i]) )
  {
   kinetu[i] = MAX_TYP+j;
   kfl[i] = 0;
  }
 return 0;
}


void FreeMem( void )
{
 mem_free( treestr );
 mem_free( tree );
 mem_free( lindex );
 mem_free( scindex );
 mem_free( scanpar );
 mem_free( sparam );
 mem_free( outtit );
 mem_free( rstr );
 mem_free( loop );
 mem_free( poutelem );
 mem_free( outpel );
 mem_free( ktype );
 mem_free( params[0] );
 mem_free( eff[0] );
}
