#include "copyleft.h"

/*
    GEPASI - a simulator of metabolic pathways and other dynamical systems
    Copyright (C) 1989, 1992, 1993  Pedro Mendes
*/

/*************************************/
/*                                   */
/*         GWSIM - Simulation        */
/*        MS-WINDOWS front end       */
/*                                   */
/*        Simulation files I/O       */
/*                                   */
/*          QuickC/WIN 1.0           */
/*                                   */
/*   (include here compilers that    */
/*   compiled GWSIM successfully)    */
/*                                   */
/*************************************/


#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>
#include <sys\types.h>
#include <sys\stat.h>
#include "commdlg.h"
#include "globals.h"
#include "iotop.h"
#include "iosim.h"
#include "gep2.h"
#include "simgvar.h"
#include "gaussw.h"
#include "strtbl.h"

#pragma alloc_text( CODE13, TopIn, SimToBuf, WriteSim, BufToSim, ReadSim, setdb )

/*
	set up several database variables
*/
int setdb( int swapnames )
{
 /* set up the strings describing each step							*/
 step_string( );

 /* reduce the stoicheiometry matrix								*/
 if( reduce( swapnames ) != 0 )
  return -1;

 /* if there are no steps with internal metabolites signal and stop	*/
 if( (!nsteps) || (!nmetab) )
  return IDS_ERR_BAD_EMPTY;
 /* setup the parameters structure									*/
 if( SetParams() == -1 )
  return IDS_ERR_NOEXEC;

 /* setup the outpel structure										*/
 if( SetOutpEl() == -1 )
  return IDS_ERR_NOEXEC;

 return 0;
}

/*
	read topology file with validity check
*/

int TopIn( LPSTR tname )
{
 unsigned char  nmod;
 BOOL			badtop;
 int			i, j, nRc;

 if( !(nRc = ReadTop( tname ) ) )
 {
  badtop = FALSE;
  if( (nsteps<=0) || (nsteps>MAX_STEP) ||
      (totmet<=0) || (totmet>MAX_MET)    )
   badtop = TRUE;
  /* check if topology is complete							*/
  for( i=0; i<nsteps; i++ )
  {
   if( kinetu[i]==NOT ) badtop = TRUE;
   if( ktype[kinetu[i]].nmodf > 0 )
   {
    /* count the number of assigned modfs for each reaction	*/
    for( j=0, nmod=0; j<totmet; j++ )
     if( (*loop)[i][j] != 0 ) nmod++;
    if( ktype[kinetu[i]].nmodf != nmod ) badtop = TRUE;
   }
  }

  /* if topology is incomplete, signal and return				*/
  if( badtop )
  {
   tname[0] = '\0';
   return IDS_ERR_BAD_TOP;
  }

  nRc = setdb( 1 ); /* 1 means swap metabolite and step names	*/

  /* exit and signal no errors								*/
  return nRc;
 }
 else return IDS_ERR_LOAD;
}

/*
	write simulation parameters to file
*/

void SimToBuf( LPSTR Buff, LPSTR FName )
{
 int i, j;
 char *p;
 char auxstr[260];

 /* write the kinetic constants, one line per step						*/
 for( i=0; i<nsteps; i++ )
 {
  for( j=0; j<(int)ktype[kinetu[i]].nconst; j++ )
  {
   lstrcat( Buff, (LPSTR) " " );
   gcvt( *(params[i]+j), 20, auxstr );
   lstrcat( Buff, (LPSTR) auxstr );
  }
  lstrcat( Buff, (LPSTR) "\n" );
 }

 /* write the concentrations, one per line								*/
 for( i=0; i<totmet; i++ )
 {
  gcvt( xu[i], 20, auxstr );
  lstrcat( Buff, (LPSTR) auxstr );
  lstrcat( Buff, (LPSTR) "\n" );
 }

 /* write the units in one  line										*/
 wsprintf( (LPSTR) auxstr, "%s %s\n", (LPSTR) options.concu, (LPSTR) options.timeu );
 lstrcat( Buff, (LPSTR) auxstr );
 /* write log, dyn and ss options in one line							*/
 sprintf( auxstr, "%d %d %lu %.20le %.20le %.20le %.20le %d %d %d\n",
          options.debug, options.dyn, options.pfo, options.endtime,
          options.reltol, options.abstol, options.hrcz, options.adams,
          options.bdf, options.ss);
 lstrcat( Buff, (LPSTR) auxstr );
 /* write txt options in one line										*/
 sprintf( auxstr, "%d %d %d %d %d %d %d\n", options.txt,
          options.structan, options.staban, options.stdela,
          options.nonela, options.stdcc, options.noncc );
 lstrcat( Buff, (LPSTR) auxstr );
 /* write the dat options in one line									*/
 sprintf( auxstr, "%d %d %d %d %d %d %d %d %d\n", options.dat, options.datsep,
          options.datwidth, options.dattit, options.datmca, options.datss,
          totsel, options.append, options.quotes );
 lstrcat( Buff, (LPSTR) auxstr );
 /* write the dat file title											*/
 sprintf( auxstr, "%s\n", DatName );
 lstrcat( Buff, (LPSTR) auxstr );
 /* write each output item in one line (index and title)                */
 for( i=0; i<totsel; i++ )
  for( j=0; j<noutpel; j++ )
   if( outpel[j].idx == (unsigned int) (i+1) )
   {
    wsprintf( (LPSTR) auxstr, "%d %s\n", j, outpel[j].title );
    lstrcat( Buff, (LPSTR) auxstr );
   }
 /* write the scan options in one line									*/
 sprintf( auxstr, "%d %lu %d %d %d\n", options.scan, options.scandens,
          options.scanlog, totscan, nlinks );
 lstrcat( Buff, (LPSTR) auxstr );
 /* write each scan item in one line  					                */
 for( i=0; i<totscan; i++ )
  for( j=0; j<nscanpar; j++ )
   if( spar[j].idx == (i+1) )
   {
    sprintf( auxstr, "%d %.20le %.20le %lu %d\n", j, spar[j].low,
             spar[j].high, spar[j].dens, spar[j].log );
    lstrcat( Buff, (LPSTR) auxstr );
   }
 /* write each linked item in one line									*/
 for( i=0; i<nlinks; i++ )
  for( j=0; j<nscanpar; j++ )
   if( spar[j].lidx == i )
   {
    sprintf( auxstr, "%d %d %d %20le\n", j, spar[j].linkedto,
             spar[j].operation, spar[j].factor );
    lstrcat( Buff, (LPSTR) auxstr );
   }
}


int WriteSim( LPSTR FName )
{
 HCURSOR hSaveCursor;
 GLOBALHANDLE hBuff;
 LPSTR Buff;
 int i, j, ch1;
 WORD bufsize;
 OFSTRUCT OfStruct;

 /* load the wait cursor												*/
 hSaveCursor = SetCursor(hHourGlass);

 /*bufsize = (WORD) (3*nsteps + (8*nsteps + 6 + NAME_L)*totmet);		*/
 bufsize = (WORD) 10240;
 hBuff = GlobalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT , bufsize );
 Buff = GlobalLock( hBuff );
 /* write the information on the buffer									*/
 TopToBuf( Buff );
 if( (ch1 = OpenFile( FName, &OfStruct, OF_CREATE | OF_WRITE )) != -1 )
 {
  /* write the buffer to the file and close it							*/
  _lwrite( ch1, (LPSTR) Buff, (WORD) lstrlen( Buff ) );
  ch1 = _lclose( ch1 );
 }
 if( ch1 == -1 )
 {
  GlobalUnlock( hBuff );
  GlobalFree( hBuff );
  SetCursor(hSaveCursor);
  return IDS_ERR_SAVE;
 }

 /* set the flags, meaning 'kinetic type not saved' */
 for( i=0; i<nsteps; i++ ) kfl[i] = 1;
 /* check if model contains user-defined kinetic types					*/
 if( nudf>0 )
 {
  for( i=0; i<nsteps; i++ )
   if( (kinetu[i] > MAX_TYP-1) && (kfl[i]) )
   {
    /* write the tree function to the buffer								*/
    TreeToBuf( kinetu[i]-MAX_TYP, (LPSTR) Buff );
    /* reopen the file and write this tree to it							*/
    if( (ch1 = OpenFile( (LPSTR) NULL, &OfStruct, OF_REOPEN | OF_WRITE )) != -1 )
    {
     /* write the buffer to the file and close it							*/
     _lseek( ch1, (LONG) 0, 2 );
     _lwrite( ch1, (LPSTR) Buff, (WORD) lstrlen( Buff ) );
     ch1 = _lclose( ch1 );
    }
    if( ch1 == -1 )
    {
     GlobalUnlock( hBuff );
     GlobalFree( hBuff );
     SetCursor(hSaveCursor);
     return IDS_ERR_SAVE;
    }
    /* mark all other reactions with this type */
    for( j=i; j<nsteps; j++ ) if(kinetu[j]==kinetu[i]) kfl[j] = 0;
   }
 }


 Buff[0] = '\0';
 SimToBuf( Buff, FName );
 /* append buffer to  the file											*/
 if( (ch1 = OpenFile( (LPSTR) NULL, &OfStruct, OF_REOPEN | OF_WRITE )) != -1 )
 {
  /* write the buffer to the file and close it							*/
  _lseek( ch1, (LONG) 0, 2 );
  _lwrite( ch1, (LPSTR) Buff, (WORD) lstrlen( Buff ) );
   ch1 = _lclose( ch1 );
 }
 if( ch1 == -1 )
 {
  GlobalUnlock( hBuff );
  GlobalFree( hBuff );
  SetCursor(hSaveCursor);
  return IDS_ERR_SAVE;
 }

 /* release the memory block											*/
 GlobalUnlock( hBuff );
 GlobalFree( hBuff );
 SetCursor(hSaveCursor);
 return 0;
}


/*
	read buffer with simulation parameters and store them
*/

int BufToSim( char *Buff )
{
 int i, j, log, lkt, op;
 unsigned long dens;
 unsigned u_dummy;
 double faux, low, high, fac;
 char structfile[128], auxstr[128];
 int nRc;
 struct stat fst;

  /* read the kinetic constants from the buffer, one line per step	*/
  for(i=0;i<nsteps;i++)
  {
   Buff = strchr( Buff, '\n' );
   if( Buff != NULL ) Buff++;
   else return IDS_ERR_BAD_KPARAM;
   for( j=0; j<(int)ktype[kinetu[i]].nconst; j++ )
   {
    Buff = strchr( Buff, ' ' );
    for( ;*Buff == ' '; Buff++);
    if ( (Buff == NULL) || (sscanf( Buff,"%le", &faux ) < 1) )
     return IDS_ERR_BAD_KPARAM;
    *(params[i]+j) = faux;
   }
  }

  /* read the concentrations, one per line							*/
  for(i=0;i<totmet;i++)
  {
   Buff = strchr( Buff, '\n' );
   if( Buff != NULL ) Buff++;
   else return IDS_ERR_BAD_CONCENT;
   if ( (Buff == NULL) || (sscanf( Buff,"%le", &faux ) < 1) )
    return IDS_ERR_BAD_CONCENT;
   xu[i] = faux;
  }
  /* read the units in one line											*/
  Buff = strchr( Buff, '\n' );
  if( Buff != NULL )
   Buff++;
  if ( (Buff == NULL) || (sscanf( Buff,"%s %s", &options.concu, &options.timeu ) < 2) )
   return IDS_ERR_BAD_UNITS;
  /* read log, dyn and ss options in one line							*/
  Buff = strchr( Buff, '\n' );
  if( Buff != NULL )
   Buff++;
  if ( (Buff == NULL) || (sscanf( Buff,"%d %d %ld %le %le %le %le %d %d %d",
          &options.debug, &options.dyn, &options.pfo, &options.endtime,
          &options.reltol, &options.abstol, &options.hrcz, &options.adams,
          &options.bdf, &options.ss ) < 10 ) )
   return IDS_ERR_BAD_OPTIONS;
  /* read txt options in one line										*/
  Buff = strchr( Buff, '\n' );
  if( Buff != NULL )
   Buff++;
  if ( (Buff == NULL) || (sscanf( Buff,"%d %d %d %d %d %d %d",
          &options.txt, &options.structan, &options.staban,
          &options.stdela, &options.nonela, &options.stdcc,
          &options.noncc ) < 7 ) )
   return IDS_ERR_BAD_OPTIONS;
  /* read the dat options in one line									*/
  Buff = strchr( Buff, '\n' );
  if( Buff != NULL )
   Buff++;
  if ( (Buff == NULL) || (sscanf( Buff,"%d %d %d %d %d %d %d %d %d",
        &options.dat, &options.datsep, &options.datwidth,
        &options.dattit, &options.datmca, &options.datss,
        &totsel, &options.append, &options.quotes ) < 9 ) )
   return IDS_ERR_BAD_OPTIONS;
  /* read the dat filename in one line									*/
  Buff = strchr( Buff, '\n' );
  if( Buff != NULL )
   Buff++;
  if ( (Buff == NULL) || (sscanf( Buff,"%s", DatName ) != 1 ) )
   return IDS_ERR_BAD_OPTIONS;

  /* read each output item in one line (index only)						*/
  for( i=0; i<totsel; i++ )
  {
   Buff = strchr( Buff, '\n' );
   if( Buff != NULL )
    Buff++;
   if ( (Buff == NULL) || (sscanf( Buff,"%d", &j ) < 1 ) )
    return IDS_ERR_BAD_OPTIONS;
   if( ver_no < (float) 2.05 ) outpel[j+totmet].idx = i+1;
   else outpel[j].idx = i+1;
  }

 /* read the scan options in one line									*/
  Buff = strchr( Buff, '\n' );
  if( Buff != NULL )
   Buff++;
  if ( (Buff == NULL) || (sscanf( Buff, "%d %lu %d %d %d", &options.scan,
                          &options.scandens, &options.scanlog,
                          &totscan, &nlinks ) < 5 ) )
   return IDS_ERR_BAD_OPTIONS;


 /* read each scan item in one line                 */
  for( i=0; i<totscan; i++ )
  {
   Buff = strchr( Buff, '\n' );
   if( Buff != NULL )
    Buff++;
   if ( (Buff == NULL) || (sscanf( Buff,"%d %le %le %lu %d", &j,
                                   &low, &high, &dens, &log ) < 5 ) )
    return IDS_ERR_BAD_OPTIONS;
   spar[j].idx  = i+1;
   spar[j].low  = low;
   spar[j].high = high;
   spar[j].dens = dens;
   spar[j].log  = log;
  }

 /* read each linked item in one line                 */
  for( i=0; i<nlinks; i++ )
  {
   Buff = strchr( Buff, '\n' );
   if( Buff != NULL )
    Buff++;
   if ( (Buff == NULL) || (sscanf( Buff,"%d %d %d %le", &j, &lkt, &op,
                                   &fac ) < 4 ) )
    return IDS_ERR_BAD_OPTIONS;
   spar[j].lidx  = i;
   spar[j].linkedto  = lkt;
   spar[j].operation = op;
   spar[j].factor = fac;
  }
 return 0;
}


/*
	read simulation parameters from file
*/

int ReadSim( LPSTR FName )
{
 HCURSOR hSaveCursor;
 HANDLE hBuff;
 unsigned int bufsize;
 char *Buff;
 int i, j, k, fl, ty, ch1, nRc;
 OFSTRUCT OfStruct;
 struct stat fst;
 WORD sizetr;

 /* display the wait cursor											*/
 hSaveCursor = SetCursor(hHourGlass);

 /* open the file													*/
 if( (ch1 = OpenFile( FName, &OfStruct, OF_READ )) != -1 )
 {
  /* get the file statistics										*/
  fstat( ch1, &fst );

  /* set the buffer to the size of the file plus one				*/
  bufsize = (unsigned int) fst.st_size;

  /* allocate space for the read buffer and lock it					*/
  hBuff = LocalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT, (WORD) (bufsize+1) );
  if( hBuff == NULL )
  {
   SetCursor(hSaveCursor);
   return IDS_ERR_LOAD;
  }
  Buff = (char *) LocalLock( hBuff );

  /*read the file and close it										*/
  if( read( ch1, Buff, bufsize ) == -1 )
  {
   LocalUnlock( hBuff );
   LocalFree( hBuff );
   SetCursor(hSaveCursor);
   return IDS_ERR_LOAD;
  }
  if( close( ch1 ) == -1 )
  {
   LocalUnlock( hBuff );
   LocalFree( hBuff );
   SetCursor(hSaveCursor);
   return IDS_ERR_LOAD;
  }

  /* set the number of user-def functions to zero					*/
  nudf = 0;
  nrateq = MAX_TYP;
  sizetr = 4;
  GlobalUnlock( hTreeStr );
  GlobalFree( hTreeStr );
  hTreeStr = GlobalAlloc( GMEM_ZEROINIT, sizetr );
  if( hTreeStr == NULL )
  {
   LocalUnlock( hBuff );
   LocalFree( hBuff );
   SetCursor(hSaveCursor);
   return IDS_ERR_NOEXEC;
  }
  treestr = GlobalLock( hTreeStr );
  treeptr = treestr;

  /* read the topology section 										*/
  if( nRc = BufToTop( Buff, &Buff ) )
  {
   LocalUnlock( hBuff );
   LocalFree( hBuff );
   SetCursor(hSaveCursor);
   return nRc;
  }

  for( i=0; i<nsteps; i++ ) kfl[i] = 1;

  /* check if model contains user-defined kinetic types				*/
  /* if there are any, check if they have to be read				*/
  for( i=0; i<nsteps; i++ )
   if( (kinetu[i] >= MAX_TYP) && (kfl[i]) )
   {
    /* look for a form-feed											*/
    Buff = strchr( Buff, '\f' );
    Buff = BufToTree( ++Buff );
    for( j=0, ty=-1; j<nudf; j++ )
     if( lstrcmp( (LPSTR) tree[j].descr, (LPSTR) tr.descr ) == 0 )
     {
      ty = j;
      break;
     }
    if( ty == -1 )
    {
     if( (nRc=addtr( i )) !=0 )
     {
      LocalUnlock( hBuff );
      LocalFree( hBuff );
      SetCursor(hSaveCursor);
      return nRc;
     }
     fl = 1;
    }
    else
     for( j=i,k=kinetu[i]; j<nsteps; j++ )
      if( (kinetu[j]==k) && (kfl[j]) )
      {
       kinetu[j] = MAX_TYP+ty;
       kfl[j] = 0;
      }
   }

  /* setup the relevant databases									*/
  if( nRc = setdb( 0 ) )
  {
   LocalUnlock( hBuff );
   LocalFree( hBuff );
   SetCursor(hSaveCursor);
   return nRc;
  }


  /* read the simulation section 										*/
  nRc = BufToSim( Buff );

  LocalUnlock( hBuff );
  LocalFree( hBuff );
  SetCursor(hSaveCursor);
  return nRc;
 }
 else
 {
  SetCursor(hSaveCursor);
  return IDS_ERR_LOAD;
 }
}

