/* Copyright 1995-96 Jon Griffiths.  See the file "jlib.doc" for details. */
#include <jlib.h>

#ifndef MIN
#define MIN(a,b)   (((a) < (b)) ? (a) : (b))
#endif
#ifndef MAX
#define MAX(a,b)   (((a) < (b)) ? (b) : (a))
#endif

/* vertically clip a side of a triangle */
#define V_CLIP(cx1,cy1,cx2,cy2)                      \
    if(cy1 < cy2){                                   \
      if(cy1 < 0){                                   \
         cx1 += ((cx2-cx1)*(-cy1)/(cy2-cy1));        \
	 cy1 =  0;                                   \
       }                                             \
       if(cy2 > height){                             \
          cx2 -= ((cx2-cx1)*(cy2-height)/(cy2-cy1)); \
          cy2 =  height;                             \
       }                                             \
   }                                                 \
   else{                                             \
       if(cy1 > height){                             \
	  cx1 += ((cx2-cx1)*(cy1-height)/(cy1-cy2)); \
	  cy1 =  height;                             \
       }                                             \
       if(cy2 < 0){                                  \
	  cx2 -= ((cx2-cx1)*(-cy2)/(cy1-cy2));       \
	  cy2 =  0;                                  \
       }                                             \
   }

/* Use Bresenham's line algorithm to trace the first edge of a triangle. */
JINLINE void first_edge( int x1, int y1, int x2, int y2, int max_x )
{
   int dx, dy, xf, yf, a, b, t, i;

   if((x1==x2) && (y1==y2)){
      if((x1<0)&&(x1<max_x)){
         left[y1]=right[y1] = x1;
      }
      return;
   }

   if (x2>x1) {
      dx = x2-x1;
      xf = 1;
   }
   else {
      dx = x1-x2;
      xf = -1;
   }

   if (y2>y1) {
      dy = y2-y1;
      yf = 1;
   }
   else {
      dy = y1-y2;
      yf = -1;
   }

   if (dx>dy) {
      a = dy+dy;
      t = a-dx;
      b = t-dx;
      for(i=0;i<=dx;i++) {
            if(x1<=0){
               left[y1] = 0;
            }
            else{
               if(x1>=max_x){
                  right[y1] = max_x;
               }
               else{
                  left[y1]=right[y1] = x1;
               }
            }
            x1 += xf;
         if(t<0){
            t+=a;
         }
         else{
            t+=b;
            y1+=yf;
         }
      }
   }
   else {
      a = dx+dx;
      t = a-dy;
      b = t-dy;
      for(i=0;i<=dy;i++) {
         if(x1<=0){
            left[y1]= 0;
         }
         else{
            if(x1>=max_x){
               right[y1] = max_x;
            }
            else{
               left[y1]=right[y1] = x1;
            }
         }
         y1 += yf;
         if (t<0) {
            t += a;
         }
         else {
            t += b;
            x1 += xf;
         }
      }
   }
}

/* Use Bresenham's line algorithm to trace an edge of the triangle. */
JINLINE void next_edge( int x1, int y1, int x2, int y2, int max_x )
{
   int dx, dy, xf, yf, a, b, t, i;

   if(x1==x2 && y1==y2){
     if(x1<left[y1]){
        if(x1<=0){
          left[y1] = 0;
        }
        else{
          left[y1] = x1;
        }
     }
     if(x1>right[y1]) {
        if(x1>=max_x){
           right[y1] = max_x;
        }
        else{
           right[y1] = x1;
        }
     }
     return;
   }

   if (x2>x1) {
      dx = x2-x1;
      xf = 1;
   }
   else {
      dx = x1-x2;
      xf = -1;
   }

   if (y2>y1) {
      dy = y2-y1;
      yf = 1;
   }
   else {
      dy = y1-y2;
      yf = -1;
   }

   if (dx>dy) {
      a = dy+dy;
      t = a-dx;
      b = t-dx;
      for (i=0;i<=dx;i++) {
            if(x1<left[y1]){
              if(x1<=0){
                left[y1] = 0;
              }
              else{
                left[y1] = x1;
              }
            }
            if(x1>right[y1]) {
              if(x1>=max_x){
                 right[y1] = max_x;
              }
              else{
                 right[y1] = x1;
              }
            }
            x1 += xf;
         if (t<0) {
            t += a;
         }
         else {
            t += b;
            y1 += yf;
         }
      }

   }
   else {
      a = dx+dx;
      t = a-dy;
      b = t-dy;
      for (i=0;i<=dy;i++) {
         if (x1<left[y1]) {
              if(x1<=0){
                left[y1] = 0;
              }
              else{
                left[y1] = x1;
              }
         }
         if (x1>right[y1]) {
              if(x1>=max_x){
                 right[y1] = max_x;
              }
              else{
                 right[y1] = x1;
              }
         }
         y1 += yf;
         if (t<0) {
            t += a;
         }
         else {
            t += b;
            x1 += xf;
         }
      }
   }
}


/*+------------------------------------------------------------------------+*/
/*| Draw a non-clipping filled triangle.                                   |*/
/*+------------------------------------------------------------------------+*/
void buff_filled_triangleNC(buffer_rec *buff,int x1,int y1,int x2,int y2,
                           int x3,int y3,UBYTE col )
{
   int iy,lowest_bound,highest_bound,max_x=B_MAX_X(buff);
   int *lp,*rp;

   JLIB_ENTER("buff_filled_triangleNC");

   highest_bound=MAX(y1,MAX(y2,y3));
   lowest_bound=MIN(y1,MIN(y2,y3));

   /* reset the minumum amount of the edge tables */
   for (iy=lowest_bound;iy<=highest_bound;iy++) {
      left[iy] = MAX_XRES;
      right[iy] = -1;
   }
      
   /* define edges */
   first_edge(x1,y1,x2,y2,max_x);
   next_edge(x2,y2,x3,y3,max_x);
   next_edge(x3,y3,x1,y1,max_x);

   lp=left+lowest_bound;
   rp=right+lowest_bound;

   /* fill horizontal spans of pixels from left[] to right[] */
   for(iy=lowest_bound;iy<=highest_bound;iy++) {
       buff_draw_h_lineNC(buff,*lp,iy,*rp,col);
       lp++;rp++;
   }

   JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+*/
/*| Draw a clipping filled triangle.                                       |*/
/*+------------------------------------------------------------------------+*/
void buff_filled_triangle(buffer_rec *buff,int x1,int y1,int x2,int y2,
                          int x3,int y3,UBYTE col )
{
   int iy,height,max_x,nx1,ny1,nx2,ny2,nx3,ny3;
   int clip=0,lowest_bound,highest_bound,*lp,*rp;

   JLIB_ENTER("buff_filled_triangle");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(buff);
#endif

   height=B_MAX_Y(buff);
   max_x=B_MAX_X(buff);

   /* clip x cases */
   highest_bound=MAX(x1,MAX(x2,x3));
   lowest_bound =MIN(x1,MIN(x2,x3));

   if((highest_bound<0)||(lowest_bound>max_x)){
      JLIB_LEAVE;
      return;
   }

   if((highest_bound>max_x)||(lowest_bound<0)){
      clip=1;
   }

   /* clip y cases and set lowest bounds */
   highest_bound=MAX(y1,MAX(y2,y3));
   lowest_bound =MIN(y1,MIN(y2,y3));

   if((highest_bound<0)||(lowest_bound>height)){
      JLIB_LEAVE;
      return;
   }

   if(highest_bound>height){
      highest_bound=height;
   }
   if(lowest_bound<0){
      lowest_bound=0;
   }

   /* reset the minumum amount of the edge tables */
   for (iy=lowest_bound;iy<=highest_bound;iy++) {
      left[iy] = MAX_XRES;
      right[iy] = -1;
   }

   nx1=x2;ny1=y2;    /* create line pairs for possible clipped sides */
   nx2=x3;ny2=y3;
   nx3=x1;ny3=y1;

   if(y1>ny1){
      JLIB_SWAP(y1,ny1);
      JLIB_SWAP(x1,nx1);
   }
   if((ny1>=0)&&(y1<=height)){
      if((y1<=0)||(ny1>=height)){
         if(x1>nx1){
            JLIB_SWAP(x1,nx1);
            JLIB_SWAP(y1,ny1);
         }
         V_CLIP(x1,y1,nx1,ny1);
      }
      first_edge(x1,y1,nx1,ny1,max_x);
   }

   if(y2>ny2){
      JLIB_SWAP(y2,ny2);
      JLIB_SWAP(x2,nx2);
   }
   if((ny2>=0)&&(y2<=height)){
      if((y2<=0)||(ny2>=height)){
         if(x2>nx2){
            JLIB_SWAP(x2,nx2);
            JLIB_SWAP(y2,ny2);
         }
         V_CLIP(x2,y2,nx2,ny2);
      }
      next_edge(x2,y2,nx2,ny2,max_x);
   }

   if(y3>ny3){
      JLIB_SWAP(y3,ny3);
      JLIB_SWAP(x3,nx3);
   }
   if((ny3>=0)&&(y3<=height)){
      if((y3<=0)||(ny3>=height)){
         if(x3>nx3){
            JLIB_SWAP(x3,nx3);
            JLIB_SWAP(y3,ny3);
         }
         V_CLIP(x3,y3,nx3,ny3);
      }
      next_edge(x3,y3,nx3,ny3,max_x);
   }

   if(clip){
      /* check effect of horizontal clipping on vertical bounds */
      while((left[lowest_bound]>max_x)||(right[lowest_bound]<0)){
            lowest_bound++;
      }
      while((left[highest_bound]>max_x)||(right[highest_bound]<0)){
            highest_bound--;
      }
   }

   lp=left+lowest_bound;
   rp=right+lowest_bound;

   /* fill horizontal spans of pixels from left[] to right[] */
   for(iy=lowest_bound;iy<=highest_bound;iy++) {
       buff_draw_h_lineNC(buff,*lp,iy,*rp,col);
       lp++;rp++;
   }

   JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+*/
/*| Draw a non-clipping triangle outline.                                  |*/
/*+------------------------------------------------------------------------+*/
void buff_draw_triangleNC(buffer_rec *buff,int x1,int y1,int x2,int y2,
                                int x3,int y3, UBYTE col)
{
 JLIB_ENTER("buff_draw_triangleNC");

 buff_draw_lineNC(buff,x1,y1,x2,y2,col);
 buff_draw_lineNC(buff,x2,y2,x3,y3,col);
 buff_draw_lineNC(buff,x3,y3,x1,y1,col);

 JLIB_LEAVE
}


/*+------------------------------------------------------------------------+*/
/*| Draw a clipping triangle outline.                                      |*/
/*+------------------------------------------------------------------------+*/
void buff_draw_triangle(buffer_rec *buff,int x1,int y1,int x2,int y2,
                                int x3,int y3, UBYTE col)
{

 JLIB_ENTER("buff_draw_triangle");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(buff);
#endif

 buff_draw_line(buff,x1,y1,x2,y2,col);
 buff_draw_line(buff,x2,y2,x3,y3,col);
 buff_draw_line(buff,x3,y3,x1,y1,col);

 JLIB_LEAVE
}
