/* 
 * pal.c - A simple palette editor
 * 
 * Copyright (c) 1996 by Lennart Steinke
 * 
 * Note:    The palette is saved with 6 (0...63) bit for each component,
 * so that the standard PC paint programs can load it. You need to shift
 * every byte 2 bits to the left to get a palette, which can be used with
 * Jlib.
 * 
 * This is "mailware": if you're using this stuff, send me a postcard, or
 * email me.
 * 
 * Lennart Steinke Hofener Str. 16 74366 Kirchheim/N. GERMANY
 */

#include <stdlib.h>
#include <string.h>
#include <jlib.h>



/* defines */

#define TRUE	1
#define FALSE	0

#define QUIT	0
#define PLAY	1

#define CANCEL	27
#define SELECT	13

#define PAL_X	10
#define PAL_Y	10

#define BUTTON_WIDTH	50
#define BUTTON_HEIGHT	20

#define OK_BUTTON_X		20
#define OK_BUTTON_Y		175
#define CANCEL_BUTTON_X	100
#define CANCEL_BUTTON_Y 175
#define FADE_BUTTON_X	180
#define FADE_BUTTON_Y   140
#define SET_BUTTON_X	240
#define SET_BUTTON_Y	140

#define LEVER_X	180

#define NUM_BUTTONS 4
#define ID_DONE		0
#define ID_OK		1
#define ID_CANCEL	2
#define ID_FADE		3
#define ID_SET		4

/* types */

typedef struct {
	USHORT          x, y, width, height;
	USHORT          min_value, max_value, act_value;
} lever_rec;

typedef struct {
	USHORT          x, y, width, height;
	USHORT          id;
	char            text[20];
} button_rec;

/* makros */

#define SET_LEVER_VAL(lever, value)  (lever)-> act_value=(value)
#define GET_LEVER_VAL(lever) (lever)->act_value

#define INDEX_2_X(i) (PAL_X+ (i) /16 *10)
#define INDEX_2_Y(i) (PAL_Y+ (i) %16 *10)

#define SET_BUTTON(b,x_,y_,w,h,i,c) (b)->x=(x_);		\
									(b)->y=(y_);		\
									(b)->width=(w);		\
                                    (b)->height=(h);	\
                                    (b)->id=(i);		\
                        			strncpy((b)->text,(c),20)


/* globals */

buffer_rec     *offscreen;

UBYTE           graph_enabled = FALSE;
UBYTE           kb_enabled = FALSE;
lever_rec       c1[3], c2[3];
UBYTE           col[2];
button_rec      b[NUM_BUTTONS];
UBYTE           pal[768];
char           *input = NULL, *output = NULL;



/* prototypes */

void            init(void);
void            done(void);
void            fatal_error(char *s);
void            draw(void);
void            re_draw(void);
void            draw_lever(lever_rec * lever);
void            draw_colors(void);
void            draw_button(button_rec * button);
void            screen_set_mixed_pal(void);
USHORT          check_click(USHORT mx, USHORT my, USHORT button);
void            read_pal(void);
void            write_pal(void);
void            buff_draw_outline_char(buffer_rec * buf, char ch, USHORT x, USHORT y, UBYTE col);

int
main(int argc, char **argv)
{

        int             status = 1, a, x, y, buttons, id=ID_OK;
	USHORT          s, e, h, d, i;
	float           fr, fg, fb;

	col[0] = 31;
	col[1] = 63;

	init();

	if (argc == 2) {
		input = argv[1];
		output = argv[1];
		read_pal();
	}
	if (argc == 3) {
		input = argv[1];
		output = argv[2];
		read_pal();
	}

	for (a = 0; a < 3; a++) {

		c1[a].x = LEVER_X;
		c1[a].y = 20 + a * 12;
		c1[a].width = 65;
		c1[a].height = 10;
		c1[a].min_value = 0;
		c1[a].max_value = 63;
		c1[a].act_value = (pal[col[0] * 3 + a]) >> 2;

		c2[a].x = LEVER_X;
		c2[a].y = 85 + a * 12;
		c2[a].width = 65;
		c2[a].height = 10;
		c2[a].min_value = 0;
		c2[a].max_value = 63;
		c2[a].act_value = (pal[col[1] * 3 + a]) >> 2;
	}

	SET_BUTTON(&b[0], OK_BUTTON_X, OK_BUTTON_Y, BUTTON_WIDTH, BUTTON_HEIGHT, ID_OK, "OK");
	SET_BUTTON(&b[1], CANCEL_BUTTON_X, CANCEL_BUTTON_Y, BUTTON_WIDTH, BUTTON_HEIGHT, ID_CANCEL, "CANCEL");
	SET_BUTTON(&b[2], FADE_BUTTON_X, FADE_BUTTON_Y, BUTTON_WIDTH, BUTTON_HEIGHT, ID_FADE, "FADE");
	SET_BUTTON(&b[3], SET_BUTTON_X, SET_BUTTON_Y, BUTTON_WIDTH, BUTTON_HEIGHT, ID_SET, "SET E<-S");


	draw();

	while (status != QUIT) {


		mouse_get_status(&x, &y, &buttons);

		switch ((id = check_click(x, y, buttons))) {
			case ID_OK:
				status = QUIT;
				break;

			case ID_CANCEL:
				status = QUIT;
				break;

			case ID_SET:
				pal[col[1] * 3 + 0] = pal[col[0] * 3 + 0];
				pal[col[1] * 3 + 1] = pal[col[0] * 3 + 1];
				pal[col[1] * 3 + 2] = pal[col[0] * 3 + 2];

				for (a = 0; a < 3; a++)
					SET_LEVER_VAL(&c2[a], (pal[col[1] * 3 + a] >> 2));

				screen_put_pal(col[1], pal[col[1] * 3 + 0], pal[col[1] * 3 + 1], pal[col[1] * 3 + 2]);
				re_draw();
				break;

			case ID_FADE:
				s = col[0];
				e = col[1];

				if (s > e) {
					h = s;
					s = e;
					e = h;
				}

				d = e - s;

				if (!d)
					break;

				s *= 3;
				e *= 3;


				fr = (pal[e + 0] - pal[s + 0]) / (float) d;
				fg = (pal[e + 1] - pal[s + 1]) / (float) d;
				fb = (pal[e + 2] - pal[s + 2]) / (float) d;

				for (a = 0, i = s; a < d; a++, i += 3) {
					pal[i + 0] = pal[s + 0] + a * fr;
					pal[i + 1] = pal[s + 1] + a * fg;
					pal[i + 2] = pal[s + 2] + a * fb;
				}

				screen_block_set_pal(pal);
				break;

		}
	}


	done();
	if (id == ID_OK) {
		printf("\nsaving pal.\n");
		write_pal();
	}

 return 0;
}


void
init(void)
{



	offscreen = buff_init(SCREEN_WIDTH, SCREEN_HEIGHT);

	if (offscreen == NULL)
		fatal_error("init(): Not enough memory to allocate double buffer");

	screen_set_video_mode();
	graph_enabled = TRUE;
	screen_set_mixed_pal();

	if (mouse_present() != MOUSE_PRESENT) {
		fatal_error("init(): Unable to init mouse device.");
	}

	mouse_show_pointer();
}


void done(void){
	mouse_closedown();
	screen_restore_video_mode();
	graph_enabled = FALSE;
}

void
draw(void)
{

	int             a, c, d;
	char           *rgb[3] =
	{"red", "green", "blue"};

	c = 0;
	for (a = 0; a < 16; a++)
		for (d = 0; d < 16; d++) {
			buff_draw_rect(offscreen,
						   PAL_X + a * 10, PAL_Y + d * 10,
						   PAL_X + a * 10 + 8, PAL_Y + d * 10 + 9, c);

			if (c == col[0])
				buff_draw_outline_char(offscreen, 'S', PAL_X + a * 10 + 2, PAL_Y + d * 10, 15);
			if (c == col[1])
				buff_draw_outline_char(offscreen, 'E', PAL_X + a * 10 + 2, PAL_Y + d * 10, 15);

			c++;
		}

	for (a = 0; a < NUM_BUTTONS; a++)
		draw_button(&b[a]);

	draw_colors();

	for (a = 0; a < 3; a++) {
		draw_lever(&c1[a]);
		buff_draw_string(offscreen, rgb[a], c1[a].x + c1[a].width + 20, c1[a].y, 15);
		draw_lever(&c2[a]);
		buff_draw_string(offscreen, rgb[a], c2[a].x + c2[a].width + 20, c2[a].y, 15);
	}

	mouse_hide_pointer();
	screen_blit_fs_buffer(offscreen);
	mouse_show_pointer();

}

void
re_draw(void)
{

	UBYTE           a;

	draw_colors();

	for (a = 0; a < 3; a++) {
		draw_lever(&c1[a]);
		draw_lever(&c2[a]);
	}

	mouse_hide_pointer();
	screen_blit_fs_buffer(offscreen);
	mouse_show_pointer();

}

void
draw_lever(lever_rec * lever)
{

	USHORT          x;

	buff_draw_rect(offscreen, lever->x, lever->y, lever->x + lever->width, lever->y + lever->height, 0);
	buff_draw_box(offscreen, lever->x, lever->y, lever->x + lever->width, lever->y + lever->height, 15);
	x = lever->x + lever->act_value + 1;
	buff_draw_box(offscreen, x - 1, lever->y, x + 1, lever->y + lever->height, 15);
}

void
draw_button(button_rec * button)
{

	buff_draw_box(offscreen, button->x, button->y,
			  button->x + button->width, button->y + button->height, 15);

	buff_draw_string(offscreen, button->text, button->x + 5, button->y + 5, 15);
}

void
screen_set_mixed_pal(void)
{
	int             a, index;

	/* create the palette and set it */
	for (a = 0, index = 0; a < 256; a += 8, index += 3) {
		/* 32 shades of gray */
		pal[index] = a;
		pal[index + 1] = a;
		pal[index + 2] = a;
	}

	for (a = 0; a < 256; a += 8, index += 3) {
		/* 32 shades of red */
		pal[index] = a;
		pal[index + 1] = 0;
		pal[index + 2] = 0;
	}

	for (a = 0; a < 256; a += 8, index += 3) {
		/* 32 shades of green */
		pal[index] = 0;
		pal[index + 1] = a;
		pal[index + 2] = 0;
	}

	for (a = 0; a < 256; a += 8, index += 3) {
		/* 32 shades of blue */
		pal[index] = 0;
		pal[index + 1] = 0;
		pal[index + 2] = a;
	}

	for (a = 0; a < 256; a += 8, index += 3) {
		/* 32 shades of yellow */
		pal[index - 3] = a;
		pal[index - 2] = a;
		pal[index - 1] = 0;
	}
	pal[index - 3] = 255;
	pal[index - 2] = 255;
	pal[index - 1] = 0;


	for (a = 0; a < 256; a += 8, index += 3) {
		/* 32 shades of cyan */
		pal[index] = a / 2;
		pal[index + 1] = a;
		pal[index + 2] = a;
	}

	for (a = 0; a < 256; a += 8, index += 3) {
		/* 32 shades of pink */
		pal[index] = a;
		pal[index + 1] = 0;
		pal[index + 2] = a;
	}

	for (a = 0; a < 256; a += 8, index += 3) {
		/* 32 shades yellow->red */
		pal[index] = 255;
		pal[index + 1] = 255 - a;
		pal[index + 2] = 0;
	}

	screen_block_set_pal(pal);
}

void
draw_colors(void)
{

	char            s[4];

	buff_draw_box(offscreen, LEVER_X, 10, LEVER_X + 66, 18, 15);
	buff_draw_rect(offscreen, LEVER_X + 1, 11, LEVER_X + 65, 18, col[0]);

	sprintf(s, "%3i", col[0]);
	buff_draw_rect(offscreen, LEVER_X + 86, 10, LEVER_X + 120, 20, 0);
	buff_draw_string(offscreen, s, LEVER_X + 86, 10, 15);

	buff_draw_box(offscreen, LEVER_X, 75, LEVER_X + 66, 75 + 8, 15);
	buff_draw_rect(offscreen, LEVER_X + 1, 76, LEVER_X + 65, 75 + 8, col[1]);

	sprintf(s, "%3i", col[1]);
	buff_draw_rect(offscreen, LEVER_X + 86, 75, LEVER_X + 120, 85, 0);
	buff_draw_string(offscreen, s, LEVER_X + 86, 75, 15);
}

USHORT
check_click(USHORT mx, USHORT my, USHORT button)
{

	int             a, index;
	USHORT          x, y;

	/* Was there a click on the palette ? */
	if (mx >= PAL_X && mx < PAL_X + 160 && my >= PAL_Y && my < PAL_Y + 160) {

		index = ((mx - PAL_X) / 10) * 16 + ((my - PAL_Y) / 10);

                if (BUTTON_CHECK(button, MOUSE_B_LEFT)) {

			x = INDEX_2_X(col[0]);
			y = INDEX_2_Y(col[0]);
			buff_draw_rect(offscreen, x, y, x + 8, y + 9, col[0]);

			col[0] = index;
			x = INDEX_2_X(col[0]);
			y = INDEX_2_Y(col[0]);

			buff_draw_outline_char(offscreen, 'S', x + 2, y, 15);

			x = INDEX_2_X(col[1]);
			y = INDEX_2_Y(col[1]);

			buff_draw_outline_char(offscreen, 'E', x + 2, y, 15);

			for (a = 0; a < 3; a++)
				c1[a].act_value = (pal[col[0] * 3 + a]) >> 2;

			re_draw();

			return ID_DONE;
		}
		else {
                        if (BUTTON_CHECK(button, (MOUSE_B_MIDDLE | MOUSE_B_RIGHT))) {

				x = INDEX_2_X(col[1]);
				y = INDEX_2_Y(col[1]);
				buff_draw_rect(offscreen, x, y, x + 8, y + 9, col[1]);

				col[1] = index;
				x = INDEX_2_X(col[1]);
				y = INDEX_2_Y(col[1]);
				buff_draw_outline_char(offscreen, 'E', x + 2, y, 15);

				x = INDEX_2_X(col[0]);
				y = INDEX_2_Y(col[0]);
				buff_draw_outline_char(offscreen, 'S', x + 2, y, 15);

				for (a = 0; a < 3; a++)
					c2[a].act_value = (pal[col[1] * 3 + a]) >> 2;

				re_draw();

				return ID_DONE;
			}
		}
	}

	/* From this point forward only left button clicks do something */
        if (!BUTTON_CHECK(button, MOUSE_B_LEFT))
		return ID_DONE;

	for (a = 0; a < NUM_BUTTONS; a++) {
		if (mx > b[a].x && mx < b[a].x + b[a].width && my > b[a].y && my < b[a].y + b[a].height) {
			return b[a].id;
		}
	}

	for (a = 0; a < 3; a++) {
		if (mx > c1[a].x && mx < c1[a].x + c1[a].width - 1 && my > c1[a].y && my < c1[a].y + c1[a].height) {
			c1[a].act_value = mx - c1[a].x;
			pal[col[0] * 3 + a] = c1[a].act_value << 2;

			screen_put_pal(col[0], pal[col[0] * 3], pal[col[0] * 3 + 1], pal[col[0] * 3 + 2]);

			re_draw();
			return ID_DONE;
		}

		if (mx > c2[a].x && mx < c2[a].x + c2[a].width - 1 && my > c2[a].y && my < c2[a].y + c2[a].height) {
			c2[a].act_value = mx - c2[a].x;
			pal[col[1] * 3 + a] = c2[a].act_value << 2;

			screen_put_pal(col[1], pal[col[1] * 3], pal[col[1] * 3 + 1], pal[col[1] * 3 + 2]);
			re_draw();
			return ID_DONE;
		}
	}

	return ID_DONE;
}

void
fatal_error(char *s)
{

	if (graph_enabled) {
		screen_restore_video_mode();
	}

	if (kb_enabled) {
		kb_closedown();
	}


	printf("\nFatal error:\n\t%s\n\n", s);
	fflush(stdout);
	exit(0);
}

void
read_pal(void)
{

	FILE           *f;
	int             a;

	f = fopen(input, "rb");
	if (!f) {
		fatal_error("Not able to open input file.");
	}
	fread(pal, sizeof(UBYTE), 768, f);
	fclose(f);

	for (a = 0; a < 768; a++) {
		pal[a] = pal[a] << 2;
	}
	screen_block_set_pal(pal);
}

void
write_pal(void)
{

	FILE           *f;
	int             a;

	for (a = 0; a < 768; a++) {
		pal[a] = pal[a] >> 2;
	}

	if (output)
		f = fopen(output, "wb");
	else
		f = fopen("out.pal", "wb");

	if (!f) {
		fatal_error("Not able to open output file.");
	}

	fwrite(pal, sizeof(UBYTE), 768, f);
	fclose(f);
}

void
buff_draw_outline_char(buffer_rec * buf, char ch, USHORT x, USHORT y, UBYTE col)
{
	int             a, b;

	for (a = 0; a < 3; a++)
		for (b = 0; b < 3; b++)
			buff_draw_char(buf, ch, x + a, y + b, 0);

	buff_draw_char(buf, ch, x + 1, y + 1, col);
}
