/*
* This file is part of the MGUI (Morello Graphic User Interface)
* library. Copyright 1995-2000 Vincenzo Morello.
*
* This file describes the C++ frame for the C MGUI library.
*
* Version 3.0  Jul 20, 2000
*
* All MGUI classes have the Cm suffix. Use something other,
* such as Ca, for suffixing application defined classes.
*
* All GUI Objects (buttons, labels, menu etc.) should be created via
* the new operator.
*
* All GUI Objects (both C and C++) are automatically destroyed when the
* CmShell instance, or any derived one, is deleted.
*
* The CmContainer class is a Superclass for CmShell and CmForm.
* All GUI object contructors have a pointer to this class as
* the first argument. This keeps the MGUI object hierarchy.
*
*/

#ifndef _MGUIPP_H
#define _MGUIPP_H

#include "mgui.h"

/**
 Base MGUI Class
 */
class CmMGUI {
protected:
	virtual inline void beep(void) { MBeep(); }

	int getKeyboardState(void);
	const char *getSysDir(void);

	int	paletteMode(void);
	int	getScreenWidth(void);
	int	getScreenHeight(void);
	int	getNPlanes(void);
	long	getNColors(void);
	MTColor	getBlackColor(void);
	MTColor	getWhiteColor(void);

	void enableCustomizing(void);
	char *getStringOption(const char *obj_name, const char *opt_name);
	char *getTextOption(const char *obj_name);
	int  getIntOption(const char *obj_name, const char *opt_name, int *ret);
	int  getFontOption(const char *obj_name, MTFont *ret);
	int  getBGColorOption(const char *obj_name, int *pr, int *pg, int *pb);
	int  getFGColorOption(const char *obj_name, int *pr, int *pg, int *pb);
	int  getShadowOption(const char *obj_name, int *shadow, int *i_t, int *o_t);

	void setTextOption(const char *obj_name, const char *text);
	void setIntOption(const char *obj_name, const char *rname, int val);
	void setFGColorOption(const char *obj_name, int r, int g, int b);
	void setBGColorOption(const char *obj_name, int r, int g, int b);
	void setShadowOption(const char *obj_name, int shadow, int in_t, int out_t);
	void setFontOption(const char *obj_name, MTFont font);
	void saveOptions(void);

	void logInit(int level, int nr, int nvr, int nc, int nvc, const char *fname);
	void log(int level, const char *fmt, ...);
	void maskedLog(int level, CARD32 mask, const char *fmt, ...);
	void logSetLevel(int level);
	void logSetMask(CARD32 a);
	CARD32 logGetMask(void);
	void logSetMaskBits(CARD32 a);
	void logResetMaskBits(CARD32 a);
	void setTraceFileName(const char *fname);
	void trace(const char *fmt,...);

	void getDefaultBackgroundRGB(int *pr, int *pg, int *pb);
	void setDefaultBackgroundRGB(int r, int g, int b);
	void getDefaultForegroundRGB(int *pr, int *pg, int *pb);
	void setDefaultForegroundRGB(int r, int g, int b);

	int  textCharX(const char *text, MTFont font, int x);
	int  textHeight(const char *text, MTFont font);
	int  charHeight(char car, MTFont font);
	int  textWidth(const char *text, MTFont font);
	int  textNWidth(const char *text, MTFont font, int len);
	int  charWidth(char car, MTFont font);

	int  spoolFile(const char *fname);
	int  spoolTempFile(const char *fname);

	void mainLoop(void);
	void loopWhileFlag(int *pflag, int value);
	void loopWhileEvents(int discard);
	void nextEvent(MEvent *pe, int wait);
	void processEvent(MEvent *pe);

	int messageDialog(const char *t, const char *msg, const char *btn1, const char *btn2, ...);
	int inputPrompt(const char *t, const char *msg, char *text, int len, const char *ok_btn,
	    const char *canc_btn);
	char *textPrompt(const char *t, const char *msg, const char *text, int w, int h, const char *ok,
	 const char *cancel);
	int  fileSelection(const char *title, const char *filter, char *fname, char *dname,
		 int change_cur_dir);
	char *getCurrentDirectory(void);
	void setInputPromptRGB(int r, int g, int b);
	void setMessageDialogRGB(int r, int g, int b);
	void setFileSelectionRGB(int r, int g, int b);
	void getInputPromptRGB(int *pr, int *pg, int *pb);
	void getMessageDialogRGB(int *pr, int *pg, int *pb);
	void getFileSelectionRGB(int *pr, int *pg, int *pb);
	void setFileSelectionTextStrings(
	    char *file_list, char *dir_list, char *drive_list, char *filter,
	    char *filename, char *curr_dir, char *ok_button, char *close_button);
};

/**
   Application Class.
   Provides the application's starting and terminating
   methods. The application program must define a derived
   instance to provide the pure virtual start() method.
   The program can be terminated at any point by calling
   CmAppl::end(code);
*/

class CmAppl : public CmMGUI {
	friend void MGUIMain(int argc, char **argv);
	static CmAppl *app_inst;
protected:
	CmAppl(void);
public:
/// The program starts receiving the main's ordinary arguments argc and argv.
	virtual void start(int argc, char **argv)=0;
/// Terminates the program freeing allocated resources and then calling exit(code)
	static void end(int code);
};


class CmMenuItem;

// Callback class definition
typedef class CmMGUI CmCallback;

typedef void (CmCallback::*VOID_CB)(void);
typedef void (CmCallback::*EVENT_CB)(MEvent *);
typedef void (CmCallback::*LONG_CB)(long);
typedef void (CmCallback::*INT_CB)(int);
typedef void (CmCallback::*MENUITEM_CB)(int);
typedef void (CmCallback::*ITEMSEL_CB)(CmMenuItem *);
typedef void (CmCallback::*STRING_CB)(char *);
typedef void (CmCallback::*NEWBLOCK_CB)(SBL_NEW_BLOCK *);
typedef void (CmCallback::*ITEMSEEK_CB)(SBL_ITEM_SEEK *);
typedef void (CmCallback::*EDITVAL_CB)(EDIT_VAL *);
typedef void (CmCallback::*EDITCURS_CB)(EDIT_CURS *);
typedef void (CmCallback::*EDITCHANGE_CB)(EDIT_CHANGE *);
typedef void (CmCallback::*LISTACT_CB)(LIST_ACT *);
typedef void (CmCallback::*LISTSEL_CB)(LIST_SEL *);
typedef void (CmCallback::*DRAWAREAPREF_CB)(DRAWAREA_PREF *);

/**
   Timeout Class.
*/

class CmTimeout : public CmMGUI {
	friend void _TimeoutCB_(TIMEOUT_ID id, void *ud);
	TIMEOUT_ID tid;
	CARD32	msec;
	int continuous;
	int stopped;
	VOID_CB callback;
	CmCallback *owner;

public:
	virtual ~CmTimeout(void);
	CmTimeout(CmCallback *own, VOID_CB cb, CARD32 msec, int continuous=False);
	void start(CARD32 msec);
	void start(void);
	void stop(void);
};

/**
 Abstract Color Class. See derived classes
*/

class CmColor {
protected:
	MTColor id;
public:
/// Doesn't allocate the color, simply creates another id
	CmColor(MTColor cc=0) { id = cc;}
/// internal use
	operator MTColor() { return id; }
/// returns the color's R-G-B components.
	inline void getRGB(int *r, int *g, int *b);
};

/**
  Ordinary color class. This is a read-only shared color used to set
  widget's background/foreground and for drawing primitives.
 */
class CmSharedColor : public CmColor {
public:
	virtual ~CmSharedColor(void);
/// creates a color with the R-G-B components specified in parameters (r,g,b)
	CmSharedColor(unsigned r, unsigned g, unsigned b);
/// creates a color as a shade of the color c. shift specifies the shade intensity (darker if < 100, brighter if > 100)
	CmSharedColor(CmColor *c, int shift=100);
};

/**
  Color cell class. This is a read-write non-shared color. It is available
  only when the video system runs in palette mode (usually 16 or 256 colors).
 */
class CmRWColor : public CmColor {
public:
	virtual ~CmRWColor(void);
	CmRWColor(unsigned r, unsigned g, unsigned b);
	void setRGB(int r, int g, int b);
};

/**
 BMP IMAGE Class
*/

class CmImage : public MIMAGE_DESCR {
protected:
	void allocImage(int imgw, int imgh, int np);
public:
	virtual ~CmImage(void);
	CmImage(const char *fname);
	CmImage(int imgw, int imgh, int np);

	inline void install(void);
	inline void uninstall(void);
	int  getWidth(void)   { return img_w; }
	int  getHeight(void)  { return img_h; }
	int  getNColors(void) { return ncolors; }
	int  getNPlanes(void) { return nplanes; }
	inline void getPixelRGB(int x, int y, int *pr, int *pg, int *pb);
	inline void setPixelRGB(int x, int y, int r, int g, int b);
	inline int  getPixelIndex(int x, int y);
	inline void setPixelIndex(int x, int y, int i);
	inline void getCMapEntry(int i, int *pr, int *pg, int *pb);
	inline void setCMapEntry(int i, int r, int g, int b);
	
	inline void merge(CmImage *src, int xo, int yo, int transparent);
	inline void setMask(CmImage *mask);
	CmImage *clone(void);
	CmImage *createImageMaskByCMapIndex(int index);
	CmImage *createImageMaskByRGB(int r, int g, int b);
	CmImage *createImageMaskByPixel(int x, int y);
	inline void computeBackgroundRGB(int *pr, int *pg, int *pb);
};

typedef CmImage CmBMPImage;

/**
 GIF IMAGE Class
*/
class CmGIFImage : public CmImage {
public:
	CmGIFImage(const char *fname);
};

/**
 TIFF IMAGE Class
*/
class CmTIFFImage : public CmImage {
public:
	CmTIFFImage(const char *fname);
};

/**
 Cursor Class
*/
class CmCursor {
	MTCursor cur;
public:
	operator MTCursor() { return cur;}
	CmCursor(CmImage *bits, int xs, int ys) {cur=MCreateCursor(bits,xs,ys);}
};

/**
 Font Class
*/
class CmFont {
	MTFont font;
	int loaded;
public:
	operator MTFont() { return font;}
	CmFont(int ff=DEFAULT_FONT)
		{ font = ff; loaded=0;}
	CmFont(const char *family, int h, int bold=0, int italic=0)
		{ font = MLoadFont(family,h,bold,italic); loaded=1;}
	~CmFont()
		{ if (loaded) MUnloadFont(font);}
	static const char *getFirstFamily(void) { return MGetFirstFontFamily();}
	static const char *getNextFamily(void) { return MGetNextFontFamily();}
};

class CmShell;
class CmForm;
class CmContainer;
class CmObject;

/**
 Abstract Object Class
*/

class CmObject : public CmMGUI {

	friend class CmOptionEdit;
	friend void _ObjectDestroyCB_(MOBJECT p, void *a, void *ud);
	friend void _EventPreProcessCB_(MOBJECT p, MEvent *pe, void *ud);

	void *user_data; 

protected:
	MOBJECT  obj;
	EVENT_CB eventCB;

	void reserved(void);
	virtual ~CmObject(void);
public:
	operator MOBJECT() {return obj; };

	inline void drawText(int x, int y, int w, int h, const char *text,
			MTFont font, CmColor *fgc, int align);
	inline void drawNText(int x, int y, int w, int h, const char *text,
			MTFont font, CmColor *fgc, int align, int n);
	inline void drawBGText(int x, int y, int w, int h, const char *text,
			MTFont font, CmColor *bgc, CmColor *fgc, int align);
	inline void drawBGNText(int x, int y, int w, int h, const char *text,
			MTFont font, CmColor *bgc, CmColor *fgc, int align, int n);
	inline void drawShadowedText(int x, int y, int w, int h, const char *text,
			MTFont font, int sh, CmColor *fg1, CmColor *fg2, int align);
	inline void drawShadowedNText(int x, int y, int w, int h, const char *text,
			MTFont font, int sh, CmColor *fg1, CmColor *fg2, int align, int n);
	inline void drawPoint(int x, int y, CmColor *col);
	inline void drawLine(int x1, int y1, int x2, int y2, CmColor *col);
	inline void drawPolyline(MPOINT *points, int n, CmColor *col);
	inline void drawFilledPolyline(MPOINT *points, int n, CmColor *col);
	inline void drawBox(int x, int y, int w, int h, CmColor *bgc);
	inline void drawFilledBox(int x, int y, int w, int h, CmColor *bgc);
	inline void drawCircle(int x, int y, int r, CmColor *col);
	inline void drawFilledCircle(int x, int y, int r, CmColor *col);
	inline void drawArc(int x, int y, int r, int a1, int a2, CmColor *col);
	inline void drawFilledArc(int x, int y, int r, int a1, int a2, CmColor *c);
	inline void drawImage(CmImage *gi, int img_x, int img_y, int w, int h,
			int x, int y);
	inline void drawWindow(void);
	inline void drawShadow(void);
	inline void drawSubShadow(int x, int y, int w, int h, CmColor *tsc, CmColor *bsc, int thick);
	inline void drawBorder(int x, int y, int w, int h, CmColor *col, int thick);

	inline void drawXorLine(int x1, int y1, int x2, int y2);
	inline void drawXorBox(int x, int y, int w, int h);

	inline void setSensitive(void);
	inline void setUnsensitive(void);
	inline void setSensitivity(int true_false);
	inline void setSelectable(void);
	inline void setUnselectable(void);
	inline void setColor(CmColor *bgc, CmColor *fgc);
	inline void setBackgroundRGB(unsigned r, unsigned g, unsigned b);
	inline void setForegroundRGB(unsigned r, unsigned g, unsigned b);
	inline void setText(const char *text);
	inline void setName(const char *name);
	inline void setFont(MTFont font, int resize_obj=0);
	inline void setShadow(int type, int in=0, int out=0);
	inline void setResize(int resize_w, int resize_h);
	inline void setSize(int width, int height);
	inline void setWidth(int width);
	inline void setHeight(int height);
	inline void setCompositeSize(int width, int height);
	inline void setCompositeWidth(int width);
	inline void setCompositeHeight(int height);
	inline void setPosition(int x, int y);
	inline void setX(int x);
	inline void setY(int y);
	inline void setBackgroundImage(CmImage *gi, int mode);
	inline void setBackgroundImageFile(const char *fname, int mode);
	inline void setColorFromBackgroundImage(void);
	inline void setTransparency(CARD8 level);
	inline CARD8 getTransparency(void);
	inline void setAttachment(int top, int bottom, int left, int right);
	inline void setOffset(int t_off, int b_off, int l_off, int r_off);
	inline void setCursor(MTCursor cursor);
	inline void setPopupHelp(const char *str, int delay, int duration);
	inline void setLabelHelp(const char *str, CmObject *label);
	inline void setUserData(void *ud) { user_data = ud; };
	inline void changePopupHelpText(const char *str);
	inline void changeLabelHelpText(const char *str);
	inline void topAttach(int att_type, int off, CmObject *obj);
	inline void bottomAttach(int att_type, int off, CmObject *obj);
	inline void leftAttach(int att_type, int off, CmObject *obj);
	inline void rightAttach(int att_type, int off, CmObject *obj);
	inline void align(int h_align, int v_align);

	inline void getText(char *text);
	inline const char *getName(void);
	inline void *getUserData(void) { return user_data; };
	inline MTFont getFont(void);
	inline void getShadow(int *type, int *in, int *out);
	inline int isSensitive(void);
	inline int isSelectable(void);
	inline int isMapped(void);
	inline int getWidth(void);
	inline int getHeight(void);
	inline int getCompositeWidth(void);
	inline int getCompositeHeight(void);
	inline int getX(void);
	inline int getY(void);
	CmColor *getBackgroundColor(void);
	CmColor *getForegroundColor(void);
	CmContainer *getParent(void);
	CmShell *getShell(void);
	inline CmImage *getBackgroundImage(void);

	inline void resize(int w, int h);
	inline void redraw(void);
	inline void select(void);
	inline void map(void);
	inline void unmap(void);
	inline void enableOptions(const char *name, CARD32 mask);
	inline void saveOptions(void);
	inline void unlink(int resize_obj);
	inline void focusLink(CmObject *obj_to_link);
	void reparent(CmContainer *new_parent, CmObject *where);
	void link(CmContainer *parent, CmObject *where, int resize_parent);

	void setEventCallback(CmCallback *pp, EVENT_CB ff, CARD32 mask);
#ifdef WIN_MGUI
	inline void print(DOC_ID, MPRINT_OPTIONS *);
#endif
};

/**
 Abstract Container Class
*/

class CmContainer : public CmObject {
public:
	virtual void addChild(CmObject *child)=0;

	CmObject *getSelectedObject(void);
	CmObject *getFirstChild(void);
	CmObject *getNextChild(void);
	inline int getNChildren(void);
};

/**
 SHELL Object
*/

class CmShell : public CmContainer {
	friend void _WMCloseCB_(MOBJECT p, void *a, void *ud);
	VOID_CB closeCB;
public:
	~CmShell(void);
	CmShell(const char *text=0, int flags=0);

	inline Window getWindow(void);

	inline void realize(void);
	inline void realize(int x, int y);
	inline void realize(int x, int y, int w, int h);
	inline void unrealize(void);
	inline void raise(void);
	void addChild(CmObject *child);

	void setWMCloseCallback(CmCallback *pp, VOID_CB ff);
};

/**
 Abstract Form Class
*/

class CmForm : public CmContainer {
protected:
	CmForm(void){};
public:
	inline void setSpacing(int offset);
	inline void setFocusWrap(int true_false);
	inline void setFocusHighlight(int true_false);
	inline void selectFirstChild(void);
	inline void selectLastChild(void);

	void addChild(CmObject *object);
};

/**
 PosForm Class
*/

class CmPosForm : public CmForm {
public:
	CmPosForm(CmContainer *parent);
};

/**
 RowForm Class
*/

class CmRowForm : public CmForm {
public:
	CmRowForm(CmContainer *parent);
};

/**
 ColForm Class
*/
class CmColForm : public CmForm {
public:
	CmColForm(CmContainer *parent);
};

/**
 Abstract TableForm Class
 */
class CmAbsTableForm : public CmForm {
public:
    inline void showGrid(void);
    inline void hideGrid(void);
    inline void setRowHeight(int row, int height);
    inline void setColWidth(int col, int width);
    inline void setColFont(int col, MTFont f);
    inline void setColAlign(int col, int al);
    inline void setColBackgroundRGB(int col, int r, int g, int b);
    inline void setColForegroundRGB(int col, int r, int g, int b);
    inline void setCellText(int x, int y, const char *text);
    inline void getCellText(int x, int y, char *text);
    inline void setCellFont(int x, int y, MTFont f);
    inline void setCellAlign(int x, int y, int h_al, int v_al);
    inline void setCellBackgroundRGB(int x, int y, int r, int g, int b);
    inline void setCellForegroundRGB(int x, int y, int r, int g, int b);
    CmObject *getCellObject(int x, int y);
    inline void deleteRow(int row);
    inline void deleteDataRows(void);
};

/**
 TableForm Class
 */
class CmTableForm : public CmAbsTableForm {
public:
    CmTableForm(CmContainer *parent, int n_columns);
};


/**
 ScrolledTableForm Class
 */
class CmScrolledTableForm : public CmAbsTableForm {
public:
    CmScrolledTableForm(CmContainer *parent, int n_columns,
	int clip_w, int clip_h,	int head_rows=0, int head_cols=0);
};

/**
 Abstract Paged Form
 */
class CmAbsPagedForm : public CmForm {
public:
	inline void setActivePage(int page);
	inline void disablePageKeys(void);
	inline void enablePageKeys(void);
};

/**
 PagedPos Form
 */
class CmPagedPosForm : public CmAbsPagedForm {
public:
	CmPagedPosForm(CmContainer *parent, int n_pages);
};

/**
 PagedRow Form
 */
class CmPagedRowForm : public CmAbsPagedForm {
public:
	CmPagedRowForm(CmContainer *parent, int n_pages);
};

/**
 PagedCol Form
 */
class CmPagedColForm : public CmAbsPagedForm {
public:
	CmPagedColForm(CmContainer *parent, int n_pages);
};

/**
 PagedPos Form Extended
 */
class CmPagedPosFormExt : public CmAbsPagedForm {
public:
	CmPagedPosFormExt(CmContainer *parent, int n_pages);
};

/**
 PagedRow Form Extended
 */
class CmPagedRowFormExt : public CmAbsPagedForm {
public:
	CmPagedRowFormExt(CmContainer *parent, int n_pages);
};

/**
 PagedCol Form Extended
 */
class CmPagedColFormExt : public CmAbsPagedForm {
public:
	CmPagedColFormExt(CmContainer *parent, int n_pages);
};

/**
 TabbedPos Form
 */
class CmTabbedPosForm : public CmAbsPagedForm {
public:
	CmTabbedPosForm(CmContainer *parent, int n_pages, MTFont f=HELV_SMALL);
};

/**
 TabbedRow Form
 */
class CmTabbedRowForm : public CmAbsPagedForm {
public:
	CmTabbedRowForm(CmContainer *parent, int n_pages, MTFont f=HELV_SMALL);
};

/**
 TabbedCol Form
 */
class CmTabbedColForm : public CmAbsPagedForm {
public:
	CmTabbedColForm(CmContainer *parent, int n_pages, MTFont f=HELV_SMALL);
};

/**
 Abstract Clipped Form
 */
class CmAbsCForm : public CmForm {
public:
	inline void setClipX(int clip_x);
	inline void setClipY(int clip_y);
	inline int	getClipX(void);
	inline int	getClipY(void);
};

/**
 Abstract Scrolled Form
 */
class CmAbsScrolledForm : public CmAbsCForm {
public:
	inline void setIncr(int h_inc, int v_inc);
};

/**
 Scrolled PosForm
 */
class CmScrolledPosForm : public CmAbsScrolledForm {
public:
	CmScrolledPosForm(CmContainer *parent, int w, int h);
};

/**
 Scrolled RowForm
 */
class CmScrolledRowForm : public CmAbsScrolledForm {
public:
	CmScrolledRowForm(CmContainer *parent, int w, int h);
};

/**
 Scrolled ColForm
 */
class CmScrolledColForm : public CmAbsScrolledForm {
public:
	CmScrolledColForm(CmContainer *parent, int w, int h);
};

/**
 Clipped PosForm
 */
class CmClippedPosForm : public CmAbsCForm {
public:
	CmClippedPosForm(CmContainer *parent, int w, int h);
};

/**
 Clipped RowForm
 */
class CmClippedRowForm : public CmAbsCForm {
public:
	CmClippedRowForm(CmContainer *parent, int w, int h);
};

/**
 Clipped ColForm
 */
class CmClippedColForm : public CmAbsCForm {
public:
	CmClippedColForm(CmContainer *parent, int w, int h);
};

/**
 LABEL Object
*/

class CmLabel : public CmObject {
public:
	CmLabel(CmContainer *parent, const char *text=0, MTFont font=DEFAULT_FONT);
	inline void setMargin(int wm, int hm);
	inline void setHSpacing(int hs);
	inline void setAlignment(int align);
};

/**
 Abstract Edit Class
*/

class CmAbsEdit : public CmObject {
	friend void _EditNewValueCB_(MOBJECT p, char *text, void *ud);
	friend void _EditValueChangeCB_(MOBJECT p, EDIT_CHANGE *ec, void *ud);
	STRING_CB newValueCB;
	EDITCHANGE_CB valueChangeCB;
protected:
	CmAbsEdit(void){};
public:
	inline void changeText(int off, int len, const char *new_t, int chg_len);
	inline void getSubString(int start, int end, char *str);
	inline int  findSubStringForward(int off, const char *str, int icase);
	inline int  findSubStringBackward(int off, const char *str, int icase);
	inline void setSelection(int start, int end);
	inline void getSelection(int *pstart, int *pend);
	inline char *getAllocatedText(void);
	inline void setCursorPos(int pos);
	inline int  getCursorPos(void);

	void setNewValueCallback(CmCallback *, STRING_CB);
	void setValueChangeCallback(CmCallback *, EDITCHANGE_CB);
};

/**
 Abstract EditField Class
*/

class CmAbsEditField : public CmAbsEdit {
	friend void _EditActCB_(MOBJECT p, char *text, void *ud);
	friend void _EditValCB_(MOBJECT p, EDIT_VAL *ev, void *ud);
	friend void _EditHotKeyCB_(MOBJECT p, void *a, void *ud);
	STRING_CB  activateCB;
	EDITVAL_CB validateCB;
	VOID_CB    hotKeyCB;
protected:
	CmAbsEditField(void){};
public:
	inline void setFilter(int filter, const char *extra_chars);
	inline void setTextExt(const char *text, int validated, int notify);
	inline void setValidatedColors(CmColor *bgc, CmColor *fgc);
	inline void setInvalidatedColors(CmColor *bgc, CmColor *fgc);
	inline void setValidationStatus(int validated);
	inline void selectForReplace(void);
	inline const char *text(void);
	inline int  getFilter(void);
	inline int  isValidated(void);

	void setActivateCallback(CmCallback *, STRING_CB);
	void setHotKeyCallback(CmCallback *, VOID_CB, int);
	void setValidateCallback(CmCallback *, EDITVAL_CB);
};

/**
 EditField Classes
*/

class CmEditField : public CmAbsEditField {
public:
	CmEditField(CmContainer *parent, const char *text, int len,
		int clip_len=-1, MTFont font=DEFAULT_FONT);
};

/**
 Hidden EditField Classes
*/
class CmHiddenEditField : public CmAbsEditField {
public:
	CmHiddenEditField(CmContainer *parent, const char *text, int len,
		int clip_len=-1, MTFont font=DEFAULT_FONT, char shown='*');
};

/**
 Spin EditField Classes
*/
class CmSpinEditField : public CmAbsEditField {
	friend void _SpinCB_(MOBJECT p, MVALUE a, void *ud);
	INT_CB spinCB;
public:
	CmSpinEditField(CmContainer *parent, const char *text, int len,
		int clip_len, MTFont font,
		CmCallback *owner, INT_CB cb);
};

/**
 Abstract EditWindow Classes
*/
class CmAbsEditWindow : public CmAbsEdit {
	friend void _EditCursorCB_(MOBJECT p, EDIT_CURS *ec, void *ud);
	EDITCURS_CB cursCB;
public:
	inline void setCursorXY(int x, int y);
	inline void getCursorXY(int *px, int *py);
	inline void setAutoIndent(int on_off);
	inline void setTabSize(int size);
	inline void setClipWindow(int clip_w, int clip_h);

	void setCursorCallback(CmCallback *, EDITCURS_CB);
};

/**
 EditWindow Classes
*/

class CmEditWindow : public CmAbsEditWindow {
public:
	CmEditWindow(CmContainer *parent, const char *text, int width, int height,
		MTFont font=DEFAULT_FONT);
};

/**
 Scrolled EditWindow Classes
*/
class CmScrolledEditWindow : public CmAbsEditWindow {
public:
	CmScrolledEditWindow(CmContainer *parent, const char *text, int width, int height,
		MTFont font=DEFAULT_FONT);
};

/**
 Abstract Button Object
*/

class CmAbsButton : public CmObject {
public:
	inline void setImages(const char *normal, const char *arm=0, const char *uns=0);
	inline void changeImageFiles(const char *normal, const char *arm, const char *uns,
		const char *norm_on, const char *arm_on, const char *uns_on);
	inline CmImage *changeNormalImage(CmImage *new_normal);
	inline CmImage *changeArmedImage(CmImage *new_armed);
	inline CmImage *changeUnsensitiveImage(CmImage *new_unsens);
	inline CmImage *changeNormalOnImage(CmImage *new_normal_on);
	inline CmImage *changeArmedOnImage(CmImage *new_armed_on);
	inline CmImage *changeUnsensitiveOnImage(CmImage *new_unsens_on);
	inline void setTextVsImageAlignment(CARD8 align);
	inline void setOnMouseOverForegroundColor(CmColor *fgc);
	inline void setOnMouseOverForegroundRGB(int r, int g, int b);
	inline void setShadowOnMouseOver(int thick);
	inline void setAccelerator(int key);
};

/**
 Push Button Object
*/

class CmPushButton : public CmAbsButton {
	friend void _PushButtonActivateCB_(MOBJECT p, void *a, void *ud);
	VOID_CB activateCB;
public:
	CmPushButton(CmContainer *p, const char *text, MTFont font=DEFAULT_FONT);
	CmPushButton(CmContainer *p, CmImage *norm, CmImage *arm=0, CmImage *uns=0,
		CmImage *norm_on=0, CmImage *arm_on=0, CmImage *uns_on=0);
	CmPushButton(CmContainer *p, const char *norm, const char *arm, const char *uns=0,
		const char *norm_on=0, const char *arm_on=0, const char *uns_on=0);
	CmPushButton(CmContainer *p, const char *text, MTFont font, 
		CmImage *norm, CmImage *arm=0, CmImage *uns=0,
		CmImage *norm_on=0, CmImage *arm_on=0, CmImage *uns_on=0);
	CmPushButton(CmContainer *p, const char *text, MTFont font,
		const char *norm, const char *arm=0, const char *uns=0,
		const char *norm_on=0, const char *arm_on=0, const char *uns_on=0);

	inline void setRepeatDelay(int first, int repeat);
	inline void setMargin(int wm, int hm);
	inline void setHSpacing(int hs);
	inline void setAlignment(int align);
	void setCallback(CmCallback *owner, VOID_CB cb);
};

/**
 Toggle Button Object
*/

class CmToggleButton : public CmAbsButton {
	friend void _ToggleButtonActivateCB_(MOBJECT p, MVALUE a, void *ud);
	INT_CB activateCB;
public:
	CmToggleButton(CmContainer *par, const char *text, MTFont font=DEFAULT_FONT);
	CmToggleButton(CmContainer *p, CmImage *norm, CmImage *arm=0, CmImage *uns=0,
		CmImage *norm_on=0, CmImage *arm_on=0, CmImage *uns_on=0);
	CmToggleButton(CmContainer *p, const char *norm, const char *arm, const char *uns=0,
		const char *norm_on=0, const char *arm_on=0, const char *uns_on=0);
	CmToggleButton(CmContainer *p, const char *text, MTFont font, 
		CmImage *norm, CmImage *arm=0, CmImage *uns=0,
		CmImage *norm_on=0, CmImage *arm_on=0, CmImage *uns_on=0);
	CmToggleButton(CmContainer *p, const char *text, MTFont font,
		const char *norm, const char *arm=0, const char *uns=0,
		const char *norm_on=0, const char *arm_on=0, const char *uns_on=0);

	inline void arm(void);
	inline void disarm(void);

	inline int isArmed(void);

	void setCallback(CmCallback *owner, INT_CB cb);
};

/**
 Radio Button Object
*/

class CmRadioButton : public CmAbsButton {
	friend void _RadioButtonActivateCB_(MOBJECT p, MVALUE a, void *ud);
	INT_CB activateCB;
public:
	CmRadioButton(CmContainer *par, const char *text, MTFont font=DEFAULT_FONT);
	CmRadioButton(CmContainer *p, CmImage *norm, CmImage *arm=0, CmImage *uns=0,
		CmImage *norm_on=0, CmImage *arm_on=0, CmImage *uns_on=0);
	CmRadioButton(CmContainer *p, const char *norm, const char *arm, const char *uns=0,
		const char *norm_on=0, const char *arm_on=0, const char *uns_on=0);
	CmRadioButton(CmContainer *p, const char *text, MTFont font, 
		CmImage *norm, CmImage *arm=0, CmImage *uns=0,
		CmImage *norm_on=0, CmImage *arm_on=0, CmImage *uns_on=0);
	CmRadioButton(CmContainer *p, const char *text, MTFont font,
		const char *norm, const char *arm=0, const char *uns=0,
		const char *norm_on=0, const char *arm_on=0, const char *uns_on=0);

	inline void arm(void);
	inline void disarm(void);

	inline int isArmed(void);

	void setCallback(CmCallback *owner, INT_CB cb);
};

/**
 Pixmap Object
*/

class CmPixmap : public CmObject {
public:
	CmPixmap(CmContainer *par, CmImage *normal);
	CmPixmap(CmContainer *par, const char *fname);

	inline void setImage(CmImage *normal);
	inline void setImage(const char *fname);
};

/**
 DrawArea Object
*/
class CmDrawArea : public CmObject {
	friend void _DrawAreaExposeCB_(MOBJECT, MEvent *, void *);
	friend void _DrawAreaInputCB_(MOBJECT, MEvent *, void *);
	friend void _DrawAreaResizeCB_(MOBJECT, DRAWAREA_PREF *, void *);
	EVENT_CB exposeCB;
	EVENT_CB inputCB;
	DRAWAREAPREF_CB resizeCB;
public:
	CmDrawArea(CmContainer *parent, int w, int h);

	void setExposeCallback(CmCallback *, EVENT_CB);
	void setInputCallback(CmCallback *, EVENT_CB);
	void setResizeCallback(CmCallback *, DRAWAREAPREF_CB);
};

/**
 Status Bar Object
*/

class CmStatusBar : public CmObject {
public:
	CmStatusBar(CmContainer *parent, const char *text=0, int max=10, MTFont font=DEFAULT_FONT);

	inline void setMax(int max);
	inline void setPos(int pos);
	inline void setSliderColor(CmColor *col);
	inline void setSliderRGB(int r, int g, int b);
};


/**
 Scroll Bar Object
*/

class CmScrollBar : public CmObject {
	friend void _ScrollBarValueCB_(MOBJECT p, MVALUE a, void *ud);
	LONG_CB changeCB;
public:
	CmScrollBar(CmContainer *parent, int type, long n, long max);

	inline void setMax(long max);
	inline void setPos(long pos);
	inline void setN(long n);
	inline void setIncr(int incr);
	inline void setValues(long max, long n, long pos);
	inline long getMax(void);
	inline long getPos(void);
	inline long getN(void);

	void setCallback(CmCallback *, LONG_CB);
};

/**
 SFile Object
*/

class CmSFile : public CmObject {
public:
	CmSFile(CmContainer *parent, const char *fname, MTFont font=DEFAULT_FONT,
		int nrow=10, int ncol=20);

	inline void setFile(const char *fname);
	inline void setHMargin(int hm);
};

/**
 SText Object
*/

class CmSText : public CmObject {
public:
	CmSText(CmContainer *parent, const char *text, MTFont font=DEFAULT_FONT,
		int nrow=10, int ncol=20);

	inline void setText(const char *text);
};

/**
 FormattedText Object
*/

class CmFormattedText : public CmObject {
public:
	CmFormattedText(CmContainer *parent, const char *text, int clip_w,
		MTFont font=DEFAULT_FONT);
	inline void setMargin(int w_m, int h_m);
	inline void setHSpacing(int h_s);
};

class CmSFormattedText : public CmObject {
public:
	CmSFormattedText(CmContainer *parent, const char *text, int clip_w,
		int clip_h, MTFont font=DEFAULT_FONT);
	inline void setMargin(int w_m, int h_m);
	inline void setHSpacing(int h_s);
};

/**
 Abstract List Class
*/

class CmAbsList : public CmObject {
	friend class CmOptionEdit;
	friend void _ListActivateCB_(MOBJECT p, LIST_ACT *la, void *ud);
	friend void _ListPickCB_(MOBJECT p, LIST_ACT *la, void *ud);
	friend void _ListSelectionCB_(MOBJECT p, LIST_SEL *la, void *ud);
	LISTACT_CB activateCB;
	LISTACT_CB pickCB;
	LISTSEL_CB selectionCB;
protected:
	CmAbsList(void){};
public:
	inline void setSortCallback(STRCMP_CB cb);
	inline void setHead(const char *text);
	inline void setHeadColor(CmColor *bgc, CmColor *fgc);
	inline void setHMargin(int hm);
	inline void setItemCheckStatus(const char *item, int status);

	inline LIST_ACT *getFirstItem(void);
	inline LIST_ACT *getFirstSelectedItem(void);
	inline LIST_ACT *getNextItem(void);
	inline LIST_ACT *getNextSelectedItem(void);
	inline LIST_ACT *getFirstUnselectedItem(void);
	inline LIST_ACT *getNextUnselectedItem(void);
	inline long      getNItem(void);
	inline LIST_ACT *getItem(const char *item);
	inline LIST_ACT *getElement(long n);
	inline LIST_ACT *getSelected(void);
	inline int       getItemCheckStatus(const char *item);
	inline void	  getHead(char *buff);

	inline void selectItem(const char *item);
	inline void deselectItem(const char *item);
	inline void selectElement(long n);
	inline void deselectElement(long n);
	inline void freeze(void);
	inline void thaw(void);
	inline void reset(void);
	inline void clear(void);
	inline int  addItem(const char *item, void *data);
	inline int  delItem(const char *item);
	inline int  addElement(long n, const char *item, void *data);
	inline int  delElement(long n);
	inline void changeItem(const char *old, const char *new_item, void *u_data);
	inline void changeElement(long n, const char *new_item, void *u_data);
	inline void enableMultiSelection(void);

	void setCallback(CmCallback *, LISTACT_CB);
	void setPickCallback(CmCallback *, LISTACT_CB);
	void setSelectionCallback(CmCallback *, LISTSEL_CB);
};

/**
 Scrolled List Class
*/
class CmSList : public CmAbsList {
public:
	CmSList(CmContainer *parent, MTFont font, int row_len,
		int nrow, int ncol);
};

/**
 Edit List Class
*/
class CmEditList : public CmAbsList {
public:
	CmEditList(CmContainer *parent, MTFont font, EL_TEMPLATE *elt,
		int ne, int nrow, int ncol);
	int     addItem(char **argv, void *u_data);
	int     addItemFromEdits(void *u_data);
	void    changeElement(long n, char **new_ele, void *data);
	void    delElement(long n);
	void    seekItem(char **argv, long *pos, void **u_data);
	void    getItemField(const char *item, int ii, char *buff);
	void    getEditText(int ii, char *buff);
	void    setEditText(int ii, const char *buff);
};

/**
 Clipped ScrolledList Class
*/
class CmClippedSList : public CmAbsList {
	friend void _ClippedListNewBlockCB_(MOBJECT, SBL_NEW_BLOCK *, void *);
	friend void _ClippedListItemSeekCB_(MOBJECT, SBL_ITEM_SEEK *, void *);
	NEWBLOCK_CB newBlockCB;
	ITEMSEEK_CB itemSeekCB;
public:
	CmClippedSList(CmContainer *parent,
		CmCallback *owner, NEWBLOCK_CB nb, ITEMSEEK_CB is,
		MTFont font, int nrow, int ncol, int xc, long xn=0);
};

/**
 OptionEdit Class
*/

class CmOptionEdit : public CmAbsEditField {
	CmAbsList *slist;
public:
	~CmOptionEdit(void);
	CmOptionEdit(CmContainer *parent, const char *text, int len, int clip_len=-1,
		int nrow=10, MTFont font=DEFAULT_FONT);
	CmAbsList *getSList(void) { return slist; };
};

/**
 Menu Class
*/

class CmMenuBar : public CmObject {
public:
	CmMenuBar(CmContainer *parent, MTFont font);
	inline void setAcceleratorKey(int key);
};

class CmMenu;

/**
 Abstract MenuLine Class
*/
class CmMenuLine : public CmMGUI {
protected:
	friend class CmMenu;
	MENU_ITEM id;
	MOBJECT pdobj;
	CmMenuLine *next;
	virtual ~CmMenuLine(void) {}
};

/**
 MenuSeparator Class
*/
class CmMenuSeparator : public CmMenuLine {
public:
	CmMenuSeparator(CmMenu *pd);
};

/**
 Abstract MenuItem Class
*/
class CmAbsMenuItem : public CmMenuLine {
	friend class CmMenu;
	friend void _MenuItemActivateCB_(MENU_ITEM item, MVALUE status, void *ud);
	MENUITEM_CB itemCB;
	CmCallback *owner;
protected:
	CmAbsMenuItem(CmMenu *pd, CmCallback *ff, MENUITEM_CB cb);
public:
	inline void setText(const char *text);
	inline void setSensitive(void);
	inline void setUnsensitive(void);
	inline void setAcceleratorKey(int key, const char *a_text=0);
	inline int  setCheckStatus(int true_false);
	inline int  getCheckStatus(void);
	inline const char *getText(void);
	inline CmMenu *getMenu(void);
	inline int  toggleCheckStatus(void);
};

/**
 MenuItem Class
*/
class CmMenuItem : public CmAbsMenuItem {
public:
	CmMenuItem(CmMenu *pd, const char *text, CmCallback *ff=0, MENUITEM_CB cb=0, int acc=0, const char *at=0);
};

/**
 Radio MenuItem Class
*/
class CmMenuRadioItem : public CmAbsMenuItem {
public:
	CmMenuRadioItem(CmMenu *pd, const char *text, CmCallback *ff=0, MENUITEM_CB cb=0, int acc=0, const char *at=0);
};

/**
 Toggle MenuItem Class
*/
class CmMenuToggleItem : public CmAbsMenuItem {
public:
	CmMenuToggleItem(CmMenu *pd, const char *text, CmCallback *ff=0, MENUITEM_CB cb=0, int acc=0, const char *at=0);
};

/**
 The container for menu items.
 Can be created as a stand alone container (popup), a container for a menu bar,
 a container as child of an item (subpulldown).
 */
class CmMenu : public CmObject {
	friend class CmMenuSeparator;
	friend class CmAbsMenuItem;
	CmMenuLine *list;
	ITEMSEL_CB itemSelCB;
public:
	virtual ~CmMenu(void);
	CmMenu(CmMenuBar *menu, const char *title);
	CmMenu(CmMenuItem *it);
	CmMenu(MTFont font=HELV_MEDIUM);
	void setText(const char *text);
	const char *getText(void);
	void popup(int x, int y);

	void setItemSelectionCallback(CmCallback *, ITEMSEL_CB);
};

/*******************************************************************/
// INLINE FUNCTIONS
/*******************************************************************/

/*
 * Inline CmObject functions
 */

inline void CmColor::getRGB(int *r, int *g, int *b)
{
	MQueryColor(id, r, g, b);
}

inline void CmObject::drawText(int x, int y, int w, int h, const char *text,
		MTFont font, CmColor *fgc, int align)
{
	MDrawText(obj, x, y, w, h, text, font, *fgc, align);
}

inline void CmObject::drawNText(int x, int y, int w, int h, const char *text,
		MTFont font, CmColor *fgc, int align, int n)
{
	MDrawNText(obj, x, y, w, h, text, font, *fgc, align, n);
}

inline void CmObject::drawBGText(int x, int y, int w, int h, const char *text,
		MTFont font, CmColor *bgc, CmColor *fgc, int align)
{
	MDrawBGText(obj, x, y, w, h, text, font, *bgc, *fgc, align);
}

inline void CmObject::drawBGNText(int x, int y, int w, int h, const char *text,
		MTFont font, CmColor *bgc, CmColor *fgc, int align, int n)
{
	MDrawBGNText(obj, x, y, w, h, text, font, *bgc, *fgc, align, n);
}

inline void CmObject::drawShadowedText(int x, int y, int w, int h, const char *text,
		MTFont font, int sh, CmColor *fgc1, CmColor *fgc2, int align)
{
	MDrawShadowedText(obj, x, y, w, h, text, font, sh, *fgc1, *fgc2, align);
}

inline void CmObject::drawShadowedNText(int x, int y, int w, int h, const char *text,
		MTFont font, int sh, CmColor *fgc1, CmColor *fgc2, int align, int n)
{
	MDrawShadowedNText(obj, x, y, w, h, text, font, sh, *fgc1, *fgc2, align, n);
}

inline void CmObject::drawPoint(int x, int y, CmColor *col)
{
	MDrawPoint(obj, x, y, *col);
}

inline void CmObject::drawLine(int x1, int y1, int x2, int y2, CmColor *col)
{
	MDrawLine(obj, x1, y1, x2, y2, *col);
}

inline void CmObject::drawPolyline(MPOINT *points, int n, CmColor *col)
{
	MDrawPolyline(obj, points, n, *col);
}

inline void CmObject::drawFilledPolyline(MPOINT *points, int n, CmColor *col)
{
	MDrawFilledPolyline(obj, points, n, *col);
}

inline void CmObject::drawBox(int x, int y, int w, int h, CmColor *col)
{
	MDrawBox(obj, x, y, w, h, *col);
}

inline void CmObject::drawFilledBox(int x, int y, int w, int h, CmColor *col)
{
	MDrawFilledBox(obj, x, y, w, h, *col);
}

inline void CmObject::drawCircle(int x, int y, int r, CmColor *col)
{
	MDrawCircle(obj, x, y, r, *col);
}

inline void CmObject::drawFilledCircle(int x, int y, int r, CmColor *col)
{
	MDrawFilledCircle(obj, x, y, r, *col);
}

inline void CmObject::drawArc(int x, int y, int r, int a1, int a2, CmColor *col)
{
	MDrawArc(obj, x, y, r, a1, a2, *col);
}

inline void CmObject::drawFilledArc(int x, int y, int r, int a1, int a2, CmColor *c)
{
	MDrawFilledArc(obj, x, y, r, a1, a2, *c);
}

inline void CmObject::drawImage(CmImage *gi, int img_x, int img_y, int w, int h,
		int x, int y)
{
	MDrawImage(obj, gi, img_x, img_y, w, h, x, y);
}

inline void CmObject::drawXorLine(int x1, int y1, int x2, int y2)
{
	MDrawXorLine(obj, x1, y1, x2, y2);
}

inline void CmObject::drawXorBox(int x, int y, int w, int h)
{
	MDrawXorBox(obj, x, y, w, h);
}

inline void CmObject::drawWindow(void)
{
	MDrawWindow(obj);
}

inline void CmObject::drawShadow(void)
{
	MDrawShadow(obj);
}

inline void CmObject::drawSubShadow(int x, int y, int w, int h, CmColor *tsc, CmColor *bsc, int thick)
{
	MDrawSubShadow(obj, x, y, w, h, *tsc, *bsc, thick);
}

inline void CmObject::drawBorder(int x, int y, int w, int h, CmColor *col, int thick)
{
	MDrawBorder(obj, x, y, w, h, *col, thick);
}

inline void CmObject::setSensitive(void)
{
	MObjectSetSensitive(obj);
}

inline void CmObject::setUnsensitive(void)
{
	MObjectSetUnsensitive(obj);
}

inline void CmObject::setSensitivity(int true_false)
{
	MObjectSetSensitivity(obj, true_false);
}

inline void CmObject::setSelectable(void)
{
	MObjectSetSelectable(obj);
}

inline void CmObject::setUnselectable(void)
{
	MObjectSetUnselectable(obj);
}

inline void CmObject::setColor(CmColor *bgc, CmColor *fgc)
{
	MObjectSetColor(obj, *bgc, *fgc);
}

inline void CmObject::setBackgroundRGB(unsigned r, unsigned g, unsigned b)
{
	MObjectSetBackgroundRGB(obj, r, g, b);
}

inline void CmObject::setForegroundRGB(unsigned r, unsigned g, unsigned b)
{
	MObjectSetForegroundRGB(obj, r, g, b);
}

inline void CmObject::setText(const char *text)
{
	MObjectSetText(obj, text);
}

inline void CmObject::setBackgroundImage(CmImage *gi, int mode)
{
	MObjectSetBackgroundImage(obj, gi, mode);
}

inline void CmObject::setBackgroundImageFile(const char *fn, int mode)
{
	MObjectSetBackgroundImageFile(obj, fn, mode);
}

inline void CmObject::setColorFromBackgroundImage(void)
{
	MObjectSetColorFromBackgroundImage(obj);
}

inline void CmObject::setTransparency(CARD8 level)
{
	MObjectSetTransparency(obj, level);
}

inline CARD8 CmObject::getTransparency(void)
{
	return MObjectGetTransparency(obj);
}

inline void CmObject::setName(const char *name)
{
	MObjectSetName(obj, name);
}

inline void CmObject::setFont(MTFont font, int resize)
{
	MObjectSetFont(obj, font, resize);
}

inline void CmObject::setShadow(int type, int in, int out)
{
	MObjectSetShadow(obj, type, in, out);
}

inline void CmObject::setResize(int resize_w, int resize_h)
{
	MObjectSetResize(obj, resize_w, resize_h);
}

inline void CmObject::align(int h_align, int v_align)
{
	MObjectAlign(obj, h_align, v_align);
}

inline void CmObject::setCursor(MTCursor cursor)
{
	MObjectSetCursor(obj, cursor);
}

inline void CmObject::setPopupHelp(const char *str, int delay, int duration)
{
	MObjectSetPopupHelp(obj, str, delay, duration);
}

inline void CmObject::setLabelHelp(const char *str, CmObject *label)
{
	MObjectSetLabelHelp(obj, str, label->obj);
}

inline void CmObject::changePopupHelpText(const char *str)
{
	MObjectChangePopupHelpText(obj, str);
}

inline void CmObject::changeLabelHelpText(const char *str)
{
	MObjectChangeLabelHelpText(obj, str);
}

inline void CmObject::getText(char *text)
{
	MObjectGetText(obj, text);
}

inline int CmObject::getWidth(void)
{
	return MObjectGetWidth(obj);
}

inline int CmObject::getHeight(void)
{
	return MObjectGetHeight(obj);
}

inline int CmObject::getCompositeWidth(void)
{
	return MObjectGetCompositeWidth(obj);
}

inline int CmObject::getCompositeHeight(void)
{
	return MObjectGetCompositeHeight(obj);
}

inline int CmObject::getX(void)
{
	return MObjectGetX(obj);
}

inline int CmObject::getY(void)
{
	return MObjectGetY(obj);
}

inline MTFont CmObject::getFont(void)
{
	return MObjectGetFont(obj);
}

inline void CmObject::resize(int w, int h)
{
	MObjectResize(obj, w, h);
}

inline void CmObject::redraw(void)
{
	MObjectRedraw(obj);
}

inline void CmObject::select(void)
{
	MSelectObject(obj);
}

inline void CmObject::map(void)
{
	MObjectMap(obj);
}

inline void CmObject::unmap(void)
{
	MObjectUnmap(obj);
}

inline int CmObject::isSensitive(void)
{
	return MObjectGetSensitivity(obj);
}

inline void CmObject::enableOptions(const char *name, CARD32 mask)
{
	MObjectEnableOptions(obj, name, mask);
}

inline void CmObject::saveOptions(void)
{
	MObjectSaveOptions(obj);
}

inline void CmObject::unlink(int resize)
{
	MDeleteObject(obj, resize);
}

inline void CmObject::setAttachment(int top, int bottom, int left, int right)
{
	MObjectSetAttachment(obj, top, bottom, left, right);
}

inline void CmObject::setOffset(int t_off, int b_off, int l_off, int r_off)
{
	MObjectSetOffset(obj, t_off, b_off, l_off, r_off);
}

inline void CmObject::setSize(int width, int height)
{
	MObjectSetSize(obj, width, height);
}

inline void CmObject::setWidth(int width)
{
	MObjectSetWidth(obj, width);
}

inline void CmObject::setHeight(int height)
{
	MObjectSetHeight(obj, height);
}

inline void CmObject::setCompositeSize(int width, int height)
{
	MObjectSetCompositeSize(obj, width, height);
}

inline void CmObject::setCompositeWidth(int width)
{
	MObjectSetCompositeWidth(obj, width);
}

inline void CmObject::setCompositeHeight(int height)
{
	MObjectSetCompositeHeight(obj, height);
}

inline void CmObject::setPosition(int x, int y)
{
	MObjectSetPosition(obj, x, y);
}

inline void CmObject::setX(int x)
{
	MObjectSetX(obj, x);
}

inline void CmObject::setY(int y)
{
	MObjectSetY(obj, y);
}

inline void CmObject::focusLink(CmObject *obj2)
{
	MFormFocusLink(obj, obj2->obj);
}

inline void CmObject::topAttach(int att_type, int off, CmObject *p)
{
	MTopAttach(obj, att_type, off, p ? p->obj : NULL);
}

inline void CmObject::bottomAttach(int att_type, int off, CmObject *p)
{
	MBottomAttach(obj, att_type, off, p ? p->obj : NULL);
}

inline void CmObject::leftAttach(int att_type, int off, CmObject *p)
{
	MLeftAttach(obj, att_type, off, p ? p->obj : NULL);
}

inline void CmObject::rightAttach(int att_type, int off, CmObject *p)
{
	MRightAttach(obj, att_type, off, p ? p->obj : NULL);
}

inline CmImage *CmObject::getBackgroundImage(void)
{
	return (CmImage *)MObjectGetBackgroundImage(obj);
}

#ifdef WIN_MGUI
inline void CmObject::print(DOC_ID did, MPRINT_OPTIONS *po)
{
	MPrintObject(obj, did, po);
}
#endif

inline void CmImage::install(void)
{
	MInstallImage(this);
}

inline void CmImage::uninstall(void)
{
	MUninstallImage(this);
}

inline void CmImage::getPixelRGB(int x, int y, int *pr, int *pg, int *pb)
{
	MImageGetPixelRGB(this, x, y, pr, pg, pb);
}

inline void CmImage::setPixelRGB(int x, int y, int r, int g, int b)
{
	MImageSetPixelRGB(this, x, y, r, g, b);
}

inline int  CmImage::getPixelIndex(int x, int y)
{
	return MImageGetPixelIndex(this, x, y);
}

inline void CmImage::setPixelIndex(int x, int y, int i)
{
	MImageSetPixelIndex(this, x, y, i);
}

inline void CmImage::getCMapEntry(int i, int *pr, int *pg, int *pb)
{
	MImageGetCMapEntry(this, i, pr, pg, pb);
}

inline void CmImage::setCMapEntry(int i, int r, int g, int b)
{
	MImageSetCMapEntry(this, i, r, g, b);
}

inline void CmImage::merge(CmImage *src, int xo, int yo, int transparent)
{
	MImageMerge(this, src, xo, yo, transparent);
}

inline void CmImage::setMask(CmImage *msk)
{
	mask = msk;
}

inline void CmImage::computeBackgroundRGB(int *pr, int *pg, int *pb)
{
	MImageComputeBackgroundRGB(this, pr, pg, pb);
}

/*
 * Inline CmShell functions
 */
inline Window CmShell::getWindow(void)
{
	return MShellGetWindow(obj);
}

inline void CmShell::realize(void)
{
	MShellRealize(obj);
}

inline void CmShell::realize(int x, int y)
{
	MShellRealizeXY(obj, x, y);
}

inline void CmShell::realize(int x, int y, int w, int h)
{
	MShellRealizeXYWH(obj, x, y, w, h);
}

inline void CmShell::unrealize(void)
{
	MShellUnrealize(obj);
}

inline void CmShell::raise(void)
{
	MShellRaise(obj);
}

inline int	CmContainer::getNChildren(void)
{
	return MFormGetNChildren(obj);
}
/*
 *  Inline CmForm functions
 */


inline void CmForm::setSpacing(int offset)
{
	MFormSetSpacing(obj, offset);
}

inline void CmForm::setFocusWrap(int true_false)
{
	MFormSetFocusWrap(obj, true_false);
}

inline void CmForm::setFocusHighlight(int true_false)
{
	MFormSetFocusHighlight(obj, true_false);
}

inline void CmForm::selectFirstChild(void)
{
	MSelectFirstChild(obj);
}

inline void CmForm::selectLastChild(void)
{
	MSelectLastChild(obj);
}

inline void CmAbsScrolledForm::setIncr(int h_inc, int v_inc)
{
	MScrolledFormSetIncr(obj, h_inc, v_inc);
}

inline void CmAbsCForm::setClipX(int clip_x)
{
	MFormSetClipX(obj, clip_x);
}

inline void CmAbsCForm::setClipY(int clip_y)
{
	MFormSetClipY(obj, clip_y);
}

inline int CmAbsCForm::getClipX(void)
{
	return MFormGetClipX(obj);
}

inline int CmAbsCForm::getClipY(void)
{
	return MFormGetClipY(obj);
}

inline void CmAbsPagedForm::setActivePage(int page)
{
	MPagedFormSetActivePage(obj, page);
}

inline void CmAbsPagedForm::disablePageKeys(void)
{
	MPagedFormDisablePageKeys(obj);
}

inline void CmAbsPagedForm::enablePageKeys(void)
{
	MPagedFormEnablePageKeys(obj);
}

inline void CmAbsTableForm::showGrid(void)
{
    MTableFormShowGrid(obj);
}

inline void CmAbsTableForm::hideGrid(void)
{
    MTableFormHideGrid(obj);
}

inline void CmAbsTableForm::setColFont(int col, MTFont f)
{
    MTableFormSetColFont(obj, col, f);
}

inline void CmAbsTableForm::setRowHeight(int row, int height)
{
    MTableFormSetRowHeight(obj, row, height);
}

inline void CmAbsTableForm::setColWidth(int col, int width)
{
    MTableFormSetColWidth(obj, col, width);
}

inline void CmAbsTableForm::setColAlign(int col, int al)
{
    MTableFormSetColAlign(obj, col, al);
}

inline void CmAbsTableForm::setColBackgroundRGB(int col, int r, int g, int b)
{
    MTableFormSetColBackgroundRGB(obj, col, r, g, b);
}

inline void CmAbsTableForm::setColForegroundRGB(int col, int r, int g, int b)
{
    MTableFormSetColForegroundRGB(obj, col, r, g, b);
}

inline void CmAbsTableForm::setCellText(int x, int y, const char *text)
{
    MTableFormSetCellText(obj, x, y, text);
}

inline void CmAbsTableForm::getCellText(int x, int y, char *text)
{
    MTableFormGetCellText(obj, x, y, text);
}

inline void CmAbsTableForm::setCellFont(int x, int y, MTFont f)
{
    MTableFormSetCellFont(obj, x, y, f);
}

inline void CmAbsTableForm::setCellAlign(int x, int y, int h_al, int v_al)
{
    MTableFormSetCellAlign(obj, x, y, h_al, v_al);
}

inline void CmAbsTableForm::setCellBackgroundRGB(int x, int y, int r, int g, int b)
{
    MTableFormSetCellBackgroundRGB(obj, x, y, r, g, b);
}

inline void CmAbsTableForm::setCellForegroundRGB(int x, int y, int r, int g, int b)
{
    MTableFormSetCellForegroundRGB(obj, x, y, r, g, b);
}

inline void CmAbsTableForm::deleteRow(int row)
{
    MTableFormDeleteRow(obj, row);
}

inline void CmAbsTableForm::deleteDataRows(void)
{
    MTableFormDeleteDataRows(obj);
}

inline int CmEditList::addItem(char **argv, void *u_data)
{
	return MEditListAddItem(obj, argv, u_data);
}

inline int CmEditList::addItemFromEdits(void *u_data)
{
	return MEditListAddItemFromEdits(obj, u_data);
}

inline void CmEditList::changeElement(long n, char **new_ele, void *data)
{
	MEditListChangeElement(obj, n, new_ele, data);
}

inline void CmEditList::delElement(long n)
{
	MEditListDelElement(obj, n);
}

inline void CmEditList::seekItem(char **argv, long *pos, void **u_data)
{
	MEditListSeekItem(obj, argv, pos, u_data);
}

inline void CmEditList::getItemField(const char *item, int ii, char *buff)
{
	MEditListGetItemField(obj, item, ii, buff);
}

inline void CmEditList::getEditText(int ii, char *buff)
{
	MEditListGetEditText(obj, ii, buff);
}

inline void CmEditList::setEditText(int ii, const char *buff)
{
	MEditListSetEditText(obj, ii, buff);
}

inline CmImage *CmAbsButton::changeNormalImage(CmImage *new_normal)
{
	return (CmImage *)MDrawnButtonChangeNormalImage(obj, new_normal);
}

inline CmImage *CmAbsButton::changeArmedImage(CmImage *new_armed)
{
	return (CmImage *)MDrawnButtonChangeArmedImage(obj, new_armed);
}

inline CmImage *CmAbsButton::changeUnsensitiveImage(CmImage *new_unsens)
{
	return (CmImage *)MDrawnButtonChangeUnsensitiveImage(obj, new_unsens);
}

inline CmImage *CmAbsButton::changeNormalOnImage(CmImage *new_normal)
{
	return (CmImage *)MDrawnButtonChangeNormalOnImage(obj, new_normal);
}

inline CmImage *CmAbsButton::changeArmedOnImage(CmImage *new_armed)
{
	return (CmImage *)MDrawnButtonChangeArmedOnImage(obj, new_armed);
}

inline CmImage *CmAbsButton::changeUnsensitiveOnImage(CmImage *new_unsens)
{
	return (CmImage *)MDrawnButtonChangeUnsensitiveOnImage(obj, new_unsens);
}

inline void CmAbsButton::setTextVsImageAlignment(CARD8 align)
{
	MButtonSetTextVsImageAlignment(obj, align);
}

inline void CmAbsButton::setOnMouseOverForegroundColor(CmColor *fgc)
{
	MButtonSetOnMouseOverForegroundColor(obj, *fgc);
}

inline void CmAbsButton::setOnMouseOverForegroundRGB(int r, int g, int b)
{
	MButtonSetOnMouseOverForegroundRGB(obj, r, g, b);
}

inline void CmAbsButton::setShadowOnMouseOver(int thick)
{
	MButtonSetShadowOnMouseOver(obj, thick);
}

inline void CmAbsButton::setImages(const char *norm, const char *arm, const char *uns)
{
	MButtonChangeImageFiles(obj, norm, arm, uns, 0, 0, 0);
}

inline void CmAbsButton::changeImageFiles(
			const char *norm, const char *arm, const char *uns,
			const char *norm_on, const char *arm_on, const char *uns_on)
{
	MButtonChangeImageFiles(obj, norm, arm, uns, norm_on, arm_on, uns_on);
}


inline void CmAbsButton::setAccelerator(int key)
{
	MButtonSetAccelerator(obj, key);
}

inline void CmPushButton::setMargin(int wm, int hm)
{
	MPButtonSetMargin(obj, wm, hm);
}

inline void CmPushButton::setHSpacing(int hs)
{
	MPButtonSetHSpacing(obj, hs);
}

inline void CmPushButton::setAlignment(int align)
{
	MPButtonSetAlignment(obj, align);
}

inline void CmRadioButton::arm(void)
{
	MRButtonActivate(obj);
}
inline void CmRadioButton::disarm(void)
{
	MRButtonDeactivate(obj);
}

inline int CmRadioButton::isArmed(void)
{
	return MRButtonGetActivationStatus(obj);
}

inline void CmToggleButton::arm(void)
{
	MTButtonActivate(obj);
}

inline void CmToggleButton::disarm(void)
{
	MTButtonDeactivate(obj);
}

inline int CmToggleButton::isArmed(void)
{
	return MTButtonGetActivationStatus(obj);
}


inline void CmFormattedText::setMargin(int w_m, int h_m)
{
	MFormattedTextSetMargin(obj, w_m, h_m);
}

inline void CmFormattedText::setHSpacing(int h_s)
{
	MFormattedTextSetHSpacing(obj, h_s);
}

inline void CmSFormattedText::setMargin(int w_m, int h_m)
{
	MFormattedTextSetMargin(obj, w_m, h_m);
}

inline void CmSFormattedText::setHSpacing(int h_s)
{
	MFormattedTextSetHSpacing(obj, h_s);
}

inline void CmScrollBar::setMax(long max)
{
	MScrollBarSetMax(obj, max);
}

inline void CmScrollBar::setPos(long pos)
{
	MScrollBarSetPos(obj, pos);
}

inline void CmScrollBar::setN(long n)
{
	MScrollBarSetN(obj, n);
}

inline void CmScrollBar::setIncr(int incr)
{
	MScrollBarSetIncr(obj, incr);
}

inline void CmScrollBar::setValues(long max, long n, long pos)
{
	MScrollBarSetValues(obj, max, n, pos);
}

inline long CmScrollBar::getMax(void)
{
	return MScrollBarGetMax(obj);
}

inline long CmScrollBar::getPos(void)
{
	return MScrollBarGetPos(obj);
}

inline long CmScrollBar::getN(void)
{
	return MScrollBarGetN(obj);
}

inline void CmStatusBar::setMax(int max)
{
	MStatusBarSetMax(obj, max);
}

inline void CmStatusBar::setPos(int pos)
{
	MStatusBarSetPos(obj, pos);
}

inline void CmStatusBar::setSliderColor(CmColor *c)
{
	MStatusBarSetSliderColor(obj, *c);
}

inline void CmStatusBar::setSliderRGB(int r, int g, int b)
{
	MStatusBarSetSliderRGB(obj, r, g, b);
}

inline void CmPixmap::setImage(CmImage *img)
{
	MPixmapSetImage(obj, img);
}

inline void CmPixmap::setImage(const char *fname)
{
	MPixmapSetImageFile(obj, fname);
}

inline void CmAbsEdit::changeText(int off, int len, const char *new_t, int chg_len)
{
	MEditChangeText(obj, off, len, new_t, chg_len);
}

inline void CmAbsEdit::getSubString(int start, int end, char *str)
{
	MEditGetSubString(obj, start, end, str);
}

inline int	 CmAbsEdit::findSubStringForward(int off, const char *str, int icase)
{
	return MEditFindSubStringForward(obj, off, str, icase);
}

inline int	 CmAbsEdit::findSubStringBackward(int off, const char *str, int icase)
{
	return MEditFindSubStringBackward(obj, off, str, icase);
}

inline void CmAbsEdit::setSelection(int start, int end)
{
	MEditSetSelection(obj, start, end);
}

inline void CmAbsEdit::getSelection(int *pstart, int *pend)
{
	MEditGetSelection(obj, pstart, pend);
}

inline char *CmAbsEdit::getAllocatedText(void)
{
	return MObjectGetText(obj, 0);
}

inline void CmAbsEdit::setCursorPos(int pos)
{
	MEditSetCursorPos(obj, pos);
}

inline int	 CmAbsEdit::getCursorPos(void)
{
	return MEditGetCursorPos(obj);
}

inline void CmAbsEditField::setFilter(int filter, const char *extra_chars)
{
	MEditSetFilter(obj, filter, extra_chars);
}

inline void CmAbsEditField::setTextExt(const char *text, int validated, int notify)
{
	MEditSetTextExt(obj, text, validated, notify);
}

inline void CmAbsEditField::setValidatedColors(CmColor *bgc, CmColor *fgc)
{
	MEditSetValidatedColors(obj, *bgc, *fgc);
}

inline void CmAbsEditField::setInvalidatedColors(CmColor *bgc, CmColor *fgc)
{
	MEditSetNotValidatedColors(obj, *bgc, *fgc);
}

inline void CmAbsEditField::setValidationStatus(int validated)
{
	MEditSetValidationStatus(obj, validated);
}

inline void CmAbsEditField::selectForReplace(void)
{
	MEditSetReplace(obj);
}

inline const char *CmAbsEditField::text(void)
{
	return MEditValue(obj);
}

inline int  CmAbsEditField::getFilter(void)
{
	return MEditGetFilter(obj);
}

inline int  CmAbsEditField::isValidated(void)
{
	return MIsEditValidated(obj);
}

inline void CmAbsEditWindow::setCursorXY(int x, int y)
{
    MEditSetCursorXY(obj, x, y);
}

inline void CmAbsEditWindow::getCursorXY(int *px, int *py)
{
    MEditGetCursorXY(obj, px, py);
}

inline void CmAbsEditWindow::setAutoIndent(int on_off)
{
    MEditSetAutoIndent(obj, on_off);
}

inline void CmAbsEditWindow::setTabSize(int size)
{
    MEditSetTabSize(obj, size);
}

inline void CmAbsEditWindow::setClipWindow(int clip_w, int clip_h)
{
    MEditSetClipWindow(obj, clip_w, clip_h);
}

inline void CmLabel::setMargin(int wm, int hm)
{
	MLabelSetMargin(obj, wm, hm);
}

inline void CmLabel::setHSpacing(int hs)
{
	MLabelSetHSpacing(obj, hs);
}

inline void CmLabel::setAlignment(int align)
{
	MLabelSetAlignment(obj, align);
}

inline void CmMenuBar::setAcceleratorKey(int key)
{
	MMenuBarSetAcc(obj, key);
}

inline void CmAbsMenuItem::setText(const char *text)
{
	MMenuItemSetText(id, text);
}

inline void CmAbsMenuItem::setSensitive(void)
{
	MMenuItemSetSensitive(id);
}

inline void CmAbsMenuItem::setUnsensitive(void)
{
	MMenuItemSetUnsensitive(id);
}

inline void CmAbsMenuItem::setAcceleratorKey(int key, const char *a_text)
{
	MMenuItemSetAccel(id, key, a_text);
}

inline int  CmAbsMenuItem::setCheckStatus(int true_false)
{
	return MMenuItemSetCheckStatus(id, true_false);
}

inline int  CmAbsMenuItem::getCheckStatus(void)
{
	return MMenuItemGetCheckStatus(id);
}

inline const char *CmAbsMenuItem::getText(void)
{
	return MMenuItemGetText(id);
}

inline int  CmAbsMenuItem::toggleCheckStatus(void)
{
	return MMenuItemToggleCheckStatus(id);
}

inline CmMenu *CmAbsMenuItem::getMenu(void)
{
	return (CmMenu *)MObjectGetUserData(pdobj);
}

inline void CmMenu::setText(const char *text)
{
	MMenuSetTitleText(obj, text);
}

inline const char *CmMenu::getText(void)
{
	return MMenuGetTitleText(obj);
}

inline void CmMenu::popup(int x, int y)
{
	MPopupMenu(obj, x, y);
}

inline void CmSFile::setFile(const char *fname)
{
	MSFileSetFile(obj, fname);
}

inline void CmSFile::setHMargin(int hm)
{
	MSBLSetHMargin(obj, hm);
}

inline void CmSText::setText(const char *text)
{
	MSTextSetString(obj, text);
}

inline void CmAbsList::setSortCallback(STRCMP_CB cb)
{
	MSListSetSortCallback(obj, cb);
}

inline void CmAbsList::setHead(const char *text)
{
	MSBLSetHead(obj, text);
}

inline void CmAbsList::getHead(char *buff)
{
	MSBLGetHead(obj, buff);
}

inline void CmAbsList::setHeadColor(CmColor *bgc, CmColor *fgc)
{
	MSBLHeadSetColor(obj, *bgc, *fgc);
}

inline void CmAbsList::setHMargin(int hm)
{
	MSBLSetHMargin(obj, hm);
}

inline void CmAbsList::setItemCheckStatus(const char *item, int status)
{
	MSListSetItemCheckStatus(obj, item, status);
}

inline LIST_ACT *CmAbsList::getFirstItem(void)
{
	return MSListGetFirstItem(obj);
}

inline LIST_ACT *CmAbsList::getFirstSelectedItem(void)
{
	return MSListGetFirstSelectedItem(obj);
}

inline LIST_ACT *CmAbsList::getNextItem(void)
{
	return MSListGetNextItem(obj);
}

inline LIST_ACT *CmAbsList::getNextSelectedItem(void)
{
	return MSListGetNextSelectedItem(obj);
}

inline LIST_ACT *CmAbsList::getFirstUnselectedItem(void)
{
	return MSListGetFirstUnselectedItem(obj);
}

inline LIST_ACT *CmAbsList::getNextUnselectedItem(void)
{
	return MSListGetNextUnselectedItem(obj);
}

inline long CmAbsList::getNItem(void)
{
	return MSListGetNItem(obj);
}

inline LIST_ACT *CmAbsList::getItem(const char *item)
{
	return MSListGetItem(obj, item);
}

inline LIST_ACT *CmAbsList::getElement(long n)
{ 
	return MSListGetElement(obj, n);
}

inline LIST_ACT *CmAbsList::getSelected(void)
{
	return MSListGetSelected(obj);
}

inline int CmAbsList::getItemCheckStatus(const char *item)
{
	return MSListGetItemCheckStatus(obj, item);
}

inline void CmAbsList::selectItem(const char *item)
{
	MSListSelectItem(obj, item);
}

inline void CmAbsList::deselectItem(const char *item)
{
	MSListDeselectItem(obj, item);
}

inline void CmAbsList::selectElement(long n)
{
	MSListSelectElement(obj, n);
}

inline void CmAbsList::deselectElement(long n)
{
	MSListDeselectElement(obj, n);
}

inline void CmAbsList::freeze(void)
{
	MSListFreeze(obj);
}

inline void CmAbsList::thaw(void)
{
	MSListThaw(obj);
}

inline void CmAbsList::reset(void)
{
	MSListReset(obj);
}

inline void CmAbsList::clear(void)
{
	MSListClear(obj);
}

inline int CmAbsList::addItem(const char *item, void *data)
{
	return MSListAddItem(obj, item, data);
}

inline int CmAbsList::delItem(const char *item)
{
	return MSListDelItem(obj, item);
}

inline int CmAbsList::addElement(long n, const char *item, void *data)
{
	return MSListAddElement(obj, n, item, data);
}

inline int CmAbsList::delElement(long n)
{
	return MSListDelElement(obj, n);
}

inline void CmAbsList::changeItem(const char *old, const char *new_item, void *u_data)
{
	MSListChangeItem(obj, old, new_item, u_data);
}

inline void CmAbsList::changeElement(long n, const char *new_item, void *u_data)
{
	MSListChangeElement(obj, n, new_item, u_data);
}

inline void CmAbsList::enableMultiSelection(void)
{
	MSListEnableMultiSel(obj);
}


inline int CmMGUI::getKeyboardState(void)
{
	return MGetKeyboardState();
}

inline const char *CmMGUI::getSysDir(void)
{
	return MGetSysDir();
}

inline int CmMGUI::paletteMode(void)
{
	return MPaletteMode();
}

inline int CmMGUI::getScreenWidth(void)
{
	return MGetScreenWidth();
}

inline int CmMGUI::getScreenHeight(void)
{
	return MGetScreenHeight();
}

inline int CmMGUI::getNPlanes(void)
{
	return MGetNPlanes();
}

inline long CmMGUI::getNColors(void)
{
	return MGetNColors();
}

inline MTColor CmMGUI::getBlackColor(void)
{
	return MBlackColor();
}

inline MTColor CmMGUI::getWhiteColor(void)
{
	return MWhiteColor();
}

inline void CmMGUI::enableCustomizing(void)
{
	MEnableCustomizing();
}

inline char *CmMGUI::getStringOption(const char *obj_name, const char *opt_name)
{
	return MGetStringOption(obj_name, opt_name);
}

inline char *CmMGUI::getTextOption(const char *obj_name)
{
	return MGetTextOption(obj_name);
}

inline int CmMGUI::getIntOption(const char *obj_name, const char *opt_name, int *ret)
{
	return MGetIntOption(obj_name, opt_name, ret);
}

inline int CmMGUI::getFontOption(const char *obj_name, MTFont *ret)
{
	return MGetFontOption(obj_name, ret);
}

inline int CmMGUI::getBGColorOption(const char *obj_name, int *pr, int *pg, int *pb)
{
	return MGetBGColorOption(obj_name, pr, pg, pb);
}

inline int CmMGUI::getFGColorOption(const char *obj_name, int *pr, int *pg, int *pb)
{
	return MGetFGColorOption(obj_name, pr, pg, pb);
}

inline int CmMGUI::getShadowOption(const char *obj_name, int *shadow, int *i_t, int *o_t)
{
	return MGetShadowOption(obj_name, shadow, i_t, o_t);
}

inline void CmMGUI::setTextOption(const char *obj_name, const char *text)
{
	MSetTextOption(obj_name, text);
}

inline void CmMGUI::setIntOption(const char *obj_name, const char *rname, int val)
{
	MSetIntOption(obj_name, rname, val);
}

inline void CmMGUI::setFGColorOption(const char *obj_name, int r, int g, int b)
{
	MSetFGColorOption(obj_name, r, g, b);
}

inline void CmMGUI::setBGColorOption(const char *obj_name, int r, int g, int b)
{
	MSetBGColorOption(obj_name, r, g, b);
}

inline void CmMGUI::setShadowOption(const char *obj_name, int shadow, int in_t, int out_t)
{
	MSetShadowOption(obj_name, shadow, in_t, out_t);
}

inline void CmMGUI::setFontOption(const char *obj_name, MTFont font)
{
	MSetFontOption(obj_name, font);
}

inline void CmMGUI::saveOptions(void)
{
	MSaveOptions();
}

inline void CmMGUI::logInit(int level, int nr, int nvr, int nc, int nvc, const char *fname)
{
	MLogInit(level, nr, nvr, nc, nvc, fname);
}

inline void CmMGUI::logSetLevel(int level)
{
	MLogSetLevel(level);
}

inline void CmMGUI::logSetMask(CARD32 a)
{
	MLogSetMask(a);
}

inline CARD32 CmMGUI::logGetMask(void)
{
	return MLogGetMask();
}

inline void CmMGUI::logSetMaskBits(CARD32 a)
{
	MLogSetMaskBits(a);
}

inline void CmMGUI::logResetMaskBits(CARD32 a)
{
	MLogResetMaskBits(a);
}

inline void CmMGUI::setTraceFileName(const char *fname)
{
	MSetTraceFileName(fname);
}

inline int CmMGUI::spoolFile(const char *fname)
{
	return MSpoolFile(fname);
}

inline int CmMGUI::spoolTempFile(const char *fname)
{
	return MSpoolTempFile(fname);
}

inline void CmMGUI::getDefaultBackgroundRGB(int *pr, int *pg, int *pb)
{
	MGetDefaultBackgroundRGB(pr, pg, pb);
}

inline void CmMGUI::setDefaultBackgroundRGB(int r, int g, int b)
{
	MSetDefaultBackgroundRGB(r, g, b);
}

inline void CmMGUI::getDefaultForegroundRGB(int *pr, int *pg, int *pb)
{
	MGetDefaultForegroundRGB(pr, pg, pb);
}

inline void CmMGUI::setDefaultForegroundRGB(int r, int g, int b)
{
	MSetDefaultForegroundRGB(r, g, b);
}

inline int CmMGUI::textCharX(const char *text, MTFont font, int x)
{
	return MTextCharX(text, font, x);
}

inline int CmMGUI::textHeight(const char *text, MTFont font)
{
	return MTextHeight(text, font);
}

inline int CmMGUI::charHeight(char car, MTFont font)
{
	return MCharHeight(car, font);
}

inline int CmMGUI::textWidth(const char *text, MTFont font)
{
	return MTextWidth(text, font);
}

inline int CmMGUI::textNWidth(const char *text, MTFont font, int len)
{
	return MTextNWidth(text, font, len);
}

inline int CmMGUI::charWidth(char car, MTFont font)
{
	return MCharWidth(car, font);
}

inline void CmMGUI::mainLoop(void)
{
	MMainLoop();
}

inline void CmMGUI::loopWhileFlag(int *pflag, int value)
{
	MLoopWhileFlag(pflag, value);
}

inline void CmMGUI::loopWhileEvents(int discard)
{
	MLoopWhileEvents(discard);
}

inline void CmMGUI::nextEvent(MEvent *pe, int wait)
{
	MNextEvent(pe, wait);
}

inline void CmMGUI::processEvent(MEvent *pe)
{
	MProcessEvent(pe);
}

inline int CmMGUI::inputPrompt(const char *t, const char *msg, char *text, int len, const char *ok_btn,
	 const char *canc_btn)
{
	return MInputPrompt(t, msg, text, len, ok_btn, canc_btn);
}

inline char *CmMGUI::textPrompt(const char *t, const char *msg, const char *text, int w, int h, const char *ok,
	 const char *cancel)
{
	return MTextPrompt(t, msg, text, w, h, ok, cancel);
}

inline int CmMGUI::fileSelection(const char *title, const char *filter, char *fname, char *dname,
		 int change_cur_dir)
{
	return MFileSelection(title, filter, fname, dname, change_cur_dir);
}

inline char *CmMGUI::getCurrentDirectory(void)
{
	return MGetCurrentDirectory();
}

inline void CmMGUI::setInputPromptRGB(int r, int g, int b)
{
	MSetInputPromptRGB(r, g, b);
}

inline void CmMGUI::setMessageDialogRGB(int r, int g, int b)
{
	MSetMessageDialogRGB(r, g, b);
}

inline void CmMGUI::setFileSelectionRGB(int r, int g, int b)
{
	MSetFileSelectionRGB(r, g, b);
}

inline void CmMGUI::getInputPromptRGB(int *pr, int *pg, int *pb)
{
	MGetInputPromptRGB(pr, pg, pb);
}

inline void CmMGUI::getMessageDialogRGB(int *pr, int *pg, int *pb)
{
	MGetFileSelectionRGB(pr, pg, pb);
}

inline void CmMGUI::getFileSelectionRGB(int *pr, int *pg, int *pb)
{
	MGetFileSelectionRGB(pr, pg, pb);
}

inline void CmMGUI::setFileSelectionTextStrings(char *file_list, char *dir_list, char *drive_list,
    char *filter, char *filename, char *curr_dir, char *ok_button, char *close_button)
{
	MSetFileSelectionTextStrings(file_list, dir_list, drive_list,
		filter, filename, curr_dir, ok_button, close_button);
}

inline void CmMGUI::trace(const char *fmt,...)
{
    va_list ap;

    va_start(ap, fmt);
    vTrace(fmt, ap);
}

#ifdef OLD_MGUI

typedef CmClippedRowForm CmClippedForm;
typedef CmScrolledRowForm CmScrolledForm;

#endif

#endif
