/*===================================================================
 rootfind.c

 Version 1.2

 Written by:
   Brent Worden
   WordenWare
   email:  brent@worden.org

 Copyright (c) 1999-2001 WordenWare

 Created:  May 09, 1999
 Revised:  March 26, 2001
===================================================================*/

#include "algorthm.h"
#include "numerror.h"
#include "rootfind.h"
#include "utility.h"
#include <math.h>

NUMERICS_EXPORT double bisection(Function func, double x0, double x1, double tol)
{
	double f0 = func(x0);
	double f1 = func(x1);

	if(f0 * f1 > 0.0){
		NUMERICS_ERROR("bisection", "func(x0) and func(x1) must have oppisite signs");
	} else {
		double fmid;
		double xmid;

		while(!areEqual(x0, x1, tol)){
			xmid = average(x0, x1);
			fmid = func(xmid);
			if(fmid * f0 > 0.0){
				f0 = fmid;
				x0 = xmid;
			} else {
				x1 = xmid;
			}
		}
	}
	return average(x0, x1);
}

NUMERICS_EXPORT BOOL bracketOut(Function func, double *x0, double *x1)
{
	double f0, f1;
	int j;
	BOOL found = FALSE;

	f0 = func(*x0);
	f1 = func(*x1);
	for(j = 0; j < NUMERICS_ITMAX && !found; ++j){
		if(f0 * f1 < 0.0){
			found = TRUE;
		} else {
			if(fabs(f0) < fabs(f1)){
				*x0 += 1.6 * (*x0 - *x1);
				f0 = func(*x0);
			} else {
				*x1 += 1.6 * (*x1 - *x0);
				f1 = func(*x1);
			}
		}
	}
	return found;
}

double bisectionWithState(FunctionWithState func, void* state, double x0, double x1, double tol)
{
	double f0 = func(x0, state);
	double f1 = func(x1, state);

	if(f0 * f1 > 0.0){
		NUMERICS_ERROR("bisection", "func(x0) and func(x1) must have oppisite signs");
	} else {
		double fmid;
		double xmid;

		while(!areEqual(x0, x1, tol)){
			xmid = average(x0, x1);
			fmid = func(xmid, state);
			if(fmid * f0 > 0.0){
				f0 = fmid;
				x0 = xmid;
			} else {
				x1 = xmid;
			}
		}
	}
	return average(x0, x1);
}

BOOL bracketOutWithState(FunctionWithState func, void* state, double *x0, double *x1)
{
	double f0, f1;
	int j;
	BOOL found = FALSE;

	f0 = func(*x0, state);
	f1 = func(*x1, state);
	for(j = 0; j < NUMERICS_ITMAX && !found; ++j){
		if(f0 * f1 < 0.0){
			found = TRUE;
		} else {
			if(fabs(f0) < fabs(f1)){
				*x0 += 1.6 * (*x0 - *x1);
				f0 = func(*x0, state);
			} else {
				*x1 += 1.6 * (*x1 - *x0);
				f1 = func(*x1, state);
			}
		}
	}
	return found;
}

NUMERICS_EXPORT double falsePosition(Function func, double x0, double x1, double tol)
{
	double f0 = func(x0);
	double f1 = func(x1);

	if(f0 * f1 > 0.0){
		NUMERICS_ERROR("falsePosition", "func(x0) and func(x1) must have oppisite signs");
	} else {
		double f;
		double x;
		int n = 0;

		do {
			x = x1 - f1 * (x1 - x0) / (f1 - f0);
			f = func(x);
			if(f * f1 < 0.0){
				x0 = x1;
				f0 = f1;
			}
			x1 = x;
			f1 = f;
			++n;
		} while(n < NUMERICS_ITMAX && !areEqual(x0, x1, tol));
		if(n >= NUMERICS_ITMAX){
			NUMERICS_ERROR("falsePosition", "Failed to converge to root.");
		}
	}
	return x1;
}

NUMERICS_EXPORT double newton(Function func, Function der, double x0, double tol)
{
	int n = 0;
	double x1 = x0;

	do {
		x0 = x1;
		x1 = x0 - func(x0) / der(x0);
		++n;
	} while(n < NUMERICS_ITMAX && !areEqual(x0, x1, tol));
	if(n >= NUMERICS_ITMAX){
		NUMERICS_ERROR("newton", "Failed to converge to root.");
	}

	return x1;
}

NUMERICS_EXPORT double secant(Function func, double x0, double x1, double tol)
{
	double f0 = func(x0);
	double f1 = func(x1);
	int n = 0;
	double x;

	do {
		x = x1 - f1 * (x1 - x0) / (f1 - f0);
		x0 = x1;
		f0 = f1;
		x1 = x;
		f1 = func(x);
		++n;
	} while(n < NUMERICS_ITMAX && !areEqual(x0, x1, tol));
	if(n >= NUMERICS_ITMAX){
		NUMERICS_ERROR("secant", "Failed to converge to root.");
	}

	return x1;
}

/*===================================================================
 Revision History

 Version 1.0 - 05/09/1999 - New.
 Version 1.1 - 01/11/2000 - Added bracketOut.
 Version 1.2 - 03/26/2001 - Added bracketOutWithState and
                            bisectionWithState
===================================================================*/
