/* DEMO.C

	Copyright (c) S Morphet, 1994-1995

   Example program for STEPH library.  This program demonstrates some of
   the features of Steph.  Steph has many other features, described in the
   programmers manual.

	Compile with Microsoft C as a large memory model program,
	and link to STEPH_L.LIB.

	EDITOR.C is an example program showing a text editor working
	with Steph.

*/

/* You always need to include steph.h */
/* You may need to change this line to set a path to steph.h */

#include <steph.h>

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <string.h>

/******************************************************************/


/* Functions in DEMO.C */

/* Various demo functions */
void fn1( void );
void fn2( void );
void fn4( void );

/* Function to finish program */
void fnexit( void );

/* Beep */
void beep(void);

/* Functions to test f1 key in dialogue box */
void passf1( void );
void actf1( void );

/* Functions to perform system level key trapping */
void _mysysfilter( void );
void _mysysaction( void );

/* An initialise function */
void myinit( void );

/******************************************************************/


/* the MAIN function */

void main( void )
{
   /* Steph will try to use the current graphics mode */
   /* Here we assume 16 colour text mode is already set. */

   /* Initialise Steph */
   if ( !_steph_initialise( M_TEXT ) )
   {
      perror("Steph couldn't start in the current graphics mode.\n");
      exit(0);
   }

   /* set up menus and hooked functions */

   if( ! _menu_build( 0, "&Test", /* this is the menu title */
         "&Dialogue",             /* a menu item */
         "=",                     /* horizontal bar across menu */
         "&Query",                /* another item */
         "E&xit.",                /* another item */
         NULL ) )                 /* NULL marks the end of the list */
   {
      perror("Unable to build menu 0.\n");
      return;
   }

   _menu_functions( 0,
         (vfnptr)fn1,            /* attach our fn1 */
         NOFN,                   /* can't hook to the horiz bar */
         (vfnptr)fn4,            /* another fn */
         (vfnptr)fnexit );       /* attach our exit function */

   _menu_helptext( 0,
         "Here is text for the bar option",      /* help texts */
         "Pop up a dialogue box to test",
         NOTEXT,
         "Function two doesn't do much",
         "Exit the program" );

         
   /* build another menu */

   if( ! _menu_build( 1, "^&Squeak",   /* the ^ makes this a rhs menu    */
         "$S&quelch",                  /* $ means the item starts grey   */
         "=",
         "?&Toggle Me!",               /* ? means this item is checked   */
         "Tic&kle",
         NULL ) )
   {
      perror("Unable to build menu 1.\n");
      return;
   }

   _menu_functions( 1,                        /* attach the toggle func. */
         NOFN, NOFN, _menu_toggle, NOFN );    /* to the checked item     */

   _menu_helptext( 1,
         "This is the squeak menu",
         "Squelchy",
         NOTEXT,
         "Snuffly",
         "Tickly" );

         
	/* these rest of the menus have no attached functions yet */

   if( ! _menu_build( 2, "&Whschkiput", "&item", NULL ) )
   {
      perror("Unable to build menu 2.\n");
      return;
   }
   _menu_helptext( 2,
         "The noise that a llama makes",
         "The item" );

         
   /* this whole menu is greyed */
   
   if( ! _menu_build( 3, "$&Quack", "Duck&1", "Duck&2", NULL ) )
   {
      perror("Unable to build menu 3.\n");
   }
   _menu_helptext( 3,
         "More ducks, need more attack...",
         "Top duck",
         "Another duck" );

         
   if( ! _menu_build( 4, "T&weet", "ite&m", NULL ) )
   {
      perror("Unable to build menu 4.\n");
      return;
   }
   _menu_helptext( 4,
         "This is the tweet menu",
         "There is only one bird here" );


   /* another on the right hand side */
   
   if( ! _menu_build( 5, "^&Baa", "item&1", "item&2", NULL) )
   {
      perror("Unable to build menu 5.\n");
      return;
   }
   _menu_helptext( 5,
         "This is the special wooly sheep menu",
         "Sheep 1",
         "Sheep 2" );

         
   if( ! _menu_build( 6, "&Nibble", "Don't be &sad!", NULL ) )
   {
      perror("Unable to build menu 6.\n");
      return;
   }
   _menu_helptext( 6,
         "Nibble menu",
         "Ashumbadiddly!" );

         
   /* Set up pointer to our application initialisation function */
   
   _steph_initialfunction( myinit );

   
   /* Set up the screen windows leaving all the hooks free */
   /* i.e. Nothing happens in these windows. */
   
   _window_build( 0, WP_MAIN, 10, "0 The Title", NOFN, NOFN, NOFN );
	_window_build( 1, WP_TOP, 2, "1 Topper than you!", NOFN, NOFN, NOFN );
   _window_build( 2, WP_TOP, 2, "2 Top Window.", NOFN, NOFN, NOFN );
   _window_build( 3, WP_BOT, 3, "3 Bottom Window.", NOFN, NOFN, NOFN );
   _window_build( 4, WP_BOT, 3, "4 Right at the bottom.", NOFN, NOFN, NOFN );

   /* remember, only window zero, the main window is open by default */

   
   /* fix the top two lines of window 2 */

   _window_setfixed( 2, 2 );
   

   /* Set up dialogue boxes */
  
   /* A big dialogue to demo all the controls... */
   /* This is dbox zero, centred on the screen, with ten controls. */
   /* 60 chars wide, 20 chars deep */

   if( !_dbox_build( 0, DB_CENSCR, 60, 20, 10,
         NOTEXT, "Help text for the dialogue" ) )
   {
      perror("Couldn't build dialogue box 0.\n");
      return;
   }
   
   /* Give it some horiz bars, and make dbox1 a dependant */

   _dbox_setbars( 0, 6, _NO_BAR   );
   _dbox_setdependants( 0, 1, _NO_DEPENDANT );


   /* remember - lists, combo lists and text should be in DYNAMIC memory */

   /* set up some buttons and things... */

   /* a non interactive label */
   _dbox_c_label( 0, 0, 2, 2, "This is a label." );

   /* a button, the dialogue closes when it is pressed */
   _dbox_c_button( 0, 1, 2, 4, 9, "&OK", EXIT, NOFN );

   /* a button, closes the dialogue without restoring data */
   _dbox_c_button( 0, 2, 2, 5, 9, "&Cancel", ESCAPE, NOFN );

   /* a button, opens another dialogue via fn2 */
   _dbox_c_button( 0, 3, 2, 8, 9, "&Another", NOEXIT, (vfnptr)fn2 );

   /* a check box, begins checked */
   _dbox_c_check( 0, 4, 2, 10, "Chec&k me!", 1 , NOFN   );

   /* a set of three radio buttons -  long line ->> */
   _dbox_c_radio( 0, 5, _s_buildradiolist("\x1b\x01\x00\x00\x00Text&1\0\x1b\x02\x00\x00\x00T&wo\0\x1b\x03\x00\x00\x01Three&3\0\0"), 0, NOFN   );

   /* an editable text box */
   _dbox_c_text( 0, 6, 0x1b, 5, 12, "&Text box",
      (char*)_s_strdup("TTKFaster4"), NOFN, NOFN );

   /* a vertical list, empty so Steph will ignore it... */
   _dbox_c_list( 0, 7, 0x1b, 8, 20, 4, 6, DBL_VSCROLL, "&Vertical List",
         _s_buildlist(""), EXIT, NOFN, NOFN   );

   /* a horizontal list, not empty -  long line >>>> */
   _dbox_c_list( 0, 8, 2, 13, 20, 2, 6, DBL_HSCROLL, "&Horizontal List",
         _s_buildlist("Atem1\0Btem2\0Ctem3\0Dtem4\0Etem5\0Ftem6\0Atem7\0Htem8\0Item9\0"), NOEXIT, NOFN, NOFN   );
	_dbox[0].control[8].iret = 2;

   /* a combo box, exits dialogue when selection made */
   _dbox_c_combo( 0, 9, 0x1b, 17, 10, 3, 3, "Co&mbo",
         _s_buildlist("Item1\0Item2\0Item3\0Item4\0"), EXIT, NOFN, NOFN );

         
   /* Keystroke filter and action function for dialogue zero */

   _dbox_set_global_keys( 0, passf1, actf1 );

   
   /* Another dialogue.  A dependant of dbox 0. */

   if( !_dbox_build( 1, DB_CENSCR, 30, 12, 3,
         NOTEXT, "Click OK to continue." ) )
   {
      perror("Couldn't build dialogue 1.\n");
      return;
   }
   
   _dbox_c_label( 1, 0, 2, 2, "Dialogue 1." );
   _dbox_c_button( 1, 1, 2, 5, 6, "&OK", EXIT, NOFN );
   _dbox_c_check( 1, 2, 2, 8, "&Data!", 0, NOFN );

   
   /* A third dialogue - maybe you pressed F1 for online help? */

   if( !_dbox_build( 2, DB_CENSCR, 30, 6, 2, "Help", "Joking?" ) )
   {
      perror("Couldn't build dialogue 2.\n");
      return;
   }

	_dbox_c_label( 2, 0, 2, 2, "No Help available!" );
   _dbox_c_button( 2, 1, 2, 4, 6, "&OK", EXIT, NOFN );


   /* Set the speed bar buttons, and attach the beep function to
      the rest of the bar. */

   _speed_button_set( 0, "ALT=Menu", TRUE, _menu_go );
   _speed_button_set( 1, "One", TRUE, NOFN );
   _speed_button_set( 2, "Another", TRUE, NOFN );
   
   _speed_clickfn( (vfnptr)beep );


   /* Set up filter and action functions for system level keytraps */
   
   _window_setsyskey( (vfnptr)_mysysfilter, (vfnptr)_mysysaction );

   
   /* Ask for one line of tool bar space */

   _toolbar_reserve( 1 );


   /* Set up a button, a toggle and a combo on the tool bar */
   
   _toolbar_set_button( 0, 0, 2, "<Button>", NOFN );
   _toolbar_set_toggle( 1, 0, 12, "<Toggle>", ON, NOFN, NOFN );

	/* this is a long line >>>> */
	_toolbar_set_combo( 2, 0, 22, 10, 8, "Combo", _s_buildlist("One\0Two\0Three\0Four\0Five\0Six\0Seven\0Eight\0Nine\0Ten\0Eleven\0"), 0, NOFN );


	/* give control to steph */

	_steph_go();

   /* if execution reaches this point, it is because no menus or windows
      had been defined, and Steph refused to start. */
      
   /* The program usually ends when fnexit() (below) is called from
      the menu */
}


/******************************************************************/

/* a function to make a beep */

void beep(void) { printf("\a"); }

/******************************************************************/

/* open dialogue box zero */

void fn1( void ) { _dbox_go( 0 ); }

/******************************************************************/

/* open dialogue box one */

void fn2( void ) { _dbox_go( 1 ); }

/******************************************************************/

/* open a query box, and report on the result with a flash box */

void fn4( void )
{
  char *text;

	/* build and open the query using dialogue 3 as a store */

	switch (_query_box( DB_CENSCR,
			_s_buildlist("Some text\0Some more text\0"),
			_s_buildlist( "Button1\0But2\0" ), "Helptext" ) )
	{
		case QB_NONE:
		{
			text =_s_buildlist("No button was pressed\0Press a key.\0");
			break;
		}
		case 0:
		{
			text = _s_buildlist("The first button was pressed\0Press a key.\0");
			break;
		}
		case 1:
		{
			text = _s_buildlist("The second button was pressed\0Press a key.\0");
			break;
		}

	}

	/* display the result in a flash box */

	_flash_box( DB_CENSCR, text );

	/* wait for a key, then remove it from the buffer */
	_s_wait_for_key();

	_flash_remove();
}


/******************************************************************/

/* an exit function like this is the only way out of Steph. */

void fnexit( void )
{
   _steph_closedown();
   exit(0);
}

/******************************************************************/

/* the initialise function, runs just after steph_go is called */

void myinit( void )
{
   /* open the other windows that we built */

   _window_open( 1, OPEN );
   _window_open( 2, OPEN );
   _window_open( 3, OPEN );
   _window_open( 4, OPEN );

   /* make 2 the active window */
   _window_make_active( 2 );

   /* put some text into window 2, make the window bigger to see it all */

   _wb_newline( 2 );
   _wb_settext( 2, _s_strdup("Wombat.") );

   _wb_newline( 2 );
   _wb_settext( 2, _s_strdup("Mongoose.") );

   _wb_newline( 2 );
   _wb_settext( 2, _s_strdup("Aardvark.") );

   _wb_newline( 2 );
   _wb_settext( 2, _s_strdup("Gerbil.") );

   _wb_newline( 2 );
   _wb_settext( 2, _s_strdup("Cat.") );

   _wb_newline( 2 );
   _wb_settext( 2, _s_strdup("Pigeon.") );

   _wb_newline( 2 );
   _wb_settext( 2, _s_strdup("Other animals.") );

   /* insert a blank line after line 2 (Aardvark) */

   _wb_gotoline( 2, 2 );
   _wb_newline(2);

   /* update the windows after having opened them etc. */

   _window_draw();
   _window_refresh_all();

}


/******************************************************************/

/* filter for system keystrokes - pass only F1:    00,59 dec   00,3b hex */

void _mysysfilter( void )
{
   /* reset the 'match found' flag */
   _window_spec.syskeyflag = FALSE;

   /* set the flag if the key is F1 */
   if( _window_spec.key1 == 0 )
      if( _window_spec.key2 == 59 )
         _window_spec.syskeyflag = TRUE;
}


/* open dialogue 2 if F1 is pressed */
/* note system keystrokes are inhibited when dialogues are open */

void _mysysaction( void )
{
   switch( _window_spec.key1 )
   {
      case 0:
         switch( _window_spec.key2 )
         {
            case 59:
            {
               _dbox_go( 2 );
            }

            default:
            {}
         }

      default:
      {}
   }
}

/******************************************************************/

/* dialogue keystroke filter for dialogue 0 */
/* pass the F1 key only - c.f. _mysysfilter above */

void passf1( void )
{
   /* pass only f1 for dbx global keys */

   if( ( _dbox_spec.filter_key != 0 ) || ( _dbox_spec.filter_key2 != 59) )
   {
      _dbox_spec.filter_key = 0;
      _dbox_spec.filter_key2 = 0;
   }
}


/* open dialogue 1 if F1 pressed in dialogue */
/* note system keystrokes are inhibited when dialogues are open */

void actf1( void )
{
   /* call a dbox if you press f1 in this dbox */
   _dbox_go( 1 );
}
 
/******************************************************************/

/* end of DEMO.C */

/* there are a number of other functions... */
/* read the manual file STEPH.DOC, and see EDITOR.C for more information */
