(* eas.pas            Easter-date calculations    (terry stancliffe 1999
                                                   email: tpat@cwcom.net)
   usage:   eas yyyy
   input:   - a year number (AD)
   outputs: - dates of Easter Sunday(s), New Style and Old Style
            - current calendar difference, NS-OS (days)

   procedures  easg()  and  easj()

   (compile and run ok with TurboPascal ver 3)
*)

(* These Easter algorithms use checkable quantities (Golden Number,
Epact, Dominical Letter, paschal full moon date, etc), for direct
comparison with traditional Easter tables -- both for the Gregorian
(New Style) calendar and its Easter cycle, and for the Julian (Old
Style) calendar and its corresponding Easter cycle. *)

(* ------------------------------------------------------------- *)

(* procedure easg

   Easter calculation for Gregorian (NS) calendar --
   checks ok with a variety of tabular data from ESAE 1961 etc. *)

procedure easg(yr:integer; var mm,dd, es: integer);

   (* input:    yr -- year number AD
      outputs:  mm -- 3 or 4 (for March or April Easter)
                dd -- day of month of Easter Sunday
                es -- Easter Sunday in days forward from 0 March *)

var   gn,              (* Golden Number *)
      ci, kk, adj,     (* Gregorian centurial adjustment of epact *)
      ep,              (* Epact *)
      pfm,
      ymod,
      dl:              (* Dominical Letter *)
           integer;

begin

(* --> Golden number *)                     (* gn in [1..19] *)

(* Year 0 (1 BC) is given GN =1: year AD 18 had GN =19; all other years
repeat the same sequence. *)

if yr>=0 then   gn:=  1 + yr - 19*(yr div 19)
         else   gn:=  1 + yr - 19*((yr+1) div 19 ) +19;


(* --> Gregorian centurial adjustment of epact *)

(* See rules in notes below and in eas.doc *)

if yr>=0 then   ci:= yr div 100
         else   ci:= ((yr+1) div 100 ) -1;  (* ci 19 is for 19xx, etc *)

kk:= ((ci+8) div 25) - 1;

adj:=  (ci + 5 - ((ci+8) div 4) - ((ci+9-kk) div 3) );

     (* adj +7 is for 15xx-16xx; adj +8 is for 17xx-18xx; adj +9 is for
        19xx-21xx, etc (with a +25-cy repeat-period for the increments,
        and eventually to be reduced modulo 30), these values are to be
        SUBTRACTed from the Julian epact value (not explicitly
        calculated here, but obtainable by putting adj=0 in the formula
        below), but adj is to be ADDed to the 19gn term in the epact
        formula below.

        Comparison with traditional sources:  adj here is the row-
        difference (in the expanded epact table reproduced, in Coyne
        et al 1983 pp 212-3, from Clavius 1612 pp 110-111, see also
        file eas.doc) between the top row of epacts (Julian)
        (adj=0) and the row for the required century (adj= number
        of rows below top row). In the Calendar Act 1751 and the
        Book of Common Prayer, the adjustments are given relative
        to the values for 15xx-16xx, (which were in the 7th row
        below the Julian row in the Clavius table) setting that as
        zero, i.e. an offset of 7.  Thus, in the BCP and Calendar
        Act, 19xx-21xx has an adjustment of 2, -- here 2+7 =9, and
        so on.   *)


(* --> Epact -- including centurial adjustment *)   (* ep in [0..29] *)

(* Tabular age of moon on 1 January in given year, minus 1 day. *)

ep:=  29 - ( 19*gn + adj + 2 ) mod 30 ;


(* --> paschal (tabular) full moon date *)          (* pfm in [21..49] *)

(* Full moon is defined as 14th day counting tabular new moon as
   day 1, and paschal f.m. is first full-moon date falling on or
   after 21 March. *)

pfm:= 21 + ((53-ep) mod 30);
   if ( (ep=24) or ((ep=25) and (gn in [12..19])) ) then pfm:= pfm-1;

                      (* pfm is expressed as an offset from 0 March *)
                      (* pfm includes effect of special rules for
                         epact 24 and second epact 25 *)


(* --> Dominical (Sunday) letter *)

(* Letters A-G are written in repeating sequence against all days of the
calendar year (except 29 Feb in a leap year) starting with A on 1 January:
the Sunday letter used in the Easter calculation is then the letter that
falls against all the Sundays in March and April. In the Gregorian calendar
the DL has a 400-year cycle. *)

if yr>=0  then   ymod:=  yr - 400* (yr div 400)
          else   ymod:=  yr - 400*((yr+1) div 400) + 400;
dl:=   7 - (ymod + (ymod div 4) - (ymod div 100) + 6) mod 7;

                                            (* dl in 1..7 for A..G *)


(* --> Easter date *)

es:= 4 + dl + ( ((pfm-4-dl) div 7) +1)*7;
                (* es in [22..56], meaning 22-March through 25-April *)

mm:= 3 + (es-1) div 31;   dd:= 1 + (es-1) mod 31;
                                          (* sorts out month and day *)


(* for debugging or data-checking, unbracket next lines to print
   intermediates:
writeln('y:',yr:5,'  gn:',gn:2,'  adj:',adj:2,
        '  ep:',ep:2,'  dl: ',chr(dl+64),'  pfm: ',pfm:2,
        '   es:',es:2,'  m:',mm:1,'  d:',dd:2);
*)

end; (* of proc easg *)

(* =================================================================== *)

(* procedure easj

   Easter calculation for Julian (OS) calendar --
   checks ok with a variety of tabular data from ESAE 1961 etc. *)

procedure easj(yr:integer; var mm,dd, es: integer);

    (* input:    yr -- year number AD
      outputs:  mm -- 3 or 4 for March or April Easter
                dd -- day of month of Easter Sunday
                es -- Easter Sunday in days forward from 0 March *)

var   gn,              (* Golden Number *)
      ep,              (* Epact *)
      pfm,
      ymod, dl:        (* Dominical Letter *)
       integer;

(* Easter dates are given by procedure easj (a) according to the
   traditional pre-1582 Easter cycle as recorded by Dionysius
   Exiguus in AD 525 (see Coyne et al 1983 and ESAE 1961) and (b)
   expressed in the Julian calendar of days and months, which is
   offset from the Gregorian calendar by 10 days in 15xx-16xx, 11
   days in 17xx, 12 days in 18xx, 13 days in 19xx-20xx, and
   generally in yr AD by
           ( (yr div 100) - (yr div 400) - 2 ) days
   for 1 March onwards (January and February have the same
   offset as the preceding year if that was different). *)

begin

(* --> Golden number *)                     (* gn [1..19] *)

if yr>=0 then   gn:=  1 + yr - 19*(yr div 19)
         else   gn:=  1 + yr - 19*((yr+1) div 19 ) +19;


(* --> Epact (OS) *)                        (* ep in [0..29] *)

ep:=  29 - ( 19*gn + 2 ) mod 30 ;


(* --> paschal full moon date (OS) *)       (* pfm in [21..49] *)

pfm:= 21 + ((53-ep) mod 30);
                      (* pfm is expressed as an offset from 0 March *)


(* --> Dominical (Sunday) letter (OS) *)

if yr>=0 then   ymod:=  yr - 28* (yr div 28)
         else   ymod:=  yr - 28*((yr+1) div 28) + 28;
dl:=  7 - (ymod +(ymod div 4) + 4) mod 7 ;
                                            (* dl 1..7 for A..G *)


(* --> Easter date *)

es:= 4 + dl + ( ((pfm-4-dl) div 7) +1)*7;
                (* es in [22..56], meaning 22-March through 25-April *)

mm:= 3 + (es-1) div 31;   dd:= 1 + (es-1) mod 31;
                                          (* sorts out month and day *)


(* for debugging or data-checking, unbracket next lines to print
   intermediates:
writeln('y:',yr:5,'  gn:',gn:2,'      ','  ',
        '  ep:',ep:2,'  dl: ',chr(dl+64),'  pfm: ',pfm:2,
        '   es:',es:2,'  m:',mm:1,'  d:',dd:2);
*)

end; (* of proc easj *)

(* =================================================================== *)

(* Test code for exercising procedures easg and easj *)

type str3 = string[3];
const mnthn: array[3..12] of str3 =
  ('Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec');
const mnthl: array[3..12] of integer =
  (31,30,31,30,31,31,30,31,30,31);

var yyr,mmm,ddd,ior, cc, caldiff, esd,esm: integer;
    ch: char;
    mnth: string[6];

const m3 = ' March';
      m4 = ' April';

begin (*main*)

lowvideo;

if paramcount<>1 then
writeln('usage:   eas yyyy',#13,#10,#13,#10,
        'input:   - a year number (AD)',#13,#10,
        'outputs: - dates of Easter Sunday(s), New Style and Old Style',
        #13,#10,
        '         - current calendar difference, NS-OS (days)',#13,#10)
else
begin
               (* get year number from command line *)
   val (paramstr(1),yyr,ior);
   if (yyr<-800) then
       writeln('Sorry! I can''t go back as far as ',yyr)
   else
   begin
               (* get Gregorian Easter and show result *)
   easg (yyr,mmm,ddd, esd);
   if mmm=3  then  mnth:= m3  else  mnth:= m4;
   writeln ('Gregorian/New Style Easter ',yyr:4,': ',ddd:2, mnth,'.');

               (* get Julian Easter *)
   easj   (yyr,mmm,ddd, esd);
   if mmm=3  then  mnth:= m3  else  mnth:= m4;

               (* translate OS Julian Easter date to NS equivalent *)
   if yyr>=0 then cc:= yyr div 100
             else cc:= ((yyr+1) div 100 ) -1;
   caldiff:= cc - ((cc+8) div 4);
   esm:= 3;  esd:= esd+ caldiff;
   while (esd>mnthl[esm]) do
     begin   esd:= esd - mnthl[esm];   esm:= esm+1;   end;

               (* show Julian Easter result with NS equivalent *)
   writeln ('Julian /  Old Style Easter ',yyr:4,': ',ddd:2, mnth,' (OS)',
            ' ( = ',esd,' ',mnthn[esm],' (NS)).');

               (* show difference in calendars for the year *)
   writeln ('(Difference in Mar/Apr ',yyr:4,' between NS and OS calendars: ',
            caldiff,' days.)');

   end
end
end. (*main*)

