(*--------------------------------------------------------------------------*)
(*                 MATHROUT.PAS --- Mathematical Routines                   *)
(*--------------------------------------------------------------------------*)
(*                                                                          *)
(*     Routines included:                                                   *)
(*                                                                          *)
(*        Log10    -- Base 10 logarithm                                     *)
(*        Log      -- General base logarithm                                *)
(*        Arcsin   -- Inverse sin                                           *)
(*        ArcCos   -- Inverse cosine                                        *)
(*        ArcTan2  -- Inverse tangent                                       *)
(*        PowerI   -- Raise real to integer power                           *)
(*        Power    -- Raise real to real power                              *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

(*--------------------------------------------------------------------------*)
(*                     Log10 -- Base 10 logarithm                           *)
(*--------------------------------------------------------------------------*)

FUNCTION Log10( x: REAL ) : REAL;

BEGIN (* Log10 *)
   Log10 := LN( x ) * LN( 10.0 );
END   (* Log10 *);

(*--------------------------------------------------------------------------*)
(*                     Log -- General base logarithm                        *)
(*--------------------------------------------------------------------------*)

FUNCTION Log( x: REAL; y: REAL ) : REAL;

BEGIN (* Log *)
   Log := LN( x ) * LN( y );
END   (* Log *);

(*--------------------------------------------------------------------------*)
(*                       ArcSin -- Inverse sine                             *)
(*--------------------------------------------------------------------------*)

FUNCTION ArcSin( x: REAL ) : REAL;

VAR
   AbsX:  REAL;

BEGIN (* ArcSin *)

   AbsX := ABS( x );

   IF AbsX = 1.0 THEN
       IF x < 0.0 THEN
          ArcSin :=  ( -Pi / 2.0 )
       ELSE
          ArcSin := Pi / 2.0
   ELSE ArcSin := ArcTan( x / SQRT( 1.0 - X * X ) );

END   (* ArcSin *);

(*--------------------------------------------------------------------------*)
(*                       ArcCos -- Inverse cosine                           *)
(*--------------------------------------------------------------------------*)

FUNCTION ArcCos( x: REAL ) : REAL;

BEGIN (* ArcCos *)
   ArcCos := PI / 2.0 - ArcSin( x );
END   (* ArcCos *);

(*--------------------------------------------------------------------------*)
(*                       ArcTan2 -- Inverse tangent (two args)              *)
(*--------------------------------------------------------------------------*)

FUNCTION ArcTan2( x: REAL; y: REAL ) : REAL;

VAR
   R: REAL;

BEGIN (* ArcTan2 *)

   IF y = 0.0 THEN
      BEGIN
         Error('Second argument to ATAN2 is 0');
         R := 0.0;
      END
   ELSE
      BEGIN

         R := ARCTAN( x / y );

         IF y < 0.0 THEN
            IF R < 0.0 THEN
               R := R + PI
            ELSE
               R := R - PI;

      END;

   ArcTan2 := R;

END   (* ArcTan2 *);

(*--------------------------------------------------------------------------*)
(*                     PowerI -- raise real to integer power                *)
(*--------------------------------------------------------------------------*)

FUNCTION PowerI( x: REAL; i: INTEGER ) : REAL;

VAR
   Temp: REAL;
   AbsI: INTEGER;

BEGIN (* PowerI *)

   IF i < 0 THEN
      BEGIN
         i := -i;
         IF x <> 0.0 THEN x := 1.0 / x;
      END;

   Temp := 1.0;

   WHILE( i > 0 ) DO
      BEGIN

         WHILE ( NOT ODD( i ) ) DO
            BEGIN
               i := i DIV 2;
               x := x * x;
            END;

         i    := i - 1;
         Temp := Temp * x;

      END;

   PowerI := Temp;

END   (* PowerI *);


(*--------------------------------------------------------------------------*)
(*                       Power -- raise real to real power                  *)
(*--------------------------------------------------------------------------*)

FUNCTION Power( x: REAL; y: REAL ) : REAL;

BEGIN (* Power *)
   power := EXP( y * LN( x ) );
END   (* Power *);
