/*****************************************************/
/*                                                   */
/*                 CD Audio Library                  */
/*                                                   */
/*       Copyright (c) 1996 by Alister Perrott       */
/*                                                   */
/*              al@aperrott.demon.co.uk              */
/*                                                   */
/*****************************************************/

/* Last Edited: 1/7/96  16:23 */

/***************************************************************************/



/****************************************************************************

List of functions:

CloseCDtray
Find_CD_Drive
GetAudioChannelInfo
GetAudioTrackLength
GetCDplayStatus
GetDeviceStatus
GetFirstAudioTrackNumber
GetLastAudioTrackNumber
GetNumberOfAudioTracks
GetQ_ChannelInfo
GetUPCcode
InitialiseCD
LockCDtray
OpenCDtray
PlayCDtrack
PlayFromCDtrack
ResumeCDplay
SetAudioChannelInfo
StopCDplay
UnlockCDtray


By Category:

Initialisation:

Find_CD_Drive
InitialiseCD


Information:

GetAudioChannelInfo
GetAudioTrackLength
GetCDplayStatus
GetDeviceStatus
GetFirstAudioTrackNumber
GetLastAudioTrackNumber
GetNumberOfAudioTracks
GetQ_ChannelInfo
GetUPCcode


Play control:

PlayCDtrack
PlayFromCDtrack
ResumeCDplay
StopCDplay


Volume control:

GetAudioChannelInfo
SetAudioChannelInfo


Device control:

CloseCDtray
LockCDtray
OpenCDtray
UnlockCDtray


****************************************************************************/


/* Prototypes */


/***************************************************************************/

/*
This function MUST be called before any of the other CD-audio functions.
It checks for the presence of a CD-ROM drive and initialises the buffers
required by the play routines.

If successful it returns zero (CD_OK). 

	Return                Meaning
	======                =======
	CD_OK                 CD-ROM found and buffers initialised
	No_CD_Drive           No CD-ROMs or MSCDEX
	No_Free_DOS_Mem       Unable to allocate DOS memory

*/

char Find_CD_Drive();


/***************************************************************************/

/*
This function reads the audio information for the current disk (track info).
It should be called after 'Find_CD_Drive' and each time a new disk is
inserted. If the function returns 'CD_Drive_Not_Ready' it should be called
again - until it succeeds.

If successful it returns zero (CD_OK). 

 Return                Meaning
 ======                =======
 CD_OK                 Audio track information has been read.
 No_CD_Drive           No CD-ROMs or MSCDEX
 No_Free_DOS_Mem       Unable to allocate DOS memory
 CD_Drive_Not_Ready		Either no disc or disc has just been loaded

*/


char InitialiseCD();


/***************************************************************************/

/* This function plays the selected audio track.

 Return                Meaning
 ======                =======
 CD_OK                 The command has executed successfully
 No_CD_Drive           No CD-ROMs or MSCDEX
 No_Free_DOS_Mem       Unable to allocate DOS memory
 CD_Drive_Not_Ready		Either no disc or disc has just been loaded
 CD_Unknown_Command    The command is unsupported
	CD_Invalid_Parameter  An invalid value was passed to the function

*/

char PlayCDtrack( unsigned char TrackNumber );

/***************************************************************************/

/* This function plays the selected audio track and all audio tracks that
follow it.

 Return                Meaning
 ======                =======
 CD_OK                 The command has executed successfully
 No_CD_Drive           No CD-ROMs or MSCDEX
 No_Free_DOS_Mem       Unable to allocate DOS memory
 CD_Drive_Not_Ready		Either no disc or disc has just been loaded
 CD_Unknown_Command    The command is unsupported
	CD_Invalid_Parameter  An invalid value was passed to the function

*/


char PlayFromCDtrack( unsigned char TrackNumber );

/***************************************************************************/

/* This function stops the drive when it is playing an audio track. Play can
be resumed by calling 'ResumeCDplay'.

 Return                Meaning
 ======                =======
 CD_OK                 The command has executed successfully
 No_CD_Drive           No CD-ROMs or MSCDEX
 No_Free_DOS_Mem       Unable to allocate DOS memory
 CD_Drive_Not_Ready		Either no disc or disc has just been loaded
 CD_Unknown_Command    The command is unsupported

*/

char StopCDplay();

/***************************************************************************/

/* This function is used to resume play when it has been interrupted by a
'StopCDplay' command.

 Return                Meaning
 ======                =======
 CD_OK                 The command has executed successfully
 No_CD_Drive           No CD-ROMs or MSCDEX
 No_Free_DOS_Mem       Unable to allocate DOS memory
 CD_Drive_Not_Ready		Either no disc or disc has just been loaded
 CD_Unknown_Command    The command is unsupported

*/

char ResumeCDplay();

/***************************************************************************/

/* This function opens the CD-ROM tray.

 Return                Meaning
 ======                =======
 CD_OK                 The command has executed successfully
 No_CD_Drive           No CD-ROMs or MSCDEX
 No_Free_DOS_Mem       Unable to allocate DOS memory
 CD_Drive_Not_Ready		Either no disc or disc has just been loaded
 CD_Unknown_Command    The command is unsupported

*/

char OpenCDtray();

/***************************************************************************/

/* This function closes the CD-ROM tray. NOTE: This function is not supported
by all devices (though it works fine on mine).

 Return                Meaning
 ======                =======
 CD_OK                 The command has executed successfully
 No_CD_Drive           No CD-ROMs or MSCDEX
 No_Free_DOS_Mem       Unable to allocate DOS memory
 CD_Drive_Not_Ready		Either no disc or disc has just been loaded
 CD_Unknown_Command    The command is unsupported

*/

char CloseCDtray();

/***************************************************************************/

/* This function locks the CD-ROM tray. NOTE: This function is not supported
by all devices (though it works fine on mine).

 Return                Meaning
 ======                =======
 CD_OK                 The command has executed successfully
 No_CD_Drive           No CD-ROMs or MSCDEX
 No_Free_DOS_Mem       Unable to allocate DOS memory
 CD_Drive_Not_Ready		Either no disc or disc has just been loaded
 CD_Unknown_Command    The command is unsupported

*/

char LockCDtray();

/***************************************************************************/

/* This function unlocks the CD-ROM tray. NOTE: This function is not
supported by all devices (though it works fine on mine).

 Return                Meaning
 ======                =======
 CD_OK                 The command has executed successfully
 No_CD_Drive           No CD-ROMs or MSCDEX
 No_Free_DOS_Mem       Unable to allocate DOS memory
 CD_Drive_Not_Ready		Either no disc or disc has just been loaded
 CD_Unknown_Command    The command is unsupported

*/

char UnlockCDtray();

/***************************************************************************/

/* This function checks whether the drive is currently playing a disk.

One of the following values will be returned; depending on the current
state:

	CD_Playing
	CD_Stopped
	CD_Paused

*/

char GetCDplayStatus();

/***************************************************************************/

/* This function returns a 32-bit value. Bit 0 is the least significant bit.

The bits are interpreted as follows:

  Bit 0     0    Door closed
            1    Door open
  
  Bit 1     0    Door locked
            1    Door unlocked
  
  Bit 2     0    Supports only cooked reading
            1    Supports cooked and raw reading
  
  Bit 3     0    Read only
            1    Read/write
  
  Bit 4     0    Data read only
            1    Data read and plays audio/video tracks
  
  Bit 5     0    No interleaving
            1    Supports interleaving
  
  Bit 6     0    Reserved
  
  Bit 7     0    No prefetching
            1    Supports prefetching requests
  
  Bit 8     0    No audio channel manipulation
            1    Supports audio channel manipulation
  
  Bit 9     0    Supports HSG addressing mode
            1    Supports HSG and Red Book addressing modes
  
  Bit 10-31 0    Reserved (all 0)

*/

unsigned long GetDeviceStatus();

/***************************************************************************/

/* This function returns a pointer to a string containing the current disk's
UPC/EAN (Universal Product Code). NB Many CDs do not have a UPC entry. If no
code is found the string will consist of a single ASCII zero character. */

char * GetUPCcode();

/***************************************************************************/

/* This function returns the current track number and the running time from
its start.
It is assumed that a prior call has been make to 'GetCDplayStatus' in order
to verify that a track is being played.

The returned value is a pointer to a character array as follows:

Offset   Value
======   =====
  0      Track number
  1      Index number
  2      Minutes elapsed from start of track
  3      Seconds elapsed from start of track
  4      Frames elapsed from start of track
  5      Minutes elapsed from start of disk
  6      Seconds elapsed from start of disk
  7      Frames elapsed from start of disk

*/

unsigned char * GetQ_ChannelInfo();

/***************************************************************************/

/* This function returns the number of audio tracks that were found on the
currently registered CD. A value of zero indicates that a valid disc has not
yet been registered. */

unsigned char GetNumberOfAudioTracks();

/***************************************************************************/

/* This function returns the number of the last audio track found on the
currently registered CD. A value of zero indicates that a valid disc has not
yet been registered. */

unsigned char GetLastAudioTrackNumber();

/***************************************************************************/

/* This function returns the number of the first audio track found on the
currently registered CD. A value of zero indicates that a valid disc has not
yet been registered. */

unsigned char GetFirstAudioTrackNumber();

/***************************************************************************/

/* This function returns the length (in frames) of an audio track on the
currently registered CD. A value of zero indicates that a valid disc has not
yet been registered. One frame is equal to 1/75 of a second. */

unsigned long GetAudioTrackLength( unsigned char TrackNumber );

/***************************************************************************/


/* Volume control functions */


/***************************************************************************/

/* This function returns a pointer to a structure containing the current
audio channel assignments and output volumes:

Offset   Value
======   =====
  0      Input channel (0,1,2 or 3) for output channel 0
  1      Volume control (0 - 255) for output channel 0
  2      Input channel (0,1,2 or 3) for output channel 1
  3      Volume control (0 - 255) for output channel 1
  4      Input channel (0,1,2 or 3) for output channel 2
  5      Volume control (0 - 255) for output channel 2
  6      Input channel (0,1,2 or 3) for output channel 3
  7      Volume control (0 - 255) for output channel 3

*/

unsigned char * GetAudioChannelInfo();

/***************************************************************************/

/* This function takes a pointer to a structure containing the new audio
channel assignments and output volumes. It is intended to provide playback
control of audio information on the disk. It allows input channels on the
CD-ROM to be assigned to specific output speaker connections. The purpose
of this function is to allow two independent channels to be recordedin
different languages for exampleand to play back only one of them at a time
or to be able to manipulate an audio signal so that the source appears to
moveto make a sound seem to move from left to right for example.

Output channel 0 is the left channel, 1 is right, 2 is left prime, and 3 is
right prime. The Red Book specification allows for 4 audio channels. The two
"prime" channels (2 and 3) extend stereo to quadrophonic stereo.

An audio volume setting of 0 means off. Drives that don't support 4 output
audio channels may ignore output to channels 2 and 3. Assignment of input
channels 2 and 3 to output channels 0 and 1 may be treated as though the
volume control for that channel is 0.

Drives that do not support variable audio control will treat a setting of 0
as off and 1-0xff as on. Drives that support less than 256 volume settings
will do their best to break up the 256 settings among the settings they can
support. E.g. if there are 16 settings supported, then the first setting
will cover 0x01-0x10, the second 0x11-0x20...the sixteenth 0xf1-0xff. Drives
that can't play a single channel in both must play only that one channel and
try to suppress the other if possible. Drives that can't swap channels
should play the channel that was moved in its normal channel.


Offset   Value
======   =====
  0      Input channel (0,1,2 or 3) for output channel 0
  1      Volume control (0 - 255) for output channel 0
  2      Input channel (0,1,2 or 3) for output channel 1
  3      Volume control (0 - 255) for output channel 1
  4      Input channel (0,1,2 or 3) for output channel 2
  5      Volume control (0 - 255) for output channel 2
  6      Input channel (0,1,2 or 3) for output channel 3
  7      Volume control (0 - 255) for output channel 3

*/

unsigned char SetAudioChannelInfo( unsigned char * );

/***************************************************************************/


/* Definitions */


/***************************************************************************/

/* The maximum number of Audio tracks for which the play routines need to 
keep records. If this value is changed, cdaudio.c will need to be recompiled
for the buffer sizes to alter. The largest value that this can be set to is
255 - as determined by the maximum track number for which MSCDEX will return
information. */

#define MaxAudioTracks 5 /* 100 */

/***************************************************************************/

/* Set 'Debugging' to 1 to display debugging info. */

#define Debugging 0

/***************************************************************************/

/* Return codes */

#define CD_OK 0
#define No_CD_Drive 1
#define No_Free_DOS_Mem 2
#define CD_Drive_Not_Ready 3
#define CD_Unknown_Command 4
#define CD_Invalid_Parameter 5
#define CD_Not_Yet_Initialised 6


#define CD_Playing 0
#define CD_Stopped 1
#define CD_Paused 2

/***************************************************************************/

