/*
**  Program:	<stat.c>
**
**  Purpose:	Provides status control for BYE-PC testing and
**		control of modem functions.
**
**  Author:	R.E. Starr, Jr.
**
**  Revisons:
**
**     1.00 (12/01/86) -- First release
**     1.01 (03/30/87) -- Modfiied for change in CSW.
**     1.02 (03/30/87) -- Added new status functions implemented
**			  in BYE200
**     1.03 (08/08/87) -- Modified for byexface version 2.0
**     1.04 (10/23/87) -- Added name setting capability
*/

#include    <stdio.h>
#include    <conio.h>
#include    <process.h>
#include    <ctype.h>
#include    <string.h>
#include    "bit.h"
#include    "byexface.h"

#define YES 1
#define NO  0
#define ON  1
#define OFF 0

#define VER 1		/* version of STAT running */
#define REV 4

#define BYE_VER 3	/* BYE version# 3.00 or > required */
#define BYE_REV 0


    /* function delcarations */

int binary();

void null_cmd();
void commands();
void remote();
void send_tx();
void show_baud(), show_stat();
void show_nulls();
void show_rxsize();
void set_mdmbreak();
void set_mdmcd();
void set_dtr();
void set_mdmpause();
void set_timer();
void get_rx();
void set_status();
void set_mdmtrap();
void set_name();


    /* static command table */

#define PCMDS  (sizeof(cmdpri) / sizeof(struct key))

struct key
    {
    char *keyword;	    /* pointer to key word */
    void (*key_fxn)();	    /* pointer to function */
    };
struct key cmdpri[] =			/* primary command set */
    {
    "BAUD", show_baud,
    "BOOT", _bye_warmboot,
    "BREAK", set_mdmbreak,
    "CD", set_mdmcd,
    "DTR", set_dtr,
    "FLUSH", _bye_rxflush,
    "HELP", commands,
    "NAME", set_name,
    "NULLS", show_nulls,
    "PAUSE", set_mdmpause,
    "QUEUE", show_rxsize,
    "REMOTE", remote,
    "RX", get_rx,
    "STATUS", set_status,
    "TIME", set_timer,
    "TRAP", set_mdmtrap,
    "TX", send_tx
    };

#define SCMDS  (sizeof(cmdsec) / sizeof(struct key))

struct key cmdsec[] =			/* secondary command set */
    {
    "OFF", null_cmd,
    "ON", null_cmd
    };


main(argc, argv)

 int argc;
 char *argv[];
    {
    char *ptr;
    int n, rtn;

    rtn = _bye_check(BYE_VER, BYE_REV);
    if (rtn)
	{
	switch(rtn)
	    {
	    case 1:
		printf("\nBYE-PC is not loaded!\n");
		break;
	    case 2:
		printf("\nBYE-PC loaded is the wrong Version!\n");
		break;
	    case 3:
		printf("\nBYE-PC loaded is the wrong Revision!\n");
		break;
	    default:
		printf("\nBYE-PC returned invalid error code!\n");
		break;
	    }
	exit(1);
	}
    if (argc > 1 && (_bye_getcsw() == 0xffff))
	{
	n = binary(argv[1], cmdpri, PCMDS);
	if (n != EOF)
	    (*cmdpri[n].key_fxn)(argv[2]);
	else
	    {
	    commands();
	    exit(1);
	    }
	}
    else
	show_stat();
    exit(0);
    }


/*
**
**  Function:	int binary(*word, tab, n);
**
**  Paramters:	char *word --> pointer to string to search for
**		struct tab --> command table structure
**		int n	   --> number of commands in command table
**
**  Purpose:	Performs a binary search of the command table structure
**		passed in the structure 'tab'. The strings are compared
**		without regard to case.
**
**  Return:	EOF = command not found, else return the index of the
**		command in the command table (0 to n).
**
*/

int binary(word, tab, n)

 char *word;
 struct key tab[];
 int n;
    {
    int low, high, mid, cond;

    low = 0;
    high = n - 1;
    while (low <= high)
	{
	mid = (low + high) / 2;
	if ((cond = strcmpi(word, tab[mid].keyword)) < 0)
	    high = mid - 1;
	else if (cond > 0)
	    low = mid + 1;
	else
	    return(mid);
	}
    return(-1);
    }


/*
**
**  Function:	void null_cmd();
**
**  Paramters:	<none>
**
**  Purpose:	Empty command for structure tables
**
**  Return:	<none>
**
*/

void null_cmd()
 {
 return;
 }


/*
**
**  Function:	void commands();
**
**  Paramters:	<none>
**
**  Purpose:	Displays a list of all valid commands
**
**  Return:	<none>
**
*/

void commands()
 {
 printf("\n\t** BYE-PC STAT Version %1d.%-2.2d **\n\n", VER, REV);
 printf("BAUD ............. Displays Current Baud Rate On-Line.\n");
 printf("BOOT ............. Hang up and warm boot system.\n");
 printf("BREAK {ON/OFF} ... Sets Remote ^C & ^S option state.\n");
 printf("CD {ON/OFF} ...... Sets Carrier Lost Detect State.\n");
 printf("DTR {ON/OFF} ..... Toggle DTR line state.\n");
 printf("FLUSH ............ Flush Rx-Queue of all data.\n");
 printf("NULLS [#] ........ Displays and Sets Number of nulls (0-9).\n");
 printf("NAME [string]..... Displays and Sets Callers name (63 chars max).\n");
 printf("PAUSE {ON/OFF} ... Sets the ^S pause timeout state.\n");
 printf("QUEUE ............ Displays Rx-Queue size.\n");
 printf("REMOTE {ON/OFF} .. Disables remote console.\n");
 printf("RX {#} ........... Rx up to 'n' characters & display.\n");
 printf("STATUS [#] ....... Set caller status level (0-255)\n");
 printf("TIME [0-255] ..... Show/set time allowed on the system.\n");
 printf("TRAP {ON/OFF} .... Filter/Trap all ^C & ^S characters.\n");
 printf("TX {literal} ..... Transmit the string literal to remote.\n");
 printf("\nNote: {} are required arguments and [] indicates optional.\n\n");
 }


/*
**
**  Function:	void show_stat();
**
**  Paramters:	<none>
**
**  Purpose:	Displays current caller status data
**
**  Return:	<none>
**
*/

void show_stat()
 {
 char d, name[64];
 int baud, vers, rev;
 unsigned x, t1, t2, t3;

 x = _bye_getcsw();
 if (x == 0xffff)
    printf("\n(System Operator)");
 _bye_version(&vers, &rev);	/* read version number of BYE */
 printf("\nSTAT %1d.%-2.2d under BYE-PC Version %1d.%-2.2d\n\n", VER, REV, vers, rev);

 _bye_getname(name);
 printf("Caller on line       : %s\n", name);
 printf("Modem Line Status    : (");
 if (_bye_getcd())
     printf("On");
 else
     printf("Off");
 printf(" line)\n");

 printf("Current Baud Rate    : ");
 baud = _bye_baud();
 switch(baud)
     {
     case 0:
	 printf("300bps");
	 break;
     case 1:
	 printf("1200bps");
	 break;
     case 2:
	 printf("2400bps");
	 break;
     default:
	 printf("(High Speed!)");
     }
 printf("\nNulls set to         : %1d\n", _bye_getnulls());
 d = (char)((x >> 8) & 0x0F) + 'A';
 printf("Maximum Drive Access : %c:\n", d);
 printf("DOS Path Restriction : ");
 if (x & BIT_2)
     printf("(Un-Restricted)\n");
 else
     printf("(Restricted)\n");
 printf("CD Command Status    : ");
 if (x & BIT_3)
     printf("(Active)\n");
 else
     printf("(Inactive)\n");
 printf("File Send access     : ");
 if (x & BIT_4)
     printf("(Active)\n");
 else
     printf("(Inactive)\n");
 printf("File Receive access  : ");
 if (x & BIT_5)
     printf("(Active)\n");
 else
     printf("(Inactive)\n");
 t1 = _bye_gettimer();		    /* minutes passed since log on */
 t2 = _bye_gettimeon(); 	    /* time allowed this call */
 printf("Time since logged on : %u minute(s)\n", t1);
 printf("Time limit this call : ");
 if (t2)
    printf("%u minute(s)\n", t2);
 else
    printf("(unlimited)\n");
 printf("Time left this call  : ");
 if (t2)
    printf("%u minute(s)\n", t2 - t1);
 else
    printf("(unlimited)\n");
 if (x == 0xffff)
    printf("\n{STAT HELP or ?} for a list of commands...\n\n");
 }


/*
**  Function:	void remote()
**
**  Purpose:	disables remote tx/rx status.
**
*/

void remote(str)

 char str[];
    {
    int n;
    extern struct cmdsec;

    n = binary(str, cmdsec, SCMDS);
    switch(n)
	{
	case EOF:
	    printf("\nInvalid LOCAL option!\n");
	    return;
	case 0:
	    printf("\nRemote OFF\n");
	    _bye_stdout(0);	/* set local tx-state */
	    _bye_stdin(0);	/* set remote rx-state */
	    break;
	case 1:
	    _bye_stdout(1);	/* set local tx-state */
	    _bye_stdin(1);	/* set remote rx-state */
	    printf("\nRemote ON\n");
	    break;
	}
    }


/*
**  Function:	void show_baud()
**
**  Purpose:	Shows the Baud rate of the caller on line.
**
*/

void show_baud()
 {
 int baud;

 printf("\nModem ");
 if (_bye_getcd())
    printf("ON");
 else
    printf("OFF");
 printf(" line at ");

 baud = _bye_baud();
 switch(baud)
    {
    case 0:
	printf("300bps");
	break;
    case 1:
	printf("1200bps");
	break;
    case 2:
	printf("2400bps");
	break;
    default:
	printf("(High Speed!)");
    }
 printf("\n");
 }


/*
**  Function:	void show_vers()
**
**  Purpose:	Shows the version/revison of BYE-PC running.
**
*/

void show_vers()
 {
 int ver, rev;

 _bye_version(&ver, &rev);
 printf("\nBYE-PC Version %1d.%-2.2d\n", ver, rev);
 }


/*
**  Function:	void show_nulls()
**
**  Purpose:	Shows the # of nulls sent before each char.
**
*/

void show_nulls(str)

 char str[];
    {
    int n, x;

    if (str[0] == NULL)
	{
	x = _bye_getnulls();		    /* read # of nulls */
	printf("\nNumber of nulls: %1d\n", x);
	}
    else
	{
	n = atoi(str);
	_bye_setnulls(n);
	x = _bye_getnulls();		    /* read # of nulls */
	printf("\nNulls set to: %1d  [%-2.2xh]\n", x, n);
	}
    }


/*
**  Function:	void show_rxsize()
**
**  Purpose:	Shows the # chars in the rx queue.
**
*/

void show_rxsize()
 {
 printf("\nRx-Queue size: %d\n", _bye_rxsize());
 }


/*
**  Function:	void send_tx(string)
**
**  Purpose:	Send a string of text to modem.
**
*/

void send_tx(s)

 char s[];
    {
    int i;

    printf("\n[START Data-Tx]\n");
    for (i = 0; i < strlen(s); ++i)
	_bye_putc(s[i]);
    printf("\n[END of Data-Tx]\n");
    }


/*
**  Function:	void get_rx(n)
**
**  Purpose:	Receive 'n' characters from the modem.
**
*/

void get_rx(str)

 char str[];
    {
    int n, i, err;
    unsigned c;

    err = 0;
    printf("\n[START Data-Rx]\n");
    n = atoi(str);
    for (i = 0; i < n; ++i)
	{
	err = 0;
	while(1)
	    {
	    c = _bye_getc();
	    if (c == EOF)
		{
		tdelay(8*1);
		if (err++ > 6)
		    {
		    printf("[TIME-OUT]\n");
		    break;
		    }
		}
	    else
		{
		printf("%c", (int)c);
		break;
		}
	    }
	}
    printf("\n[END of Data-Rx]\n");
    }


/*
**  Function:	void mdm_flush()
**
**  Purpose:	Flushes rx-queue of all rx-data.
**
*/

void mdm_flush()
 {
 _bye_rxflush();
 printf("\nRx-Queue Flushed...\n");
 }


/*
**  Function:	void set_dtr()
**
**  Purpose:	Set the DTR line to the modem.
**
*/

void set_dtr(str)

 char str[];
    {
    int n;
    extern struct cmdsec;

    n = binary(str, cmdsec, SCMDS);
    switch(n)
	{
	case EOF:
	    printf("\nInvalid DTR option!\n");
	    return;
	case 0:
	    _bye_dtr(0);
	    break;
	case 1:
	    _bye_dtr(1);
	    break;
	}
    }


/*
**  Function:	void set_mdmcd()
**
**  Purpose:	Set the CD status from modem.
**
*/

void set_mdmcd(str)

 char str[];
    {
    int n;
    extern struct cmdsec;

    n = binary(str, cmdsec, SCMDS);
    switch(n)
	{
	case EOF:
	    printf("\nInvalid CD option!\n");
	    return;
	case 0:
	    _bye_setcd(0);
	    printf("\nModem CD status OFF...\n");
	    break;
	case 1:
	    _bye_setcd(1);
	    printf("\nModem CD status ON...\n");
	    break;
	}
    }


/*
**  Function:	void set_mdmbreak()
**
**  Purpose:	Set ^C status flag in BYE.
**
*/

void set_mdmbreak(str)

 char str[];
    {
    int n;
    extern struct cmdsec;

    n = binary(str, cmdsec, SCMDS);
    switch(n)
	{
	case EOF:
	    printf("\nInvalid BREAK option!\n");
	    return;
	case 0:
	    _bye_setbreak(CTRL_NOBRK);
	    printf("\nRemote BREAK status OFF...\n");
	    break;
	case 1:
	    _bye_setbreak(CTRL_BRK);
	    printf("\nRemote BREAK status ON...\n");
	    break;
	}
    }


/*
**  Function:	void set_mdmpause()
**
**  Purpose:	Set ^S timeout status flag in BYE.
**
*/

void set_mdmpause(str)

 char str[];
    {
    int n;
    extern struct cmdsec;

    n = binary(str, cmdsec, SCMDS);
    switch(n)
	{
	case EOF:
	    printf("\nInvalid PAUSE option!\n");
	    return;
	case 0:
	    _bye_setbreak(CTRL_NOTOUT);
	    printf("\nRemote PAUSE timeout OFF...\n");
	    break;
	case 1:
	    _bye_setbreak(CTRL_TOUT);
	    printf("\nRemote PAUSE timeout ON...\n");
	    break;
	}
    }


/*
**  Function:	void set_mdmtrap()
**
**  Purpose:	Sets pass/ignore all ^C and ^S status flag in BYE.
**
*/

void set_mdmtrap(str)

 char str[];
    {
    int n;
    extern struct cmdsec;

    n = binary(str, cmdsec, SCMDS);
    switch(n)
	{
	case EOF:
	    printf("\nInvalid TRAP option!\n");
	    return;
	case 0:
	    _bye_setbreak(CTRL_NOTRAP);
	    printf("\nRemote TRAP ^C & ^S status OFF...\n");
	    break;
	case 1:
	    _bye_setbreak(CTRL_TRAP);
	    printf("\nRemote TRAP ^C & ^S status ON...\n");
	    break;
	}
    }



/*
**  Function:	void set_status(str)
**
**  Purpose:	Sets/shows caller status level in BYE.
**
*/

void set_status(str)

 char str[];
    {
    int n;

    if (str[0] == NULL)
	printf("\nCaller Status Level: %-4.4xh\n", _bye_getcsw());
    else
	{
	n = atoi(str);
	if (n <= 0 || n > 255)
	    n = 0;
	_bye_setcsw(n);
	printf("\nCaller Status Set to: %-4.4xh\n", _bye_getcsw());
	}
    }


/*
**  Function:	void set_timer()
**
**  Purpose:	Set ^S timeout status flag in BYE.
**
*/

void set_timer(str)

 char str[];
    {
    int n, x;

    if (str[0] == NULL)
	{
	x = _bye_gettimer();
	printf("\nMinutes on system: %d\n", x);
	}
    else
	{
	n = atoi(str);
	if (n < 0)
	    n = 0;
	if (n > 255)
	    n = 255;
	_bye_settimeon(n);
	printf("\nTime on system set to: %d minutes\n", n);
	}
    }


/*
**  Function:	void set_name(str)
**
**  Purpose:	Sets/shows caller status level in BYE.
**
*/

void set_name(str)

 char str[];
    {
    char name[65];
    int n;

    _bye_getname(name);
    if (str[0] == NULL)
	printf("\nCaller on line: %s\n", name);
    else
	{
	strncpy(name, str, 63);
	_bye_setname(name);
	_bye_getname(name);
	printf("\nCaller Name Set to: %s\n", name);
	}
    }


