#include <stdio.h>
#include <stdlib.h>
#include <conio.h>

#define NORMAL        0x7         /* Normal video attribute */
#define BOLD          0x8         /* Bold video attribute */
#define UNDERLINED    0xA         /* Underlined video attribute */
#define REVERSE       0x70        /* Reverse video attribute */

/****************************************************************************/
/* Function prototypes                                                      */

void VTInit( void );
void ConOut( unsigned char );
static void atnrm( unsigned char );
static void (*ttstate)() = atnrm;
static void atescf( unsigned char );
static void AnsiParse( unsigned char );
static void ExtParse( unsigned char );
static void AnsiModeSet( char, int );
static void ExtModeSet( char, int );
static void SetChar0( unsigned char );
static void SetChar1( unsigned char );
static void SetDouble( unsigned char );
static void TransmitId( void );
static void VTBell( void );

/****************************************************************************/
/* Global variables                                                         */

unsigned originmode;              /* Origin mode, relative or absolute */
unsigned insertmode;              /* Insert mode, off or on */
unsigned autowrap;                /* Automatic wrap mode, off or on */
unsigned newline;                 /* Newline mode, off or on,  GLOBAL data!*/
unsigned cursorvisible;           /* Cursor visibility, on or hidden */
unsigned reversebackground;       /* Reverse background attribute, on or off*/
unsigned screenwid;               /* Screen column width */
unsigned char log;                /* Flag to indicate char logging is on */

/****************************************************************************/
/* External variables                                                       */



/****************************************************************************/
/* Local static data                                                        */

char term_id_str[]= "[?1;2c";     /* VT100 id string */
#define lansarg 10                /* Max number of ANSI arguments */
static int nansarg = 0;           /* Index for ANSI argument list */
static int ansargs[lansarg];      /* Room for 10 ANSI arguments */
static unsigned char lastc;       /* Saved last character */


/*****************************************************************************/
/*****************************************************************************/

/*  V T I N I T  --   */

void VTinit() {

    if (GetVTSetup() == 0) {      /* If there are no saved values for */
        screenwid = 80;           /* VT emulation parameters then provid'em */
        newline = 0;
        autowrap = 0;
        insertmode = 0;
        cursorvisible = 1;
        reversebackground = 0;
        log = 0;
    }


    Setvattr( NORMAL );
    ttstate = atnrm;              /* initial output state is normal */
    SetScroll(0,0);
    ClearScreen();
    SetCharSet(0, 'B');
    SetCharSet(1, 'B');
    MapCharSet(0);
    ClearAllTabs();
    InitTabs();
    SetScreenWidth(screenwid);
    SetCursorVisibility(cursorvisible);
    SetBackGround(reversebackground);
    SetCurs(1,1);
    SaveCursor();
    lastc = '\0';
}



/*  C O N O U T  --  Put a character to the terminal screen */

void ConOut(unsigned char c) {

   (*ttstate) (c);
   if (log)
       writelog(c);

   lastc = c;

}

/*  A T N R M  --  local routine to process an arbitrary character */

static void atnrm(c) unsigned char c; {

   if ( c >= 32 && c < 0x80 )
      chrwrite(c);

   else switch (c) {

      case 27:                    /* Escape */
          ttstate = atescf;       /* next state parser is esc follower */
          break;

      case '\r':                  /* Carriage return */
          SetCurs(1,0);
          break;

      case '\n':                  /* Line feed */
          if (newline)
              SetCurs(1,0);

      case 11:                    /* Vertical tab */
      case 12:                    /* Form feed */
          ScrollUp();
          break;

      case '\a':                  /* Ring terminal bell */
          VTBell();
          break;

      case 8:                     /* back space */
          SetRelCurs( -1, 0 );
          break;

      case '\t':                  /* Horizontal tab */
          DoTab();
          break;


      case 14:                    /* Map G1 to current */
          MapCharSet(1);
          break;

      case 15:                    /* Map G0 to current */
          MapCharSet(0);
          break;

      case 5:                     /* transmit answer back */
          break;

      default:

      }
}




/*  A T E S C F  --  escape sequence follower */

static void atescf(c) unsigned char c; {
   register char *termid = term_id_str;

   switch (c) {

        case  '[':                /* Parse ansi args */
           memset(ansargs,0,sizeof(ansargs));
           nansarg = 0;
           ttstate = AnsiParse;
           break;

        case  'D':                /* Index */
           ScrollUp();
           ttstate = atnrm;
           break;

        case  'E':                /* Carriage return/line feed combination */
           SetCurs(1,0);
           ScrollUp();
           ttstate = atnrm;
           break;

        case  'M':                /* Reverse Index */
           ScrollDown();
           ttstate = atnrm;
           break;

        case  'H':                /* Set a tab stop */
           SetTabStop();
           ttstate = atnrm;
           break;

        case  '7':                /* Save cursor description */
           SaveCursor();
           ttstate = atnrm;
           break;

        case  '8':                /* Restore cursor description */
           RestoreCursor();
           ttstate = atnrm;
           break;

        case  '=':                /* Enable application keypad */
           SetKeyPad(1);
           ttstate = atnrm;
           break;

        case  '>':                /* Enable numeric keypad */
           SetKeyPad(0);
           ttstate = atnrm;
           break;

        case  'c':                /* Reset terminal to power on values */
           VTInit();
           ttstate = atnrm;
           break;

        case  '(':                /* Select character set G0 */
           ttstate = SetChar0;
           break;

        case  ')':                /* Select character set G1 */
           ttstate = SetChar1;
           break;

        case  '#':                /* Set double high/wide characters */
           ttstate = SetDouble;
           break;

        case 24:
        case 26:                  /* Cancel escape sequence */
           ttstate = atnrm;
           break;


        case  'Z':                /* Transmit the terminal ID */
           TransmitId();
           ttstate = atnrm;
           break;

        case  '\\':
           ttstate = atnrm;
           break;

        case  '<':
           ttstate = atnrm;
           break;

        case  'P':
           ttstate = atnrm;
           break;

        case  '*':
           ttstate = atnrm;
           break;

        default:                  /* unrecognized so ignore */
           ttstate = atnrm;
           break;
        }

}


/*  A N S I P A R S E  --  parse ansi arguments */

static void AnsiParse(c) unsigned char c; {
   register int i;
   register int j;

   c &= 0x7F;

   switch(c) {

        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
           ansargs[nansarg] = (ansargs[nansarg] * 10) + ( c - '0');
           break;

        case ';':                 /* Argument separator */
           if ( ++nansarg > lansarg)
              ttstate = atnrm;
           break;

        case 'h':                 /* Set ANSI mode */
           for( i = 0, ++nansarg; i < nansarg && i <= lansarg; i++)
              AnsiModeSet(ansargs[i],1);
           ttstate = atnrm;
           break;

        case 'l':                 /* Reset ANSI mode */
           for( i = 0, ++nansarg; i < nansarg && i <= lansarg; i++)
              AnsiModeSet(ansargs[i],0);
           ttstate = atnrm;
           break;


        case 'H':                 /* Address cursor to line and column */
        case 'f':
           i = ansargs[0];
           j = ansargs[1];
           if ( i == 0)
              i = 1;
           if ( j == 0)
              j = 1;
           SetCurs(j,i);

           ttstate = atnrm;
           break;

        case 'J':                 /* Erase screen */
           if (ansargs[0] == 0){      /* from cursur to end of the screen */
               ClearEOS();
               }
           else if (ansargs[0] == 1){ /* from home position to cursur */
               ClearBOS();
               }
           else if (ansargs[0] == 2){ /* whole screen */
               ClearScreen();
               }
           ttstate = atnrm;
           break;


        case 'K':                 /* Erase Line */
           if (ansargs[0] == 0)       /* from cursur to end of the line */
               ClearEOL();
           else if (ansargs[0] == 1)  /* start of line to cursur */
               ClearBOL();
           else if (ansargs[0] == 2){ /* whole line */
               ClearBOL();
               ClearEOL();
               }
           ttstate = atnrm;
           break;


        case 'm':                 /* Select screen attribute */
           ttstate = atnrm;
           if ( ++nansarg <= lansarg) {
              for ( i = 0; i < nansarg; i++) {
                 switch (ansargs[i]) {
                    case 0:
                    case 22:
                       if (i == 0)
                          SetVattr( NORMAL );
                       else
                          AddVattr(NORMAL);
                       break;
                    case 1:
                       if (i == 0)
                          SetVattr( BOLD);
                       else
                          AddVattr(BOLD);
                       break;
                    case 2:
                       SubVattr( BOLD );
                       break;
                    case 4:
                       if (i == 0)
                          SetVattr( UNDERLINED );
                       else
                          AddVattr( UNDERLINED );
                       break;
                    case 5:
                       if (i == 0)
                          SetVattr( BLINK );
                       else
                          AddVattr( BLINK );
                       break;
                    case 7:
                       if (i == 0)
                          SetVattr( REVERSE );
                       else
                          AddVattr( REVERSE );
                       break;
                    case 24:
                       SubVattr( UNDERLINED );
                       break;
                    case 25:
                       SubVattr( BLINK );
                       break;
                    case 27:
                       SubVattr( REVERSE );
                       break;
                    default:
                       break;
                 }
              }
           }
           break;

       case '?':                  /* Extended mode set/reset */
           if (lastc == '[')
              ttstate = ExtParse;
           else
              ttstate = atnrm;
           break;

        case 'r':                 /* Define scrolling region */
           SetScroll(ansargs[0],ansargs[1]);
           ttstate = atnrm;
           break;

        case 'A':                 /* Move cursor up */
           if (ansargs[0] == 0)
              SetRelCurs(0,-1);
           else
              SetRelCurs(0,-ansargs[0]);

           ttstate = atnrm;
           break;

        case 'B':                 /* Move cursor down */
           if (ansargs[0] == 0)
              SetRelCurs(0,1);
           else
              SetRelCurs(0,ansargs[0]);
           ttstate = atnrm;
           break;

        case 'C':                 /* Move cursor right */
           if (ansargs[0] == 0)
              SetRelCurs(1,0);
           else
              SetRelCurs(ansargs[0],0);
           ttstate = atnrm;
           break;

        case 'D':                 /* Move cursor left */
           if (ansargs[0] == 0)
              SetRelCurs(-1,0);
           else
              SetRelCurs(-ansargs[0],0);
           ttstate = atnrm;
           break;

        case 'g':                 /* Tab stop set/reset */
           if (ansargs[0] == 0)
               ClearTabStop();
           else if (ansargs[0] == 3)
               ClearAllTabs();
           ttstate = atnrm;
           break;


        case 'c':                 /* Transmit the terminal ID */
           TransmitId();
           break;

        case 24:
        case 26:                  /* Cancel escape sequence */
           ttstate = atnrm;
           break;


        default:                  /* unrecognized so ignore */
           ttstate = atnrm;
           break;
   }
}


/* E X T P A R S E  -- Parse extended mode Set/Reset */

static void ExtParse(unsigned char c) {
    register int i;

    switch (c) {

        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
           ansargs[nansarg] = (ansargs[nansarg] * 10) + ( c - '0');
           break;

        case ';':                 /* Argument separator */
           if ( ++nansarg > lansarg)
              ttstate = atnrm;
           break;

        case 'h':
           for( i = 0, ++nansarg; i < nansarg && i <= lansarg; i++)
              ExtModeSet(ansargs[i],1);
           ttstate = atnrm;
           break;

        case 'l':
           for( i = 0, ++nansarg; i < nansarg && i <= lansarg; i++)
              ExtModeSet(ansargs[i],0);
           ttstate = atnrm;
           break;

        case 24:
        case 26:                  /* Cancel escape sequence */
           ttstate = atnrm;
           break;

        default:
           ttstate = atnrm;
           break;
    }
}


/* A N S I M O D E S E T  -- Set/Reset ANSI mode   ,  ESC [ P1,,, Pn h/l */

static void AnsiModeSet( char c, register int mode) {

    switch (c ) {
        case 2:                   /* Lock/unlock keyboard */
            break;


        case 4:                   /* Insert/Replace setting */
            insertmode = mode;
            break;

        case 12:                  /* Echo */
            break;

        case 20:                  /* New Line mode */
            newline = mode;
            break;

        default:
           break;
    }
}


/* E X T M O D E S E T  --  Set/Reset extended mode after ESC [ ? */

static void ExtModeSet(char c, int mode) {

    switch (c) {
        case 1:
            SetCursorKey(mode);   /* set cursor key mode */
            break;

        case 3:                   /* Set the screen width */
            if (mode)
               SetScreenWidth(132);
            else
               SetScreenWidth(80);
            break;

        case 4:                   /* Jump/smooth scroll (not supported) */
            break;

        case 5:                   /* Set the background attribute */
            SetBackGround(mode);
            break;


        case 6:                   /* Set the scrolling origin */
            originmode = mode;
            break;

        case 7:                   /* set the autowrap mode */
            autowrap = mode;
            break;

        case 8:                   /* Auto repeat on/off */
            break;

        case 18:                  /* Page print terminator */
            break;

        case 19:                  /* Print region */
            break;

        case 25:                  /* Display/Hide cursor */
            SetCursorVisibility(mode);
            break;

        default:
            break;
    }
}



/* A T C H R S E T 0 -- Set the current character set for G0 */

void SetChar0( unsigned char c) {
  SetCharSet( 0, c );
  ttstate = atnrm;
}

/* A T C H R S E T 1 -- Set the current character set for G1 */

void SetChar1( unsigned char c) {
  SetCharSet( 1, c );
  ttstate = atnrm;
}


/* S E T D O U B L E -- Set the current line to double high and/or wide */

void SetDouble( unsigned char c) {

  switch (c) {
     case '5':                    /* Single width */
     case '6':                    /* Double width */
     case '3':                    /* Double height/width */
     case '4':                    /* Bottom half of double height/width */
     default:
  }
  ttstate = atnrm;

}

/* T R A N S M I T I D -- Transmit the terminal id to the host */

static void TransmitId() {
   register char *termid = term_id_str;

   ttoc(27);
   while(*termid != '\0')         /* Do until end of id string */
      ttoc(*termid++);               /* Transmit each character of string */
}


/*  V T B E L L  --  Do a VT100 style bell */

static void VTBell() {

   sound(880);
   delay(125);
   nosound();

}

