/*******************************************************

    The CalcPlus Class Library Version 1.0,
    Author: Vladimir Schipunov, Copyright (C) 1996

    This library is free software. Permission to use,
    copy, modify and redistribute the CalcPlus library
    for any purpose is hereby granted without fee,
    provided that the above copyright notice appear
    in all copies.

*******************************************************/

#include <string.h>
#include <ctype.h>
#include <time.h>
#if !defined(_MSC_VER) && !defined(__WIN32__) // for MSVC 1.5 & BCC32 4.5
    #include <strstream.h>
#else
    #include <strstrea.h>
#endif
#include "calctype.h"

ostream& operator << ( ostream& o, const CType& t )
{
    t.print( o );
    return o;
}

istream& operator >> ( istream& i, CType& t )
{
    t.get( i );
    return i;
}

CType& CType::operator=( const CType& t )
{
    int n = size();
    if( type()!=t.type() || n!=t.size())
        cerr << "Types or sizes of objects are not the same" << endl;
    else
        memcpy( ptr(), t.data(), n );
    return *this;
}

void CType::read( char* s )
{
    istrstream in( s, strlen(s) );
    get( in );
}

static char buf[512];
const char* CType::s( char* p ) const
{
    if( !p )
        p = buf;
    ostrstream o( p, sizeof( buf ));
    o << *this << '\0';
    return p;
}

CType::empty() const
{
    const char* p = (const char*)data();
    long n = size();
    for( int i = 0; i < n; i++ )
        if( p[i] )
            return 0;
    return 1;
}

CType::compare( const CType& t ) const
{
    return (type()==t.type() && size()==t.size()) ?
        memcmp( data(), t.data(), size() ) : -1;
}

#define BOOL_TRUE  "TRUE"
#define BOOL_FALSE "FALSE"

void CLong::get( istream& in ){ long x; in >> x; n = x; } // for ZTC 3.1
void CLong::print( ostream& out ) const { out << n; }
void CDouble::get( istream& in ){ double x; in >> x; n = x; }
void CDouble::print( ostream& out ) const { out << n; }
void CBool::print( ostream& out ) const { out << (n ? BOOL_TRUE:BOOL_FALSE); }

void CBool::get( istream& in )
{
    in >> buf;
    for( int i = 0; i < sizeof( buf ); i++ )
        buf[i] = (char)toupper( buf[i] );
    n = !strcmp( buf, BOOL_FALSE ) ? 0 :
        !strcmp( buf, BOOL_TRUE  ) ? 1 : 2;
    if( n==2 )
        cerr << "Bad initializer for boolean" << endl;
}

CString::CString( const char* s, int len ) :
    CType( idString ), zero( len ? 0 : 1 )
{
    n = len ? len : strlen( s )+1;
    p = new char[n];
    if( len )   memcpy( p, s, n );
    else        strcpy( p, s );
}

CString::CString( const CString& s ) :
    CType( idString ), n( s.n ),
    p( new char[s.n] ), zero( s.zero )
{
    if( zero )
        strcpy( p, s.p );
    else
        memcpy( p, s.p, n );
}

CString::CString( int len ) :
    CType( idString ), n( len ),
    p( new char[len] ), zero( 1 )
{
}

const char *CString::s( char *p ) const
{
    if( !p && zero )
        return *this;
    if( !p )
        p = buf;
    memcpy( p, CString::p, n );
    if( !zero )
        p[n] = 0;
    return p;
}

void CString::print( ostream& out ) const { out << p; }
void CString::get( istream& in ) { in.read( p, n ); }
void CNil::print( ostream& out ) const { out << "NIL"; }

CType& CLong::operator=( const CType& t )
{
    if( t.type()!=idLong )
        return CType::operator=(t);
    n = ((CLong&)t).n;
    return *this;
}

CType& CString::operator=( const CType& t )
{
    if( type()!=t.type() || size()==t.size())
        return CType::operator=(t);
    const CString& s = (const CString&)t;
    delete[] p;
    p = new char[s.n];
    memcpy( p, s.p, n );
    return *this;
}

CType& CString::operator=( const char* s )
{
    int len = strlen( s )+1;
    if( size()+zero!=len )
    {
        delete[] p;
        p = new char[ n = len ];
    }
    memcpy( p, s, n );
    return *this;
}

char& CString::operator[]( int i ) const
{
    static char err;
    if( i<0 || i>=n || !p )
    {
        cerr << "Error in string index" << endl;
        return err;
    }
    return p[i];
}

CString& CString::right( char c )
{
    int i;
    for( i = 0; i < n; i++ )
        if( p[n-i-1] != ' ' )
            break;
    if( i < n )
        for( int j = n-1; j >= 0; j-- )
            p[j] = j-i >= 0 ? p[j-i] : c;
    return *this;
}

CString& CString::left( char c )
{
    int i;
    for( i = 0; i < n; i++ )
        if( p[i] != ' ' )
            break;
    int j;
    if( i < n )
        for( j = 0; j < n; j++ )
            p[j] = j+i < n ? p[j+i] : c;
    for( j = n-1; j >= 0; j-- )
        if( p[j] != ' ' && p[j] != 0 )
            break;
        else
            p[j] = c;
    return *this;
}

CString::operator==( const char* s ) const
{
    return size()==strlen( s ) && !memcmp(s,p,size());
}

CArray::CArray( const CArray& t ) :
    CType( t ), array( new CType*[ t.n ]),
    n( t.n ), structure( t.structure ?
        (CArray*)(t.structure->copy()) : 0 )
{
    for( int i = 0; i < n; i++ )
        array[i] = t.array[i]->copy();
}

CArray::~CArray()
{
    clear();
    delete structure;
}

void CArray::add( CType *t )
{
    CType **q = new CType*[ n+1 ];
    for( int i = 0; i < n; i++ )
        q[i] = array[i];
    q[ n++ ] = t;
    delete[] array;
    array = q;
}

void CArray::clear()
{
    for( int i = 0; i < n; i++ )
        delete array[i];
    delete[] array;
    array = 0;
    n = 0;
}

CArray::compare( const CType& t ) const
{
    if( t.type() != idArray )
        return 1;
    CArray &a = (CArray&) t;
    if( n != a.n )
        return 1;
    for( int i = 0; i < n; i++ )
        if( array[i]->compare( *a.array[i] ))
            return 1;
    return 0;
}

void CArray::print( ostream& o ) const
{
    o << '{';
    for( int i = 0; i < n; i++ )
    {
        o << *array[i];
        if( i!=n-1 )
            o << ',';
    }
    o << '}';
}

CType& CArray::operator = ( const CType& t )
{
    if( t.type()==idArray )
        return operator=( (CArray&)t );
    return CType::operator=( t );
}

CArray& CArray::operator = ( const CArray&t )
{
    clear();
    for( int i = 0; i < t.n; i++ )
        add( t.array[i]->copy());
    return *this;
}

CType& CArray::operator[]( int i )
{
    if( i<0 || i>=n )
    {
        cerr << "Wrong index in array "
             << "[" << i << "]" << endl;
        return *this;
    }
    if( !array[i] )
    {
        cerr << "Element not defined "
             << "[ << i << ]" << endl;
        return *this;
    }
    return *array[i];
}

CType& CArray::operator[]( const char* s )
{
    return operator[]( index( s ));
}

CArray::index( const char* s ) const
{
    if( !structure )
    {
        cerr << "No structure in array "
             << "[" << s << "]" << endl;
        return -1;
    }
    for( int i = 0; i < structure->n; i++ )
        if( structure->array[i] &&
            structure->array[i]->type()==idString &&
                ((CString&)(*structure->array[i]))==s )
            return i;
    cerr << "Field not found " << s << endl;
    return -1;
}

CArray::remove( int i )
{
    if( !n || i<0 || i>=n )
    {
        cerr << "Wrong index in array "
             << "[" << i << "]" << endl;
        return 1;
    }
    CType** t = new CType*[n-1];
    for( int j = 0; j < n-1; j++ )
        t[j] = j<i ? array[j] : array[j+1];
    delete array[i];
    delete[] array;
    array = t;
    n--;
    return 0;
}

int CDate::MonthIsFirst = 0;

inline void dateOut( ostream& o, int n )
{
    if( !n )          o << "  ";
    else if( n < 10 ) o << '0' << n;
    else              o << n;
}

void CDate::print( ostream& o ) const
{
    dateOut( o, MonthIsFirst ? m : d ); o << '/';
    dateOut( o, MonthIsFirst ? d : m ); o << '/';
    dateOut( o, y % 100 );
}

#define PRE_DATE i >> c; if( c!='0' ) i.putback(c); n=0; i >> n;
void CDate::get( istream& i )
{
    char c; int n;
    PRE_DATE; d = n; i >> c;
    PRE_DATE; m = n; i >> c;
    PRE_DATE; y = n;
    if( y < 100 && y )
        y += 1900;
    if( MonthIsFirst )
    {
        int tmp = d;
        d = m;
        m = tmp;
    }
}

static isleap( unsigned y )
{
    return y%400==0 || (y%4 == 0 && y%100 != 0);
}

static m2d( int m )
{
    return (m*3057-3007)/100;
}

static long y2d( int y )
{
    return y*365L+y/4-y/100+y/400;
}

long CDate::i() const
{
    long n = d+m2d( m );
    if ( m > 2 )
        n -= isleap(y) ? 1 : 2;
    n += y2d( y-1 );
    return n;
}

void CDate::set( long n )
{
    int i;
    for( i = (int)(n*400L/146097L );
        y2d( i ) < n; i++ ){}
    y = i;
    i = (int) (n - y2d( i-1 ));
    if( i > 59 )
    {
        i += 2;
        if( isleap( y ))
            i -= i > 62 ? 1 : 2;
    }
    m = (int)(( i*100+3007 )/3057);
    d = i - m2d( m );
}

CDate::CDate() : CType( idDate )
{
    time_t timer = time( 0 );
    struct tm *t = localtime( &timer );
    y = t->tm_year+1900;
    m = t->tm_mon+1;
    d = t->tm_mday;
}

