     {  M C 6 8 7 0 5  -  S e r i e s   A s s e m b l e r

                 D. R. Brooks      April 1988

                 Code mnemonics as Motorola, plus MSG 'string' defines msg.
                 All expressions are Absolute.
                 The assembler is one-pass, into core.
                 Operand expressions may include
                    Operands:
                       Symbol      - a pre-declared name
                       Dec. Const. - decimal digits
                       Hex. Const. - '$' then hex. digits
                       Char Const. - 1 or 2 chars. in quotes, as 'AB'
                       Current Loc - '$', not followed by hex. digit

                    Operators:
                       ( ) ^ * / & | ~ + -   in order of precedence

                    Logical Operators:
                       &   And
                       |   Or
                       ~   Exclusive Or

                 Address expressions may be preceded by one undefined symbol,
                   followed by '+' or '-' (no other operator allowed).

                 Immediate operands are preceded by '#'
                 Hexadecimal constants are preceded by '$'
                 '$' (not followed by hex.) denotes Current Address

                 If several errors occur on one line, only the first is logged

           NOTE: The error "1-Byte address too large" will be flagged from a
                 forward reference expression, when the expression is RESOLVED
                 Look immediately above the flagged code line, for the list of
                 locations here resolved; one of them will be the error.
                 From non-forward referenced expressions, the "too large"
                 will be flagged at the line containing the expression.


           PSEUDO-OPS: The following Pseudo-ops. are supported:
                 INCLUDE  'file'    - Include named file in the assembly
                 TITLE  'string'    - Use <string> in page title (also Ejects)
                 EJECT              - Force listing to new page
                 PAGE  n1, n2       - Sets n1 lines/page, n2 cols./line.
                 CODE               - Selects the EPROM memory bank pointer
                 DATA               - Selects the  RAM  memory bank pointer
                 IF                 - Conditional Assembly, op-2     = op-3
                 IFNOT              -  ditto              , op-2 not = op-3
                 ENDIF              -  ditto, end conditional block
                 LISCN              - Allows listing suppressed cond. asm.
                 NOLCN              - Inhibits ditto [LISCN is default]
 ***********************************************************************}

Function assemble :boolean;          {Assembler mainline}

{**** Global data declarations, used by all sections ****}
type
   argt        = string[128];
   ttype       = (delimiter, variable, number, hexnum, charconst, curlocn);
                                                   {Expression token types}

   dictlink    = ^dictrecord;         {Items to maintain the Symbol Dict'ry}
   dictrtype   = (declared, undechdr, fwdref); {Dictionary record types}
   memaloc     = (signed, unsigned, gword);   {Generate byte or word for item}

   dictrecord  = record               {Variant record for Symbol Dictionary}
      link     : dictlink;                {Pointer to next in list}
      dicsymb  : symbol;                  {The subject Symbol}
      case curdictype : dictrtype of      {Record-type Indicator variable}
         declared:                    {Symbol fully Declared}
            (dclocn   : integer);         {its Memory location or value}
         undechdr:                    {Undeclared Symbol ie Fwd. Referenced}
            ();                           {No additional data}
         fwdref:                      {Forward-References to the Above}
            (offset   : integer;          {Any relative offset}
             mspace   : memaloc;          {Memory space allocated}
             frlocn   : integer);         {Memory locn. of the Reference}
         end;                         {End the CASE}

const
                           {Machine Addressing mode permissions}
      INH1  : integer =   1;
      IMM1  : integer =   2;        {The modes are stored in "optable", }
      DIR1  : integer =   4;        { as bit-mapped items. }
      EXT1  : integer =   8;
      REL1  : integer =  16;
      BSC1  : integer =  32;
      BTB1  : integer =  64;
      IX1   : integer = 128;
      IX11  : integer = 256;
      IX12  : integer = 512;

                           {The addressing modes, as Machine opcode variants}
   varINH  : integer = $00;
   varIMM  : integer = $20;
   varDIR  : integer = $30;
   varEXT  : integer = $40;
   varREL  : integer = $00;
   varBSC  : integer = $00;
   varBTB  : integer = $00;
   varIX   : integer = $70;
   varIX1  : integer = $60;
   varIX2  : integer = $50;


var
   {Data items local to Assembler}
   rdline, lstline   : string[255];  {Line as input, & copy for Listing}
   stlab, opcode     : symbol;       {Statement label (if any), & opcode}
   operand           : array[1..3] of argt;    {Unpack the argument fields}
   liston,                           {Writing a LIST file}
   listcon,                          {Conditional listing suppressed}
   abortrun,                         {Job cancelled by ^C }
   badline,                          {Source error by Lexical Analyser}
   condasm,                          {Conditional assembly - asm. if TRUE}
   memoflo           : boolean;      {Code overflows memory}
   pausepos,                         {Position of "Pause" message}
   currentopcode,                    {Point to current op-code table entry}
   opcodebase,                       {Currently-assembling op-code}
   listfield,                        {Object-code listing field}
   errorcode,                        {Coded error}
   grp,                              {1 or 2 - Data or Code memory bank}
   thiscode          : integer;      {Start of current Instr.}
   dicthead,                         {Dictionary-list head}
   dicptr            : dictlink;     {Supplementary pointer}
   includit          : argt;         {Name of INCLUDE file, if any}
   cpa               : array[1..2] of integer;  {DATA & CODE pointers}

                                  {Variables for the Listing-file page layout}
   pagewidth,                        {No. of columns printable}
   linesperpage,                     {Total physical lines}
   listpageline,                     {Line-no. on this page}
   pagenum           : integer;      {Current page number}
   pagetitle         : argt;         {Title block}

                                     {Variables for the Expression Parser}
   prog              : argt;
   token             : symbol;
   toktype           : ttype;
   code, t           : integer;
   plusminus,                        {Required +/- next}
   undefok           : boolean;      {Undefined (forward-ref.) symbol OK?}
   fwdptr            : dictlink;     {Pointer to such fwd. ref. entry}


{******************* Simple Utility Subprograms ************************
 N.B. Bit-7 is used by the Lexical Analyser, to mark chars. within quoted
      strings. As subsequently seen, all lines have the bounding, and inner
      duplicate, quotes removed, and string elements marked by Bit-7 ON.
      Hence string contents will NEVER match any standard character tests.}

Function hard (c :char) :byte;       {Deletes the string-marker - Bit 7}
begin
   hard := ord(c) and $7f;
   end;

Function soft (c :char) :char;       {Applies the string-marker - Bit 7}
begin
   soft := chr(ord(c) or $80);
   end;

Function issoft(c:char) :boolean;    {Is this char. quoted?}
begin
   issoft := c >= #128;
   end;

Function listing : boolean;          {Test all listing controls}
begin
   listing:= liston and (listcon or condasm);
   end;

Procedure flagerror(x :integer);     {Flags error, if none already}
begin
   if (errorcode = 0) then errorcode:= x;
   end;

Function todisk : boolean;           {Listing to disk}
begin
   todisk:= (listname <> 'CON:') and
            (listname <> 'con:');
   end;         

Procedure listheading;               {Writes the Page-titles}
const
   formfeed : char = ^L;

var
   titlesize : integer;
   
begin
   if (listing and (listpageline > (linesperpage-6))) then begin
      if  ((pagenum >0) and todisk) then write(lstfile,formfeed);
      pagenum:= pagenum+1;
      listpageline:= 0;
      titlesize:= pagewidth-49;
      writeln(lstfile,'MC68705 Assembler ver ', version, '  ',
                       copy(pagetitle,1,titlesize):titlesize,
                       today:10,
                       ' Page', pagenum:4);
      writeln(lstfile);
      end;
   listpageline:= listpageline+1;
   end;

Procedure logerror;                  {Outputs error to Listing}
type
   ermesg      = string[32];

const
   ercode      : array[1..19] of ermesg = (        {Table of error messages}
                 'Invalid Char. in Label/Opcode',
                 'Unrecognised Op-Code',
                 'Undefined Symbol(s):',
                 'Unlabelled EQU',
                 'Multiple Definition',
                 'Dictionary Format Error',
                 '1-Byte Address too Large',
                 'ORG must not be labelled',
                 'MSG requires a quoted string',
                 'Syntax error in Expression',
                 'Unbalanced brackets in Exprn.',
                 'Expected an Expression',
                 'Invalid Char. in Expression',
                 'Program under/overflows Memory',
                 'Forward Reference not allowed',
                 '"X" is a Reserved Symbol',
                 'Invalid Addressing Mode',
                 'Bit Number must be 0-7',
                 'INCLUDE file not found' );

begin
   listheading;
   if listing then writeln(lstfile, '* * * * * * * * *',
                      errorcode:4,' **Error: ',ercode[errorcode]);
   errcount:= errcount+1;
   if todisk or (not liston) then begin
      writeln('     ',lstline);
      writeln('*****',errorcode:4,' **Error: ',ercode[errorcode]);
      end
   end;

Procedure listsource(level, linum :integer);         {Source Listing driver}
const
   bankind : array[1..2] of char =('D','C');         {Memory bank indicator}

begin
   if listing then begin
      listheading;
      write(lstfile,level:2,':',linum:4,' ');        {Write to Listing File}
      if prefix[2] in ['0'..'2'] then write(lstfile,bankind[grp])
                                 else write(lstfile,' ');
      writeln(lstfile,prefix,copy(lstline,1,pagewidth-30));
      end
   end;

{********************* Symbol-Dictionary Manager **********************
  The Symbol Dictionary is maintained as a singly-linked list, in the Heap
  (freestore) space. See above for the record formats used.
  Records are stored in ascending collating sequence, of Symbols.
  Each distinct symbol will have either a DECLARED or UNDECLARED-HEADER
  record, as appropriate. An Undec-Hdr will be followed by one FORWARD-REF
  record, for each reference outstanding, to that symbol. The FWDREF records
  have a copy of the Symbol name, purely to make list-scanning easier. When
  the symbol is declared, the FWDREF records are removed from the list.
  FWDREF records may appear in any order, following their UNDEC-HDR record.
  At start-up, the dictionary is initialised to contain low & high-values.}

Function makeblank(name :symbol; prev, foll :dictlink) :dictlink;
var                     {Builds a new dictionary entry between PREV & FOLL}
   temp : dictlink;

begin
   new(temp);                          {Allocate space}
   prev^.link:= temp;
   temp^.link:= foll;                  {Link into list}
   makeblank := temp;
   temp^.dicsymb:= name;
   temp^.curdictype:= undechdr;        {Default type for creation}

   end;


Function dictionary(name :symbol; reference :boolean) :dictlink;
 { Routine to scan Dictionary and create new item if not found.
   If REFERENCE, and the symbol is undeclared, a new Fwd-Ref. is also built.
   DICTIONARY returns a pointer to the Fwd-Ref (if any), else to the Symbol}

var
   tempd1, tempd2 :dictlink;

begin
   tempd1:= dicthead;                  {Traverse list to find Symbol}
   tempd2:= dicthead^.link;
   while (name > tempd2^.dicsymb) do begin
      tempd1:= tempd2;
      tempd2:= tempd2^.link;
      end;

   if name <> tempd2^.dicsymb then     {Create new Symbol entry}
      tempd2:= makeblank(name, tempd1, tempd2);

   dictionary:= tempd2;

   if (reference and                           {May need a forward ref.}
      (tempd2^.curdictype <> declared)) then begin  {... only if undeclared}
         tempd1:= tempd2^.link;                {Build fwd-ref AFTER tempd2}
         tempd2:= makeblank(name, tempd2, tempd1);
         tempd2^.curdictype:= fwdref;               {Fill in the record}
         dictionary:= tempd2;
         end;

   end;

{********************* Arithmetic-Expression Parser *******************
    The Parser is of the Recursive-Descent type, based on that by Schildt
  ("Advanced Turbo-Pascal programming & techniques").
    It has been modified to handle integers exclusively, and enhanced to
  recognise logical operators, hexadecimal & character constants and 
  Symbol references.
    Symbols, if already defined, are considered equivalent to integer
  constants. An undefined symbol reference may only occur as the first
  token in the expression, in which case the rest of the expression
  yields a forward-ref. offset, and the entire expression is equivalent
  to a forward reference. No distinction is made between symbols defined
  in Data or Code spaces. }


Function isdelim (ch :char) :boolean;    {is 'ch' a valid delimiter}
begin
   isdelim:= (pos(ch,' +-/*^&|~()') <> 0);
   end;

Procedure gettoken;                      {get next token from source line}
var
   temp : argt;

begin
   token:= '';                           {null initially}
   if prog[t] = ' ' then
      token:= ' '                        {End of Expression Marker}
   else
   if ((prog[t] in ['+', '-']) or
      ((not plusminus) and (prog[t] in 
            ['*', '/', '&', '|', '~', '^', '(', ')'] ))) then begin
      toktype:= delimiter;               {Delimiter/Operator}
      token  := prog[t];                 {After fwd. ref., only +/- allowed}
      t:= t+1;
      plusminus:= false;                 {Turn off the +/- control}
      end
   else
   if (prog[t] in symchar) then begin      {Symbol Reference}
      while ((prog[t] in symchar) or
             (prog[t] in digit)) do begin
         token:= token+prog[t];          {build token}
         t:= t+1;
         end;
      toktype:= variable;
      end
   else
   if (prog[t] in digit) then begin      {Decimal Constant}
      while (prog[t] in digit) do begin
         token:= token+prog[t];          {build number}
         t:= t+1;
         end;
      toktype:= number;
      end
   else
   if (prog[t] = '$') then begin         {'$' - Current Locn. or Hex. Const.}
      t:= t+1;                                  {skip the '$'}
      if (hex(prog[t]) >= 0) then begin         {Hexadecimal String}
         while (hex(prog[t]) >= 0) do begin
            token:= token+prog[t];
            t:= t+1;
            end;
         toktype:= hexnum;
         end
      else begin                                {Current Location}
         token:= '$';
         toktype:= curlocn;
         end
      end
   else
   if (issoft(prog[t])) then begin       {Quoted String - Char. Constant}
      while (issoft(prog[t])) do begin
         token:= token+prog[t];          {collect the quoted string}
         t:= t+1;
         end;
      toktype:= charconst;
      end
   else begin                            {Invalid char.}
      flagerror (13);
      t:= t+1;                           {Skip it}
      end
   end;

Function pwr(a,b :integer) :integer;           {raise a to power b}
var
   t    : integer;
   temp : integer;

begin
   if b=0 then
      pwr:= 1
   else begin
      temp:= a;
      for t:= b downto 2 do a:= a*temp;
      pwr:= a;
      end
   end;

Procedure arith(op :char; var result, operand :integer); {perform arith. functs.}
begin
   case op of
      '+' : result:= result + operand;
      '-' : result:= result - operand;
      '*' : result:= result * operand;
      '/' : result:= result div  operand;
      '^' : result:= pwr(result, operand);
      '&' : result:= result and operand;
      '|' : result:= result or operand;
      '~' : result:= result xor operand;
      end
   end;

{******************* Expression Parser ********************}

Procedure    level2(var result :integer);  forward; {Recursive Declaration}

Procedure primitive(var result :integer);  {Action: Operands}
var
   dicent : dictlink;                      {Pointer to Dict. for any Symbol}
   temp   : integer;

begin                                      {Returns numeric value}
   case toktype of
      number:
         val(token,result,code);           {Simple decimal const.}
      variable: begin
         dicent:= dictionary(token,undefok);  {Search Dictionary for symbol}
         if (dicent^.curdictype = declared) then
            result:= dicent^.dclocn
         else begin
            result:= 0;
            if (undefok) then begin
               prefix[20]:= '?';           {Flag fwd. ref.}
               fwdptr:= dicent;
               plusminus:= true;           {Only legal oprs. after fwd. ref.}
               end
            else
               flagerror (15);             {Fwd. is illegal}
            end
         end;
      hexnum: begin
         result:= 0;                       {Hexadecimal Constant}
         for temp:= 1 to length(token) do
            result:= (result shl 4) + hex(token[temp]);
         end;
      charconst: begin
         result:= 0;                       {Character Constant}
         for temp:= 1 to length(token) do
            result:= (result shl 8) + hard(token[temp]);
         end;
      curlocn:
         result:= thiscode;                {Current Location}
      else
         flagerror (10);                   {Exprn. syntax error}
      end;                                 {End the CASE}
   undefok:= false;                        {No more forward refs. allowed}
   gettoken;
   end;

Procedure level6(var result :integer);  {Action: Brackets }
begin
   if (token[1]='(') and (toktype=delimiter) then begin
      gettoken;                      {parenthesized expression}
      level2(result);
      if token[1]<>')' then flagerror (11); {brackets unbalanced}
      gettoken;
      end
   else
      primitive(result);
   end;

Procedure level5(var result :integer);  {Action: Delimiters }
var
   op   : char;

begin
   op:= ' ';
   if ((toktype=delimiter) and ((token[1]='+') or (token[1]='-'))) then begin
      op:= token[1];                 {unary plus or minus}
      gettoken;
      end;
   level6(result);
   if op='-' then result:= -result;  {unary minus}
   end;

Procedure level4(var result :integer);  {Action: ^ }
var
   hold : integer;

begin
   level5(result);
   if token[1]='^' then begin
      gettoken;
      level4(hold);
      arith('^', result, hold);
      end
   end;

Procedure level3(var result :integer);  {Action: * / & | ~ }
var
   op   : char;
   hold : integer;

begin
   level4(result);
   op:= token[1];
   while op in ['*', '/', '&', '|', '~' ] do begin
      gettoken;
      level4(hold);
      arith(op, result, hold);
      op:= token[1];
      end
   end;

Procedure level2;                          {Action: + - }
var
   op   : char;
   hold : integer;

begin
   level3(result);
   op:= token[1];
   while ((op='+') or (op='-')) do begin
      gettoken;
      level3(hold);
      arith(op, result, hold);
      op:= token[1];
      end
   end;

Procedure getexp(field :integer; var result :integer; fwdok :boolean);
                                           {Parser Mainline evaluator}
begin
   plusminus:= false;
   prog:= operand[field];                    {Choose the operand}
   undefok:= (fwdok and (prog[1] in symchar)); {Forward ref. 1st. only}
   fwdptr:= nil;                             {No fwd. ref. yet}
   t:= 1;                                    {Intz. string pointer}
   gettoken;
   if length(token)<>0 then level2(result)
                       else flagerror (12);  {Exprn. not found}
   end;

{********************* Object-Code Generator Section ******************
  The Object-Code Generator receives a partially digested source line,
  and emits the necessary object formats, to the pseudo-EPROM (a byte
  array in main store).
  This section also handles Symbol-Table maintenance, forward references,
  etc.}

Function cando(locn :integer; data :byte) :boolean; {Try to post to memory}
begin
   if ((locn >= 0) and (locn < (memmax+1))) then begin
      memory[locn]:= data;
      cando:= true;
      end
   else begin
      if (not memoflo) then begin
         flagerror(14);            {Memory oflo. error - log once only}
         memoflo:= true;
         end
      end
   end;

                                   {Place a 1- or 2-byte value in memory}
Procedure placeval (size :memaloc; mempos, value :integer);
var                                {Used normally to resolve fwd. refs.}
   dummy  :boolean;

begin
   case size of
      gword: begin                 {a WORD value}
                dummy:= cando(mempos, hi(value)); {NB byte order on 6800}
                dummy:= cando(mempos+1, lo(value));
                end;
     signed: begin                 {a signed BYTE value}
                if ((value < -128) or (value > 127)) then
                   flagerror (7);                 {out of range}
                dummy:= cando(mempos, lo(value));
                end;
   unsigned: begin                 {unsigned BYTE value}
                if ((value < 0) or (value > 255)) then
                   flagerror (7);                 {out of range}
                dummy:= cando(mempos, lo(value));
                end
      end
   end;

Procedure defsymbol (dicentry :dictlink; address :integer); {Declare a Symbol}
var
   dicstep :dictlink;
   resolv  :string[20];

begin
   case dicentry^.curdictype of
      declared: flagerror (5);            {Multiply declared}
      fwdref:   flagerror (6);            {Dictionary Error}
      undechdr: begin
                if (dicentry^.dicsymb ='X') then
                   flagerror (16);           {"X" is reserved for Index modes}
                dicentry^.curdictype := declared; {Make it "declared"}
                dicentry^.dclocn:= address;
                dicstep:= dicentry^.link;    {Resolve forward refs.}
                while (dicstep^.curdictype = fwdref) do begin
                   resolv:= prefix;                  {Place in Listing}
                   prefix:= '@@@@ =..xx Resolved';
                   hexword(1,dicstep^.frlocn);
                   if (dicstep^.mspace = gword) then
                      hexword(7,(address+dicstep^.offset))   {List as Byte/Word}
                   else
                      hexbyte(9,(lo(address+dicstep^.offset)));
                   listheading;
                   if listing then writeln(lstfile,'    @',prefix);
                   prefix:= resolv;
                   placeval(dicstep^.mspace,
                            dicstep^.frlocn,
                            address+dicstep^.offset); {Post value in Mem.}
                   dicentry^.link:= dicstep^.link;
                   dispose(dicstep);                  {Discard entry}
                   dicstep:= dicentry^.link;
                   end
                end
      end
   end;

Procedure definelabel;                    {Defines any label on Current Line}
begin
   if stlab = '' then begin
      if opcode = 'EQU' then flagerror (4);      {EQU must be labelled}
      end
   else begin
      if opcode = 'ORG' then flagerror (8);      {ORG must not be labelled}
      dicptr:= dictionary(stlab,false);          {name in dictionary}
      if opcode <> 'EQU' then defsymbol(dicptr,thiscode); {Defined, unless EQU}
      end
   end;

{******************** Code-Generator Slave Routines **********}

Procedure post (value :byte);             {Lay down 1 byte of code}
begin
   if cando(cpa[grp], value) then begin
      cpa[grp]:= cpa[grp]+1;
      if (listfield < 19) then begin
         hexbyte(listfield,value);        {Object-code listing}
         listfield:= listfield+3;
         end
      end
   end;

Procedure postop(value :byte);            {Same, but list instr. time also}
begin
   post(value);                           {Lay down code}
   with ExTable[value div 16] do hexchar(19,cycles[value and 15]);
   end;

Procedure genbytex(value :integer; class :memaloc);     {Explicit BYTE data}
begin
   if (fwdptr <> nil) then begin          {Do forward reference}
      fwdptr^.offset:= value;
      fwdptr^.mspace:= class;
      fwdptr^.frlocn:= cpa[grp];
      value:= 0;
      end;
   placeval(class,cpa[grp],value);        {Range checks in 1 byte}
   post(lo(value));                       {Update pointers}
   end;

Procedure genbyte(fldnum, incr : integer; class :memaloc);
var                                           {BYTE data, optional offset}
   value : integer;

begin
   getexp(fldnum,value,true);             {Get expression}
   value:= value+incr;                    {Offset if any}
   genbytex(value,class);
   end;

Procedure genwordx(value :integer);       {Explicit WORD value}
begin
   if (fwdptr <> nil) then begin          {Link up fwd. reference}
      fwdptr^.offset:= value;
      fwdptr^.mspace:= gword;
      fwdptr^.frlocn:= cpa[grp];
      value:= 0;
      end;
   post(hi(value));                       {Store integer: note byte order}
   post(lo(value));
   end;

Procedure genword(fldnum :integer);  {Lay down WORD data, fwd. ref. if reqd.}
var
   value : integer;

begin
   getexp(fldnum,value,true);             {Interpret field as data}
   genwordx(value);
   end;

Procedure genmsg;                    {Do the MSG Command}
var
   value :byte;
   ptr   :integer;

begin
   ptr:= 1;
   if issoft(copy(operand[1],1,1)) then begin
      while issoft(copy(operand[1],ptr,1)) do begin
         value:= hard(copy(operand[1],ptr,1));
         post(value);
         ptr:= ptr+1;
         end
      end
   else
      flagerror (9);                 {MSG must have a string}
   end;

Procedure blankblock;                {Reserve blank memory block}
var
   value : integer;

begin
   getexp(1,value,false);            {Get size - no fwd. refs.}
   cpa[grp]:= cpa[grp]+value;        {Advance pointer}
   hexword(16,value);                {Listing}
   end;

Procedure makeorigin;                {Assign code Origin}
var
   value : integer;

begin
   getexp(1,value,false);            {Get the ORG - no fwd. ref.}
   cpa[grp]:= value;                 {Load from there}
   hexword(16,value);                {Listing}
   end;

Procedure makeequate;                {Generate an EQU Function}
var
   dicpt : dictlink;
   value : integer;

begin
   getexp(1,value,false);            {Equate expression - no fwd. refs.}
   if (stlab <> '') then begin
      dicpt:= dictionary(stlab,false);  {search dict., do NOT create fwd. ref.}
      defsymbol(dicptr,value);
      hexword(16,value);             {EQU value for listing}
      end
   end;

{***************** Assemble Machine Instructions **************

 The following routines (and the Global address-mode constants) are highly
 target-machine dependent. These routines, and the separate op-code file,
 would need changing for a different target machine.
 Most of the rest could be used with almost any byte-oriented machine,
 provided care is taken over the ordering of 2-byte data items.}

Function bitoper  : integer;  {Returns 2nd byte of Bit instructions}
var
   bitnum : integer;

begin
   getexp(1,bitnum,false);      {Get bit no., must be absolute}
   if (not (bitnum in [0..7])) then begin
      flagerror (18);                   {Out of range}
      bitnum:= 0;
      end;

   opcodebase:= opcodebase + bitnum*2;  {Adjust opcode}
   getexp(2,bitnum,true);               {Operand address}
   bitoper:= bitnum;
   end;


Function longaddr(                     {Long-Mode Addressing?}
                  indexed :boolean;    {X-register offset used?}
                  target  :integer)    {Target address/offset}
                  :boolean;          
var
   switch,              {Decision-table key}
   adrmode,             {Op-code Offset for addressing mode}
   modeflg,             {Standardised legal-modes indicator}
   direct,              {Code offset for 1-byte addressing}
   extend : integer;    {Code offset for 2-byte addressing}

const                   {Bit-fields for Decision table key}
   defined : integer = 4;  {Operand fully defined}
   twobyte : integer = 2;  {Operand known larger than 1 byte}
   extok   : integer = 1;  {Two-byte addresses are legal}

begin
   with optable[currentopcode] do begin
      if indexed then begin
         modeflg:= modes shr 6;         {Select the Indexed opcodes}
         direct := varIX1;
         extend := varIX2;              {the opcode variants}
         end
      else begin
         modeflg:= modes;               {Use the Direct/Extended modes}
         direct := varDIR;
         extend := varEXT;
         end;

      switch:= 0;                   {Set up the decision-table pointer}
      if (fwdptr=nil)               then switch:= switch+defined;
      if (not (target in [0..255])) then switch:= switch+twobyte;
      if ((modeflg and EXT1)<>0)     then switch:= switch+extok;
      if ((modeflg and DIR1)=0)      then switch:= 8;  {Illegal modes}

      case switch of                {Decision table for addressing mode}
         0: adrmode:= direct;
         1: adrmode:= extend;
         2: adrmode:= direct;
         3: adrmode:= extend;
         4: adrmode:= direct;
         5: adrmode:= direct;
         6: begin
               flagerror (7);         {Operand oversize}
               adrmode:= direct;
               end;
         7: adrmode:= extend;
         8: begin
               flagerror (17);        {Illegal addressing mode}
               adrmode:= extend;
               end;
         end;                         {End the CASE}


      opcodebase:= opcodebase + adrmode;  {Adjust the opcode}
      longaddr:= (adrmode = extend);
      end
   end;


Procedure makeinstruction;            {Build a Machine Instruction: mainline}
var
   byte1, byte2 : integer;

begin
   with optable[currentopcode] do begin                  {Branch by Modes}
      opcodebase:= stub;              {Start of the opcode}

      if ((modes and INH1) <>0) then begin                {INHERENT}
         postop(stub);
         end
      else
      if ((modes and BSC1) <>0) then begin                {BIT SET/CLEAR}
         byte2:= bitoper;             {Get bit no. & operand}
         postop(opcodebase);
         post(byte2);
         end
      else
      if ((modes and BTB1) <>0) then begin                {BIT TEST & BRANCH}
         byte2:= bitoper;
         postop(opcodebase);
         post(byte2);
         genbyte(3,-(thiscode+3),signed);    {Target: relative}
         end
      else
      if (copy(operand[1],1,1) ='#') then begin          {IMMEDIATE}
         delete(operand[1],1,1);      {Strip the '#'}
         if ((modes and IMM1) =0) then flagerror (17)
                                 else begin
            postop(opcodebase+varIMM);
            genbyte(1,0,unsigned);           {Immediate argument}
            end
         end
      else
      if ((operand[1] =' ')   and
          (operand[2] ='X ')) then begin                 {INDEXED, no offset}
         if ((modes and IX1) =0) then flagerror (17)
                                else postop(stub +varIX);
         end
      else
      if ((modes and REL1) <>0) then begin                {RELATIVE}
         postop(opcodebase);
         genbyte(1,-(thiscode+2),signed);    {Make it relative}
         end
      else begin
         getexp(1,byte1,true);                  {INDEXED, DIRECT, EXTEND}
         if longaddr((operand[2]='X '),byte1) then begin
            postop(opcodebase);                            {Long Address}
            genwordx (byte1);
            end
         else begin
            if ((operand[2]='X ') and     {Indexed, and}
                (fwdptr=nil) and         {fully defined, and}
                (byte1=0)) then          {address =0}
                   postop(stub+varIX)      {Use the short-index mode}
            else begin
               postop(opcodebase);                         {Short Address}
               genbytex(byte1,unsigned);
               end
            end
         end
      end
   end;

{******************** Pseudo-Opcode Functions *****************}

Function quoted(fld :integer) :argt;              {Extracts quoted string}
var
   temp :argt;
   ptr  :integer;

begin
   temp:= operand[fld];
   ptr := 1;
   while issoft(temp[ptr]) do begin
      temp[ptr]:= chr(hard(temp[ptr]));
      ptr:= ptr+1;
      end;
   delete(temp,ptr,length(temp)-ptr+1);
   quoted:= temp;
   end;

Procedure includefile;                            {Run an Include file}
begin
   includit:= quoted(1);
   end;

Procedure ejectpage;                              {Throw listing to new page}
begin
   listpageline:= 1000;
   end;

Procedure settitle;                               {Assign page title}
begin
   pagetitle:= quoted(1);
   ejectpage;                    {Page eject also}
   end;

Procedure pagemode;                               {Assign page height & width}
begin
   getexp(1,linesperpage,false);
   getexp(2,pagewidth,false);
   end;

Procedure setcond(sense :boolean);                {Sets conditional assy. switch}
var
   x, y : integer;
   
begin
   getexp(1,x,false);            {Get the arguments}
   getexp(2,y,false);            {Fwd. references illegal, here}
   condasm:= (sense = (x = y));
   end;

{******************** Code Generator Mainline *****************}

Procedure makecode;                         {Object-Code Generator}
var
   point, index            : integer;
   entry                   : symbol;
begin
   if condasm then begin
      thiscode:= cpa[grp];              {Move up pointer to new instr.}
      definelabel;                      {Define any label present}
      listfield:= 7;                    {1st. object-code listing col.}
      end;
   
   point:= 64;                       {Begin binary search for the op-code}
   index:= 32;
   with optable[point] do entry:= mnemonic;
   while ((entry <> opcode) and (index >= 0)) do begin
      if   entry  > opcode then point:= point - index
                           else point:= point + index;
      if index > 0 then index:= index div 2
                   else index:= -1;
      with optable[point] do entry:= mnemonic;
      end;

with optable[point] do begin
 if (stub= -15) and
    (index >= 0)   then condasm:= true;    {ENDIF done directly}
 if condasm then begin                 {All others subject to cond. assy.}
      if index < 0 then flagerror (2)               {Unrecognised opcode}
                   else
      case stub of                   {If -ve, special functions}
         -1: makeorigin;             {ORG Statement}
         -2: genbyte(1,0,unsigned);  {FCB (ie Byte Data) Statement}
         -3: genword(1);             {FDB (ie Word Data) Statement}
         -4: blankblock;             {RMB (ie Reserve Memory Block) Statement}
         -5: makeequate;             {EQU Statement}
         -6: genmsg;                 {MSG (ie ASCII Text) Statement}
         -7: includefile;            {INCLUDE extra source file}
         -8: settitle;               {TITLE: new title + Eject}
         -9: ejectpage;              {EJECT: throw listing to new page}
        -10: pagemode;               {PAGE: assign listing format}
        -11: grp:= 2;                {CODE: set bank indicator}
        -12: grp:= 1;                {DATA: ditto}
        -13: setcond(true);          {IF:    condasm:= fld2=fld3 }
        -14: setcond(false);         {IFNOT: condasm:= not (fld2=fld3) }
        -15: condasm:= true;         {ENDIF: duplicate, to validate entry}
        -16: listcon:= true;         {LISCN: list conditional-exclusions}
        -17: listcon:= false;        {NOLCN: do not list same}
        else begin                   {Non-negative, ie machine opcodes}
                currentopcode:= point;
                makeinstruction;
             end                     {End the Case ELSE}
         end;                        {End the CASE}
      if stub > -7 then
         hexword(2,thiscode);        {Listing: code address}
      end                            {Conditional assembly control}
    end                              {Record WITH block}
   end;                              {End the Procedure}

{****************** Lexical Analysis Section ****************************
  The Lexical Analyser removes such irrelevancies as comments, null lines,
  and string quotes. It offers the line, split into 3 (blank delimited)
  zones, to the Object-Code Generator.}

Procedure makesym(allalpha :boolean; source :argt; var target :symbol);
                                     {Forms Field 1 and 2 items}
var
   srpt :integer;

begin
   target:= copy(source,1,8);
   if length(source) > 0 then begin
      badline:= badline or (allalpha and not (source[1] in symchar));
      for srpt:= 1 to length(target) do
         badline:= badline or not ((target[srpt] in symchar)
                                or (target[srpt] in digit));
      end
   end;

{********* Lexical Analyser mainline *****************}

Function lexan
(srcname :argt; level :integer) :boolean;  {Lexical Analyser}
var                                           {Recursive on INCLUDES}
   ptr,fptr,linum                 : integer;
   istring, first, comment        : boolean;
   srcfile                        : text;

{********** Procedures to trap ^S and ^C keys *********}

function dotest :char;                       {Trap key, intercept ^C}
var
   x : char;

begin
   if keypressed then begin
      read(kbd, x);
      if (x = ^C) then abortrun:= true;
      dotest:= x;
      end
   else
      dotest:= ^L;         {No keystroke - anything except ^S will do}
   end;

procedure mpause(highlight :boolean);      {Set "Pause" in high or low video}
var
   savx, savy : integer;                   {Save cursor posn.}

begin
   savx:= wherex;
   savy:= wherey;
   setwin(0);                              {Large window}
   gotoxy(pausepos,1);                     {To the "Pause" line}
   if highlight then highvideo
                else lowvideo;
   write('Pause');
   lowvideo;
   setwin(6);
   gotoxy(savx,savy);                      {Restore screen}
   end;

procedure testkeys;                           {The actual keystroke tester}
var
   x : char;

begin
   x:= dotest;             {Look for a ^S - ^C trapped implicitly}
   if (x = ^S) then begin
      mpause(true);                      {Highlight "Pause"}
      repeat
         x:= dotest;                     {Get another key, or ^L if none}
         until ((x = ^S) or (x = ^C));   {Wait for a second ^S}
      mpause(false);                     {"Pause" in low video, again}
      end;
   end;

{******************************************************}

procedure showprogress;                       {Trace assembly, on CRT}
var
   savecursx, savecursy : integer;            {Save cursor posn.}

begin
   highvideo;
   savecursx:= wherex;
   savecursy:= wherey;
   setwin(0);                     {Larger window, here}
   gotoxy(2,5);                   {Trace of the Assembly}
   write('Include-File Depth', level:3, ', Line', linum:4);
   setwin(6);                     {Small window, again}
   gotoxy(savecursx, savecursy);
   lowvideo;
   end;

begin
assign(srcfile,srcname);
{$I-}                                         {Open, without error traps}
reset(srcfile);
{$I+}                                         {Normal error handling}
if (IORESULT =0) then begin
   lexan:= true;                              {Source file opened successfully}
   linum:= 0;
   condasm:= true;                            {Cond. assy. ON, initially}
   while not(eof(srcfile) or abortrun) do begin
      testkeys;                         {Check for ^S, ^C}
      includit:= '';                    {No INCLUDE file}
      prefix:='                    ';
      badline:= false;
      errorcode:= 0;
      readln(srcfile,rdline);           {Get the source line}

      lstline:= rdline;                 {Unaltered form, for the Lister}
      ptr:= 1;                             {Expand tabs, to tidy listing}
      while (ptr <= length(lstline)) do begin
         if (lstline[ptr] =^I) then begin
            lstline[ptr]:= ' ';
            ptr:= ptr+1;
            while (((ptr-1) mod 8) <>0) do begin
               insert(' ',lstline,ptr);
               ptr:= ptr+1;
               end
            end
         else ptr:= ptr+1;
         end;

      linum:= linum+1;                  {Source line count}
      if ((linum mod 16) =1) then showprogress;
      if length(rdline) >1 then begin
         rdline:= rdline + ' ';            {So the string handler works OK}
         istring:= false;
         ptr:= 1;                  {First, strip string-quotes}
         while ptr <= length(rdline) do begin
            if rdline[ptr]='''' then begin                   {Look for Quote}
               delete(rdline,ptr,1);                        {Kill it}
               if istring then begin                        {Already in string, so}
                  if rdline[ptr]<>'''' then istring:= false;    {check end}
                  end
               else begin
                  istring:= true;             {start of string}
                  end
               end;
            if istring then rdline[ptr]:= soft(rdline[ptr])
                       else if rdline[ptr] in lower then
                            rdline[ptr]:= upcase(rdline[ptr]);
            ptr:= ptr+1;
            end;
         delete(rdline,length(rdline),1);    {Then drop the temporary trailing space}

         for ptr:=1 to 3 do operand[ptr]:= ''; {Ready to split up into 3 fields}
         fptr:= 1; ptr:= 1; istring:=false;    {Point for first field}
         first:= true; comment:= false;
         while (ptr <= length(rdline)) and
               (fptr < 4) do begin
                  if rdline[ptr] in whitespace then begin
                     if not istring then begin
                        fptr:= fptr+1;
                        istring:= true;        {Start of whitespace}
                        end
                     end
                  else begin
                     istring:= false;          {Non-whitespace}
                     if (first and (rdline[ptr]='*')) then comment:=true;
                     first:= false;
                     operand[fptr]:= operand[fptr] + rdline[ptr];
                     end;
                  ptr:= ptr+1;
                  end;


         makesym (true,operand[1],stlab);
         makesym(false,operand[2],opcode);
         if badline then flagerror (1);

         operand[1]:= ''; operand[2]:= '';        {Split RHS into 3 fields}
         ptr:= 1;  fptr:= 1;
         while ((fptr < 3) and (ptr <= length(operand[3]))) do begin
            if copy(operand[3],ptr,1) =',' then
               fptr:= fptr+1
            else
               operand[fptr]:= operand[fptr] + copy(operand[3],ptr,1);
            ptr:= ptr+1;
            end;
         if ptr > 1 then delete(operand[3],1,ptr-1);
         for ptr:= 1 to 3 do operand[ptr]:= operand[ptr] + ' '; {Blank ends}

         if (not(comment) and (errorcode =0)) then makecode;
         end;

      listsource(level,linum);                       {Write to Listing File}

      if (includit <> '') then begin
         if (not lexan(includit,level+1)) then        {INCLUDE File}
            flagerror(19);                               {Flag not-found}
         end;

      if ((errorcode>0) and (not comment)) then logerror;
      end;
   close(srcfile);
   if abortrun then lexan:= false;
   end
else
   lexan:= false;                                 {Failed to open source file}
end;

{********************* Assembler Mainline ******************************}

begin
   liston:= (listname <> nofile);
   for grp:= 0 to memmax do
      memory[grp]:= 0;                {Flush out 'EPROM'}
   grp:= 2;                           {CODE Group by default}
   assemble:= false;                  {Trashed memory}
   listcon:= true;                    {Initially, list everything}
   cpa[1]:= 0;                        {Default start point}
   cpa[2]:= 0;                        {Both banks}
   memoflo := false;                  {No code oflo. yet}
                             {Build dummy first & last dictionary entries}
   new (dicthead);                     {The first...}
   dicthead^.dicsymb:= '   ';
   dicthead^.curdictype  := declared;
   dicthead^.dclocn := -1;
   new (dicptr);                       {And last entry}
   dicthead^.link   := dicptr;
     dicptr^.link   := nil;
     dicptr^.dicsymb:= '\\\';
     dicptr^.curdictype  := declared;
     dicptr^.dclocn := -1;
   {$I-}
   if liston then rewrite(lstfile);    {Protected open of List file}
   {$I+}
   if (IOResult =0) then begin
      linesperpage:= 66;               {Page layout defaults}
      pagewidth   := 79;
      listpageline:= 1000;
      pagetitle   := '';
      pagenum     := 0;

      abortrun:= false;
      errcount:= 0;
      gotoxy(1,1);
      highvideo;
      write(' Assembling...        ');
      lowvideo;
      write('^S=');
      pausepos:= wherex;
      writeln('Pause ^C=Cancel');
      highvideo;
      writeln(' Src= ', srcname);
      if liston then writeln(' Lst= ', listname)
                else writeln;
      lowvideo;
      setwin(6); gotoxy(1,1);                {Small window, for listing}
      if (lexan(srcname,1)) then begin       {perform the actual Assembly}
         listcon:= true;                    {Diagnostics: list always}
         listheading;
         if liston then writeln(lstfile);
         badline:= false;                   {Ready for Dictionary test}
         errorcode:= 3;                     {The undefined symbols error}
         while dicthead <> nil do begin  {Find undefineds, and clear dictionary}
  {
         if liston then begin
            write(lstfile,dicthead^.dicsymb:10, '    ');
            case dicthead^.curdictype of
               declared: write(lstfile,'OK');
               undechdr: write(lstfile,'UNDEC');
               fwdref:   write(lstfile,dicthead^.offset:8,dicthead^.frlocn:8);
               end;
            writeln(lstfile);
            end
  }
            dicptr:= dicthead^.link;
            if dicthead^.curdictype = undechdr then begin  {Log undeclared's}
               if (not badline) then logerror;        {Output the first time}
               badline:= true;
               listheading;
               if liston then writeln(lstfile,dicthead^.dicsymb:41);
               end;
            dispose(dicthead);              {Then/otherwise, scrap this item}
            dicthead:= dicptr;              {Traverse the list}
            end;

       if liston then begin
         if badline then begin
            listheading;
            writeln(lstfile);
            end;
         listheading;
         writeln(lstfile, '    ',
                '------------------- End of Assembly -------------------');
         writeln(lstfile);
         end;
         altered:= true;                     {Memory contents changed}
         if errcount=0 then begin
            assemble:= true;
            prefix:= 'No Errors Detected';
            end
         else begin
            str(errcount:3,prefix);
            prefix:= prefix + ' Error(s) Detected';
            end;
         if liston then begin
            write(lstfile,ENDFILE);          {Final EOF to list file}
            close(lstfile);                  {Ignore errors, here}
            end
         end
      else begin
         if abortrun then prefix:= 'Assembly Cancelled'
                     else prefix:= '"' + srcname + '" Not Found';
         {$I-}
       if liston then begin
         close(lstfile);
         erase(lstfile);      {Don't bother with errors here}
         end;
         {$I+}
         end
      end
   else
      prefix:= 'Can''t open ' + listname;
   setwin(0);                                    {Regular VDU window}
   end;
