#include <stdio.h>
#include "string.h"
#include "cmdline.h"
#include "umem.h"
#include "data.h"
#include "section.h"
#include "public.h"
#include "maps.h"
#include "lexpr.h"
#include "partit.h"
#include "interp.h"

extern BOOL phiused,prm_relocatable;
extern LIST *sectionlist;
extern HASHREC **publichash;
extern BOOL prm_relocatable;
extern LDESCRIPTION *partitions;
extern PUBLIC *entrypoint;

static void stringtobuf(char *buf, char *string, int len)
{
	int slen,tlen,i;
	slen = tlen = strlen(string);
	if (phiused) {
		for (i=0; i < tlen; i++)
			if (string[i] & 0x80)
				slen--;
	}
	buf[0] = 0;
	strcpy(buf,string);
	if (slen < len) {
		for (i=0; i <len-slen; i++)
			buf[i+tlen] = ' ';
		buf[i+tlen] = 0;
	}
		
}
void putattribs(char *attribs, SECTION *sect)
{
	char *p = attribs;
  if (sect->flags & SECTION_ABSOLUTE)
		*p++ = 'A';
  if (sect->flags & SECTION_COMMON)
		*p++ = 'C';
  if (sect->flags & SECTION_EQUAL)
		*p++ = 'E';
  if (sect->flags & SECTION_MAX)
		*p++ = 'M';
  if (sect->flags & SECTION_NOW)
		*p++ = 'N';
  if (sect->flags & SECTION_POSTPONE)
		*p++ = 'P';
  if (sect->flags & SECTION_ROM)
		*p++ = 'R';
  if (sect->flags & SECTION_SEPERATE)
		*p++ = 'S';
  if (sect->flags & SECTION_UNIQUE)
		*p++ = 'U';
  if (sect->flags & SECTION_RAM)
		*p++ = 'W';
  if (sect->flags & SECTION_EXEC)
		*p++ = 'X';
  if (sect->flags & SECTION_ZERO)
		*p++ = 'Z';
	*p++ = 0;
}
/*
 * Dump class names out
 */
static void DumpMajorSections(FILE *out)
{
	LIST *p = sectionlist;
  if (p) {
  	fputc('\n',out);
		phifg(DKGREEN,out);
  	fprintf(out," Start    Length  \t\t\tSection\t\tAttribs\n");

		phifg(DKBLUE,out);
		while (p) {
			char attribs[40],buf[100];
			SECTION *q = p->data;
			long offset, length;
			stringtobuf(buf,q->name,16);
			putattribs(attribs,q);
			length = q->abssize;
			offset = q->absbase;
			fprintf(out," %08lX %08lX\t\t\t%s\t%s", offset, length, buf, attribs);
			if (q->virtual)
				fprintf(out,"\n\t\t virtual = %8lX\n",q->virtual);
			else
				fprintf(out,"\n");
			p = p->link;
		}
	}
}
/*
 * Overlay map
 */
static void DumpPartitions(FILE *out)
{
	if (partitions) {
		LDESCRIPTION *p = partitions;
		phifg(DKGREEN,out);
		fprintf(out,"\n\nPartition/overlay map\n");
		phifg(DKRED,out);
		while (p) {
			if (p->type == TY_PARTITION) {
				LDESCRIPTION *o = p->downlink;
				fprintf(out,"\nPARTITION: %s", p->name);
				if (o->virtual)
					fprintf(out,"\t\tvirtual = %8lX\n",p->virtual);
				else
					fprintf(out,"\n");
				while (o) {
					if (o->type == TY_OVERLAY) {
						LDESCRIPTION *r = o->downlink;
						long address,size,roundsize,align;
						char buf[100];
						address = o->address;
						size = o->size;
						roundsize = o->roundsize;
						align = o->align;
						if (size < 0)  size = 0;
						if (roundsize < 0) roundsize = 0;
						if (align <0) align = 0;
						stringtobuf(buf,o->name,14);
						fprintf(out,"  %s addr=%8lX align=%6lX size=%8lX roundsize=%6lX\n",
						      buf,address,align,size,roundsize);
						while (r) {
							if (r->type == TY_REGION) {
								stringtobuf(buf,r->name,12);
								address = r->address;
								size = r->size;
								roundsize = r->roundsize;
								align = r->align;
								if (size < 0)  size = 0;
								if (roundsize < 0) roundsize = 0;
								if (align <0) align = 0;
								fprintf(out,"    %s addr=%8lX align=%6lX size=%8lX roundsize=%6lX\n",
								      buf,address,align,size,roundsize);
							}
							r = r->sidelink;
						}
					}
					o = o->sidelink;
				}
			}
			p = p->sidelink;
		}
	}
}	
/* 
 * Dump detailed segment map
 */
static void DumpAllSections(FILE *out)
{
	LIST *p = sectionlist;
	if (p) {
		phifg(DKGREEN,out);
	  fprintf(out,"\n\nDetailed map of sections\n\n");
		phifg(DKBLUE,out);
		while (p) {
			SECTION *r = p->data;
			while (r) {
				long len, offset;
				char attribs[40];
				char *pos,buf[100];
				stringtobuf(buf,r->name,16);
				putattribs(attribs, r->parent);
				offset = r->parent->absbase + r->base;
				len = r->size;
				pos = strrchr(r->modname,'\\');
				if (!pos)
					pos = r->modname;
				else
					pos++;
				fprintf(out," %08lX %08lX   %04X S=%s M=%-12s A=%s"	,
							offset, len, r->align, buf, r->modname, attribs);
				if (r->virtual)
					fprintf(out," VIRTUAL\n");
				else
					fprintf(out,"\n");
    		r = r->link;
	 		}
			p = p->link;
		}
	}
}
/*
 * Count the number of publics
 */
static uint CountPublics()
{
  int i,rv = 0;
  for (i=0; i< HASH_TABLE_SIZE; i++) {
    PUBLIC *p = publichash[i];
		while(p) {
			rv++;
			p = p->link;
		}
	}
  return(rv);
}
/*
 * Move the publics to a list
 */
static void PublicsToList(PUBLIC **list)
{
  int i, pos=0;
  for (i=0; i < HASH_TABLE_SIZE; i++) {
    PUBLIC *p = publichash[i];
		while (p) {
			list[pos++] = p;
			p = p->link;
		}
	}
}
/*
 * Output the public list
 */
static void OutputPublics(FILE *out, PUBLIC **list, uint len)
{
  int i;
  static char *idle = "idle", *notidle = "";
	phifg(DKORANGE,out);
  for (i=0; i < len; i++) {
		PUBLIC *p = list[i];
		long address = p->offset;
	  char *cs=notidle,buf[100];
		stringtobuf(buf,p->name,12);
		if (p->sect)
			address += p->sect->parent->absbase + p->sect->base - p->sect->parent->virtual;
    if (!p->referenced)
			cs = idle;
		if (!prm_relocatable)
    	fprintf(out," %08lX  %-6s %s\n",address, cs, buf);
		else
    	fprintf(out," %08lX  %-6s %s %-12s\n",address, cs, buf,p->sect->parent->name);
  }
}
/*
 * Sort the public list by name
 */
static void SortByName( PUBLIC **list, uint len)
{
  int i,j;
  for (i=0; i <len; i++)
    for (j=i+1; j< len; j++)
			if (phicmp(list[i]->name,list[j]->name) > 0) {
        PUBLIC *temp = list[i];
				list[i] = list[j];
				list[j] = temp;
			}
}
/*
 * Sort the public list by value
 */
static void SortByValue( PUBLIC **list, uint len)
{
  int i,j;
  for (i=0; i <len; i++)
    for (j=i+1; j< len; j++) {
			long addri = list[i]->offset;
			long addrj = list[j]->offset;
			if (list[i]->sect)
				addri+=list[i]->sect->base + list[i]->sect->parent->absbase;
			if (list[j]->sect)
				addrj+=list[j]->sect->base + list[j]->sect->parent->absbase;
			if (addri < addrj) {
        PUBLIC *temp = list[i];
				list[i] = list[j];
				list[j] = temp;
			}
		}
}
/*
 * Main routine to sort and dump publics
 */
static void DumpPublics(FILE *out)
{
  PUBLIC **list;
  uint len = CountPublics();
  list = (PUBLIC **)AllocateMemory(len*sizeof(PUBLIC *));
  PublicsToList(list);
  
	phifg(DKGREEN,out);
  fprintf(out,"\n  Address     Publics by Name\n\n");
	SortByName(list,len);
	OutputPublics(out, list,len);
	phifg(DKGREEN,out);
  fprintf(out,"\n  Address     Publics by Value\n\n");
	SortByValue(list,len);
	OutputPublics(out, list,len);
  DeallocateMemory(list);
}
/*
 * Routine to write the map file
 */
void WriteMapFile(char *name, uint mode)
{
  FILE *out;
  if (mode == MAP_NONE) return;

  if ((out =fopen(name,"w")) == 0)
    fatal("Can't create map file %s", name);

	if (phiused)
		fputc(0x1f,out);
	phifg(DKGREEN,out);
	if (prm_relocatable)
		fprintf(out,"Relocatable map file\n\n");
	else
		fprintf(out,"Absolute map file\n\n");
  DumpMajorSections(out);
	DumpPartitions(out);

	if (entrypoint && !prm_relocatable) 
		fprintf(out,"\nEntry Point: \t%08lX\n\n",entrypoint->offset + entrypoint->sect->base + entrypoint->sect->parent->absbase);
  if (mode == MAP_NORMAL) {
    fclose(out);
		return;
  }
  if (mode == MAP_EXTENDED) {
		DumpAllSections(out);
  }
  DumpPublics(out);
  fclose(out);
}