/*
 * Assembler core
 *
 * Copyright (c) 1996 LADSOFT
 *
 * David Lindauer, camille@bluegrass.net
 *
 * Core for a retargetable assembler
 *
 * This program may be freely used and redistributed, as long as the 
 * copyright notice remains intact.  The author holds no liabilty;
 * all risks, both direct and consequential, are the responsibilty of
 * the user.  You use this program at your own risk!
 *
 */
/*
 * dc.c
 *
 * defining integer data
 *
 * note that functions here depend on internal buffer sizes... this
 * is one of the places where you can out-=do the internal buffers of 
 * the assembler 
 */
#include <stdio.h>
#include <string.h>
#include "utype.h"
#include "umem.h"
#include "interp.h"
#include "asm.h"
#include "input.h"
#include "dc.h" 
#include "olist.h"
#include "gencode.h"
#include "expr.h"
#include "extern.h"
#include "fpmath.h"

extern BOOL newline;
extern BYTE pcbuf[BUFLEN];
extern int opcodesize;
extern long org;
extern BOOL ifskip;
extern int pass;
extern BOOL phiused;
extern char endian;

LIST *datalist=0;

/*dafault mode */
static long currentmode;

void dcinit(void)
{
	datalist = 0;      
	currentmode = 0x3210;
}
/* read a string into the internal data structure */
void dcstring(char *string)
{
	DC_DATA *p;
	if (ifskip)
		return;
  p = AllocateMemory(sizeof(DC_DATA));
	p->type = DC_STRING;
	p->U.string = string;
	AppendToList(&datalist, p);
}
/* read an expression into the internal data structure */
void dcexp(EXPRESSION *exp)
{
  DC_DATA *p;
	if (ifskip) {
		delexpr(exp,0);
		return;
	}
	p = AllocateMemory(sizeof(DC_DATA));
	p->type = DC_VALUE;
	p->U.exp = exp;
	AppendToList(&datalist, p);
}
/* once all data is read into internal structure, process it and
 * create the image  
 */
void processdc(int type, EXPRESSION *label)
{
  LIST *p = datalist;
	long oldorg;
  unsigned elems = 0;
	if (ifskip) {
		dolist();
		return;
	}
	/* alignment */
  if (type == TWORD || type == TLONG)
		align();
	datalist = 0;
	oldorg = org;
	while (p) {
		DC_DATA *q = p->data;
		LIST *t = p->link;
		int i;
		/* swap value */
		long swapval = q->U.exp->swapper !=0xffff ? q->U.exp->swapper : currentmode;
		/* dump the data */
		switch(q->type) {
			case DC_VALUE: {
				switch(type) {
					case TPACKED: {
							BYTE fp[12];
							BYTE pack[12];
							if (q->U.exp->isfp)
								memcpy(fp,q->U.exp->x.floatptr,12);
							else
								LongFloat(fp,q->U.exp->x.value);
							FloatPacked(pack,fp);
							memcpy(pcbuf+opcodesize,pack,12);
							opcodesize += 12;
						}
						break;

						
					case TEXTENDED: {
							BYTE fp[12];
							if (q->U.exp->isfp)
								memcpy(fp,q->U.exp->x.floatptr,12);
							else
								LongFloat(fp,q->U.exp->x.value);
							memcpy(pcbuf+opcodesize,fp,12);
							opcodesize += 12;
						}
						break;
					case TSINGLE: {
							BYTE single[4];
							BYTE fp[12];
							if (q->U.exp->isfp)
								memcpy(fp,q->U.exp->x.floatptr,12);
							else
								LongFloat(fp,q->U.exp->x.value);
							FloatSingle(single,fp);
							memcpy(pcbuf+opcodesize,single,4);
							opcodesize += 4;
						}
						break;
					case TDOUBLE: {
							BYTE dbl[8];
							BYTE fp[12];
							if (q->U.exp->isfp)
								memcpy(fp,q->U.exp->x.floatptr,12);
							else
								LongFloat(fp,q->U.exp->x.value);
							FloatDouble(dbl,fp);
							memcpy(pcbuf+opcodesize,dbl,8);
							opcodesize += 8;
						}
						break;
					case TLONG:
						if (q->U.exp->isfp) {
							long val;
						  FloatLong(&val,q->U.exp->x.floatptr);
							val = doswap(swapval, val);
							if (endian == 'L') {
								pcbuf[opcodesize+3] = val >> 24;
								pcbuf[opcodesize+2] = (val >> 16) & 0xff; 
								pcbuf[opcodesize+1] = (val >> 8) & 0xff; 
								pcbuf[opcodesize+0] = (val) & 0xff; 
							}
							else {
								pcbuf[opcodesize] = val >> 24;
								pcbuf[opcodesize+1] = (val >> 16) & 0xff; 
								pcbuf[opcodesize+2] = (val >> 8) & 0xff; 
								pcbuf[opcodesize+3] = (val) & 0xff; 
							}
							opcodesize += 4;
						}
						else {
							q->U.exp->x.value = doswap(swapval, q->U.exp->x.value);
							putdword(q->U.exp, FALSE);
						}
						break;
					case TWORD:
						if (q->U.exp->isfp) {
							long val;
							FloatLong(&val,q->U.exp->x.floatptr);
							val = doswap(swapval, val);
							if (endian == 'L') {
								pcbuf[opcodesize+1] = (val >> 8) & 0xff; 
								pcbuf[opcodesize+0] = (val) & 0xff; 
							}
							else {
								pcbuf[opcodesize+0] = (val >> 8) & 0xff; 
								pcbuf[opcodesize+1] = (val) & 0xff; 
							}
							opcodesize += 2;
						}
						else {
							q->U.exp->x.value = doswap(swapval, q->U.exp->x.value);
							putword(q->U.exp, FALSE);
						}
						break;
					case TBYTE:
					default:
						if (q->U.exp->isfp) {
							long val;
							FloatLong(&val,q->U.exp->x.floatptr);
							pcbuf[opcodesize] = val & 0xff; 
							opcodesize++;
						}
						else {
							q->U.exp->x.value = doswap(swapval, q->U.exp->x.value);
  						pcbuf[opcodesize] = (BYTE) q->U.exp->x.value;
/*  						linkadr(q->U.exp,TBYTE,FALSE); */
							opcodesize++;
						}
						break;
				}
				delexpr(q->U.exp,0);
				}
				break;
			case DC_STRING: {
				int count = 0;
				char buffer[800];
				switch(type) {
					case TLONG:
						count = phistreamtoflat(buffer,q->U.string,4,TRUE);
						break;
					case TWORD:
						count = phistreamtoflat(buffer,q->U.string,2,TRUE);
						break;
					case TBYTE:
						count = phistreamtoflat(buffer,q->U.string,1,FALSE);
						if (phiused)
							for (i=0; i<count; i++)
								buffer[i] &= 0x7f;
						break;
					default:
						break;
				}
				for (i=0; i < count; i++)
					pcbuf[opcodesize++] = buffer[i];
				DeallocateMemory(q->U.string);
  			if ((type == TWORD || type == TLONG) && (opcodesize + org) & 1)
					pcbuf[opcodesize++] = 0;
				}
				break;
		}
		DeallocateMemory(p->data);
		DeallocateMemory(p);
		p = t;
		
	}
	/* set the size attribute */
	if (label) {
		label->size = opcodesize;
		label->x.value = oldorg;
		label->datatype = type;
	}
	dolist();           
	copytomem();
}
/* process a phi-text type DC */
void processphitext(EXPRESSION *label)
{
	LIST *p = datalist;
	long oldorg;
	if (ifskip) {
		dolist();
		return;
	}
	datalist = 0;
	oldorg = org;
	while (p) {
		DC_DATA *q = p->data;
		LIST *t = p->link;
		int i;
		/* process the data */
		switch(q->type) {
			case DC_VALUE:
  				pcbuf[opcodesize] = (BYTE) q->U.exp->x.value;
  				linkadr(q->U.exp,TBYTE,FALSE);
					opcodesize++;
				delexpr(q->U.exp,0);
				break;
			case DC_STRING: {
				int count;
				char buffer[800];
				count = strlen(q->U.string);
				strcpy(buffer, q->U.string);
				for (i=0; i < count; i++)
					pcbuf[opcodesize++] = buffer[i];
				DeallocateMemory(q->U.string);
				}
				break;
		}
		DeallocateMemory(p->data);
		DeallocateMemory(p);
		p = t;
		
	}
	/* set up the size and org */
	if (label) {
		label->size = opcodesize;
		label->x.value = oldorg;
	}
	dolist();           
	copytomem();
}
/*
 * process pre-initialized blocks
 */
void processdcb(int type, EXPRESSION *label, EXPRESSION *exp1, EXPRESSION *exp2)
{
  int i;
	long val = doswap(exp2->swapper != 0xffff ? exp2->swapper : currentmode,exp2->x.value);
	long oldorg;
	if (ifskip) {
		delexpr(exp1,exp2);
		dolist();
		return;
	}
	if (type != TBYTE)
		align();
	/* clip value */
	switch(type) {
		case TBYTE:
				val = val & 0xff;
				break;
		case TWORD:
				val = val & 0xffff;
				break;
	}
	oldorg = org;
	/* dump the data */
	if (exp1->isdef && exp2->isdef && numericExpression(exp1) && numericExpression(exp2) && exp1->x.value >= 0) {
		long len = exp1->x.value * (1 << (type - 1));
		if (!len) {
			align();
		}
		else {
		  if (len > BUFLEN)
				Error("DCB data too large");
			else {
				for (i=0; i < exp1->x.value; i++) {
					switch(type) {
						case TBYTE:
							pcbuf[opcodesize++] = (BYTE) val;
							break;
						case TWORD: {
								pcbuf[opcodesize++] = (val & 0xff00) >>8;
								pcbuf[opcodesize++] = val & 0xff;
							}
							break;
						case TLONG: {
								pcbuf[opcodesize++] = (val & 0xff000000L) >>24;
								pcbuf[opcodesize++] = (val & 0xff0000L) >> 16;
								pcbuf[opcodesize++] = (val & 0xff00) >>8;
								pcbuf[opcodesize++] = val & 0xff;
							}
							break;
					}
				}
			}
		}
	}
	else 
		Error("Invalid expression in DCB statement");
	/* set the size */
	if (label) {
		label->size = opcodesize;
		label->x.value = oldorg;
		if (pass == 1)
			label->datatype = alloc_array(type,exp1->x.value);
	}
	dolist();
	copytomem();
}
void setmode(long newmode)
{
	currentmode = newmode;
}