/*
 * Assembler core
 *
 * Copyright (c) 1996 LADSOFT
 *
 * David Lindauer, camille@bluegrass.net
 *
 * Core for a retargetable assembler
 *
 * This program may be freely used and redistributed, as long as the 
 * copyright notice remains intact.  The author holds no liabilty;
 * all risks, both direct and consequential, are the responsibilty of
 * the user.  You use this program at your own risk!
 *
 */
/*
 * macros.c
 *
 *macro handling
 */
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "utype.h"
#include "umem.h"
#include "asm.h"
#include "input.h"
#include "macros.h"
#include "interp.h"
#include "gencode.h"

extern HASHREC ** hashtable;
extern char *bufptr;
extern char buf[BUFLEN];
extern BOOL newline;
extern int pass;
extern char *macroptr;
extern BOOL ifskip;
extern PHITEXT phiparms;
extern BOOL phiused;
extern long memory_left;
extern int macstyle,ifstyle;

PHITEXT macrophiparms;

LIST *macrolist = 0;
int macrolevel = 0;
int macinvocation = 1;

void macini (void)
{
	macstyle = 1;
	macrolist = 0;
	macinvocation = 1;
	macrolevel = 0;
}
/* macro listing */
static void domaclist(void)
{
	if (macstyle !=3) {
		dolist2();
	}
}
/* macro args are stored as indexes to the argument number */
static char *putnum(int num, char *bptr)
{
	sprintf(bptr,"%d",num);
	bptr += strlen(bptr);
  return(bptr);
}
/* error msg */
static void illformedarg()
{
			Error("Ill Formed Macro Argument");
}
/* decode a macro line given the argument info */
static void decode(char *buf, MACINFO *minfo)
{
	char *ptr = minfo->macro->lines[minfo->linenum++];
	char *bptr = buf;
	int bank = phiparms.bank;
	BOOL foundcomment = FALSE;
	while (*ptr) {
		
		/* phi text bank switch */
		if (phiused && ((*ptr & 0xf0) == 0x90))
			bank = *ptr & 0x0f;

				if (*ptr == '\\' && !foundcomment) {
					/* handling for normal args */
					if (isdigit(*(ptr+1))) {
						int temp = 0;
						while(isdigit(*++ptr))
							temp = temp*10+*ptr-'0';
						if (temp == 0) {
							/* #0 is the type info */
							*bptr++ = minfo->macro->type;
						}
						else {
							/* other args come from the argument list */
							temp -= 1;
							if (temp < minfo->argcount) {
								strcpy(bptr,minfo->args[temp]);
								bptr += strlen(bptr);
							}
						}
					}
					else {
						int i;
						/* If we get here, looking for a special arg */
						switch(*(ptr+1)) {
							case '@':
								/* invocation number */
								bptr = putnum(minfo->invoke,bptr);
								break;
							case '#':
								/* argcount */
								bptr = putnum(minfo->argcount,bptr);
								break;
							case '*':
								/* concatenate all args */
								for (i=0; i < minfo->argcount; i++) {
									strcpy(bptr,minfo->args[i]);
								  bptr = bptr + strlen(bptr);
								}
								break;				
							case ':':
								/*& label used on argument invocation */
								if (minfo->name) {
									strcpy(bptr,minfo->name);
									bptr += strlen(bptr);
								}
								break;
							case '!':
								/* macro name */
								strcpy(bptr,minfo->macro->name);
								bptr+= strlen(bptr);
								break;
							default:
								ptr -= 2;
						}
						ptr += 2;
					}
				}
				else {
					/* if we get here we are just going to copy a char */
					int val;
					if (phiused) {
						if (*ptr & 0x80)
							val = 0;
						else
							val = (bank << 7) + *ptr;
					}
					else 
						val = *ptr;
					if ( iscommentchar(val))
						foundcomment = TRUE;
					*bptr++ = *ptr++;
				}
	}
	*bptr = 0;
}
/* Encode a macro to numeric argument form */
static void encode(MACRODEF *macro, int line, char *buf)
{
	char lbuf1[BUFLEN];
	char lbuf2[BUFLEN];
	int i=0,p=0,bank=0;
	char *compos = 0;
	strcpy(lbuf1,buf);
	/* find the beginning of the comments */
	/* note we are NOT allowed to use embedded comments in macros */
	if (phiused) {
		char sbuf[400], *sbufp = sbuf;
		i = phistreamtoflat(sbuf,buf,2,FALSE)-2;
		while (i) {
			int val;
			if (iscommentchar(val=(*sbufp++ <<7) +*sbufp++)) {
				compos = lbuf1 + p;
				break;
			}
			if ((val>>7) != bank) {
				bank = val >> 7;
				p++;
			}
			p++;
			i--;
		}
	}
	else
		compos = strchr(lbuf1,';');
	/* replace all arguments with their arg number */
	for (i=0; i < macro->argcount; i++) {
		char *pos;
		while ((pos = strstr(lbuf1,macro->args[i])) != 0) {
			char *bpos1 = lbuf1, *bpos2 = lbuf2;
			if (compos && pos >=compos)
				break;
			while (bpos1 < pos)
				*bpos2++ = *bpos1++;
			bpos1+=strlen(macro->args[i]);
			sprintf(bpos2,"\\%02d",i+1);
			strcat(bpos2,bpos1);
			strcpy(lbuf1, lbuf2);
		}
	}
	macro->lines[line] = AllocateMemory(strlen(lbuf1)+1);
	strcpy(macro->lines[line],lbuf1);
}
/*
 * set up for a REPT macro
 */
int setrept(EXPRESSION *count)                     
{
	MACRODEF *macro;
	LIST *list=0, *xlist=0;
	int i=0,argcount = 0,ct;
	MACINFO *minfo;

	
	if (!numericExpression(count)) {
		Error("non-constant in REPT directive");
		ct = 1;
	}
	else
		ct = count->x.value;
	macrolevel++;
	domaclist();
	macrolevel--;	

	/* create ,acrp */
	macro = AllocateMemory(sizeof(MACRODEF));
	macro->type = 0;
	macro->name = AllocateMemory(5);
	strcpy(macro->name, "t$$$");
	macro->args = 0;
	/* read in the macro */
	while (TRUE) {
		char *abptr;
		if (!nextLine( buf, BUFLEN)) {
			Error("Unexpected end of file");
			newline = TRUE;
			return(0);
		}
		if (strstr(buf, "ENDM") || strstr(buf,"endm") || strstr(buf,".ENDM") || strstr(buf,".endm"))
			break;
		if (!ifskip) {
			abptr = AllocateMemory(strlen(buf) + 1);
			strcpy(abptr, buf);
		}
		AppendToList(&list,abptr);
		argcount++;
	}
	if (ifskip)
		return;
	/* Copy args to macro data area */
	if (argcount) {
		macro->lines = AllocateMemory(argcount * sizeof (char *));
		for (i=0; i < argcount; i++) {
			xlist = list->link;
			macro->lines[i] = list->data;
			DeallocateMemory(list);
			list = xlist;
		}
		macro->linecount = argcount;
	}
	else
		macro->linecount = 0;

	/* We are going to be executing immediately, make the invocation info */
	minfo = AllocateMemory(sizeof(MACINFO));
	list = AllocateMemory(sizeof(LIST));
	minfo->macro = macro;
	minfo->linenum = 0;
	minfo->type = MT_REPT;
	minfo->name = 0;
	list->data = minfo;
	list->link = macrolist;
	macrolist = list;
	macrolevel++;
	
	if (macrolevel == 1)
		macrophiparms = phiparms;
	minfo->args = 0;
	minfo->argcount = 0;
	minfo->invoke = macinvocation++;
	minfo->maxcount = ct;
	minfo->count = 0;

	if (!ct)
		CloseMacro();

	delexpr(count,0);
	newline = TRUE;
	return(' ');
}
/* read an argument from the definition line */
int getmacroarg(char *argbuf, int curchar, BOOL onlysym)
{
		int i = 0;
		/* pass whitespace */
		if (iswhitespacechar(curchar))
			curchar = skipSpace(&bufptr);
		if (iscommentchar(curchar))
			return curchar;
		/* read in a string */
		if (curchar == '"' || curchar == '\'') {
			int thechar = curchar;
			i+= installphichar(curchar,argbuf,i);
			curchar = parsechar(&bufptr);
			while (curchar != thechar && curchar != '\n' && !iscommentchar(curchar)) {
				i+=installphichar(curchar,argbuf,i);
				curchar = parsechar(&bufptr);
			}
			i+=installphichar(thechar,argbuf,i);
			if (curchar != thechar)
				illformedarg();
			curchar = parsechar(&bufptr);
		}
		/* read in a string */
		else if (curchar == '<') {
			int thechar = '>';
			curchar = parsechar(&bufptr);
			while (curchar != thechar && curchar != '\n' && !iscommentchar(curchar)) {
				i+=installphichar(curchar,argbuf,i);
				curchar = parsechar(&bufptr);
			}
			if (curchar != thechar)
				illformedarg();
			curchar = parsechar(&bufptr);
		}
		else
			if (onlysym)
				/* for totally symbolc args */
				if (!issymchar(curchar) && !(phiused &&(curchar & 0x80))) {
					curchar = parsechar(&bufptr);
					illformedarg();
				}
				else
					while ((issymchar(curchar) || (phiused &&(curchar & 0x80)))) {
						i+=installphichar(curchar,argbuf,i);
						curchar = parsechar(&bufptr);
					}
			else
				/* for any symbol allowed in an arg but ',' */
				while (curchar != ',' && curchar != '\n' && !iscommentchar(curchar) && !iswhitespacechar(curchar)) {
					i+=installphichar(curchar,argbuf,i);
					curchar = parsechar(&bufptr);
				}
	
		/* clean up */
		argbuf[i] = 0;
		if ((argbuf[i-1] & 0xf0) == 0x90)
			argbuf[i-1] = 0x90;
		if (iswhitespacechar(curchar))
			curchar = skipSpace(&bufptr);
		return(curchar);
}
/*
 * define a macro
 */
int	createMacro(EXPRESSION *def)
{
	
	char *arglist[MAXMACROARGS];
	char argbuf[BUFLEN];
	MACRODEF *macro;
	int curchar;
	char type = 'W';
  LIST *list=0, *xlist = 0;
	int i=0, argcount = 0;

  if (def == 0) {
		Error("Missing name in macro construct");
		return(0);
	}
	if (!ifskip) {
  	def->islabel = FALSE;
  	def->ismacro = TRUE;
	}
	curchar = skipSpace(&bufptr);
	/* Get type info */
	if (curchar == '.') {
		curchar = parsechar(&bufptr);
		if (curchar != '\n' && !iscommentchar(curchar)) {
			type = curchar;
			curchar = parsechar(&bufptr);
		}
		else {
			bufptr--;
			putphiback(curchar);
			curchar = '.';
		}
	}
	/* read in args */
	while (curchar != '\n' && !iscommentchar(curchar)) {
		curchar = getmacroarg(argbuf,curchar,TRUE);
		if (argcount == MAXMACROARGS) {
			Error("Too many args in macro");
			argcount--;
		}
		if (pass == 1 && !ifskip) {
			arglist[argcount] = AllocateMemory(strlen(argbuf)+1);
			strcpy(arglist[argcount++],argbuf);
		}
		if (curchar != '\n' && curchar != ',' && !iscommentchar(curchar))
			illformedarg();
		if (curchar == ',')
			curchar = parsechar(&bufptr);
	}
	/* skip rest of line */
	while (curchar != '\n')
		curchar = parsechar(&bufptr);
	/* make the macro (pass 1 only) */
  if (pass == 1 && !ifskip) {
		macro = AllocateMemory(sizeof(MACRODEF));
		macro->type = type;
		macro->name = AllocateMemory(strlen(def->name) + 1);
		strcpy(macro->name, def->name);
		def->x.xdata = macro;
		if (argcount) {
			macro->args = AllocateMemory(sizeof(char *)*argcount);
			for (i=0; i < argcount; i++)
				macro->args[i] = arglist[i];
		}
		else
			macro->args = 0;
		macro->argcount = argcount;
	}

	argcount = 0;
	dolist2();

	/* read in the macro definitions */
	while (TRUE) {
		if (!nextLine( buf, BUFLEN)) {
			Error("Unexpected end of file");
			newline = TRUE;
			return(0);
		}
		dolist2();
		if (strstr(buf, "ENDM") || strstr(buf,"endm") || strstr(buf,".ENDM") || strstr(buf,".endm"))
			break;
		if (pass == 1 && !ifskip) {
			char *abptr;
			abptr = AllocateMemory(strlen(buf) + 1);
			strcpy(abptr, buf);
			AppendToList(&list,abptr);
			argcount++;
		}
	}
	if (pass == 1 && !ifskip) {
		/* encode to numeric args */
		if (argcount) {
			macro->lines = AllocateMemory(argcount * sizeof (char *));
			for (i=0; i < argcount; i++) {
				xlist = list->link;
				encode(macro, i, list->data);
				DeallocateMemory(list->data);
				DeallocateMemory(list);
				list = xlist;
			}
			macro->linecount = argcount;
		}
		else
			macro->linecount = 0;
	}
	if (!ifskip) 
		delexpr(def,0);
	newline = TRUE;
	return('\n');
}

/* invoke a macro */
void OpenMacro(EXPRESSION *label, EXPRESSION *exp)
{
	char *arglist[MAXMACROARGS];
	char argbuf[BUFLEN];
  MACINFO *minfo;
	int curchar;
  LIST *list;
	int i=0, argcount = 0;
	if (ifskip) {
		newline = TRUE;
		dolist();
		return;
	}
	curchar = skipSpace(&bufptr);
	/* if followed by the 'macro' keyword we are actually redefining */
	if (!strncmp(bufptr-1,"MACRO",5) || !strncmp(bufptr-1,"macro",5) || 
				!strncmp(bufptr-1,".macro",6) || !strncmp(bufptr-1,".MACRO",6)) {
		bufptr += 4;
		curchar = createMacro(exp);
		return;
	}
	/* otherwise create the invocation info */
	minfo = AllocateMemory(sizeof(MACINFO));
	list = AllocateMemory(sizeof(LIST));
	minfo->macro = exp->x.xdata;
	minfo->linenum = 0;
	minfo->type = MT_MACRO;
	if (label && label->name) {
		minfo->name = AllocateMemory(strlen(label->name) + 1);
		strcpy(minfo->name, label->name);
	}
	else
		minfo->name = 0;
	list->data = minfo;
	list->link = macrolist;
	macrolist = list;
	macrolevel++;
	
	/* read in the invocation arguments */
	while (curchar != '\n' && !iscommentchar(curchar)) {
		curchar = getmacroarg(argbuf,curchar,FALSE);
		if (argcount == MAXMACROARGS) {
			Error("Too many args in macro");
			argcount--;
		}
		arglist[argcount] = AllocateMemory(strlen(argbuf)+1);
		strcpy(arglist[argcount++],argbuf);
		if (curchar != '\n' && curchar != ',' && !iscommentchar(curchar))
			Error("Ill Formed Macro Argument");
		if (curchar == ',')
			curchar = parsechar(&bufptr);
	}
	while (curchar != '\n')
		curchar = parsechar(&bufptr);
	domaclist();
	if (macrolevel == 1)
		macrophiparms = phiparms;
	/* move arguments to invocation */
	if (argcount) {
		minfo->args = AllocateMemory(sizeof(char *)*argcount);
		for (i=0; i < argcount; i++)
			minfo->args[i] = arglist[i];
	}
	else
		minfo->args = 0;
	minfo->argcount = argcount;
	minfo->invoke = macinvocation++;

	delexpr(label,exp);
	newline = TRUE;

}
/* close a macro, deallocate invocation space */
void CloseMacro(void)
{
	int i;
	LIST *list = macrolist;
  MACINFO *minfo=list->data;     	
	if (minfo->type == MT_REPT)
		RemoveMacro(minfo->macro);
	for (i=0; i < minfo->argcount; i++)
		DeallocateMemory(minfo->args[i]);
	DeallocateMemory(minfo->args);
	if (minfo->name)
		DeallocateMemory(minfo->name);
	DeallocateMemory(minfo);
	macrolist = macrolist->link;
	DeallocateMemory(list);
	macrolevel--;
}
/*
 * exit from a macro prematurely */
void macroexit(void)
{
	if (ifskip)
		return;
  if (macrolevel)
		CloseMacro();
	else
		Error("MEXIT/MEND has no function outside of macro");
	if (macstyle ==4)
		dolist2();
}
/* called by the input routines to fetch a line from a macro
 */
BOOL GetMacroLine(void)
{
	MACINFO *minfo = macrolist->data;
	while (minfo->macro->linecount == minfo->linenum) {
		/* check for a REPT of a macro */
		if (minfo->type == MT_REPT) {
			if (++minfo->count < minfo->maxcount) {
				minfo->linenum = 0;
				break;
			}
		}
		/* close the current macro */
		CloseMacro();
		if (macrolevel == 0)
			return(FALSE);
		minfo = macrolist->data;
	}
	/* decode the line and exit */
	if (minfo->type == MT_MACRO)
		decode(buf,minfo);
	else
		strcpy(buf,minfo->macro->lines[minfo->linenum++]);
	return(TRUE);
}
/* remove a macro definition */
void RemoveMacro(MACRODEF *macro)
{
	while (macro->argcount)
		DeallocateMemory(macro->args[--macro->argcount]);
	while (macro->linecount)
		DeallocateMemory(macro->lines[--macro->linecount]);
	DeallocateMemory(macro->args);
	DeallocateMemory(macro->lines);
	DeallocateMemory(macro->name);
	DeallocateMemory(macro);
}