/*
 * Assembler core
 *
 * Copyright (c) 1996 LADSOFT
 *
 * David Lindauer, camille@bluegrass.net
 *
 * Core for a retargetable assembler
 *
 * This program may be freely used and redistributed, as long as the 
 * copyright notice remains intact.  The author holds no liabilty;
 * all risks, both direct and consequential, are the responsibilty of
 * the user.  You use this program at your own risk!
 *
 */
/*
 * sections.c
 *
 * handle sections
 */
#include <stdio.h>   
#include <string.h>
#include <malloc.h>
#include <memory.h>
#include "cmdline.h"
#include "umem.h"
#include "asm.h"
#include "input.h"
#include "sections.h"
#include "gencode.h"
#include "struct.h"
#include "loclab.h"

extern long org;
extern BYTE *outputbuf;
extern BOOL ifskip;
extern int opcodesize;
extern int pass;
extern BOOL prm_debug;

int basetype;
uint sectionnum, lastsectionnum, allocsize;
LIST *sections;
SECTIONREC *cursection;

static char mainname[] = "MAIN";
char linename[] = "??LINE";

void section_init(void)
{
	if (pass == 1) {
		basetype = 0;
		cursection = 0;
		sectionnum = 0;
		lastsectionnum = 0;
		allocsize = 0;
			sections = 0;
	}
	if (prm_debug) {
		switchnamedsection(linename);
	}
  switchnamedsection(mainname);

}
/* get section number num */
SECTIONREC *getsection(int num)
{
	LIST *p = sections;
	while (p) {
		SECTIONREC *q = p->data;
		if (!--num)
			return(q);
		p = p->link;
	}
	return(0);
}
/* find a section based on its name */
int findsection(char *name)
{
	LIST *p = sections;
	int i = 1;
	while (p) {
		SECTIONREC *q = p->data;
		if (!strcmp(q->name, name))
			return(i);
		i++;
		p = p->link;
	}
	return(0);
}
/* add a new section */
void addsection(EXPRESSION *section, int type)
{
	SECTIONREC *p;
	LIST *q,*nl;
	/* if it exists return it */
	int sn = findsection(section->name);
	if (sn) {
		p = getsection(sn);
		if (type)
			p->type = type;
	}
	else {
		LIST *r;
		/* else make the new section */
		if (strchr(section->name , '$') || strchr(section->name, '_') || strchr(section->name, '.'))
			Error("Ill-formed section name");
		p = AllocateMemory(sizeof(SECTIONREC));
		p->name = AllocateMemory(strlen(section->name) +1 );
		strcpy(p->name, section->name);
		outputbuf = 0;
		allocsize = 0;
		sectionalloc();
  	p->buffer = outputbuf;
		p->type = type;
		p->org = 0;
		p->highest = 0;
		p->allocsize = allocsize;
		p->fixuplist = 0;
		p->fixuppos = &p->fixuplist;
		q = sections;
		r = &sections;
		sn = 1;
		while (q) {
			sn++;
			r = q;
			q = q->link;
		}
		nl = AllocateMemory(sizeof(LIST));
		r->link = nl;
		nl->link = 0;
		nl->data = p;
	}
	/* set the current section to this */
	cursection = p;
	setorg(p->org);
	outputbuf = p->buffer;
	allocsize = p->allocsize;
	sectionnum = sn;  
	basetype = p->type;
}
/*
 * run through all sections after pass 1 and set the high value so
 * we can check for mismatch at the end of pass 2
 */
void setsectionhighs(void)
{
	LIST *p = sections;
	switchoutofsection(0,FALSE);
	while (p) {
		SECTIONREC *q = p->data;
		q->highest = q->org;
		q->org = 0;
		p = p->link;
	}
	switchoutofsection(0,FALSE);
	section_init();
}
/*
 * Check for phase errors at end of pass 2
 */
void checksectionhighs(void)
{
	LIST *p = sections;
	switchoutofsection(0,FALSE);
	while (p) {
		SECTIONREC *q = p->data;
		if (q->highest != q->org) {
			char buf[100];
			sprintf(buf,"Phase error section %s", q->name);
			Error(buf);
		}
		p = p->link;
	}
}
/*
 * create a new section named 'name' 
 * Used during initialization to create MAIN or LINES
 */
void switchnamedsection(char *name)
{
	EXPRESSION *locexpr = AllocateMemory(sizeof(EXPRESSION));
		locexpr->name = AllocateMemory(strlen(name)+1);
		strcpy(locexpr->name, name);
		locexpr->reg1 = locexpr->reg2 = 0;
		locexpr->link = 0;
		locexpr->isdef = FALSE;
		locexpr->isintermed = TRUE;
		locexpr->x.value = 0;
		locexpr->size = 0;
		locexpr->islabel = FALSE;
		locexpr->isextern = FALSE;
		locexpr->defpass = 0;
		locexpr->isopcode = FALSE;
		locexpr->ismacro = FALSE;
		locexpr->isoperand = FALSE;
		locexpr->ischangeable = FALSE;
		locexpr->ismovemreg = FALSE;
  	locexpr->isstruct = FALSE;
	  locexpr->rel = FALSE;
		locexpr->relmode = RM_NONE;
		locexpr->section = 0;
		locexpr->ispublic = FALSE;
		locexpr->islocalpub = FALSE;
		locexpr->xrefcount = 0;
		locexpr->swapper = 0xffff;
		locexpr->relpc = FALSE;
		locexpr->isfp = FALSE;
		locexpr->floatspec = FALSE;
		locexpr->id = 0;
    locexpr->datatype = 0;
	  switchsection(locexpr,FALSE,0);
}
/* 
 * do a section switch */
void switchsection(EXPRESSION *section, BOOL tolist, int type)
{
	SECTIONREC *p;
	int i = findsection(section->name);
	section->isintermed = TRUE;
	if (ifskip) {
		dolist();
		delexpr(section, 0);
		return;
	}
	StructCheck();
	if (sectionnum) {
		cursection->org = org;
		cursection->buffer = outputbuf;
		cursection->allocsize = allocsize;
		cursection->type = basetype;
	}
  if (!section) {
		p = getsection(lastsectionnum);
		if (p) {
			setorg(p->org);
			outputbuf = p->buffer;
			allocsize = p->allocsize;
			cursection = p;
			basetype = p->type;
			if (type != p->type)
				Error("Can't change section type");
		}
		sectionnum = lastsectionnum;
	}
	else {
		if (i) {
			p = getsection(i);
			setorg(p->org);
			cursection = p;
			outputbuf = p->buffer;
			allocsize = p->allocsize;
			sectionnum = i;
			basetype = p->type;
			if (type != p->type)
				Error("Can't change section type");
		}
		else
			addsection(section,type);
	}
	delexpr(section, 0);
	loclabinit();
  if (tolist)
  	dolist();
}
/* Switch to an OFFSET region */
void switchoutofsection(long neworg, BOOL tolist)
{
  SECTIONREC *p = getsection(sectionnum);
	if (ifskip) {
		dolist();
		return;
	}
	StructCheck();
  if (p) {
		p->org = org;
		p->allocsize = allocsize;
		p->buffer = outputbuf;
		lastsectionnum = sectionnum;
	}

	basetype = 0;
	cursection = 0;
  sectionnum = 0;
	outputbuf = 0;
  setorg(neworg);
  opcodesize = 0;
	allocsize = 0;
	loclabinit();
	if (tolist)
		dolist();
}
/*
 * resize a section data area when it gets too much data
 */
void sectionalloc(void)
{
	if (allocsize >= MAXSECTIONALLOC) {
		Error("Object file too big, break up sources");
		setorg(0);
		return;
	}
	allocsize+=SECTIONALLOC;
	outputbuf = ReallocateMemory(outputbuf,allocsize);
	memset(outputbuf+allocsize-SECTIONALLOC,0,SECTIONALLOC);
}	