/*============================================================================
 *	D2D/98	- disc to disc copy for IBM-PC/AT compatible machines
 *
 *	Copyright (c) 1987 - 1992, 1998 by
 *		Ulrich Windl
 *		Alte Regensburger Strae 11a
 *		D-93149 Nittenau
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *	Uli says: "All you programmers of classic C - forget it; it's Turbo-C"
 *		  "All you UNIX programmers - forget about portability"
 *
 *	Borland C++ 2.0
 *	Version 2.32.1.4
 *
 *	Last update:	11/12/87	* Media Byte implemented
 *			11/17/87	* 1.44MB for PS/2, all-flag
 *			12/17/87	* new chk_drive() function, etc.
 *	1.20		 1/26/88	* 720/730kB for PS/2; PS/2 flag
 *	1.21		 2/10/88	* usage() receives valid PS/2 flag
 *	1.22		 4/14/88	* handle bad clusters on source
 *	     4/18/88  &  4/28/88	* "1.22 new bug": fixed
 *	1.23		 7/19/88	* several improvements *
 *	1.30		11/09/88	* replaced INT 24 handler, works
 *					  under DOS-mode of OS/2(!)
 *	1.31		11/15/88	* do not abort on bad destination disk
 *	1.32		 5/02/89	* removed clumsy code, more speed
 *	1.33		 5/10/89	* improved security_test()
 *	1.34		 5/19/89	* removed 1.33 bug
 *	1.35		 8/21/89	* ATs have 3.5" drives too
 *	2.00  9/28/90 - 10/03/90	* use EMS, copy non-DOS disks
 *	2.01 10/06/90 - 10/12/90	* fixed bug, count fragments in FAT
 *	2.02 		 5/04/91	* fixed DOS-version bug, deal with
 *					  garbage at end of FAT correctly
 *	2.10  5/23/91 -  5/31/91	* added XMS support
 *	2.11  6/05/91 -	 6/16/91	* Windows 3.0 can't lock XMS
 *					* create new volume serial numbers
 *	2.12		 8/14/91	* fixed bug in EMS info
 *	2.13		 9/08/91	* {EMS,XMS}.avail now set properly
 *	2.14  9/11/91 -	 9/26/91	* fixed bugs in formatting code
 *d2d-2.c'v
 * Revision 2.32.1.4  1998/12/25  23:36:50  UhW
 * Added new option to disable boot code on destination disks (anti-virus).
 * Reordered code to process option parsing. `getCR()' will always print '\n'.
 *
 * Revision 2.32.1.3  1998/12/25  22:57:22  UhW
 * Added new options (from release 3.3): ``unconditionally format destination
 * disk'', ``turn off bell for non-critical notifications'', ``try to recover
 * from read errors'', and ``overwrite destination disk without asking''.
 * Print additional release and edition information.
 *
 * Revision 2.32.1.2  1998/12/24  23:26:30  UhW
 * Merged some fixed from `D2D 3.2 Professional': Mainly add better checks
 * for BIOS Parameter Blocks, and improve testing for other problems.  Made
 * environment variable `TMPDIR' known.
 *
 * Revision 2.32.1.1  1998/12/24  22:30:14  UhW
 * This is ``D2D/98, edition `Sophia''', released under the GPL to celebrate
 * the 10th anniversary of the first release in 1988.
 *
 * Revision 2.32  1992/04/18  19:18:15  UhW
 * Changed option "/t" to accept drive and directory for temporary file.
 * Modified forward() function accordingly.
 *
 * Revision 2.31  1992/04/13  21:33:10  UhW
 * Fixed bug in private XMS management. XMS accepted if there is room for at
 * least two tracks.
 *
 * Revision 2.30  1992/04/12  21:55:55  UhW
 * Modified read_disc() and write_disc() functions: Code in main loops should
 * be clearer now. Added ..._read_init() and ..._write_init() functions that
 * initialize a new state (RAM, XMS, EMS, DISK). Added resetDisks() function
 * that is called after writing a disk. New function invalidateDisk() is
 * called whenever the writing phase is aborted. Main variants have only been
 * tested!
 *
 * Revision 2.29  1992/04/06  21:37:06  UhW
 * Added new "smart-verify" mode that verifies only if formatting is not done.
 * SLACK-constant used in memory requirement. Option "/t" uses the specified
 * drive's default directory to place the swap file into it.
 *
 * Revision 2.28  1992/04/05  18:26:05  UhW
 * Added strings for error class and error locus in function ERROR(). Changed
 * few messages. Added requirement message.
 *
 * Revision 2.27  1992/04/03  20:15:19  UhW
 * Fixed bug in EMScheck() that caused EMS to be ignored if more than 127 pages
 * were available. Modified code to print space of the correct disk and made
 * the status display compatible with that of EMS and XMS. Improved code that
 * allocates memory.
 *
 * Revision 2.26  1992/03/23  22:03:25  UhW
 * Fixed bug in maxTrack(): single allocated clusters are counted now; moved
 * fragmentation counting into second pass. Fragments are counted exactly
 * now. Added output in percent and fixed "%#03x" format to "0x%03x" which
 * works with BC++ 2.0. Code in first pass should be stable again!
 *
 * Revision 2.25  1992/03/16  22:06:16  UhW
 * Unified XMS and EMS memory reports: both display used and available amounts
 * of memory. Increased DEMO sizes from 64kB to 96kB each.
 *
 * Revision 2.24  1992/03/11  23:05:20  UhW
 * Fixed bug "destPB_valid has wrong value". Changed input handling. IBM seems
 * to use "FAT" instead of "FAT12"; bootfix() modified accordingly. Added
 * version string to serial number.
 *
 * Revision 2.23  1992/03/09  23:34:09  UhW
 * Fixed bugs in argument parser, fixed bug in /t option, cleaned up usage(),
 * moved bootfix(), added/changed some messages.
 *
 * Revision 2.22  1992/02/29  21:28:49  UhW
 * Began to clean up code for setting the media type; removed PS2_flag.
 * Swap file can now be created outside of the root directory (via environment
 * variables). 2.88MB drives not completely implemented. Added new error
 * message if device type is unknown. Created include file "user.h"
 *
 * Revision 2.21  1992/02/15  21:57:49  UhW
 * Added more BPBs and changed the syntax of the command-line. Fixed up a bug
 * in the error handler. Smoothed code of maxTrack() calculation. Fixed bug
 * in displaying bad clusters (couldn't display single bad clusters).
 *
 * Revision 2.20  1992/01/14  22:25:38  UhW
 * Modified format check routine and added a new verify switch. Verify will
 * be performed immediately after writing a track. Default is no verify pass.
 *
 * Revision 2.19  1992/01/08  21:02:51  UhW
 * Just fixed up the copyright list
 *
 * Revision 2.18  1992/01/08  20:30:30  UhW
 * Tried to shorten usage() information: Verbose help is generated by the new
 * procedure help(). A new switch "/?" has been added to explicitly produce
 * verbose help. The previous behavior is identical for no arguments.
 *
 * Revision 2.17  1991/12/19  19:47:37  UhW
 * Added support of user-definable XMS and EMS usage. The procedure usage()
 * has been rewritten together with the argument parser. A new procedure
 * forward() is used by the current parser. The strout() procedure has been
 * modified to handle '\n' correctly. The user can only decrease the amount
 * of memory the program uses (XMS, EMS). XMSsetup() and EMSsetup() have been
 * modified to use the maximum of requested and available memory.
 *
 * Revision 2.16  1991/11/06  20:01:49  UhW
 * Bytes in the FAT are unsigned; the new compiler needs "unsigned char" to
 * produce correct code. The code to analyze the FAT has been rewritten:
 * Now there's a separate function to access single FAT entries; there are
 * also two passes through the FAT (1: get last cluster, 2: find bad blocks)
 *
 * Revision 2.15  1991/11/06  17:31:26  UhW
 * Prepared to be used with RCS (co -kv required).
 * Default drive '@' is no longer allowed in function chk_drive().
 *============================================================================
 */

#define	LICENCE		"GPL"
#define	RELEASE		"/98"
#define	EDITION		"Sophia"

#include	<stdio.h>
#include	<stdlib.h>
#include	<ctype.h>
#include	<string.h>
#include	<time.h>

#include	<errno.h>
#include	<fcntl.h>
#include	<alloc.h>

#include	<conio.h>		/* direct console input required */
#include	<dos.h>
#include	<dir.h>
#include	<io.h>

#include	<assert.h>
#include	<sys/stat.h>
#include	"ems.h"
#include	"xms.h"
#include	"d2d.h"

/* modes of "getDevPar()": */
#define	GDP_BUILD	1	/* return "BUILD BPB" BPB */
#define GDP_DEFAULT	0	/* return default BPB */

/* modes of "setDevPar()": */
#define	BPB_BUILD	1	/* set "BUILD BPB" for specified drive */
#define	BPB_DEFAULT	0	/* set default BPB */

static	char	SCCS_ID[] =	"@(#)" __FILE__ " 2.32.1.4 "
				LICENCE " " EDITION "\t"
				__DATE__ "\t" __TIME__;
static	char	*RCS_ID	=	"$" "Id: d2d-2.c'v 2.32.1.4 1998/12/25 23:36:50 UhW Exp" " $";
static	char		*argv0;		/* argv[0] */

#ifdef	DEBUG
# define	BUG(levl)	if ( (levl) <= dbuglevl )
int	dbuglevl	= 0;		/* DEBUG level */
#endif

#define	HARDDISC	"HARDDISK"	/* environment variable */
#define	IBM_VAR		"IBM"		/* IBM flag */

static	char		*copyrght[]	= {
	"Copyright (c) 1987 - 1992, 1998 by ",
	"Ulrich Windl\n" };

static	const char	T_ready[]	= "\tPress RETURN when ready:";
static	const char	T_noFloppy[]	= "\"%c:\" is no floppy drive\n";
static	const char	T_invalid[]	= "\nDrive %c: does not exist\n";
static	const char	T_FATAL[]	= "\nFATAL: ";

/* some default BPBs */
static	const a_BPB	BPB160	= { 512, 1, 1, 2,  64,  320, 0xfe, 1,  8, 1 };
static	const a_BPB	BPB180	= { 512, 1, 1, 2,  64,  360, 0xfc, 2,  9, 1 };
static	const a_BPB	BPB320	= { 512, 2, 1, 2, 112,  640, 0xff, 1,  8, 2 };
static	const a_BPB	BPB320S	= { 512, 2, 1, 2, 112,  640, 0xfa, 1,  8, 1 };
static	const a_BPB	BPB360	= { 512, 2, 1, 2, 112,  720, 0xfd, 2,  9, 2 };
static	const a_BPB	BPB360S	= { 512, 2, 1, 2, 112,  720, 0xfc, 2,  9, 1 };
static	const a_BPB	BPB640	= { 512, 2, 1, 2, 112, 1280, 0xfb, 2,  9, 2 };
static	const a_BPB	BPB720	= { 512, 2, 1, 2, 144, 1440, 0xf9, 3,  9, 2 };
static	const a_BPB	BPB1M2	= { 512, 1, 1, 2, 224, 2400, 0xf0, 7, 15, 2 };
static	const a_BPB	BPB1M4	= { 512, 1, 1, 2, 224, 2880, 0xf0, 9, 18, 2 };
static	const a_BPB	BPB2M8	= { 512, 0, 1, 2,   0, 5760, 0xf0, 0, 36, 2 };

static	BPB_MODE	type		= BPB_DOS;	/* type of BPB to use */
static	short		copies		= 1;		/* number of copies */
static	char		ovflow_D	= '\0';		/* overflow drive */
static	char		all_flag	= FALSE;	/* copy all tracks *1.10* */
static	int		drives		= 0;		/* counter for drives */
static	int		cur_dsk;			/* current disc */
static	char		source_D;			/* source and */
static	char		dest_D;				/* destination drive */
static	EMS_STATE	EMS;				/* EMS state info */
static	XMS_STATE	XMS;				/* XMS state info */
static	DISK_STATE	DISK;				/* fixed disk info */
static	char		ovf_name[128]	= "@:\\";	/* scratch directory */
static	SWAPMODE	swapmode	= SM_RAM;	/* swap mode */
static	SWAPSTATE	swap		= { 0, 0, 0 };	/* swap state */
static	A_DEVPAR	PB;				/* Device Parameters */
static	A_DEVPAR	sourcePB;		/* holds source default format */
static	char		sourcePB_valid	= FALSE;
static	A_DEVPAR	destPB;			/* holds destination format */
static	char		destPB_valid	= FALSE;
static	char		do_form;			/* format flag */
static	char		verify		= FALSE;	/* verify flag */
static	char		smart_verify	= FALSE;	/* smart-verify flag */
static	char		new_serial;
static	char		IBM	= '1';		/* 100% IBM compatible */
static	char		alwaysFormat	= FALSE;	/* unconditional format */
static	char		skipCheck	= FALSE;	/* skip security_test */
static	char		beQuiet		= FALSE;	/* don't use bell */
static	char		dataRecovery	= FALSE;	/* try to recover data */
static	char		disableBoot	= FALSE;	/* disable boot code */
static	int		SigInt	= 0;		/* ^C Interrupt count */
static	int		enabled;		/* enable error messages */
static	struct LISThead {
		LINK	*first;			/* first link in list */
		LINK	*last;			/* last link in list */
	}		Troot;			/* list head of track-list */

/*
 *	getCR - wait for return key to be pressed
 */
static	int	getCR(void)
{
	int	key;
	int	status	= 0;

	do {
		key = getch();
		if ( key == '\3' || SigInt > 0 )
		{
			status = -1;
			break;
		}
	} while ( key != '\r' );
	fputc('\n', stderr);
	return(status);
}

/*
 *	help - print verbose explanation of arguments
 */
static	void	help(void)
{
	fprintf(stderr,				/* 1.30 */
		"/#\tnumber of copies (/12)\n"
		"/a\tcopy all tracks\n"
		"/b<str>\toverride BPB of disc (/b720kB)\n"
		"\t(<str> is one of: 160k,180k,320k,320S,360k,360S,720k,1.2M,1.4M,2.8M)\n"
		"/D\tdisable boot code on destination\n"
		"/e#\tspecify EMS usage in kB (/e128)\n"
		"/n\tnon IBM-compatible BIOS (seldom helps)\n"
		"/o\tkeep old serial number\n"
		"/q\tdon't use bell\n"
		"/r\tdata recovery -- continue after read errors\n"
		"/t<str>\ttemporary drive [and path] for swap file (/tC:\\tmp)\n"
		"/u\tunconditionally format destination disk\n"
		"/v[s]\t[smart ]verify after write (/vs)\n"
		"/w\twrite destination disk without asking\n"
		"/x#\tspecify XMS usage in kB (/x1024)\n"
		"/?\tdisplay this explanation\n"
		"\n");
	fprintf(stderr, "Copyright (c) 1987-1992, 1998 by\n");
	fprintf(stderr,
		"Ulrich Windl, Alte Regensburger Strae 11a, D-93149 Nittenau\n");
}

/*
 *	usage - print information and die
 */
static	void	usage(const char *arg)
{
	fprintf(stderr, "Version %s\n", SCCS_ID + 4);
	fprintf(stderr, "D2D" RELEASE
		" is free software and comes with ABSOLUTELY NO WARRANTY.\n");
	fprintf(stderr, "See file COPYING for details...\n\n");
	fprintf(stderr,	"usage: %s from: to: ", argv0);
#ifdef	DEBUG
	fprintf(stderr, "[%cd#] ", OPT_CHAR);
#endif
	fprintf(stderr,
		"[%c#] [%ca] [%cb<str>] [%cD]\n"
		"\t\t[%ce#] [%cn] [%co] [%cq] [%cr] [%ctd:[<path>]] [%cu]\n"
		"\t\t[%cv[s]] [%cw] [%cx#] [%c?]\n",
		OPT_CHAR, OPT_CHAR, OPT_CHAR, OPT_CHAR, OPT_CHAR, OPT_CHAR,
		OPT_CHAR, OPT_CHAR, OPT_CHAR, OPT_CHAR, OPT_CHAR, OPT_CHAR,
		OPT_CHAR, OPT_CHAR, OPT_CHAR);
	fprintf(stderr, "\n");
	if ( arg != NULL )
	{
		fprintf(stderr, "argument \"%s\" not understood\n", arg);
	}
	else
	{
		fprintf(stderr, "Press RETURN");
		if ( getCR() == 0 )
			help();
	}
	exit(1);
}

/*
 *	DOS error - display dosexterr
 */
static	void	ERROR(void)
{
	struct DOSERROR	err;
	char		*es;
	char		*cs;
	char		*ls;

	dosexterr(&err);
	fprintf(stderr, "\nDOS ERROR %d: class=%d, locus=%d, action=%d\n",
		err.de_exterror, err.de_class, err.de_locus, err.de_action);
	switch ( err.de_exterror )
	{
		case 19:
			es = "Write protect violation"; break;
		case 20:
			es = "Unknown unit"; break;
		case 21:
			es = "Device not ready"; break;
		case 22:
			es = "Unknown command"; break;
		case 23:
			es = "CRC error"; break;
		case 24:
			es = "Bad request structure length"; break;
		case 25:
			es = "Seek error"; break;
		case 26:
			es = "Unknown media"; break;
		case 27:
			es = "Sector not found"; break;
		case 28:
			es = "Printer out of paper"; break;
		case 29:
			es = "Write fault"; break;
		case 30:
			es = "Read fault"; break;
		case 31:
			es = "General failure"; break;
		case 32:
			es = "Sharing violation"; break;
		case 33:
			es = "Lock violation"; break;
		case 34:
			es = "Invalid disk change"; break;
		default:
			es = "";
	}
	switch ( err.de_class )
	{
		case 1:
			cs = "Out of resource"; break;
		case 2:
			cs = "Temporary situation"; break;
		case 3:
			cs = "Authorization problem"; break;
		case 4:
			cs = "Internal error"; break;
		case 5:
			cs = "Hardware failure"; break;
		case 6:
			cs = "System failure"; break;
		case 7:
			cs = "Application error"; break;
		case 8:
			cs = "Not found"; break;
		case 9:
			cs = "Invalid format or type"; break;
		case 10:
			cs = "Locked"; break;
		case 11:
			cs = "Storage medium problem"; break;
		case 12:
			cs = "Existing"; break;
		case 13:
			cs = "Unknown"; break;
		default:
			cs = "";
	}
	switch ( err.de_locus )
	{
		case 1:
			ls = "Unknown"; break;
		case 2:
			ls = "Block device"; break;
		case 3:
			ls = "Network"; break;
		case 4:
			ls = "Character device"; break;
		case 5:
			ls = "Memory"; break;
		default:
			ls = "";
	}
	fprintf(stderr, "(%s, %s, %s)\n", es, cs, ls);
}

/*
 *	check for floppy (uses ioctl)
 */
static	int	chk_drive(char drive)
{
	union	REGS	reg;

	reg.h.ah = IOCTL;			/* IOCTL */
	reg.h.al = 0x08;			/* Is device removable ? */
	reg.h.bl = (unsigned) drive - 'A' + 1;
	if ( reg.h.bl == 0 )
	{
		fprintf(stderr, T_invalid, drive);
		exit(2);
	}
	switch ( intdos(&reg, &reg) )
	{
		case 0x0f:
			fprintf(stderr, T_invalid, drive);
			exit(2);
		case 0x00:
			return(0);
		case 0x01:
		default:
			return(1);		/* not removable */
	}
}

/*
 *	smart bell -- conditionally ring bell
 */
static	void	smartBell(int alwaysRing)
{
	if ( !beQuiet || alwaysRing )
	{
		putc('\a', stderr);
		fflush(stderr);
	}
}

/*
 *	int 23 control-break handler
 */
static	int	i23h(void)
{
	++SigInt;
	return(1);
}

/*
 *	wstr - write string to console (ROM-BIOS)
 */
static	void	wstr(const char *text)
{
	extern	int	directvideo;

	directvideo = 0;
	for ( ; *text; ++text )
		putch(*text);
}

/*
 *	flushin - flush pending input key-strokes *1.32*
 */
static	void	flushin(void)
{
	while ( kbhit() )
		getch();
}

/*
 *	int 24 critical error handler
 */
static	int	i24h(unsigned errval, unsigned ax, unsigned bp, unsigned si)
{
	static	char	*errors[] = {
		"Write protect violation",	/* 00 */
		"Unknown unit",			/* 01 */
		"Device not ready",		/* 02 */
		"Unknown command",		/* 03 */
		"CRC error",			/* 04 */
		"Bad request structure length",	/* 05 */
		"Seek error",			/* 06 */
		"Unknown media",		/* 07 */
		"Sector not found",		/* 08 */
		"Printer out of paper",		/* 09 */
		"Write fault",			/* 0A */
		"Read fault",			/* 0B */
		"General failure",		/* 0C */
		"Sharing violation",		/* 0D */
		"Lock violation",		/* 0E */
		"Invalid disk change"		/* 0F */
			};
	register int	flags;
	register int	i;

	if ( !enabled )
		return(_HARDERR_FAIL);			/* fail the call (by force) */
	if ( (ax & 0x8000) == 0 )
	{
		putch( (ax & 0xff) + 'A' );
	}
	else
	{
		const unsigned char far	*devhdr	= MK_FP(bp, si);
		/* pointer to device driver header of failing device driver. */

		if ( (*( (const unsigned far *) (devhdr + 4)) & 0x8000) == 0 )
		{
			putch( 'A' + (ax & 0xff) );
			wstr(": Bad FAT, aborting process\r\n");
			return(_HARDERR_ABORT);
		}
		for ( devhdr += 10, i = 0; i < 8; ++i )
			putch(*devhdr++);
	}
	wstr(": ");
	wstr(errors[errval & 0x0f]);
	wstr(ax & 1 ? " (Write)" : " (Read)");

	/*   Now decide how to handle the error...
	 */
	if ( (flags = (ax & 0x3800) >> (3 + 8)) != 0 )
	{
		char	k;

		wstr(" - R)etry ? ");
		flushin(); k = getche();
		wstr("\r\n");
		if ( k == 'Y' || k == 'y' || k == 'R' || k == 'r' )
			return(_HARDERR_RETRY);	/* Retry */
		return(_HARDERR_FAIL);		/* Fail or abort */
	}
	wstr("\r\n");
	return( (flags & 1) ? _HARDERR_FAIL : _HARDERR_ABORT );
						/* Abort or retry */
}

/*
 *	get drive parameter
 */
static	int	getDevPar(char drive, A_DEVPAR *par, int mode)
{
	union REGS	reg;
	struct SREGS	seg;

	reg.h.ah = IOCTL;			/* Get Device Parameters */
	reg.h.al = GENERIC_IOCTL;
	reg.h.bl = (unsigned) drive - 'A' + 1;
	reg.h.ch = RAWIO;
	reg.h.cl = 0x60;			/* Get Device Parameters */
	reg.x.dx = (unsigned) FP_OFF(par);
	par->SpecFns = mode;			/* kind of BPB returned */
	seg.ds = (unsigned) FP_SEG(par);	/* set segment */
	if ( intdosx(&reg, &reg, &seg), reg.x.cflag != 0 )
	{
		ERROR();
		return(TRUE);
	}
#ifdef DEBUG
	BUG(5)	fprintf(stderr,
			"getDevPar(%d): Cyl:%d Sec:%d MediaType:%d Total:%d\n",
			mode, par->NuOfCyl, par->DevBPB.SecPerTrk,
			par->MediaType, par->DevBPB.TotalSec);
#endif
	return(FALSE);
}

/*
 *	set device parameters
 */
static	void	setDevPar(char drive, A_DEVPAR *par, int mode)
{
	union	REGS	reg;
	struct	SREGS	seg;

	if ( IBM != '1' )
		return;
	reg.h.ah = IOCTL;			/* Set Device Parameters */
	reg.h.al = GENERIC_IOCTL;
	reg.h.bl = (unsigned) drive - 'A' + 1;
	reg.h.ch = RAWIO;
	reg.h.cl = 0x40;			/* Set Device Parameters */
	reg.x.dx = (unsigned) FP_OFF(par);
	seg.ds = (unsigned) FP_SEG(par);	/* set segment */
	par->SpecFns = mode | 0 | 4;		/* set new default DevBPB */
	if ( intdosx(&reg, &reg, &seg), reg.x.cflag != 0 )
	{
		ERROR();
	}
#ifdef DEBUG
	BUG(5) fprintf(stderr,
		       "setDevPar(%d): Cyl:%d Sec:%d MediaType:%d Total:%d\n",
		       mode, par->NuOfCyl, par->DevBPB.SecPerTrk,
		       par->MediaType, par->DevBPB.TotalSec);
#endif
}

/*
 *	flush disk buffers (check for disk changes)
 */
static	void	resetDisks(void)
{
	union REGS	regs;

	regs.h.ah = 0x0d;
	intdos(&regs, &regs);
}

/*
 *	clean up and terminate with status ``result''
 */
static	void	cleanup(int result)
{
	setdisk(cur_dsk);
	if ( XMS.avail )
	{
#if	!30
		XMSunlock(XMS.handle);
#endif
		XMSfree(XMS.handle);
	}
	if ( EMS.avail )			/* try to free EMS pages */
		EMSfree(EMS.handle);
	if ( ovflow_D )	   		/* try to delete tempfile */
		unlink(ovf_name);
	if ( destPB_valid && dest_D != source_D )
		setDevPar(dest_D, &destPB, BPB_DEFAULT);
	if ( sourcePB_valid )			/* release fixed BPBs */
		setDevPar(source_D, &sourcePB, BPB_DEFAULT);
	resetDisks();
	exit(result);
}

/*
 *	check for interrupt (^C) and terminate if allowed (via 'cleanup')
 */
static void Interrupt(int allowed)
{
	if ( SigInt > 0 )
	{
		smartBell(0);
		fprintf(stderr, "\nInterrupt\n");
		if ( allowed )			/* terminate if allowed */
			cleanup(1);
	}
}

/*
 *	set default track layout
 */
static	void	setdefTrk(register A_DEVPAR *DPref)
{
	register int	sec;
	int		sectors;	/* sectors per track */
	int		bytes;		/* bytes per sector */

	sectors = DPref->DevBPB.SecPerTrk;
	bytes   = DPref->DevBPB.BytPerSec;
	for ( sec = 0; sec < sectors; ++sec )
	{
		DPref->TrkLay.Sector[sec].number = sec + 1;	/* sector ID */
		DPref->TrkLay.Sector[sec].size   = bytes;
					/* bytes per sector */
	}
	DPref->TrkLay.SecPerTrk = DPref->DevBPB.SecPerTrk;
}

/*
 *	"de-reference FAT pointer"
 *	(= return the contents of the given location in the FAT)
 */
static	unsigned	FAT(const unsigned char *image, unsigned index)
{
	unsigned	one_word;

	one_word = index / 2 + index;
	one_word = image[one_word] + (image[one_word + 1] << 8);
	return( (index & 1) ? one_word >> 4 : one_word & 0x0fff );
}

/*
 *	return rounded-up quotient
 */
static	unsigned	rudiv(unsigned num, unsigned denom)
{
	return( ((unsigned long) num + denom - 1) / denom );
}

/*
 *	check BPB
 *	return values:
 *	0 if ok
 *	1 if impossible values
 *	2 if inconsistent values
 *	3 if unsupported values
 */
static	int	BPBcheck(const a_BPB *BPBptr)
{
	if ( BPBptr->BytPerSec == 0 ||
	     BPBptr->SecPerClu == 0 ||
	     BPBptr->RootEntr == 0 ||
	     BPBptr->TotalSec == 0 ||
	     BPBptr->SecPerFAT == 0 ||
	     BPBptr->SecPerTrk == 0 ||
	     BPBptr->Heads == 0 )
		return(1);
	if ( BPBptr->ResSec + BPBptr->NumOfFATs * BPBptr->SecPerFAT +
	     rudiv(BPBptr->RootEntr * DIR_ENTRY_SIZE, BPBptr->BytPerSec) >
	     BPBptr->TotalSec )
		return(2);
	if ( BPBptr->BytPerSec > SECT_SIZE ||
	     BPBptr->SecPerTrk > MAX_SECT ||
	     BPBptr->TotalSec / BPBptr->SecPerTrk > MAX_TRACK ||
	     BPBptr->Heads > 2 ||
	     BPBptr->ResSec + BPBptr->SecPerFAT > BPBptr->SecPerTrk ||
	     BPBptr->BytPerSec * (unsigned long) BPBptr->SecPerTrk > 0x8000UL )
		return(3);

	return(0);
}

/*
 *	return number of used (= allocated) tracks
 */
static	unsigned	maxTrack(char drive)
{
	RWTpacket		packet;
	union REGS		reg;
	struct SREGS		seg;
	unsigned		tracks;
	unsigned		maxclust;	/* number of last cluster */
	unsigned		FATlength;
	unsigned char		*FATref;	/* memory FAT image */
	register unsigned	FATindx;	/* current FAT entry */
	register unsigned	FATnxt;		/* next FAT entry */
	unsigned		FATmax;		/* last used cluster */
	unsigned		fragments;	/* number of fragments */
	unsigned		directory;	/* clusters of directory */
	unsigned		SecPerCyl;	/* sectors per cylinder */

	sourcePB_valid = !getDevPar(drive, &sourcePB, GDP_DEFAULT);
						/* keep BPB */
	setdefTrk(&sourcePB);
	if ( type == BPB_DOS )
	{
		if ( getDevPar(drive, &PB, GDP_BUILD) ||
		     BPBcheck(&PB.DevBPB) != 0 )
		{
			fprintf(stderr, "invalid or unsupported BPB\n");
			return(0);		/* failed to get BPB */
		}
	}
	else
	{
		PB = sourcePB;			/* begin with default format */
		switch ( type )
		{
			case BPB_160:
				PB.DevBPB = BPB160; break;
			case BPB_180:
				PB.DevBPB = BPB180; break;
			case BPB_320:
				PB.DevBPB = BPB320; break;
			case BPB_320S:
				PB.DevBPB = BPB320S; break;
			case BPB_360:
				PB.DevBPB = BPB360; break;
			case BPB_360S:
				PB.DevBPB = BPB360S; break;
			case BPB_640:
				PB.DevBPB = BPB640; break;
			case BPB_720:
				PB.DevBPB = BPB720; break;
			case BPB_1M2:
				PB.DevBPB = BPB1M2; break;
			case BPB_1M4:
				PB.DevBPB = BPB1M4; break;
			case BPB_2M8:
				PB.DevBPB = BPB2M8; break;
			default:
				return(0);
		}
		if ( BPBcheck(&PB.DevBPB) != 0 )
		{
			fprintf(stderr, "invalid or unsupported BPB\n");
			return(0);
		}
	}
	setdefTrk(&PB);
	SecPerCyl = PB.DevBPB.SecPerTrk * PB.DevBPB.Heads;
	if ( PB.DevBPB.TotalSec / SecPerCyl != PB.NuOfCyl )
	{
#ifdef	DEBUG
	BUG(1)	fprintf(stderr, "Cyl: %u -> %u\n", PB.NuOfCyl,
			PB.DevBPB.TotalSec / SecPerCyl);
#endif
		PB.NuOfCyl = PB.DevBPB.TotalSec / SecPerCyl;
	}
	setDevPar(drive, &PB, BPB_DEFAULT);
	directory = PB.DevBPB.ResSec + PB.DevBPB.NumOfFATs * PB.DevBPB.SecPerFAT
		    + rudiv(PB.DevBPB.RootEntr * DIR_ENTRY_SIZE,
			    PB.DevBPB.BytPerSec);
	maxclust = rudiv(PB.DevBPB.TotalSec - directory, PB.DevBPB.SecPerClu);
#ifdef	DEBUG
	BUG(1)	fprintf(stderr, "Highest cluster = %u (%u-%u)\n", maxclust,
			rudiv(PB.DevBPB.TotalSec, PB.DevBPB.SecPerClu),
			rudiv(directory, PB.DevBPB.SecPerClu));
#endif
	if ( all_flag || type != BPB_DOS )			/*1.10,2.00*/
		return(PB.DevBPB.TotalSec / PB.DevBPB.SecPerTrk);
	FATlength = PB.DevBPB.SecPerFAT;
#ifdef	DEBUG
	BUG(5) fprintf(stderr, "FAT has %u sectors\n", FATlength);
#endif
	reg.h.ah = IOCTL;
	reg.h.al = GENERIC_IOCTL;
	reg.x.bx = (unsigned) drive - 'A' + 1;
	reg.h.ch = RAWIO;
	reg.h.cl = 0x61;	/* Read Track on Device */
	reg.x.dx = (unsigned) FP_OFF(&packet);
	seg.ds   = (unsigned) FP_SEG(&packet);	/* set segment */
	packet.SpecFns = 0;			/* all bits cleared */
	packet.Head = 0;			/* begin with head 0 */
	packet.Cyl  = 0;			/* on the very first track */
	packet.FstSec = PB.DevBPB.ResSec;	/* read first FAT sector */
	packet.nOfSec = FATlength;		/* read the whole FAT */
	if ( (packet.Transfer = (unsigned long)
		malloc(FATlength * PB.DevBPB.BytPerSec)) == NULL )
	{
		smartBell(1);
		fprintf(stderr, "%sno RAM for reading FAT\n", T_FATAL);
		cleanup(4);
	}
	if ( intdosx(&reg, &reg, &seg), reg.x.cflag != 0 )
	{
		ERROR();
		smartBell(1);
		fprintf(stderr, "%scan't read %u sectors of FAT\n",
			T_FATAL, FATlength);
		cleanup(4);
	}
	FATref = (unsigned char far *) packet.Transfer;
	/* use (12-bit) FAT to determine used disc area */
	FATindx = 2;
	FATmax = 0;
	do
	{
#ifdef	DEBUG
		BUG(8)	fprintf(stderr, "FATindx=%#03x, FATmax=%#03x\n",
				FATindx, FATmax);
#endif
		/*! follow allocation chain in FAT */
		for ( ;
		      FATindx < maxclust &&
		      (FATnxt = FAT(FATref, FATindx)) != 0 &&
		      FATnxt != 0x0ff7;
		      FATindx = FATnxt
		    )
		{
			/*! "FATref[FATindx]" contained "FATnxt" !*/
			/*! "FATnxt" belongs to chain !*/
#ifdef	DEBUG
			BUG(9)	fprintf(stderr, "FAT[0x%03x]=0x%03x\n",
					FATindx, FATnxt);
#endif
			if ( FATmax < FATindx )
				FATmax = FATindx;
			if ( FATnxt >= 0x0ff7 )	/* last cluster in chain */
				break;
			if ( FATnxt > maxclust )
			{
				fprintf(stderr,
					"FAT[0x%03x]=0x%03x: invalid cluster\n",
					FATindx, FATnxt);
				break;
	  		}
			if ( FATindx >= FATnxt )
	  		{
#ifdef	DEBUG
				BUG(1)	fprintf(stderr,
						"FAT[0x%03x]=0x%03x: break\n",
						FATindx, FATnxt);
#endif
				break;
	  		}
		} /* for */
		/*! end of allocation chain or bad FAT !*/
	} while ( ++FATindx < maxclust );
	/*! "FATmax" was last cluster !*/

	/* scan through the used clusters for bad blocks (second pass) */
	for ( fragments = 0, FATindx = 2; FATindx <= maxclust; ++FATindx )
	{
		FATnxt = FAT(FATref, FATindx);
		if ( FATnxt == 0x0ff7 )
		{
			unsigned	to;

			for ( to = FATindx + 1;
			      to <= maxclust && FAT(FATref, to) == 0x0ff7;
			      ++to )
				;
			if ( FATindx == to - 1 )
				fprintf(stderr, "0x%03x: bad cluster",
					FATindx);
			else
				fprintf(stderr, "0x%03x - 0x%03x: bad clusters",
					FATindx, to - 1);
			fprintf(stderr,	to - 1 <= FATmax ?
				" (must be readable)\n" : " (not read)\n");
			FATindx = to;
		}
		else if ( FATnxt != 0 && FATnxt < 0x0ff7 &&
			  FATnxt != FATindx + 1)
		{
#ifdef	DEBUG
			BUG(4)	fprintf(stderr,	"FAT[0x%03x]=0x%03x: fragment\n",
					FATindx, FATnxt);
#endif
			++fragments;
		}
		
	}

	FATmax += rudiv(directory, PB.DevBPB.SecPerClu);
	tracks = rudiv(FATmax * PB.DevBPB.SecPerClu, PB.DevBPB.SecPerTrk);
	if ( tracks >= PB.DevBPB.TotalSec / PB.DevBPB.SecPerTrk )
		tracks = PB.DevBPB.TotalSec / PB.DevBPB.SecPerTrk;
#ifdef	DEBUG
	BUG(5)	fprintf(stderr,	"%u (+ %u) clusters -> %u tracks)\n",
			FATmax, rudiv(directory, PB.DevBPB.SecPerClu), tracks);
#endif
	free( (void far *) packet.Transfer );
	if ( fragments > 1 )
	{
		fprintf(stderr, "%u fragments (%lu%%) found in FAT\n",
			fragments,
			(100L * fragments + maxclust / 2) / maxclust);
	}
	return(tracks);
}

/*
 *	error on read/write/format
 */
static	void	RWFerror(const char *type, unsigned cyl, unsigned head)
{
	smartBell(1);
	fprintf(stderr, "%s%s error (Cylinder %2u, Head %u)\n",
		T_FATAL, type, cyl, head);
}

/*
 *	enter new track into list.
 *	head is address of root pointer
 */
static	void	enter(struct LISThead *head, unsigned long track)
{
	register LINK	*current;

	if ( (current = (LINK *) malloc( sizeof(LINK) )) == (LINK *) NULL )
	{
		smartBell(1);
		fprintf(stderr, "%sno room for new link", T_FATAL);
		cleanup(4);
	}
	current->next = (LINK *) NULL;
	current->trackRef = track;
	if ( head->last == (LINK *) NULL )	/* first element */
		head->first = current;
	else
		head->last->next = current;	/* append "current" link */
	head->last = current;
}

/*
 * open temporary swap file
 */
static	int	swap_file(void)
{
	int	tmp;

	if ( !ovflow_D )
	{
		smartBell(1);
		fprintf(stderr,	"%sno swap disc!?\n", T_FATAL);
		cleanup(2);
	}
	/* try to open temp-file */
#ifdef	DEBUG
	BUG(6)	fprintf(stderr, "file=\"%s\"\n", ovf_name);
#endif
	if ( (tmp = open(ovf_name, FM_WRITE, S_IWRITE)) == ERR )
	{
		smartBell(1);
		fprintf(stderr,	"%scan't initialize temporary file \"%s\"\n",
			T_FATAL, ovf_name);
		cleanup(4);
	}
	return(tmp);
}

/*
 * initialize state for writing to XMS
 */
static	int	XMS_write_init(XMS_move_param *const paramp,
			       const XMS_offset addr,
			       const XMS_offset len)
{
#ifdef	DEBUG
	BUG(1)	fprintf(stderr, "(XMS write on)\n");
#endif
	swapmode = SM_XMS;
	XMS.tracks = 0;
	XMS.lastoffs = 0L;
	paramp->length = len;
	paramp->source_handle = 0;
	paramp->source_offset = addr;
	paramp->dest_handle = XMS.handle;
	swap.XMS = 1;
	return(0);
}

/*
 * initialize state for writing to EMS
 */
static	int	EMS_write_init(int *const offsetp)
{
#ifdef	DEBUG
	BUG(1)	fprintf(stderr, "(EMS write on)\n");
#endif
	swapmode = SM_EMS;
	*offsetp = 0;			/* page offset */
	EMS.tracks = 0;
	EMS.lastpg = 1;			/* last used (mapped) page */
	if ( EMSmap(0, EMS.lastpg - 1, EMS.handle) ||
	     EMSmap(1, EMS.lastpg, EMS.handle) )
	{
		smartBell(1);
		fprintf(stderr, "%sEMS map error\n", T_FATAL);
		return(-1);
	}
	swap.EMS = 1;
	return(0);
}

/*
 * initialize state for writing to disk
 */
static	int	DISK_write_init(int *const ip)
{
#ifdef	DEBUG
	BUG(1)	fprintf(stderr, "(swap on)\n");
#endif
	swapmode = SM_DISC;
	if ( (*ip = swap_file()) == ERR )
	{
		return(-1);
	}
	swap.disc = 1;
	return(0);
}

/*
 * try to recover by reading as many sectors as possible from the bad track
 */
static	void	recoverRead(const RWTpacket *rp, const int bytesPerSector)
{
	RWTpacket	readPacket;
	union REGS	inReg, outReg;
	struct SREGS	segReg;
	int		s;

	inReg.h.ah = IOCTL;		/* Read A Track */
	inReg.h.al = GENERIC_IOCTL;
	inReg.x.bx = source_D - 'A' + 1;
	inReg.h.ch = RAWIO;
	inReg.h.cl = 0x61;		/* read track */
	inReg.x.dx = (unsigned) FP_OFF(&readPacket);
	segReg.ds = (unsigned) FP_SEG(&readPacket);
	readPacket = *rp;
	readPacket.nOfSec = 1;
	fprintf(stderr, "Trying to find good sectors C%02uH%u:S",
		readPacket.Cyl, readPacket.Head);
	for ( s = 0; s < rp->nOfSec; ++s )
	{
		fprintf(stderr, "%02u",	s);
		fflush(stderr);
		Interrupt(FALSE);
		if ( SigInt > 0 )
			break;
		readPacket.FstSec = s;
		intdosx(&inReg, &outReg, &segReg);
		fprintf(stderr, "%c\b\b\b",
			outReg.x.cflag == 1 ? '-' : '+');
		if ( outReg.x.cflag == 1 )
		{
			memset((void far *) readPacket.Transfer, '\0',
			       bytesPerSector);
			sprintf((char far *) readPacket.Transfer,
				"C%02u:H%u:S%02u",
				readPacket.Cyl, readPacket.Head,
				readPacket.FstSec);
		}
		readPacket.Transfer += bytesPerSector;
	}
	fprintf(stderr, "\n");
}

/*
 * read disc into memory (optionally into XMS, EMS, and onto fixed disc)
 */
static	void	read_disc(char drive)
{
	unsigned	usedTracks;
	unsigned	track;
	RWTpacket	request;		/* requested disc position */
	unsigned	tracksize;		/* allocation unit */
	int		tmp;			/* temporary filepointer */
	XMS_move_param	move;			/* block move parameters */
	int		pgofs;			/* page offset (0-0x8000) */
	union REGS	inreg, outreg;		/* registers for DOS call */
	struct SREGS	seg;

	Troot.first = Troot.last = (LINK *) NULL;	/* clear list */
	Interrupt(TRUE);
	fprintf(stderr, "\nInsert *SOURCE* disc into drive %c:\n%s", drive,
		T_ready);
	flushin();
	if ( getCR() )
		cleanup(1);
	if ( setdisk(drive - 'A') < drive - 'A' )
	{
		fprintf(stderr, T_invalid, drive);
		cleanup(2);
	}
	usedTracks = maxTrack(drive);		/* optimize if possible */
	fprintf(stderr, "Reading %u tracks...\n", usedTracks);
	tracksize = PB.DevBPB.BytPerSec * PB.DevBPB.SecPerTrk;
	inreg.h.ah = IOCTL;		/* Read A Track */
	inreg.h.al = GENERIC_IOCTL;
	inreg.x.bx = (unsigned) drive - 'A' + 1;
	inreg.h.ch = RAWIO;
	inreg.h.cl = 0x61;
	inreg.x.dx = (unsigned) FP_OFF(&request);
	seg.ds = (unsigned) FP_SEG(&request);
	request.SpecFns = 0;			/* prepare read */
	request.FstSec  = 0;
	request.nOfSec  = PB.DevBPB.SecPerTrk;
	request.Head	= 0;			 /* default for SS (1 head) */
	for ( track = 0; track < usedTracks; ++track )
	{
		if ( PB.DevBPB.Heads == 2 )
		{
			request.Cyl = track / 2;
			request.Head = track % 2;
		}
		else
			request.Cyl = track;
		if ( swapmode == SM_RAM )
		{
			if ( (request.Transfer =
			      (unsigned long) malloc(tracksize)) == NULL )
			{
				fprintf(stderr,
					"\n!! unexpected MEMORY OVERFLOW !!\n");
				cleanup(4);
			}
			if ( coreleft() < (unsigned long) SLACK )
			{
				fprintf(stderr, "[Base memory exhausted]\n");
				if ( XMS.avail )
				{
					if ( XMS_write_init(&move,
							    request.Transfer,
							    tracksize) )
					{
						cleanup(4);
					}
				}
				else if ( EMS.avail )
				{
					if ( EMS_write_init(&pgofs) )
					{
						cleanup(4);
					}
				}
				else
				{
					if ( DISK_write_init(&tmp) )
					{
						cleanup(4);
					}
				} /* if .. else */
			} /* if ( coreleft() ) */
		} /* if ( SM_RAM ) */
		else if ( swapmode == SM_XMS )
		{
			if ( XMS.lastoffs + tracksize >= XMS.avail * 1024L )
			{
				fprintf(stderr, "[XMS exhausted]\n");
				if ( EMS.avail )
				{
					if ( EMS_write_init(&pgofs) )
					{
						cleanup(4);
					}
				}
				else
				{
					if ( DISK_write_init(&tmp) )
					{
						cleanup(4);
					}
				}
			} /* if ( XMS available ) */
		}
		else if ( swapmode == SM_EMS )
		{
			if ( (pgofs += tracksize) >= PAGESIZE )
			{
				pgofs -= PAGESIZE;
				if ( ++EMS.lastpg >= EMS.avail )
				{
					fprintf(stderr, "[EMS exhausted]\n");
					if ( DISK_write_init(&tmp) )
					{
						cleanup(4);
					}
				}
				else if ( EMSmap(0, EMS.lastpg - 1, EMS.handle)
					  || EMSmap(1, EMS.lastpg, EMS.handle) )
				{
					smartBell(1);
					fprintf(stderr, "%sEMS map error\n",
						T_FATAL);
					cleanup(4);
				}
			} /* if ( EMS page full ) */
		}
		/* no action required for other cases */

#ifdef	DEBUG
		BUG(9)	fprintf(stderr, "read(%d,%d)->%p\n", request.Cyl,
					request.Head, request.Transfer);
#endif
		if ( intdosx(&inreg, &outreg, &seg), outreg.x.cflag != 0 )
		{
			ERROR();
			RWFerror("READ", request.Cyl, request.Head);
			Interrupt(FALSE);
			if ( dataRecovery )
			{
				recoverRead(&request, PB.DevBPB.BytPerSec);
				if ( SigInt > 0 )
				{
					if ( swap.disc )
						close(tmp);
					cleanup(1);
				}
			}
			else
			{
				if ( swap.disc )
					close(tmp);
				cleanup(4);
			}
		}
		if ( swapmode == SM_RAM )	/* keep pointer to track */
		{
			enter(&Troot, request.Transfer);
		}
		else if ( swapmode == SM_XMS )
		{
			move.dest_offset = XMS.lastoffs;
#ifdef	DEBUG
			BUG(9)	fprintf(stderr, "XMSmove ->0x%lx\n",
					move.dest_offset);
#endif
			if ( XMSmove(&move) )
			{
				smartBell(1);
				fprintf(stderr, "%sXMS move failed: 0x%02x\n",
					T_FATAL, XMS_error);
				cleanup(4);
			}
			XMS.lastoffs += tracksize;
			++XMS.tracks;
		}
		else if ( swapmode == SM_EMS )
		{
#ifdef	DEBUG
			BUG(9)	fprintf(stderr, "EMSmove ->%p(%d)\n",
					EMS.frame + pgofs, EMS.lastpg);
#endif
			memcpy(EMS.frame + pgofs, (char far *) request.Transfer,
			       tracksize);
			++EMS.tracks;
		} /* if ( swap on EMS ) */
		else				/* swap on disc */
		{
			if ( _write(tmp, (char far *) request.Transfer,
				    tracksize) != tracksize )
			{
				smartBell(1);
				fprintf(stderr, "%sfile write error on %s\n",
					T_FATAL, ovf_name);
				close(tmp);
				cleanup(4);
			}		/* ``request.Transfer'' written */
		}
	} /* for ( track ) */
	if ( swap.XMS || swap.EMS || swap.disc )
	{
		close(tmp);			/* close tempfile if exits */
		free( (char far *) request.Transfer );	/* free file buffer */
	}
	if ( usedTracks == 0 )
	{
		fprintf(stderr, "Failed to read source disc\n");
		cleanup(2);				/* general failure */
	}
	setDevPar(drive, &sourcePB, GDP_DEFAULT);	/* restore BPB */
	Interrupt(TRUE);
}

/*
 *	test destination disc *1.33*
 */
static	int	security_test(char drive)
{
	struct ffblk	f1st;
	char		label[] = "@:\\*.*";
	const char	T_ATN[] = "ATTENTION: ";
	char		cont;
	A_DEVPAR	par;			/* destination disc's BPB */
	int		old_state = enabled;

	label[0] = drive;			/* try to find label */
	enabled = 0;
	if ( findfirst(label, &f1st, FA_LABEL) )
	{
		if ( errno == ENOENT && _doserrno != 3 )	/* formatted */
		{
			smartBell(0);
			fprintf(stderr, "%serase formatted disc ?", T_ATN);
		}
		/* in other cases the "i24h" handler gains control */
		else
		{
			do_form = TRUE;		/* no valid DOS format */
			enabled = old_state;
			return(FALSE);		/* overwrite it! */
		}
	}
	else
	{
		register char	*ptr	= f1st.ff_name;

		while (*ptr && *ptr != '.')	/* remove '.' from label */
			++ptr;
		if ( *ptr )
			strcpy(ptr, ptr + 1);
		smartBell(0);
		fprintf(stderr, "%serase disc named \"%s\" ?",
			T_ATN, f1st.ff_name);
	}
	enabled = old_state;
	fprintf(stderr, " Continue (y,n) "); flushin();
	do {
		if ( skipCheck )
		{
			cont = 'y';
			break;
		}
		cont = getch();
		if ( cont == '\3' || SigInt > 0 ||
		     (cont = tolower(cont)) == 'n' )
		{
			fprintf(stderr, "no\n");
			return(TRUE);	/* do NOT touch it! */
		}
	} while ( cont != 'y' );
	fprintf(stderr, "yes\n");
	/* get the actual device (=disc) parameters to examine the format */
	if ( getDevPar(drive, &par, GDP_BUILD) ||
	     BPBcheck(&par.DevBPB) != 0 ||
	     par.DevBPB.Heads != PB.DevBPB.Heads ||
	     par.DevBPB.SecPerTrk != PB.DevBPB.SecPerTrk ||
	     par.DevBPB.TotalSec != PB.DevBPB.TotalSec )
		do_form = TRUE;
	return(FALSE);			/* overwrite it! */
}

/*
 *	return address of LINK describing track image (non destructive)
 */
static	unsigned long	trackbuf(register LINK **root)
{
	register LINK	*store;

	if ( (store = *root) != NULL )
	{
		*root = store->next;		/* advance read pointer */
		return( store->trackRef );
	}
	return(NULL);
}

/*
 *	return TRUE if `bootSec' points to a valid DOS boot record
 */
static	int	bootRecord(const unsigned char far *bootSec)
{
	return ( (bootSec[0] == 0xe9 ||
		  bootSec[0] == 0xeb && bootSec[2] == 0x90) &&
		 bootSec[0x1fe] == 0x55 && bootSec[0x1ff] == 0xaa );
}

/*
 *	disable existing DOS boot record
 */
static	void	unBoot(unsigned char far *bootSec)
{
#ifdef	DEBUG
	BUG(1)	fprintf(stderr, "unBoot()\n");
#endif
	if ( bootRecord(bootSec) )
	{
		bootSec[0] = 0xeb;	/* L:	JMP SHORT L */
		bootSec[1] = 0xfe;
		bootSec[2] = 0x90;
		fprintf(stderr, "[Boot code disabled]\n");
	}
}

/*
 *	generate new volume serial number (if already present)
 */
static	void	bootfix(unsigned char far *bootsec)
{
	unsigned long	serial;
	time_t		t;

#ifdef	DEBUG
	BUG(1)	fprintf(stderr, "bootfix()\n");
#endif
	if ( bootRecord(bootsec) )
	{
#ifdef	DEBUG
		BUG(5)	fprintf(stderr, "DOS bootsector found\n");
#endif
		if ( bootsec[0x26] == 0x29 &&
		     memcmp(bootsec + 0x36, "FAT", 3) == 0 &&
		     *(const long *) (bootsec + 0x27) != 0L )
		{
#ifdef	DEBUG
			BUG(3)	fprintf(stderr, "extended bootrecord found\n");
#endif
			serial = time(&t);
			fprintf(stderr,
				"[Volume serial number is %04lX-%04lX]\n",
				serial >> 16, serial & 0xffff);
			*(long *) (bootsec + 0x27) = serial;
		}
	}
}

/*
 * initialize state for XMS read
 */
static	int	XMS_read_init(XMS_move_param *const paramp,
			      const XMS_offset addr, const XMS_offset len)
{
#ifdef	DEBUG
	BUG(1)	fprintf(stderr, "(XMS read on)\n");
#endif
	swapmode = SM_XMS;
	paramp->length = len;
	paramp->source_handle = XMS.handle;
	paramp->source_offset = 0L;
	paramp->dest_handle = 0;
	paramp->dest_offset = addr;
	if ( XMSmove(paramp) )
	{
		smartBell(1);
		fprintf(stderr,	"%sXMS move failed: 0x%02x\n",
			T_FATAL, XMS_error);
		return(-1);
	}
	paramp->source_offset += len;
	return(0);
}

/*
 * initialize state for EMS read
 */
static	int	EMS_read_init(void far *const addr, const int len,
			      int *const pagep, int *const offsetp,
			      int *const tracksp)
{
#ifdef	DEBUG
	BUG(1)	fprintf(stderr, "(EMS read on)\n");
#endif
	swapmode = SM_EMS;
	*pagep = 1;
	if ( EMSmap(0, 0, EMS.handle) ||
	     EMSmap(1, 1, EMS.handle) )
	{
		smartBell(1);
		fprintf(stderr, "%sEMS map error\n", T_FATAL);
		return(-1);
	}
	*offsetp = 0;
	memcpy(addr, EMS.frame + *offsetp, len);
	*offsetp += len;
	*tracksp = 1;
	return(0);
}

/*
 * initialize state for DISK read
 */
static	int	DISK_read_init(int *const ip, void far *addr,
			       const unsigned len)
{
#ifdef	DEBUG
	BUG(1)	fprintf(stderr, "(DISK read on)\n");
#endif
	swapmode = SM_DISC;
	if ( (*ip = open(ovf_name, FM_READ, 0)) == ERR )
	{
		smartBell(1);
		fprintf(stderr,	"%scan't open tempfile %s\n",
	 		T_FATAL, ovf_name);
		return(-1);
	}
	if ( _read(*ip, addr, len) != len)
	{
		smartBell(1);
		fprintf(stderr, "%scan't read from tempfile %s\n",
			T_FATAL, ovf_name);
		return(-1);
	}
	return(0);
}

/*
 * try to invalidate partial copy of disk
 */
static	void	invalidateDisk(const char drive)
{
	union REGS	regs;
	struct SREGS	sregs;
	A_FormPacket	packet;

	packet.Head = 0;
	packet.Cyl = 0;
	packet.SpecFns = 0;
	regs.h.ah = IOCTL;
	regs.h.al = GENERIC_IOCTL;
	regs.x.bx = (unsigned) drive - 'A' + 1;
	regs.h.ch = RAWIO;
	regs.h.cl = 0x42;			/* Format/Verify */
	regs.x.dx = (unsigned) FP_OFF(&packet);
	sregs.ds = (unsigned) FP_SEG(&packet);
	intdosx(&regs, &regs, &sregs);
}

/*
 *	write memory contents to disc
 */
static	int	write_disc(char drive)
{
	register unsigned	trk;		/* current track */
	const unsigned		source_tracks
				= PB.DevBPB.TotalSec / PB.DevBPB.SecPerTrk;
	LINK			*infoRoot	= Troot.first;
	RWTpacket		Wpacket;	/* write packet */
	A_FormPacket		Fpacket;	/* format packet */
	union REGS		formReg;	/* format parameters */
	union REGS		wrReg;		/* write parameters */
	union REGS		reg;		/* DOS result registers */
	struct SREGS		wrSeg;		/* write segment register */
	struct SREGS		Fseg;		/* format segment register */
	XMS_move_param		move;		/* XMS block move parameters */
	int			curp;		/* current page */
	int			ems_trks;	/* # of tracks read from EMS */
	int			pgofs;		/* page offset */
	int			tmp	= ERR;	/* tempfile pointer */
	char			*tmpbuf	= NULL;	/* tempfile buffer */
	const unsigned		tracksize
				= PB.DevBPB.BytPerSec * PB.DevBPB.SecPerTrk;

	do_form = alwaysFormat;
	smartBell(0);
	fprintf(stderr,
	"\nInsert *DESTINATION* disc into drive %c: (%d cop%s left)\n%s",
		drive, copies, copies != 1 ? "ies" : "y", T_ready);
	flushin();
	if ( getCR() )
		cleanup(1);
	destPB_valid = !getDevPar(drive, &destPB, GDP_DEFAULT);  /* preserve BPB */
	if ( security_test(drive) )
		return(TRUE);			/* signal error! *1.33* */
	if ( destPB.DevBPB.Heads == PB.DevBPB.Heads &&
	     destPB.DevBPB.SecPerTrk == PB.DevBPB.SecPerTrk &&
	     destPB.DevBPB.TotalSec == PB.DevBPB.TotalSec )
	{
		PB.MediaType = 0;
	}
	else
	{
		PB.MediaType = 0;
		switch ( destPB.DevType )
		{
			case 1:	/* 1.2MB */
				if ( PB.NuOfCyl <= 40 )
					PB.MediaType = 1;	/* 360kB */
				else
					PB.MediaType = 2;	/* 720kB */
				break;
			case 2:	/* 720kB */
				if ( PB.NuOfCyl <= 40 )
					PB.MediaType = 1;	/* 360kB */
				break;
			case 7:	/* 1.44MB */
				PB.MediaType = 1;	/* 720kB */
				break;
			case 8:	/* 2.88MB */
				/* not known yet */
			default:
				smartBell(1);
				fprintf(stderr,
					"%sinvalid media for device type %u\n",
					T_FATAL, PB.DevType);
				cleanup(2);
		}
#ifdef	DEBUG
		BUG(1)	fprintf(stderr, "device type=%u, media type=%u\n",
				(unsigned) PB.DevType, (unsigned) PB.MediaType);
		BUG(2)	fprintf(stderr, "cylinders=%u, heads=%u, sectors=%u\n",
				PB.NuOfCyl, PB.DevBPB.Heads,
				PB.DevBPB.SecPerTrk);
#endif
	}
	if ( destPB.NuOfCyl != PB.NuOfCyl )
	{
		fprintf(stderr,
			"Warning: writing %u cylinders in a %u cylinder drive\n",
			PB.NuOfCyl, destPB.NuOfCyl);
	}
	setDevPar(drive, &PB, BPB_BUILD);	/* fix format of source */
	Interrupt(TRUE);
	wrReg.h.ah = IOCTL;			/** prepare WRITE **/
	wrReg.h.al = GENERIC_IOCTL;
	wrReg.x.bx = (unsigned) drive - 'A' + 1;
	wrReg.h.ch = RAWIO;
	wrReg.h.cl = 0x41;			/* Write Track on Device */
	wrReg.x.dx = (unsigned) FP_OFF(&Wpacket);
	Wpacket.SpecFns = 0;			/* all bits cleared */
	Wpacket.Head = 0;			/* default: Head 0 */
	Wpacket.FstSec = 0;			/* read 1st sector */
	Wpacket.nOfSec = PB.DevBPB.SecPerTrk;	/* whole tracks only */
	wrSeg.ds = (unsigned) FP_SEG(&Wpacket);	/* set segment */
	formReg.h.ah = IOCTL;			/** prepare FORMAT **/
	formReg.h.al = GENERIC_IOCTL;
	formReg.x.bx = (unsigned) drive - 'A' + 1;
	formReg.h.ch = RAWIO;
	formReg.h.cl = 0x42;			/* Format/Verify */
	formReg.x.dx = (unsigned) FP_OFF(&Fpacket);
	Fseg.ds = (unsigned) FP_SEG(&Fpacket);
	if ( smart_verify )
		verify = !do_form;	/* only verify if not formatting */

	if ( do_form || verify )
	{
		Fpacket.Head = 0;		/* default: Head 0 */
		Fpacket.Cyl = 0;
		Fpacket.SpecFns = 1;		/**1.23* Format status check */
		intdosx(&formReg, &reg, &Fseg);
		if ( (Fpacket.SpecFns & 2) != 0 )
		{
			fprintf(stderr, "(%u cylinders, %u heads, %u sectors, "
					"device type=%u, media type=%u)\n",
				PB.NuOfCyl, PB.DevBPB.Heads,
				PB.DevBPB.SecPerTrk, (unsigned) PB.DevType,
				(unsigned) PB.MediaType);
			fprintf(stderr, "Your (ROM-) BIOS does not support "
					"this format\n");
			cleanup(2);
		}
	}

	Fpacket.SpecFns = 0;			/* do format */
	if ( do_form )
		fprintf(stderr, "Formatting%s c", verify ? "," : " while");
	else
		fprintf(stderr, "C");
	fprintf(stderr, "opying%s...\n", verify ? " and verifying" : "");
	Interrupt(TRUE);			/* last chance */
	if ( disableBoot )
		unBoot((unsigned char far *) (Troot.first->trackRef));
	if ( new_serial )
		bootfix((unsigned char far *) (Troot.first->trackRef));
	swapmode = SM_RAM;
	for ( trk = 0; trk < source_tracks; ++trk )
	{
		if ( PB.DevBPB.Heads > 1 )
			Fpacket.Cyl = trk >> 1, Fpacket.Head = trk & 1;
		else
			Fpacket.Cyl = trk;
		Wpacket.Cyl = Fpacket.Cyl;
		Wpacket.Head = Fpacket.Head;
		if ( do_form )			/* format track */
		{
			intdosx(&formReg, &reg, &Fseg);
			if ( reg.x.cflag != 0 )
			{
				ERROR();
				RWFerror("FORMAT", Fpacket.Cyl, Fpacket.Head);
				if ( tmp != ERR )
					close(tmp);
				/* Try to make disk unusable */
				invalidateDisk(drive);
				return(TRUE);	/* signal error */
			}
		}
		if ( swapmode == SM_RAM )
		{
			if ( (Wpacket.Transfer = trackbuf(&infoRoot)) == NULL )
			{
				if ( (tmpbuf = malloc(tracksize)) == NULL )
				{
					smartBell(1);
					fprintf(stderr,
						"%sno RAM for track buffer\n",
						T_FATAL);
					invalidateDisk(drive);
					cleanup(4);
				}
				Wpacket.Transfer = (unsigned long) tmpbuf;
				if ( swap.XMS && XMS.lastoffs > 0 )
				{
					if ( XMS_read_init(&move,
							   Wpacket.Transfer,
							   tracksize) )
					{
						invalidateDisk(drive);
						cleanup(4);
					}
				}
				else if ( swap.EMS && EMS.tracks > 0 )
				{
					if ( EMS_read_init(tmpbuf, tracksize,
							   &curp, &pgofs,
							   &ems_trks) )
					{
						invalidateDisk(drive);
						cleanup(4);
					}
				}
				else if ( swap.disc )	/* read data from disc */
				{
					if ( DISK_read_init(&tmp, tmpbuf,
							    tracksize) )
					{
						invalidateDisk(drive);
						cleanup(4);
					}
				}
				else	/* all data has been written */
				{
#ifdef	DEBUG
					BUG(1)	fprintf(stderr, "(done)\n");
#endif
					swapmode = SM_DONE;
				}
			}
		}
		else if ( swapmode == SM_XMS )
		{
			if ( move.source_offset >= XMS.lastoffs )
			{
				if ( swap.EMS )
				{
					if ( EMS_read_init(tmpbuf, tracksize,
							   &curp, &pgofs,
							   &ems_trks) )
					{
						invalidateDisk(drive);
						cleanup(4);
					}
				}
				else if ( swap.disc )
				{
					if ( DISK_read_init(&tmp, tmpbuf,
							    tracksize) )
					{
						invalidateDisk(drive);
						cleanup(4);
					}
				}
				else
					swapmode = SM_DONE;
			}
			else
			{
				if ( XMSmove(&move) )
				{
					smartBell(1);
					fprintf(stderr,
						"%sXMS move failed: 0x%02x\n",
						T_FATAL, XMS_error);
					invalidateDisk(drive);
					cleanup(4);
				}
				move.source_offset += tracksize;
			}
		}
		else if ( swapmode == SM_EMS )
		{
			if ( ++ems_trks > EMS.tracks )
			{
				if ( swap.disc )
				{
					if ( DISK_read_init(&tmp, tmpbuf,
							    tracksize) )
					{
						invalidateDisk(drive);
						cleanup(4);
					}
				}
				else
					swapmode = SM_DONE;
			}
			else	/* there is still another track */
			{
				if ( pgofs >= PAGESIZE )
				{
					++curp;
					pgofs -= PAGESIZE;
					if ( EMSmap(0, curp - 1, EMS.handle) ||
					     EMSmap(1, curp, EMS.handle) )
					{
						smartBell(1);
						fprintf(stderr,
							"%sEMS map error\n",
							T_FATAL);
						invalidateDisk(drive);
						cleanup(4);

					}
				}
				memcpy(tmpbuf, EMS.frame + pgofs, tracksize);
				pgofs += tracksize;
			}
		}
		else if ( swapmode == SM_DISC )
		{
			int	size	= _read(tmp, tmpbuf, tracksize);
			
			if ( size == 0 )
				swapmode = SM_DONE;
			else if ( size != tracksize )
			{
				smartBell(1);
				fprintf(stderr,
					"%sread from tempfile returned %d\n",
					T_FATAL, size);
				invalidateDisk(drive);
				cleanup(4);
			}
		}
		if ( swapmode != SM_DONE )	/* more data to write */
		{
#ifdef	DEBUG
			BUG(9)	fprintf(stderr, "write(%d,%d) <- %p\n",
					Wpacket.Cyl, Wpacket.Head,
					Wpacket.Transfer);
#endif
			intdosx(&wrReg, &reg, &wrSeg);
			if ( reg.x.cflag != 0 )
			{
				ERROR();
				RWFerror("WRITE",
					 Wpacket.Cyl, Wpacket.Head);
				/* Try to make disk unusable */
				invalidateDisk(drive);
				return(TRUE);	/* signal error */
			}
			if ( verify )
			{
				formReg.h.cl = 0x62;	/* verify a track */
				intdosx(&formReg, &reg, &Fseg);
				formReg.h.cl = 0x42;	/* format and verify */
				if ( reg.x.cflag != 0 )
				{
					ERROR();
					RWFerror("VERIFY", Fpacket.Cyl,
						 Fpacket.Head);
					/* Try to make disk unusable */
					invalidateDisk(drive);
					return(TRUE);	/* signal error */
				}
			}
		} /* if ( swapmode != SM_DONE ) */
#ifdef	DEBUG
		else
		{
			BUG(1)	fprintf(stderr, "DONE in track %d\n", trk);
		}
#endif
	} /* for ( trk ) */
	if ( tmpbuf != NULL )
		free(tmpbuf);			/* free tempfile buffer */
	if ( swap.disc )
	{
		close(tmp);
	}
	setDevPar(drive, (source_D == dest_D) ? &sourcePB : &destPB,
		  GDP_DEFAULT);			/* restore BPB */
	return(FALSE);				/* signal OK */
}

/*
 * numstr - convert large numbers into `short' format
 */
static	char	*numstr(unsigned long num, char str[])
{
	const char	*cp	= "%luB";

	if ( num > 9999L )
	{
		num /= 1024L;
		if ( num > 9999L )
		{
			num /= 1024;
			if ( num > 9999L )	/* wow! */
				num /= 1024L, cp = "%luGB";
			else
				cp = "%luMB";
		}
		else
			cp = "%lukB";
	}
	sprintf(str, cp, num);
	return(str);
}

/*
 * strout - print string and begin a new line if string would not fit
 * on current line. Only printable characters and '\n' are handled correctly.
 * strout(NULL) resets the internal position.
 */
static	void	strout(const char *str)
{
	static	int	len	= 0;
	const char	*cp;
	int		l;

	if ( str == NULL )		/* reset position */
		len = 0;
	else if ( (cp = strchr(str, '\n')) != NULL )
	{
		for ( ; str != cp; ++str )
			fputc(*str, stderr);
		fputc('\n', stderr);
		len = 0;
		strout(cp + 1);		/* process rest of string */
	}
	else
	{
		l = strlen(str);
		if ( (len += l) >= 80 )	/* won't fit on this line */
		{
			fputc('\n', stderr);
			len = l;
		}
		fprintf(stderr, "%s", str);
	}
}

/*
 * XMScheck - test XMM/XMS and set up XMS-info
 */
static	int	XMScheck(void)
{
	XMS_version_info	info;
	XMS_extended_info	meminfo;
	XMS_handle		handle;
#if	!30
	XMS_offset		offset;
#endif
	char			line[80];

	XMS.avail = 0L;
	if ( !XMSsetup() )
	{
#ifdef	DEBUG
		BUG(99)	fprintf(stderr, "XMS driver found\n");
#endif
		if ( XMSget_version_info(&info) ||
		     XMSquery_extended_memory(&meminfo) )
			return(-1);
		if ( meminfo.largest_free_block < XMS.request )
			XMS.request = meminfo.largest_free_block;
		if ( XMS.request > 0 && !XMSalloc(XMS.request, &handle) &&
#if	!30
		     !XMSlock(handle, &offset) )
#else
		     1 )
#endif
		{
			unsigned		i;
			volatile unsigned	j;
			XMS_move_param		m;
			int			status	= 0;

#ifdef	DEBUG
			BUG(99)	fprintf(stderr, "XMS memory block ok\n");
#endif
			m.length = sizeof(i);
			m.source_handle = 0;
			m.source_offset = (XMS_offset) &i;
			m.dest_handle = handle;
			for ( i = 0; i < XMS.request; ++i )
			{
				m.dest_offset = 1024L * i;
				if ( XMSmove(&m) )
				{
					status = -1;
					break;
				}
			}
#ifdef	DEBUG
			BUG(99)	fprintf(stderr, "XMS set memory block ok\n");
#endif
			m.source_handle = handle;
			m.dest_offset = (XMS_offset) &j;
			m.dest_handle = 0;
			for ( i = 0; i < XMS.request; ++i )
			{
				m.source_offset = 1024L * i;
				if ( XMSmove(&m) || i != j )
				{
#ifdef	DEBUG
					BUG(99)	fprintf(stderr,
				"XMS get memory block fail at 0x%08x: %u:%u\n",
						m.source_offset, i, j);
#endif
					status = -1;
					break;
				}
			}
#ifdef	DEBUG
			BUG(99)	fprintf(stderr, "XMS get memory block ok\n");
#endif
#if	!30
			if ( !XMSunlock(handle) && !XMSfree(handle) && !status )
#else
			if ( !XMSfree(handle) && !status )
#endif
			{
				char		availstr[8], totalstr[8];
				unsigned long	avail, total;

#ifdef	DEBUG
				BUG(99)	fprintf(stderr,
						"XMS free memory block ok\n");
#endif
				avail = XMS.request * 1024L;
				total = meminfo.total_extended_memory * 1024L;
				sprintf(line, "[%s/%s XMS %d.%d] ",
					numstr(avail, availstr),
					numstr(total, totalstr),
					info.XMS_version / 100,
					info.XMS_version % 100);
				strout(line);
				XMS.avail = XMS.request;
				if ( avail > 2L * MAX_SECT * SECT_SIZE )
					return(0);
			}
			else
			{
				strout("[XMS error] ");
			}
		}
	}
	XMS.avail = 0;
	return(-1);
}

/*
 * EMScheck - test EMS and set up EMS-info
 */
static	int	EMScheck(void)
{
	int		status	= 0;
	unsigned	frame, tot, handle, version;
	char		line[80];

	EMS.avail = 0;
	if ( !EMSsetup() )
	{
#ifdef	DEBUG
		BUG(99)	fprintf(stderr,	"EMS driver ok\n");
#endif
		if ( EMSpageframe(&frame) || EMSpagecount(&EMS.avail, &tot) )
			return(-1);
		if ( EMS.avail < EMS.request )
			EMS.request = EMS.avail;
		if ( !EMSalloc(EMS.request, &handle) )
		{
			unsigned	i;

#ifdef	DEBUG
			BUG(99)	fprintf(stderr,	"EMS alloc pages ok\n");
#endif
			EMS.frame = MK_FP(frame, 0);
			for ( i = 0; i < EMS.request; ++i )
			{
				if ( EMSmap(0, i, handle) )
				{
					status = -1;
					break;
				}
				*(unsigned *) EMS.frame = i;
			}
#ifdef	DEBUG
			BUG(99)	fprintf(stderr,	"EMS set pages ok\n");
#endif
			for ( i = 0; i < EMS.request; ++i )
			{
				if ( EMSmap(0, i, handle) ||
				     *(unsigned *) EMS.frame != i )
				{
#ifdef	DEBUG
					BUG(99)	fprintf(stderr,
					"EMS verify fail page %u:%u\n",
							i, *EMS.frame);
#endif
					status = -1;
					break;
				}
			}
#ifdef	DEBUG
			BUG(99)	fprintf(stderr,	"EMS verify pages ok\n");
#endif
			if ( !EMSfree(handle) && !status &&
			     !EMSversion(&version) )
			{
				char	avail[8], total[8];

#ifdef	DEBUG
				BUG(99)	fprintf(stderr,	"EMS free pages ok\n");
#endif
				sprintf(line, "[%s/%s EMS %d.%d at %04X0] ",
					numstr(EMS.request * 16L * 1024L, avail),
					numstr(EMS.avail * 16L * 1024L, total),
					(version >> 4) & 0x0f, version & 0x0f,
					frame);
				strout(line);
				EMS.avail = EMS.request;
				if ( EMS.avail > 1 )
					return(0);
			}
			else
			{
				strout("[EMS error] ");
			}
		}
	}
	EMS.avail = 0;
	return(-1);
}

/*
 * findhd - try to find harddisc
 */
static	char	findhd(void)
{
	static	const char	*evars[]	=
		{
			HARDDISC, "TMP", "TEMP", "TMPDIR"
		};
	const char		*envstr;
	int			i;
	char			drive;

	for ( i = 0; i < sizeof(evars) / sizeof(char *); ++i )
	{
		if ( (envstr = getenv(evars[i])) != NULL && envstr[1] == ':' )
		{
			drive = toupper(envstr[0]);
#ifdef	DEBUG
			fprintf(stderr, "finddh %s=%s\n", evars[i], envstr);
#endif
			if ( chk_drive(drive) == 1 )
			{
				/* try to use path as new scratch directory */
				if ( envstr[2] != '\0' )
				{
					char	last;

					strcpy(ovf_name + 2, envstr + 2);
					last = ovf_name[strlen(ovf_name) - 1];
					if ( last != '\\' && last != '/' &&
					     last != ':' )
						strcat(ovf_name, "\\");
				}
				ovf_name[0] = drive;
				return(drive);
			}
		}
	}
	return('\0');
}

/*
 * swapCheck - test swap device
 */
static	int	swapCheck(void)
{
	int		fd;
	union REGS	regs;
	unsigned char	drive	= ovflow_D - 'A';
	struct dfree	info;
	char		line[80];

	DISK.avail = 0;
	if ( ovflow_D && DISK.request > 0 )
	{
		/* try to open temp-file */
		strcat(ovf_name, "$D2D$XXXXXX");
		mktemp(ovf_name);
#ifdef	DEBUG
		BUG(99)	fprintf(stderr, "file=\"%s\"\n", ovf_name);
#endif
		if ( (fd = open(ovf_name, FM_CREATE, S_IWRITE)) == ERR )
		{
			ovflow_D = '\0';
			strout("[swap file error] ");
			return(-1);
		}
		regs.h.ah = IOCTL;
		regs.h.al = 0x00;
		regs.x.bx = fd;
		intdos(&regs, &regs);
		if ( regs.x.cflag == 0 && (regs.x.dx & 0x0080) == 0 )
			drive = regs.x.dx & 0x001f;
		close(fd);
		getdfree(drive + 1, &info);
		if ( info.df_sclus != 0xffff )
		{
			char			request[8], avail[8];
			const unsigned long	factor
					= (unsigned long) info.df_sclus *
					  info.df_bsec;

			DISK.avail = (unsigned) (info.df_avail * factor / 1024);
			if ( DISK.avail < DISK.request )
				DISK.request = DISK.avail;
			sprintf(line, "[%s/%s on disc %c:] ",
				numstr(DISK.request * 1024L, request),
				numstr(info.df_avail * factor, avail),
				drive + 'A');
			strout(line);
			DISK.avail = DISK.request;
			if ( DISK.avail * 1024L >= MAX_SECT * SECT_SIZE )
				return(0);
		}
		unlink(ovf_name);
	}
	DISK.avail = 0;
	ovflow_D = '\0';
	return(-1);
}

/*
 *	forward - commandline parser utility
 *	Try to go forward `n' characters if `n' is not zero; otherwise try to
 *	go forward until the next occurrence of OPT_CHAR. If it is not possible
 *	to go far enough, then empty the string. If n > 0 and a remaining rest
 *	does not begin with OPT_CHAR, then call usage().
 */
static	void	forward(char **strp, int n)
{
	char	*cp;

	if ( n > 0 )
	{
		for ( cp = *strp; **strp && n > 0; --n )
			++*strp;
		if ( n > 0 || **strp && **strp != OPT_CHAR )
			usage(n > 0 ? cp : *strp);
	}
	else
	{
		for ( cp = *strp; *cp != '\0' && *cp != OPT_CHAR; ++cp )
			;
		*strp = cp;		/* **strp == '\0' */
	}
}

/*
 *	main program checks for arguments
 */
void	main(int argc, char *argv[])
{
	char		*envref;	/* result of getenv() */
	unsigned	dos_version;	/* correct MS-DOS version */
	int		requirement;	/* extra storage requirement in kB */
	char		line[80];	/* text buffer for status */
	unsigned	u;		/* general purpose unsigned */

#ifdef	DEBUG
	fprintf(stderr, "<<<%s>>>\n", SCCS_ID);
	BUG(1)	fprintf(stderr, "MSDOS %d.%d (%lu bytes free)\n",
			 _osmajor, _osminor, coreleft());
#endif
	argv0 = argv[0];		/* preserve argv[0] */
	swab((char *) &_version, (char *) &dos_version, sizeof(unsigned));
	if ( dos_version < ((3 << 8) | 20) )
	{	/* *1.20*,*2.00* */
		fprintf(stderr, "%s wants DOS 3.20 at least\n", argv0);
		fprintf(stderr, "But this looks like DOS %d.%d\n",
			dos_version >> 8, dos_version & 0xff);
		exit(1);
	}
	cur_dsk = getdisk();		/* get current disc */
	ovflow_D = findhd();

	if ( (envref = getenv(IBM_VAR)) != NULL && envref[1] == '\0' )
	{
		IBM = *envref;
#ifdef	DEBUG
		fprintf(stderr, "env %s=%s\n", IBM_VAR, envref);
#endif
	}
	new_serial = 1;
	XMS.request = MAX_DISC;
	EMS.request = (MAX_DISC + 15) / 16;
	while ( --argc )
	{
		if ( **++argv == OPT_CHAR )	/* argument is option */
		{
			while ( **argv == OPT_CHAR )
			{
				switch ( *++*argv )	/* next char of argument */
				{
#ifdef	DEBUG
				    case 'd':
					if ( *++*argv && isdigit(**argv) )
						dbuglevl = atoi(*argv);
					else if ( --argc && (dbuglevl = atoi(*++argv)) > 0 )
						;
					else
						usage(*argv - 2);
					fprintf(stderr, "DEBUG level = %d\n", dbuglevl);
					forward(argv, 0);
					break;
#endif
				    case 'A':			/* 1.10 */
				    case 'a':
					all_flag = TRUE;
					forward(argv, 1);
					break;
				    case 'B':
				    case 'b':
					if ( *++*argv )
					{
					    if ( !strncmp(*argv, "160k", 4) )
						type = BPB_160;
					    else if ( !strncmp(*argv, "180k", 4) )
						type = BPB_180;
					    else if ( !strncmp(*argv, "320k", 4) )
						type = BPB_320;
					    else if ( !strncmp(*argv, "320S", 4) )
						type = BPB_320S;
					    else if ( !strncmp(*argv, "360k", 4) )
						type = BPB_360;
					    else if ( !strncmp(*argv, "360S", 4) )
						type = BPB_360S;
					    else if ( !strncmp(*argv, "640k", 4) )
						type = BPB_640;
					    else if ( !strncmp(*argv, "720k", 4) )
						type = BPB_720;
					    else if ( !strncmp(*argv, "1.2M", 4) )
						type = BPB_1M2;
					    else if ( !strncmp(*argv, "1.4M", 4) )
						type = BPB_1M4;
					    else if ( !strncmp(*argv, "2.8M", 4) )
						type = BPB_2M8;
					    else
						usage(*argv - 2);
					    forward(argv, 4);
					    break;
					}
					else
						usage(*argv - 2);
				    case 'D':
					disableBoot = TRUE;
					forward(argv, 1);
					break;
				    case 'E':
				    case 'e':
				    	if ( *++*argv && isdigit(**argv) )
				    	{
				    		u = rudiv(atoi(*argv), 16);
				    		if ( u <= EMS.request )
				    			EMS.request = u;
				    		forward(argv, 0);
				    		break;
				    	}
				    	else
				    		usage(*argv - 2);
				    case 'N':
				    case 'n':
					IBM = '0';
					forward(argv, 1);
					break;
				    case 'O':
				    case 'o':
					new_serial = 0;
					forward(argv, 1);
					break;
				    case 'Q':
				    case 'q':
					beQuiet = TRUE;
					forward(argv, 1);
					break;
				    case 'R':
				    case 'r':
				        dataRecovery = TRUE;
					forward(argv, 1);
					break;
				    case 'T':
				    case 't':
					if ( *++*argv && (*argv)[1] == ':' )
					{
						char	*cp = *argv;

						forward(argv, 0);
						strcpy(ovf_name, cp);
						ovflow_D = toupper(*cp);
						cp = ovf_name + (*argv - cp);
						if ( cp[-1] != '\\' &&
						     cp[-1] != '/' &&
						     cp[-1] != ':' )
							*cp++ = '\\';
						*cp = '\0';
						break;
					}
					else
						usage(*argv - 2);
				    case 'U':
				    case 'u':
				        alwaysFormat = TRUE;
				        forward(argv, 1);
				        break;
				    case 'V':
				    case 'v':
				    	if ( (*argv)[1] == 'S' ||
					     (*argv)[1] == 's' )
					{
						smart_verify = TRUE;
						forward(argv, 2);
					}
					else
					{
						verify = TRUE;
			    			forward(argv, 1);
			    		}
				    	break;
				    case 'W':
				    case 'w':
				        skipCheck = TRUE;
				        forward(argv, 1);
				        break;
				    case 'X':
				    case 'x':
				    	if ( *++*argv && isdigit(**argv) )
				    	{
				    		u = atoi(*argv);
				    		if ( u <= XMS.request )
				    			XMS.request = u;
				    		forward(argv, 0);
				    		break;
				    	}
				    	else
				    		usage(*argv - 2);
				    case '?':
				    	usage(NULL);
				    default:
					if ( isdigit(**argv) &&
					     (copies = atoi(*argv)) > 0 )
					{
						forward(argv, 0);
						break;
					}
					else
						usage(*argv - 1);
				} /* switch */
			} /* while ( **argv == OPT_CHAR ) */
			if ( **argv )
				usage(*argv);	/* unprocessed rest of arg */
		}
		else			/* argument is not an option */
		{
			if ( strcmp(*argv + 1, ":") == 0 && drives < 2 )
			{
				if ( drives == 0 )	/* 1st name */
				{
					source_D = toupper(**argv);
					++drives;
				}
				else if ( drives == 1 )	/* 2nd name */
				{
					dest_D = toupper(**argv);
					++drives;
				}
			}
			else
				usage(*argv);
		} /* if ... else */
	} /* while ( --argc ) */
	if ( drives < 2 )
	{
		if ( drives == 1 )
		{
			fprintf(stderr, "(using drive %c: as destination)\n",
				source_D);
			dest_D = source_D;
		}
		else
			usage(NULL);
	}
	if ( chk_drive(source_D) != 0 )		/* Fixed Disk & others */
	{
		fprintf(stderr, T_noFloppy, source_D);
		exit(2);
	}
	if ( chk_drive(dest_D) != 0 )
	{
		fprintf(stderr, T_noFloppy, dest_D);
		exit(2);
	}
	if ( ovflow_D && chk_drive(ovflow_D) != 1 )
	{
		fprintf(stderr, "\"%c:\" is no fixed disc\n", ovflow_D);
		exit(2);
	}
	fprintf(stderr,
		"Free Disc-to-Disc copy program (D2D" RELEASE "), edition `"
		EDITION "'\n");
	strout(NULL); strout(copyrght[0]); strout(copyrght[1]);
	/* begin status display */
	requirement = MAX_DISC;
	sprintf(line, "[%lukB Base Memory] ", coreleft() / 1024L);
	strout(line);
	requirement -= (int) ((coreleft() - coreleft() / 128L - SLACK) / 1024L);
	if ( XMS.request > requirement )
		XMS.request = requirement;
	if ( !XMScheck() )
	{
		if ( XMSalloc(XMS.avail, &XMS.handle) ||
#if	!30
		     XMSlock(XMS.handle, &XMS.offset) )
#else
		     0 )
#endif
		{
			strout("Can't allocate XMS memory\n");
			exit(4);
		}
		requirement -= XMS.avail;
	}
	if ( EMS.request > requirement / 16 )
		EMS.request = rudiv(requirement, 16);
	if ( !EMScheck() )
	{
		char	name[9];

		if ( EMSalloc(EMS.avail, &EMS.handle) )
		{
			strout("Can't allocate EMS pages\n");
			exit(4);
		}
		sprintf(name, "D2D-%04u", EMS.avail);
		EMSsetname(EMS.handle, name);
		requirement -= EMS.avail * 16;
		if ( requirement < 0 )
			requirement = 0;
	}
	DISK.request = requirement;
	if ( !swapCheck() )
	{
		requirement -= DISK.avail;
	}
	strout("\n\n");
	if ( requirement > 0 )
	{
		char	number[8];

		fprintf(stderr,
			"You will need %s more memory to copy all disc types.\n",
			numstr(requirement * 1024L, number));
	}
	fprintf(stderr,
		"Copy from drive %c: to drive %c: producing %d %s%s.\n",
		source_D, dest_D, copies,
		(verify || smart_verify) ? "verified " : "",
		(copies == 1) ? "copy" : "copies");
	ctrlbrk(i23h);				/* catch BREAK and ERRORs */
	harderr(i24h);
	enabled = 1;
	read_disc(source_D);			/* read source disc */
	while ( copies )
	{
		resetDisks();
		if ( !write_disc(dest_D) )	/**1.32**/
			--copies;
	}
	fprintf(stderr, "You succeeded in copying the disc.\n");
	cleanup(0);				/* exit normally */
}
