Unit SortInps;
(* SortDemo Version 2.4: 19 Mar 1995                                        *)

(* Copyright (C) 1995 Tapirsoft, Harald Selke                               *)
(* Based on a programme by K.L. Noell.                                      *)
(* Implementation of the input procedures for the SortDemo package          *)
(* See SortDemo.Doc for full documentation.                                 *)
(*                                                                          *)
(* This programme is free software; you can redistribute it and/or modify   *)
(* it under the terms of the GNU General Public License (version 1) as      *)
(* published by the Free Software Foundation.                               *)
(*                                                                          *)
(* This programme is distributed in the hope that it will be useful,        *)
(* but WITHOUT ANY WARRANTY; without even the implied warranty of           *)
(* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *)
(* GNU General Public License for more details.                             *)
(*                                                                          *)
(* You should have received a copy of the GNU General Public License        *)
(* along with this programme; if not, write to the Free Software            *)
(* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                *)


{$A+}                        (* Word alignment on                   *)
{$B-}                        (* Boolean complete evaluation off     *)
{$D+}                        (* Debug information on                *)
{$E+}                        (* Emulate coprocessor if necessary    *)
{$F-}                        (* Don't force far calls               *)
{$G-}                        (* Don't generate 80286 code           *)
{$I+}                        (* I/O checking on                     *)
{$L+}                        (* Generate local symbol information   *)
{$N-}                        (* No numeric processing               *)
{$O-}                        (* No overlaying                       *)
{$R+}                        (* Range checking on                   *)
{$S+}                        (* Stack checking on                   *)
{$V-}                        (* No var-string checking              *)
{$X-}                        (* No extended syntax allowed          *)

Interface

Const NMax = 1024;                                (* Maximum number of keys *)

Type  String12 = String [12];

Var   A    : Array [0..NMax] Of Integer;               (* A [0] is sentinel *)
      XMax, YMax, XDist, YDist,
      DotColour, BackColour : Integer;

(* A contains the numbers to be sorted; the first key is stored in A [1],   *)
(* A [0] is supposed to contain -1 as sentinel for some algorithms.         *)
(* XMax is supposed to contain the value of GetMaxX, YMax that of GetMaxY.  *)
(* XDist and YDist give the distance of the pixels that correspond to the   *)
(* numbers in horizontal and vertical direction, respectively.              *)
(* DotColour is the colour of the pixels, BackColour that of the background.*)
(* XMax and BackColour are not used within this module, they are defined    *)
(* here for sake of completeness.                                           *)

(* Two procedures to draw and erase a dot. *)

Procedure DrawDot (i, j : Integer);

Procedure EraseDot (i, j : Integer);

(* In the following procedures, n is the number of keys to be initialized.  *)
(* The keys' values are between 1 and range, including these two.           *)

Procedure RandomInit (n, range : Integer);

Procedure ReversInit (n, range : Integer);

Procedure SortedInit (n, range : Integer);

Procedure AlmostInit (n, range, perc : Integer);
(* perc is the maximum percentage of keys out of order. *)

Function FileInit (IFName : String12; Var n, range : Integer;
                   show : Boolean) : Integer;

Implementation
Uses Graph;

Procedure DrawDot;
Begin
  If (XDist = 1) And (YDist = 1) Then
    PutPixel ((i-1)*XDist, YMax - j*YDist, DotColour)
  Else
  Begin
    SetFillStyle (SolidFill, DotColour);
    Bar ((i-1) * XDist, YMax - j * YDist,
         i * XDist - 1, YMax - (j-1) * YDist - 1)
  End
End;

Procedure EraseDot;
Begin
  If (XDist = 1) And (YDist = 1) Then
    PutPixel ((i-1)*XDist, YMax - j*YDist, BackColour)
  Else
  Begin
    SetFillStyle (SolidFill, BackColour);
    Bar ((i-1) * XDist, YMax - j * YDist,
         i * XDist - 1, YMax - (j-1) * YDist - 1)
  End
End;

Procedure RandomInit;
(* Initializes the array A with random values between 1 and range and plots *)
(* them on the screen. A [0] is initialized with -1 to serve as sentinel    *)
(* for some algorithms.                                                     *)
Var i : Integer;
Begin                                                         (* RandomInit *)
  A [0] := -1;
  For i := 1 To n Do
  Begin
    A [i] := Random (range) + 1;
    DrawDot (i, A [i]);
  End
End;   (* RandomInit *)

Procedure ReversInit;
(* Initializes the array A with the numbers 1 to range in reverse order and *)
(* plots them on the screen. A [0] is initialized with -1 to serve as sen-  *)
(* tinel for some algorithms.                                               *)
Var i : Integer;
    scale : Real;
Begin                                                         (* ReversInit *)
  scale := (range-1) / (n-1);
  A [0] := -1;
  For i := 1 To n Do
  Begin
    A [i] := Trunc ((n-i) * scale) + 1;
    DrawDot (i, A [i]);
  End;
End;   (* ReversInit *)

Procedure SortedInit;
(* Initializes the array A with the numbers 1 to range and plots them on    *)
(* the screen. A [0] is initialized with -1 as sentinel.                    *)
Var i : Integer;
    scale : Real;
Begin                                                         (* SortedInit *)
  scale := (range-1) / (n-1);
  A [0] := -1;
  For i := 1 To n Do
  Begin
    A [i] := Trunc ((i-1) * scale) + 1;
    DrawDot (i, A [i]);
  End;
End;   (* SortedInit *)

Procedure AlmostInit;
(* Initializes the array A with numbers from 1 to range, so that at most a  *)
(* percentage of perc elements are in the wrong place. perc is a constant   *)
(* defined at the top of the programme. The dots are plotted on the screen. *)
(* A [0] is initialized with -1 as sentinel.                                *)
Var i, j, k, m, tmp : Integer;
    scale : Real;
Begin                                                         (* AlmostInit *)
  scale := (range-1) / (n-1);
  A [0] := -1;
  For i := 1 To n Do A [i] := Trunc ((i-1) * scale) + 1;
  m := n * perc Div 200;
  For i := 1 To m Do
  Begin
    j := Random (n-1) + 1; k := Random (n-1) + 1;
    tmp := A [j]; A [j] := A [k]; A [k] := tmp
  End;
  For i := 1 To n Do DrawDot (i, A [i]);
End;   (* AlmostInit *)

Function FileInit;
(* Initializes the array A with numbers read from the text file IFName      *)
(* which must contain at least 2 nonnegative numbers. Only the first XMax   *)
(* numbers will be read, even if the file is longer. If the maximal value   *)
(* is greater than YMax, the whole input is scaled down to fit to the       *)
(* screen. The dots are plotted on the screen. A [0] is initialized with -1 *)
(* as sentinel.                                                             *)
Var i, IORes, MaxValue : Integer;
    InputFile : Text;
    Buf : Array [1..5000] Of Char;
    c : Char;
    scale : Real;
Begin                                                           (* FileInit *)
  If Pos ('.', IFName) = 0 Then IFName := IFName + '.Inp';
  {$I-}
  Assign (InputFile, IFName);
  SetTextBuf (InputFile, Buf);
  Reset (InputFile);
  {$I+}
  IORes := IOResult;
  If IORes <> 0 Then Begin FileInit := 10; Exit End
  Else
  Begin
    A [0] := -1;
    MaxValue := 0;
    i := 1;
    While (Not EoF (InputFile)) And (i <= XMax) Do
    Begin
      ReadLn (InputFile, A [i]);
      If A [i] > MaxValue Then MaxValue := A [i];
      If A [i] < 0 Then
      Begin
        FileInit := 11;
        Close (InputFile);
        Exit
      End;
      Inc (i)
    End;
    Dec (i);
    Close (InputFile);
    If i < 2 Then
    Begin
      FileInit := 12;
      Exit
    End;
    n := i;
    If MaxValue > YMax Then
    Begin
      scale := YMax / MaxValue;
      For i := 1 To n Do A [i] := Trunc (A [i] * scale);
      MaxValue := YMax
    End;
    range := MaxValue;
    XDist := XMax Div n;
    YDist := YMax Div range;
    If show Then For i := 1 To n Do DrawDot (i, A [i]);
  End;
  FileInit := 0
End;   (* FileInit *)

End.
