/*
 * C code editing commands for Epsilon 5.0
 *
 * Copyright (C) 1985, 1986, 1987 Free Software Foundation, Inc.
 * Copyright (C) 1991, K. Shane Hartman and the Free Software Foundation, Inc.
 *
 * This file was part of GNU Emacs.
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details (the file COPYING)
 *
 * If you want a copy of the GNU General Public License, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * Most of this code was ported to EEL by me from Gnu Emacs lisp sources.  
 * The remainder was created by me to facilitate the port.  All code is
 * Copyleft by K. Shane Hartman and the Free Software Foundation.  See the
 * file COPYING for specific terms of the license.
 *
 * This file is a straight replacement for Lugaru's C.E.  Just compile
 * load and save state.  You may want to play with the control variables
 * to get the style you want.  See below.
 *
 * WARNING: The Gnu C indenter is very good, but searches a lot.  If your
 * machine is wimpy, I would stick with Lugaru's indenter which is faster
 * and dumber.  I use a 486/33 so I don't notice the extra computation.  Also,
 * I use a limited lookback parser to handle most of the simple cases which
 * assumes the indentation style used in this file.  The general indenter is 
 * used for hairy cases.  It is disabled by default, but see variable
 * c-indenter-shortcut.  Without the quick-indenter, large functions require
 * noticeable but tolerable time to indent.  The reason is that Lugaru's
 * regex searches really suck.
 *
 * This file is a straight replacement for Lugaru's c.e.  Note the improved
 * C-M-F, C-M-B commands and the new C-M-A, C-M-E, C-M-U, C-M-Q commands
 *
 * Note I have only debugged the Gnu C style, if your style is different,
 * you may have to make some modifications although I have looked at
 * various settings of the control variables which seem to work.  Don't
 * know about some of the more subtle cases.
 *
 *
 * Revision History: for epsilon 5.0
 * Send bug reports, bug fixes to shane@ai.mit.edu
 *
 * Version 1.0:  06/01/91 shane@ai.mit.edu Initial version.
 * Version 1.1:  06/20/91 shane@ai.mit.edu Some bug fixes.
 * Version 1.2:  06/28/91 shane@ai.mit.edu Improve performance of looking_at
 *         slightly.
 * Version 1.3:  07/08/91 shane@ai.mit.edu Removed is_word_char, use Lugaru's.
 *         Also added variable c_tab_always_indent.
 * Version 1.4:  07/12/91 shane@ai.mit.edu Made C-M-A a little smarter.
 * Version 1.5:  07/19/91 shane@ai.mit.edu Fix problem with indenting lines
 *         after disgusting c++ comments.
 * Version 1.6:  07/18/91 More fun with c++ comments.  Fix 
 *         c_backward_to_noncomment to understand them.
 * Version 1.7:  07/18/91 More fun with c++ comments.  Fix 
 *         bug in skip_c_comments to understand them. Make
 *         c_backward_to_start_of_if stop if it detects a syntax error.  Stop
 *         binding case_fold many times.  Do it once in outermost call and use
 *         unwind-protect (C's miserable excuse for it, that is).
 * Version 1.8:  08/05/91 shane@ai.mit.edu Improve performance of CR between 
 *         functions if c_indenter_shortcut is T.  Flush
 *         c_backward_to_start_of_do, I don't think it does anything.  Do case
 *         folding in C-M-Q (indent-function).  Halt infinite loop for 
 *         #thing ... \ at file begin.  Make the effect of c_brace_offset and
 *         c_case_offset more predictable for settings I don't use.  Remove
 *         unused variable c_continued_brace_offset.  Use c_brace_offset for
 *         this.
 * Version 1.9:  08/12/91 johnk@wrq.com (John Kercheval) Fix the indent-function 
 *         command to use spots to remember the region it is indenting since
 *         the region changes while indenting.  Use beginning_of_defun to
 *         determine the true function start rather than the macro
 *         BEGINNING_OF_DEFUN(). Broke out beginning_of_defun command guts and
 *         renamed command beginning-of-defun to start-of-defun to support
 *         this (the command recenters the window). Fix command do-c-indent to
 *         handle tab correctly when !c_tab_always_indent and epsilon is doing
 *         space to tab translation.  Fix c_indenter_1 to handle statement
 *         after case correctly (use current_indentation not current_column).
 *               08/12/91 shane@ai.mit.edu Fix indenter to work correctly when
 *         there are no characters after the insertion point.
 * Version 1.10: 08/13/91 shane@ai.mit.edu Restore c_backward_to_start_of_do.  
 *         It handles do - while statements after ifs, etc.
 */

#include "eel.h"

/* 
 * These are also used by Larry Osterman's comment.e, available from Lugaru
 * BBS, FTP.AI.MIT.EDU and elsewhere.  A good package for manipulating
 * comments.
 */
buffer char *comment_start;

buffer char *comment_end;

buffer short comment_column = 40;

int matchend = 0;

typedef struct pstate
{
  int incomment;
  int instring;
  int level;
  int containing_cexp;
  int quoted;
  int beginning_of_defun;
} PSTATE;

/* Enable much faster shortcut (most cases) for Shane's indentation style, 
 * if indentation looks funny for you, set this to 0.  The full indenter
 * is always correct for all settings of the variables below.
 */
int c_indenter_shortcut = 0;

/*
 * The following variables control indentation style.  They are set for
 * Gnu C style (almost, I use less indentation).  Change them till you
 * get the style you want.
 */

/* Indentation of C statements with respect to containing block. */
int c_indent_level = 2;

/* Imagined indentation of a C open brace that actually follows a statement. */
int c_brace_imaginary_offset = 0;

/* Extra indentation for braces, compared with other text in same context. */
int c_brace_offset = 0;

/* Indentation level of declarations of C function arguments.
 * Note this must be 0 for the quick indenter to work.
 */
int c_argdecl_indent = 0;

/* Offset of C label lines relative to usual indentation. */
int c_label_offset = -2;

/* Offset of C case statements relative to usual indentation. */
int c_case_offset = 0;

/* Extra indent for lines not starting new statements. */
int c_continued_statement_offset = 2;

/* Non-zero means TAB in C mode should always reindent the current line,
 * regardless of where in the line point is when the TAB command is used.
 */
int c_tab_always_indent = 1;

#define LOOKING_AT(pat) looking_at (1, pat, (char *) 0)

#define INDENT_TO(indent) to_column(indent)

#define SKIP_CHARS_BACKWARD(set) skip_chars (-1, (set))
#define SKIP_CHARS_FORWARD(set) skip_chars (1, (set))

#define FORWARD_SEXP() traverse_cexp (1)
#define BACKWARD_SEXP() traverse_cexp (-1)

#define SEARCH_FORWARD(str) search (1, (str))
#define SEARCH_BACKWARD(str) search (-1, (str))

#define BOLP() (current_column () == 0)

#define BEGINNING_OF_DEFUN() \
   (re_search (-1, "^[{A-Za-z0-9$_]+[^A-Za-z0-9$_:]") ? \
    ((to_begin_line()), 1) : \
    ((point = 0), 0))

#undef void
#define void int

#undef NULL 
#define NULL ((char *) 0)

void skip_chars(dir, set)
int dir;
char *set;
{
  int last = size ();
  int offset = dir < 0 ? -1 : 0;
  
  while (point < last && point > 0 && index (set, character (point + offset)))
    point += dir;
}


void current_indentation()
{
  int orig = point;
  int indent;
  
  to_indentation ();
  indent = current_column ();
  point = orig;
  return (indent);
}


int looking_at (dir, pat, res)
int dir;
char *pat;
char *res;
{
  int start = point;
  int ret = 0;
  char fmh[256];
  
  matchend = start;
  if (!res) res = fmh;
  if (parse_string (dir, pat, res))
  {
    matchend = point;
    ret = 1;
  }
  point = start;
  return (ret);
}

char matching_ends[] = "][]}{})()\"\"''";
char opening_ends[] = "[{(";
char closing_ends[] = "]})";
  
int matchend = 0;

char matching_end (c)
char c;
{
  char *s = index (matching_ends, c);
  
  if (s == NULL) 
    return (0);
  else 
    return (s[1]);
}

#define opening_end(c) (index (opening_ends, (c)) != NULL)

#define closing_end(c) (index (closing_ends, (c)) != NULL)

#ifndef opening_end
int opening_end (c) 
char c;
{
  return (index (opening_ends, c) != NULL);
}
#endif

#ifndef closing_end
int closing_end (c) 
char c;
{
  return (index (closing_ends, c) != NULL);
}
#endif


int is_slashified (dir)
int dir;
{
  int pos = point;
  int slash_count = 0;

  if (dir > 0) pos--;
  while ((pos-- > 0) && (character (pos) == '\\'))
    slash_count++;
  return (slash_count & 1);
}

void skip_c_comments(dir)
int dir;
{
  int offset = 1;
  int last = size ();
  
  if (dir != 1) 
  {
    dir = -1;
    offset = 0;
  }
  while (re_search (dir, "[^ \t\n\f]") && (point > 0) && (point < last))
  {
    char c = character (point - offset);
    char cc = character (point - offset + dir);
    
    if (c == '/')
    {
      if (cc == '*')
        search (dir, dir > 0 ? "*/" : "/*");
      else if (cc == '/')		/* c++ comment */
      {
        if (dir > 0)
          nl_forward ();
        else
          point--;
      }
    }
    else
    {
      if (dir > 0) point -= dir;
      break;
    }
  }
}

/*
 * This should use parse_partial_cexp but I did not port some of the
 * functionality it should have to do this.  This is probably faster and
 * I have no more time to devote to this project, so I have left it.
 */
int traverse_cexp (dir)
int dir;
{
	int level = 0;
  int orig = point;
  char c = 0;
  char start = 0;
  char end = 0;
  int offset = 1;
  char buf[2];
  char patbuf[32];
  
  if (dir != 1) 
  {
    dir = -1;
    offset = 0;
  }
  if (dir > 0 && point < size () && index (":;?,", character (point)))
    point++;
  skip_c_comments (dir);
  if (!index (matching_ends, character (point)))
  {
    if (re_search(dir, "[][)(}{\"';:,? \t\n\f]"))
    {
      point -= dir;
      return (1);
    }
    return (0);
  }
  else if ((opening_end(character (point)) && dir > 0) ||
           (closing_end(character (point)) && dir < 0) ||
           (character (point) == '"') ||
           (character (point) == '\''))
  {
    if (dir < 0) point++;
    strcpy (patbuf, "[][)(}{\"']|/%*|%*/|//");
    while (re_search (dir, patbuf))
    {
      buf[0] = c = character (point - offset);
      buf[1] = 0;
      if (start == 0) 
      {
        start = c;
        end = matching_end (start);
        if (!end) 
          start = 0;
        else
        {
          strcpy (patbuf, "%X|%X|[\"']|/%*|%*/|//");
          patbuf[1] = start;
          patbuf[4] = end;
        }
      }
      if (c == '"' || c == '\'')
      { 
        char strpat[8];
        
        strcpy (strpat, "%X|\n");
        strpat[1] = c;
        while (re_search (dir, buf) && is_slashified (dir));
        if (character (point - 1) == '\n')
          break;
      }
      else if (c == '*')
      {
        search (dir, dir > 0 ? "*/" : "/*");
      }
      else if (c == '/')		/* c++ comment */
      {
        if (dir > 0)
          nl_forward ();
      }
      if (c == start)
        level++;
      if (c == end && !--level)
        break;
    }
    return (level == 0 && start != 0);
  }
  else
  {
    point = orig;
    return (0);
  }
}

int parse_partial_cexp (from, to, state)
int from;
int to;
PSTATE *state;
{
  char c = 0;
  int stack[64];
  
  state->instring = 0;
  state->incomment = 0;
  state->containing_cexp = -1;
  state->quoted = 0;
  state->level = -1;
  point = from;
  while (re_search (1, "[][(){}\"']|/%*|//") && (point < to))
  {
    c = character (point - 1);
    if (opening_end(c))
    {
      state->level++;
      if (state->level > (sizeof (stack) / sizeof (stack[0])))
        error ("Nesting too deep: %d:", state->level);
      stack[state->level] = point - 1;
    }
    else if (closing_end(c))
    { 
      if ((state->level >= 0) && 
          (matching_end (character (stack[state->level])) == c))
        state->level--;
      else if (state->level >= 0)
      {
        point = stack[state->level];
        error ("Looking for %c", character (stack[state->level]));
      }
    }
    else if (c == '"' || c == '\'')
    {
      char strpat[8];
      
      state->quoted = -1;
      strcpy (strpat, "[X\n]");
      strpat[1] = c;
      while (re_search (1, strpat) && is_slashified (1) && (point < to))
        ;
      if (point < to) state->quoted = 0;
    }
    else if (c == '*' || c == '/')
    {
      state->incomment = 1;
      state->quoted = -2;
      if (c == '*')
        SEARCH_FORWARD("*/");
      else
        nl_forward ();
      if (point <= to) 
      {
        state->incomment = 0;
        state->quoted = 0;
      }
    }
  }
  if (state->level >= 0) 
    state->containing_cexp = stack[state->level];
}

/*
 * Indenting
 */

#if 0
/* This will indent comments nicely but is not hooked up */
int c_comment_indent()
{
  int ret = 0;

  if (LOOKING_AT("^/%*"))
    ;                                   /* Existing comment at bol stays. */
  else
  {
    int opoint = point;

    {
      int save_excursion = point;
      
      to_begin_line();
      if (LOOKING_AT("[ \t]*}[ \t]*(\n|/%*)"))
        /*
         * A comment following a solitary close-brace 
         * should have only one space. 
         */
      {
        SEARCH_FORWARD("}");
        ret = current_column () + 1;
      }
      else if (LOOKING_AT("^#[ \t]*endif[ \t]*") ||
               LOOKING_AT("^#[ \t]*else[ \t]*"))
        ret = 7;                        /* 2 spaces after #endif */
      else if ((point = opoint),
               SKIP_CHARS_BACKWARD(" \t"),
               (comment_column == 0 && BOLP()))
        /*
         * If comment_column is 0, and nothing but space
         * before the comment, align it at 0 rather than 1. 
         */
        ret = 0;
      else
      {
        ret = current_column () + 1;    /* Else indent at comment column */
        if (comment_column > ret) 
          ret = comment_column;         /* except leave at least one space. */
      }
      point = save_excursion;
    }
  }
  return (ret);
}
#endif


/* 
 * Indent current line as C code.
 */

int c_indenter_1(state)
PSTATE *state;
{
  int indent;
  int beg = 0;
  int pos;

  if (point == size ())
  {
    /* 
     * So that indenting in a brand new file works.  I did not want to 
     * change (and debug) all the other routines; this makes them work
     * correctly when there are no characters after the indent point.
     */
    insert (' ');        
    point--;             
  }
  pos = size () - point;
  indent = calculate_c_indent (state);
  to_begin_line();
  beg = point;
  if (indent == -1)                     /* NIL */
    indent = current_indentation ();
  else if (indent == -2)                /* T */
    indent = calculate_c_indent_in_comment ();
  else if (LOOKING_AT("[ \t]*#"))
    indent = 0;
  else
  {
    SKIP_CHARS_FORWARD(" \t");
    if (LOOKING_AT("case[ \t'(]|default:"))
    {
      if (state->containing_cexp >= 0)
      {
        int save_excursion = point;
        
        point = state->containing_cexp;
        indent = current_indentation () + c_case_offset + c_indent_level;
        point = save_excursion;
      }
      if (indent < 0) indent = 0;
    }
    else if (LOOKING_AT("[A-Za-z0-9$_]+:"))
    {
      indent += c_label_offset;
      if (indent < 0) indent = 0;
    }
    else if (LOOKING_AT("else[ \t\n]"))
    {
      int save_excursion = point;
      
      c_backward_to_start_of_if (state->beginning_of_defun);
      indent = current_indentation ();
      point = save_excursion;
    }
    else if (LOOKING_AT("}[ \t]*else"))
    {
      int save_excursion = point;
      
      point++;
      BACKWARD_SEXP();
      indent = current_indentation ();
      point = save_excursion;
    }
    else if (LOOKING_AT("while[ \t\n]"))
    {
      int save_excursion = point;
      
      if (!c_backward_to_start_of_do (state->beginning_of_defun))
      {
        point = save_excursion;
        goto next;                      /* Lisp can do some things C can't */
      }
      /* This is a `while' that ends a do-while. */
      indent = current_indentation ();
      point = save_excursion;
    }
    else
    next:
      if (character (point) == '}')
        indent -= (c_indent_level - c_brace_offset);
      else if (character (point) == '{')
        indent += c_brace_offset;
  }
  to_indentation ();
  if (current_column () != indent)
    INDENT_TO(indent);
  /* 
   * If initial point was within line's indentation,
   * position after the indentation.  Else stay at same point in text. 
   */
  if (size () - pos > point)
    point = size () - pos;
  return (indent);
}

int c_indenter()
{
  PSTATE state;
  int old_case_fold = case_fold;
	jmp_buf *old_level = top_level;       /* to undo case fold on abort */
  jmp_buf this_level;
  int ret;
  
  case_fold = 0;
  top_level = &this_level;
  if (ret = setjmp (top_level))         /* the verbose unwind-protect */
  {
    top_level = old_level;
    case_fold = old_case_fold;
    longjmp (top_level, ret);
  }
  state.beginning_of_defun = -1;
  ret = c_indenter_1 (&state);
  top_level = old_level;
  case_fold = old_case_fold;
  return (ret);
}

void c_backward_to_exp_start(lim)
int lim;
{
  if (index (")\"", character (point - 1)))
    BACKWARD_SEXP();
  to_begin_line();
  if (point <= lim)
    point = lim + 1;
  SKIP_CHARS_FORWARD(" \t");
}


/*
 * If point follows a `do' statement, move to beginning of it and return t.
 * Otherwise return nil and don't move point.
 */
int c_backward_to_start_of_do(limit)
int limit;
{
  int save_excursion;
  
  if (limit < 0) 
  {
#if 0
    save_excursion = point;
    BEGINNING_OF_DEFUN();
    limit = point;
    point = save_excursion;
#else
    limit = 0;
#endif    
  }
  {
    int first = 1;
    int startpos = point;
    int done = 0;
    
    while (!done)
    {
      c_backward_to_noncomment (limit);
      if (!BACKWARD_SEXP())
        done = 2;
      else if (LOOKING_AT("do[^A-Za-z0-9_$]"))
        done = 1;
      /* 
       * Otherwise, if we skipped a semicolon, we lose.
       * (Exception: we can skip one semicolon before getting
       * to the last token of the statement, unless that token
       * is a close brace).
       */
      else 
      {
        save_excursion = point;
        if (FORWARD_SEXP())
        {
          if (!first && character (point - 1) == '}')
            done = 2;
          if (!done && character (point) == ';' && character (point - 1) == '}')
            done = 2;
          if (!done && (character (point) == ';'))
          {
            if (first)
            {
              if (character (point - 1) == ')' && BACKWARD_SEXP())
              {
                if (BACKWARD_SEXP() && 
                    LOOKING_AT("while") && 
                    c_backward_to_start_of_do (limit))
                  continue;
              }
            }
            if (!first) 
              done = 2;
            first = 0;
          }
          point = save_excursion;
        }
      }
      /* If we go too far back in the buffer, we lose. */
      if (point < limit && !done)
        done = 2;
    }
    if (done != 1)
      point = startpos;
    return (done == 1);
  }
}


/* Move to the start of the last "unbalanced" `if'. */
void c_backward_to_start_of_if(limit)
int limit;
{
  if (limit < 0)
  {
#if 0
    int save_excursion = point;
    
    BEGINNING_OF_DEFUN();
    limit = point;
    point = save_excursion;
#else
    limit = 0;
#endif    
  }
  {
    int if_level = 1;
    
    while ((point > 0) && (if_level > 0))
    {
      if (!BACKWARD_SEXP())
      {
        say ("Syntax error");
        break;
      }
      if (LOOKING_AT("else[^A-Za-z0-9$_]"))
        if_level++;
      else if (LOOKING_AT("if[^A-Za-z0-9$_]"))
        if_level--;
      else if (point < limit)
      {
        if_level = 0;
        point = limit;
      }
    }
  }
}


void c_backward_to_noncomment(lim)
int lim;
{
  int opoint = 0;
  int stop = 0;
  
  if (lim < 0) lim = 0;
  while (!stop)
  {
    SKIP_CHARS_BACKWARD(" \t\n\f");
    opoint = point;
    if (point >= 2 + lim)
    {
      point -= 2;
      if (LOOKING_AT("%*/"))
      {
        if (!SEARCH_BACKWARD("/*"))
          point = lim > 0 ? lim : 0;
        continue;
      }
      point += 2;
    }
    if (point <= lim)
      stop = 1;
    else
    {
      to_begin_line();
      SKIP_CHARS_FORWARD(" \t");
      if ((character (point) != '#') && !LOOKING_AT("//"))
      {
        PSTATE ps;
        
        parse_partial_cexp (point, opoint, &ps);
        if (ps.incomment && looking_at (-1, "//.*", NULL))
          point = matchstart;
        else
          stop = 1;
      }
      point = opoint;
      if (!stop) 
        to_begin_line();
    } 
  }
}


/*
 * Return the indentation amount for line inside a block comment.
 */
int calculate_c_indent_in_comment()
{
  int indent = 0;
  int save_excursion = point;
  
  to_begin_line();
  if (point > 0)
  {
    SKIP_CHARS_BACKWARD(" \t\n");
    to_begin_line();
    SKIP_CHARS_FORWARD(" \t");
    indent = current_column ();
    if (LOOKING_AT("/%*"))
      indent++;
  }
  point = save_excursion;
  return (indent);
}


/*
 * This lookback parser was not part of the original code.  I added it
 * to speed the beast up in most cases.  It works for my indentation style;
 * it may not work for yours since it makes more assumptions than the
 * general indenter.  It is disabled by default, set the variable
 * c-indenter-shortcut to 1 if you want to try it out.  It is much faster
 * if you indent like the style in this file but it WILL misindent code
 * that does not use braces for cues in complicated code.  The general
 * indenter can handle everything.
 */
int calculate_simple_c_indent(indent_point)
int indent_point;
{
  int save_excursion = point;
  int ret = -1;
  int lim = point - 512;
  
  if (!c_indenter_shortcut) return (-1);
  if (lim < 0) lim = -1;
  point = indent_point;
  to_indentation ();
  indent_point = point;
  if (character (point) == '}')
  {
    point++;
    if (BACKWARD_SEXP())
    {
      int save_excursion_1 = point;
      
      to_indentation ();
      if (point == save_excursion_1)
        ret = current_column () - c_brace_offset + c_indent_level;
    }
    goto finish;
  }
  c_backward_to_noncomment (lim);
  if (point > 0)
  {
    if (character (point - 1) == '{')
    {
      int save_excursion_1 = point;
      
      point--;
      c_backward_to_noncomment (lim);
      if (character (point - 1) == ')')
        c_backward_to_exp_start (lim);
      else
      {
        point = save_excursion_1;
        to_indentation ();
      }
      ret = current_column () + c_indent_level;
      if (character (point) == '{')
        ret -= c_brace_offset;
    }
    else if (character (point - 1) == '}')
    {
      to_indentation ();
      ret = current_column () - c_brace_offset;
    }
    else if (character (point - 1) == ')')
    {
      c_backward_to_exp_start (lim);
      if (LOOKING_AT("(if|else|switch|for)[ \t\n(]"))
      {
        ret = current_column ();
        point = indent_point;
        if (character (point) != '{')
          ret += c_indent_level;
      }
    }
    else if (character (point - 1) == ';')
    {
      point--;
      c_backward_to_exp_start (lim);
      if (!LOOKING_AT("for[ \t\n(]"))
      {
        int save_excursion_1 = point;
        
        skip_c_comments (-1);
        if (character (point) == ';')
        {
          point = save_excursion_1;
          ret = current_indentation ();
        }
        else if (index ("{}", character (point)))
        {
          point = save_excursion_1;
          ret = current_indentation ();
        }
        else
        {
          to_indentation ();
          if (LOOKING_AT("(else|if|for|while|switch)[ \t\n(]"))
            ret = current_column ();
          else if (LOOKING_AT("case[ \t'(]|default:"))
            ret = current_column () + c_indent_level - c_case_offset;
        }
        point = indent_point;
        if (LOOKING_AT("case[ \t'(]|default:"))
          ret -= c_indent_level;
      }
    }
    else if (character (point - 1) == ':')
    {
      to_indentation ();
      if (LOOKING_AT("case[ \t'(]|default:"))
      {
        ret = current_column () + c_indent_level;
        point = indent_point;
        if (LOOKING_AT("case[ \t'(]|default:"))
          ret -= c_indent_level;
        else if (character (point) == '{')
          ret -= c_case_offset + c_indent_level;
      }
    }
  }
finish:
  if (ret >= 0)
  {
    point = indent_point;
    if (LOOKING_AT("case[ \t'(]|default:"))
      ret += c_case_offset;
  }
  point = save_excursion;
  return (ret);
}


/*
 * Return appropriate indentation for current line as C code.
 * In usual case returns an integer: the column to indent to.
 * Returns -1 if line starts inside a string, -2 if in a comment.
 */
int calculate_c_indent(state)
PSTATE *state;
{
  int save_excursion = point;
  int parse_start = 0;
  int indent_point;
  int ret = 0;
  int containing_cexp = -1;
  PSTATE ps;
  
  if (!state)
  {
    state = &ps;
    state->beginning_of_defun = -1;
  }
  state->containing_cexp = containing_cexp;
  to_begin_line();
  if (point > 0)
  {
    indent_point = point;
    ret = calculate_simple_c_indent (indent_point);
    if (ret >= 0) goto finish;
    if (state->beginning_of_defun == -1)
    {
      BEGINNING_OF_DEFUN();
      state->beginning_of_defun = point;
    }
    point = state->beginning_of_defun;
    ret = 0;
    do
    {
      parse_start = point;
      parse_partial_cexp (point, indent_point, state);
      containing_cexp = state->containing_cexp;
    } while (point < indent_point);
    if (state->instring || state->incomment)
    {
      /* return -1 or -2 if should not change this line */
      ret = state->quoted;
    }
    else if (containing_cexp < 0)
    {
      /* Line is at top level.  May be data or function definition,
       * or may be function argument declaration.
       * Indent like the previous top level line
       * unless that ends in a closeparen without semicolon,
       * in which case this line is the first argument decl.
       */
      point = indent_point;
      SKIP_CHARS_FORWARD(" \t");
      if (character (point) == '{')
        ret = 0;                        /* Unless it starts a function body */
      else if (c_argdecl_indent > 0)    /* Handle puky un*x style arglist */
      {
        c_backward_to_noncomment (parse_start > 0 ? parse_start : 0);
        /* 
         * look at previous line that's at column 0
         * to determine whether we are in top-level decls
         * or function's arg decls.
         */
        re_search (-1, "^[^ \f\t\n#]");
        if (LOOKING_AT("[A-Za-z0-9$_]+[^\"\n=]*%("))
        {
          point = matchend - 1;
          FORWARD_SEXP();
          SKIP_CHARS_FORWARD(" \t\f");
          if ((point < indent_point) && !index (",;", character (point)))
            ret = c_argdecl_indent;
        }
      }
    }
    else if (character (containing_cexp) != '{')
    {
      /*
       * line is expression, not statement:
       * indent to just after the surrounding open.
       */
      point = containing_cexp + 1;
      ret = current_column ();
    }
    else
    {
      /* 
       * Statement level.  Is it a continuation or a new statement
       * Find previous non-comment character.
       */
      point = indent_point;
      c_backward_to_noncomment (containing_cexp);
      /* 
       * Back up over label lines, since they don't
       * affect whether our line is a continuation.
       */
      while ((character (point - 1) == ',') ||
             ((character (point - 1) == ':') && 
              ((character (point - 2) == '\'') || is_word_char (point - 2))))
      {
        if (character (point) == ',')
        {
          point--;
          c_backward_to_exp_start (containing_cexp);
        }
        to_begin_line();
        c_backward_to_noncomment (containing_cexp);
      }
      /* 
       * Check for a preprocessor statement or its continuation lines.
       * Move back to end of previous non-preprocessor line.
       */
      {
        int found = point;
        int stop = 0;
        int save_excursion_1;
        
        while (!stop)
        {
          save_excursion_1 = point;
          if (point > 0) to_end_line();
          if ((character (point - 1)) != '\\')
          {
            point = save_excursion_1;
            /* 
             * This line is not preceded by a backslash.
             * So either it starts a preprocessor command
             * or any following continuation lines
             * should not be skipped.
             */
            to_indentation();
            if (character (point) == '#')
            {
              to_end_line();
              found = point;
            }
            else
              stop = 1;
          }
          else
            nl_reverse();
        }
        point = found;
        /* Now we get the answer. */
        save_excursion_1 = point;
        point = indent_point;
        SKIP_CHARS_FORWARD(" \t");
        /*
         * Don't treat a line with a close-brace
         * as a continuation.  It is probably the
         * end of an enum type declaration.
         */
        if ((character (point)) != '}')
        {
          point = save_excursion_1;
          if ((point > 0) && !index (",;{}", character (point - 1)))
          {
            /*
             * This line is continuation of preceding line's statement
             * indent c_continued_statement_offset more than the
             * previous line of the statement.
             */
            c_backward_to_exp_start (containing_cexp);
            ret = current_column ();
            point = indent_point;
            SKIP_CHARS_FORWARD(" \t");
            if (character (point) !=  '{')
              ret += c_continued_statement_offset;
          }
          else
            goto new_statement;
        }
        else
        new_statement:         
        {
          /* 
           * This line starts a new statement.
           * Position following last unclosed open.
           */
          point = containing_cexp;
          /*
           * Is line first statement after an open-brace or after a case.
           * If no, find that first statement and indent like it. 
           */
          point++;
          {
            int colon_line_end = 0;
            
            while (SKIP_CHARS_FORWARD(" \t\n"),
                   LOOKING_AT("#|/%*|case[ \t\n'(].*:|[a-zA-Z0-9_$]*:"))
            {
              /* Skip over comments and labels following openbrace. */
              if (character (point) == '#')
                nl_forward(); 
              else if (character (point) == '/')
              {
                point += 2;
                SEARCH_FORWARD("*/");
              }
              else
              {
                /* case or label: */
                save_excursion_1 = point;
                to_end_line();
                colon_line_end = point;
                point = save_excursion_1;
                SEARCH_FORWARD(":");
              }
            }
            /* 
             * The first following code counts
             * if it is before the line we want to indent.
             */
            if ((point < indent_point) || 
                (colon_line_end > 0) && (colon_line_end < indent_point))
            {
              if (colon_line_end > point)
                ret = current_indentation() - c_label_offset;
              else if ((colon_line_end > 0) &&
                       (colon_line_end > containing_cexp))
              {
                save_excursion_1 = point;
                point = colon_line_end;
                to_indentation ();
                if (LOOKING_AT("case[ \t'(]|default:"))
                {
                  ret = current_column ();
                  point = indent_point;
                  SKIP_CHARS_FORWARD(" \t");
                  if (!LOOKING_AT("[{}]"))
                    ret += c_indent_level;
                  else
                    ret -= c_case_offset;
                  point = save_excursion_1;
                }
                else
                {
                  point = save_excursion_1;
                  ret = current_column ();
                }
              }
              else
                goto no_previous;
            }
            else
            no_previous:
            {
              /*
               * If no previous statement,
               * indent it relative to line brace if on (or the last case
               * statement).  For open brace in column zero, don't let
               * statement start there too.  If c_indent_level is zero,
               * use c_brace_offset + c_continued_statement_offset instead.
               * For open-braces not the first thing in a line,
               * add in c_brace_imaginary_offset.
               */
              point = containing_cexp;
              if (BOLP() && !c_indent_level)
                ret = c_brace_offset + c_continued_statement_offset;
              else 
                ret = c_indent_level - c_brace_offset;
              /*
               * Move back over whitespace before the openbrace.
               * If openbrace is not first nonwhite thing on the line,
               * add the c_brace_imaginary_offset.
               */
              SKIP_CHARS_BACKWARD(" \t");
              if (!BOLP())
                ret += c_brace_imaginary_offset;
              /*
               * If the openbrace is preceded by a parenthesized exp,
               * move to the beginning of that;
               * possibly a different line. 
               */
              if (character (point - 1) == ')')
                BACKWARD_SEXP();
              /* Get initial indentation of the line we are on. */
              ret += current_indentation ();
            }
          }
        }
      }
    }
  }
finish:
  point = save_excursion;
  return (ret);
}


/*
 * Commands
 */

keytable c_tab;			/* key table for c mode */

char c_mode_name[] = "GnuC";


command c_mode()
{
  mode_keys = c_tab;		/* use these keys */
  c_tab[')'] = c_tab[']'] = Matchdelim ? (short) show_matching_delimiter : 0;
	indenter = c_indenter;
  comment_start = "/* ";
  comment_end = "*/";
	auto_indent = 1;
	major_mode = c_mode_name;
	try_calling("c-mode-hook");
	make_mode();
}

/* make this the default mode for .c, .cpp, .h, .e, and .y files */
suffix_c()	{ c_mode(); }
suffix_cpp()	{ c_mode(); }
suffix_h()	{ c_mode(); }
suffix_e()	{ c_mode(); }
suffix_y()	{ c_mode(); }


command do_c_indent() on c_tab['\t']
{
  int save_excursion = point;

  if (!c_tab_always_indent)
  {
    /* If not in indentation, do a tab */
    SKIP_CHARS_BACKWARD(" \t");
    if (!BOLP())
    {
      point = save_excursion;
      to_column (current_column () + tab_size - 
                 current_column () % tab_size);
      return;
    }
    point = save_excursion;        
  }
  c_indenter();
}

command do_c_newline() on c_tab['\n']
{
  int *spot = alloc_spot();
  
  to_indentation ();
  if (LOOKING_AT("#|(case|else)[ \t]|[A-Za-z0-9$_]+:|while[ \t]+%(.*%);"))
    c_indenter ();
  point = *spot;
  free_spot (spot);
  normal_character ();                  /* Also indents */
}


command c_open() on c_tab['{']
{
  normal_character ();
  c_indenter ();
}


command c_close() on c_tab['}']
{
  normal_character ();
  c_indenter ();
  if (Matchdelim)
    find_delimiter ();
}


command forward_cexp() on c_tab[ALT(CTRL('F'))]
{
  int start = point;
  int last = size ();
  
  if (!traverse_cexp (iter < 0 ? -1 : 1))
  {
    point = start;
    quick_abort ();
  }
  while (index (",:;\\", character (point)) && (point < last)) point++;
}


command backward_cexp() on c_tab[ALT(CTRL('B'))]
{
  int start = point;
  
  while ((point > 0) && index (",:;\\ \t\n\f", character (point - 1))) point--;
  if (!traverse_cexp (iter < 0 ? 1 : -1))
  {
    point = start;
    quick_abort ();
  }
}


command kill_cexp() on c_tab[ALT(CTRL('K'))]
{
  int start = point;
  int end;
  
  if (!traverse_cexp (iter < 0 ? -1 : 1))
  {
    point = start;
    quick_abort ();
  }
  end = point;
  if (end < start)
  {
    point = end;
    end = start;
    start = point;
  }
  else
  {
    point = start;
  }
  iter = 0;
  do_save_kill (start, end);
}


void beginning_of_defun ()
{
  int last = point;
  
  while (BEGINNING_OF_DEFUN())
  {
    int next = point;
    int save_excursion;
    
    if (character (point) == '{')
    {
      while (point > 0)
      {
        nl_reverse();
        save_excursion = point;
        to_begin_line();
        next = point;
        SKIP_CHARS_FORWARD(" \t");
        if (point == save_excursion)
        {
          nl_forward();
          break;
        }
        point = next;
        if (LOOKING_AT("#|(*.%*/)|(//)"))
        {
          nl_forward();
          break;
        }
      }
      break;
    }
    skip_c_comments (-1);
    save_excursion = point;
    to_begin_line();
    if (character (point) == '#')
    {
      point = next;
      break;
    }
    to_end_line();
    if (character (point - 1) == '\\')
    {
      point = next;
      break;
    }    
    last = next;
    point = save_excursion;
    if ((character (point) == '}') || 
        ((character (point) == ';') && 
         (point > 0) &&
         ((character (point - 1) == '}') ||
          (character (point - 1) == ')') ||
          (BACKWARD_SEXP(), skip_c_comments (-1), character (point) == '='))))
    {
      point = last;
      break;
    }
    if (point == 0) break;
    to_end_line();
    point++;
  }
}


command start_of_defun() on c_tab[ALT(CTRL('A'))]
{
  iter = 0;
  beginning_of_defun ();
  window_start = prev_screen_line(0);
}


command end_of_defun() on c_tab[ALT(CTRL('E'))]
{
  int orig = point;
  
  iter = 0;
  FORWARD_SEXP();
  point++;
  if (BEGINNING_OF_DEFUN())
  {
    while (FORWARD_SEXP() && (character (point - 1) != '}'));
    if (character (point - 1) != '}')
      point = orig;
    else if (character (point) == ';')
      point++;
  }
}


command up_level() on c_tab[ALT(CTRL('U'))]
{
  PSTATE state;
  int orig = point;
  
  iter = 0;
  BEGINNING_OF_DEFUN();
  while (point <= orig)
  {
    parse_partial_cexp (point, orig + 1, &state);
  }
  if (state.containing_cexp >= 0)
    point = state.containing_cexp;
  else
  {
    point = orig;
    error ("No containing cexp");
  }
}


command indent_function() on c_tab[ALT(CTRL('Q'))]
{
  PSTATE ps;
  int *save_excursion = alloc_spot ();
  int *stop = 0;
  int old_case_fold = case_fold;
  jmp_buf *old_level = top_level;     /* to undo case fold on abort */
  jmp_buf this_level;
  int ret;

  iter = 0;
  top_level = &this_level;
  case_fold = 0;
  if (ret = setjmp (top_level)) 
  {
    free_spot (save_excursion);
    if (stop) free_spot (stop);
    top_level = old_level;
    case_fold = old_case_fold;
    longjmp (top_level, ret);
  }
  beginning_of_defun ();
  ps.beginning_of_defun = point;
  end_of_defun ();
  stop = alloc_spot ();
  point = ps.beginning_of_defun;
  while (point < *stop)
  {
    c_indenter_1 (&ps);
    nl_forward ();
    check_abort ();
  }
  point = *save_excursion;
  free_spot (save_excursion);
  free_spot (stop);
  case_fold = old_case_fold;
  top_level = old_level;
  this_cmd = 0;
}

