/* Mouse extensions for Epsilon V6.0 */

/* The following copyright and trademark notice applies to some of the code
 * herein; all other material is Copyright (c) 1987 by Robert Lenoil, with
 * free copying allowed for any purpose, provided that this copyright notice
 * is included.
 *
 * last major change:     Jul '92 by /cmb/
 *
 * Clemens Beckstein, IMMD-8, Uni Erlangen, W. Germany
 * email (INTERNET):  clemens@immd8.informatik.uni-erlangen.de 
 */

/* This file adds the following mouse functions to Epsilon:
 * click left                 sets point at mouse cursor
 * double click left          sets point at mark and mark at mouse cursor
 * drag left                  sets mark at start and point at finish
 * click right                deletes from mark to mouse cursor
 * double click right         yanks at mouse cursor
 * click left in echo area    set point relative to position in echo area
 *                            (column 0 = top of buffer)
 * click right in echo area   set mouse cursor column relative to point's
 *                            position in buffer
 * bump top/bottom edge       scroll down/up
 */

#include "eel.h"
#include "lowlevel.h"

/* This file modifies the following Epsilon commands/procedures:
COMMAND              SOURCE FROM VERSION
when_idle		6.0
next_video		6.0
set_video		6.0

   And adds the following commands/procedures:
COMMAND           WRITTEN FOR VERSION
flush_mouse_buttons     5.0
hide_mouse_cursor       3.1
get_button_press_info	6.0
get_button_rel_info	6.0
mouse                   5.01
mouse_button_pressed    5.01
init_mouse              6.0
mouse_to_cursor         3.1
process_mouse_command   5.0
read_mouse_motion	6.0
read_mcursor_pos	6.0
refresh_for_mouse       3.1
screen_set_point        6.0
set_mouse_cursor        3.1
show_mouse_cursor       3.1
start_up                5.01

   And defines the following globals:
GLOBAL            WRITTEN FOR VERSION
doubleclick_interval    4.0
mouse_present           3.1
mouse_active            5.01

Modified (behaviour of double-click right) and debugged
the function  process_mouse_command() ...   /cmb/ Nov '87

Switched from pause() to delay() primitive (Epsilon version
4.0) in mouse_button_pressed() and process_mouse_command(),
and added buffer specific auto_quote_allowed switch
to getkey() ...                             /cmb/ Mar '89

Changed getkey() and completely rewrote screen_set_point()
to conform to the new Epsilon version 5.0 features; added
corresponding definitions of COMMAND_MAX and command_keys.
Also drastically simplified init_mouse() (the routines to
change video modes are more clever now) ...
                                            /cmb/ Aug '90
                  
Did a further clean up ...                  /cmb/ Oct '90

Added horizontal scrolling and fixed several bugs
in init_mouse() and screen_set_point() ...  
                                            /cmb/ Mar '91
						    
Made start_up() initialize mouse_present to 0 and mouse()
not to produce an error if no mouse present ...
					    /cmb/ May '91

Added support for 80x43, 132x25 and 132x44 video modes
on SIGMA Legend.  Mouse support is disabled for non 80 column
modes (hence the need to distinguish mouse_present and
mouse_active) ...                           /cmb/ Sep '91

Moved mouse support out of getkey() using the new Epsilon
version 6.0 when_idle() hook and simplified set_screen_point()
using the new window_at_cursor() primitive...
					    /cmb/ Jun '92

Did a major cleanup in light of soon to come OS/2 support
and simplified init_mouse()...
					    /cmb/ Jul '92
*/

int mouse_present = 0;            /* non-zero if system has a mouse */
int mouse_active = 0;             /* non-zero if mouse support is activated */
short button_status, button_presses, button_releases,
   hor_mpos, vert_mpos, hor_motion, vert_motion;
int doubleclick_interval = 40;    /* in centiseconds */

#define MOUSE_SERVICES 0x33       /* Mouse interrupt */
#define LEFT_BUTTON 0
#define RIGHT_BUTTON 1
#define BMASK(button) (1<<button) /* button mask for button status word */
#define CHAR_SIZE 8               /* pixels per character */
#define SCROLL_VERT -1            /* prev_cmd code for mouse-induced scroll */
#define SCROLL_HORZ -2            /* prev_cmd code prior to mouse scroll */


/* START_UP calls init_mouse. */
start_up()
{  
	mouse_present=0;             /* initially assume no mouse present */
	init_mouse();
}

when_idle()
{		   /* At top-level when not in the echo area
                      and if no popup is displayed, allow mouse input */
	if (mouse_active && !number_of_popups() 
		         && !in_echo_area) {
		mouse_to_cursor();
		flush_mouse_buttons();
		show_mouse_cursor();
		while (!char_avail())
			if (mouse_button_pressed()) {
				prev_cmd = 0;
				process_mouse_command();
				     /* in case they hit other button too */
				flush_mouse_buttons();
			}
		hide_mouse_cursor();
	}
}

/* MOUSE toggles whether the mouse is active or not */
command mouse()
{  if (iter == 0 || (!has_arg && mouse_active))
   {  mouse_active = 0;
      say("Mouse off.");
   }
   else
   {  init_mouse();
      if (mouse_active)
         say("Mouse support activated.");
      else
         say("Mouse missing or unsupported video mode.");
   }
   iter = 1;
}

/* init_mouse performs mouse initialization 
 * (dependent on the current video mode) 
 */
init_mouse()
{
   if (!mouse_present) {
      m_regs.w.ax = 0;			      /* Mouse correctly installed? */
      do_interrupt(MOUSE_SERVICES, &m_regs);
   }
   mouse_present = mouse_active = m_regs.w.ax;
   if (screen_cols != 80)                      /* non 80 column video mode! */
      mouse_active = 0;
   if (mouse_active) {
      m_regs.w.ax = 8;		       /* Set min and max vertical position */
      m_regs.w.cx = 0;		       /* for mouse */
      m_regs.w.dx = screen_lines * CHAR_SIZE - 1;
      do_interrupt(MOUSE_SERVICES, &m_regs);
   }
}

/* READ_MOUSE_MOTION reads and then clears the mouse motion counters */
read_mouse_motion()
{
   m_regs.w.ax = 11;
   do_interrupt(MOUSE_SERVICES, &m_regs);
   hor_motion = m_regs.w.cx;
   vert_motion = m_regs.w.dx;
}

/* SHOW_MOUSE_CURSOR turns on the mouse cursor and clears the mickey count. */
show_mouse_cursor()
{  m_regs.w.ax = 1;
   do_interrupt(MOUSE_SERVICES, &m_regs);
   read_mouse_motion();
}

/* HIDE_MOUSE_CURSOR hides the mouse cursor */
hide_mouse_cursor()
{  m_regs.w.ax = 2;
   do_interrupt(MOUSE_SERVICES, &m_regs);
}

/* REFRESH_FOR_MOUSE hides the mouse cursor, refreshes the screen, then
 * reenables the mouse cursor.
 */
refresh_for_mouse()
{  hide_mouse_cursor();
   refresh();
   show_mouse_cursor();
}

/* FLUSH_MOUSE_BUTTONS resets the stored number of button presses */
flush_mouse_buttons()
{  get_button_press_info(LEFT_BUTTON);
   get_button_press_info(RIGHT_BUTTON);
}

/* GET_BUTTON_PRESS_INFO gets the button press information
 * and clears it afterwards
 */
get_button_press_info(button)
{
   m_regs.w.ax = 5;
   m_regs.w.bx = button;
   do_interrupt(MOUSE_SERVICES, &m_regs);
   button_status = m_regs.w.ax;
   button_presses = m_regs.w.bx;
   hor_mpos = m_regs.w.cx;
   vert_mpos = m_regs.w.dx;
}

/* GET_BUTTON_REL_INFO gets the button release information
 * and clears it afterwards
 */
get_button_rel_info(button)
{
   m_regs.w.ax = 6;
   m_regs.w.bx = button;
   do_interrupt(MOUSE_SERVICES, &m_regs);
   button_status = m_regs.w.ax;
   button_releases = m_regs.w.bx;
   hor_mpos = m_regs.w.cx;
   vert_mpos = m_regs.w.dx;
}

/* READ_MCURSOR_POS gets position and button status of mouse */
read_mcursor_pos()
{
   m_regs.w.ax = 3;
   do_interrupt(MOUSE_SERVICES, &m_regs);
   button_status = m_regs.w.bx;
   hor_mpos = m_regs.w.cx;
   vert_mpos = m_regs.w.dx;
}

/* MOUSE_BUTTON_PRESSED returns the status of the mouse buttons (button is
 * down if status & BMASK(button) non-zero).  It also handles vertical screen
 * (or horizontal) scrolling if the mouse is bumping against the top or bottom
 * (resp. top or bottom) of the screen.
 * 
 * redone for Epsilon V5.0 ...                                /cmb/ Mar '91  
 */
mouse_button_pressed()			
{
   short status, row, col;

   for(;;) {
      read_mcursor_pos();

      status = button_status & 
         (BMASK(LEFT_BUTTON) + BMASK(RIGHT_BUTTON));
      col = hor_mpos / CHAR_SIZE;
      row = vert_mpos / CHAR_SIZE;
      
      read_mouse_motion();

      /* Check if not much horizontal motion */
      if (!(hor_motion / 8) && 
          (((row == 0) && (vert_motion < 0)) || 
          (row == (screen_lines - 1) && (vert_motion > 0)))) {
         if (prev_cmd == SCROLL_VERT) {
            window_scroll(vert_motion / 4);
            refresh_for_mouse();
         } else {
            delay(30,COND_KEY);
            prev_cmd = SCROLL_VERT; 
            read_mouse_motion();
         }
      } else {
         if ((prev_cmd == SCROLL_VERT) && 
             (row > 0) && (row < (screen_lines - 1)))
            prev_cmd = 0;     
         if (!(vert_motion / 8) && 
             (((col == 0) && (hor_motion < 0)) || 
              (col == (screen_cols - 1) && (hor_motion > 0)))
             && (row < screen_lines - 2)) {
            if (prev_cmd == SCROLL_HORZ) {
               if ((col == 0) && 
                   (display_column > (0 + -hor_motion / 4) - 1)) {
                  int old_iter = iter, old_arg = has_arg;
                  int old_prev = prev_cmd, old_this = this_cmd;
                  
                  iter = -hor_motion / 4;
                  has_arg = 0;
                  prev_cmd = 0;
                  scroll_right();
                  refresh_for_mouse();
                  iter = old_iter, has_arg = old_arg;
                  prev_cmd = old_prev, this_cmd = old_this;
                  break;
               } 
               if (col == screen_cols -1) {
                  int old_iter = iter, old_arg = has_arg;
                  int old_prev = prev_cmd, old_this = this_cmd;
                  
                  iter = hor_motion / 4;
                  has_arg = 0;
                  prev_cmd = 0;
                  scroll_left();
                  refresh_for_mouse();
                  iter = old_iter, has_arg = old_arg;
                  prev_cmd = old_prev, this_cmd = old_this;
                  break;
               }
            } else {
               delay(30,COND_KEY);
               prev_cmd = SCROLL_HORZ;
               read_mouse_motion();
            }
         } else {
            break;
         }
      }
   } 
   return status;
}


process_mouse_command()
{  short status;

   delay(doubleclick_interval,COND_KEY);  /* give user time to double click */
   get_button_press_info(LEFT_BUTTON);

   status = button_status;
   switch (button_presses)
   {  case 1:
         /* Check if user clicked in echo area */
         if ((vert_mpos / CHAR_SIZE) == screen_lines - 1)
         {  /* Position window over section of buffer relative to where
             * the user clicked in the echo area, with the left-hand
             * side corresponding to the top of the buffer.
             */
            point = size() * hor_mpos / (screen_cols - 1) / CHAR_SIZE;
            build_first = 1;
            break;
         }

         /* Single-click left - set point at mouse cursor */
         screen_set_point(vert_mpos, hor_mpos);
         refresh_for_mouse();
         if (status & BMASK(LEFT_BUTTON))
         {  /* The button is still being held down.  Set mark at point, then
             * set point at new mouse cursor when button is released.
             */
            set_mark();
            while (mouse_button_pressed() & BMASK(LEFT_BUTTON));
	    get_button_rel_info(LEFT_BUTTON);
            screen_set_point(vert_mpos, hor_mpos);
         }
         break;

      case 2:
         /* Double-click left - set mark at mouse cursor, then set point and
          * mouse cursor to previous mark.
          */
         screen_set_point(vert_mpos, hor_mpos);
         exchange_point_and_mark();
         refresh_for_mouse();
         mouse_to_cursor();
         return;

      case 0:
         /* No left clicks, try the right button */
         get_button_press_info(RIGHT_BUTTON);

         switch (button_presses)
         {  /* int *point_save, pos; */            /*  /cmb/ Nov '87  */

            case 0:
               return;
            case 1:
               /* Check if user clicked in echo area */
               if ((vert_mpos / CHAR_SIZE) == screen_lines - 1)
               {  /* Yes, set column to indicate percentage of file */
                  set_mouse_cursor(vert_mpos,
			  size() > 0 ? point * CHAR_SIZE *
					  (screen_cols - 1) / size()
				     : 0);
                  break;
               }  /* else fall through to case 2 */

               /* Click right - delete from mark to mouse cursor */
            case 2:
               /* Double click right - yank at mouse cursor
        (and move point to mouse cursor...  /cmb/ Nov '87) */
               /* point_save = alloc_spot(); */     /* /cmb/ Nov '87  */
               if (screen_set_point(vert_mpos, hor_mpos))
               {  if (button_presses == 1)
                     kill_region();
                  else
                     yank();
                  refresh_for_mouse();
                  mouse_to_cursor();
                  /* point = *point_save; */       /*  /cmb/ Nov '87  */
               }
               else
                  ding();
               /* free_spot(point_save); */        /*  /cmb/ Nov '87  */
               break;

            default:
               goto too_many_clicks;
         }
         break;

      default:
      too_many_clicks:
         ding();
         return;
   }
   hide_mouse_cursor();
   maybe_refresh();
   show_mouse_cursor();
}

/* SCREEN_SET_POINT takes mouse row and column coordinates and sets point
 * there.  If the coordinates lie in another window, it is made current.
 * Non-zero is returned if the specified row is an editable line, otherwise
 * (modeline or echo area) zero is returned.
 *
 * revised and simplified for version 6.0 of Epsilon...   /cmb/ Jun '92
 */
screen_set_point(row, col)
{  int vert_bottom, hor_right, htl, vtl;
   short old_window = window_number;

             /* change coordinates from pixels to characters */
   row /= CHAR_SIZE; col /= CHAR_SIZE;
   if (row == screen_lines -1) return 0;     /* hit echo area... */

             /* Find which window the coordinates lie in */
   window_handle = window_at_coords(row,col);
   vtl = window_edge(VERTICAL,TOPLEFT);
   vert_bottom = vtl + window_height - 1;
   if (row == vert_bottom)
	goto hit_border; 		     /* hit horizontal mode line */
   htl = window_edge(HORIZONTAL,TOPLEFT);
   hor_right = htl + window_width;
   if (hor_right < screen_cols && col == hor_right - 1)
	goto hit_border;	             /* hit vertical mode line */
   point = window_start;	 /* got the right window; position point */
   point = next_screen_line(row - vtl);
   if (display_column < 0)
	move_to_column(current_column() + col - htl);
   else
        move_to_column(display_column + col - htl);
   return 1;
hit_border:
   window_number = old_window;   /* row = border area, don't do anything */
   ding();			 /* inform user about it */
   return 0;
}

/* MOUSE_TO_CURSOR warps the mouse cursor to the hardware cursor */
mouse_to_cursor()
{
   m_regs.b.ah = 3;  /* read cursor position */
   m_regs.b.bh = 0;
   do_interrupt(VIDEO_IO, &m_regs);
   set_mouse_cursor(m_regs.b.dh * CHAR_SIZE, m_regs.b.dl * CHAR_SIZE);
}

/* SET_MOUSE_CURSOR sets the mouse cursor to the given row and column */
set_mouse_cursor(row, col)
{  m_regs.w.ax = 4;
   m_regs.w.cx = col;
   m_regs.w.dx = row;
   do_interrupt(MOUSE_SERVICES, &m_regs);
}

/* make sure mouse is (re-) initialized when video modes are changed */
new_mouse_next_video()
{
   mouse_next_video();
   if (mouse_present)
      init_mouse();
}

new_mouse_set_video()
{
   mouse_set_video();
   if (mouse_present)
      init_mouse();
}

REPLACE_FUNC("mouse","next_video");
REPLACE_FUNC("mouse","set_video");
