                                                
/*
**  PC-Dragon Main Module / 6809 Engine
**
**  Copyright 1993-97 by Paul D. Burgin. All rights reserved.
*/

#include "alloc.h"
#include "stdio.h"
#include "conio.h"
#include "graphics.h"
#include "dos.h"
#include "bios.h"
#include "string.h"
#include "stdlib.h"
#include "ctype.h"
#include "dir.h"
#include "direct.h"
#include "time.h"
#include "setjmp.h"
#include "io.h"

#include "build.h"
#include "types.h"
#include "extern.h"
#include "6809cpu.h"
#include "6809regs.h"
#include "key.h"
#include "macros.h"
#include "doio.h"
#include "snapshot.h"

	/* Flag bits of condition code register. */
	unsigned char		e_flag, f_flag, h_flag, i_flag;
	unsigned char		n_flag, z_flag, v_flag, c_flag;

	/* 8 bit registers (excluding cc reg). */
	reg8				double_accumulator[2], dp_reg;

	/* 16 bit registers. */
	reg16				x_reg, y_reg, u_reg, s_reg, pcr;

	/* 16 bit unused register. */
	reg16				null_reg;

	/* Table of register addresses. */
	reg16 *reg_table[] =
	{
		d_reg_addr, &x_reg, &y_reg, &u_reg, &s_reg, &pcr,
		&null_reg, &null_reg,
		(reg16 *)&a_reg, (reg16 *)&b_reg,
		&null_reg,
		(reg16 *)&dp_reg,
		&null_reg, &null_reg, &null_reg, &null_reg
	};

	/* IRQ interrupt variables. */
	unsigned int		irq_rate = D_IRQ_RATE;	/* Cycles per IRQ.         */
	unsigned int		irq_cycles;				/* Cycles since last IRQ.  */
	unsigned int		ins_cycles;				/* Cycles for current ins. */

	/* Counters dependent on IRQ. */
	unsigned int		refresh_count;		/* Video mode change.       */
	unsigned int		button_counter;		/* Joystick button update.  */
	unsigned int		kbcnt;				/* F-key check in PC mode.  */
	unsigned int		key_scan_reset;		/* Pseudo-hardware keyboard */
											/* buffer junking timeout.  */

	/* Keyboard emulation type. */
	boolean				new_int9_set = FALSE; /* Current INT9 handler. */

	/* O/S breakpoint keyboard input. */
	boolean				drag_usedkey;		/* Flag reduces thrashing. */
	unsigned char		keyin;				/* ASCII of key received.  */
	unsigned char		old_case = 0xff;	/* Last write to $149.     */

	/* Hardware model. */
	hw_type				arch;					/* Architecture emulated.  */
	hw_type				req_arch = dontcare;	/* Architecture requested. */
	boolean				mapmode1;				/* Current map mode flag.  */
	boolean				wipe_mem = TRUE;		/* Startup mem wipe flag.  */

	/* Debug data. */
	boolean				in_debug = FALSE;
	boolean				ignore_illegal_opcodes = FALSE;

	/* Address space. */
	unsigned char far 	*memory = NULL;		/* Main 64K memory.      */
	unsigned char far 	*rom2	= NULL;		/* Dragon 64 second ROM. */
	unsigned char far 	*hiram	= NULL;		/* Upper 32K ram.        */

	/* Variables for decoding. */
	unsigned char		indirect;			/* Indirection flag.            */
	unsigned char		opcode;				/* Current opcode.              */
	boolean				ext1, ext2;			/* Page 1 and page 2 flags.     */
	instr_type			i_type;				/* Type (group) of instruction. */
	instr_mode			a_mode;				/* Current addressing mode.     */

	/* Command line snapshot/cartridge file names/variables. */
	unsigned char		cartfile[2][MAXPATH+4];
	unsigned char		cartnamed = 0;
	boolean				snapboot = FALSE;

	/* Jump buffer used for hard resets. */
	jmp_buf				hard_reset;

	/* Bootup vector. */
	unsigned int		bootup;

	/* System components. */
	char				bootparallel, numparallel;
	char				bootserial, numserial;
	boolean				hasjoystick;

	/* Effective address. */
	unsigned int		addr;

	/* Register pointers, instruction buffer, instruction size flags, etc. */
	reg8				*reg8_,  *r8temp_;
	reg16				*reg16_, *r16temp_;
	unsigned char		i_buffer[2];
	__register			__r;
	instr_bits			__b, __b2;
	boolean				do_jump;

	/* Instruction operand values. */
	unsigned char		v8;
	unsigned int		v16;

	/* Temp values used in arithmetic. */
	unsigned char		t8, s8;
	unsigned int		t16;

	/* Temp values used by the macros. */
	unsigned char		mac8;
	unsigned int		mac16;

	/* Temporaries used by breakpoint patching loop. */
	const unsigned int	*ptr;
	unsigned char		doio_service;

	/* Speed measurement variables. */
	clock_t         	last_time, this_time;
	unsigned int		two_second_count;

	/* Drive and directory for PC-Dragon and cwd on entry. */
	unsigned char old_drive[MAXDRIVE], old_directory[MAXDIR] = "@";
	unsigned char pcd_drive[MAXDRIVE], pcd_directory[MAXDIR] = "@";

	/* Path for editor, assembler and 6809 source code. */
	unsigned char path_editor[MAXPATH] = "edit";
	unsigned char path_assembler[MAXPATH] = "as9";
	unsigned char name_editfile[MAXPATH+2] = "source.a09";

	/* Variables for options/configuration file. */
	const unsigned char config_file[] = "pcdragon.cfg";
	FILE *options_file, *name_stream;
	extern unsigned char os[];

	/* Text strings used for /O errors. */
	unsigned char bad_pmode[]	= "Bad Pmode In /O Option";
	unsigned char bad_dollar[]	= "Missing $ In /O Option";
	unsigned char bad_len[]		= "Bad String Length In /O Option";

	/* Flag for part completed snapshot. */
	boolean snapshot_in_progress = FALSE;

	/* Flag for whether bootup snapshot should be loaded. */
	boolean load_bootsnap = TRUE;

	/* Bootup snapshot names. */
	unsigned char bootsnapdefault[] = "bootsnap";
	unsigned char *bootsnap;

	/* DOIO breakpoint tables (ordering important). */
	const unsigned int tandy_breakpoints[] =
		{ 0xa1cb, 0xa9de, 0xa755, 0xa749, 0xa7e9, 0xa77c, 0xa82a };
	const unsigned int d32_breakpoints[] =
		{ 0xbbe5, 0xbd52, 0xbda5, 0xbdad, 0xbddc, 0xbde7, 0xbe12 };
	const unsigned int d64_breakpoints[] =
		{ 0x3aeb, 0x3c57, 0x3caa, 0x3cb2, 0x3ce1, 0x3cec, 0x3d17 };
	boolean install_breakpoints = TRUE;

	/* Function prototypes for instructions. */
	void near i_abx(void);		void near i_adc(void);
	void near i_add8(void);		void near i_add16(void);
	void near i_and(void);		void near i_asr(void);
	void near i_bit(void);		void near i_clr(void);
	void near i_cmp8(void);		void near i_cmp16(void);
	void near i_com(void);		void near i_cwai(void);
	void near i_daa(void);		void near i_dec(void);
	void near i_eor(void);		void near i_exg(void);
	void near i_inc(void);		void near i_jmp(void);
	void near i_ld8(void);		void near i_ld16(void);
	void near i_lea(void);		void near i_lsl(void);
	void near i_lsr(void);		void near i_mul(void);
	void near i_neg(void);		void near i_nop(void);
	void near i_or(void);		void near i_psh(void);
	void near i_pul(void);		void near i_rol(void);
	void near i_ror(void);		void near i_rti(void);
	void near i_rts(void);		void near i_sbc(void);
	void near i_sex(void);		void near i_st8(void);
	void near i_st16(void);		void near i_sub8(void);
	void near i_sub16(void);	void near i_swi(void);
	void near i_sync(void);		void near i_tfr(void);
	void near i_tst(void);		void near i_reset(void);
	void near i_orcc(void);		void near i_andcc(void);
	void near i_doio(void);		void near illegal(void);

	/* Instruction jump table (ordering important). */
	const void near (*execute_instruction[])(void) =
	{
		i_abx,	i_adc,	i_add8,	i_add16,i_and,	i_asr,	i_bit,	i_clr,
		i_cmp8,	i_cmp16,i_com,	i_cwai, i_daa,	i_dec,	i_eor,	i_exg,
		i_inc,	i_jmp,	i_ld8,	i_ld16,	i_lea,	i_lsl,	i_lsr,	i_mul,
		i_neg,	i_nop,	i_or,	i_psh,	i_pul,	i_rol,	i_ror,	i_rti,
		i_rts,	i_sbc,	i_sex,	i_st8,	i_st16,	i_sub8,	i_sub16,i_swi,
		i_sync,	i_tfr,	i_tst,	i_reset,i_orcc,	i_andcc,i_doio,	illegal
	};

/* Change current drive and directory. */
void set_directory(unsigned char which_dir)
{
	unsigned char *to_drive = old_drive, *to_directory = old_directory;
	unsigned char old_path[MAXPATH];

	if (which_dir > 0)
	{
		/* Remember current working directory. */
		getcwd(old_path, MAXPATH);
		if (old_path[strlen(old_path) - 1] != '\\')
			strcat(old_path, "\\");
		strcat(old_path, "_");
		_splitpath(old_path, old_drive, old_directory, NULL, NULL);
		if (strlen(old_directory) > 1)
			old_directory[strlen(old_directory)-1] = '\0';

		/* New directory is PC-Dragon directory. */
		to_drive		= pcd_drive;
		to_directory	= pcd_directory;
	}

	if (to_directory[0] != '@')
	{
		_chdrive(to_drive[0] - '@');
		chdir(to_directory);
	}
}

/* General exit function. */
void quit(unsigned char *text, unsigned char exit_type)
{
	/* exit_type is 0 - error message on exit, at command line      */
	/*              1 - error message on exit, from within emulator */
	/*              2 - no error to print,     at command line      */
	/*              3 - normal termination with message             */
	/*              4 - normal termination with no message          */

	switch (exit_type)
	{
		case 4: text = NULL;
		case 1:
		case 3:	/* Close down graphics system if necessary. */
				if (graphinit)
					closegraph();

				/* Restore original keyboard interrupt handler. */
				if (new_int9_set)
					Set_Old_Int9();

				/* Restore 80 column text mode and cursor. */
				textmode(C80);
				textattr(LIGHTGRAY);
				clrscr();
				_setcursortype(_NORMALCURSOR);
				if (exit_type != 1)
				{
					put_title(FALSE);
					if (text != NULL)
					{
						if (strlen(text) > 79)
							text[79] = '\0';
						window(40 - (strlen(text) >> 1),23,80,23);
						textattr(MAGENTA);
						cprintf(text);
					}
					window(1,24,80,24);
					break;
				}
		case 0:	/* Exit message. */
				cprintf("\r\nPC-DRAGON: %s\r\n", text);
	}

	/* Delete swap and temporary files. */
	fcloseall();
	if (access("pcd.swp",2) == 0)
		remove("pcd.swp");
	if (access("pcd.tmp",2) == 0)
		remove("pcd.tmp");

	/* Return to previous drive & directory. */
	set_directory(0);

	/* Return to dos. */
	exit(exit_type);
}

/* OS breakpoint keyboard input. */
void near check_sw_keyboard(void)
{
	if (drag_usedkey && iskey())
	{
		if ((keyin = getch2(FALSE)) != 0)
			drag_usedkey	= FALSE;
	}

	if (!drag_usedkey)
	{
		drag_usedkey = TRUE;

		/* Swap case of letter keys. */
		if (islower(keyin) && old_case)
			a_reg.u_value = keyin - 0x20;
		else if (isupper(keyin))
			a_reg.u_value = keyin + 0x20;
		else
			a_reg.u_value = keyin;
		old_case = TRUE;
	}
	else
		a_reg.u_value = 0;

	z_flag = (a_reg.u_value == 0);
	n_flag = (a_reg.u_value >> 7);
	v_flag = 0;
}

/* Read a byte from memory. */
unsigned char near get_mem8(register unsigned int address)
{
	if (address < 0xff00)
	{
		if ((arch == dragon32) || (address < 0x8000))
			return(memory[address]);
		else if (mapmode1)
			return(hiram[address-0x8000]);
		else if ((address >= 0xc000) || (memory[0xff22] & 0x04)
				|| (arch == tandy))
			return(memory[address]);
		else
			return(rom2[address-0x8000]);
	}

	/* Map 2 bytes of ROM for immediate data. */
	else if (address == I_DATA)
		return(i_buffer[0]);
	else if (address == (I_DATA+1))
		return(i_buffer[1]);

	/* Intercept hardware reads. */
	return(get_sampias(address));
}

/* Write a byte to memory. */
void near set_mem8(register unsigned int address, register unsigned char value)
{
	/* Intercept screen updates. */
	if ((address >= screen_base) && (address <= screen_end)
			&& (memory[address] != value))
		video_out(value, address);

	if (address <= 0x01d9)
	{
		/* Intercept cassette filename writes. */
		if (address >= 0x01d2)
			filemem[address - 0x01d2] = value;

		/* Intercept cursor position changes. */
		else if ((address == 0x88) || (address == 0x89))
		{
			memory[address] = value;	/* This makes the assignment    */
										/* below redundant - but it's   */
										/* faster to do the check here! */
			update_cursor();
		}

		/* Intercept keyboard case changes. */
		else if (address == 0x0149)
			old_case = value;
	}

	/* Intercept hardware writes. */
	if (address >= 0xff00)
		set_sampias(value, address);

	/* Write new value to effective address if RAM area. */
	else if (address < ROM_START)
		memory[address] = value;
	else if ((arch != dragon32) && (address < 0xff00) && mapmode1)
		hiram[address-0x8000] = value;
}

/* get_mem8 variant for routines in far modules. */
unsigned char far_get_mem8(unsigned int address)
{
	return (get_mem8(address));
}

/* get_mem16 variant for routines in far modules. */
unsigned int far_get_mem16(unsigned int address)
{
	return (get_mem16(address));
}

/* set_mem8 variant for routines in far modules. */
void far_set_mem8(unsigned int address, unsigned char value)
{
	set_mem8(address, value);
}

/* Set individual flags from cc register input. */
/* (Used by other modules -- cannot be near).   */
void far_set_cc_reg(register unsigned char cc_input)
{
	e_flag =  (cc_input >> 7);
	f_flag = ((cc_input >> 6) & 0x01);
	h_flag = ((cc_input >> 5) & 0x01);
	i_flag = ((cc_input >> 4) & 0x01);
	n_flag = ((cc_input >> 3) & 0x01);
	z_flag = ((cc_input >> 2) & 0x01);
	v_flag = ((cc_input >> 1) & 0x01);
	c_flag =  (cc_input & 0x01);
}

/*
**  6 8 0 9   I N S T R U C T I O N S
*/

/* These are the routines which execute the functionality */
/* of the individual 6809 instructions.                   */
/*                                                        */
/* Nearly all of the 'functions' used below are actually  */
/* macros which are expanded in-line.                     */

void near i_jmp(void)
{
	do_jump = TRUE;
	if (a_mode == RELATIVE)
	{
		/* A branch instruction. */
		if ((opcode == BSR) || (opcode == LBSR))
		{
			s_reg.u_value -= 2;
			set_mem16(s_reg.u_value, pcr.u_value);
		}
		else
		{
			switch (opcode - 0x21)
			{
				case BRN:	do_jump = FALSE;
							break;
				case BHI:	do_jump = (!c_flag & !z_flag);
							break;
				case BLS:	do_jump = (c_flag | z_flag);
							break;
				case BHS:	do_jump = (!c_flag);
							break;
				case BLO:	do_jump = (c_flag);
							break;
				case BNE:	do_jump = (!z_flag);
							break;
				case BEQ:	do_jump = (z_flag);
							break;
				case BVC:	do_jump = (!v_flag);
							break;
				case BVS:	do_jump = (v_flag);
							break;
				case BPL:	do_jump = (!n_flag);
							break;
				case BMI:	do_jump = (n_flag);
							break;
				case BGE:	do_jump = !(v_flag ^ n_flag);
							break;
				case BLT:	do_jump = (v_flag ^ n_flag);
							break;
				case BGT:	do_jump = (!z_flag) & !(v_flag ^ n_flag);
							break;
				case BLE:	do_jump = z_flag | (v_flag ^ n_flag);
							break;
			}
		}
	}
	else
	{
		if (opcode > JSR_MIN)
		{
			/* A JSR. */
			s_reg.u_value -= 2;
			set_mem16(s_reg.u_value, pcr.u_value);
		}
	}
	if (do_jump)
		pcr.u_value = addr;
}

void near i_ld8(void)
{
	t8 = reg8_->u_value = get_mem8(addr);
	{
		register unsigned char g8;

		g8 = t8;
		set_nz8(g8);
	}
	v_flag = 0;
}

void near i_ld16(void)
{
	{
		register unsigned int g16;

		reg16_->u_value = g16 = get_mem16(addr);
		set_nz16(g16);
	}
	v_flag = 0;
}

void near i_st8(void)
{
	set_mem8(addr, (t8 = reg8_->u_value));
	{
		register unsigned char g8;

		g8 = t8;
		set_nz8(g8);
	}
	v_flag = 0;
}

void near i_st16(void)
{
	t16 = reg16_->u_value;
	set_mem16(addr, t16);
	{
		register unsigned int g16;

		g16 = t16;
		set_nz16(g16);
	}
	v_flag = 0;
}

void near i_psh(void)
{
	register unsigned int g8 = i_buffer[0];

	v16 = reg16_->u_value;
	if (g8 & 0x80)
	{
		reg16_->u_value -= 2;
		set_mem16(reg16_->u_value, pcr.u_value);
	}
	if (g8 & 0x40)
	{
		if (__r == S_REG)
		{
			reg16_->u_value -= 2;
			set_mem16(reg16_->u_value, u_reg.u_value);
		}
		else
		{
			reg16_->u_value -= 2;
			set_mem16(reg16_->u_value, s_reg.u_value);
		}
	}
	if (g8 & 0x20)
	{
		reg16_->u_value -= 2;
		set_mem16(reg16_->u_value, y_reg.u_value);
	}
	if (g8 & 0x10)
	{
		reg16_->u_value -= 2;
		set_mem16(reg16_->u_value, x_reg.u_value);
	}
	if (g8 & 0x08)
	{
		set_mem8(--(reg16_->u_value), dp_reg.u_value);
	}
	if (g8 & 0x04)
	{
		set_mem8(--(reg16_->u_value), b_reg.u_value);
	}
	if (g8 & 0x02)
	{
		set_mem8(--(reg16_->u_value), a_reg.u_value);
	}
	if (g8 & 0x01)
	{
		set_mem8(--(reg16_->u_value), get_cc_reg());
	}
	ins_cycles += (v16 - reg16_->u_value);
}

void near i_pul(void)
{
	register unsigned int g8 = i_buffer[0];

	v16 = reg16_->u_value;
	if (g8 & 0x01)
	{
		far_set_cc_reg(get_mem8((reg16_->u_value)++));
	}
	if (g8 & 0x02)
	{
		a_reg.u_value = get_mem8((reg16_->u_value)++);
	}
	if (g8 & 0x04)
	{
		b_reg.u_value = get_mem8((reg16_->u_value)++);
	}
	if (g8 & 0x08)
	{
		dp_reg.u_value = get_mem8((reg16_->u_value)++);
	}
	if (g8 & 0x10)
	{
		x_reg.u_value = get_mem16(reg16_->u_value);
		reg16_->u_value += 2;
	}
	if (g8 & 0x20)
	{
		y_reg.u_value = get_mem16(reg16_->u_value);
		reg16_->u_value += 2;
	}
	if (g8 & 0x40)
	{
		if (__r == S_REG)
		{
			u_reg.u_value = get_mem16(reg16_->u_value);
		}
		else
		{
			s_reg.u_value = get_mem16(reg16_->u_value);
		}
		reg16_->u_value += 2;
	}
	if (g8 & 0x80)
	{
		pcr.u_value = get_mem16(reg16_->u_value);
		reg16_->u_value += 2;
	}
	ins_cycles += (reg16_->u_value - v16);
}

void near i_lea(void)
{
	{
		register unsigned int aa;
		register unsigned char rr = __r;

		reg16_->u_value = aa = addr;
		if ((rr == X_REG) || (rr == Y_REG))
			z_flag = (aa == 0);
	}
}

void near i_lsr(void)
{
	get8();
	{
		register unsigned char g8;

		c_flag = ((g8 = v8) & 0x01);
		z_flag = ((g8 >>= 1) == 0);
		v8 = g8;
		n_flag = 0;
	}
	set8();
}

void near i_lsl(void)
{
	get8();
	{
		register unsigned char g8;

		c_flag = ((g8 = v8) >> 7);
		v_flag = c_flag ^ ((g8 >> 6) & 0x01);
		g8 <<= 1;
		set_nz8(g8);
		v8 = g8;
	}
	set8();
}

void near i_ror(void)
{
	get8();
	{
		register unsigned char g8, h8;

		h8 = (c_flag << 7);
		c_flag = ((g8 = v8) & 0x01);
		v8 = g8 = (g8 >> 1) | h8;
		set_nz8(g8);
	}
	set8();
}

void near i_rol(void)
{
	get8();
	{
		register unsigned char g8;

		t8 = c_flag;
		c_flag = ((g8 = v8) >> 7);
		v_flag = c_flag ^ ((g8 >> 6) & 0x01);
		v8 = g8 = (g8 << 1) | t8;
		set_nz8(g8);
	}
	set8();
}

void near i_asr(void)
{
	get8();
	{
		register unsigned char g8;

		c_flag = ((g8 = v8) & 0x01);
		v8 = g8 = ((g8 >> 1) | (g8 & 0x80));
		set_nz8(g8);
	}
	set8();
}

void near i_clr(void)
{
	v8 = n_flag = v_flag = c_flag = 0;
	z_flag = 1;
	set8();
}

void near i_abx(void)
{
	x_reg.u_value += b_reg.u_value;
}

void near i_mul(void)
{
	{
		register unsigned char g8;

		c_flag = ((g8 = b_reg.u_value) >> 7);
		z_flag = ((d_reg_u_value = (a_reg.u_value * g8)) == 0);
	}
}

void near i_nop(void)
{
}

void near i_rts(void)
{
	pcr.u_value = get_mem16(s_reg.u_value);
	s_reg.u_value += 2;
}

void near i_dec(void)
{
	get8();
	{
		register unsigned char g8;

		v_flag = ((g8 = v8) == 0x80);
		v8 = g8 = g8 - 1;
		set_nz8(g8);
	}
	set8();
}

void near i_inc(void)
{
	get8();
	{
		register unsigned char g8;

		v_flag = ((g8 = v8) == 0x7f);
		v8 = g8 = g8 + 1;
		set_nz8(g8);
	}
	set8();
}

void near i_tfr(void)
{
	/* Set source register. */
	__r = (i_buffer[0] >> 4);
	set_reg();

	if (__b == B8)
		exg_get8()
	else
		v16 = reg16_->u_value;

	/* Assign to destination register. */
	__r = (i_buffer[0] & 0x0f);
	set_reg();
	if (__b == B8)
	{
		v8 = (unsigned char)(v16 & 0x00ff);
		exg_set8()
	}
	else
		reg16_->u_value = v16;
}

void near i_exg(void)
{
	/* Set first register. */
	__r = (i_buffer[0] >> 4);
	set_reg();

	if ((__b2 = __b) == B8)
	{
		exg_get8()
		r8temp_ = reg8_;
	}
	else
		r16temp_ = reg16_;

	/* Swap with second register. */
	__r = (i_buffer[0] & 0x0f);
	set_reg();
	if (__b == B8)
	{
		if (__b2 == B8)
			v8 = (unsigned char)(v16 & 0x00ff);
		else
			v8 = (unsigned char)(r16temp_->u_value & 0x00ff);
		exg_get8()
		exg_set8()
		__r = (i_buffer[0] >> 4);
		if (__b2 == B8)
		{
			v8 = (unsigned char)(v16 & 0x00ff);
			reg8_ = r8temp_;
			exg_set8()
		}
		else
			r16temp_->u_value = v16;
	}
	else
	{
		if (__b2 == B8)
		{
			v8 = (unsigned char)(reg16_->u_value & 0x00ff);
			reg16_->u_value = v16;
			__r = (i_buffer[0] >> 4);
			reg8_ = r8temp_;
			exg_set8()
		}
		else
		{
			v16 = reg16_->u_value;
			reg16_->u_value = r16temp_->u_value;
			r16temp_->u_value = v16;
		}
	}
}

void near i_com(void)
{
	get8();
	{
		register unsigned char g8;

		g8 = v8 = ~v8;
		set_nz8(g8);
	}
	v_flag = 0;
	c_flag = 1;
	set8();
}

void near i_neg(void)
{
	get8();
	{
		register unsigned char g8;

		c_flag = ((g8 = v8) != 0);
		v_flag = (g8 == 0x80);
		v8 = g8 = 1 + (~g8);
		set_nz8(g8);
	}
	set8();
}

void near i_bit(void)
{
	t8 = reg8_->u_value & get_mem8(addr);
	{
		register unsigned char g8;

		g8 = t8;
		set_nz8(g8);
	}
	v_flag = 0;
}

void near i_or(void)
{
	t8 = (reg8_->u_value |= get_mem8(addr));
	{
		register unsigned char g8;

		g8 = t8;
		set_nz8(g8);
	}
	v_flag = 0;
}

void near i_and(void)
{
	t8 = (reg8_->u_value &= get_mem8(addr));
	{
		register unsigned char g8;

		g8 = t8;
		set_nz8(g8);
	}
	v_flag = 0;
}

void near i_orcc(void)
{
	t8 = (get_cc_reg() | get_mem8(addr));
	{
		register unsigned char g8;

		g8 = t8;
		set_nz8(g8);
	}
	far_set_cc_reg(t8);
	v_flag = 0;
}

void near i_andcc(void)
{
	t8 = (get_cc_reg() & get_mem8(addr));
	{
		register unsigned char g8;

		g8 = t8;
		set_nz8(g8);
	}
	far_set_cc_reg(t8);
	v_flag = 0;
}

void near i_eor(void)
{
	t8 = (reg8_->u_value ^= get_mem8(addr));
	{
		register unsigned char g8;

		g8 = t8;
		set_nz8(g8);
	}
	v_flag = 0;
}

void near i_tst(void)
{
	get8();
	{
		register unsigned char g8;

		g8 = v8;
		set_nz8(g8);
	}
	v_flag = 0;
}

void near i_rti(void)
{
	__r = S_REG;
	set_reg();
	i_buffer[0] = 0x01;
	i_pul();
	i_buffer[0] = (e_flag) ? 0xfe : 0x80;
	i_pul();
}

void near i_sex(void)
{
	{
		register unsigned char g8;

		g8 = (a_reg.u_value = (b_reg.s_value < 0) ? 0xff : 0x00);
		set_nz8(g8);
	}
}

void near i_cwai(void)
{
	far_set_cc_reg(get_cc_reg() & i_buffer[0]);

	/* Trigger immediate IRQ (not exactly bona-fide behaviour). */
	irq_cycles = irq_rate;
}


void near i_sync(void)
{
	/* Trigger immediate IRQ (not exactly bona-fide behaviour). */
	irq_cycles = irq_rate;
}

void near i_swi(void)
{
	/* Set E flag. */
	e_flag = TRUE;

	/* Push entire machine state onto stack. */
	__r = S_REG;
	set_reg();
	i_buffer[0] = 0xff;
	i_psh();

	/* Get new PCR from appropriate vector. */
	if (ext1)
		pcr.u_value = get_mem16(0xfff4);
	else if (ext2)
		pcr.u_value = get_mem16(0xfff2);
	else
	{
		f_flag = i_flag = TRUE; /* Only SWI1 disables interrupts. */
		pcr.u_value = get_mem16(0xfffa);
	}
}

void near i_add8(void)
{
	s8 = reg8_->u_value = ((v8 = reg8_->u_value) + (t8 = get_mem8(addr)));
	set_nz8(s8);
	c_flag = (s8 < v8);
	set_va8(0,v8,t8);
}

void near i_add16(void)
{
	{
		register unsigned int g16, h16;

		g16 = get_mem16(addr);
		h16 = reg16_->u_value = ((t16 = reg16_->u_value) + g16);
		set_nz16(h16);
		c_flag = (h16 < g16);
		set_va16(t16,g16);
	}
}

void near i_sub8(void)
{
	s8 = reg8_->u_value = ((v8 = reg8_->u_value) - (t8 = get_mem8(addr)));
	set_nz8(s8);
	c_flag = (s8 > v8);
	set_vs8(0,v8,t8);
}

void near i_sub16(void)
{
	{
		register unsigned int g16, h16;

		g16 = reg16_->u_value;
		h16 = get_mem16(addr);
		c_flag = (g16 < h16);
		t16 = reg16_->u_value = g16 - h16;
		set_nz16(t16);
		set_vs16(g16,h16);
	}
}

void near i_cmp8(void)
{
	c_flag = ((s8 = reg8_->u_value) < (t8 = get_mem8(addr)));
	v8 = s8 - t8;
	set_nz8(v8);
	set_vs8(0,s8,t8);
}

void near i_cmp16(void)
{
	{
		register unsigned int g16, h16;

		g16 = get_mem16(addr);
		h16 = reg16_->u_value;
		c_flag = (h16 < g16);
		t16 = h16 - g16;
		set_nz16(t16);
		set_vs16(h16,g16);
	}
}

void near i_adc(void)
{
	s8 = reg8_->u_value = ((v8 = reg8_->u_value) + (t8 = get_mem8(addr)) + c_flag);
	set_nz8(s8);
	set_va8(c_flag,v8,t8);
	if (s8 != v8)
		c_flag = (s8 < v8);
}

void near i_sbc(void)
{
	s8 = reg8_->u_value = ((v8 = reg8_->u_value) - ((t8 = get_mem8(addr)) + c_flag));
	set_nz8(s8);
	set_vs8(c_flag,v8,t8);
	if (s8 != v8)
		c_flag = (s8 > v8);
}

void near i_daa(void)
{
	v8 = reg8_->u_value;
	t8 = (h_flag || ((v8 & 0x0f) >= 0x0a)) ? 6 : 0;
	t8 += ((c_flag || (v8 >= 0xa0) ||
		((v8 >= 0x90) && ((v8 & 0x0f) >= 0x0a))) ? 0x60 : 0);
	s8 = reg8_->u_value = v8 + t8;
	set_nz8(s8);
	c_flag |= (s8 < v8);
}

void near i_doio(void)
{
	switch (i_buffer[0])
	{
		case doio_keyboard:		/* Keyboard update. */
								if (new_int9_set)
								{
									__r = S_REG;
									set_reg();
									i_buffer[0] = (arch == tandy) ? 0x54 : 0x14;
									i_psh();
									return;
								}
								else
									check_sw_keyboard();
								break;

		case doio_joystick:		/* Update joystick positions. */
								update_joysticks(0x15a);
								break;

		case doio_bit_in:		/* Bit input from cassette. */
								read_bit();
								b_reg.u_value = memory[0x92] - c_flag;
								break;

		case doio_byte_in:		/* Byte input from cassette. */
								read_byte();
								set_mem8(0x83,0);
								v_flag = c_flag = 0;
								n_flag = z_flag = 1;
								break;

		case doio_cassette_off:	/* Cassette off. */
								close_files(FALSE);
								f_flag = i_flag = FALSE;
								set_mem8(0xff21,get_mem8(0xff21) & 0xf7);
								break;

		case doio_cassette_on:	/* Cassette on for reading. */
								break;

		case doio_byte_out:		/* Byte output to cassette. */
								write_byte();
								y_reg.u_value = 0xbe69;
								b_reg.u_value = 0;
								c_flag = 1;
								break;
	}
	pcr.u_value = get_mem16(s_reg.u_value);
	s_reg.u_value += 2;
}

void near i_reset(void)
{
	/* Retrieve initial program counter from restart vector. */
	pcr.u_value = get_mem16(bootup);

	/* Initialise dp and cc registers. */
	dp_reg.u_value = 0;
	i_flag = f_flag = TRUE;
}

/* Handler for illegal instructions. */
void near illegal(void)
{
	unsigned char		pcr_string[21], opcode_string[21];
	unsigned int		illegal_pcr;


	illegal_pcr = (pcr.u_value - (ext1 | ext2)) - 1;
	sprintf(pcr_string,    "   PCR    = $%04X", illegal_pcr);
	sprintf(opcode_string, "   Opcode = $%s%02X",
		(ext1 ? "10" : (ext2 ? "11" : "")), opcode);

	if (!ignore_illegal_opcodes)
	{
		switch(selection_box(8,21,3,FALSE,FALSE,FALSE,1,
			" ILLEGAL INSTRUCTION","",
			pcr_string,
			opcode_string,"",
			"   1 - Ignore",
			"   2 - Ignore All",
			"   3 - Control Menu"))
		{
			case 2:	ignore_illegal_opcodes = TRUE;
					break;

			case 3:	emu_keys(CONTROL_MENU); /* Jump to control menu. */
					if (in_debug && !await_breakpoint && !debug_disp)
					{
						/* Rewind PC if just requested debugger. */
						pcr.u_value = illegal_pcr;
					}
					break;
		}
	}
}

/* Generate FIRQ (used to execute cartridge games). */
void firq(void)
{
	memory[0xff23] |= 0x80;

	/* Clear E flag. */
	e_flag = FALSE;

	/* Push partial machine state onto stack. */
	__r = S_REG;
	set_reg();
	i_buffer[0] = 0x81;
	i_psh();

	/* Set I flag so that no IRQ's will occur   */
	/* and get FIRQ handler address from $FFF6. */
	i_flag = TRUE;
	pcr.u_value = get_mem16(0xfff6);
}

/* Reset 6809, SAM and PIA chips. */
void reset_hardware(void)
{
	unsigned int hw_address;

	/* Rewind any open cassettes. */
	close_files(TRUE);

	/* Reset 6809 engine. */
	i_reset();

	/* Reset timers. */
	irq_cycles = kbcnt = refresh_count = 0;

	/* Reset PIA's. */
	for (hw_address = 0xff00; hw_address < 0xff40; hw_address++)
		memory[hw_address] = 0;

	/* Pullup resistor to select first ROM in Dragon 64. */
	if (arch == dragon64)
		memory[0xff22] = 0x04;

	/* Reset SAM. */
	new_screen_size	= 0;
	new_screen_base	= 0;
	mapmode1		= FALSE;

	/* Setup default dragon video attributes. */
	set_vmode(0);
}

/* Examine ROM file to decide which architecture to emulate. */
void detect_arch(void)
{
	/* Detect architecture type. */
	switch (memory[0xbec0])
	{
		case 0x00:	arch = dragon32;
					break;

		case 0x07:	arch = dragon64;
					break;

		case 0x80:	arch = tandy;
					break;

		default:	quit("ROM File Appears To Be Corrupt",0);
	}
}

/* Convert hexadecimal input to decimal. */
unsigned int xtod(unsigned char *ins, unsigned char *ach, unsigned char len)
{
	unsigned int retval = 0;
	unsigned char xch;

	while (len-- > 0)
	{
		xch = toupper(ins[*ach]);
		(*ach)++;
		if (!isxdigit(xch))
			quit("Bad Hex String In /O Option",0);
		retval <<= 4;
		retval |= ((xch > '9') ? (xch - 55) : (xch - '0'));
	}

	return(retval);
}

/* Output list of command line/config file options. */
void options_list(unsigned char **argv)
{
	printf("\n%s%d.%02d%s%s\n%s%s%s\n\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n",
		"PC-DRAGON EMULATOR & DEBUGGER V", VER_MAJOR, VER_MINOR, BETA,
		" (C)1993-97 PAUL BURGIN",
		"Usage: ", argv[0], " [options] [progfile[.pak/dgn]] [@source[.a09]]",
		"\t/32\tEmulate Dragon 32",
		"\t/64\tEmulate Dragon 64",
		"\t/T\tEmulate Tandy Color Computer II",
		"\t/K\tReal Mode Keyboard Emulation~",
		"\t/M\tView Reference Manual Only",
		"\t/V\tPrint Version Information Only",
		"\t/J\tEnable PC Joystick~",
		"\t/G\tDigital Joystick Emulation~",
		"\t/Rr\tSet Joystick Sampling Rate r",
		"\t/P\tAdvanced Printer Support~",
		"\t/CR\tTurn <CR> To <CR><LF> Translation Off",
		"\t/CRLF\tTurn <CR> To <CR><LF> Translation On",
		"\t/LPTp\tSelect Parallel Printer Port p",
		"\t/COMs\tSelect Serial Printer Port s",
		"\t/Br\tSelect Serial Baud Rate (r=1200,2400,4800,9600)",
		"\t/FILE\tRedirect Printer Output To File PRINTER.OUT",
		"\t/NP\tDisable LPT And COM Ports",
		"\t/NB\tDo Not Install ROM Breakpoints~",
		"\t/NN\tDo Not Prompt For Cassette Names~");

	printf("\n-- Press Any Key -- ");
	new_int9_set = FALSE;
	getch2(TRUE);

	printf("\n\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s",
		"\t/SB\tIgnore Bootup Snapshot~",
		"\t/SD\tIgnore Debugger State In Snapshot Files~",
		"\t/SO\tIgnore Options In Snapshot Files~",
		"\t/SX\tIgnore PC-Dragon Extensions In Snapshot Files~",
		"\t/SC\tCompress Snapshot Files~",
		"\t/SLd\tSet Snapshot Save Length To d Bytes (d=1..65280)",
		"\t/SPr\tSnapshot Save Portability (r=N[none], D[dragon], A[any])",
		"\t/SVm\tUse Snapshot Format Of Debug CoCo2 V1.m (m=2,3,4)",
		"\t/Fk's'\tAssign String s To Key Alt-Fk (k=1..10) Or Alt-k (k=A..Z)",
		"\t/Aa\tArtifacting Mode a (a=O[off], B[blue], R[red])",
		"\t/Cc\tCursor Mode c (c=O[off], U[underline], S[solid])",
		"\t/X\tDisplay Inverse Text As Lower Case~",
		"\t/Ofsm$x\tConfigure Borders And Colours (See Manual)",
		"\t/Q\tQuadruple Interrupt Frequency",
		"\t/U\tQuarter Interrupt Frequency",
		"\t/IAp\tUse Assembler Path p",
		"\t/IEp\tUse Editor Path p",
		"\t/II\tIgnore Illegal Opcodes~",
		"\t/ISl\tSwapping Level (s=0[none], 1[shell], 2[editor], 3[max])",
		"\n~ toggleable options\n");
}

/* Process a string of options. */
void option_string(unsigned char *opt_string, char **argv)
{
	unsigned char	arg_char = 0;
	char			local_string[MAXPATH];

	while (opt_string[arg_char] != '\0')
	{
		if ((opt_string[arg_char] == '/') || (opt_string[arg_char] == '-'))
		{
			arg_char += 2;

			switch (toupper(opt_string[arg_char - 1]))
			{

			case '3':	req_arch = dragon32;
						if (opt_string[arg_char] == '2')
							arg_char++;
						if (toupper(opt_string[arg_char]) == 'K')
							arg_char++;
						break;

			case '6':	req_arch = dragon64;
						if (opt_string[arg_char] == '4')
							arg_char++;
						if (toupper(opt_string[arg_char]) == 'K')
							arg_char++;
						break;

			case 'A':	switch (toupper(opt_string[arg_char++]))
						{
							case 'O':	set_artifact(0);	break;
							case 'B':	set_artifact(1);	break;
							case 'R':	set_artifact(2);	break;

							default:	quit("Error In /A Option",0);
						}
						break;

			case 'B':	if (strncmp(&opt_string[arg_char],"1200",4) == 0)
							set_comspeed(_COM_1200);
						else if (strncmp(&opt_string[arg_char],"2400",4) == 0)
							set_comspeed(_COM_2400);
						else if (strncmp(&opt_string[arg_char],"4800",4) == 0)
							set_comspeed(_COM_4800);
						else if (strncmp(&opt_string[arg_char],"9600",4) == 0)
							set_comspeed(_COM_9600);
						else
							quit("Error In /B Option",0);
						arg_char += 4;
						break;

			case 'C':   if (strnicmp(&opt_string[arg_char],"RLF",3) == 0)
						{
							arg_char += 3;
							crlf = TRUE;
						}
						else if (strnicmp(&opt_string[arg_char],"OM1",3) == 0)
						{
							if (!change_lpt(COM1))
								quit("COM1 Not Found",0);
							arg_char += 3;
						}
						else if (strnicmp(&opt_string[arg_char],"OM2",3) == 0)
						{
							if (!change_lpt(COM2))
								quit("COM2 Not Found",0);
							arg_char += 3;
						}
						else switch (toupper(opt_string[arg_char++]))
						{
							case 'R':	crlf = FALSE;				break;

							case 'S':	cursortype = _SOLIDCURSOR;	break;
							case 'U':	cursortype = _NORMALCURSOR;	break;
							case 'O':	cursortype = _NOCURSOR;		break;

							default:	quit("Error In /C Option",0);
						}
						break;

			case 'D':	in_debug = !in_debug;
						break;

			case 'F':   if (strnicmp(&opt_string[arg_char],"ILE",3) == 0)
						{
							arg_char += 3;
							goto prnt_file;
						}

						if ((tolower(opt_string[arg_char]) >= 'a')
								&& (tolower(opt_string[arg_char]) <= 'z'))
						{
							unsigned char kword = tolower(opt_string[arg_char++]) - 'a';
							unsigned char delim;

							if (strlen(&opt_string[arg_char]) < 2)
								quit("Error In BASIC Keyword Format",0);
							delim = opt_string[arg_char++];
							strncpy(keywords[kword],&opt_string[arg_char],10);
							if (strchr(keywords[kword],delim) == NULL)
								quit("Mismatched BASIC Keyword Delimiter (Or Too Long)",0);
							*strchr(keywords[kword],delim) = '\0';
							arg_char += strlen(keywords[kword]) + 1;
						}
						else switch(opt_string[arg_char])
						{
							case '#':	opt_string[arg_char] = '9' + 2;
							case '1':	if (opt_string[arg_char+1] == '0')
											opt_string[++arg_char] = '9' + 1;
							case '2':
							case '3':
							case '4':
							case '5':
							case '6':
							case '7':
							case '8':
							case '9':	{
											unsigned char preset = opt_string[arg_char++] - '1';
											unsigned char delim;

											if (strlen(&opt_string[arg_char]) < 2)
												quit("Error In Function String Format",0);
											delim = opt_string[arg_char++];
											strncpy(presets[preset],&opt_string[arg_char],PRESET_LEN+1);
											if (strchr(presets[preset],delim) == NULL)
												quit("Mismatched Function String Delimiter (Or Too Long)",0);
											*strchr(presets[preset],delim) = '\0';
											arg_char += strlen(presets[preset]) + 1;
										}
										break;

							default:
							prnt_file:	change_printfile(!printfile);
										break;
						}
						break;

			case 'G':   digital_joysticks = !digital_joysticks;
						break;

			case 'H':
			case '?':	options_list(argv);
			case '!':	quit("",2);

			case 'I':   switch(toupper(opt_string[arg_char++]))
						{
							case 'A':	strcpy(path_assembler, &opt_string[arg_char]);
										arg_char = strlen(opt_string);
										break;

							case 'E':	strcpy(path_editor, &opt_string[arg_char]);
										arg_char = strlen(opt_string);
										break;

							case 'S':   swapping = opt_string[arg_char++] - '0';
										if (swapping > 3)
											quit("Swapping Level Out Of Range",0);
										break;

							case 'I':	ignore_illegal_opcodes = !ignore_illegal_opcodes;
										break;

							default:	quit("Bad /I Option",0);
						}
						break;

			case 'J':   if (strnicmp(&opt_string[arg_char],"OY",2) == 0)
							arg_char += 2;
						if (hasjoystick)
							realjoy = !realjoy;
						else
							quit("Game Adapter Not Found",0);
						break;

			case 'K':	new_int9_set = !new_int9_set;
						break;

			case 'L':	if (strnicmp(&opt_string[arg_char],"PT1",3) == 0)
						{
							if (!change_lpt(LPT1))
								quit("LPT1 Not Found",0);
						}
						else if (strnicmp(&opt_string[arg_char],"PT2",3) == 0)
						{
							if (!change_lpt(LPT2))
								quit("LPT2 Not Found",0);
						}
						else
							quit("Error In /LPT Option",0);
						arg_char += 3;
						break;

			case 'M':	view_manual();
						quit("",2);

			case 'N':	switch(toupper(opt_string[arg_char++]))
						{
							case 'N':	ask_filenames = !ask_filenames;
										break;

							case 'B':	install_breakpoints = !install_breakpoints;
										break;

							case 'P':	change_lpt(NO_PORTS);
										break;

							default:	quit("Bad /N Option",0);
						}
						break;

			case 'O':	{
							unsigned char o_screen	= opt_string[arg_char+1] - '0';
							unsigned char o_vmode	= opt_string[arg_char+2] - '0';

							if (o_screen > 1)
								quit("Bad Pmode Or Screen In /O Option",0);

							switch(toupper(opt_string[arg_char]))
							{
								case 'B':	if (o_vmode > 1)
												quit(bad_pmode,0);
											if (opt_string[arg_char+3] != '$')
												quit(bad_dollar,0);
											arg_char += 4;
											if (strlen(&opt_string[arg_char]) != 2)
												quit(bad_len,0);
											border_ary[o_vmode][o_screen] = xtod(opt_string,&arg_char,2) & 63;
											break;

								case 'T':   if (opt_string[arg_char+2] != '$')
												quit(bad_dollar,0);
											arg_char += 3;
											if (strlen(&opt_string[arg_char]) != 2)
												quit(bad_len,0);
											fore_ary[o_screen] = xtod(opt_string,&arg_char,1);
											back_ary[o_screen] = xtod(opt_string,&arg_char,1);
											break;

								case 'G':	if (o_vmode > 1)
												quit(bad_pmode,0);
											if (opt_string[arg_char+3] != '$')
												quit(bad_dollar,0);
											arg_char += 4;
											if (strlen(&opt_string[arg_char]) != 12)
												quit(bad_len,0);
											vmode_ary[o_vmode][o_screen][0] = xtod(opt_string,&arg_char,2) & 63;
											vmode_ary[o_vmode][o_screen][1] = xtod(opt_string,&arg_char,2) & 63;
											vmode_ary[o_vmode][o_screen][2] = xtod(opt_string,&arg_char,2) & 63;
											vmode_ary[o_vmode][o_screen][3] = xtod(opt_string,&arg_char,2) & 63;
											vmode_ary[o_vmode][o_screen][4] = xtod(opt_string,&arg_char,2) & 63;
											vmode_ary[o_vmode][o_screen][5] = xtod(opt_string,&arg_char,2) & 63;
											break;

								default:	quit("Bad Function In /O Option",0);
							}
						}
						break;

			case 'P':	printsimple = !printsimple;
						break;

			case 'Q':	irq_rate >>= 2;
						break;

			case 'R':	if (sscanf(&opt_string[arg_char],"%[0-9]",local_string) != 1)
							quit("Incorrect /R Option Format",0);
						if (atoi(local_string) < 256)
							hardjoy_resolution = atoi(local_string);
						else
							quit("/R Option Out Of Range",0);
						arg_char += strlen(local_string);
						break;

			case 'S':	switch(toupper(opt_string[arg_char++]))
						{
							case 'B':	load_bootsnap = !load_bootsnap;
										break;

							case 'C':	compress = !compress;
										break;

							case 'D':	load_ds_default = !load_ds_default;
										break;

							case 'O':	snap_opt_default = !snap_opt_default;
										break;

							case 'X':	load_ext_default = !load_ext_default;
										break;

							case 'V':	switch(toupper(opt_string[arg_char++]))
										{
											case '2':	version_default = V1_2; break;
											case '3':	version_default = V1_3; break;
											case '4':	version_default = V1_4; break;

											default:	quit("Requested PAK Version Not Supported",0);
										}
										break;

							case '2':	version_default = V1_2; break;
							case '3':	version_default = V1_3; break;
							case '4':	version_default = V1_4; break;

							case 'L':	if (sscanf(&opt_string[arg_char],"%[0-9]",local_string) != 1)
											quit("Incorrect /SL Option Format",0);
										snap_length = atoi(local_string);
										if ((snap_length < 1) || (snap_length > 0xff00))
											quit("/SL Value Must Be In Range $0001-$FF00",0);
										arg_char += strlen(local_string);
										break;

							case 'P':	switch(toupper(opt_string[arg_char++]))
										{
											case 'N':	write_port = this_rom; break;
											case 'D':	write_port = any_dragon; break;
											case 'A':	write_port = any_rom; break;

											default:	quit("Unrecognised Portability Option",0);
										}
										break;

							default:	quit("Invalid Snapshot Option",0);
						}
						break;

			case 'T':	req_arch = tandy;
						break;

			case 'U':	if (irq_rate >= 17800)
							quit("IRQ Rate Is Already Minimum",0);
						irq_rate <<= 2;
						break;

			case 'V':	printf("\n%s\n%s%d.%02d%s %s\n%s%s\n",
							"PC-DRAGON EMULATOR & DEBUGGER (C)1993-97 PAUL BURGIN",
							"Version: ", VER_MAJOR, VER_MINOR, BETA, regstr,
							"Built:   ", __DATE__);
						quit("",2);

			case 'W':	wipe_mem = !wipe_mem;
						break;

			case 'X':	lower_case = !lower_case;
						break;

			default:	printf("\nPC-DRAGON: Unknown Option \"%c%c\"\n",
							opt_string[arg_char-2], opt_string[arg_char-1]);
						quit("",2);

			}
		}
		else if (arg_char == 0)
		{
			if (opt_string[arg_char] == '@')
			{
				strcpy(name_editfile, &opt_string[1]);
				if (name_editfile[0] == '\0')
					quit("Error In Source File Name",0);
				if (strext(name_editfile) == NULL)
					strcat(name_editfile,".a09");
			}
			else
			{
				if (options_file != name_stream)
					cartnamed = 0;
				if (cartnamed < 2)
				{
					name_stream = options_file;
					strncpy(cartfile[cartnamed++],opt_string,MAXPATH+4);
				}
				else
					quit("More Than Two Snapshot/Cartridge Files Specified",0);
			}
			arg_char = strlen(opt_string);
		}
		else
		{
			printf("\nPC-DRAGON: Garbled Option \"%s\"\n",
				&opt_string[arg_char]);
			quit("",2);
		}
	}
}

/* Process config file and command line options. */
void options(int argc, char **argv)
{
	int				arg_num, arg_ch, fch;
	unsigned char	opt_line[MAXPATH];

	/* Process options in config file. */
	option_string(os, argv);
	if ((options_file = fopen(config_file,"rt")) != NULL)
	{
		do
		{
			arg_ch = 0;
			do
			{
				fch = fgetc(options_file);
				if ((fch != EOF) && (fch != '\n') && (fch != ' ')
						&& (fch != '\t') && (arg_ch < MAXPATH))
					opt_line[arg_ch++] = fch;
			} while ((fch != EOF) && (fch != '\n')
				&& (fch != ' ') && (fch != '\t'));
			if ((arg_ch > 0) && (opt_line[0] != '#'))
			{
				opt_line[arg_ch] = '\0';
				option_string(opt_line, argv);
			}
		} while (fch != EOF);
		fclose(options_file);
	}
	options_file = NULL;

	/* Process options on command line. */
	for (arg_num = 1; arg_num < argc; arg_num++)
		option_string(argv[arg_num], argv);
}

/* Main function. */
void main(int argc, char **argv)
{
	/* Check system hardware. */
	{
		unsigned int hardware;

		hardware = biosequip();
		bootparallel = numparallel = (hardware >> 14) & 0x03;
		bootserial = numserial = (hardware >> 9) & 0x0007;
		hasjoystick = ((hardware & 0x0100) == 0); /* Is this correct? */
	}

	/* Choose default printer based on available hardware. */
	check();
	if (numparallel > 0)
		lpt = LPT1;
	else if (numserial > 0)
		lpt = COM1;
	else
		lpt = NO_PORTS;

	/* Establish full path to executable. */
	{
		unsigned char *pcd_path;

		pcd_path = searchpath(argv[0]);
		_splitpath(pcd_path, pcd_drive, pcd_directory, NULL, NULL);
		if (strlen(pcd_directory) > 1)
			pcd_directory[strlen(pcd_directory)-1] = '\0';
	}

	/* Change to PC-Dragon directory. */
	set_directory(1);

	/* Process command line options. */
	options(argc, argv);

	/* Allocate some space required by cassette.c for name storage. */
	if ((last_filename = (unsigned char *)malloc(MAXPATH)) == NULL)
		quit("Cannot Allocate Cassette Name Buffer",0);

	/* Allocate screen buffers. */
	if ((info_screen = (unsigned char *)malloc(4000)) == NULL)
		quit("Cannot Allocate Info Screen Buffer",0);
	if ((debug_screen = (unsigned char *)malloc(4000)) == NULL)
		quit("Cannot Allocate Debug Screen Buffer",0);
	if ((norm_screen = (unsigned char *)malloc(2000)) == NULL)
		quit("Cannot Allocate Dialogue Box Buffer",0);

	/* Allocate 64K for address space of emulated dragon. */
	if ((memory = (unsigned char far *)farmalloc(MEMORY_SIZE+1)) == NULL)
		quit("Cannot Allocate 64K Address Space",0);
	if (wipe_mem)
		_fmemset(memory,0,0xff00);

	/* Load Dragon ROM. */
	if (!load_dgn_file("dragrom",0))
	{
		if (!load_dgn_file("dragon",0))
		{
			if (!load_dgn_file("tandyrom",0))
			{
				if (!load_dgn_file("coco",0))
					quit("Cannot Find ROM File",0);
			}
		}
	}

	/* Detect architecture type. */
	detect_arch();

	/* Change architecture if requested. */
	if ((req_arch != dontcare) && (arch != req_arch))
	{
		switch (req_arch)
		{
			case dragon32:	if (!load_dgn_file("d32rom",0))
								quit("Error Reading D32ROM.DGN",0);
							break;

			case dragon64:	if (!load_dgn_file("d64rom",0))
								quit("Error Reading D64ROM.DGN",0);
							break;

			case tandy:		if (!load_dgn_file("tandyrom",0))
								quit("Error Reading TANDYROM.DGN",0);
							break;
		}
		detect_arch();
		if (arch != req_arch)
			quit("ROM File Does Not Agree With Filename",0);
	}

	/* Patch emulation breakpoints with DOIO instruction. */
	if (install_breakpoints)
	{
		ptr = (arch == tandy) ? tandy_breakpoints : d32_breakpoints;
		for (doio_service = 0; doio_service < doio_last; doio_service++)
		{
			addr = *ptr++;
			memory[addr]		= doio_opcode;
			memory[addr + 1]	= doio_service;
		}
	}

	/* Allocate extra space for Dragon 64 or Tandy CoCo. */
	if (arch != dragon32)
	{
		/* Allocate upper 32K RAM. */
		if ((hiram = (unsigned char far *)farmalloc(0x8000)) == NULL)
			quit("Cannot Allocate Upper 32K RAM",0);

		if (wipe_mem)
			_fmemset(hiram,0,0x8000);

		/* Allocate memory for 2nd Dragon 64 ROM. */
		if (arch == dragon64)
		{
			if ((rom2 = (unsigned char far *)farmalloc(0x4000)) == NULL)
				quit("Cannot Allocate Memory For Second ROM",0);

			/* Load second ROM file. */
			rom2loaded = load_dgn_file("d64rom2",1);

			/* Add breakpoints and '64' signature to 2nd ROM. */
			if (rom2loaded && (rom2[0x0001] == 0xf3))
			{
				if (install_breakpoints)
				{
					ptr = d64_breakpoints;
					for (doio_service = 0; doio_service < doio_last; doio_service++)
					{
						addr = *ptr++;
						rom2[addr]		= doio_opcode;
						rom2[addr + 1]	= doio_service;
					}
				}
				rom2[0x3ff0] = '6';
				rom2[0x3ff1] = '4';
			}
		}
	}

	/* Reset extended services write-only memory. */
	memory[0xffec] = 0xff;	memory[0xffed] = 0xe0;
	memory[0xffee] = 0x00;	memory[0xffef] = 0x00;

	/* Calculate how many seconds 10000 IRQ's should take. */
	should_be_seconds = (double)irq_rate / 89.0;

	/* Disable scrolling of windows. */
	_wscroll = FALSE;

	/* Install new keyboard interrupt handler. */
	if (new_int9_set)
		Set_New_Int9();

	/* Initial title message. */
	textmode(C80);
	textattr(BLACK);
	clrscr();
	_setcursortype(_NOCURSOR);
	put_title(TRUE);
	getch2(TRUE);

	/* Change to 40 character colour mode for dragon text mode emulation. */
	setup_screen();

	/* Re-install new keyboard interrupt handler. */
	if (new_int9_set)
		Set_New_Int9();

	/* Reset other hardware. */
	reset_hardware();
	new_screen_base = 0x0400;
	execute_vmode(TRUE);

	/* Initialise static items in PAK image. */
	init_snap_info();

	/* Set snapshot save length if not already specified by user. */
	if (snap_length > 0xff00)
		snap_length = (arch == dragon32) ? 0x8000 : 0xff00;

	/* Load startup snapshot (always V1.4 with no state/options & ext=on). */
	load_debug_state = snap_load_options = FALSE;
	version = V1_4;
	load_extensions = TRUE;
	if (load_bootsnap)
		snapboot = load_snapshot(bootsnap,NULL);
	load_debug_state	= load_ds_default;
	snap_load_options	= snap_opt_default;
	version				= version_default;
	load_extensions		= load_ext_default;
	strcpy(snapshot_name,"snapshot");

	/* Load chosen snapshot/cartridge files. */
	while (cartnamed-- > 0)
	{
		if (!load_dgn_file(cartfile[cartnamed],2))
		{
			if (load_snapshot(cartfile[cartnamed],NULL))
			{
				snapboot = TRUE;
			}
			else
			{
				quit("Error Finding Or Reading Specified Program File",1);
			}
		}
	}

	/* Preserve machine state for snapshot loads. */
	setjmp(hard_reset);
	drag_usedkey = TRUE;

	/* Record starting time. */
	last_time = two_second_count = 0;
	this_time = clock();

	/* Loop forever interpreting 6809 instructions. */
	for (;;)
	{
		/* Return to debugger if enabled. */
		if (in_debug)
			debugger();

		/* Check for emulator control keys. */
		if (new_int9_set)
		{
			/* Check 'Real Mode' keyboard. */
			if (keys[kF1] | keys[kF2] | keys[kF3] | keys[kF4] | keys[kF5]
			  | keys[kF6] | keys[kF7] | keys[kF8] | keys[kF9] | keys[kF10]
			  | keys[kLEFTCTRL] | keys[kRIGHTCTRL]
			  | keys[kLEFTALT] | keys[kRIGHTALT])
			{
				service_emu_keys();
			}
		}
		else if (kbcnt != 0)
		{
			/* Check 'PC Mode' keyboard. */
			kbcnt = 0;

			/* Use key last input by Dragon OS routine, if      */
			/* there was one. Else check the PC's input buffer. */
			if (drag_usedkey && iskey())
			{
				if ((keyin = getch2(FALSE)) != 0)
					drag_usedkey = FALSE;
			}
			else
			{
				/* Check for Ctrl-Alt combination. */
				if ((bioskey(2) & 0x0c) == 0x0c)
				{
					/* Junk keyboard buffer. */
					drag_usedkey = TRUE;
					while (iskey())
						getch2(TRUE);

					/* Jump to control menu. */
					emu_keys(CONTROL_MENU);
				}
			}
		}

		/* Read next instruction opcode. */
		opcode = get_mem8(pcr.u_value++);

		/* Check page of opcode, read another byte if necessary. */
		{
			register unsigned int op;
			register unsigned int ee;

			ee  =  ext1 = ((op = opcode) == EXT1);
			ee |= (ext2 = (op == EXT2));

			while (ee)
			{
				op = (opcode = get_mem8(pcr.u_value++));
				ee = ((op == EXT1) | (op == EXT2));
			}
		}

		{
			register unsigned int i_size;

			{
				register ins_t *ip;

				/* Lookup details of instruction in table. */
				ip = &i_table[opcode * sizeof(instruction)];

				/* Calculate size of rest of instruction. */
				i_type		= *ip++;
				a_mode		= *ip++;
				__r			= *ip++;
				ins_cycles	= *ip++;
				i_size		= *ip++;
			}

			/* Transform branches into long branches if required. */
			if (ext1 && (a_mode == RELATIVE))
				i_size++;

			{
				register boff = 0;

				/* Fetch rest of instruction. */
				while (--i_size)
					i_buffer[boff++] = get_mem8(pcr.u_value++);
			}
		}

		{
			register unsigned char rr = __r;

			/* Process selected page 1 and page 2 instructions. */
			if ((i_type == SUB16) && (rr == D_REG))
			{
				if (ext1)
					i_type = CMP16;
				else if (ext2)
				{
					i_type = CMP16;
					__r = U_REG;
				}
			}
			else if (ext1 && (rr == U_REG))
			{
				__r = S_REG;
			}
			else if (rr == X_REG)
			{
				if (ext1)
					__r = Y_REG;
				else if (ext2)
					__r = S_REG;
			}
		}

		/* Setup pointer to implicit register. */
		set_reg();

		/* Calculate effective address. */
		switch (a_mode)
		{
			case INDEXED:	switch ((i_buffer[0] >> 5) & 0x03)
							{
								case OFF_X:		r16temp_ = &x_reg;
												break;
								case OFF_Y:		r16temp_ = &y_reg;
												break;
								case OFF_U:		r16temp_ = &u_reg;
												break;
								case OFF_S:		r16temp_ = &s_reg;
												break;
							}
							if (i_buffer[0] < 0x80) /* 4 bit plus sign. */
							{
								if ((i_buffer[0] & 0x10) == 0)
									addr = r16temp_->u_value + (i_buffer[0] & 0x000f);
								else
									addr = r16temp_->u_value + (i_buffer[0] | 0xfff0);
								ins_cycles++;
							}
							else
							{
								indirect = i_buffer[0] & 0x10;
								switch (i_buffer[0] & 0x0f)
								{

								case 0: /* Allows illegal (,r+). */
									addr = r16temp_->u_value++;
									ins_cycles += 2;
									break;

								case 1:
									addr = r16temp_->u_value;
									r16temp_->u_value += 2;
									ins_cycles += 3;
									break;

								case 2: /* Allows illegal (,-r). */
									addr = --r16temp_->u_value;
									ins_cycles += 2;
									break;

								case 3:
									r16temp_->u_value -= 2;
									addr = r16temp_->u_value;
									ins_cycles += 3;
									break;

								case 4:
									addr = r16temp_->u_value;
									break;

								case 5:
									addr = (r16temp_->u_value + b_reg.s_value);
									ins_cycles++;
									break;

								case 6:
									addr = (r16temp_->u_value + a_reg.s_value);
									ins_cycles++;
									break;

								case 7: /* 7 is illegal. */
									addr = r16temp_->u_value;
									break;

								case 8:
									addr = r16temp_->u_value + (signed char)get_mem8(pcr.u_value++);
									ins_cycles++;
									break;

								case 9:
									addr = r16temp_->u_value + get_mem16(pcr.u_value);
									pcr.u_value += 2;
									ins_cycles += 4;
									break;

								case 10: /* 10 is illegal. */
									addr = r16temp_->u_value;
									break;

								case 11:
									addr = r16temp_->u_value + d_reg_u_value;
									ins_cycles += 4;
									break;

								case 12:
									addr = pcr.u_value + (signed char)get_mem8(pcr.u_value) + 1;
									pcr.u_value++;
									ins_cycles++;
									break;

								case 13:
									addr = pcr.u_value + get_mem16(pcr.u_value) + 2;
									pcr.u_value += 2;
									ins_cycles += 5;
									break;

								case 14: /* 14 is illegal. */
									addr = r16temp_->u_value;
									break;

								case 15:
									addr = get_mem16(pcr.u_value);
									pcr.u_value += 2;
									indirect = 0xff;
									ins_cycles += 2;
									break;
								}
								if (indirect != 0x00)
								{
									addr = get_mem16(addr);
									ins_cycles += 3;
								}
							}
							break;

			case IMMEDIATE:	addr = I_DATA;
							break;

			case DIRECT:	addr = (dp_reg.u_value << 8) + i_buffer[0];
							break;

			case EXTENDED:	addr = (i_buffer[0] << 8) + i_buffer[1];
							break;

			case RELATIVE:	if (!ext1 && (opcode != LBRA) && (opcode != LBSR))
								addr = pcr.u_value + (signed char)i_buffer[0];
							else
								addr = pcr.u_value + (i_buffer[0] << 8) + i_buffer[1];
							break;
		}

		/* Call appropriate emulator function. */
		execute_instruction[i_type]();

		/* Update and check IRQ counter. */
		if ((irq_cycles += ins_cycles) >= irq_rate)
		{
			/* Reset IRQ cycles counter. */
			irq_cycles = 0;

			/* Update video refresh cycles counter. */
			if (++refresh_count > REFRESH_RESOLUTION)
			{
				refresh_count = 0;

				if (!debug_disp)
				{
					/* Check whether major mode changes required. */
					if ((new_vmode != vmode)
							|| (new_screen_base != screen_base)
							|| (new_screen_size != screen_size))
						execute_vmode(FALSE);

					/* Otherwise check whether minor mode changes required. */
					else if ((new_palette != palette)
							|| (new_four_colours != four_colours))
						set_vmode(memory[0xff22]);
				}
			}

			/* Update emulator keyboard scan counter. */
			kbcnt++;

			/* Update pseudo-hardware keyboard reset counter. */
			if (key_scan_reset < KEY_SCAN_RESET)
			{
				if (++key_scan_reset == KEY_SCAN_RESET)
					key_scan_counter = KEY_SCAN_RESOLUTION + KEY_SCAN_DEADBAND;
			}

			/* Update joystick button scan rate counter. */
#if BUTTON_RESOLUTION > 1
			if (button_counter < BUTTON_RESOLUTION)
				button_counter++;
#else
			button_counter = 1;
#endif

			/* Update PIA-0 CR-B interrupt bit 7, and    */
			/* check whether PIA-0 IRQ-B output enabled. */
			/* If it is, cause IRQ in emulated machine.  */
			memory[0xff03] |= 0x80;
			if ((memory[0xff03] & 0x01) && !i_flag)
			{
				/* Set E flag. */
				e_flag = TRUE;

				/* Push entire machine state onto stack. */
				__r = S_REG;
				set_reg();
				i_buffer[0] = 0xff;
				i_psh();

				/* Set I flag so that no further IRQ's will occur */
				/* and get IRQ handler address from $FFF8.        */
				i_flag = TRUE;
				pcr.u_value = get_mem16(0xfff8);
			}

			/* Time to update speed gauge? */
			if (++two_second_count >= 100)
			{
				two_second_count = 0;

				last_time = this_time;
				this_time = clock();
			}

			/* Finish off snapshot if necessary. */
			if (snapshot_in_progress)
			{
				snapshot_in_progress = FALSE;
				complete_snapshot();
			}
		}
	}
}
