/********************************************/
/*** Copyright 1991, 1992 Alexander Pruss ***/
/***        Some rights reserved          ***/
/********************************************/

/** SEE LICENSE.DOC for LICENSING INFORMATION **/

/*
 *
 * Encode a file using the OBFuscated methods
 *
 */

/*
 * Revision:
 *  0.00  -- 1-2 encoding only
 *  0.10  -- 3-4 encoding added
 *  0.10b -- converted to stdio, extractor added
 *  0.10c -- -n option added
 *  0.10d -- W initiator changed to < in 3->4 method
 *        -- a possible major bug fixed, which might have caused
 *           problems on systems with large prefetch queues
 *        -- bug in Makefile fixed
 *
 */

/**
 ** export version 0.10d
 **/

#include <stdio.h>
#include <string.h>
#include <dir.h>
#include <dos.h>
#include <sys/stat.h>

#include "getopt.h"
#include "header.h"

#define B_SIG  "S&$ig(Ap"
#define B_SLEN 8

#define SUUSTYLE "uu00"
#define SABSTYLE "ab00"
#define E_LEN 4

#define Fwrite(fd,buf,len) fwrite((buf),1,(len),(fd))
#define Fread(fd,buf,len) fread((buf),1,(len),(fd))

extern char decoded_to_end;
extern char file_size_mod3;

typedef struct
{
  char sighead[B_SLEN];
  char etype[E_LEN];
  char len[8];  /* HEX--long */
  char trailer[40];
} sig;

unsigned long filelen;

char uutrailer[]="<\n";
char ntrailer[]="\n";

char wrapname[70];
int newwrapname=0;

char topwrap[]=
"This file has been encoded with the COMT encoder.\n"
"This encoding renders it a text file but makes it still possible to run the\n"
"program in its text file format.\n\n"
"Simply cut out the portion of the file between the --BEGIN and --END lines,\n"
"save as `%s', making sure you have left no blank lines at the top,\n"
"and that the first line of the file begins with `ENC.COM.'\n"
"Make sure you save as a DOS text file (you can check by TYPE'ing the file,\n"
"and seeing if all the text is nicely aligned at the left hand margin, and\n"
"there are no graphics or control characters).  Then run %s.\n\n"
"--BEGIN clear text of %s via COMT--CUT HERE--REMOVE THIS LINE TOO!\n";

char botwrap[]=
"--END clear text of %s via COMT--CUT HERE.\n";

enum styles
{
  UUSTYLE, ABSTYLE, AUTO
} method=AUTO;

enum logic
{
  FALSE=0, TRUE
} wrap=FALSE;
enum logic extractmode=FALSE;

sig mainsig;

unsigned uuencode_block(char *outblock, char *inblock, unsigned len);
unsigned nencode_block(char *outblock, char *inblock, unsigned len);

#define IBSIZE (3*2048) /* must be divisible by 3 */
#define OBSIZE (3*2*2250)
int n_inbuf;
char inbuf[IBSIZE];
char outbuf[OBSIZE];


enum styles decide(unsigned long l)
{
  long uu,ab;
  if(method!=AUTO) return method;
  uu=l*4/3+sizeof(uheader)-1+sizeof(uutrailer)-1;
  ab=l*2+sizeof(nheader)-1+sizeof(ntrailer)-1;
  if(uu<ab) return UUSTYLE; else return ABSTYLE;
}

int encode(char *outblock, char *inblock, unsigned len)
{
  switch(method)
  {
    case ABSTYLE: return nencode_block(outblock, inblock, len);
    case UUSTYLE: len=(len+2)/3;
       /* this maps 0->0 and handles last block right */
                  return uuencode_block(outblock, inblock, len);
  }
}


void makesig(unsigned long length)
{
  char num[9];
  strncpy(mainsig.sighead,B_SIG,B_SLEN);
  switch(method)
  {
    case ABSTYLE: strncpy(mainsig.etype,SABSTYLE,E_LEN);
      strncpy(mainsig.trailer,ntrailer,sizeof(ntrailer)-1); break;
    case UUSTYLE: strncpy(mainsig.etype,SUUSTYLE,E_LEN);
      strncpy(mainsig.trailer,uutrailer,sizeof(uutrailer)-1); break;
  }
  sprintf(num,"%.8lX",length);
  strncpy(mainsig.len,num,8);
}

int skipsig(FILE *in)
{
  int found=0;
  while(NULL!=fgets(inbuf,IBSIZE,in))
    if(!strncmp(inbuf,B_SIG,B_SLEN))
    {
      found=1;
      break;
    }
  if(!found)
  {
    fputs("Cannot find the COMT signature.\n",stderr);
    return -1;
  }
  memcpy(&mainsig,inbuf,sizeof mainsig);
  if(!strncmp(mainsig.etype,SUUSTYLE,E_LEN))
    method=UUSTYLE;
  else if(!strncmp(mainsig.etype,SABSTYLE,E_LEN))
    method=ABSTYLE;
  else
  {
    strncpy(outbuf,mainsig.etype,E_LEN);
    outbuf[E_LEN]=0;
    fprintf(stderr,"I do not know `%s' encoding.  Probably this version is too old.\n"
       ,outbuf);
    return -1;
  }
  sscanf(mainsig.len,"%8lX",&filelen);
  return 0;
}



void sendhead(FILE *out, unsigned long len, char *name)
{
  if(wrap)
  {
    if(newwrapname) sprintf(outbuf,topwrap,wrapname,wrapname,wrapname);
     else sprintf(outbuf,topwrap,name,name,name);
    Fwrite(out,outbuf,strlen(outbuf));
  }
  switch(method)
  {
    case UUSTYLE: Fwrite(out,uheader,sizeof uheader-1); break;
    case ABSTYLE: Fwrite(out,nheader,sizeof nheader-1); break;
  }
  makesig(len);
  Fwrite(out,&mainsig,strlen((char *)&mainsig));
}

#define SSTR(f,x)  Fwrite((f),(x),sizeof(x)-1)

void sendtail(FILE *out, char *name)
{
  static char uutail[]="\nz!#\n";  /* # in 4th position */
  switch(method)
  {                  /*  01234 */
    case UUSTYLE: uutail[3]='0'+(n_inbuf%3); /* How many of last 3 are real */
                  SSTR(out,uutail); break;
    case ABSTYLE: SSTR(out,"\nZ\n"); break;
  }
  if(wrap)
  {
    sprintf(outbuf,botwrap,newwrapname?wrapname:name);
    Fwrite(out,outbuf,strlen(outbuf));
  }
}

unsigned int fill_inbuf(FILE *f)
{
  n_inbuf=Fread(f,inbuf,IBSIZE);
  if(n_inbuf<0) return 0; else return n_inbuf;
}

unsigned int fill_outbuf(FILE *f)
{
  n_inbuf=Fread(f,outbuf,OBSIZE-256);
  if(n_inbuf<0) return 0;
  outbuf[n_inbuf]=0;
  fgets(outbuf+n_inbuf,256,f);  /* make sure we end on an EOL */
  n_inbuf+=strlen(outbuf+n_inbuf);
  return n_inbuf;
}


void usage(int rval)
{
  static char umsg[]=
"comt [-1 -3 -w -nname|-x|-h] name[.COM] [outputname]\n"
"Options:\n"
" -x  Extract from file\n"
" -1  Use 1->2 encoding\n"
" -3  Use 3->4 encoding\n"
" -w  Wrap in a text wrapper\n"
" -nname  Set name to wrap as\n"
" -h  This help information\n\n"
"By default the input file is overwritten.  If -w is specified, the\n"
"output file name.ASC is by default created, unless the outputname\n"
"is specified.  Default extension on outputname is .COM except in -w\n"
"mode where .ASC is used.\n\n"
"COMT ver. 0.10d is Copyright 1991,1992 Alexander Pruss.  Some rights reserved.\n"
"  All use is permitted limited only by the law of the land and natural law.\n"
"  I hereby specify that only the exact version I produced belongs to me.\n"
"  Any modification, however trivial, renders my rights to this null and void.\n"
"  Thus, if you change this copyright message this program shall no longer\n"
"  belong to me.  I permit any such modification, and I permit any removal\n"
"  of credit to myself for this programme.  NO WARRANTY OF ANY KIND IS IMPLIED.\n"
"  Interpretation of natural law is left to the user.\n";
  Fwrite(stdout,umsg,sizeof umsg-1);
  exit(rval);
}

void notfound(char *name)
{
  Fwrite(stderr,outbuf,sprintf(outbuf,"I cannot find `%s'.\n",name));
  exit(3);
}

void err(char *d)
{
  Fwrite(stderr,outbuf,sprintf(outbuf,"Error %s.\n",d));
  exit(4);
}

int have_ext(char *path)
{
  char drive[3];
  char dir[80];
  char name[9];
  char ext[5];
  return (EXTENSION&fnsplit(path,drive,dir,name,ext))?1:0;
}


main(int argc, char **argv)
{
  FILE *in;
  FILE *out;
  char tmp[15]="ARPXXXXX";
  char temp[86];
  char inname[84];
  char outname[84];
  int in_keep=0;
  struct stat sb;
  int option;
  unsigned long totdec;
  unsigned int curdec;

  char drive[3];
  char dir[80];
  char name[9];
  char ext[5];
  int flags;

  opterr=1;

  if(argc==1) usage(2);
  while (-1!=(option=getopt(argc,argv,"13whxn:?")))
    switch (option)
    {
      case '1': method=ABSTYLE; break;
      case '3': method=UUSTYLE; break;
      case 'w': wrap=1; break;
      case 'h': usage(0); break;
      case 'x': extractmode=1; break;
      case '?': usage(0); break;
      case 'n': strcpy(wrapname,optarg); newwrapname=1; break;
    }

  flags=fnsplit(argv[optind],drive,dir,name,ext);
  if(!(EXTENSION & flags))
  {
    sprintf(inname,"%s.COM",argv[optind]);
    if(extractmode) strcpy(ext,".COM");
  }
  else strcpy(inname,argv[optind]);
  if(access(inname,0)) notfound(inname);
  if(argc>optind+1)
  {
   if(!have_ext(argv[optind+1]))
    sprintf(outname,wrap?"%s.ASC":"%s.COM",argv[optind+1]);
     else strcpy(outname,argv[optind+1]);
   if(!stricmp(outname,inname)) usage(2);
   in_keep=1;
  }
  else if(extractmode && stricmp(ext,".COM"))
  {
    /* extracting a non-COM file to by default a .COM file */
    fnmerge(outname,drive,dir,name,".COM");
    in_keep=1;
  }
  else if(!wrap)
  {
    tmpnam(tmp);
    sprintf(temp,"%s%s%s",drive,dir,tmp);
    if(-1==rename(inname,temp)) err("renaming input file");
    strcpy(outname,inname);
  }
  else
  {
    fnmerge(outname,drive,dir,name,".ASC");
    if(!stricmp(outname,inname)) usage(2);
    in_keep=1;
  }
  out=fopen(outname,extractmode?"wb":"w");
  if(out==NULL) err("creating output file");
  in=fopen(in_keep?inname:temp,extractmode?"r":"rb");
  if(in==NULL || extractmode && skipsig(in))
  {
    if(in_keep)
    {
      fclose(out);
      unlink(outname);
    }
    else
    {
      fclose(out);
      unlink(outname);
      rename(temp,inname);
    }
    err("reading input file");
  }

  if(!extractmode)
  {
    fstat(fileno(in),&sb);

    method=decide(sb.st_size);
    sendhead(out,sb.st_size,inname);

    encode(NULL,NULL,0);
    while(fill_inbuf(in))
    {
      Fwrite(out,outbuf,encode(outbuf,inbuf,n_inbuf));
      if(n_inbuf<IBSIZE) break;
    }
    fclose(in);
    sendtail(out,inname);
    fclose(out);
    if(!in_keep) unlink(temp);
    return 0;
  }
  else
  { /* extract */
    totdec=0;
    while(fill_outbuf(in))
    {
      if(method==UUSTYLE)
        curdec=uudecode_block(outbuf,n_inbuf);
      else
        curdec=ndecode_block(outbuf,n_inbuf);
      totdec+=curdec;
      if(totdec>filelen) curdec-=(unsigned int)(totdec-filelen);
      Fwrite(out,outbuf,curdec);
      if(decoded_to_end || filelen+20<totdec) /* prevent big problems */
      {
        if(method==UUSTYLE)
          totdec-=(3-file_size_mod3)%3;
        break;
      }
    }
    fclose(in);
    fclose(out);
    if(!in_keep) unlink(temp);
    if(filelen!=totdec)
    {
      fprintf(stderr,
        "Decoded length of %ld does not match header length of %ld.",
        totdec,filelen);
      return 1;
    }
    return 0;
  }

}
