/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 *
 *      Vector and matrix manipulation routines.
 *
 *      See readme.txt for copyright information.
 */

#include "3dview.h"

OBJECT3D * The3Dview::AddToWorld(int num, VTX first[])
{
	if (num<=0 || num>256) return 0;
	OBJECT3D *p = new OBJECT3D;
	if (world==0) world = end = p;
	p->num_vertex = num;
	p->vertex = new VTX[num];
	p->next = 0;
	memcpy(p->vertex, first, sizeof(VTX)*num);
	return end = p;
}

OBJECT3D * The3Dview::AddToWorld(int num, VTX first, ...)
{
	return AddToWorld(num, &first);
}

void The3Dview::DeleteWorld(void)
{
	OBJECT3D *p = world, *pp;
	while(p)
	{
		pp = p->next;
		delete p;
		p = pp;
	}
}

float dot_product_f(float x1, float y1, float z1, float x2, float y2, float z2)
{
   return (x1 * x2) + (y1 * y2) + (z1 * z2);
}

void apply_matrix_f(MATRIX_f *m, float x, float y, float z, float *xout, float *yout, float *zout)
{
    *xout=x*m->v[0][0]+y*m->v[0][1]+z*m->v[0][2]+m->t[0];
    *yout=x*m->v[1][0]+y*m->v[1][1]+z*m->v[1][2]+m->t[1];
    *zout=x*m->v[2][0]+y*m->v[2][1]+z*m->v[2][2]+m->t[2];
}

void persp_project_f(float x, float y, float z, float *xout, float *yout)
{
   float z1 = 1.0 / z;
   *xout = ((x * z1) * _persp_xscale_f) + _persp_xoffset_f;
   *yout = ((y * z1) * _persp_yscale_f) + _persp_yoffset_f;
}

MATRIX_f identity_matrix_f = 
{
   {
      /* 3x3 identity */
      { 1.0, 0.0, 0.0 },
      { 0.0, 1.0, 0.0 },
      { 0.0, 0.0, 1.0 },
   },

   /* zero translation */
   { 0.0, 0.0, 0.0 }
};


/* get_translation_matrix_f:
 *  Floating point version of get_translation_matrix().
 */
void get_translation_matrix_f(MATRIX_f *m, float x, float y, float z)
{
   *m = identity_matrix_f;

   m->t[0] = x;
   m->t[1] = y;
   m->t[2] = z;
}

/* get_scaling_matrix_f:
 *  Floating point version of get_scaling_matrix().
 */
void get_scaling_matrix_f(MATRIX_f *m, float x, float y, float z)
{
   *m = identity_matrix_f;

   m->v[0][0] = x;
   m->v[1][1] = y;
   m->v[2][2] = z;
}



/* get_x_rotate_matrix_f:
 *  Floating point version of get_x_rotate_matrix().
 */
void get_x_rotate_matrix_f(MATRIX_f *m, float r)
{
   float c = floatcos(r);
   float s = floatsin(r);

   *m = identity_matrix_f;

   m->v[1][1] = c;
   m->v[1][2] = -s;

   m->v[2][1] = s;
   m->v[2][2] = c;
}


/* get_y_rotate_matrix_f:
 *  Floating point version of get_y_rotate_matrix().
 */
void get_y_rotate_matrix_f(MATRIX_f *m, float r)
{
   float c = floatcos(r);
   float s = floatsin(r);

   *m = identity_matrix_f;

   m->v[0][0] = c;
   m->v[0][2] = s;

   m->v[2][0] = -s;
   m->v[2][2] = c;
}


/* get_z_rotate_matrix_f:
 *  Floating point version of get_z_rotate_matrix().
 */
void get_z_rotate_matrix_f(MATRIX_f *m, float r)
{
   float c = floatcos(r);
   float s = floatsin(r);

   *m = identity_matrix_f;

   m->v[0][0] = c;
   m->v[0][1] = -s;

   m->v[1][0] = s;
   m->v[1][1] = c;
}


/* magical formulae for constructing rotation matrices */
#define MAKE_ROTATION_f(x, y, z)                \
   float sin_x = floatsin(x);                   \
   float cos_x = floatcos(x);                   \
						\
   float sin_y = floatsin(y);                   \
   float cos_y = floatcos(y);                   \
						\
   float sin_z = floatsin(z);                   \
   float cos_z = floatcos(z);                   \
						\
   float sinx_siny = sin_x * sin_y;             \
   float cosx_siny = cos_x * sin_y;

#define R00_f (cos_y * cos_z)
#define R10_f ((sinx_siny * cos_z) - (cos_x * sin_z))
#define R20_f ((cosx_siny * cos_z) + (sin_x * sin_z))

#define R01_f (cos_y * sin_z)
#define R11_f ((sinx_siny * sin_z) + (cos_x * cos_z))
#define R21_f ((cosx_siny * sin_z) - (sin_x * cos_z))

#define R02_f (-sin_y)
#define R12_f (sin_x * cos_y)
#define R22_f (cos_x * cos_y)



/* get_rotation_matrix:
 *  Constructs a 3d transformation matrix, which will rotate points around
 *  all three axis by the specified amounts (given in the Allegro fixed 
 *  point, 256 degrees to a circle format).
 *  Floating point version of get_rotation_matrix().
 */
void get_rotation_matrix_f(MATRIX_f *m, float x, float y, float z)
{
   MAKE_ROTATION_f(x, y, z);

   m->v[0][0] = R00_f;
   m->v[0][1] = R01_f;
   m->v[0][2] = R02_f;

   m->v[1][0] = R10_f;
   m->v[1][1] = R11_f;
   m->v[1][2] = R12_f;

   m->v[2][0] = R20_f;
   m->v[2][1] = R21_f;
   m->v[2][2] = R22_f;

   m->t[0] = m->t[1] = m->t[2] = 0;
}


/* get_align_matrix:
 *  Aligns a matrix along an arbitrary coordinate system.
 *  Floating point version of get_align_matrix().
 */
void get_align_matrix_f(MATRIX_f *m, float xfront, float yfront, float zfront, float xup, float yup, float zup)
{
   float xright, yright, zright;

   normalize_vector_f(&xfront, &yfront, &zfront);
   normalize_vector_f(&xup, &yup, &zup);

   cross_product_f(xfront, yfront, zfront, xup, yup, zup, &xright, &yright, &zright);
   cross_product_f(xright, yright, zright, xfront, yfront, zfront, &xup, &yup, &zup);

   m->v[0][0] = xfront;
   m->v[0][1] = yfront;
   m->v[0][2] = zfront;

   m->v[1][0] = xup;
   m->v[1][1] = yup;
   m->v[1][2] = zup;

   m->v[2][0] = xright;
   m->v[2][1] = yright;
   m->v[2][2] = zright;
}



/* get_vector_rotation_matrix:
 *  Constructs a 3d transformation matrix, which will rotate points around
 *  the specified x,y,z vector by the specified angle (given in the Allegro 
 *  fixed point, 256 degrees to a circle format), in a clockwise direction.
 *  Floating point version of get_vector_rotation_matrix().
 */
void get_vector_rotation_matrix_f(MATRIX_f *m, float x, float y, float z, float a)
{
   float c = floatcos(a);
   float s = floatsin(a);
   float cc = 1 - c;

   normalize_vector_f(&x, &y, &z);

   m->v[0][0] = (cc * x * x) + c;
   m->v[0][1] = (cc * x * y) + (z * s);
   m->v[0][2] = (cc * x * z) - (y * s);

   m->v[1][0] = (cc * x * y) - (z * s);
   m->v[1][1] = (cc * y * y) + c;
   m->v[1][2] = (cc * z * y) + (x * s);

   m->v[2][0] = (cc * x * z) + (y * s);
   m->v[2][1] = (cc * y * z) - (x * s);
   m->v[2][2] = (cc * z * z) + c;

   m->t[0] = m->t[1] = m->t[2] = 0;
}



/* get_transformation_matrix:
 *  Constructs a 3d transformation matrix, which will rotate points around
 *  all three axis by the specified amounts (given in the Allegro fixed 
 *  point, 256 degrees to a circle format), scale the result by the
 *  specified amount (itofix(1) for no change of scale), and then translate
 *  to the requested x, y, z position.
 *  Floating point version of get_transformation_matrix().
 */
void get_transformation_matrix_f(MATRIX_f *m, float scale, float xrot, float yrot, float zrot, float x, float y, float z)
{
   MAKE_ROTATION_f(xrot, yrot, zrot);

   m->v[0][0] = R00_f * scale;
   m->v[0][1] = R01_f * scale;
   m->v[0][2] = R02_f * scale;

   m->v[1][0] = R10_f * scale;
   m->v[1][1] = R11_f * scale;
   m->v[1][2] = R12_f * scale;

   m->v[2][0] = R20_f * scale;
   m->v[2][1] = R21_f * scale;
   m->v[2][2] = R22_f * scale;

   m->t[0] = x;
   m->t[1] = y;
   m->t[2] = z;
}

/* get_camera_matrix: 
 *  Constructs a camera matrix for translating world-space objects into
 *  a normalised view space, ready for the perspective projection. The
 *  x, y, and z parameters specify the camera position, xfront, yfront,
 *  and zfront is an 'in front' vector specifying which way the camera
 *  is facing (this can be any length: normalisation is not required),
 *  and xup, yup, and zup is the 'up' direction vector. Up is really only
 *  a 1.5d vector, since the front vector only leaves one degree of freedom
 *  for which way up to put the image, but it is simplest to specify it
 *  as a full 3d direction even though a lot of the information in it is
 *  discarded. The fov parameter specifies the field of view (ie. width
 *  of the camera focus) in fixed point, 256 degrees to the circle format.
 *  For typical projections, a field of view in the region 32-48 will work
 *  well. Finally, the aspect ratio is used to scale the Y dimensions of
 *  the image relative to the X axis, so you can use it to correct for
 *  the proportions of the output image (set it to 1 for no scaling).
 *  Floating point version of get_camera_matrix().
 */
void get_camera_matrix_f(MATRIX_f *m, float x, float y, float z, float xfront, float yfront, float zfront, float xup, float yup, float zup, float fov, float aspect)
{
   MATRIX_f camera, scale;
   float xside, yside, zside, width, d;

   /* make 'in-front' into a unit vector, and negate it */
   normalize_vector_f(&xfront, &yfront, &zfront);
   xfront = -xfront;
   yfront = -yfront;
   zfront = -zfront;

   /* make sure 'up' is at right angles to 'in-front', and normalize */
   d = dot_product_f(xup, yup, zup, xfront, yfront, zfront);
   xup -= d * xfront; 
   yup -= d * yfront; 
   zup -= d * zfront;
   normalize_vector_f(&xup, &yup, &zup);

   /* calculate the 'sideways' vector */
   cross_product_f(xup, yup, zup, xfront, yfront, zfront, &xside, &yside, &zside);

   /* set matrix rotation parameters */
   camera.v[0][0] = xside; 
   camera.v[0][1] = yside;
   camera.v[0][2] = zside;

   camera.v[1][0] = xup; 
   camera.v[1][1] = yup;
   camera.v[1][2] = zup;

   camera.v[2][0] = xfront; 
   camera.v[2][1] = yfront;
   camera.v[2][2] = zfront;

   /* set matrix translation parameters */
   camera.t[0] = -(x*xside  + y*yside  + z*zside);
   camera.t[1] = -(x*xup    + y*yup    + z*zup);
   camera.t[2] = -(x*xfront + y*yfront + z*zfront);

   /* construct a scaling matrix to deal with aspect ratio and FOV */
   width = floattan(64.0 - fov/2);
   get_scaling_matrix_f(&scale, width, -aspect*width*4/3, -1.0);

   /* combine the camera and scaling matrices */
   matrix_mul_f(&camera, &scale, m);
}



/* qtranslate_matrix:
 *  Adds a position offset to an existing matrix.
 *  Floating point version of qtranslate_matrix().
 */
void qtranslate_matrix_f(MATRIX_f *m, float x, float y, float z)
{
   m->t[0] += x;
   m->t[1] += y;
   m->t[2] += z;
}


/* qscale_matrix:
 *  Adds a scaling factor to an existing matrix.
 *  Floating point version of qscale_matrix().
 */
void qscale_matrix_f(MATRIX_f *m, float scale)
{
   int i, j;

   for (i=0; i<3; i++)
      for (j=0; j<3; j++)
	 m->v[i][j] *= scale;
}


/* matrix_mul:
 *  Multiplies two matrices, storing the result in out (this must be
 *  different from the two input matrices). The resulting matrix will
 *  have the same effect as the combination of m1 and m2, ie. when
 *  applied to a vector v, (v * out) = ((v * m1) * m2). Any number of
 *  transformations can be concatenated in this way.
 *  Floating point version of matrix_mul().
 */
void matrix_mul_f(MATRIX_f *m1, MATRIX_f *m2, MATRIX_f *out)
{
   MATRIX_f temp;
   int i, j;

   if (m1 == out) {
      temp = *m1;
      m1 = &temp;
   }
   else if (m2 == out) {
      temp = *m2;
      m2 = &temp;
   }

   for (i=0; i<3; i++) {
      for (j=0; j<3; j++) {
	 out->v[i][j] = (m1->v[0][j] * m2->v[i][0]) +
			(m1->v[1][j] * m2->v[i][1]) +
			(m1->v[2][j] * m2->v[i][2]);
      }

      out->t[i] = (m1->t[0] * m2->v[i][0]) +
		  (m1->t[1] * m2->v[i][1]) +
		  (m1->t[2] * m2->v[i][2]) +
		  m2->t[i];
   }
}



/* vector_length: 
 *  Computes the length of a vector, using the son of the squaw...
 *  Floating point version of vector_length().
 */
float vector_length_f(float x, float y, float z)
{
   return sqrt(x*x + y*y + z*z);
}


/* normalize_vector: 
 *  Converts the specified vector to a unit vector, which has the same
 *  orientation but a length of one.
 *  Floating point version of normalize_vector().
 */
void normalize_vector_f(float *x, float *y, float *z)
{
   float length = 1.0 / vector_length_f(*x, *y, *z);

   *x *= length;
   *y *= length;
   *z *= length;
}


/* cross_product:
 *  Calculates the cross product of two vectors.
 *  Floating point version of cross_product().
 */
void cross_product_f(float x1, float y1, float z1, float x2, float y2, float z2, float *xout, float *yout, float *zout)
{
    *xout = (y1 * z2) - (z1 * y2);
    *yout = (z1 * x2) - (x1 * z2);
    *zout = (x1 * y2) - (y1 * x2);
}


/* polygon_z_normal:
 *  Helper function for backface culling: returns the z component of the
 *  normal vector to the polygon formed from the three vertices.
 *  Floating point version of polygon_z_normal().
 */
float polygon_z_normal_f(VTX *v1, VTX *v2, VTX *v3)
{
   return ((v2->x-v1->x) * (v3->y-v2->y)) - ((v3->x-v2->x) * (v2->y-v1->y));
}


/* scaling factors for the perspective projection */
float _persp_xscale_f = 160.0;
float _persp_yscale_f = 100.0;
float _persp_xoffset_f = 160.0;
float _persp_yoffset_f = 100.0;


/* set_projection_viewport:
 *  Sets the viewport used to scale the output of the persp_project() 
 *  function.
 */
void set_projection_viewport(int x, int y, int w, int h)
{
   _persp_xscale_f = w/2;
   _persp_yscale_f = h/2;
   _persp_xoffset_f = x + w/2;
   _persp_yoffset_f = y + h/2;
}

/* polygon3d_f:
 *  Floating point version of polygon3d().
 */
void polygon3d_f(The3Dview *bmp, int type, int vc, VTX *vtx[])
{
	int array[256][2];

	for(int pom=0;pom<vc;pom++)
	{
		array[pom][0] = (int)vtx[pom]->x;
		array[pom][1] = (int)vtx[pom]->y;
	}
	if (type==0) bmp->WindowDrawPolygon(vc, array);
	else bmp->WindowFillPolygon(vc, array);
}

/* triangle3d_f:
 *  Draws a 3d triangle. Dodgy assumption alert! See comments for triangle().
 */
void triangle3d_f(The3Dview *bmp, int type, VTX *v1, VTX *, VTX *)
{
   polygon3d_f(bmp, type, 3, &v1);
}

/* quad3d_f:
 *  Draws a 3d quad. Dodgy assumption alert! See comments for triangle().
 */
void quad3d_f(The3Dview *bmp, int type, VTX *v1, VTX *, VTX *, VTX *)
{
   polygon3d_f(bmp, type, 4, &v1);
}

void wire3d_f(The3Dview *bmp, int x1, int y1, int x2, int y2, int col)
{
   bmp->WindowLine(x1,y1,x2,y2,col);
}

The3Dview::The3Dview(int x, int y, int w, int h):
	Window(x,y,w,h)
{
	assert(image = (char *)malloc(w*h));
	/* set up the viewport for the perspective projection */
	set_projection_viewport(0, 0, w,h);
	world = end = 0;
}


