/******************************************************************************
Fractal Infinity - COMMAND LINE PARSER FUNCTIONS - fr_par.c.
This file Copyright (C) 1999 Graeme Fenwick- see documentation for details.
******************************************************************************/

#include "fr_par.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

int arg_parse(char *arg, int sspace, int *pint, long double *plod,
              char *pcha, char pist[]);

/********** ARG_PARSE: Separates out info from switch-arg, and returns bit-mask
                       corresponding to what program could scan from it */

int arg_parse(char *arg, int sspace, int *pint, long double *plod,
              char *pcha, char pist[])
{
   char   *numpoint;
   int    s_plod, s_pint, s_pist;                      /* scan-success flags */

   if (strlen(arg) < 2 || arg[0] != '/')
      return 0;                             /* didn't match "/<char>" format */
   *pcha = islower(arg[1]) ? toupper(arg[1]) : arg[1];  /* read switch chrct */
   if (strlen(arg) == 2) 
      return LET_PARSE;                                /* return letter only */
   numpoint = (arg[2] == ':') ? arg + 3 : arg + 2;
   s_plod = (sscanf(numpoint, "%Lf", plod) == 1);               /* scan ldb? */
   s_pint = (sscanf(numpoint, "%d", pint) == 1);                /* scan int? */
   s_pist = (strlen(numpoint) < sspace) &&
            (*numpoint != '\0') &&
            (sscanf(numpoint, "%s", pist) == 1);             /* scan string? */
   return (s_plod * LDB_PARSE) + (s_pint * INT_PARSE) +
          (s_pist * STR_PARSE) + LET_PARSE;    /* return success bit-pattern */
}

/********** CLI_PARSE: Checks all CLI args are valid switches, and modifies
                       default values where appropriate */

int cli_parse(int argc, char **argv, int *color, int *xres, int *yres, 
              int *svx, long double *xc, long double *yc, long double *range)
{
   int           flags, round, pint;
   long double   plod;
   char          let;

   for (round = 1; round < argc; ++round) {
      flags = arg_parse(argv[round], -1, &pint, &plod, &let, NULL);
      if (!(flags & LET_PARSE))               /* initial letter must be okay */
         return 0;
      else if (let == 'X' && (flags & LDB_PARSE))
         *xc = plod;                /* assign long double switch value to xc */
      else if (let == 'Y' && (flags & LDB_PARSE))
         *yc = plod;
      else if (let == 'R' && (flags & LDB_PARSE))
         *range = plod;
      else if (let == 'H' && (flags & INT_PARSE) && pint >= 640)
         *xres = pint;                           /* assign int value to xres */
      else if (let == 'V' && (flags & INT_PARSE) && pint >= 400)
         *yres = pint;
      else if (let == 'C' && (flags & INT_PARSE) && pint >= 8)
         *color = pint;
      else if (let == 'E')
         *svx = 1;
      else if (let == '?')
         correct_usage(1);                          /* explicit info request */
      else
         return 0;           /* illegal switch - calling env handles failure */
   }
   return 1;                             /* all CLI args parsed successfully */
}

/********** CORRECT_USAGE: Displays correct program usage- can be called
                           normally (explicitly) or because of wrong cli use */

void correct_usage(int normal)
{
   printf("%s\n\n", (normal) ? "Generates Images of the Mandlebrot Set." :
      "Invalid Switch or Switch value. Usage is as follows :-");
   printf("%s\n%s\n\n%s%s\n%s\n%s%s\n%s\n%s%s\n%s\n%s%s\n%s\n%s%s\n%s\n",
      "FRACTAL [/X[:]xcoord] [/Y[:]ycoord] [/R[:]range] [/H[:]hres]",
      "        [/V[:]vres] [/C[:]depth] [/E] [/?]",
      "  /X         Initial fractal X coordinate (at center).       ",
      "        Default  0",
      "    xcoord   (any value).",
      "  /Y         Initial fractal Y coordinate (at center).       ",
      "                 0",
      "    ycoord   (any value).",
      "  /R         Range (width) of fractal area displayed.        ",
      "                 5",
      "    range    (any value).",
      "  /H         Horizontal Screen Resolution.                   ",
      "               640",
      "    hres     (supported resolutions only- see documentation).",
      "  /V         Vertical Screen Resolution.                     ",
      "               480",
      "    vres     (supported resolutions only- see documentation).");
   printf("%s%s\n%s\n%s\n%s\n\n%s\n%s\n",
      "  /C         Specifies Color Depth (in bits).                ",
      "                16",
      "    depth    (24, 16, 15, or 8- see documentation).",
      "  /E         Auto-Generate/ Save/ Exit.                               ",
      "  /?         Usage Information (this screen).                         ",
      "This is a brief summary of the startup parameters only. Full",
      "documentation is available in the accompanying text files.");
   exit(normal ? EXIT_SUCCESS : EXIT_FAILURE);
}


