#!/bin/sh
# the next line restarts using wish \
exec wish "$0" "$@"

# Xforge: A Small X-Windows interface to 'gforge' landscape generator

# Requires gforge-1.2a, plus Tcl 7.4 and Tk 4.0 [or later versions (?)]
# get gforge from http://www-leland.stanford.edu/~beale/land/gforge/

# The Tcl/Tk packages are available from various web sites like
#   ftp://ftp.smli.com/
#   ftp://ftp.aud.alcatel.com/tcl/
#   ftp://ftp.cs.berkeley.edu/ucb/tcl/

# Xforge is Copyright 1995 John P Beale 9/18/95 5:35 am
# released under GPL Gnu Public Licence version 2
# ----------------------------------------------------------------

set font -*-helvetica-*-r-*-*-*-100-*-*-*-*-*-*
set font2 -*-helvetica-*-r-*-*-*-120-*-*-*-*-*-*
set fontout -*-helvetica-*-r-*-*-*-100-*-*-*-*-*-*
# set fontout -*-fixed-medium-*-*-*-14-*-*-*-*-*-*-*

# generate a small temporary image
if {![file exists output.pgm]} {
    exec gforge -se 35 -me 20 -po 2.0 -limit -1.0 1 -dim 2.8 \
	    -type pg8
}

# Set title and allow window resizing.
wm title . "Xforge 0.02"

# Create a frame for buttons and entry.
frame .top -borderwidth 1
pack .top -side top -fill x

# Create the command buttons.
button .top.quit -text Quit -command exit -width 3 -font $font 
button .top.save -text Save -command Save -width 3 -font $font
set butrun [button .top.run -text "View" -width 3 -command Run \
          -font $font ]
set toplab [label .top.title -text "Gforge Landscapes" -width 25 \
           -font $font2 ]
pack .top.quit .top.save .top.run .top.title  -side right

#-------- various entry boxes
label .top.l -text Meshsize -padx 2 -font $font
entry .top.e1 -width 4 -bd 1 -relief ridge -textvariable sf(mesh) \
	-font $font

button .top.l2 -text Seed -padx 2 -pady 0 -font $font -bd 1 \
        -command {incr sf(seed); Run}
entry .top.e2 -width 5 -bd 1 -relief ridge -textvariable sf(seed) \
	-font $font

pack .top.l .top.e1 .top.l2 .top.e2  -side left
 

# Set up keybinding equivalents to the buttons
bind Entry <Return> Run 
bind Entry <Control-c> Stop
# focus .top.cmd

# Create a text widget to log the output
frame .t
set log [text .t.log -width 60 -height 6 -wrap word \
	-borderwidth 2 -relief raised -setgrid true\
	-yscrollcommand {.t.scroll set} -font $fontout]
scrollbar .t.scroll -command {.t.log yview}
pack .t.scroll -side right -fill y
pack .t.log -side left -fill both -expand true
pack .t -side top -fill both -expand true


# Init some global variables
set running 0         ;# not yet running a program
set disp_size 300     ;# pixel dimension of image window
set iformat pg8       ;# kind that image photo widget can load
set sf(seed) 0        ;# intial random seed value

set imblank [image create photo -file output.pgm -palette 240 ]

# ------------- Create labelled scale widgets
# sc is an array of lists holding scalebar varible parameters, settings
# sc_minval sc_maxval ivar_name txtlabel scalefac fvar_name frame Fini Iini
#   0         1         2         3        4        5         6    7    8

set sc(mesh)  {1 100 mesh "Meshsize" 5 meshf lf 100 20}
set sc(dim)   {1 500 dim "Dim" 0.01 dimf rf 2.5 250}
# set sc(seed)  {0 100 seed "Seed" 1 seedf lf 23 23 }
set sc(pow)   {1 300 power "Pow" 0.01 powerf lf 1.2 120 }
set sc(lowlim) {-100 0 lowlim "LowLim" 0.01 lowlimf rf -1.0 -100}
set sc(hpff)  {0 100 hpf "HPfilter F" 0.01 xx lf 0.0 0 }
set sc(hpfo)  {0 100 hpf "HPfilter O" 0.5 xx rf 1.0 2 }
set sc(bpff)  {1 1000 xx "BPfilter F" 0.001 xx lf 0.2 200}
set sc(bpfq)  {1 1000 bpf "BPfilter Q" 0.2 bpfq rf 0.5 2}
set sc(crat1) {1 100 xx "NumCraters" 0.1 xx lf 1.0 10}
set sc(crat2) {1 100 xx "CraterDepth" 0.1 xx rf 1.0 10}
set sc(peakx) {0 100 peakx "PeakX (%)" 0.01 xsf lf 0.5 50}
set sc(peaky) {0 100 peaky "PeakY (%)" 0.01 ysf rf 0.5 50}

set scalevars "mesh dim pow lowlim hpff hpfo bpff \
               bpfq crat1 crat2 peakx peaky"

# set initial values of array vars
proc SetVarArray { wlist } {
  global sc sf si
  foreach a $wlist {
    set si($a) [lindex $sc($a) 8]
    set sf($a) [lindex $sc($a) 7]
  }
}

# Update the scale indicator value (int or float)
proc UpdateVal { var val }  {  
  global autoup sc si sf
    set scalefac [lindex $sc($var) 4]
    set sf($var) [expr $si($var) * $scalefac ]
}

# set variables to their initial values
SetVarArray $scalevars

# -------------generate all the scale bars -------------

proc PackScales { wlist } {
  global sc si sf svar1 svar2 svar3 font
  foreach a $wlist {
    set pf sb.[lindex $sc($a) 6]    ;# which toplevel frame: l or r
    frame .$pf.$a -bd 0
    set svar1($a) [scale .$pf.$a.scl -from [lindex $sc($a) 0] \
            -to [lindex $sc($a) 1] -font $font \
	    -length 100 -width 14 -variable si($a) \
            -orient horizontal -bd 1 -showvalue false \
	    -command "UpdateVal $a" -bd 2 ]
    set svar2($a) [button .$pf.$a.txt -text [lindex $sc($a) 3] \
            -bd 0 -width 8 -font $font ]
    set svar3($a) [ entry .$pf.$a.val -textvariable sf($a) -bd 1 \
           -relief ridge -width 5 -font $font ]
    pack .$pf.$a.txt .$pf.$a.val -side left
    pack .$pf.$a.scl -side left -fill x -expand true
    pack .$pf.$a -side top -fill x -expand true 
  }
}

frame .sb -relief ridge -bd 2           ;# main scalebar frame
frame .sb.lf         ;# scalebar left side
frame .sb.rf         ;# scalebar right side

PackScales $scalevars
pack .sb.lf .sb.rf -side left -fill x -expand true
pack .sb -side top -fill x -expand true

# commands to enable and disable ('grey out') scalebars
proc Disable { a } {
  global sb sc svar1 svar2 svar3
  $svar1($a) config -state disabled
  $svar2($a) config -state disabled
  $svar3($a) config -state disabled
}

proc Enable { a } {
  global sb sc svar1 svar2 svar3
  $svar1($a) config -state normal
  $svar2($a) config -state normal
  $svar3($a) config -state normal
}

proc UpdateEnabled { a blist } {
  global sb 
  if $sb($a) {set cmd Enable} else {set cmd Disable}
      foreach b $blist {
	$cmd $b
      }
}

proc AutoUp {} {
    global autoup running
#    bind Scale <ButtonRelease-1> { }
    if {$autoup && !$running} {
	Run
    }
}
Disable crat1
Disable crat2
Disable peakx
Disable peaky

# automatically update image when mouse leaves scale bar... NOT!
# bind Scale <ButtonRelease-1> {AutoUp}

# ------------ a line of buttons at mid-screen ------------
frame .bb -relief ridge -bd 2
checkbutton .bb.a1 -text "Set_PeakXY" -variable sb(peakset) -font $font \
	-command {UpdateEnabled peakset {peakx peaky} }
checkbutton .bb.a2 -text "Use Filters" -variable sb(filters) -font $font \
	-command {UpdateEnabled filters {hpff hpfo bpff bpfq} }
checkbutton .bb.a3 -text "Add Craters" -variable sb(craters) -font $font \
	-command {UpdateEnabled craters {crat1 crat2} }
checkbutton .bb.a4 -text "View Tiled" -variable sb(tiled) -command { \
        . config -cursor watch; View; update idletasks; . config -cursor \
        arrow }  -font $font
pack .bb.a1 .bb.a2 .bb.a3 -side right

set sb(filters) 1   ;# default is to use filters

label .bb.l3 -text Viewsize -padx 2 -font $font
entry .bb.e3 -width 4 -bd 1 -relief ridge -textvariable disp_size
pack .bb.l3 .bb.e3 .bb.a4 -side left
pack .bb -side top -fill x -expand true

# ----- create the button which actually displays the image ----
set im [image create photo -file output.pgm -palette 240 ]
$imblank configure -width $disp_size -height $disp_size 
$imblank copy $im -to 0 0 $disp_size $disp_size
set buts [button .butscene -image $imblank -command Run]
pack .butscene -side bottom -fill y -expand true

# Run gforge, read text output onto screen
proc Run  {} {
	global running input log butrun iformat toplab \
            sc sf si sb font font2
        . config -cursor watch  ;# indicate we're running
        set running 1
	$butrun config -text Wait -command Stop -font $font
        $toplab config -font $font2 -text "   working...   "
        update idletasks                            ;# repaint display

	set arglist {gforge} ;# start of command line
	if {$sb(peakset)} {
          lappend arglist -peak $sf(peakx) $sf(peaky)
	}
	lappend arglist -se $sf(seed)
        lappend arglist -me $sf(mesh)
	lappend arglist -po $sf(pow)
	lappend arglist -limit $sf(lowlim) 1
	lappend arglist -dim $sf(dim)
	if { $sb(craters) } {
	    lappend arglist -craters $sf(crat1) $sf(crat2)
	}
	if { $sb(filters) } {
	    lappend arglist -hpf $sf(hpff) $sf(hpfo)
	    lappend arglist -bpf $sf(bpff) $sf(bpfq)
        }
	lappend arglist -type $iformat
        if [catch {open "|$arglist |& cat"} input] {
		$log insert end $arglist\n
	} else {
		fileevent $input readable Log
		$log insert end $arglist\n
	}
}

# ----------------------  View gforge image (output.pgm) -------------
proc View {} {
    global buts imblank disp_size sc sf sb log
    $imblank blank              ;# set image to blank
    $buts config -image $imblank ;# erase old image
    set im [image create photo -file output.pgm -palette 240 ]
    set zfac [expr $disp_size / $sf(mesh) ]
    set cpstring { -zoom $zfac }
    if { $zfac > 1.0 } {
       if { $sb(tiled) } {
	    $imblank configure -width $disp_size -height $disp_size 
	    $imblank copy $im -to 0 0 $disp_size $disp_size
	} else {
	    set ss [expr $zfac * $sf(mesh)]
	    $imblank configure -width $ss -height $ss
            $imblank copy $im -zoom $zfac ;# scaled version of im
	}
        $buts config -image $imblank ;# scaled, untiled    
    } else {
      $buts config -image $im      ;# display unscaled, untiled
    }  
}

# Read and log output from the program
proc Log {} {
	global input log
	if [eof $input] {
	    View             
	    Stop
	} else {
		gets $input line
#		$log insert end $line\n ;# record prog output
		$log see end
	}
}

proc Save {} {
    global iformat
    set iformat tga
    Run
    set iformat pg8
}

# Stop the program and fix up the button
proc Stop {} {
	global input butrun running toplab font
        set running 0
#        bind Scale <ButtonRelease-1> {AutoUp}
	catch {close $input}
	$butrun config -text "View" -command Run
	$toplab config -text "Gforge Landscapes"
        update idletasks                            ;# repaint display	
	. config -cursor arrow
}


$log insert end "Welcome to Xforge v0.01, a Gforge graphical front-end. "
$log insert end "This is an incomplete, alpha test version of xforge. "
$log insert end "Hit 'View' to see a new image, 'Save' to write 'output.tga'. "
$log insert end "Change resolution with 'Meshsize'.  Have fun... jpb 9/20/95\n"
$log insert end "---------------\n"
# Give an initial image after a short delay
# after 2000 Run

