/*
labels.c

copyright 1988, 1989, 1990, 1991  Robert C. Becker, Lantern Systems

a table of data defining stroke drawn characters for use with the
label() function where label() is passed a pointer to a format string
and a list of variables (very similar to printf() ).

the variables slant and a_r (aspect ratio) are external to the
labeling program.  they define slant of the character and the aspect
ratio of the drawn character, respectively.

Characters below 0x20 are defined as non-printing.  It is optional to
print or not print by calling disp_fns ().  There is no character for
0x7F currently defined (IBM uses delta).  Although characters above 0x7F
are available, none are defined.  A label string with characters from 
0x7F to 0xFF will return unpredictable results.

Character vectors associate one-for-one with pen control codes.
The pen control codes consist of 0, 1, 0xFE and 0xFF.  They decode as

	0	move to the associated coordinate
	1	draw from the current location to the associated coordinate
	0xFE	move to the associated coordinate and end this character
	0xFF	draw to the associated coordinate and end this character

*/


#include "hpgraph.h"


static char copyright[] =  "copyright 1988-1991, Robert C. Becker, Lantern Systems";

static int no_char[] =  { 0,0 };	/* non-printing character */
static unsigned char no_char_p[] = { 0x0ff };	/* pen control for non-printing char */


/* The structure char___deft contains information which describes the basic
font structure.  The basic font design is assumed to be in a cell, and the
cell contains a grid, and some point on the grid is chosen for the origin
of the font.  It is assumed that the characters are left justified in the
cell.  It is not required that characters fill an entire cell.  Fonts in
which the characters do not completely fill the cell (from right to left,
that is) will have widths which are less than x_inc, the origin-to-origin
intercharacter spacing.

fontname: This is the name of the font.  It need not be unique.

char_v: adr of array of pointers to char stroke vectors.

char_p: adr of array of pointer to pen control codes.

last_char: value of the highest defined character in this font.  References
	to characters above this number are ignored.

ascender: highest part of any char in the font (pipe in this case).

descender: height of lowest part of any char in the font.

width: width of char cell (0 -> variable size [not yet implimented]).  This
	is not necessarily the origin-to-origin x-spacing from one char to 
	another (i.e., pitch).

base: this number is base of the char for lorg's 1, 4, & 7.

center: center of char for lorg's 2, 5, & 8.

top: top of char for lorg's 3, 6, & 9.

x_inc: char-to-char std spacing

y_inc: line-to-line std spacing

scale: scale factor to achieve uniform font heights for identical scale factors



standard character font (scale = 1.0)

	--------------------
16	|                  |  <- top
15	|                  |
14	|                  |
13	|                  |  <- ascender (i.e., the top of '|')
12	|  *               |
11	|  *         *     |
10	|  *       *       |
9	|  *     *         |  __ center (8.5)
8	|  *   *           |
7	|  * *   *         |
6	|  *       *       |
5	|  *         *     |
4	|  *           *   |
3	|                  |
2	|                  |
1	|                  |   <- descender (such as the tail in 'y' 
0	|                  |
-1	|                  |   <- base
	--------------------
	 0 1 2 3 4 5 6 7 8 

                         ^ width
*/


/*_______________________________________*/

static int sh[] = {4,12,  2,12,  1,11,  2,10,  4,10,  5,9,  4,8,  1,8, 4,7,  4,3,  7,7,  7,3,  4,5,  7,5 };
static unsigned char sh_p[] = { 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 0xff };
static int sx[] = { 4,12,  2,12,  1,11,  2,10,  4,10,  5,9,  4,8,  1,8, 4,7,  7,3,  7,7,  4,3  };
static unsigned char sx_p[] = { 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 0xff };
static int ex[] = { 4,12,  1,12,  1,8,  4,8,  1,10,  3,10, 4,7,  7,3,  7,7,  4,3  };
static unsigned char ex_p[] = { 0, 1, 1, 1, 0, 1, 0, 1, 0, 0xff };
static int et[] = { 4,12,  1,12,  1,8,  4,8,  1,10,  3,10, 3,7,  7,7,  5,7, 5,3 };
static unsigned char et_p[] = { 0, 1, 1, 1, 0, 1, 0, 1, 0, 0xff };
static int eq[] = { 4,12,  1,12,  1,8,  4,8,  1,10,  3,10, 4,7,  5,8,  6,8,  7,7,  7,5,  6,4,  5,4,  4,5,  4,7,  6,5,  7,4 };
static unsigned char eq_p[] = { 0, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0xff };
static int ak[] = { 1,8,  3,12,  5,8,  2,10,  4,10,  4,7,  4,3,  6,7,  5,5,  4,5,  5,5,  6,3 };
static unsigned char ak_p[] = { 0, 1, 1, 0, 1, 0, 1, 0, 1, 1, 0, 0xff };
static int bl[] = { 1,8,  1,12, 3,12,  4,11,  3,10,  1,10,  4,10,  5,9,  4,8,  1,8,  5,7,  5,3,  8,3 };
static unsigned char bl_p[] = { 0, 1, 1, 1, 1, 0, 1, 1, 1, 1, 0, 1, 0xff };
static int bs[] = { 1,8,  1,12, 3,12,  4,11,  3,10,  1,10,  4,10,  5,9,  4,8,  1,8,  7,7,  5,7,  4,6,  5,5,  
			7,5,  8,4,  7,3,  4,3 };
static unsigned char bs_p[] = { 0, 1, 1, 1, 1, 0, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 0xff };
static int ht[] = { 1,8,  1,12,  4,8,  4,12,  1,10,  4,10,  3,7,  7,7,  5,7,  5,3 };
static unsigned char ht_p[] = { 0, 1, 0, 1, 0, 1, 0, 1, 0, 0xff };
static int lf[] = { 1,12,  1,8,  4,8,  8,8,  5,8,  5,3,  5,5,  7,5 };
static unsigned char lf_p[] = { 0, 1, 1, 0, 1, 1, 0, 0xff };
static int vt[] = { 1,12,  3,8,  4,12,  3,7,  7,7,  5,7,  5,3 };
static unsigned char vt_p[] = { 0, 1, 1, 0, 1, 0, 0xff };
static int ff[] = { 4,12,  1,12,  1,8,  1,10,  3,10,  7,7,  4,7,  4,3,  4,5,  6,5 };
static unsigned char ff_p[] = { 0, 1, 1, 0, 1, 0, 1, 1, 0, 0xff };
static int cr[] = { 4,11,  3,12,  2,12,  1,11,  1,9,  2,8,  3,8,  4,9,  4,3,  4,7,  6,7,  7,6,  6,5,  4,5,  6,5,  7,3 };
static unsigned char cr_p[] = { 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 0, 0xff };
static int so[] = { 4,12,  2,12,  1,11,  2,10,  4,10,  5,9,  4,8,  1,8, 4,7,  5,8,  6,8,  7,7,  7,5,  6,4,  5,4,  4,5,  4,7 };
static unsigned char so_p[] = { 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 0xff };
static int si[] = { 4,12,  2,12,  1,11,  2,10,  4,10,  5,9,  4,8,  1,8, 5,7,  7,7,  6,7,  6,3,  5,3,  7,3 };
static unsigned char si_p[] = { 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 0xff };
static int dl[] = { 1,8,  1,12,  3,12,  4,11,  4,9,  3,8,  1,8,  5,7,  5,3,  8,3 };
static unsigned char dl_p[] = { 0, 1, 1, 1, 1, 1, 1, 0, 1, 0xff };
static int d1[] = { 1,8,  1,12,  3,12,  4,11,  4,9,  3,8,  1,8, 4,6,  5,7,  5,3,  4,3,  6,3  };
static unsigned char d1_p[] = { 0, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 0xff };
static int d2[] = { 1,8,  1,12,  3,12,  4,11,  4,9,  3,8,  1,8, 4,6,  5,7,  6,7,  7,6,  7,5,  4,4,  4,3,  7,3  };
static unsigned char d2_p[] = { 0, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 0xff };
static int d3[] = { 1,8,  1,12,  3,12,  4,11,  4,9,  3,8,  1,8, 5,7,  7,7,  8,6,  7,5,  6,5,  7,5,  8,4,  7,3,  5,3  };
static unsigned char d3_p[] = { 0, 1, 1, 1, 1, 1, 1, 0, 1,1, 1, 1, 0, 1, 1, 0xff };
static int d4[] = { 1,8,  1,12,  3,12,  4,11,  4,9,  3,8,  1,8,  6,3,  6,7,  4,4,  7,4 };
static unsigned char d4_p[] = { 0, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0xff };
static int nk[] = { 1,8,  1,12,  4,8,  4,12,  4,7,  4,3,  6,7,  5,5,  4,5,  5,5,  6,3  };
static unsigned char nk_p[] = { 0, 1, 1, 1, 0, 1, 1, 1, 1, 1, 0xff  };
static int sy[] = { 4,12,  2,12,  1,11,  2,10,  4,10,  5,9,  4,8,  1,8,  4,7,  6,5,  8,7,  6,5,  6,3  };
static unsigned char sy_p[] = { 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 0xff  };
static int eb[] = { 4,12,  1,12,  1,8,  4,8,  1,10,  3,10,  4,3,  4,7,  6,7,  7,6,  6,5,  4,5,  7,5,  8,4,  7,3,  4,3 };
static unsigned char eb_p[] = { 0, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 0, 1, 1, 1, 0xff  };
static int cn[] = { 4,11,  3,12,  2,12,  1,11,  1,9,  2,8,  3,8,  4,9,  4,3,  4,7,  7,3,  7,7 };
static unsigned char cn_p[] = { 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0xff  };
static int em[] = { 4,12,  1,12,  1,8,  4,8,  1,10,  3,10,  4,3,  4,7,  6,5,  8,7,  8,3 };
static unsigned char em_p[] = { 0, 1, 1, 1, 0, 1, 0, 1, 1, 1, 0xff  };
static int sb[] = { 4,12,  2,12,  1,11,  2,10,  4,10,  5,9,  4,8,  1,8, 4,3,  4,7,  6,7,  7,6,  6,5,  4,5,  7,5,  8,4,  7,3,  4,3 };
static unsigned char sb_p[] = { 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 0, 1, 1, 1, 0xff };
static int ec[] = { 4,12,  1,12,  1,8,  4,8,  1,10,  3,10,  7,6,  6,7,  5,7,  4,6,  4,4,  5,3,  6,3,  7,4 };
static unsigned char ec_p[] = { 0, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 1, 1, 0xff };
static int fs[] = { 4,12,  1,12,  1,8,  1,10,  3,10,  7,7,  5,7,  4,6,  5,5,  7,5,  8,4,  7,3,  4,3 };
static unsigned char fs_p[] = { 0, 1, 1, 0, 1, 0, 1, 1, 1, 1, 1, 1, 0xff };
static int gr[] = { 4,11,  3,12,  2,12,  1,11,  1,9,  2,8,  4,8,  4,9,  3,9,  4,3,  4,7,  6,7,  7,6,  6,5,  4,5,  6,5,  7,3 };
static unsigned char gr_p[] = { 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 0, 0xff };
static int rs[] = { 1,8,  1,12,  3,12,  4,11,  3,10,  1,10,  3,10,  4,8,  7,7,  5,7,  4,6,  5,5,  7,5,  8,4,  7,3,  4,3 };
static unsigned char rs_p[] = { 0, 1, 1, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 1, 1, 0xff };
static int us[] = { 1,12,  1,9,  2,8,  3,8,  4,9,  4,12,  7,7,  5,7,  4,6,  5,5,  7,5,  8,4,  7,3,  4,3 };
static unsigned char us_p[] = { 0, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 0xff };




static int spa[] = { 0,0, 0,0 };		/* 20h " " space */
static unsigned char spa_p[] = { 0, 0x0fe };		/* special case of no drawing, but occupying space */
static int excl[] = { 4,12, 4,6, 4,5, 4,4 };	/* 21h	"!" point */
static unsigned char excl_p[] = {  0,1,0,0x0ff };
static int dquote[] = { 4,12, 4,10, 6,12, 6,10 };	/* 22h double quote */
static unsigned char dqte_p[] = {  0,1,0,0x0ff };
static int sharp[] = { 3,4, 3,12, 5,12, 5,4, 7,7, 1,7, 1,9, 7,9 };	/* 23h pound sign (sharp) */
static unsigned char sharp_p[] = { 0,1,0,1,0,1,0,0x0ff };
static int dollar[] = { 7,10, 6,11, 2,11, 1,10, 1,9, 2,8, 4,8, 5,7, 6,7, 7,6,
		7,5, 6,4, 2,4, 1,5, 4,3, 4,12 };	/* 24h "$" */
static unsigned char dllr_p[] = { 0,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0x0ff };
static int percent[] = { 2,12, 1,11, 2,10, 3,11, 2,12, 8,12, 1,5, 6,7, 5,6, 6,5, 7,6, 6,7 };	/* 25h "%" */
static unsigned char pct_p[] = { 0,1,1,1,1,0,1,0,1,1,1,0x0ff };
static int amper[] = { 7,4, 1,10, 1,11, 2,12, 3,12, 4,11, 4,10, 1,7, 1,5, 2,4, 5,4, 7,6 };	/* 26h ampersand */
static unsigned char amper_p[] = { 0,1,1,1,1,1,1,1,1,1,1,0x0ff };
static int squote[] = { 6,12, 5,9 };	/* 27h single quote */
static unsigned char sqte_p[] = { 0,0x0ff };
static int lparen[] = { 5,12, 3,10, 3,6, 5,4 };	/* 28h left parenthese */
static unsigned char lprn_p[] = { 0,1,1,0x0ff };
static int rparen[] = { 4,12, 6,10, 6,6, 4,4 };	/* 29h	right parenthese */
static unsigned char rprn_p[] = { 0,1,1,0x0ff };
static int star[] = { 4,5, 4,11, 2,10, 6,6, 7,8, 1,8, 2,6, 6,10 };	/* 2ah asterisk */
static unsigned char star_p[] = { 0,1,0,1,0,1,0,0x0ff };
static int plus[] = { 4,10, 4,6, 2,8, 6,8 };	/* 2bh "+" */
static unsigned char plus_p[] = { 0,1,0,0x0ff };
static int comma[] = { 5,4, 4,4, 4,5, 5,5, 5,3, 4,2 };	/* 2ch "," */
static unsigned char comma_p[] = { 0,1,1,1,1,0x0ff };
static int minus[] = { 2,8, 6,8 };	/* 2dh "-" */
static unsigned char minus_p[] = { 0,0x0ff };
static int period[] = { 5,5, 5,4, 4,4, 4,5, 5,5 };	/* 2eh "." */
static unsigned char prd_p[] = { 0,1,1,1,0x0ff };
static int fract[] = { 1,5, 7,11 };	/* 2fh "/" */
static unsigned char fract_p[] = { 0,0x0ff };
/*______________________________________*/

static int zero[] = { 2,12, 6,12, 7,11, 7,5, 6,4, 2,4, 1,5, 1,11, 2,12, 7,11, 1,5 };	/* 30h "0" */
static unsigned char zero_p[] = { 0,1,1,1,1,1,1,1,1,0,0x0ff };
static int one[] = { 3,11, 4,12, 4,4, 3,4, 5,4 };	/* 31h "1" */
static unsigned char one_p[] = { 0,1,1,0,0x0ff };
static int two[] = { 1,11, 2,12, 6,12, 7,11, 7,9, 5,7, 3,7, 1,5, 1,4, 7,4 };	/* 32h "2" */
static unsigned char two_p[] = { 0,1,1,1,1,1,1,1,1,0x0ff };
static int three[] = { 1,11, 2,12, 6,12, 7,11, 7,9, 6,8, 3,8, 6,8, 7,7, 7,5, 6,4, 2,4, 1,5 };	/* 33h "3" */
static unsigned char three_p[] = { 0,1,1,1,1,1,1,0,1,1,1,1,0x0ff };
static int four[] = { 6,4, 6,12, 1,7, 1,6, 7,6 };	/* 34h "4" */
static unsigned char four_p[] = { 0,1,1,1,0x0ff };
static int five[] = { 7,12, 1,12, 1,8, 6,8, 7,6, 7,5, 6,4, 2,4, 1,5 };	/* 35h "5" */
static unsigned char five_p[] = { 0,1,1,1,1,1,1,1,0x0ff };
static int six[] = { 6,12, 3,12, 1,10, 1,5, 2,4, 6,4, 7,5, 7,7, 6,8, 1,8 };	/* 36h "6" */
static unsigned char six_p[] = { 0,1,1,1,1,1,1,1,1,0x0ff };
static int seven[] = { 1,12, 7,12, 7,10, 4,7, 4,4 };	/* 37h "7" */
static unsigned char seven_p[] = { 0,1,1,1,0x0ff };
static int eight[] = { 2,8, 1,9, 1,11, 2,12, 6,12, 7,11, 7,9, 6,8, 2,8, 1,7,
		1,5, 2,4, 6,4, 7,5, 7,7, 6,8 };	/* 38h "8" */
static unsigned char eight_p[] = { 0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0x0ff };
static int nine[] = { 7,8, 2,8, 1,9, 1,11, 2,12, 6,12, 7,11, 7,5, 6,4, 2,4 };	/* 39h "9" */
static unsigned char nine_p[] = { 0,1,1,1,1,1,1,1,1,0x0ff };
/*_______________________________________*/

static int colon[] = { 5,10, 5,9, 4,9, 4,10, 5,10, 5,5, 5,4, 4,4, 4,5, 5,5 };	/* 3ah ":" */
static unsigned char colon_p[] = { 0,1,1,1,1,0,1,1,1,0x0ff };
static int semi[] = { 5,10, 5,9, 4,9, 4,10, 5,10, 5,4, 4,4, 4,5, 5,5, 5,3, 4,2 };	/* 3bh ';' */
static unsigned char semi_p[] = { 0,1,1,1,1,0,1,1,1,1,0x0ff };
static int lss[] = { 5,11, 2,8, 5,5 };	/* 3ch "less than" sign */
static unsigned char lss_p[] = { 0,1,0x0ff };
static int equal[] = { 1,7, 7,7, 1,9, 7,9 };	/* 3dh equal sign */
static unsigned char equal_p[] = { 0,1,0,0x0ff };
static int gtr[] = { 3,11, 6,8, 3,5 };	/* 3eh "greater than" sign */
static unsigned char gtr_p[] = { 0,1,0x0ff };
static int q_mark[] = { 1,11, 2,12, 6,12, 7,11, 7,9, 6,8, 5,8, 4,7, 4,6, 4,5, 4,4 };	/* 3fh "?" */
static unsigned char q_mark_p[] = { 0,1,1,1,1,1,1,1,1,0,0x0ff };
static int atsgn[] = { 6,6, 4,6, 3,7, 3,8, 4,9, 5,9, 6,8, 6,6, 7,6, 7,9,
		5,11, 3,11, 1,9, 1,6, 3,4, 7,4 };	/* 40h "at" sign */
static unsigned char atsgn_p[] = { 0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0x0ff };


static int a_41[] = { 1,4, 1,10, 3,12, 5,12, 7,10, 7,4, 7,8, 1,8 };
static unsigned char a_41p[] = { 0,1,1,1,1,1,0,0x0ff };
static int b_42[] = { 2,4, 2,12, 1,12, 5,12, 6,11, 6,9, 5,8, 2,8, 4,8, 6,8, 7,7,
 		7,5, 6,4, 1,4 };
static unsigned char b_42p[] = { 0,1,0,1,1,1,1,1,0,1,1,1,1,0x0ff };
static int c_43[] = { 7,6, 5,4, 3,4, 1,6, 1,10, 3,12, 5,12, 7,10 };
static unsigned char c_43p[] = { 0,1,1,1,1,1,1,0x0ff };
static int d_44[] = { 2,4, 2,12, 1,12, 5,12, 7,10, 7,6, 5,4, 1,4 };
static unsigned char d_44p[] = { 0,1,0,1,1,1,1,0x0ff };
static int e_45[] = { 7,4, 1,4, 1,12, 7,12, 1,8, 5,8 };
static unsigned char e_45p[] = { 0,1,1,1,0,0x0ff };
static int f_46[] = { 1,4, 1,12, 7,12, 1,8, 5,8 };
static unsigned char f_46p[] = { 0,1,1,0,0x0ff };
static int g_47[] = { 5,7, 7,7, 7,5, 6,4, 3,4, 1,6, 1,10, 3,12, 5,12, 7,10 };
static unsigned char g_47p[] = { 0,1,1,1,1,1,1,1,1,0x0ff };
static int h_48[] = { 1,4, 1,12, 1,8, 7,8, 7,4, 7,12 };
static unsigned char h_48p[] = { 0,1,0,1,0,0x0ff };
static int i_49[] = { 2,4, 6,4, 4,4, 4,12, 2,12, 6,12 };
static unsigned char i_49p[] = { 0,1,0,1,0,0x0ff };
static int j_4a[] = { 1,6, 3,4, 5,4, 6,6, 6,12, 4,12, 8,12 };
static unsigned char j_4ap[] = { 0,1,1,1,1,0,0x0ff };
static int k_4b[] = { 1,4, 1,12, 1,6, 6,11, 3,8, 7,4 };
static unsigned char k_4bp[] = { 0,1,0,1,0,0x0ff };
static int l_4c[] = { 1,12, 1,4, 7,4 };
static unsigned char l_4cp[] = { 0,1,0x0ff };
static int m_4d[] = { 1,4, 1,12, 4,8, 7,12, 7,4 };
static unsigned char m_4dp[] = { 0,1,1,1,0x0ff };
static int n_4e[] = { 1,4, 1,12, 7,4, 7,12 };
static unsigned char n_4ep[] = { 0,1,1,0x0ff };
static int o_4f[] = { 5,4, 3,4, 1,6, 1,10, 3,12, 5,12, 7,10, 7,6, 5,4 };
static unsigned char o_4fp[] = { 0,1,1,1,1,1,1,1,0x0ff };
static int p_50[] = { 1,4, 1,12, 6,12, 7,11, 7,9, 6,8, 1,8 };
static unsigned char p_50p[] = { 0,1,1,1,1,1,0x0ff };
static int q_51[] = { 5,4, 3,4, 1,6, 1,10, 3,12, 5,12, 7,10, 7,6, 5,4, 5,6, 7,4 };
static unsigned char q_51p[] = { 0,1,1,1,1,1,1,1,1,0,0x0ff };
static int r_52[] = { 1,4, 1,12, 6,12, 7,11, 7,9, 6,8, 1,8, 4,8, 7,5, 7,4 };
static unsigned char r_52p[] = { 0,1,1,1,1,1,1,0,1,0x0ff };
static int s_53[] = { 7,11, 6,12, 2,12, 1,11, 1,9, 2,8, 6,8, 7,7, 7,5, 6,4, 2,4, 1,5 };
static unsigned char s_53p[] = { 0,1,1,1,1,1,1,1,1,1,1,0x0ff };
static int t_54[] = { 1,12, 7,12, 4,12, 4,4 };
static unsigned char t_54p[] = { 0,1,0,0x0ff };
static int u_55[] = { 7,12, 7,6, 5,4, 3,4, 1,6, 1,12 };
static unsigned char u_55p[] = { 0,1,1,1,1,0x0ff };
static int v_56[] = { 1,12, 4,4, 7,12 };
static unsigned char v_56p[] = { 0,1,0x0ff };
static int w_57[] = { 1,12, 1,4, 4,8, 7,4, 7,12 };
static unsigned char w_57p[] = { 0,1,1,1,0x0ff };
static int x_58[] = { 1,12, 1,11, 7,5, 7,4, 1,4, 1,5, 7,11, 7,12 };
static unsigned char x_58p[] = { 0,1,1,1,0,1,1,0x0ff };
static int y_59[] = { 1,12, 1,11, 4,8, 7,12, 7,11, 4,8, 4,4 };
static unsigned char y_59p[] = { 0,1,1,0,1,1,0x0ff };
static int z_5a[] = { 1,12, 7,12, 7,11, 1,5, 1,4, 7,4 };
static unsigned char z_5ap[] = { 0,1,1,1,1,0x0ff };
/*________________________
	end of upper case letters
__________________________*/
static int lsqbkt[] = { 5,12, 3,12, 3,4, 5,4 };	/* 5bh left square bracket */
static unsigned char lsqbkt_p[] = { 0,1,1,0x0ff };
static int bslash[] = { 1,11, 7,5 };	/* 5ch back-slash */
static unsigned char bslash_p[] = { 0,0x0ff };
static int rsqbkt[] = { 4,12, 6,12, 6,4, 4,4 };	/* 5dh right square bracket */
static unsigned char rsqbkt_p[] = { 0,1,1,0x0ff };
static int carat[] = { 2,9, 4,11, 6,9 };	/* 5eh "carat" */
static unsigned char carat_p[] = { 0,1,0x0ff };
static int under[] = { 1,2, 6,2 };	/* 5fh underscore */
static unsigned char under_p[] = { 0,0x0ff };
static int pick[] = { 4,12, 5,9 };	/* 60h "pick" (accent-grave) */
static unsigned char pick_p[] = { 0,0x0ff };
/*________________________________________*/

static int a_61[] = { 1,8, 2,9, 5,9, 6,8, 6,5, 7,4, 6,5, 5,4, 2,4, 1,5, 1,6, 2,7, 6,7 };
static unsigned char a_61p[] = { 0,1,1,1,1,1,0,1,1,1,1,1,0x0ff };
static int b_62[] = { 1,4, 1,12, 1,7, 3,9, 5,9, 6,8, 6,5, 5,4, 3,4, 1,6 };
static unsigned char b_62p[] = { 0,1,0,1,1,1,1,1,1,0x0ff };
static int c_63[] = { 6,8, 5,9, 2,9, 1,8, 1,5, 2,4, 5,4, 6,5 };
static unsigned char c_63p[] = { 0,1,1,1,1,1,1,0x0ff };
static int d_64[] = { 6,6, 4,4, 2,4, 1,5, 1,8, 2,9, 4,9, 6,7, 6,12, 6,4 };
static unsigned char d_64p[] = { 0,1,1,1,1,1,1,1,0,0x0ff };
static int e_65[] = { 1,7, 6,7, 6,8, 5,9, 2,9, 1,8, 1,5, 2,4, 5,4, 6,5 };
static unsigned char e_65p[] = { 0,1,1,1,1,1,1,1,1,0x0ff };
static int f_66[] = { 2,4, 2,11, 3,12, 5,12, 6,11, 1,9, 3,9 };
static unsigned char f_66p[] = { 0,1,1,1,1,0,0x0ff };
static int g_67[] = { 6,6, 4,4, 2,4, 1,5, 1,8, 2,9, 5,9, 6,8, 6,2, 5,1, 2,1, 1,2 };
static unsigned char g_67p[] = { 0,1,1,1,1,1,1,1,1,1,1,0x0ff };
static int h_68[] = { 1,12, 1,4, 1,7, 3,9, 5,9, 6,8, 6,4 };
static unsigned char h_68p[] = { 0,1,0,1,1,1,0x0ff };
static int i_69[] = { 4,10, 4,10, 3,8, 4,8, 4,4 };
static unsigned char i_69p[] = { 0,1,0,1,0x0ff };
static int j_6a[] = { 6,10, 6,10, 5,8, 6,8, 6,2, 5,1, 2,1, 1,2 };
static unsigned char j_6ap[] = { 0,1,0,1,1,1,1,0x0ff };
static int k_6b[] = { 1,12, 1,4, 5,10, 1,6, 3,8, 6,5, 6,4 };
static unsigned char k_6bp[] = { 0,1,0,1,0,1,0x0ff };
static int l_6c[] = { 3,12, 4,12, 4,4, 3,4, 5,4 };
static unsigned char l_6cp[] = { 0,1,1,0,0x0ff };
static int m_6d[] = { 1,9, 1,4, 1,8, 2,9, 3,9, 4,8, 4,4, 4,8, 5,9, 6,9, 7,8, 7,4 };
static unsigned char m_6dp[] = { 0,1,0,1,1,1,1,0,1,1,1,0x0ff };
static int n_6e[] = { 1,9, 1,4, 1,7, 3,9, 5,9, 6,8, 6,4 };
static unsigned char n_6ep[] = { 0,1,0,1,1,1,0x0ff };
static int o_6f[] = { 6,8, 5,9, 2,9, 1,8, 1,5, 2,4, 5,4, 6,5, 6,8 };
static unsigned char o_6fp[] = { 0,1,1,1,1,1,1,1,0x0ff };
static int p_70[] = { 1,1, 1,9, 1,7, 3,9, 5,9, 6,8, 6,5, 5,4, 3,4, 1,6 };
static unsigned char p_70p[] = { 0,1,0,1,1,1,1,1,1,0x0ff };
static int q_71[] = { 6,1, 6,9, 6,7, 4,9, 2,9, 1,8, 1,5, 2,4, 4,4, 6,6};
static unsigned char q_71p[] = { 0,1,0,1,1,1,1,1,1,0x0ff };
static int r_72[] = { 1,4, 1,9, 1,7, 3,9, 5,9, 6,8 };
static unsigned char r_72p[] = { 0,1,0,1,1,0x0ff };
static int s_73[] = { 5,9, 2,9, 1,8, 2,7, 5,7, 6,6, 6,5, 5,4, 2,4, 1,5 };
static unsigned char s_73p[] = { 0,1,1,1,1,1,1,1,1,0x0ff };
static int t_74[] = { 3,12, 3,5, 4,4, 5,4, 6,5, 1,10, 5,10 };
static unsigned char t_74p[] = { 0,1,1,1,1,0,0x0ff };
static int u_75[] = { 1,9, 1,5, 2,4, 5,4, 6,5, 6,9, 6,4 };
static unsigned char u_75p[] = { 0,1,1,1,1,0,0x0ff };
static int v_76[] = { 1,9, 4,4, 7,9 };
static unsigned char v_76p[] = { 0,1,0x0ff };
static int w_77[] = { 1,9, 1,5, 2,4, 4,6, 4,8, 4,6, 6,4, 7,5, 7,9 };
static unsigned char w_77p[] = { 0,1,1,1,1,0,1,1,0x0ff };
static int x_78[] = { 1,4, 6,9, 1,9, 6,4 };
static unsigned char x_78p[] = { 0,1,0,0x0ff };
static int y_79[] = { 1,9, 1,5, 2,4, 4,4, 6,6, 6,9, 6,2, 5,1, 3,1 };
static unsigned char y_79p[] = { 0,1,1,1,1,0,1,1,0x0ff };
static int z_7a[] = { 1,9, 6,9, 1,4, 6,4 };
static unsigned char z_7ap[] = { 0,1,1,0x0ff };
/*-------------------------------
end of lower case
---------------------------------*/
static int lcbkt[] = { 4,12, 3,11, 3,9, 2,8, 3,7, 3,5, 4,4 };	/* 7bh left curly bracket */
static unsigned char lcbkt_p[] = { 0,1,1,1,1,1,0x0ff };
static int pipe[] = { 4,13, 4,2 };	/* 7ch vertical pipe */
static unsigned char pipe_p[] = { 0,0x0ff };
static int rcbkt[] = { 4,12, 5,11, 5,9, 6,8, 5,7, 5,5, 4,4 };	/* 7dh right curly bracket */
static unsigned char rcbkt_p[] = { 0,1,1,1,1,1,0x0ff };
static int tilde[] = { 1,10, 2,11, 3,11, 5,9, 6,9, 7,10 };	/* 7eh tilde */
static unsigned char tilde_p[] = { 0,1,1,1,1,0x0ff };

static int ch_7fh[] = { 0,0 };
static unsigned char ch_7fh_p[] = { 0x0ff };

/*-------------------------------------*/
static int _far *char_v[] = {			/* vector drawing data pointers */
	no_char, 			/* Null char */
		/* control char's */
	sh, sx, ex, et, eq, ak, bl, bs, ht, lf, vt, ff, cr, so, si,
		/* 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0ah, 0bh, 0ch, 0dh, 0eh, 0fh */
	dl, d1, d2, d3, d4, nk, sy, eb, cn, em, sb, ec, fs, gr, rs, us,
		/* 10h, 11h, 12h, 13h, 14h, 15h, 16h, 17h, 18h, 19h, 1ah, 1bh, 1ch, 1dh, 1eh, 1fh */

	spa, excl, dquote, sharp, dollar, percent, amper, squote, lparen,
		/* 20h,21h,22h,23h,24h,25h,26h,27h,28h */
	rparen, star, plus, comma, minus, period, fract, zero,
		/* 29h, 2ah, 2bh,2ch,2dh,2eh,2fh,30h */
	one, two, three, four, five, six, seven, eight, nine,
	colon, semi, lss, equal, gtr, q_mark, atsgn,
		/* 3ah;3bh;3ch;3dh;3eh;3fh;40h */
	a_41, b_42, c_43, d_44, e_45, f_46, g_47, h_48, i_49,
	j_4a, k_4b, l_4c, m_4d, n_4e, o_4f, p_50, q_51, r_52,
	s_53, t_54, u_55, v_56, w_57, x_58, y_59, z_5a,
	lsqbkt, bslash, rsqbkt, carat, under, pick,
		/* 5bh;5ch;5dh;5eh;5fh */
	a_61, b_62, c_63, d_64, e_65, f_66, g_67, h_68, i_69,
	j_6a, k_6b, l_6c, m_6d, n_6e, o_6f, p_70, q_71, r_72,
	s_73, t_74, u_75, v_76, w_77, x_78, y_79, z_7a,
	lcbkt, pipe, rcbkt, tilde, ch_7fh };
		/* 7bh;7ch;7dh;7eh;character for 07fh */

static unsigned char _far *char_p[] = {			/* pen control pointers */
	no_char_p,			/* non-printing char's */
		/* control char's */
	sh_p, sx_p, ex_p, et_p, eq_p, ak_p, bl_p, bs_p, ht_p, lf_p, vt_p, ff_p, cr_p, so_p, si_p,
		/* 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0ah, 0bh, 0ch, 0dh, 0eh, 0fh */
	dl_p, d1_p, d2_p, d3_p, d4_p, nk_p, sy_p, eb_p, cn_p, em_p, sb_p, ec_p, fs_p, gr_p, rs_p, us_p,
		/* 10h, 11h, 12h, 13h, 14h, 15h, 16h, 17h, 18h, 19h, 1ah, 1bh, 1ch, 1dh, 1eh, 1fh */

	spa_p, excl_p, dqte_p, sharp_p, dllr_p, pct_p, amper_p, sqte_p,
		/* 20h;21h;22h;23h;24h;25h;26h;27h */
	lprn_p, rprn_p, star_p, plus_p, comma_p, minus_p, prd_p,
		/* 28h;29h;2ah;2bh;2ch;2dh;2eh */
	fract_p, zero_p,
		/* 2fh; 30h */
	one_p, two_p, three_p, four_p, five_p, six_p, seven_p, eight_p, nine_p,
	colon_p, semi_p, lss_p, equal_p, gtr_p, q_mark_p, atsgn_p,
		/* 39h;3ah;3bh;3ch;3dh;3eh;3fh;40h */
	a_41p, b_42p, c_43p, d_44p, e_45p, f_46p, g_47p, h_48p, i_49p,
	j_4ap, k_4bp, l_4cp, m_4dp, n_4ep, o_4fp, p_50p, q_51p, r_52p,
	s_53p, t_54p, u_55p, v_56p, w_57p, x_58p, y_59p, z_5ap,
	lsqbkt_p, bslash_p, rsqbkt_p, carat_p, under_p, pick_p,
		/* 5bh;5ch;5dh;5eh;5fh */
	a_61p, b_62p, c_63p, d_64p, e_65p, f_66p, g_67p, h_68p, i_69p,
	j_6ap, k_6bp, l_6cp, m_6dp, n_6ep, o_6fp, p_70p, q_71p, r_72p,
	s_73p, t_74p, u_75p, v_76p, w_77p, x_78p, y_79p, z_7ap,
	lcbkt_p, pipe_p, rcbkt_p, tilde_p,
		/* 7bh;7ch;7dh;7eh */
	ch_7fh_p };	/* pen control for char at 07fh */


/*-------------------------------------*/

static struct char___deft std_chars =
	{
	"standard",	/* font name */
	char_v,		/* adr of array of pointers to char stroke vectors */
	char_p,		/* adr of array of pointer to pen control codes */
	0x7f,		/* highest char. code defined */
	13.0,		/* highest part of any char (pipe in this case) */
	1.0,		/* height of lowest part of char */
	8.0,		/* width of char cell (0 -> variable */
	-1.0,		/* base of char for lorg */
	8.5,		/* center of char for lorg */
	16.0,		/* top of char for lorg */
	9.0,		/* char-to-char std spacing */
	15.0,		/* line-to-line std spacing */
	1.0		/* scale factor for font */
	};

/*-------------------------------------*/

struct char___deft _far *std_font ( void )	/* return pointer to struct defining std char font */
	{

	return (& std_chars);
	}

/*-------------------------------------*/
