/*
 * QIC_02.H - Structures and constants for interfacing to a Wangtek PC36 or
 * similar QIC-02 compatible IBM PC tape controller.
 *
 * $Author:   Phlash  $
 * $Date:   23 Oct 1994 10:22:02  $
 * $Revision:   1.2  $
 */

#ifndef _QIC02_H
#define _QIC02_H

/*--------------------------------------------------------------------------
   The controller interface is via two IO ports, at a BASE address and
   a BASE+1 address.
   The following registers are available through these IO ports:

   Addr        Access Mode Register
   ====        =========== ========
   BASE        Read        Status register
   BASE        Write       Control register
   BASE+1      Read        Data register
   BASE+1      Write       Command / data register

   In addition to these registers, the controller uses DMA for data transfer,
   and can optionally use an interrupt for signalling Ready and Exception
   states.
----------------------------------------------------------------------------

   The tape interface library (QIC_02.LIB) provides the following basic
   controller functions:

      int qicInit(int port, int intr, int dma, int blocking);
         Attempts to initialise the controller, MUST be called first.
         0 = OK.
         -1 = No controller.

      int qicSelect(int density);
         Selects tape density to write with (QIC-11/24/120/150)
         0 = OK.
         <0 = ERROR.

      int qicReadBlocks(char far *buffer, int nBlocks);
         Reads blocks of data (512 bytes each) from the tape into buffer.
         <0 = ERROR.
         0 = End of file.
         >0 = number of blocks read.

      int qicSkipFile(void);
         Skips over the current file until a file mark is found.
         0 = OK.
         <0 = ERROR.

      int qicRewind(void);
         Rewinds the tape.
         0 = OK.
         <0 = ERROR.

      int qicWriteBlocks(char far *buffer, int nBlocks);
         Writes blocks of data (512 bytes each) to the tape from buffer.
         0 = OK.
         <0 = ERROR.
         >0 = Number of blocks written

      int qicEraseTape(void);
         Erases the tape completely.
         0 = OK.
         <0 = ERROR.

      int qicWriteFileMark(void);
         Writes and end of file marker to the tape.
         0 = OK.
         <0 = ERROR.

      int qicComplete(void);
         Waits for outstanding I/O to complete
         0 = End of file.
         <0 = ERROR.
         >0 = Number of blocks transferred

--------------------------------------------------------------------------*/

/* Typical IO port BASE addresses */
#define DEF_BASE1       0x300
#define DEF_BASE2       0x200

/* Status register bits, NB: All are active LOW logic levels */
#define STATUS_RDY      0x01
#define STATUS_EXC      0x02
#define STATUS_DIR      0x04

/* Control register bits, NB: All are active HIGH logic levels */
#define CONTROL_CLR     0x00
#define CONTROL_ONL     0x01
#define CONTROL_RES     0x02
#define CONTROL_REQ     0x04
#define CONTROL_DMA     0x08

/* OIC-02 Commands */
#define SELECT_DR0      0x01
#define SELECT_DR1      0x02
#define SELECT_DR2      0x04
#define REWIND          0x21
#define ERASE           0x22
#define RETENSION       0x24
#define WRITE_DATA      0x40
#define WRITE_MARK      0x60
#define READ_DATA       0x80
#define READ_MARK       0xA0
#define READ_STATUS     0xC0
#define QIC11_FORMAT    0x26
#define QIC24_FORMAT    0x27
#define QIC120_FORMAT   0x28
#define QIC150_FORMAT   0x29

/* QIC-02 Structures */
typedef unsigned char   BYTE;
typedef unsigned short  WORD;
typedef unsigned long   DWORD;

#pragma pack(1)
typedef struct
{
   BYTE FIL:1;           /* Status byte 0 */
   BYTE BNL:1;
   BYTE UDA:1;
   BYTE EOM:1;
   BYTE WRP:1;
   BYTE DFF:1;
   BYTE CNI:1;
   BYTE ST0:1;

   BYTE POR:1;           /* Status byte 1 */
   BYTE RE0:1;
   BYTE RE1:1;
   BYTE BOM:1;
   BYTE MBD:1;
   BYTE NDT:1;
   BYTE ILL:1;
   BYTE ST1:1;
   
   WORD dataErrors;
   WORD underRuns;
} status_t;

#pragma pack()

/* Bitfield values */
#define TRUE   1
#define FALSE  0

/* Library data */
extern int qicDebug;
extern char qicVersion[];

/* Library functions */
extern int qicInit(int newPort, int newInt, int newDma, int blocking);
extern int qicSelect(int density);
extern int qicRewind(void);
extern int qicSkipFile(void);
extern int qicReadBlocks(char far *buffer, int nBlocks);
extern int qicWriteBlocks(char far *buffer, int nBlocks);
extern int qicComplete(void);
extern int qicEraseTape(void);
extern int qicWriteFileMark(void);
extern void qicDeInit(void);

#endif /* _QIC02_H */
