/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       IPBMTST.C
**     SYSTEM   NAME:       IP Buffer Management
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1990/2/1
**
** DESCRIPTION: Contains test program for IP buffer management
**
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.0  $
** WORKFILE:    $Workfile:   IPBMTST.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/BUFM/VCS/IPBMTST.C_V  $
**              
**                 Rev 1.0   01 Feb 1991 10:16:54   etstjan
**              Initial revision.
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/BUFM/VCS/IPBMTST.C_V   1.0   01 Feb 1991 10:16:54   etstjan  $";
#endif

#include      <stdlib.h>
#include      <stdio.h>
#include      <string.h>
#include      "ipbm.h"
#define       INCL_BASE                /* needed for time functions */
#define       INCL_NOPM
#include      <os2.h>

#define MAXNUMBER 10000                 /* max # of buffers to be allocated */
#define BUFNR     3                     /* # of memory management blocks */
#define BUFSIZE   0x8000                /* size of memory management blocks */


/* global variables */
char          *PointerArray[MAXNUMBER]; /* keeps pointers to allocated data */
int           BlockSizes[MAXNUMBER];    /* size of each block */

/* prototypes */
unsigned long TimeMilSec(void);

/**************************************************************
** NAME:        TestBuf
** SYNOPSIS:    int TestBuf(void);
** DESCRIPTION: Performs validity checks on the Buffer
**              Management and gives result message.
** RETURNS:     NO_ERR -->  no error
**              MAGERR -->  magic error found
**              TOTERR -->  total size error
**              CYCERR -->  order of list error
**************************************************************/
int TestBuf(void)
{
char Dummy;
int  CheckCode;
  CheckCode = IPBMCheck();
  if (CheckCode) {
    printf("-----> A buffer check error has been found!!!"
           "   Return Code : %d\n", CheckCode);
    printf("Press <Enter> to continue...");
    scanf("%c", &Dummy);
    }
  return CheckCode;
}

/**************************************************************
** NAME:        FreeAll
** SYNOPSIS:    void FreeAll(unsigned int CurrentNr);
** DESCRIPTION: Free all buffers still allocated.
** RETURNS:
**
**************************************************************/
void FreeAll(unsigned int CurrentNr)
{
unsigned int i;

  /* free all datablocks allocated */
  printf("Freeing all still allocated blocks...");
  for (i = 0; i < CurrentNr; i++) {
    if (PointerArray[i] != NULL) {
      if (IPBufFree(PointerArray[i]) != 0) {
        printf("Error in Freeing Block #%d.\n", i);
        exit(1);
        }
      }
    TestBuf();
    }
  printf("Done.\nAll buffers freed.\n");
}

/**************************************************************
** NAME:        main
** SYNOPSIS:    void main(void);
** DESCRIPTION:
** RETURNS:
**************************************************************/

void main(void)

{
int           NrBlocks;                 /* #blocks allocated */
int           i, j, k;                  /* general purpose counter */
char          Dummy;
unsigned long TotSize;                  /* total allocated size */
unsigned int  ReqSize;
unsigned long TimeSave = TimeMilSec();

  printf("\nStarting buffer test program....\n");

  /* allocate BUFNR buffers of BUFSIZE bytes */
  if (IPBufInit(BUFNR, BUFSIZE) != 0) {
    printf("Error in Initializing Buffers. Program halted.\n");
    exit(1);
    }
  TestBuf();                                     /* check buffer validity */
  printf("Memory buffer space has been allocated\n");
  printf("You allocated %u buffers of %u bytes.\n", BUFNR, BUFSIZE);
  printf("Time on our way: %lu mSec\n", TimeMilSec() - TimeSave);
  TimeSave = TimeMilSec();

  /* First, allocate as much 100 byte buffers as possible */
  printf("Allocating as much blocks as possible...");
  NrBlocks = 0;
  while ((NrBlocks < MAXNUMBER) &&
         ((PointerArray[NrBlocks] = IPBufGet(100)) != NULL)) {
    NrBlocks++;
    }
  printf("Done.\nI could allocate %d blocks of 100 bytes."
         " (Max is %d blocks.)\n", NrBlocks, MAXNUMBER);
  TestBuf();
  printf("Allocating the buffers: %lu mSec\n", TimeMilSec() - TimeSave);

  /* now, place data in blocks */
  printf("Filling blocks with data...");
  for (j = 0; j < NrBlocks; j++) {
    for(i = 0; i < 100; i++) {
      *(PointerArray[j] + i) = (char)((j + i) % 256);
      }
    }
  printf("Done.\n");
  TestBuf();

  printf("Checking data in blocks...");
  for (j = 0; j < NrBlocks; j++) {
    for(i = 0; i < 100; i++) {
      if (*(PointerArray[j] + i) != (char)((j + i) % 256)) {
        printf("AARGGGHHHH I found an error!!\n");
        printf("BlockNr : %d  Offset : %d  Expected : %d  Found : %d\n",
                        j, i, (char)((j + i) % 256), *(PointerArray[j] + i));
        printf("Press <Enter> to continue...");
        scanf("%c", &Dummy);
        }
      }
    }
  printf("Done.\n");
  TestBuf();

  TimeSave = TimeMilSec();
  /* free blocks at random, NrBlocks * 2 times */
  printf("Freeing datablocks in random order, %d times...\n", NrBlocks * 2);
  for (j = 0; j < (2 * NrBlocks); j++) {
    i = (rand() % NrBlocks);
    if (PointerArray[i] != NULL) {
      if (IPBufFree(PointerArray[i]) != 0) {
        printf("Error in Freeing Block #%d.\n", i);
        exit(1);
        }
      TestBuf();
      PointerArray[i] = NULL;
      }
    }
  printf("Done.\n");
  printf("This took: %lu mSec\n", TimeMilSec() - TimeSave);

  /* free all datablocks allocated */
  FreeAll(NrBlocks);

  /* request datablocks, random in size */
  printf("Allocating up to %d blocks, random in size (0 to 2000 bytes)...",
          (BUFSIZE / 500) * BUFNR);
  TimeSave = TimeMilSec();
  NrBlocks = 0;
  TotSize = 0;
  for (j = 0; j < (BUFSIZE / 500) * BUFNR; j++) {
    ReqSize = (rand() % 2000);
    if ((PointerArray[NrBlocks] = IPBufGet(ReqSize)) != NULL) {
                                                 /* block could be allocated */
      BlockSizes[NrBlocks++] = ReqSize;          /* store datablock size */
      TotSize += ReqSize;
      }
    TestBuf();
    }
  printf("Done.\nI could allocate %d blocks. (Max is %d blocks.)\n",
          NrBlocks, MAXNUMBER);
  printf("Total number of bytes allocated : %lu\n", TotSize);
  printf("This took: %lu mSec\n", TimeMilSec() - TimeSave);

  /* now, place data in blocks */
  printf("Filling blocks with data...");
  for (j = 0; j < NrBlocks; j++) {
    for(i = 0; i < BlockSizes[j]; i++) {
      *(PointerArray[j] + i) = (char)((j + i) % 256);
      }
    }
  printf("Done.\n");
  TestBuf();

  printf("Checking data in blocks...");
  for (j = 0; j < NrBlocks; j++) {
    for(i = 0; i < BlockSizes[j]; i++) {
      if (*(PointerArray[j] + i) != (char)((j + i) % 256)) {
        printf("AARGGGHHHH I found an error!!\n");
        printf("BlockNr : %d  Offset : %d  Expected : %d  Found : %d\n",
                j, i, (char)((j + i) % 256), *(PointerArray[j] + i));
        printf("Press <Enter> to continue...");
        scanf("%c", &Dummy);
        }
      }
    }
  printf("Done.\n");
  TestBuf();

  /* free all datablocks allocated */
  FreeAll(NrBlocks);

  /* now, it is time for a a real stress test */
  if ((NrBlocks = (BUFSIZE / 500) * BUFNR) > MAXNUMBER) {
    NrBlocks = MAXNUMBER;
    }
  printf("\nNow, it is time for a more serious test.\n");
  printf("I will allocate blocks, random in size (0-2000 bytes),\n"
         "fill them with data and free random selected blocks.\n"
         "The maximum amount of allocated blocks is %d\n"
         "The test will be repeated %d times......\n", NrBlocks, MAXNUMBER);
  for (i = 0; i < NrBlocks; i++) {
    PointerArray[i] = NULL;                  /* first, no allocated blocks */
    }
  TotSize = 0;
  TimeSave = TimeMilSec();
  for (j = 0; j < MAXNUMBER; j++) {
    ReqSize = (rand() % 2000);             /* select random block size */
    i = (rand() % NrBlocks);               /* select random block number */
    if (PointerArray[i] != NULL) {         /* if block number used: free it */
      TotSize -= BlockSizes[i];            /* adjust currently located size */
      if (IPBufFree(PointerArray[i]) != 0) {
        printf("Error in Freeing Block #%d.\n", i);
        exit(1);
        }
      PointerArray[i] = NULL;              /* indicate block free */
      TestBuf();
      }

    if ((PointerArray[i] = IPBufGet(ReqSize)) != NULL) {
                                           /* block could be allocated */
      BlockSizes[i] = ReqSize;             /* store datablock size */
      TotSize += ReqSize;
      for(k = 0; k < BlockSizes[i]; k++) { /* fill block with some data */
        *(PointerArray[i] + k) = (char)((i + k) % 256);
        }
      }
    TestBuf();

    i = (rand() % NrBlocks);               /* free a random data block */
    if (PointerArray[i] != NULL) {

      for(k = 0; k < BlockSizes[i]; k++) { /* first check validity of data */
        if (*(PointerArray[i] + k) != (char)((i + k) % 256)) {
          printf("AARGGGHHHH I found an error!!\n");
          printf("BlockNr : %d  Offset : %d  Expected : %d  Found : %d\n",
                  i, k, (char)((i + k) % 256), *(PointerArray[i] + k));
          printf("Press <Enter> to continue...");
          scanf("%c", &Dummy);
          }
        }

      if (IPBufFree(PointerArray[i]) != 0) {
        printf("Error in Freeing Block #%d.\n", i);
        exit(1);
        }
      TestBuf();
      PointerArray[i] = NULL;              /* free this block */
      TotSize -= BlockSizes[i];            /* and adjust allocated size */
      }
    printf("\rCurrently allocated size : %06lu    Passnumber : %d   ",
            TotSize, j);
    }
  printf("\nDone.\n");

  /* free all remaining blocks */
  FreeAll(NrBlocks);
  printf("This took: %lu mSec\n", TimeMilSec() - TimeSave);

  IPBufEnd();
  printf("Program ended in a normal way.\n\n");
}

/**************************************************************
** NAME:        TimeMilSec
** SYNOPSIS:    unsigned long TimeMilSec(void);
**         
** DESCRIPTION: Returns the time in milliseconds since
**              midnight. 
** RETURNS:     The calculated time.
**************************************************************/
unsigned long TimeMilSec(void)
{
  DATETIME       TimeStruct;

  DosGetDateTime(&TimeStruct);
  return  1000L * (
          (unsigned long)TimeStruct.hours  * 3600L +
          (unsigned long)TimeStruct.minutes  * 60L +
          (unsigned long)TimeStruct.seconds      ) +
          (unsigned long)TimeStruct.hundredths * 10L;
}

