/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       sdm.c
**     SYSTEM   NAME:       BEHOLDER
**     ORIGINAL AUTHOR(S):  Dirk Wisse
**     VERSION  NUMBER:     
**     CREATION DATE:       1991/3/20
**
** DESCRIPTION: Source Destination Matrix for The Beholder.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header$";
#endif


#include <stdlib.h>
#include <stdio.h>
#include <bufm.h>
#include <string.h>
#include <dnpap.h>
#include <error.h>
#include "sdm.h"

#define SDM_ALLOC(Mem)                  IPBufGet(Mem)
#define SDM_FREE(Mem)                   IPBufFree(Mem)

SDM_MATRIX *SdmCreate
            (
                WORD AddrSize,
                WORD HostSize,
                WORD ConnSize,
                WORD HostCount,
                WORD ConnCount,
                WORD HashCount,
                WORD HashOver
            )
{
    SDM_MATRIX   *Matrix;

    Matrix=SDM_ALLOC(sizeof(SDM_MATRIX));
    if (Matrix!=NULL)
    {
        Matrix->AddrSize=AddrSize;
        Matrix->HostSize=HostSize;
        Matrix->ConnSize=ConnSize;
        Matrix->HostCount=HostCount;
        Matrix->ConnCount=ConnCount;
        Matrix->HashCount=HashCount;
        Matrix->HashOver=HashOver;
        Matrix->HashLink=SDM_ALLOC(sizeof(SDM_HASH)*HashCount);
        Matrix->HostLink=SDM_ALLOC(sizeof(SDM_HOST)*HostCount);
        Matrix->HostData=SDM_ALLOC(HostSize*HostCount);
        Matrix->ConnLink=SDM_ALLOC(sizeof(SDM_CONN)*ConnCount);
        Matrix->ConnData=SDM_ALLOC(ConnSize*ConnCount);
    }
    if (Matrix->HashLink!=NULL &&
        Matrix->HostLink!=NULL &&
        Matrix->HostData!=NULL &&
        Matrix->ConnLink!=NULL &&
        Matrix->ConnData!=NULL)
    {
        SdmClear(Matrix);
    }
    else
    {
        SdmRemove(Matrix);
        Matrix=NULL;
    }
    return Matrix;
}

void SdmRemove(SDM_MATRIX *Matrix)
{
    if (Matrix->HashLink!=NULL)
        SDM_FREE(Matrix->HashLink);
    if (Matrix->HostLink!=NULL)
        SDM_FREE(Matrix->HostLink);
    if (Matrix->HostData!=NULL)
        SDM_FREE(Matrix->HostData);
    if (Matrix->ConnLink!=NULL)
        SDM_FREE(Matrix->ConnLink);
    if (Matrix->ConnData!=NULL)
        SDM_FREE(Matrix->ConnData);
    SDM_FREE(Matrix);
}

void SdmClear(SDM_MATRIX *Matrix)
{
    int Ind;
    
    Matrix->HashFree=Matrix->HashOver;
    Matrix->HostFree=0;
    Matrix->HostUsed=-1;
    Matrix->ConnFree=0;
    if (Matrix->HashLink!=NULL)
    {
        for (Ind=0;Ind<(Matrix->HashCount-1);Ind++)
            (Matrix->HashLink)[Ind].Host=-1;
    }
    if (Matrix->HostLink!=NULL)
    {
        for (Ind=0;Ind<(Matrix->HostCount-1);Ind++)
            (Matrix->HostLink)[Ind].Conn=-1;
    }
    if (Matrix->ConnLink!=NULL)
    {
        for (Ind=0;Ind<(Matrix->ConnCount-1);Ind++)
            (Matrix->ConnLink)[Ind].Host=-1;
    }
}



int SdmHash(SDM_MATRIX *Matrix, BYTE *Addr)
{
    int Ind;

    Ind = (((unsigned)Addr[3] << 8) | (unsigned)Addr[4]) % Matrix->HashOver;
    return(Ind);
}

int SdmGetConnLink(SDM_MATRIX *Matrix, int SrcInd, int DstInd)
{
    int ConnInd;
    
    ConnInd=Matrix->HostLink[SrcInd].Conn;
    while (ConnInd>=0)
    {
        if (Matrix->ConnLink[ConnInd].Host==DstInd)
            break;
        ConnInd=Matrix->ConnLink[ConnInd].Next;
    }
    return(ConnInd);
}

int SdmCreateConnLink(SDM_MATRIX *Matrix, int SrcInd, int DstInd)
{
    int *PtrInd, NewInd;
    BYTE *DstAddr;

    if (Matrix->ConnFree>=Matrix->ConnCount)
    {
        ERR_DEB(ERR_SDM,1,"Connection table full");
        return(-1);
    }
    NewInd=Matrix->ConnFree++;
    ERR_DEB(ERR_SDM,1,"New connection index %u",NewInd);
    Matrix->ConnLink[NewInd].Host=DstInd;
    DstAddr=Matrix->HostData+Matrix->HostSize*DstInd;
    PtrInd=&(Matrix->HostLink[SrcInd].Conn);
    while (*PtrInd>=0)
    {
        DstInd=Matrix->ConnLink[*PtrInd].Host;
        if (memcmp(DstAddr,Matrix->HostData+Matrix->HostSize*DstInd,Matrix->AddrSize)<0)
            break;
        PtrInd=&(Matrix->ConnLink[*PtrInd].Next);
    }
    Matrix->ConnLink[NewInd].Next=*PtrInd;
    *PtrInd=NewInd;
    return(NewInd);
}

int SdmGetHostLink(SDM_MATRIX *Matrix, BYTE *Addr)
{
    int HashInd, HostInd;

    HashInd=SdmHash(Matrix,Addr);
    while (HashInd>=0)
    {
        if ((HostInd=Matrix->HashLink[HashInd].Host)<0)
            return(-1);
        if (memcmp(Addr,Matrix->HostData+Matrix->HostSize*HostInd,Matrix->AddrSize)==0)
            return(HostInd);
        HashInd=Matrix->HashLink[HashInd].Next;
    }
    return(-1);
}

int SdmCreateHostLink(SDM_MATRIX *Matrix, BYTE *Addr)
{
    int HostInd, NewInd, HashInd, *PtrInd;
    
    if (Matrix->HostFree>=Matrix->HostCount)
    {
        ERR_DEB(ERR_SDM,1,"Host table full");
        return(-1);
    }
    HostInd=Matrix->HostFree++;
    ERR_DEB(ERR_SDM,1,"New host index %u",HostInd);
    Matrix->HostLink[HostInd].Conn=-1;
    memcpy(Matrix->HostData+Matrix->HostSize*HostInd,Addr,Matrix->AddrSize);
    PtrInd=&(Matrix->HostUsed);
    while (*PtrInd>=0)
    {
        if (memcmp(Addr,Matrix->HostData+Matrix->HostSize*(*PtrInd),Matrix->AddrSize)<0)
            break;
        PtrInd=&(Matrix->HostLink[*PtrInd].Next);
    }
    Matrix->HostLink[HostInd].Next=*PtrInd;
    *PtrInd=HostInd;
    HashInd=SdmHash(Matrix,Addr);
    if (Matrix->HashLink[HashInd].Host==-1)
    {
        ERR_DEB(ERR_SDM,1,"New hash index %u",HashInd);
        Matrix->HashLink[HashInd].Host=HostInd;
        Matrix->HashLink[HashInd].Next=-1;
    }
    else
    {
        if (Matrix->HashFree>=Matrix->HashCount)
        {
            ERR_DEB(ERR_SDM,1,"Hash table full");
            return(-1);
        }
        NewInd=Matrix->HashFree++;
        ERR_DEB(ERR_SDM,1,"New hash index %u",NewInd);
        Matrix->HashLink[NewInd].Host=HostInd;
        Matrix->HashLink[NewInd].Next=Matrix->HashLink[HashInd].Next;
        Matrix->HashLink[HashInd].Next=NewInd;
    }
    return(HostInd);
}



void *SdmGetHost(SDM_MATRIX *Matrix, BYTE *Addr)
{
    int Ind;
    
    if ((Ind=SdmGetHostLink(Matrix,Addr))<0)
        return(NULL);
    return(Matrix->HostData + Matrix->HostSize*Ind);
}

void *SdmFirstHost(SDM_MATRIX *Matrix)
{
    int Ind;
    
    if ((Ind=Matrix->HostUsed)<0)
        return(NULL);
    return(Matrix->HostData + Matrix->HostSize*Ind);
}

void *SdmNextHost(SDM_MATRIX *Matrix, BYTE *Addr)
{
    int Ind;
    
    if ((Ind=SdmGetHostLink(Matrix,Addr))<0)
        return(NULL);
    if ((Ind=Matrix->HostLink[Ind].Next)<0)
        return(NULL);
    return(Matrix->HostData + Matrix->HostSize*Ind);
}

void *SdmCreateHost(SDM_MATRIX *Matrix, BYTE *Addr, int *New)
{
    int Ind;
         
    *New=0;
    if ((Ind=SdmGetHostLink(Matrix,Addr))<0)
    {
        *New=1;
        if ((Ind=SdmCreateHostLink(Matrix,Addr))<0)
            return(NULL);
    }        
    return(Matrix->HostData+Matrix->HostSize*Ind);
}
    
    
SDM_CONN_DATA *SdmGetConn(SDM_MATRIX *Matrix, BYTE *SrcAddr, BYTE *DstAddr)
{
    static SDM_CONN_DATA Data;
    int SrcInd, DstInd, ConnInd;
    
    if ((SrcInd=SdmGetHostLink(Matrix,SrcAddr))<0)
        return(NULL);
    if ((DstInd=SdmGetHostLink(Matrix,DstAddr))<0)
        return(NULL);
    if ((ConnInd=SdmGetConnLink(Matrix,SrcInd,DstInd))<0)
        return(NULL);
    Data.Src=Matrix->HostData + Matrix->HostSize*SrcInd;
    Data.Dst=Matrix->HostData + Matrix->HostSize*DstInd;
    Data.Conn=Matrix->ConnData + Matrix->ConnSize*ConnInd;
    return(&Data);
}

SDM_CONN_DATA *SdmFirstConn(SDM_MATRIX *Matrix)
{
    static SDM_CONN_DATA Data;
    int SrcInd, DstInd, ConnInd;
    
    if ((SrcInd=Matrix->HostUsed)<0)
        return(NULL);
    ConnInd=Matrix->HostLink[SrcInd].Conn;
    while (ConnInd<0)
    {
        if ((SrcInd=Matrix->HostLink[SrcInd].Next)<0)
            return(NULL);
        ConnInd=Matrix->HostLink[SrcInd].Conn;
    }
    DstInd=Matrix->ConnLink[ConnInd].Host;
    Data.Src=Matrix->HostData + Matrix->HostSize*SrcInd;
    Data.Dst=Matrix->HostData + Matrix->HostSize*DstInd;
    Data.Conn=Matrix->ConnData + Matrix->ConnSize*ConnInd;
    return(&Data);
}

SDM_CONN_DATA *SdmNextConn(SDM_MATRIX *Matrix, BYTE *SrcAddr, BYTE *DstAddr)
{
    static SDM_CONN_DATA Data;
    int SrcInd, DstInd, ConnInd;
    
    if ((SrcInd=SdmGetHostLink(Matrix,SrcAddr))<0)
        return(NULL);
    if ((DstInd=SdmGetHostLink(Matrix,DstAddr))<0)
        return(NULL);
    if ((ConnInd=SdmGetConnLink(Matrix,SrcInd,DstInd))<0)
        return(NULL);
    ConnInd=Matrix->ConnLink[ConnInd].Next;
    while (ConnInd<0)
    {
        if ((SrcInd=Matrix->HostLink[SrcInd].Next)<0)
            return(NULL);
        ConnInd=Matrix->HostLink[SrcInd].Conn;
    }
    DstInd=Matrix->ConnLink[ConnInd].Host;
    Data.Src=Matrix->HostData + Matrix->HostSize*SrcInd;
    Data.Dst=Matrix->HostData + Matrix->HostSize*DstInd;
    Data.Conn=Matrix->ConnData + Matrix->ConnSize*ConnInd;
    return(&Data);
}

SDM_CONN_DATA *SdmCreateConn(SDM_MATRIX *Matrix, BYTE *SrcAddr, BYTE *DstAddr, int *New)
{
    static SDM_CONN_DATA Data;
    int SrcInd, DstInd, ConnInd;
    
    *New=0;
    if ((SrcInd=SdmGetHostLink(Matrix,SrcAddr))<0)
    {
        *New|=1;
        if ((SrcInd=SdmCreateHostLink(Matrix,SrcAddr))<0)
            return(NULL);
    }
    if ((DstInd=SdmGetHostLink(Matrix,DstAddr))<0)
    {
        *New|=2;
        if ((DstInd=SdmCreateHostLink(Matrix,DstAddr))<0)
            return(NULL);
    }
    if ((ConnInd=SdmGetConnLink(Matrix,SrcInd,DstInd))<0)
    {
        *New|=4;
        if ((ConnInd=SdmCreateConnLink(Matrix,SrcInd,DstInd))<0)
            return(NULL);
    }
    Data.Src=Matrix->HostData + Matrix->HostSize*SrcInd;
    Data.Dst=Matrix->HostData + Matrix->HostSize*DstInd;
    Data.Conn=Matrix->ConnData + Matrix->ConnSize*ConnInd;
    return(&Data);
}
    
    
    
