/************************************************************************
 * ytsgdos.c - Convery Yarn files to SOUP offline reader format using grep
 *          syntax to select messages.  DOS version.  Removed wildcard
 *          handling because there were compiler differences which I did
 *          not want to play with right now.  A better idea would be to
 *          rebuild with WATCOM.
 * Copyright (C) 1995, Richard Curry Consulting: trindflo@fishnet.net
 * All Rights Reserved
 *
 * My software is generally available as free software under the terms
 * of the GNU General Public License.
 *
 * Parts of this software were copied from source code made available as
 * part of IBMs Developer's Connection.  IBM copyright information has
 * been retained in this file.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program in a file name 'COPYING'; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * SUMMARY:
 *    This file contains the code to translate/copy Yarn news files into
 *    soup offline reader format.  This initially is intended to allow
 *    cleanup of a corrupted Yarn newsbase.
 *
 *    To build: See build.cmd: expects IBM CSET/2 compiler.
 *
 *    To use:  see below
 *
 * REVISION HISTORY:
 *
 *   Date          Version  By    Purpose of Revision
 * --------------- ------- -----  ---------------------------------------
 * Thu  95-04-06    1.00    RRC   Initial Draft
 * Mon  95-12-25    3.00    RRC   Handle X-Newsgroups correctly
 *
 ************************************************************************/

/************************************************/                         
/* Author: G.R. Blair                           */
/*         BOBBLAIR @ AUSVM1                    */
/*         bobblair@bobblair.austin.ibm.com     */
/*                                              */
/* The following code is property of            */
/* International Business Machines Corporation. */
/*                                              */
/* Copyright International Business Machines    */
/* Corporation, 1991.  All rights reserved.     */
/************************************************/
/* Identification String to be put at the top of each AIXLIKE utility */
char zqrzqvjb[] = "AIXLIKE V3.0 by Bob Blair - Copyright (c) 1990, 1992, IBM";

/*-----------------------------------------------------------------------*/
/* Modification History:                                                 */
/*      Release 1       5/1/91                                           */
/* @1 05.08.91 changed fmf_init to find all files, even hidden           */
/* @2 10.19.91 upper casing a line of over 256 bytes caused Trap D       */
/* @3 05.22.92 failed on eof condition (trap d) reported by B. Kwan      */
/* @4 05.03.93 modified for IBM C/Set2 compiler                          */
/*-----------------------------------------------------------------------*/

/* ytsg: searches a file for a pattern.

   Usage:
          ytsg [-v][-o][-i] {Pattern | -e Pattern | -f StringFile} [YarnFile]
*/

/* The ytsg command searches the input file specified by the *YarnFile*
parameter (standard input by default) for news/mail messages matching a
pattern.  The ytsg command searches specifically for *Pattern*
parameters that are fixed strings.  The ytsg command outputs the message
containing the matched line to soup formatted files.  The pattern that
you are matching must be in the first 50,000 bytes of the file.
A real regular expression parser here might be nice, but for now it is
not to be.

The exit values of this command are:
  0   A match was found
  1   No match was found
  2   A syntax error was found or a file was inaccessible (even if matches were
      found).

Flags:
      -e Pattern    Specifies a pattern.  This works the same as a simple
                    pattern, but is useful when a pattern begins with a -.

      -f StringFile Specifies a file that contains strings to be matched.
                    (The file may specify up to 256 strings in this
                    implementation).

      -i            Ignores the case of letters when making comparisons.

      -o            Outputs the HEX offset of each message.  First message is
                    at offset 0.

      -v            Displays all lines except those that match the specified
                    pattern.
*/

/* -------------------------------------------------------------------------- */
/*  Maintenance history:                                                      */
/*           Jan 15 1991: Fixed case-insensitive search                       */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/* -------------------------------------------------------------------------- */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#define  INCL_BASE
#define  INCL_NOPM
#include <os2.h>
/* #include "fmf.h" */

// #define BUFSIZE    30000
#define BUFSIZE    50000
#define MAXPATTERNS  256
#define MAXLINE 255
#define MATCHFOUND     0
#define NOMATCHFOUND   1
#define SOMEERROR      2
#define YES            1
#define NO             0
#define BAILOUT       -1
#define INIT           0
#define NEXT           1
#define TAB           0x09
#define CR            0x0d
#define LF            0x0a
#define STDIN          0
#define NONPOLISH      0
#define REVERSE        1
#define BUFEND         (buf + BUFSIZE)

char *optstring = "e:f:iov";           /* valid option letters */
char *stdinC = "STDIN";
char *pattern[MAXPATTERNS] = {NULL};     /* up to MAXPATTERNS pattern ptrs */

                   /*----------------------------*/
                   /* Options and their defaults */
                   /*----------------------------*/
char *filespec = NULL;          /* file(s) to be searched */
char *StringFile = NULL;        /* name of file containing patterns */
int  logic = NONPOLISH;         /* search for matching or non-matching files? */
int  subtreesrch = NO;          /* search subtrees for matching files ? */
int  casesensitive = YES;       /* respect case in comparing to pattern? */
int  showhexoffset = NO;        /* show hex offset with matching message? */
ULONG ulOffset;                  /* ftell for current message start */

char readbuf[BUFSIZE];                  /* Work buffer                        */
#if 0
char inp_buf[65536];                    /* vbuf for speeding stdio access     */
char out_buf[65536];                    /* vbuf for speeding stdio access     */
#else
char out_buf[4096];                    /* vbuf for speeding stdio access     */
#endif

char szNewsGps[4096];                   /* String of newsgroups for this msg  */
char szCurrentGroup[1024];              /* Current group messages are sent to */

unsigned long ulMsgByteCount;           /* # of bytes in current message      */
unsigned long ulBytesToCopy;            /* # of bytes still to be copied      */
unsigned long ulBytesThisTime;          /* # of bytes to copy this iteration  */
unsigned long ulActualRead;             /* # of bytes actually read           */
unsigned long ulActualWrite;            /* # of bytes actually written        */
int retval;                             /* Return value from setvbuf          */
int fHaveGp;                            /* Flag implies there is a current gp */
unsigned short usMsgNum = 0;            /* Filename: 0000001.MSG, 0000002...  */

        /* Define file access structures */
FILE *inp_file;
FILE *out_file;
FILE *pfAreas, *pfNewscr;

        /* Define constant strings */
char *szAreasID = "AREAS";
char szNewsrcID[512];



                   /*----------------------------*/
                   /*    Function Prototypes     */
                   /*----------------------------*/
int init(int argc, char *argv[]);   /* Initialize from command line data */
int getpatterns(char *filename);    /* get patterns from string file */
void do_a_file(char *filename);     /* Logic for converting file to stream */
void do_STDIN(void);                /* Logic when input file is stdin      */
                                    /* Logic for matching lines in one file */
void do_the_stream(FILE *stream, char *filename);
void tell_usage(void);              /* Explain program usage */
                                    /* Parse command line */
int  getopt(int argc, char *argv[], char *opstring);

extern int optind;                  /* data exported by getopt() */
extern char *optarg;
                                    /* put out errors in a standard format */
void myerror(int rc, char *area, char *details);
extern char *myerror_pgm_name;      /* data imported by myerror */

int rexpres(char *pBuf, ULONG cbBuff);  /* Look for strings in buffer         */
int env_init(void);                 /* Grab required files, env. strings, etc */
void errex(char *mess_ptr);         /* Output and error and exit              */
int showmsg(void);                  /* Output a 'found' indicator             */
                                    /* Get a copy of 'Newsgroup:' string      */
int CopyNewsGroupToString( char *pDest, ULONG MaxSizeDest );
int SetupMail( void );              /* Allow for 'mail' and newsgroups        */
int SetupNewAREA( char *szNewGpName );  /* Sets up new '.MSG' file            */
int SubscribedTo( char *szTarget ); /* Checks to see if newsgroup in newsrc   */
int swal(unsigned char *adrs);      /* Swap 32 bits end-to-end                */
int fgetrsp(FILE *pf, char *bufr, int max); /* Read a line, discard >max      */
int strnlzap(char *sp);             /* Find newline and NULL terminate at NL  */
int finish_input(FILE *pf);         /* Read file until newline                */


int  rtrnstatus = NOMATCHFOUND;     /* value returned on exit from ytsg */

/*----------------------------------------------------------------------------*/
/*  main                                                                      */
/*                                                                            */
/*  mainline logic:                                                           */
/*      Perform initialization                                                */
/*      For each filespec on the command line                                 */
/*         Initialize a search for files that match the spec                  */
/*         For every file that does match the spec                            */
/*            Try to match the pattern                                        */
/*         Close the search for matching files to prepare for the next one.   */
/*                                                                            */
/*  None of this holds true if no filespec was specified.  In that case we    */
/*  take input from STDIN and try to match that.                              */
/*----------------------------------------------------------------------------*/
int main(int argc, char *argv[])
{
   int  i, filespecindx;

   myerror_pgm_name = "ytsg";
   if ( env_init() == BAILOUT )         /* Locate necessary files, env. etc   */
     return(SOMEERROR);
   if ( (filespecindx = init(argc, argv)) == BAILOUT)
     return(SOMEERROR);
   else
     if (filespecindx > 0)
       {
         for (i = filespecindx; i < argc; i++)
           {
            do_a_file(argv[i]);
           }
       }
     else
       do_STDIN();
   return(rtrnstatus);
}

/*----------------------------------------------------------------------------*/
/*  env_init                                                                  */
/*                                                                            */
/*  Do initialization:                                                        */
/*      Open files and check for environment strings which are necessary for  */
/*      the operation of this program.                                        */
/*                                                                            */
/*----------------------------------------------------------------------------*/
int env_init(void) {

  if ( getenv("HOME") == NULL )         /* Need HOME to find newsrc           */
    errex("HOME environment variable must be set to find newsrc file.\n");

  strcpy(szNewsrcID, getenv("HOME"));   /* Copy HOME address into name base   */
  strcat(szNewsrcID, "\\yarn\\newsrc.");  /* add filename string              */

  pfNewscr = fopen(szNewsrcID , "r");   /* Open list of subscribed newsgroups */
  if (pfNewscr == NULL) {
    printf("Attempting to open newsrc file at %s ", szNewsrcID);
    perror("");
    exit(3);
  }

  if ( access(szAreasID,0) == 0 )       /* If this file exists                */
    errex("AREAS file already exists. Terminating to avoid corruption.\n");

  pfAreas = fopen(szAreasID, "wb");     /* Open list of AREAS (part of SOUP)  */
  if (pfAreas == NULL)
    errex("Can't open AREAS file.\n");


  fHaveGp = FALSE;                      /* Flag no current group              */
  return YES;
}

/*----------------------------------------------------------------------------*/
/*  init                                                                      */
/*                                                                            */
/*  Do initialization:                                                        */
/*      Examine command line options and set internal variables accordingly.  */
/*      If a pattern wasn't specified with -f, take if from the next command  */
/*       line position; complain if there isn't one.                          */
/*      If there are remaining command line arguments, they are file specs.   */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/
int init(int argc, char *argv[])
{
    int rtrnindx = 0;
    int args, i;
#ifdef I16
    char c;
#else
    int c;
#endif
    char *p;
    int cu;

    for (i = 0; i < MAXPATTERNS; pattern[i++] = NULL); /* initialize patterns */
#ifdef I16
    while ( (c = (char)getopt(argc, argv, optstring)) != EOF)
#else
    while ( (c = getopt(argc, argv, optstring)) != EOF)
#endif
    {

        cu = toupper(c);
        switch (cu)
        {
            case 'V':    logic = REVERSE;
               break;
            case 'I':    casesensitive = NO;
               break;
            case 'O':    showhexoffset = YES;
               break;
            case 'E':    if (StringFile == NULL)
                           pattern[0] = optarg;
                         else
                           {
                              tell_usage();
                              return(BAILOUT);
                           }
               break;
            case 'F':    if (pattern[0] == NULL)
                           StringFile = optarg;
                         else
                           {
                              tell_usage();
                              return(BAILOUT);
                           }
               break;
            default:     tell_usage();
                         return(BAILOUT);
               break;

        } /* endswitch */
    }

    args = optind;
    if ( (pattern[0] == NULL) && (StringFile == NULL) )
      if  (args < argc)
        pattern[0] = argv[args++];

    if (StringFile != NULL)
      if (pattern[0] == NULL)
        {
          if (getpatterns(StringFile) == BAILOUT)
            return(BAILOUT);
        }
      else
        {
          tell_usage();
          return(BAILOUT);
        }

    if ( (pattern[0] == NULL) && (StringFile == NULL) )
      {
        tell_usage();
        return(BAILOUT);
      }

    if (args < argc)
      rtrnindx = args++;

    if (casesensitive == NO)  /* upper case the patterns if search is */
      if (pattern != NULL)              /* insensitive */
        for (i = 0; pattern[i]; i++)
          for (p = pattern[i]; *p; p++)
            *p = (char)toupper(*p);
    return(rtrnindx);
}

/*----------------------------------------------------------------------------*/
/*  getpatterns                                                               */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/
int getpatterns(char *filename)
{
   FILE *stream;
   char buff[MAXLINE], *p, *q;
   int i;

   i = 0;
   if ( (stream = fopen(filename, "r")) != NULL)           /* @4c */
     {
       while (fgets(buff, MAXLINE, stream) != NULL)
         {
           if (i >= MAXPATTERNS)
             {
               fprintf(stderr,
               "Only the first %d patterns in the pattern file will be used\n",
               MAXPATTERNS);
               break;
             }
           if ( (pattern[i] = (char *)malloc(strlen(buff) + 1)) != NULL )
             {
               for (p = pattern[i], q = buff;
                    *q && (*q != CR) && (*q != LF);
                    *p++ = *q++);
               *p = '\0';
               i++;
             }
           else
             {
               myerror(ERROR_NOT_ENOUGH_MEMORY, "getting space for patterns", "");
               return(BAILOUT);
             }
         }
       fclose(stream);
     }
   else
     {
       printf("Could not open StringFile %s\n", filename);
       return(BAILOUT);
     }
   return(YES);
}
/*----------------------------------------------------------------------------*/
/*  do_a_file                                                                 */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/
void do_a_file(char *filename)
{
    FILE *stream;

    if ( (stream = fopen(filename, "rb"))  != NULL)
        {
          do_the_stream(stream, filename);
          fclose(stream);
        }
      else
        {
          rtrnstatus = SOMEERROR;
//          perror(filename);
          myerror(0, "File open error", filename);
        }
}

/*----------------------------------------------------------------------------*/
/*  do_STDIN                                                                  */
/*                                                                            */
/*  Same thing as do_a_file(), really, except that we already know the handle.*/
/*                                                                            */
/*----------------------------------------------------------------------------*/
void do_STDIN()
{
   FILE *stream;

   if ( (stream = fdopen(STDIN, "rb"))  != NULL)
      do_the_stream(stream, stdinC);
   else
      {
        rtrnstatus = SOMEERROR;
//        perror("stdin->stream");
        myerror(0, "Opening STDIN as stream", "");
      }
}

/*----------------------------------------------------------------------------*/
/*  do_the_stream                                                             */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/
void do_the_stream(FILE *inp_file, char *filename)
{
  printf("Extracting messages from file %s: ", filename);

  for (;;) {                            /* Until end of file                  */
    if (feof(inp_file)) break;

        /* Read the byte count */
    if ( fread(&ulMsgByteCount, 1, 4, inp_file) != 4 ) {
      if (feof(inp_file)) break;
      else {
        perror("Error reading from input file");
        exit(3);
      }                                         
    }

    if ( (ulMsgByteCount & 0xffff0000) != 0 ) { /* > 64K in one message       */
      printf("Message length of %ld found\n", ulMsgByteCount);
    }

    ulBytesToCopy = ulMsgByteCount;
    ulBytesThisTime = min(ulBytesToCopy, sizeof(readbuf));

    ulOffset = ftell(inp_file);

        /* Read up to 64K of the message */
    ulActualRead = fread(readbuf, 1, ulBytesThisTime, inp_file);
    if ( ulActualRead != ulBytesThisTime ) {
      if (feof(inp_file)) {
        perror("Abnormal end of input file");
        printf("Requested: %ld, Actual: %ld -- proceed [Y/n]? ",
                                                ulBytesThisTime, ulActualRead);
        if ( tolower(getchar()) == 'n' ) exit(3);
            /* Fill remainder with ^Z */
        memset(readbuf+ulActualRead, 26, ulBytesThisTime-ulActualRead);
      }
      else {
        perror("Error reading from input file");
        exit(3);
      }                                         
    }

    if ( !rexpres(readbuf, ulActualRead) ) {  /* Look for a string match      */

        /* No match, clean up and move on to next message */
      ulBytesToCopy -= ulBytesThisTime; /* Tally the bytes so far             */

      while ( ulBytesToCopy!=0 ) {      /* Complete the read of the message   */

          /* Read up to 64K of the message */
        ulBytesThisTime = min(ulBytesToCopy, sizeof(readbuf));
        ulActualRead = fread(readbuf, 1, ulBytesThisTime, inp_file);
        if ( ulActualRead != ulBytesThisTime ) {
          if (feof(inp_file)) {
            perror("Abnormal end of input file in unselected message");
            printf("Requested: %ld, Actual: %ld\n",
                                                  ulBytesThisTime, ulActualRead);
          }
          else {
            perror("Error reading from input file in unselected message");
            exit(3);
          }
        }

        ulBytesToCopy -= ulBytesThisTime; /* Tally the successful read        */
      } /* While more than 64K worth of message needs to be skipped */

      continue;                         /* Done.  Read next message           */
    }

    showmsg();                          /* Indicate activity: message selected*/

    if ( !CopyNewsGroupToString(szNewsGps, sizeof(szNewsGps)) )
      *szNewsGps = '\0';                /* Mail message then                  */

        /* If we have established a group, and hence a .MSG file */
    if (fHaveGp) {

        /* If we are change in or out of mail mode */
      if ( ((*szNewsGps & *szCurrentGroup) == 0) &&
           ((*szNewsGps | *szCurrentGroup) != 0) ) {

        fclose(out_file);             /* New group, close current file      */
        fHaveGp = FALSE;
      }

        /* Check to see if the current message is part of this group */
      else if ( strstr(szNewsGps, szCurrentGroup) == NULL )  {

        fclose(out_file);               /* New group, close current file      */
        fHaveGp = FALSE;
      }
    }

        /* If we need to establish a group to put this message in */
    if (!fHaveGp) {
      char *szFirstGp, *szTestGp;

      if ( *szNewsGps == '\0' ) {       /* Switch into mail mode              */

        szTestGp = szNewsGps;           /* Point to NULL message              */
        SetupMail();                    /* Handle SOUP protocol               */
      }

      else {                            /* Normal mail group                  */

        /* Check each group in newsgroups list for a group we subscribe to */
        szFirstGp = strtok(szNewsGps," ,");
        for ( szTestGp=szFirstGp; szTestGp!=NULL; szTestGp=strtok(NULL," ,") )
          if (SubscribedTo(szTestGp)) break;

        /* If there is no group we are subscribed to, go with the first one */
        if ( szTestGp == NULL ) szTestGp = szFirstGp;

        SetupNewAREA(szTestGp);         /* Handle SOUP protocol               */
      }

      strcpy(szCurrentGroup, szTestGp);
      fHaveGp = TRUE;
    }

        /* Identify the new message in SOUP format */
    if ( *szCurrentGroup == '\0' ) {    /* If in mail mode, mail MSG format   */
      unsigned long ulTemp;

      ulTemp = ulMsgByteCount;
      swal((unsigned char *)&ulTemp);
      fwrite(&ulTemp, 1, 4, out_file);
    }
    else                                /* Else in usenet mode                */
      fprintf(out_file, "#! rnews %ld\n", ulMsgByteCount);

        /* Write whatever we previously read */
    ulActualWrite = fwrite(readbuf, 1, ulBytesThisTime, out_file);
    if ( ulActualWrite != ulBytesThisTime ) {
      perror("Error writing to output file");
      exit(3);
    }
    ulBytesToCopy -= ulBytesThisTime;   /* Tally the successful write         */

    while ( ulBytesToCopy!=0 ) {        /* Complete the copy of the message   */

        /* Read up to 64K of the message */
      ulBytesThisTime = min(ulBytesToCopy, sizeof(readbuf));
      ulActualRead = fread(readbuf, 1, ulBytesThisTime, inp_file);
      if ( ulActualRead != ulBytesThisTime ) {
        if (feof(inp_file)) {
          perror("Abnormal end of input file");
          printf("Requested: %ld, Actual: %ld -- Writing actual\n",
                                                ulBytesThisTime, ulActualRead);
            /* Fill remainder with ^Z */
          memset(readbuf+ulActualRead, 26, ulBytesThisTime-ulActualRead);
        }
        else {
          perror("Error reading from input file");
          exit(3);
        }
      }

        /* Write whatever we just read */
      if (ulBytesThisTime != 0) {
        ulActualWrite = fwrite(readbuf, 1, ulBytesThisTime, out_file);
        if ( ulActualWrite != ulBytesThisTime ) {
          perror("Error writing extended records to output file");
          exit(3);
        }
      }
      ulBytesToCopy -= ulBytesThisTime; /* Tally the successful write         */
    } /* While more than 64K worth of message needs to be copies */
  } /* For entire input file */

}

/*----------------------------------------------------------------------------*/
/*  rexpres(Buffer, BytesInBuffer)                                            */
/*    Search Buffer for any of the search strings.  Account for command       */
/*    flags.  Return TRUE if found, FALSE if not.                             */
/*                                                                            */
/*----------------------------------------------------------------------------*/
int rexpres(char *pBuf, ULONG cbBuff) {
  int match = NO;                       /* Collect an answer                  */
  char **pp;                            /* Array of pointers                  */

  if ( *pattern[0] == '*' ) match = YES;  /* Only wildcard in first version   */

  else for (pp=pattern; !match && *pp; pp++) {  /* For every pattern          */
    char *residue = pBuf;               /* Sliding window through buffer      */
    ULONG cbResidue = cbBuff;           /* # of bytes in the remaining tail   */
    char *pFirstChar;                   /* Search pointer                     */

    if (casesensitive) {                /* Simple case...look for exact str   */

      while(cbResidue) {                /* While there are chars in the tail  */
        ULONG cbSS;                     /* Bytes in search string             */

        pFirstChar = memchr(residue, **pp, cbResidue);
        if (pFirstChar == NULL) break;  /* First char not found, no match     */

        cbResidue -= (pFirstChar - residue);  /* Get size of remaining buffer */
        residue = pFirstChar;           /* Update pointer -- slide window     */

        cbSS = strlen(*pp);             /* Read total # chars in buffer       */
        if ( cbSS > cbResidue) break;   /* Can not possibly match now         */

        if ( memcmp(*pp, residue, cbSS) == 0 ) {  /* If an exact match        */
          match = YES;
          break;
        }

        cbResidue--; residue++;         /* Not the start of a match, try next */
      } /* While searching the buffer */
    } /* If case sensitive */

    else {                              /* Case insensitive -- complicated    */

      while(cbResidue) {                /* While there are chars in the tail  */
        char *pBuffer, *pSrch;          /* Work pointers                      */
        char *pLower, *pUpper;          /* Additional search pointers         */
        int cbSrch;                     /* Bytes in search string             */

            /* Check for first character in either case */
        pUpper = memchr(residue, **pp, cbResidue);
        pLower = memchr(residue, tolower(**pp), cbResidue);

        pFirstChar = min(pLower, pUpper);   /* Go with lowest address         */
        if (!pFirstChar)                /* If one of the addresses was NULL   */
          pFirstChar = max(pLower, pUpper); /* Go with the other              */
        if (!pFirstChar)                /* If both of the addresses are NULL  */
          break;                        /* Nothing of interest left in buf    */

        cbResidue -= (pFirstChar - residue);  /* Get size of remaining buffer */
        residue = pFirstChar;           /* Update pointer -- slide window     */

        match = YES;                    /* Now assume match was found         */
        for (pBuffer=residue, pSrch=*pp, cbSrch=cbResidue;
              *pSrch && cbSrch; pBuffer++, pSrch++, cbSrch-- )
                                        /* And search a char at a time        */
          if ( toupper(*pBuffer) != *pSrch)   /* For a mismatch               */
            break;

        if ( *pSrch ) match = NO;       /* Search string was not exhausted    */

        if (match) break;

        cbResidue--; residue++;         /* Not the start of a match, try next */
      } /* While searching the buffer */
    } /* else case insensitive */
  } /* else for every pattern until a match */

  if (logic == REVERSE) match = !match; /* handle reversing logic             */
  return match;
}


/******************************************************************************/
/* Major subroutines                                                          */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/*  SubscribedTo: Search the newsrc file in an attempt to locate szTarget     */
/*  as a subscribed newsgroup.  Checks for (!) unsubscribed groups.           */
/*----------------------------------------------------------------------------*/
char szNewsrcLine[1024];

int SubscribedTo( char *szTarget ) {
  char *pName;

  fseek(pfNewscr, 0, SEEK_SET);         /* Point to top of file               */

        /* While there is a line to get, read beginning of line */
  while ( fgetrsp(pfNewscr, szNewsrcLine, sizeof(szNewsrcLine)-1) )
  {
    pName = strstr( szNewsrcLine, szTarget );
    if ( pName == NULL ) continue;      /* Not this line, try another         */

    pName += strlen(szTarget);          /* Point just past the string         */
    while ( (*pName == ' ') || (*pName == '\t') ) pName++;  /* Skip whitespace*/

    if (*pName == ':') return TRUE;     /* This is a group we subscribe to    */
    if (*pName == '!') return FALSE;    /* This is a group we 'unsubscribed'  */
      /* Or...it was something else! */
  }

  return FALSE;                         /* Never found this group             */
}

/*----------------------------------------------------------------------------*/
/*  SetupNewAREA: Open a new MSG file, make entry in AREAS file, etc.         */
/*----------------------------------------------------------------------------*/
int SetupNewAREA( char *szNewGpName ) {
  char szOutFileName[16];

        /* Log new MSG file to AREAS */
  fprintf(pfAreas, "%07d\t%s\tun\n", ++usMsgNum, szNewGpName);

        /* Generate filename, open and setup output file */
  sprintf(szOutFileName, "%07d.MSG", usMsgNum);

  out_file = fopen(szOutFileName, "wb");
  if (out_file == NULL)
    errex("Can't open new output file.\n");

  retval=setvbuf(out_file, out_buf, _IOFBF, sizeof(out_buf) );
  if (retval!=0) errex("Unable to set output buffer\n");

  return TRUE;
}

/*----------------------------------------------------------------------------*/
/*  SetupMail: Open a new MSG file for mail, etc.                             */
/*----------------------------------------------------------------------------*/
int SetupMail( void ) {
  char szOutFileName[16];

        /* Log new MSG file to AREAS */
  fprintf(pfAreas, "%07d\tEmail\tbn\n", ++usMsgNum);

        /* Generate filename, open and setup output file */
  sprintf(szOutFileName, "%07d.MSG", usMsgNum);

  out_file = fopen(szOutFileName, "wb");
  if (out_file == NULL)
    errex("Can't open new output file.\n");

  retval=setvbuf(out_file, out_buf, _IOFBF, sizeof(out_buf) );
  if (retval!=0) errex("Unable to set output buffer\n");

  return TRUE;
}


/*----------------------------------------------------------------------------*/
/*  CopyNewsGroupToString: Locate 'Newsgroups:' string and copy to a buffer.  */
/*  must leave original to be copies to MSG file.                             */
/*----------------------------------------------------------------------------*/
char szNewsgpID[] = "Newsgroups:";

int CopyNewsGroupToString( char *pDest, ULONG MaxSizeDest ) {
  char *pGps, *pGpsEnd, *pGpsFault;
  unsigned long ulGpsLen;

  pGps = readbuf;                       /* Initialize search pointer          */
  for (;;) {                            /* Search for correct or NULL string  */

    pGps = strstr( pGps, szNewsgpID );
    if ( pGps == NULL ) return FALSE;   /* No list of groups found            */
    if ( (pGps == readbuf) ||           /* If string starts at top            */
         ( *(pGps-1) == '\n' ) ||       /* OR the beginning of a line         */
         ( *(pGps-1) == ' ' ) )         /* OR is not part of a larger string  */
      break;                            /* Then we have our string            */
    pGps++;                             /* Otherwise, search again            */
  }

      /* Check for end of header (Two \n's in a row) before newsgroup string */
  pGpsFault = strstr( readbuf, "\n\n" );
  if ( (pGpsFault!=NULL)  && (pGps > strstr(readbuf, "\n\n")) ) return FALSE;

  if ( (pGps != readbuf) &&             /* If string did not start at top     */
       ( *(pGps-1) != '\n' ) ) {        /* AND did not start line             */
    printf("Warning, 'Newsgroups:' identifier not at beginning of line\n");
  }

  pGps += sizeof(szNewsgpID)-1;         /* Look past the Identifier           */

  pGpsEnd = strchr(pGps, '\n');         /* Find end of newsgroups line        */
  if ( pGpsEnd == NULL ) return FALSE;  /* No end of line???                  */

  ulGpsLen = pGpsEnd-pGps;              /* Calculate the length of the line   */
  ulGpsLen = min(ulGpsLen, MaxSizeDest-1);  /* Limit the size of the string   */

  memcpy(pDest, pGps, ulGpsLen);        /* Copy the string                    */
  *(pDest+ulGpsLen) = '\0';             /* NULL terminate                     */

  return TRUE;
}


/******************************************************************************/
/* Minor subroutines                                                          */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/*  fgetrsp: read a line.  Chars beyond 'max' are discarded.  Entire line is  */
/*  guaranteed read.  Buffer is guaranteed NULL terminated.  Returns FALSE if */
/*  no line is read (presumably end of file).                                 */
/*----------------------------------------------------------------------------*/
int fgetrsp(FILE *pf, char *bufr, int max) {

  if ( fgets(bufr, max, pf) == NULL )   /* Get a 'line'                       */
    return FALSE;                       /* Normal termination mode            */

  if (!strnlzap(bufr))                  /* Attempt to kill terminating newline*/
    finish_input(pf);                   /* Not found...complete reading line  */

return TRUE;
}

/*----------------------------------------------------------------------------*/
/*  strnlzap: Look for newline in a buffer replace it with NULL (zap it).     */
/*  return TRUE if newline was found, FALSE if not.                           */
/*----------------------------------------------------------------------------*/
int strnlzap(char *sp) {

  while (*sp != '\n')                   /* Look for the newline               */
    if (!(*sp++))                       /* If we hit end of string, rtn FALSE */
      return FALSE;
  *sp = 0;                              /* Kill string at newline */
  return TRUE;                          /* Flag that we did our job */
}

/*----------------------------------------------------------------------------*/
/*  finish_input: read the file until newline or EOF is read.                 */
/*----------------------------------------------------------------------------*/
char zaparray[256];
int finish_input(FILE *pf)
{
  for (;;) {
    if ( fgets(zaparray, sizeof(zaparray), pf) == NULL ) return FALSE;
    if ( strchr(zaparray, '\n') ) return TRUE;
  }
}

/*----------------------------------------------------------------------------*/
/*  swal: Swap a long end to end (from big to little endian).                 */
/*  call with the pointer to the long to be converted.                        */
/*----------------------------------------------------------------------------*/
int swal(unsigned char *adrs) {
  unsigned char tempbyte;

  tempbyte  = *(adrs+3);
  *(adrs+3) = *adrs;
  *adrs++   = tempbyte;
  tempbyte  = *(adrs+1);
  *(adrs+1) = *adrs;
  *adrs     = tempbyte;
  return YES;
}



/*----------------------------------------------------------------------------*/
/*  errex: Prints an error message to the screen and does 'serious' error exit*/
/*----------------------------------------------------------------------------*/
void errex(char *mess_ptr) {
  printf(mess_ptr);
  exit(3);
}

/*----------------------------------------------------------------------------*/
/*  showmsg : Give user a warm fuzzy when we select a message.  If -o option  */
/*    then print HEX offset of the message within source file.                */
/*                                                                            */
/*----------------------------------------------------------------------------*/
int showmsg(void) {
  if (showhexoffset)
        printf("0x%lX-", ulOffset);
  else  putchar('.');
  return YES;
}

/*----------------------------------------------------------------------------*/
/*  tell_usage                                                                */
/*                                                                            */
/*----------------------------------------------------------------------------*/
void tell_usage()
{
  printf("\n      Adapted from software provided by IBM, Copyright 1990, 1992\n");
  printf("ytsg searches a Yarn newsbase for a pattern and makes SOUP.\n\n");
  printf("Usage:\n");
  printf("    ytsg [-vio] {Pattern | -e Pattern | -f StringFile} [YarnFile]\n");

  printf("\nFlags:\n");
  printf("    -e Pattern    Specifies a pattern.\n");
  printf("    -f StringFile Specifies a file that contains strings to be matched.\n");
  printf("    -i            Ignores the case of letters when making comparisons.\n");
  printf("    -o            Output the offset of each msg as found\n");
  printf("    -v            Displays all lines except those that match pattern.\n");
  printf("    -y            Ignore case of letters when making comparisons.\n");
}

