// Music driver code file (MUSDRV.CPP)
//
// Written 1996, 1997 by Roland Acton - public domain.
//
// This file is part of the Game Music System 1.1 distribution.

#define COMPILING_MUSDRV

#include "globals.h"
#include <stdio.h>
#ifdef TARGET_WIN95
  #include <dos.h>
#else
  #ifdef TARGET_MSDOS
    #include <dos.h>
    #ifdef USING_GCC
      #include <sys/segments.h>
    #endif
  #else
    #include "msonly.h"
  #endif
#endif
#ifdef USING_WATCOM
  #include <conio.h>
#endif
#include "inst.h"
#include "block.h"
#include "song.h"
#include "musdrv.h"
#ifdef EDITOR_HOOKS
  #include "system.h"
  #include "statbar.h"
#endif



// Sets up the play routine's internal variables.
void music_driver::set_up_driver(song::sound_cards sound_sys,
    unsigned int sb_base, unsigned int sb_int_num, unsigned int dma_chan) {
  int octavestep, notestep, tablestep;
  unsigned int volume1_step, volume2_step;
  
  tablestep = 1;
  for (octavestep = 0;octavestep < 8;octavestep++)
    for (notestep = 0;notestep < 12;notestep++) {
      note_to_freq_table[tablestep].oct_freq.bitmap.key_on = NO;
      note_to_freq_table[tablestep].oct_freq.bitmap.octave = octavestep;
      note_to_freq_table[tablestep].oct_freq.bitmap.freq_hi =
        (note_freqs[notestep] & 0xF00) >> 8;
      note_to_freq_table[tablestep].freq_lo = note_freqs[notestep] & 0x0FF;
      tablestep++;
    }
  for (volume1_step = 0;volume1_step < 64;volume1_step++)
    for (volume2_step = 0;volume2_step < 64;volume2_step++)
      volume_table[volume1_step][volume2_step] = (unsigned int)
        (((double) volume1_step / 63) * ((double) volume2_step / 63) * 63);
  for (tablestep = 0;tablestep < MAX_TRACKS;tablestep++) {
    channel_instrument[tablestep] = 0;
#ifdef EDITOR_HOOKS
    track_active[tablestep] = YES;
#endif
  }
  reset_accumulated_ticks();
#ifdef TARGET_WIN95
#else
  #ifdef USING_BORLAND
    old_timer_interrupt = 0;
    old_dsp_interrupt = 0;
  #endif
  #ifdef USING_GCC
    old_PM_timer_interrupt.pm_offset = 9999;
    old_PM_dsp_interrupt.pm_offset = 9999;
  #endif
  #ifdef USING_WATCOM
    old_timer_interrupt = 0;
    old_dsp_interrupt = 0;
  #endif
#endif
  should_song_play = NO;
  forbid_count = 0;
#ifdef EDITOR_HOOKS
  play_block_mode = NO;
  show_interrupt = NO;
#endif
  global_volume = 63;
  song::soundsystem = sound_sys;
  sb_base_address = sb_base;
  sb_interrupt_number = sb_int_num;
  dma_channel = dma_chan;
  is_sample_playing = NO;
  song::in_subroutine = NO;
  subroutine_just_ended = NO;
  song::effect_list = 0;
  handling_effects = NO;
}



// Writes a value to one of the Adlib card's registers.
void music_driver::write_adlib(unsigned char reg, unsigned char value) {
  int stepper;

  if (shadow_regs_left[reg] != value) {
    shadow_regs_left[reg] = value;
    outportb(0x388, reg);
    for (stepper = 6;stepper > 0;stepper--)
      inportb(0x388);
    outportb(0x389, value);
    for (stepper = 35;stepper > 0;stepper--)
      inportb(0x388);
  }
}



// Writes a value to one of the Soundblaster card's stereo left registers.
void music_driver::write_sb_left(unsigned char reg, unsigned char value) {
  int stepper;

  if (shadow_regs_left[reg] != value) {
    shadow_regs_left[reg] = value;
    outportb(sb_base_address, reg);
    if (song::soundsystem < song::sbpro2) {
      for (stepper = 6;stepper > 0;stepper--)
        inportb(sb_base_address);
      outportb(sb_base_address + 1, value);
      for (stepper = 35;stepper > 0;stepper--)
        inportb(sb_base_address);
    }
    else {
      outportb(sb_base_address + 1, value);
      inportb(sb_base_address);   /* Burn a little time just in case the user
                                     has a VERY fast computer. */
    }
  }
}



// Writes a value to one of the Soundblaster card's stereo right registers.
void music_driver::write_sb_right(unsigned char reg, unsigned char value) {
  int stepper;

  if (shadow_regs_right[reg] != value) {
    shadow_regs_right[reg] = value;
    outportb(sb_base_address + 2, reg);
    if (song::soundsystem < song::sbpro2) {
      for (stepper = 6;stepper > 0;stepper--)
        inportb(sb_base_address);
      outportb(sb_base_address + 3, value);
      for (stepper = 35;stepper > 0;stepper--)
        inportb(sb_base_address);
    }
    else {
      outportb(sb_base_address + 3, value);
      inportb(sb_base_address);   /* Burn a little time just in case the user
                                     has a VERY fast computer. */
    }
  }
}



/* *** Something has GOT to be done about the size of these functions!!! The
   switch statements are making the code explode! For 2.0, maybe collapse
   the write_card functions into one that takes a parameter. More
   comparisons, but probably better cache coherency. */
/* For FM instruments, sets the given hardware channel up with the
   instrument's FM definition. For samples, starts the DSP playing the
   sample. */
void music_driver::load_instrument(unsigned int inst_number,
    unsigned int channel) {
  int mod1_offset, mod2_offset;
  instrument *this_inst;
  unsigned long int this_sample_piece_size;
  scaling_volume_union sca_vol_1, sca_vol_2;
  unsigned char stereo_masks;

  if (inst_number >= MAX_INSTRUMENTS)
    inst_number = 0;
  this_inst = song::instrument_pointer[inst_number];
  if (this_inst->has_synth_component) {
// A type_synth instrument.
    channel_instrument[channel] = inst_number;
       /* In 1.1, only record the instrument number for an FM instrument,
          don't record it for a sample instrument. */
    sca_vol_2 = this_inst->scaling_volume2;
    primary_volume[channel] = 63 - sca_vol_2.bitmap.volume;
    sca_vol_2.bitmap.volume = 63 - volume_table[global_volume]
      [primary_volume[channel]];
    sca_vol_1 = this_inst->scaling_volume1;
    if (this_inst->op1mod_feedback.bitmap.additive_synthesis) {
      secondary_volume[channel] = 63 - sca_vol_1.bitmap.volume;
      sca_vol_1.bitmap.volume = 63 - volume_table[global_volume]
        [secondary_volume[channel]];
    }
    if (song::percussion_mode && channel + 5 > song::highest_track) {
// One of the special percussion tracks.
      switch (song::highest_track - channel) {
        case 4:   // Bass drum
          mod1_offset = 0x10;
          mod2_offset = 0x13;
          switch (song::soundsystem) {
            case song::adlib:
              write_adlib(AM_VIB_OFFSET+mod2_offset, this_inst->am_vib2.byte);
              write_adlib(SCALING_VOLUME_OFFSET+mod2_offset, sca_vol_2.byte);
              write_adlib(A_D_OFFSET+mod2_offset, this_inst->a_d2.byte);
              write_adlib(S_R_OFFSET+mod2_offset, this_inst->s_r2.byte);
              write_adlib(WAVE_OFFSET+mod2_offset, this_inst->wave2.byte);
            default:
              write_sb_left(AM_VIB_OFFSET+mod2_offset, this_inst->am_vib2.byte);
              write_sb_left(SCALING_VOLUME_OFFSET+mod2_offset, sca_vol_2.byte);
              write_sb_left(A_D_OFFSET+mod2_offset, this_inst->a_d2.byte);
              write_sb_left(S_R_OFFSET+mod2_offset, this_inst->s_r2.byte);
              write_sb_left(WAVE_OFFSET+mod2_offset, this_inst->wave2.byte);
          }
          break;
        case 3:   // Snare drum
          mod1_offset = 0x14;
          primary_volume[channel] = secondary_volume[channel];
             // Use operator 1's volume as the "real" volume.
          break;
        case 2:   // Tom-tom
          mod1_offset = 0x12;
          primary_volume[channel] = secondary_volume[channel];
             // Use operator 1's volume as the "real" volume.
          break;
        case 1:   // Cymbal
          mod1_offset = 0x15;
          primary_volume[channel] = secondary_volume[channel];
             // Use operator 1's volume as the "real" volume.
          break;
        case 0:   // Hi-hat
          mod1_offset = 0x11;
          primary_volume[channel] = secondary_volume[channel];
             // Use operator 1's volume as the "real" volume.
          break;
      }
      switch (song::soundsystem) {
        case song::adlib:
          write_adlib(AM_VIB_OFFSET+mod1_offset, this_inst->am_vib1.byte);
          write_adlib(SCALING_VOLUME_OFFSET+mod1_offset, sca_vol_1.byte);
          write_adlib(A_D_OFFSET+mod1_offset, this_inst->a_d1.byte);
          write_adlib(S_R_OFFSET+mod1_offset, this_inst->s_r1.byte);
          write_adlib(WAVE_OFFSET+mod1_offset, this_inst->wave1.byte);
          break;
        default:
          write_sb_left(AM_VIB_OFFSET+mod1_offset, this_inst->am_vib1.byte);
          write_sb_left(SCALING_VOLUME_OFFSET+mod1_offset, sca_vol_1.byte);
          write_sb_left(A_D_OFFSET+mod1_offset, this_inst->a_d1.byte);
          write_sb_left(S_R_OFFSET+mod1_offset, this_inst->s_r1.byte);
          write_sb_left(WAVE_OFFSET+mod1_offset, this_inst->wave1.byte);
          break;
      }
    }
    else {
// A normal melodic track.
      if (channel > 8) {
        mod1_offset = channel_offset[channel - 9];
        mod2_offset = channel_offset[channel - 9 + 9];
      }
      else {
        mod1_offset = channel_offset[channel];
        mod2_offset = channel_offset[channel + 9];
      }
/* The performance increase is worth the space all this duplicated code
   takes. */
      switch (song::soundsystem) {
        case song::adlib:
          write_adlib(AM_VIB_OFFSET+mod1_offset, this_inst->am_vib1.byte);
          write_adlib(AM_VIB_OFFSET+mod2_offset, this_inst->am_vib2.byte);
          write_adlib(SCALING_VOLUME_OFFSET+mod1_offset, sca_vol_1.byte);
          write_adlib(SCALING_VOLUME_OFFSET+mod2_offset, sca_vol_2.byte);
          write_adlib(A_D_OFFSET+mod1_offset, this_inst->a_d1.byte);
          write_adlib(A_D_OFFSET+mod2_offset, this_inst->a_d2.byte);
          write_adlib(S_R_OFFSET+mod1_offset, this_inst->s_r1.byte);
          write_adlib(S_R_OFFSET+mod2_offset, this_inst->s_r2.byte);
          write_adlib(OP1MOD_FEEDBACK_OFFSET+channel,
            this_inst->op1mod_feedback.byte);
          write_adlib(WAVE_OFFSET+mod1_offset, this_inst->wave1.byte);
          write_adlib(WAVE_OFFSET+mod2_offset, this_inst->wave2.byte);
          break;
        case song::sb:
          write_sb_left(AM_VIB_OFFSET+mod1_offset, this_inst->am_vib1.byte);
          write_sb_left(AM_VIB_OFFSET+mod2_offset, this_inst->am_vib2.byte);
          write_sb_left(SCALING_VOLUME_OFFSET+mod1_offset, sca_vol_1.byte);
          write_sb_left(SCALING_VOLUME_OFFSET+mod2_offset, sca_vol_2.byte);
          write_sb_left(A_D_OFFSET+mod1_offset, this_inst->a_d1.byte);
          write_sb_left(A_D_OFFSET+mod2_offset, this_inst->a_d2.byte);
          write_sb_left(S_R_OFFSET+mod1_offset, this_inst->s_r1.byte);
          write_sb_left(S_R_OFFSET+mod2_offset, this_inst->s_r2.byte);
          write_sb_left(OP1MOD_FEEDBACK_OFFSET+channel,
            this_inst->op1mod_feedback.byte);
          write_sb_left(WAVE_OFFSET+mod1_offset, this_inst->wave1.byte);
          write_sb_left(WAVE_OFFSET+mod2_offset, this_inst->wave2.byte);
          break;
        case song::sbpro1:
        case song::sbpro2:
          if (channel <= song::highest_stereo_left)
            stereo_masks = STEREO_LEFT_MASK;
          else
            stereo_masks = 0;
          if (channel >= song::lowest_stereo_right)
            stereo_masks |= STEREO_RIGHT_MASK;
          if (channel > 8) {
            write_sb_right(AM_VIB_OFFSET+mod1_offset, this_inst->am_vib1.byte);
            write_sb_right(AM_VIB_OFFSET+mod2_offset, this_inst->am_vib2.byte);
            write_sb_right(SCALING_VOLUME_OFFSET+mod1_offset, sca_vol_1.byte);
            write_sb_right(SCALING_VOLUME_OFFSET+mod2_offset, sca_vol_2.byte);
            write_sb_right(A_D_OFFSET+mod1_offset, this_inst->a_d1.byte);
            write_sb_right(A_D_OFFSET+mod2_offset, this_inst->a_d2.byte);
            write_sb_right(S_R_OFFSET+mod1_offset, this_inst->s_r1.byte);
            write_sb_right(S_R_OFFSET+mod2_offset, this_inst->s_r2.byte);
            if (song::soundsystem == song::sbpro2)
              write_sb_right(OP1MOD_FEEDBACK_OFFSET + channel - 9,
                this_inst->op1mod_feedback.byte | stereo_masks);
            else
              write_sb_right(OP1MOD_FEEDBACK_OFFSET + channel - 9,
                this_inst->op1mod_feedback.byte);
            write_sb_right(WAVE_OFFSET+mod1_offset, this_inst->wave1.byte);
            write_sb_right(WAVE_OFFSET+mod2_offset, this_inst->wave2.byte);
          }
          else {
            write_sb_left(AM_VIB_OFFSET+mod1_offset, this_inst->am_vib1.byte);
            write_sb_left(AM_VIB_OFFSET+mod2_offset, this_inst->am_vib2.byte);
            write_sb_left(SCALING_VOLUME_OFFSET+mod1_offset, sca_vol_1.byte);
            write_sb_left(SCALING_VOLUME_OFFSET+mod2_offset, sca_vol_2.byte);
            write_sb_left(A_D_OFFSET+mod1_offset, this_inst->a_d1.byte);
            write_sb_left(A_D_OFFSET+mod2_offset, this_inst->a_d2.byte);
            write_sb_left(S_R_OFFSET+mod1_offset, this_inst->s_r1.byte);
            write_sb_left(S_R_OFFSET+mod2_offset, this_inst->s_r2.byte);
            if (song::soundsystem == song::sbpro2)
              write_sb_left(OP1MOD_FEEDBACK_OFFSET+channel,
                this_inst->op1mod_feedback.byte | stereo_masks);
            else
              write_sb_left(OP1MOD_FEEDBACK_OFFSET+channel,
                this_inst->op1mod_feedback.byte);
            write_sb_left(WAVE_OFFSET+mod1_offset, this_inst->wave1.byte);
            write_sb_left(WAVE_OFFSET+mod2_offset, this_inst->wave2.byte);
          }
          break;
      }
    }
  }
#ifdef TARGET_WIN95
/* Samples are disabled under Win95. They wouldn't work right since the DSP
   interrupt hasn't been hooked. */
#else
  if (this_inst->has_sample_component) {
// A type_sample instrument.
    if (this_inst->sample_linear_address && song::soundsystem >= song::sb) {
/* Testing the linear address variable is necessary because there may not
   have been enough conventional memory to move the sample from the
   protected mode data area (if protected mode is being used). */
      cancel_sample_playback();
/* The speaker will be turned off if cancel_sample_playback() causes a DSP
   reset. So, turn it on explicitly. */
      write_dsp(0xD1);   // Turn speaker on.
      if (this_inst->presence == instrument::stereo) {
        if (song::soundsystem >= song::sbpro1) {
// Set hardware to stereo, turn off filter.
          outportb(sb_base_address + 4, 0xE);
          outportb(sb_base_address + 5,
            inportb(sb_base_address + 5) | 0x2 | 0x20);
          write_dsp(0x40);
          write_dsp(find_time_constant(this_inst->sample_freq << 1));
          is_sample_playing = YES;
          sample_piece_instrument = inst_number;
          sample_piece_address = this_inst->sample_linear_address;
          sample_piece_length = this_inst->sample_length;
          is_high_speed_mode_on = YES;
// Output a silent byte to resync the stereo hardware.
          write_dsp(0x80);
          write_dsp(1);
          write_dsp(0);
// Sample won't actually start playing until the next DSP interrupt.
        }
      }
      else {
        if (song::soundsystem >= song::sbpro1) {
// Set hardware to mono.
          outportb(sb_base_address + 4, 0xE);
          outportb(sb_base_address + 5,
            inportb(sb_base_address + 5) & 0xFD);
        }
        write_dsp(0x40);
        write_dsp(find_time_constant(this_inst->sample_freq));
#ifdef DEBUG_DISPLAYS
        char text_buffer[255];
        sprintf(text_buffer, "time_constant: %d   ",
        find_time_constant(this_inst->sample_freq));
        system_obj.write_line(0, 1, text_buffer);
#endif
        this_sample_piece_size = max_sample_piece_size
          (this_inst->sample_linear_address, this_inst->sample_length);
        is_sample_playing = YES;
        sample_piece_instrument = inst_number;
        sample_piece_address = this_inst->sample_linear_address
          + this_sample_piece_size;
        sample_piece_length = this_inst->sample_length
          - this_sample_piece_size;
        prepare_dma(this_inst->sample_linear_address, this_sample_piece_size);
        if (this_inst->sample_freq > 23000) {
// Use high speed mode.
          is_high_speed_mode_on = YES;
          write_dsp(0x48);
          write_dsp(this_sample_piece_size - 1);
          write_dsp((this_sample_piece_size - 1) >> 8);
          write_dsp(0x91);
        }
        else {
// Use normal mode.
          is_high_speed_mode_on = NO;
          write_dsp(0x14);
          write_dsp(this_sample_piece_size - 1);
          write_dsp((this_sample_piece_size - 1) >> 8);
        }
      }
    }
  }
#endif   // #ifdef TARGET_WIN95
}



// Resets the user's sound card.
void music_driver::reset_card() {
  int stepper;

  for (stepper = 0;stepper < MAX_TRACKS;stepper++) {
    last_note_number[stepper] = 0;
    channel_keyed_on[stepper] = NO;
    arpeggio_offset[stepper] = 0;
    frequency_offset[stepper] = 0;
    portamento_dest[stepper] = 0;
    last_portamento[stepper] = 0;
    vibrato_offset[stepper] = 0;
  }
// In case we are in percussion mode, or about to go into it.
  song::overall_am_vib.bitmap.bass_drum = NO;
  song::overall_am_vib.bitmap.snare_drum = NO;
  song::overall_am_vib.bitmap.tom_tom = NO;
  song::overall_am_vib.bitmap.cymbal = NO;
  song::overall_am_vib.bitmap.hi_hat = NO;
  update_am_vib();
  switch (song::soundsystem) {
    case song::adlib:
// Key-off all channels first to avoid channel lockup
      for (stepper = 0xB0;stepper < 0xB8;stepper++)
        force_write_adlib(stepper, 0);
      for (stepper = 0x20;stepper < 0xF6;stepper++)
        force_write_adlib(stepper, 0);
      force_write_adlib(0x08, 0);
      break;
    case song::sb:
      for (stepper = 0xB0;stepper < 0xB8;stepper++)
        force_write_sb_left(stepper, 0);
      for (stepper = 0x20;stepper < 0xF6;stepper++)
        force_write_sb_left(stepper, 0);
      force_write_sb_left(0x08, 0);
      break;
    default:
      for (stepper = 0xB0;stepper < 0xB8;stepper++) {
        force_write_sb_left(stepper, 0);
        force_write_sb_right(stepper, 0);
      }
      for (stepper = 0x20;stepper < 0xF6;stepper++) {
        force_write_sb_left(stepper, 0);
        force_write_sb_right(stepper, 0);
      }
      force_write_sb_left(0x08, 0);
      if (song::soundsystem == song::sbpro2) {
        force_write_sb_right(0x04, 0);
        force_write_sb_right(0x05, 0);
      }
      break;
  }
  if (song::soundsystem >= song::sb)
    reset_dsp();
}



// Prepares the user's card for playing.
void music_driver::set_up_card() {

  if (song::soundsystem == song::adlib) {
    force_write_adlib(0x01, 32);
    force_write_adlib(0xBD, song::overall_am_vib.byte);
  }
  else {
    force_write_sb_left(0x01, 32);
    force_write_sb_left(0xBD, song::overall_am_vib.byte);
  }
  if (song::soundsystem == song::sbpro2)
    force_write_sb_right(0x05, 1);
}



/* Plays the given note on the given hardware channel. If just_effects is
   YES, the note and instrument are ignored and only the effects are
   processed. */
void music_driver::play_one_note(unsigned char *note, unsigned int channel,
    unsigned int just_effects) {
  unsigned int freq_changed;
  unsigned long int curr_freq;
  unsigned int more_commands;
  freq_lo_hi this_freq_info;

  freq_changed = NO;
  if ((!just_effects) && (*note & 127)) {
// New note or key-off
    if ((*note & 127) == 127) {
      channel_keyed_on[channel] = NO;
      write_frequency(channel); // <-- Handles case of key off + inst change
    }
    else {
      if ((*note & 128) && (*(note+1) & 128)
        && check_for_portamento(note+2)) {
        portamento_dest[channel] =
          ((note_to_freq_table[*note & 127].oct_freq.bitmap.freq_hi << 8)
          + note_to_freq_table[*note & 127].freq_lo)
          << note_to_freq_table[*note & 127].oct_freq.bitmap.octave;
      }
      else {
        last_note_number[channel] = *note & 127;
        channel_keyed_on[channel] = YES;
        arpeggio_index[channel] = 0;
        arpeggio_offset[channel] = 0;
        frequency_offset[channel] = 0;
        vibrato_index[channel] = 0;
        vibrato_offset[channel] = 0;
        freq_changed = YES;
      }
    }
  }
  if (*note & 128) {
    note++;
    if ((!just_effects) && (*note & 127)) {
/* Instrument change. IMPORTANT: We cannot change the instrument while the
   channel is keyed on. Doing so will cause the channel to lock up. */
      if (channel_keyed_on[channel]) {
        channel_keyed_on[channel] = NO;
        write_frequency(channel);
        channel_keyed_on[channel] = YES;
        freq_changed = YES;
      }
      load_instrument(*note & 127, channel);
    }
    if (*note & 128) {
      more_commands = YES;
      note++;
// Do commands
      while (more_commands) {
        more_commands = (*note & 128);
        switch (*(note++) & 127) {
          case 0x0:
#ifdef EDITOR_HOOKS
            if (*note) {
#endif
              switch (arpeggio_index[channel]) {
                case 0:
                  arpeggio_offset[channel] = 0;
                  break;
                case 1:
                  arpeggio_offset[channel] = *note >> 4;
                  break;
                case 2:
                  arpeggio_offset[channel] = *note & 15;
                  break;
              }
              freq_changed = YES;
              if (++arpeggio_index[channel] == 3)
                arpeggio_index[channel] = 0;
#ifdef EDITOR_HOOKS
            }
#endif
            break;
          case 0x1:
            frequency_offset[channel] += *note;
            freq_changed = YES;
            break;
          case 0x2:
            frequency_offset[channel] -= *note;
            freq_changed = YES;
            break;
          case 0x3:
            // Portamento burns cpu cycles.
            if (*note)
              last_portamento[channel] = *note;
            this_freq_info = note_to_freq_table[last_note_number[channel]
              + arpeggio_offset[channel]];
            curr_freq = (((this_freq_info.oct_freq.bitmap.freq_hi
              << 8) + this_freq_info.freq_lo)
              << this_freq_info.oct_freq.bitmap.octave)
              + frequency_offset[channel];
            if (curr_freq < portamento_dest[channel]) {
              if (curr_freq + last_portamento[channel]
                > portamento_dest[channel])
                frequency_offset[channel] += portamento_dest[channel]
                  - curr_freq;
              else
                frequency_offset[channel] += last_portamento[channel];
            }
            else {
              if (curr_freq - last_portamento[channel]
                < portamento_dest[channel])
                frequency_offset[channel] += portamento_dest[channel]
                  - curr_freq;
              else
                frequency_offset[channel] -= last_portamento[channel];
            }
            freq_changed = YES;
            break;
          case 0x4:
            if (*note) {   // If zero, old values will be kept.
              vibrato_speed[channel] = *note >> 4;
              vibrato_depth[channel] = *note & 15;
            }
/* This implementation causes a new note's first tick to be played with 0
   vibrato. */
            if (vibrato_index[channel] > 31)
              vibrato_offset[channel] =
                -((vibrato_table[vibrato_index[channel] - 32]
                * vibrato_depth[channel]) / VIBRATO_DIVISOR);
            else
              vibrato_offset[channel] =
                (vibrato_table[vibrato_index[channel]]
                * vibrato_depth[channel]) / VIBRATO_DIVISOR;
            vibrato_index[channel] += vibrato_speed[channel];
            vibrato_index[channel] &= 63;   // Make it turn over at 63.
            freq_changed = YES;
            break;
          case 0xD:
            if (*note & 240)
              shift_volume(channel, (*note & 240) >> 4);
            else
              shift_volume(channel, -(*note & 15));
            break;
          default:
// These commands only need to be done when the note is first played.
            if (!just_effects)
              switch (*(note-1) & 127) {
                case 0x9:
                  song::secondary_tempo = *note;
                  break;
                case 0xC:
                  if (*note > 63)
                    set_volume(channel, 63);
                  else
                    set_volume(channel, *note);
                  break;
                case 0xB:
                  jump_indicator = *note;
                  break;
                case 0xF:
                  if (*note) {
                    if (*note == 0xFF) {
                      if (handling_effects)
                        effect_finished = YES;
                      else
                        if (song::in_subroutine)
                          subroutine_just_ended = YES;
                        else
                          stop_playing();
                    }
                    else
                      song::tempo = *note;
                  }
                  else {
                    if (song::current_pattern == song::highest_pattern)
                      jump_indicator = 0;
                    else
                      jump_indicator = song::current_pattern + 1;
                  }
                  break;
              }
            break;
        }
        note++;
      }
    }
  }
  if (freq_changed)
    write_frequency(channel);
}



// Writes the given channel's frequency to the sound card.
void music_driver::write_frequency(unsigned int channel) {
  int workvar;
  oct_freq_union writer;
  int workoct;
  unsigned int write_channel;

  if (frequency_offset[channel] || vibrato_offset[channel]
    || arpeggio_offset[channel]) {
// Some sort of effect (e.g. slide) has altered the frequency.
    writer.byte = note_to_freq_table[last_note_number[channel]
      + arpeggio_offset[channel]].oct_freq.byte;
    workoct = writer.bitmap.octave;
    workvar = (writer.bitmap.freq_hi << 8) +
      note_to_freq_table[last_note_number[channel]
      + arpeggio_offset[channel]].freq_lo
      + ((frequency_offset[channel] + vibrato_offset[channel]) >> workoct);
// *** Clean this up in 2.0 - notes should be stored in true Hz.
    if (workvar < 1) {
      workvar = note_freqs[0];
      workoct = 0;
    }
    while ((unsigned int) workvar < note_freqs[0]) {
      workvar = workvar << 1;
      workoct--;
    }
    while ((unsigned int) workvar > note_freqs[12]) {
      workvar = workvar >> 1;
      workoct++;
    }
    if (workoct < 0) {
      workoct = 0;
      workvar = note_freqs[0];
    }
    else
      if (workoct > 7) {
        workoct = 7;
        workvar = note_freqs[12];
      }
    writer.bitmap.octave = workoct;
    writer.bitmap.freq_hi = workvar >> 8;
  }
  else {
// The frequency has not been altered.
    workvar = note_to_freq_table[last_note_number[channel]].freq_lo;
    writer.byte = note_to_freq_table[last_note_number[channel]].oct_freq.byte;
  }
  writer.bitmap.key_on = channel_keyed_on[channel];
  if (song::percussion_mode)
    if (channel + 5 > song::highest_track) {
      switch ((int) song::highest_track - (int) channel) {
        case 4:   // Bass drum
          write_channel = 6;
          song::overall_am_vib.bitmap.bass_drum = writer.bitmap.key_on;
          break;
        case 3:   // Snare drum
          write_channel = 7;
          song::overall_am_vib.bitmap.snare_drum = writer.bitmap.key_on;
          break;
        case 2:   // Tom-tom
          write_channel = 8;
          song::overall_am_vib.bitmap.tom_tom = writer.bitmap.key_on;
          break;
        case 1:   // Cymbal
          song::overall_am_vib.bitmap.cymbal = writer.bitmap.key_on;
          update_am_vib();
          return;   // Cannot set frequency of cymbal.
        case 0:   // Hi-hat
          song::overall_am_vib.bitmap.hi_hat = writer.bitmap.key_on;
          update_am_vib();
          return;   // Cannot set frequency of hi-hat.
        default:
          return;   // Less than 5 channels in module.
      }
      update_am_vib();
      writer.bitmap.key_on = NO;   // Must always be off in percussion mode.
    }
    else
      write_channel = percussion_channel_map[channel];
  else
    write_channel = channel;
  switch (song::soundsystem) {
    case song::adlib:
      write_adlib(FREQ_LO_OFFSET + write_channel, (unsigned char) workvar);
      write_adlib(OCT_FREQ_OFFSET + write_channel, writer.byte);
      break;
    case song::sb:
      write_sb_left(FREQ_LO_OFFSET + write_channel, (unsigned char) workvar);
      write_sb_left(OCT_FREQ_OFFSET + write_channel, writer.byte);
      break;
    case song::sbpro1:
    case song::sbpro2:
      if (write_channel > 8) {
        write_sb_right(FREQ_LO_OFFSET + write_channel - 9,
          (unsigned char) workvar);
        write_sb_right(OCT_FREQ_OFFSET + write_channel - 9, writer.byte);
      }
      else {
        write_sb_left(FREQ_LO_OFFSET + write_channel, (unsigned char) workvar);
        write_sb_left(OCT_FREQ_OFFSET + write_channel, writer.byte);
      }
      break;
  }
}



/* Sets the volume of the given channel. new_volume can range from 0 to 63,
   with 63 being loudest. */
void music_driver::set_volume(unsigned int channel,
    unsigned int new_volume) {
  instrument *this_inst;

  this_inst = song::instrument_pointer[channel_instrument[channel]];
/* If the channel is using frequency modulation, only the volume of operator
   2 needs to be changed. */
  if (this_inst->op1mod_feedback.bitmap.additive_synthesis)
    do_volume(channel, new_volume, new_volume);
  else
    do_volume(channel, new_volume);
}



// Changes the volume of the given channel by the given amount.
void music_driver::shift_volume(unsigned int channel, int volume_shift) {
  int worknum_op2, worknum_op1;
  instrument *this_inst;
  
  this_inst = song::instrument_pointer[channel_instrument[channel]];
  worknum_op2 = primary_volume[channel] + volume_shift;
  if (worknum_op2 < 0)
    worknum_op2 = 0;
  else
    if (worknum_op2 > 63)
      worknum_op2 = 63;
/* If the channel is using frequency modulation, only the volume of operator
   2 needs to be changed. */
  if (this_inst->op1mod_feedback.bitmap.additive_synthesis) {
    worknum_op1 = secondary_volume[channel] + volume_shift;
    if (worknum_op1 < 0)
      worknum_op1 = 0;
    else
      if (worknum_op1 > 63)
        worknum_op1 = 63;
    do_volume(channel, worknum_op2, worknum_op1);
  }
  else
    do_volume(channel, worknum_op2);
}



/* Sets the volumes of the channel's operators to the indicated values.
   Volumes can range from 0 to 63, with 63 being loudest. If mod1_volume
   is set to 9999, operator 1's volume is left unchanged. */
void music_driver::do_volume(unsigned int channel, unsigned int mod2_volume,
    unsigned int mod1_volume) {
  scaling_volume_union sca_vol_2, sca_vol_1;
  instrument *this_inst;
  int mod1_offset, mod2_offset;

  this_inst = song::instrument_pointer[channel_instrument[channel]];
  sca_vol_2 = this_inst->scaling_volume2;
  sca_vol_2.bitmap.volume = 63 - volume_table[global_volume][mod2_volume];
  primary_volume[channel] = mod2_volume;
  if (song::percussion_mode && channel + 5 > song::highest_track) {
// One of the special percussion tracks.
    switch (song::highest_track - channel) {
      case 4:   // Bass drum
        mod1_offset = 0x10;
        mod2_offset = 0x13;
        switch (song::soundsystem) {
          case song::adlib:
            write_adlib(SCALING_VOLUME_OFFSET+mod2_offset, sca_vol_2.byte);
          default:
            write_sb_left(SCALING_VOLUME_OFFSET+mod2_offset, sca_vol_2.byte);
        }
        if (mod1_volume != 9999) {
          sca_vol_1 = this_inst->scaling_volume1;
          sca_vol_1.bitmap.volume = 63 - volume_table[global_volume]
            [mod1_volume];
          secondary_volume[channel] = mod1_volume;
          switch (song::soundsystem) {
            case song::adlib:
              write_adlib(SCALING_VOLUME_OFFSET+mod1_offset, sca_vol_1.byte);
            default:
              write_sb_left(SCALING_VOLUME_OFFSET+mod1_offset, sca_vol_1.byte);
          }
        }
        return;   // We're done.
      case 3:   // Snare drum
        mod1_offset = 0x14;
        break;
      case 2:   // Tom-tom
        mod1_offset = 0x12;
        break;
      case 1:   // Cymbal
        mod1_offset = 0x15;
        break;
      case 0:   // Hi-hat
        mod1_offset = 0x11;
        break;
    }
// mod2_volume is used here, since these instruments have only 1 operator.
    sca_vol_1 = this_inst->scaling_volume1;
    sca_vol_1.bitmap.volume = 63 - volume_table[global_volume][mod2_volume];
    switch (song::soundsystem) {
      case song::adlib:
        write_adlib(SCALING_VOLUME_OFFSET+mod1_offset, sca_vol_1.byte);
        break;
      default:
        write_sb_left(SCALING_VOLUME_OFFSET+mod1_offset, sca_vol_1.byte);
        break;
    }
  }
  else {
// A normal melodic track.
// Set operator 2's volume.
    switch (song::soundsystem) {
      case song::adlib:
        write_adlib(SCALING_VOLUME_OFFSET + channel_offset[channel + 9],
          sca_vol_2.byte);
        break;
      case song::sb:
        write_sb_left(SCALING_VOLUME_OFFSET + channel_offset[channel + 9],
          sca_vol_2.byte);
        break;
      case song::sbpro1:
      case song::sbpro2:
        if (channel > 8) {
          write_sb_right(SCALING_VOLUME_OFFSET + channel_offset
            [channel - 9 + 9], sca_vol_2.byte);
        }
        else {
          write_sb_left(SCALING_VOLUME_OFFSET + channel_offset[channel + 9],
            sca_vol_2.byte);
        }
        break;
    }
    if (mod1_volume != 9999) {
// Set operator 1's volume.
      sca_vol_1 = this_inst->scaling_volume1;
      sca_vol_1.bitmap.volume = 63 - volume_table[global_volume][mod1_volume];
      secondary_volume[channel] = mod1_volume;
      switch (song::soundsystem) {
        case song::adlib:
          write_adlib(SCALING_VOLUME_OFFSET + channel_offset[channel],
            sca_vol_1.byte);
          break;
        case song::sb:
          write_sb_left(SCALING_VOLUME_OFFSET + channel_offset[channel],
            sca_vol_1.byte);
          break;
        case song::sbpro1:
        case song::sbpro2:
          if (channel > 8) {
            write_sb_right(SCALING_VOLUME_OFFSET + channel_offset[channel - 9],
              sca_vol_1.byte);
          }
          else {
            write_sb_left(SCALING_VOLUME_OFFSET + channel_offset[channel],
              sca_vol_1.byte);
          }
          break;
      }
    }
  }
}



/* Calls play_one_note() for an entire line of the song. Handles the
   idiosyncrasies of making stereo work properly on the sbpro1. just_effects
   is passed in each call to play_one_note(). */
void music_driver::play_one_line(unsigned int just_effects) {
  int stepper;
  int farthest_right;
  int offset;
  int highest_melodic;

  if (song::percussion_mode) {
    if (song::highest_track > 4) {
// Song has melodic and percussion tracks.
      for (stepper = song::highest_track - 4
        ;stepper <= (int) song::highest_track;stepper++)
        if (song::current_note[stepper])   // Don't play an "empty" track.
#ifdef EDITOR_HOOKS
          if (track_active[stepper])
#endif
            play_one_note(song::current_note[stepper], stepper,
              just_effects);
      highest_melodic = song::highest_track - 5;
    }
    else {
// Song consists of only percussion tracks.
      for (stepper = song::highest_track;stepper >= 0;stepper--)
        if (song::current_note[stepper])   // Don't play an "empty" track.
#ifdef EDITOR_HOOKS
          if (track_active[stepper])
#endif
            play_one_note(song::current_note[stepper], stepper,
              just_effects);
      return;   // All tracks played.
    }
  }
  else
    highest_melodic = song::highest_track;
  switch (song::soundsystem) {
    case song::adlib:
    case song::sb:
      if (highest_melodic > 8)
        farthest_right = 8;
      else
        farthest_right = highest_melodic;
      for (stepper = farthest_right;stepper >= 0;stepper--)
        if (song::current_note[stepper])   // Don't play an "empty" track.
#ifdef EDITOR_HOOKS
          if (track_active[stepper])
#endif
            play_one_note(song::current_note[stepper], stepper,
              just_effects);
      break;
    case song::sbpro1:
      if (song::highest_stereo_left > 8)
        farthest_right = 8;
      else
        farthest_right = song::highest_stereo_left;
      if (farthest_right > highest_melodic)
        farthest_right = highest_melodic;
      for (stepper = farthest_right;stepper >= 0;stepper--)
        if (song::current_note[stepper])   // Don't play an "empty" track.
#ifdef EDITOR_HOOKS
          if (track_active[stepper])
#endif
            play_one_note(song::current_note[stepper], stepper,
              just_effects);
      if (highest_melodic >= (int) song::lowest_stereo_right) {
        if (highest_melodic > (int) song::lowest_stereo_right + 8)
          farthest_right = song::lowest_stereo_right + 8;
        else
          farthest_right = highest_melodic;
        offset = 9 - song::lowest_stereo_right;
        for (stepper = song::lowest_stereo_right;
          stepper <= farthest_right;stepper++)
          if (song::current_note[stepper])   // Don't play an "empty" track.
#ifdef EDITOR_HOOKS
            if (track_active[stepper])
#endif
              play_one_note(song::current_note[stepper], stepper + offset,
                just_effects);
      }
      break;
    case song::sbpro2:
      if (highest_melodic > 17)
        farthest_right = 17;
      else
        farthest_right = highest_melodic;
      for (stepper = farthest_right;stepper >= 0;stepper--)
        if (song::current_note[stepper])   // Don't play an "empty" track.
#ifdef EDITOR_HOOKS
          if (track_active[stepper])
#endif
            play_one_note(song::current_note[stepper], stepper,
              just_effects);
      break;
  }
}



/* Master update routine. Plays notes and advances lines and patterns. Needs
   to be called as often as required by the current tempo (normally this is
   done by the interrupt routine). */
void music_driver::update() {
  unsigned int pattern_memory;
  unsigned int block_memory;
  unsigned int line_memory;
  unsigned char *current_note_memory[MAX_TRACKS];
  int jump_indicator_memory;
  effect_info *effect_step;
  unsigned int effect_to_delete;

#ifdef EDITOR_HOOKS
  if (show_interrupt) {
// Set red background.
    outportb(VIDEO_PALETTE_MASK, 0xFF);
    outportb(VIDEO_PALETTE_WRITE, 0);
    outportb(VIDEO_PALETTE_DATA, 63);
    outportb(VIDEO_PALETTE_DATA, 0);
    outportb(VIDEO_PALETTE_DATA, 0);
  }
  interrupt_ticks++;
#endif
  if (should_song_play) {
    if (!forbid_count) {
      if (song::effect_list) {
        if (accumulated_ticks > song::secondary_tempo) {
          if (jump_indicator == -1)
            song::advance_line();
          else
#ifdef EDITOR_HOOKS
            if (play_block_mode)
              song::block_jump(song::current_block);
            else
#endif
              song::position_jump(jump_indicator);
          pattern_memory = song::current_pattern;
          block_memory = song::current_block;
          line_memory = song::current_line;
          memcpy(current_note_memory, song::current_note,
            sizeof(unsigned char *) * MAX_TRACKS);
          effect_step = song::effect_list;
          while (effect_step) {
            if (effect_step->jump_pattern == -1) {
              song::current_pattern = effect_step->effect_pattern;
              song::current_block = effect_step->effect_block;
              song::current_line = effect_step->effect_line;
              memcpy(song::current_note, effect_step->effect_note,
                sizeof(unsigned char *) * MAX_TRACKS);
              song::advance_line();
            }
            else {
              song::position_jump(effect_step->jump_pattern);
              effect_step->jump_pattern = -1;
              effect_step->pointers_valid = YES;
            }
            effect_step->effect_pattern = song::current_pattern;
            effect_step->effect_block = song::current_block;
            effect_step->effect_line = song::current_line;
            memcpy(effect_step->effect_note, song::current_note,
              sizeof(unsigned char *) * MAX_TRACKS);
            effect_step = effect_step->next_effect;
          }
          song::current_pattern = pattern_memory;
          song::current_block = block_memory;
          song::current_line = line_memory;
          memcpy(song::current_note, current_note_memory,
            sizeof(unsigned char *) * MAX_TRACKS);
          reset_accumulated_ticks();
        }
        if (accumulated_ticks > 1)
          play_one_line(YES);
        else
          play_one_line(NO);
        handling_effects = YES;
        jump_indicator_memory = jump_indicator;
        pattern_memory = song::current_pattern;
        block_memory = song::current_block;
        line_memory = song::current_line;
        memcpy(current_note_memory, song::current_note,
          sizeof(unsigned char *) * MAX_TRACKS);
        effect_step = song::effect_list;
        while (effect_step) {
          jump_indicator = -1;
          effect_finished = NO;
          if (effect_step->pointers_valid) {
/* The note-playing code does not directly change the current
   pattern/block/line, but may need to reference it. */
            song::current_pattern = effect_step->effect_pattern;
            song::current_block = effect_step->effect_block;
            song::current_line = effect_step->effect_line;
            memcpy(song::current_note, effect_step->effect_note,
              sizeof(unsigned char *) * MAX_TRACKS);
            if (accumulated_ticks > 1)
              play_one_line(YES);
            else
              play_one_line(NO);
            if (jump_indicator != -1)
              effect_step->jump_pattern = jump_indicator;
          }
          if (effect_finished) {
            effect_to_delete = effect_step->effect_id;
            effect_step = effect_step->next_effect;
            song::cancel_effect(effect_to_delete);
          }
          else
            effect_step = effect_step->next_effect;
        }
        song::current_pattern = pattern_memory;
        song::current_block = block_memory;
        song::current_line = line_memory;
        memcpy(song::current_note, current_note_memory,
          sizeof(unsigned char *) * MAX_TRACKS);
        jump_indicator = jump_indicator_memory;
        handling_effects = NO;
      }
      else {
        if (accumulated_ticks > song::secondary_tempo) {
          if (jump_indicator == -1)
            song::advance_line();
          else
#ifdef EDITOR_HOOKS
            if (play_block_mode)
              song::block_jump(song::current_block);
            else
#endif
             song::position_jump(jump_indicator);
          reset_accumulated_ticks();
        }
        if (accumulated_ticks > 1)
          play_one_line(YES);
        else
          play_one_line(NO);
      }
      if (subroutine_just_ended) {
        song::cancel_subroutine();
        subroutine_just_ended = NO;
      }
      else
        accumulated_ticks++;
    }
  }
#ifdef EDITOR_HOOKS
  if (show_interrupt) {
// Set black background.
    outportb(VIDEO_PALETTE_MASK, 0xFF);
    outportb(VIDEO_PALETTE_WRITE, 0);
    outportb(VIDEO_PALETTE_DATA, 0);
    outportb(VIDEO_PALETTE_DATA, 0);
    outportb(VIDEO_PALETTE_DATA, 0);
  }
#endif
}



// Calls update() once every timer interrupt.
#ifdef USING_BORLAND
void interrupt music_driver::timer_int(...) {
#endif
#ifdef USING_GCC
void music_driver::timer_int() {
#endif
#ifdef USING_WATCOM
static void __interrupt __far music_driver::timer_int() {
#endif
  static unsigned int old_tempo;
  
  update();
  if (old_tempo != song::tempo) {
    set_timer(compute_timer_setting(song::tempo));
    old_tempo = song::tempo;
  }
#ifdef USING_GCC
// DJGPP automatically chains to the next interrupt.
#else
  old_timer_interrupt();
#endif
}



// Clears DSP interrupt register and plays next sample segment.
#ifdef USING_BORLAND
void interrupt music_driver::dsp_int(...) {
#endif
#ifdef USING_GCC
void music_driver::dsp_int() {
#endif
#ifdef USING_WATCOM
static void __interrupt __far music_driver::dsp_int() {
#endif
  unsigned long int this_sample_piece_size;
  instrument *this_inst;
  unsigned long int repeat_address;

#ifdef DEBUG_DISPLAYS
  char text_buffer[255];
  sprintf(text_buffer, "dsp_int, sample_piece_length: %lu   ",
  sample_piece_length);
  system_obj.write_line(0, 0, text_buffer);
#endif
  inportb(sb_base_address + 0xE);   // Acknowledge interrupt with DSP.
  outportb(0x20, 0x20);   // Acknowledge interrupt with PIC 1.
  outportb(0xA0, 0x20);   // Acknowledge interrupt with PIC 2.
  if (is_sample_playing) {
    if (sample_piece_length) {
      this_sample_piece_size = max_sample_piece_size
        (sample_piece_address, sample_piece_length);
      prepare_dma(sample_piece_address, this_sample_piece_size);
      if (is_high_speed_mode_on) {
        write_dsp(0x48);
        write_dsp(this_sample_piece_size - 1);
        write_dsp((this_sample_piece_size - 1) >> 8);
        write_dsp(0x91);
      }
      else {
        write_dsp(0x14);
        write_dsp(this_sample_piece_size - 1);
        write_dsp((this_sample_piece_size - 1) >> 8);
      }
      sample_piece_address += this_sample_piece_size;
      sample_piece_length -= this_sample_piece_size;
    }
    else {
      this_inst = song::instrument_pointer[sample_piece_instrument];
// Check for a looping sample.
      if (this_inst->repeat_length) {
        repeat_address = this_inst->sample_linear_address
          + this_inst->repeat_start;
        this_sample_piece_size = max_sample_piece_size(repeat_address,
          this_inst->repeat_length);
        prepare_dma(repeat_address, this_sample_piece_size);
        if (is_high_speed_mode_on) {
          write_dsp(0x48);
          write_dsp(this_sample_piece_size - 1);
          write_dsp((this_sample_piece_size - 1) >> 8);
          write_dsp(0x91);
        }
        else {
          write_dsp(0x14);
          write_dsp(this_sample_piece_size - 1);
          write_dsp((this_sample_piece_size - 1) >> 8);
        }
        sample_piece_address = repeat_address + this_sample_piece_size;
        sample_piece_length = this_inst->repeat_length
          - this_sample_piece_size;
      }
      else
        is_sample_playing = NO;
    }
  }
  else {
// Should be impossible.
#ifdef EDITOR_HOOKS
    status_bar_obj.message_output("ALERT: DSP interrupt while no sample playing.");
#endif
  }
#ifdef USING_GCC
// DJGPP automatically chains to the next interrupt.
#else
  old_dsp_interrupt();
#endif
}



// Wedges timer_int() into the timer interrupt chain.
gms_function_return_codes music_driver::wedge_player() {

#ifdef TARGET_WIN95
// Under Win95, this routine does nothing.
  return gms_function_success;
#else
  #ifdef USING_BORLAND
  if (old_timer_interrupt)
    return gms_function_failure;   // Timer interrupt already installed.
  else {
    set_timer(compute_timer_setting(song::tempo));
    old_timer_interrupt = getvect(TIMER_INTERRUPT_VECTOR);
    setvect(TIMER_INTERRUPT_VECTOR, timer_int);
    return gms_function_success;
  }
  #endif
  #ifdef USING_GCC
  if (old_PM_timer_interrupt.pm_offset != 9999)
    return gms_function_failure;   // Timer interrupt already installed.
  else {
    set_timer(compute_timer_setting(song::tempo));
    _go32_dpmi_get_protected_mode_interrupt_vector(TIMER_INTERRUPT_VECTOR,
      &old_PM_timer_interrupt);
    GMS_timer_interrupt.pm_selector = _my_cs();
    GMS_timer_interrupt.pm_offset = (int) timer_int;
    _go32_dpmi_chain_protected_mode_interrupt_vector(TIMER_INTERRUPT_VECTOR,
      &GMS_timer_interrupt);
    return gms_function_success;
  }
  #endif
  #ifdef USING_WATCOM
  if (old_timer_interrupt)
    return gms_function_failure;   // Timer interrupt already installed.
  else {
    set_timer(compute_timer_setting(song::tempo));
    old_timer_interrupt = _dos_getvect(TIMER_INTERRUPT_VECTOR);
    _dos_setvect(TIMER_INTERRUPT_VECTOR, timer_int);
    return gms_function_success;
  }
  #endif
#endif   // #ifdef TARGET_WIN95
}



// Removes timer_int() from the timer interrupt chain.
gms_function_return_codes music_driver::remove_player() {

#ifdef TARGET_WIN95
// Under Win95, this routine does nothing.
  return gms_function_success;
#else
  #ifdef USING_BORLAND
  if (old_timer_interrupt) {
    set_timer(65535);
    setvect(TIMER_INTERRUPT_VECTOR, old_timer_interrupt);
    old_timer_interrupt = 0;
    return gms_function_success;
  }
  else
    return gms_function_failure;   // Timer interrupt not installed.
  #endif
  #ifdef USING_GCC
  if (old_PM_timer_interrupt.pm_offset != 9999) {
    set_timer(65535);
    _go32_dpmi_set_protected_mode_interrupt_vector(TIMER_INTERRUPT_VECTOR,
      &old_PM_timer_interrupt);
    old_PM_timer_interrupt.pm_offset = 9999;
    return gms_function_success;
  }
  else
    return gms_function_failure;   // Timer interrupt not installed.
  #endif
  #ifdef USING_WATCOM
  if (old_timer_interrupt) {
    set_timer(65535);
    _dos_setvect(TIMER_INTERRUPT_VECTOR, old_timer_interrupt);
    old_timer_interrupt = 0;
    return gms_function_success;
  }
  else
    return gms_function_failure;   // Timer interrupt not installed.
  #endif
#endif   // #ifdef TARGET_WIN95
}



// Wedges dsp_int() into the DSP interrupt chain.
gms_function_return_codes music_driver::wedge_dsp_handler() {
  unsigned char bit_mask;

#ifdef TARGET_WIN95
// Under Win95, this routine does nothing.
  return gms_function_success;
#else
  #ifdef USING_BORLAND
  if (old_dsp_interrupt)
    return gms_function_failure;   // Interrupt already installed.
  else {
    if (song::soundsystem >= song::sb) {
      if (vector_table[sb_interrupt_number]) {
        old_dsp_interrupt = getvect(vector_table[sb_interrupt_number]);
        setvect(vector_table[sb_interrupt_number], dsp_int);
        pic1memory = inportb(0x21);   // Store old interrupt masks.
        pic2memory = inportb(0xA1);   // 0x21 = INTs0-7, 0xA1 = INTs8-15.
// Turn on the interrupt for the desired channel.
        if (sb_interrupt_number < 8) {
          bit_mask = ~(1 << sb_interrupt_number);
          outportb(0x21, pic1memory & bit_mask);
        }
        else {
          bit_mask = ~(1 << (sb_interrupt_number - 8));
          outportb(0x21, pic1memory & 0xFB);   // Make sure cascade is on.
          outportb(0xA1, pic2memory & bit_mask);
        }
        return gms_function_success;
      }
      else {
        return gms_function_failure;   // Invalid SB interrupt number.
      }
    }
    else
      return gms_function_failure;   // No DSP to wedge onto.
  }
  #endif
  #ifdef USING_GCC
  if (old_PM_dsp_interrupt.pm_offset != 9999)
    return gms_function_failure;   // Interrupt already installed.
  else {
    if (song::soundsystem >= song::sb) {
      if (vector_table[sb_interrupt_number]) {
        _go32_dpmi_get_protected_mode_interrupt_vector
          (vector_table[sb_interrupt_number], &old_PM_dsp_interrupt);
        GMS_dsp_interrupt.pm_selector = _my_cs();
        GMS_dsp_interrupt.pm_offset = (int) dsp_int;
        _go32_dpmi_chain_protected_mode_interrupt_vector
          (vector_table[sb_interrupt_number], &GMS_dsp_interrupt);
        pic1memory = inportb(0x21);   // Store old interrupt masks.
        pic2memory = inportb(0xA1);   // 0x21 = INTs0-7, 0xA1 = INTs8-15.
// Turn on the interrupt for the desired channel.
        if (sb_interrupt_number < 8) {
          bit_mask = ~(1 << sb_interrupt_number);
          outportb(0x21, pic1memory & bit_mask);
        }
        else {
          bit_mask = ~(1 << (sb_interrupt_number - 8));
          outportb(0x21, pic1memory & 0xFB);   // Make sure cascade is on.
          outportb(0xA1, pic2memory & bit_mask);
        }
        return gms_function_success;
      }
      else {
        return gms_function_failure;   // Invalid SB interrupt number.
      }
    }
    else
      return gms_function_failure;   // No DSP to wedge onto.
  }
  #endif
  #ifdef USING_WATCOM
  if (old_dsp_interrupt)
    return gms_function_failure;   // Interrupt already installed.
  else {
    if (song::soundsystem >= song::sb) {
      if (vector_table[sb_interrupt_number]) {
        old_dsp_interrupt = _dos_getvect(vector_table[sb_interrupt_number]);
        _dos_setvect(vector_table[sb_interrupt_number], dsp_int);
        pic1memory = inportb(0x21);   // Store old interrupt masks.
        pic2memory = inportb(0xA1);   // 0x21 = INTs0-7, 0xA1 = INTs8-15.
// Turn on the interrupt for the desired channel.
        if (sb_interrupt_number < 8) {
          bit_mask = ~(1 << sb_interrupt_number);
          outportb(0x21, pic1memory & bit_mask);
        }
        else {
          bit_mask = ~(1 << (sb_interrupt_number - 8));
          outportb(0x21, pic1memory & 0xFB);   // Make sure cascade is on.
          outportb(0xA1, pic2memory & bit_mask);
        }
        return gms_function_success;
      }
      else {
        return gms_function_failure;   // Invalid SB interrupt number.
      }
    }
    else
      return gms_function_failure;   // No DSP to wedge onto.
  }
  #endif
#endif   // #ifdef TARGET_WIN95
}



// Removes dsp_int() from the DSP interrupt chain.
gms_function_return_codes music_driver::remove_dsp_handler() {

#ifdef TARGET_WIN95
// Under Win95, this routine does nothing.
  return gms_function_success;
#else
  #ifdef USING_BORLAND
  if (old_dsp_interrupt) {
    if (song::soundsystem >= song::sb) {
      setvect(vector_table[sb_interrupt_number], old_dsp_interrupt);
      outportb(0x21, pic1memory);
      outportb(0xA1, pic2memory);
      old_dsp_interrupt = 0;
      return gms_function_success;
    }
    else
      return gms_function_failure;   // Card doesn't have a DSP.
  }
  else
    return gms_function_failure;   // No interrupt installed.
  #endif
  #ifdef USING_GCC
  if (old_PM_dsp_interrupt.pm_offset != 9999) {
    if (song::soundsystem >= song::sb) {
      _go32_dpmi_set_protected_mode_interrupt_vector
        (vector_table[sb_interrupt_number], &old_PM_dsp_interrupt);
      outportb(0x21, pic1memory);
      outportb(0xA1, pic2memory);
      old_PM_dsp_interrupt.pm_offset = 9999;
      return gms_function_success;
    }
    else
      return gms_function_failure;   // Card doesn't have a DSP.
  }
  else
    return gms_function_failure;   // No interrupt installed.
  #endif
  #ifdef USING_WATCOM
  if (old_dsp_interrupt) {
    if (song::soundsystem >= song::sb) {
      _dos_setvect(vector_table[sb_interrupt_number], old_dsp_interrupt);
      outportb(0x21, pic1memory);
      outportb(0xA1, pic2memory);
      old_dsp_interrupt = 0;
      return gms_function_success;
    }
    else
      return gms_function_failure;   // Card doesn't have a DSP.
  }
  else
    return gms_function_failure;   // No interrupt installed.
  #endif
#endif   // #ifdef TARGET_WIN95
}



// Resets the sound card's DSP.
gms_function_return_codes music_driver::reset_dsp() {
  int counter;

  outportb(sb_base_address + 0x6, 1);
  for (counter = 6;counter > 0;counter--)
    inportb(0x388);
  outportb(sb_base_address + 0x6, 0);
#ifdef TARGET_LINUX
// The msonly.cpp function doesn't return the right value, causing a freeze.
  counter = 1;
#else
  for (counter = 150;counter > 0;counter--)
    if (read_dsp() == 0xAA)
      break;
#endif
  if (counter > 0)
    return gms_function_success;
  else
    return gms_function_failure;
}



void music_driver::set_global_volume(unsigned int new_volume) {
  int stepper;

  global_volume = new_volume;
// Change the volume of the currently playing notes.
  for (stepper = song::highest_track;stepper >= 0;stepper--)
    shift_volume(stepper, 0);
}
