UTILITIES
DOUGLAS BOLING 
PMVIEW.C

//=======================================================================
// PMVIEW.C -- A file browse program for the OS/2 Presentation Manager.
//
// Copyright (c) 1990 Ziff Communications
//
// Written by Douglas Boling
//=======================================================================

#define  INCL_WIN
#define  INCL_VIO
#define  INCL_AVIO

#define  AVIOROWS 98
#define  AVIOCOLS 255
#define  PATHLENGTH 255
#define  BUFFERSIZE 16384

#include <os2.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "pmview.h"

MRESULT  EXPENTRY ClientWndProc (HWND, USHORT, MPARAM, MPARAM);
MRESULT  EXPENTRY AboutDlgProc (HWND, USHORT, MPARAM, MPARAM);
MRESULT  EXPENTRY FileDlgProc (HWND, USHORT, MPARAM, MPARAM);
MRESULT  EXPENTRY ConfigDlgProc (HWND, USHORT, MPARAM, MPARAM);
MRESULT  EXPENTRY GotoDlgProc (HWND, USHORT, MPARAM, MPARAM);
MRESULT  EXPENTRY SearchDlgProc (HWND, USHORT, MPARAM, MPARAM);
MRESULT  EXPENTRY FontDlgProc (HWND, USHORT, MPARAM, MPARAM);

BOOL OpenFile (PCHAR);
PCHAR GetFileDataPtr (LONG lFilePtr, BOOL fForward, USHORT *usLimit,
		      BOOL *fEndOfFile);
LONG MoveViewPtr (LONG lFilePtr, SHORT *ptrRow, BOOL fDispASCII);
SHORT ComputeLines (LONG lOldPtr, LONG lNewPtr, BOOL fDispASCII);
VOID btoa (CHAR cNum, CHAR szOut[], BOOL fBlank);
VOID Myltoa (LONG lNum, CHAR szOut[]);
SHORT DisplayLine (HVPS hvps, SHORT sRow, LONG lLinePtr, BOOL fDispASCII);
BOOL SearchFile (LONG *lFilePtr);
VOID ScrollVis (HWND hwndFrame, HWND hwndHScroll, HWND hwndVScroll,
		HWND hwndMenu);
VOID SetASCII (HWND hwndMenu, LONG *lViewPtr);
VOID SetFontSize (HWND hwndMenu, HVPS hvps, SHORT *psCharWidth,
		  SHORT *psCharHeight);
VOID FillDriveListBox (HWND hwnd, CHAR *pcCurrentPath);
VOID FillDirListBox (HWND hwnd, CHAR *pcCurPath, BOOL fIncHidden);
VOID FillFileListBox (HWND hwnd, CHAR *pcSearchParam, BOOL fIncHidden);
VOID FillColorBox (HWND hwnd, SHORT iListID);
VOID FillCellSizeBox (HWND hwnd, SHORT iListID);
SHORT GetCellSize (SHORT sHeight, SHORT sWidth);

struct VideoStruct {
   USHORT   uslength;
   USHORT   usAdapterType;
   USHORT   usDisplayType;
} VideoData = { 6, 0, 0};

struct ParamStruct {
   CHAR     szAttribute[2];
   SHORT    sxWinPos;
   SHORT    syWinPos;
   SHORT    sWinHeight;
   SHORT    sWinWidth;
   SHORT    sCellHeight;
   SHORT    sCellWidth;
   SHORT    sTabStop;
   BOOL     fDispASCII;
   BOOL     fScrollBars;
   BOOL     fIncHiddenFiles;
} Default = {{0x20,0xf0}, -1, 0, 0, 0, 17, 8, 8, TRUE, TRUE, FALSE};

struct FileStruct {
   CHAR     szName [PATHLENGTH];
   HFILE    hHandle;
   LONG     lSize;
   BOOL     fValid;
} File = {"", 0, 0L, FALSE};

FILESTATUS FileInfo;

struct MemBuff {
   USHORT   usBSel;
   USHORT   usBOff;
   USHORT   usDSize;
   LONG     lFilePtr;
   SHORT    sUseCnt;
   BOOL     fValid;
} Buff[4];

SHORT sCellSize [10][2] = {{8, 8}, {8, 12}, {8, 14}, {7, 15},
      {7, 25}, {8, 17}, {12, 16}, {12, 20}, {12, 22}, {12, 30}};

HAB	 hab;

CHAR	 szNewFile[PATHLENGTH];
LONG	 lNewViewPtr;

CHAR	 szSearchStr[80];
BOOL	 fCheckCase = FALSE;
BOOL	 fTopSearch = FALSE;
LONG	 lLastFound = -1;

BOOL	 fSaveWinPos;
BOOL	 fSaveToDisk = FALSE;
BOOL	 fMinimized;
CHAR	 szAppName[] = "PMVIEW by Douglas Boling";
CHAR	 szKeyName[] = "Profile type 1";

//=======================================================================
// Main Proceedure.  Create message queue, parse the command line,
//    allocate memory buffers, create the window, and process the
//    message queue.
//=======================================================================
int main (int argc, char *argv[])
   {
   static   CHAR  szClientClass [] = "PMVIEW";

   static   ULONG flFrameFlags = FCF_TITLEBAR	   | FCF_SYSMENU  |
				 FCF_SIZEBORDER    | FCF_MINMAX   |
				 FCF_TASKLIST	   | FCF_ICON	  |
				 FCF_MENU	   | FCF_ACCELTABLE |
				 FCF_VERTSCROLL    | FCF_HORZSCROLL |
				 FCF_SHELLPOSITION;

   static   SHORT    fsWinPosFlags = SWP_SHOW;
   static   USHORT   usSelector;
   static   HWND     hwndFrame;
   HMQ	    hmq;
   HWND     hwndClient;
   QMSG     qmsg;
   SHORT    i;

   hab = WinInitialize (0);
   hmq = WinCreateMsgQueue (hab, 0);

   for (i = 1; i < argc && i < 4; i++)
      if (strcmp(argv[i],"\\i") == 0 || strcmp (argv[i],"-i") == 0) {
	 fMinimized = TRUE;
	 fsWinPosFlags |= SWP_MINIMIZE;
      } else
	 if (strlen (szNewFile) == 0)
	    DosQPathInfo (argv[i], 5, szNewFile, sizeof szNewFile, 0L);

   DosAllocSeg( (USHORT) 0, &usSelector, 0 );
   for (i = 0; i < 4; i++) {
      Buff[i].usBSel = usSelector;
      Buff[i].usBOff = i * BUFFERSIZE;
      Buff[i].fValid = FALSE;
   }
   i = sizeof Default;
   WinQueryProfileData (hab, szAppName, szKeyName, &Default, &i);

   WinRegisterClass (
	       hab,		    // Anchor block handle
	       szClientClass,	    // Name of class being registered
	       ClientWndProc,	    // Window procedure for class
	       CS_SIZEREDRAW,	    // Class style
	       0);		    // Extra bytes to reserve

   hwndFrame = WinCreateStdWindow (
	       HWND_DESKTOP,	    // Parent window handle
	       NULL,		    // Style of frame window
	       &flFrameFlags,	    // Pointer to control data
	       szClientClass,	    // Client window class name
	       NULL,		    // Title bar text
	       0L,		    // Style of client window
	       NULL,		    // Module handle for resources
	       ID_RESOURCE,	    // ID of resources for window
	       &hwndClient);	    // Pointer to client window handle

   if (Default.sxWinPos != -1)
      fsWinPosFlags |= SWP_MOVE | SWP_SIZE;

   WinSetWindowPos (hwndFrame, HWND_TOP, Default.sxWinPos, Default.syWinPos,
		     Default.sWinWidth, Default.sWinHeight, fsWinPosFlags);

   while (WinGetMsg (hab, &qmsg, NULL, 0, 0))
      WinDispatchMsg (hab, &qmsg);

   if (File.hHandle)
      DosClose (File.hHandle);
   DosFreeSeg (usSelector);

   WinDestroyWindow (hwndFrame);

   WinDestroyMsgQueue (hmq);
   WinTerminate (hab);
   return 0;
}
//------------------------------------------------------------------
// Open file.  Open the file, query file information, reset memory
//     buffer structure.
//------------------------------------------------------------------
BOOL OpenFile (PCHAR szNewFileName) {

   SHORT       i;
   HFILE       hNewHandle;
   USHORT      usAction;
   BOOL        fGoodRead = FALSE;

   if (DosOpen (
	szNewFileName,		       // File name
	&hNewHandle,		       // File handle
	&usAction,		       // Action taken
	0L,			       // File size
	0,			       // File attribute
	1,			       // Open type (Fail if no file.)
	0x20,			       // Open mode (Read, deny write)
	0L))			       // Reserved.
      WinAlarm (HWND_DESKTOP, WA_NOTE);
   else {
      if (File.hHandle)
	 DosClose (File.hHandle);
      strcpy (File.szName, szNewFileName);
      File.hHandle = hNewHandle;
      DosQFileInfo (File.hHandle, 1, (PBYTE) &FileInfo,
			     sizeof FileInfo);
      File.lSize = FileInfo.cbFile;
      fGoodRead = TRUE;
      //
      // Reset the memory buffer to purge the old file data.
      //
      for (i = 0; i < 4; i++) {
	 Buff[i].lFilePtr = -1;
	 Buff[i].sUseCnt = 1;
      }
   }
   return fGoodRead;
}
//------------------------------------------------------------------
// GetFileDataPtr - Return a pointer to file data pointed to by
//		    the file pointer.
//------------------------------------------------------------------
PCHAR GetFileDataPtr (LONG lFilePtr, BOOL fForward, USHORT *usLimit,
		      BOOL *fEndOfFile) {
   PCHAR    pcDataPtr;
   ULONG    ulTemp;
   SHORT    i, sLast = -1;
   SHORT    sOldCnt, sNewCnt;
   //
   // Check limits of file.
   //
   *fEndOfFile = FALSE;
   if (lFilePtr >= File.lSize) {
      lFilePtr = File.lSize;
      *fEndOfFile = fForward;
   } else if (lFilePtr <= 0L) {
      lFilePtr = 0L;
      *fEndOfFile = !fForward;
   }
   //
   // See if requested part of the file is in one of the buffers.
   //
   sOldCnt = sNewCnt = Buff[0].sUseCnt;
   for (i = 0; i < 4; i++) {
      sNewCnt = (sNewCnt < Buff[i].sUseCnt) ? Buff[i].sUseCnt : sNewCnt;
      sOldCnt = (sOldCnt > Buff[i].sUseCnt) ? Buff[i].sUseCnt : sOldCnt;
      if (Buff[i].lFilePtr != -1)
	 if (lFilePtr >= Buff[i].lFilePtr &&
		    lFilePtr < Buff[i].lFilePtr + BUFFERSIZE)
	    sLast = i;
   }
   //
   // If requested part not found, determine the oldest buffer to replace,
   //	  then read part of the file into that buffer.
   //
   if (sLast == -1) {
      sOldCnt = Buff[0].sUseCnt;
      for (i = 0; i < 4; i++)
	 if (sOldCnt >= Buff[i].sUseCnt) {
	    sOldCnt = Buff[i].sUseCnt;
	    sLast = i;
	 }
      Buff[sLast].fValid = TRUE;
      Buff[sLast].lFilePtr = (lFilePtr / BUFFERSIZE) * BUFFERSIZE;

      DosChgFilePtr (File.hHandle, Buff[sLast].lFilePtr, 0, &ulTemp);

      DosRead (File.hHandle, MAKEP (Buff[sLast].usBSel,
			   Buff[sLast].usBOff), BUFFERSIZE,
			   &Buff[sLast].usDSize);
   }
   //
   // Reset LRU count to prevent overflow
   //
   if (sNewCnt > 0x7ff0) {
      sNewCnt -= sOldCnt;
      for (i = 0; i < 4; i++)
	 Buff[i].sUseCnt = Buff[i].sUseCnt - sOldCnt;
   }
   Buff[sLast].sUseCnt = ++sNewCnt;
   //
   // Create Pointer, set limits to buffer.
   //
   pcDataPtr = MAKEP (Buff[sLast].usBSel, Buff[sLast].usBOff
			 + (USHORT) (lFilePtr - Buff[sLast].lFilePtr));
   if (fForward)
      *usLimit = Buff[sLast].usBOff + Buff[sLast].usDSize - 1;
   else
      *usLimit = Buff[sLast].usBOff;

   return pcDataPtr;
}
//------------------------------------------------------------------
// MoveViewPtr - Update view pointer by scrolling the number of
//     lines specified in ptrRow.  If limited by the top or bottom
//     of the file, modify ptrRow to indicate the actual number of
//	rows.
//------------------------------------------------------------------
LONG MoveViewPtr (LONG lFilePtr, SHORT *ptrRow, BOOL fDispASCII) {
   SHORT    i, j, sRow, sRowCount = 0, sOffset = 0;
   USHORT   usLimit;
   PCHAR    pcDataPtr;
   BOOL     fEndOfFile;

   pcDataPtr = GetFileDataPtr (lFilePtr, *ptrRow > 0, &usLimit, &fEndOfFile);
   sRow = *ptrRow;

   if (fDispASCII)
      if (sRow > 0) {
	 for (i=0; i < sRow && !fEndOfFile; i++, sOffset++){

	    for (j=0; j < AVIOCOLS && *pcDataPtr != 0x0d
		      && !fEndOfFile; j++, sOffset++)
	       if ((OFFSETOF(pcDataPtr) < usLimit))
		  pcDataPtr++;
	       else {
		  lFilePtr = ((lFilePtr + BUFFERSIZE)/ BUFFERSIZE) * BUFFERSIZE;
		  pcDataPtr = GetFileDataPtr (lFilePtr, TRUE,
					      &usLimit, &fEndOfFile);
	       }
	    if ((OFFSETOF(pcDataPtr) < usLimit))
	       pcDataPtr++;
	    else {
	       lFilePtr = ((lFilePtr + BUFFERSIZE)/ BUFFERSIZE) * BUFFERSIZE;
	       pcDataPtr = GetFileDataPtr (lFilePtr, TRUE,
					   &usLimit, &fEndOfFile);
	    }
	    if (!fEndOfFile) sRowCount++;
	 }
	 *ptrRow = sRowCount;
      } else {
	 sRow = -sRow;
	 for (i=0; i < sRow && !fEndOfFile; i++){

	    if ((OFFSETOF(pcDataPtr) > usLimit))
	       pcDataPtr--;
	    else {
	       lFilePtr = (lFilePtr / BUFFERSIZE) * BUFFERSIZE - 1;
	       pcDataPtr = GetFileDataPtr (lFilePtr, FALSE,
					   &usLimit, &fEndOfFile);
	    }
	    if (!fEndOfFile) {
	       sRowCount--;
	       sOffset--;
	    }
	    for (j=0; j < AVIOCOLS && !fEndOfFile; j++) {
	       if ((OFFSETOF(pcDataPtr) > usLimit)) {
		  if (*(pcDataPtr-1) == 0x0d)
		     break;
		  pcDataPtr--;
	       } else {
		  lFilePtr = (lFilePtr / BUFFERSIZE) * BUFFERSIZE - 1;
		  pcDataPtr = GetFileDataPtr (lFilePtr, FALSE,
					      &usLimit, &fEndOfFile);
	       }
	       if (!fEndOfFile) sOffset--;
	    }
	 }
	 *ptrRow = sRowCount;
      }
   else {
      //
      // Align on paragraph
      //
      if (lFilePtr % 16 != 0)
	 sOffset -= (SHORT)(lFilePtr % 16);
      sOffset += sRow * 16;
      *ptrRow = sRow;
      //
      //Limit Checking
      //
      if ((lFilePtr == 0 && sRow < 0)
	  ||(lFilePtr == File.lSize && sRow > 0)){
	 sOffset = 0;
	 *ptrRow = 0;
      } else if ((lFilePtr + sOffset) < 0) {
	sOffset = (SHORT) -lFilePtr;
	*ptrRow = sOffset / 16;
      } else if ((lFilePtr + sOffset) > File.lSize) {
	sOffset = (SHORT) (File.lSize - lFilePtr);
	*ptrRow = sOffset / 16;
      }
   }
   return (LONG) sOffset;
}
//------------------------------------------------------------------
// ComputeLines - Compute number of lines between two points in
//     the file.
//------------------------------------------------------------------
SHORT ComputeLines (LONG lOldPtr, LONG lNewPtr, BOOL fDispASCII) {
   USHORT   usLimit;
   SHORT    sCount = 0;
   PCHAR    pcDataPtr;
   BOOL     fEndOfFile = FALSE;

   pcDataPtr = GetFileDataPtr (lOldPtr, lOldPtr < lNewPtr, &usLimit,
			       &fEndOfFile);
   if (fDispASCII)
      if (lOldPtr < lNewPtr)
	 while (lOldPtr < lNewPtr && !fEndOfFile){
	    if (*pcDataPtr == 0x0d)
	       sCount++;
	    lOldPtr++;
	    if ((OFFSETOF(pcDataPtr) < usLimit))
	       pcDataPtr++;
	    else
	       pcDataPtr = GetFileDataPtr (lOldPtr, TRUE, &usLimit,
					   &fEndOfFile);
	 }
      else
	 while (lOldPtr > lNewPtr && !fEndOfFile) {
	    if (*pcDataPtr == 0x0d)
	       sCount--;
	    lOldPtr--;
	    if ((OFFSETOF(pcDataPtr) > usLimit))
	       pcDataPtr--;
	    else
	       pcDataPtr = GetFileDataPtr (lOldPtr, FALSE, &usLimit,
					   &fEndOfFile);
	 }
   else
      sCount = (SHORT) ((lNewPtr - lOldPtr) / 16);
   return sCount;
}
//------------------------------------------------------------------
// btoa Convert byte to ASCII
//------------------------------------------------------------------
VOID btoa (CHAR cNum, CHAR szOut[], BOOL fBlank) {
   SHORT    i;

   for (i = 2; i > 0; i--)
      if (fBlank)
	 szOut[i] = ' ';
      else {
	 szOut[i] = (cNum % 16) + 0x30;
      if (szOut[i] > 0x39) szOut[i] += (BYTE) 7;
	 cNum /= 16;
      }
   szOut[0] = ' ';
   return;
}
//------------------------------------------------------------------
// Myltoa Convert a long to ASCII with leading zeros
//------------------------------------------------------------------
VOID Myltoa (LONG lNum, CHAR szOut[]) {
   SHORT    i;

   for (i = 7; i >= 0; i--) {
      szOut[i + (i+1)/5] = (BYTE) (lNum % 16) + 0x30;
      if (szOut[i + (i+1)/5] > 0x39) szOut[i + (i+1)/5] += (BYTE) 7;
      lNum /= 16;
   }
   szOut[4] = ' ';
   return;
}
//------------------------------------------------------------------
// Display Line
//------------------------------------------------------------------
SHORT DisplayLine (HVPS hvps, SHORT sRow, LONG lLinePtr, BOOL fDispASCII) {

   CHAR     cHexChar, szTempBuff [256];
   SHORT    i, sStart, sBytesPrinted = 0;
   USHORT   usLimit;
   BOOL     fEndOfFile;
   PCHAR    pcDataPtr;

   if (fDispASCII) {
      //
      // Print as ASCII until CR or for 255 characters.
      //
      pcDataPtr = GetFileDataPtr (lLinePtr, TRUE, &usLimit, &fEndOfFile);

      for (i = 0; i <= AVIOCOLS && !fEndOfFile; i++){
	 if (*pcDataPtr == 0x0a) {
	    if ((OFFSETOF(pcDataPtr) < usLimit))
	       pcDataPtr++;
	    else {
	       lLinePtr = ((lLinePtr + BUFFERSIZE)/ BUFFERSIZE) * BUFFERSIZE;
	       pcDataPtr = GetFileDataPtr (lLinePtr, TRUE, &usLimit,
					   &fEndOfFile);
	    }
	    szTempBuff[i] = *pcDataPtr;
	    sBytesPrinted++;
	 } else
	    szTempBuff[i] = *pcDataPtr;

	 if ((OFFSETOF(pcDataPtr) < usLimit))
	    pcDataPtr++;
	 else {
	    lLinePtr = ((lLinePtr + BUFFERSIZE)/ BUFFERSIZE) * BUFFERSIZE;
	    pcDataPtr = GetFileDataPtr (lLinePtr, TRUE, &usLimit,
					&fEndOfFile);
	 }
	 sBytesPrinted++;
	 //
	 // Expand Tab Characters
	 //
	 if (szTempBuff[i] == 0x09) {
	    for (;i % Default.sTabStop != 0 && i <= AVIOCOLS; i++)
	       szTempBuff[i] = ' ';
	    szTempBuff[i] = ' ';
	 }
	 //
	 // If CR, end of line.
	 if (szTempBuff[i] == 0x0d) {
	    break;
	 }
      }
   } else {
      //
      // Convert from Hex and print.
      //
      pcDataPtr = GetFileDataPtr (lLinePtr, TRUE, &usLimit, &fEndOfFile);
      sStart = (SHORT) lLinePtr % 16;
      sBytesPrinted = 0;
      for (i = 0; i < 10; i++)
	 szTempBuff[i] = ' ';

      if (!fEndOfFile)
	 Myltoa (lLinePtr, szTempBuff);

      for (i = 0; i < 16; i++){

	 cHexChar = *pcDataPtr;

	 btoa (cHexChar, szTempBuff + 10 + (i*3) + (i/8),
	       i < sStart || fEndOfFile);
	 szTempBuff[i+60+(i/8)] = (i < sStart || fEndOfFile) ? (CHAR) 0x20
							     : cHexChar;
	 if ((OFFSETOF(pcDataPtr) < usLimit))
	    pcDataPtr++;
	 else {
	    lLinePtr = ((lLinePtr + BUFFERSIZE)/ BUFFERSIZE) * BUFFERSIZE;
	    pcDataPtr = GetFileDataPtr (lLinePtr, TRUE, &usLimit,
					&fEndOfFile);
	 }
      }
      szTempBuff[34] = ' ';
      szTempBuff[59] = ' ';
      szTempBuff[68] = ' ';
      sBytesPrinted = i - sStart;
      i += 59 + (i/8);
   }
   VioWrtCharStr (szTempBuff, i, sRow, 0, hvps);
   return sBytesPrinted;
}
//------------------------------------------------------------------
// SearchFile - Search the file for the value in szSearchStr.
//    If found, the file pointer is modified to the location in the
//     file.
//------------------------------------------------------------------
BOOL SearchFile (LONG *lFilePtr) {
   USHORT   usLimit;
   PCHAR    pcDataPtr;
   BOOL     fEndOfFile = FALSE, fFound = FALSE;
   CHAR     szCmpBuffer[165];
   SHORT    sBufferLength, sBufferCnt, sSearchLen, i;
   LONG     lLocalPtr, lMarker;

   sSearchLen = strlen (szSearchStr);
   lLocalPtr = (fTopSearch) ? 0L : *lFilePtr;

   pcDataPtr = GetFileDataPtr (lLocalPtr, TRUE, &usLimit, &fEndOfFile);

   while (!fEndOfFile && !fFound) {

      sBufferLength = (BUFFERSIZE - (SHORT)(lLocalPtr % BUFFERSIZE))
		      - sSearchLen;
      if (sBufferLength <= 0) {

	 for (i = 0; i < BUFFERSIZE - (SHORT)(lLocalPtr % BUFFERSIZE); i++)
	    *(szCmpBuffer + i) = *(pcDataPtr + i);

	 sBufferCnt = i;
	 lMarker = lLocalPtr;
	 lLocalPtr = ((lLocalPtr + BUFFERSIZE) / BUFFERSIZE) * BUFFERSIZE;

	 pcDataPtr = GetFileDataPtr (lLocalPtr, TRUE, &usLimit,
				     &fEndOfFile);
	 if (fEndOfFile)
	    break;
	 for (i = 0; i < min (sSearchLen + 1,
			      BUFFERSIZE - (usLimit % BUFFERSIZE)); i++)
	    *(szCmpBuffer + sBufferCnt + i) = *(pcDataPtr + i);

	 for (i = 0; i < sBufferCnt; i++)
	    if (fCheckCase) {
	       if (strncmp (szCmpBuffer + i, szSearchStr, sSearchLen) == 0) {
		  fFound = TRUE;
		  break;
	       }
	    } else {
	       if (strnicmp (szCmpBuffer + i, szSearchStr, sSearchLen) == 0) {
		  fFound = TRUE;
		  break;
	       }
	    }
	 sBufferLength = (BUFFERSIZE - (usLimit%BUFFERSIZE)) - sSearchLen;
      }
      if (fFound) {
	 lLocalPtr = lMarker + i;
	 break;
      }
      for (i = 0; i < sBufferLength; i++) {
	 if (fCheckCase) {
	    if (strncmp (pcDataPtr + i, szSearchStr, sSearchLen) == 0) {
	       fFound = TRUE;
	       break;
	    }
	 } else {
	    if (strnicmp (pcDataPtr + i, szSearchStr, sSearchLen) == 0) {
	       fFound = TRUE;
	       break;
	    }
	 }
      }
      lLocalPtr += i;
   }
   if (fFound)
      *lFilePtr = lLocalPtr;
   return fFound;
}
//------------------------------------------------------------------
// ScrollVis - Set Scroll Visibility
//------------------------------------------------------------------
VOID ScrollVis (HWND hwndFrame, HWND hwndHScroll, HWND hwndVScroll,
		HWND hwndMenu) {

   if (Default.fScrollBars) {

      WinSendMsg (hwndMenu, MM_SETITEMATTR,
		  MPFROM2SHORT (IDM_DISPSCROLLBARS, TRUE),
		  MPFROM2SHORT (MIA_CHECKED, MIA_CHECKED));

      WinSetParent (hwndVScroll, hwndFrame, FALSE );
      WinSetParent (hwndHScroll, hwndFrame, FALSE );
   } else {

      WinSendMsg (hwndMenu, MM_SETITEMATTR,
		  MPFROM2SHORT (IDM_DISPSCROLLBARS, TRUE),
		  MPFROM2SHORT (MIA_CHECKED, 0));

      WinSetParent (hwndVScroll, HWND_OBJECT, FALSE );
      WinSetParent (hwndHScroll, HWND_OBJECT, FALSE );
   }
   WinSendMsg ( hwndFrame , WM_UPDATEFRAME ,
   (MPARAM) (FCF_TITLEBAR | FCF_SYSMENU | FCF_MINMAX |
   FCF_MENU), NULL);

   return;
}
//------------------------------------------------------------------
// Set ASCII - Set ASCII Menu Check marks.
//------------------------------------------------------------------
VOID SetASCII (HWND hwndMenu, LONG *lViewPtr) {
   SHORT sRow = -1;
   if (Default.fDispASCII) {
      if (MIA_CHECKED && WinSendMsg (hwndMenu, MM_QUERYITEMATTR,
				     MPFROM2SHORT (IDM_DISPLAYHEX, TRUE),
				     MPFROM2SHORT (MIA_CHECKED, 0))) {
	 *lViewPtr += 2;
	 *lViewPtr += MoveViewPtr (*lViewPtr, &sRow, TRUE);
      }
      WinSendMsg (hwndMenu, MM_SETITEMATTR,
		  MPFROM2SHORT (IDM_DISPLAYHEX, TRUE),
		  MPFROM2SHORT (MIA_CHECKED, 0));

      WinSendMsg (hwndMenu, MM_SETITEMATTR,
		  MPFROM2SHORT (IDM_DISPLAYASCII, TRUE),
		  MPFROM2SHORT (MIA_CHECKED, MIA_CHECKED));
   } else {
      WinSendMsg (hwndMenu, MM_SETITEMATTR,
		  MPFROM2SHORT (IDM_DISPLAYASCII, TRUE),
		  MPFROM2SHORT (MIA_CHECKED, 0));

      WinSendMsg (hwndMenu, MM_SETITEMATTR,
		  MPFROM2SHORT (IDM_DISPLAYHEX, TRUE),
		  MPFROM2SHORT (MIA_CHECKED, MIA_CHECKED));
   }
   return;
}
//------------------------------------------------------------------
// SetFontSize - Change font size and update menu if necessary.
//------------------------------------------------------------------
VOID SetFontSize (HWND hwndMenu, HVPS hvps, SHORT *psCharWidth,
		  SHORT *psCharHeight) {
   static   CHAR  szSmallPrompt[] = "~Small Font\tCtrl+S";
   static   CHAR  szLargePrompt[] = "~Large Font\tCtrl+S";

   VioSetDeviceCellSize (Default.sCellHeight, Default.sCellWidth, hvps);
   VioGetDeviceCellSize (psCharHeight, psCharWidth, hvps);

   switch (VideoData.usAdapterType) {
      case 0:
      case 1:			       //CGA, no other cell sizes.
	 WinSendMsg (hwndMenu, MM_SETITEMATTR,
		     MPFROM2SHORT (IDM_FONTSIZE, 0),
		     MPFROM2SHORT (MIA_DISABLED, MIA_CHECKED));
	 break;

      case 2:			       //EGA
      case 3:			       //VGA
      if (Default.sCellHeight == 8 && Default.sCellWidth == 8)
	 WinSendMsg (hwndMenu, MM_SETITEMTEXT,
		     MPFROM2SHORT (IDM_FONTSIZE, 0),
		     MPFROMP (szLargePrompt));
      else
	 WinSendMsg (hwndMenu, MM_SETITEMTEXT,
		     MPFROM2SHORT (IDM_FONTSIZE, 0),
		     MPFROMP (szSmallPrompt));
	 break;

      case 7:			       //8514
	 break;
      }
   return;
}
//=======================================================================
//
// Client Window Procedure
//
//=======================================================================
MRESULT EXPENTRY ClientWndProc (HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2)
   {
   static   HWND  hwndFrame, hwndMenu, hwndVScroll, hwndHScroll;

   static   HVPS  hvps;
   static   HPS   hps;

   static   LONG  lViewPtr;
   static   BOOL  fGoMinimize;

   static   SHORT sHScroll, sVScrScale, sHPage = 20, sVPage = 20;
   static   SHORT sCharHeight, sCharWidth, sxWinSize, syWinSize;
   static   SHORT sRowsVisible = AVIOROWS;

   HDC	    hdc;
   SIZEL    sizl;
   SHORT    sRow, sDeltaRows;
   LONG     lTemp;
   SWP	    swpWinPos;
   PSWP     pswpWinPos;
   CHAR     szTempBuffer[PATHLENGTH];

   switch (msg) {

      //--------------------------------------------------------------
      // When the window is created, get and store the handles to
      //   the scroll bars and menu, read the default parameters if
      //   saved before, set the proper attributes for the menu
      //   selections, and create a AVIO presentation space.
      //--------------------------------------------------------------
      case WM_CREATE:
	 //
	 // If no file specified and not starting as an icon, Query
	 // user for a file name.
	 //
	 if (!File.fValid) {
	    if (strlen (szNewFile) == 0) {
	       if (!fMinimized)
		  if (WinDlgBox(HWND_DESKTOP, hwnd, FileDlgProc,
			 NULL, IDD_OPEN, NULL))
		     if (!(File.fValid = OpenFile (szNewFile)))
			WinMessageBox (HWND_DESKTOP, hwnd,
				       "Unable to open the file",
				       File.szName, NULL, MB_OK | MB_ERROR);
	    } else
	       if (!(File.fValid = OpenFile (szNewFile)))
		  WinMessageBox (HWND_DESKTOP, hwnd,
				 "Unable to open the file", File.szName,
				 NULL, MB_OK | MB_ERROR);
	    if (File.fValid) {
	       if (sHScroll) {
		  sHScroll = 0;
		  VioSetOrg (0, sHScroll, hvps);
	       }
	       lViewPtr = 0;
	       sVScrScale  = (SHORT) ((File.lSize / 32768) + 1);
	    }
	 }
	 fMinimized = FALSE;
	 //
	 // Query the Frame for the necessary window handles.
	 //
	 hwndFrame = WinQueryWindow (hwnd, QW_PARENT,FALSE);
	 hwndMenu = WinWindowFromID (hwndFrame, FID_MENU);
	 hwndVScroll = WinWindowFromID (hwndFrame, FID_VERTSCROLL);
	 hwndHScroll = WinWindowFromID ( hwndFrame, FID_HORZSCROLL);
	 //
	 // Set menu attributes.
	 //
	 sVScrScale  = (SHORT) ((File.lSize / 32768) + 1);
	 ScrollVis (hwndFrame, hwndHScroll, hwndVScroll, hwndMenu);
	 SetASCII(hwndMenu, &lViewPtr);
	 //
	 // Establish AVIO presentation space.
	 //
	 hdc = WinOpenWindowDC (hwnd);
	 sizl.cx = sizl.cy = 0;
	 hps = GpiCreatePS (hab, hdc, &sizl, PU_PELS | GPIF_DEFAULT |
			    GPIT_MICRO | GPIA_ASSOC);
	 VioCreatePS (&hvps, AVIOROWS, AVIOCOLS, 0, 1, NULL);
	 VioAssociate (hdc, hvps);
	 VioGetConfig (0, (PVOID) &VideoData, hvps);

	 SetFontSize (hwndMenu, hvps, &sCharWidth, &sCharHeight);
	 VioScrollUp (0, 0, -1, -1, -1, Default.szAttribute, hvps);

	 if (strlen (File.szName) != 0)
	    WinSetWindowText (hwndFrame, File.szName);
	 return 0;

      //--------------------------------------------------------------
      // Intercept the close message to abort a file open dialog box
      //   when the program is closed from an icon.
      //--------------------------------------------------------------
      case WM_CLOSE:

	    fMinimized = FALSE;
	 break;

      //--------------------------------------------------------------
      // If the Window is being restored from an icon without a vaild
      //   file, prompt the user for a file using the file open dialog
      //   box.
      //--------------------------------------------------------------
      case WM_MINMAXFRAME:

	 if (fMinimized)
	    if (!File.fValid) {
	       if (WinDlgBox(HWND_DESKTOP, hwnd, FileDlgProc, NULL,
			     IDD_OPEN, NULL)) {
		  if (!(File.fValid = OpenFile (szNewFile)))
		     WinMessageBox (HWND_DESKTOP, hwnd,
				    "Unable to open the file",
				    File.szName, NULL, MB_OK | MB_ERROR);
		  else {
		     if (sHScroll) {
			sHScroll = 0;
			VioSetOrg (0, sHScroll, hvps);
		     }
		     lViewPtr = 0;
		     sVScrScale  = (SHORT) ((File.lSize / 32768) + 1);
		     WinSetWindowText (hwndFrame, File.szName);
		  }
	       } else
		  fGoMinimize = TRUE;
	    }
	 //
	 // Record the current state of the program.
	 //
	 pswpWinPos = PVOIDFROMMP (mp1);
	 if (pswpWinPos->fs & SWP_MINIMIZE)
	    fMinimized = TRUE;
	 else
	    fMinimized = FALSE;
	 break;

      //--------------------------------------------------------------
      // When the GoMimize flag is set send the frame a minimize
      //   message.
      //--------------------------------------------------------------
      case WM_WINDOWPOSCHANGED:

	    if (fGoMinimize) {
	       fGoMinimize = FALSE;
	       WinSetWindowPos (hwndFrame, HWND_BOTTOM, 0, 0, 0, 0,
				 SWP_MINIMIZE);
	    }
	 break;

      //--------------------------------------------------------------
      // When the window is resized, record the new window size,
      //   update the horizontal position of the AVIO space and
      //   scroll bar, and notify the frame to modify its windows.
      //--------------------------------------------------------------
      case WM_SIZE:

	 sxWinSize = SHORT1FROMMP(mp2);
	 syWinSize = SHORT2FROMMP(mp2);

	 if (sHScroll > AVIOCOLS - sxWinSize / sCharWidth) {
	    sHScroll = min (max (0, sHScroll),
			     AVIOCOLS - sxWinSize / sCharWidth);
	    VioSetOrg (0, sHScroll, hvps);
	    WinInvalidateRect (hwnd, NULL, FALSE);
	 }
	 sHPage = max (sxWinSize / sCharWidth - 1, 0);
	 sVPage = max (syWinSize / sCharHeight - 1, 0);
	 sRowsVisible = min (syWinSize / sCharHeight + 1, AVIOROWS);

	 WinSendMsg ( hwndFrame, WM_UPDATEFRAME, (MPARAM) (FCF_TITLEBAR
		      | FCF_SYSMENU | FCF_MINMAX | FCF_MENU), NULL);

	 WinSendMsg (hwndHScroll, SBM_SETSCROLLBAR,
		     MPFROM2SHORT (sHScroll, 0),
		     MPFROM2SHORT (0, AVIOCOLS - sxWinSize / sCharWidth));

	 WinSendMsg (hwndHScroll, SBM_SETTHUMBSIZE,
		     MPFROM2SHORT (sxWinSize / sCharWidth,
		     AVIOCOLS - sxWinSize / sCharWidth), NULL);

	 WinSendMsg (hwndVScroll, SBM_SETSCROLLBAR,
		     MPFROM2SHORT ((SHORT)(lViewPtr/sVScrScale), 0),
		     MPFROM2SHORT (0, File.lSize/sVScrScale));

	 WinSendMsg (hwndVScroll, SBM_SETTHUMBSIZE,
		     MPFROM2SHORT( (syWinSize/sCharHeight) * 30,
				  File.lSize/sVScrScale),NULL);

	 return WinDefAVioWindowProc (hwnd, msg, mp1, mp2);

      //--------------------------------------------------------------
      // Repaint the complete AVIO space by first clearing the window
      //   then writing to the AVIO space.
      //--------------------------------------------------------------
      case WM_PAINT:

	 WinBeginPaint (hwnd, hps, NULL);
	 VioScrollUp (0, 0, -1, -1, -1, Default.szAttribute, hvps);

	 lTemp = lViewPtr;
	 for (sRow = 0; sRow < sRowsVisible; sRow++)
	    lTemp += (LONG) DisplayLine (hvps, sRow, lTemp,
					  Default.fDispASCII);
	 WinEndPaint (hps);

	 WinSendMsg (hwndVScroll, SBM_SETPOS,
		     MPFROM2SHORT ((SHORT)(lViewPtr/sVScrScale), 0),
		     NULL);
	 return 0;

      //--------------------------------------------------------------
      // Process each menu selection according to function.
      //--------------------------------------------------------------
      case WM_COMMAND:

	 switch (COMMANDMSG(&msg)->cmd) {

	 case IDM_OPEN:
	    if (WinDlgBox(HWND_DESKTOP, hwnd, FileDlgProc,
		      NULL, IDD_OPEN, NULL)) {
	       if (File.fValid = OpenFile (szNewFile)) {
		  if (sHScroll) {
		     sHScroll = 0;
		     VioSetOrg (0, sHScroll, hvps);
		  }
		  lViewPtr = 0;
		  sVScrScale  = (SHORT) ((File.lSize / 32768) + 1);

		  WinSendMsg (hwndVScroll, SBM_SETSCROLLBAR,
			      MPFROM2SHORT (0, 0),
			      MPFROM2SHORT (0, File.lSize/sVScrScale));
		  WinSendMsg (hwndVScroll, SBM_SETTHUMBSIZE,
			      MPFROM2SHORT( (syWinSize/sCharHeight) * 30,
			      File.lSize/sVScrScale),NULL);
		  WinSetWindowText (hwndFrame, File.szName);
		  WinInvalidateRect (hwnd, NULL, FALSE);
	       } else
		  WinMessageBox (HWND_DESKTOP, hwnd, "Unable to open the file",
				 File.szName, NULL, MB_OK | MB_ERROR);
	    }
	    return 0;

	 case IDM_FIND:
	    if (WinDlgBox(HWND_DESKTOP, hwnd, SearchDlgProc,
		      NULL, IDD_SEARCH, NULL))
	       if (strlen (szSearchStr) != 0)
		  WinPostMsg (hwnd, WM_COMMAND,
			      MPFROM2SHORT (IDM_REPFIND,0), mp2);
	    return 0;

	 case IDM_REPFIND:
	    if (strlen (szSearchStr) == 0) {
	       WinPostMsg (hwnd, WM_COMMAND, MPFROM2SHORT (IDM_FIND,0),
			   mp2);
	       return 0;
	    } else {
	       //
	       //   Check to see if we are on the same line as the last
	       // found string.  If so, start search at 1 byte past
	       // last found string.
	       //
	       sDeltaRows = 1;
	       lTemp = MoveViewPtr (lViewPtr, &sDeltaRows,
					   Default.fDispASCII);
	       lTemp += lViewPtr;

	       if (lTemp > lLastFound && lLastFound > lViewPtr)
		  lLastFound++;
	       else
		  lLastFound = lViewPtr;
	       //
	       //   If string found, back up the the start fo the
	       // current line.
	       //
	       WinQueryWindowText (hwndFrame, PATHLENGTH, szTempBuffer);
	       WinSetWindowText (hwndFrame, "Searching");
	       if (SearchFile (&lLastFound)) {
		  lViewPtr = lLastFound;
		  sDeltaRows = -1;
		  lViewPtr += MoveViewPtr (lViewPtr, &sDeltaRows,
					   Default.fDispASCII);
		  WinInvalidateRect (hwnd, NULL, FALSE);
		  WinSetWindowText (hwndFrame, szTempBuffer);
	       } else {
		  WinSetWindowText (hwndFrame, szTempBuffer);
		  WinMessageBox (HWND_DESKTOP, hwnd, szSearchStr,
				 "Unable to find the string",
				 NULL, MB_OK | MB_ICONHAND);
	       }
	    }
	    return 0;

	 case IDM_GOTO:
	    if (WinDlgBox(HWND_DESKTOP, hwnd, GotoDlgProc,
		      NULL, IDD_GOTO, NULL)) {
	       lViewPtr = min (max (lNewViewPtr, 0L), File.lSize);
	       WinInvalidateRect (hwnd, NULL, FALSE);
	    }
	    return 0;

	 case IDM_CONFIGURE:
	    if (WinDlgBox(HWND_DESKTOP, hwnd, ConfigDlgProc,
		      NULL, IDD_CONFIGURE, NULL)) {
	       if (fSaveToDisk) {
		  if (fSaveWinPos) {
		     WinQueryWindowPos (hwndFrame, &swpWinPos);
		     Default.sxWinPos = swpWinPos.x;
		     Default.syWinPos = swpWinPos.y;
		     Default.sWinWidth = swpWinPos.cx;
		     Default.sWinHeight = swpWinPos.cy;
		  }
		  WinWriteProfileData (hab, szAppName, szKeyName,
				       &Default, sizeof Default);
	       }
	       SetASCII(hwndMenu, &lViewPtr);
	       ScrollVis (hwndFrame, hwndHScroll, hwndVScroll, hwndMenu);
	       SetFontSize (hwndMenu, hvps, &sCharWidth, &sCharHeight);
	       sHPage = max (sxWinSize / sCharWidth - 1, 0);
	       sVPage = max (syWinSize / sCharHeight - 1, 0);
	       sRowsVisible = min (syWinSize / sCharHeight + 1, AVIOROWS);
	       WinInvalidateRect (hwnd, NULL, FALSE);
	    }
	    return 0;

	 case IDM_DISPLAYASCII:
	    if (!Default.fDispASCII) {
	       Default.fDispASCII = TRUE;
	       SetASCII (hwndMenu, &lViewPtr);
	       WinInvalidateRect (hwnd, NULL, FALSE);
	    }
	    return 0;

	 case IDM_DISPLAYHEX:
	    if (Default.fDispASCII) {
	       Default.fDispASCII = FALSE;
	       SetASCII (hwndMenu, &lViewPtr);
	       WinInvalidateRect (hwnd, NULL, FALSE);
	    }
	    return 0;

	 case IDM_DISPSCROLLBARS:
	    if (Default.fScrollBars)
	       Default.fScrollBars = FALSE;
	    else
	       Default.fScrollBars = TRUE;

	    ScrollVis (hwndFrame, hwndHScroll, hwndVScroll, hwndMenu);
	    return 0;

	 case IDM_FONTSIZE:
	    switch (VideoData.usAdapterType) {
	       case 2:				//EGA
		  if (Default.sCellHeight == 8 && Default.sCellWidth == 8){
		     Default.sCellHeight = sCellSize[1][1];
		     Default.sCellWidth = sCellSize[1][0];
		  } else {
		     Default.sCellHeight = sCellSize[0][1];
		     Default.sCellWidth = sCellSize[0][0];
		  }
		  break;

	       case 3:				//VGA
		  if (Default.sCellHeight == 8 && Default.sCellWidth == 8){
		     Default.sCellHeight = sCellSize[2][1];
		     Default.sCellWidth = sCellSize[2][0];
		  } else {
		     Default.sCellHeight = sCellSize[0][1];
		     Default.sCellWidth = sCellSize[0][0];
		  }
		  break;

	       case 7:				//8514
		  if (!WinDlgBox(HWND_DESKTOP, hwnd, FontDlgProc,
			 NULL, IDD_PICKFONT, NULL))
		     return 0;
		  break;
	    }
	    SetFontSize (hwndMenu, hvps, &sCharWidth, &sCharHeight);
	    sHPage = max (sxWinSize / sCharWidth - 1, 0);
	    sVPage = max (syWinSize / sCharHeight - 1, 0);
	    sRowsVisible = min (syWinSize / sCharHeight + 1, AVIOROWS);
	    WinInvalidateRect (hwnd, NULL, FALSE);
	    return 0;

	 case IDM_ABOUT:
	    WinDlgBox(HWND_DESKTOP, hwnd, AboutDlgProc,
		      NULL, IDD_ABOUT, NULL);
	    return 0;

	 case IDM_EXIT:
	    WinSendMsg(hwnd, WM_CLOSE, 0L, 0L);
	    return 0;
      }
      //--------------------------------------------------------------
      // Convert cursor keys to scroll bar messages.
      //--------------------------------------------------------------
      case WM_CHAR:

	 switch (CHARMSG (&msg) -> vkey) {

	    case VK_UP:
	    case VK_DOWN:
	    case VK_PAGEUP:
	    case VK_PAGEDOWN:
	       return WinSendMsg (hwndVScroll, msg, mp1, mp2);

	    case VK_LEFT:
	    case VK_RIGHT:
	       return WinSendMsg (hwndHScroll, msg, mp1, mp2);

	    case VK_HOME:
	       if (lViewPtr != 0) {
		  lViewPtr = 0;
		  WinInvalidateRect (hwnd, NULL, FALSE);
	       }
	       break;

	    case VK_END:
	       if (lViewPtr != File.lSize - 1) {
		  lViewPtr = File.lSize - 1;
		  WinInvalidateRect (hwnd, NULL, FALSE);
	       }
	       break;
	 }
	 return WinDefWindowProc (hwnd, msg, mp1, mp2);

      //-----------------------------------------------------------
      // Process Horizonal Scroll Bar Messages by scrolling across
      //    the AVIO space.
      //-----------------------------------------------------------
      case WM_HSCROLL:

	 switch (SHORT2FROMMP (mp2)) {

	    case SB_LINELEFT:

	       sHScroll -= 1;
	       break;

	    case SB_PAGELEFT:

	       sHScroll -= sHPage;
	       break;

	    case SB_LINERIGHT:

	       sHScroll += 1;
	       break;

	    case SB_PAGERIGHT:

	       sHScroll += sHPage;
	       break;

	    case SB_SLIDERTRACK:

	       sHScroll = SHORT1FROMMP (mp2);
	       break;
	 }
	 sHScroll = min (max (0, sHScroll),
			  AVIOCOLS - sxWinSize / sCharWidth);

	 if (sHScroll != SHORT1FROMMR (WinSendMsg (hwndHScroll,
					SBM_QUERYPOS, NULL, NULL))) {
	    VioSetOrg (0, sHScroll, hvps);
	    WinSendMsg (hwndHScroll, SBM_SETPOS,
			MPFROM2SHORT (sHScroll, 0), NULL);
	 }
	 return 0;

      //-----------------------------------------------------------
      // Process Vertical Scroll Bar Messages by scrolling the AVIO
      //    space and writing the new lines.
      //-----------------------------------------------------------
      case WM_VSCROLL:

	 switch (SHORT2FROMMP (mp2)) {

	    case SB_LINEUP:

	       sDeltaRows = -1;
	       break;

	    case SB_PAGEUP:

	       sDeltaRows = -sVPage;
	       break;

	    case SB_LINEDOWN:

	       sDeltaRows = 1;
	       break;

	    case SB_PAGEDOWN:

	       sDeltaRows = sVPage;
	       break;

	    case SB_SLIDERTRACK:

	       lTemp = lViewPtr;

	       lViewPtr = (LONG) (SHORT1FROMMP(mp2) * sVScrScale);

	       sDeltaRows = ComputeLines (lTemp, lViewPtr,
			       Default.fDispASCII);
	       sDeltaRows = (sDeltaRows > sRowsVisible) ? sRowsVisible : sDeltaRows;
	       sDeltaRows = (sDeltaRows < -sRowsVisible) ? -sRowsVisible : sDeltaRows;
	       break;

	    default:
	       return 0;
	 }
	 if (SHORT2FROMMP (mp2) != SB_SLIDERTRACK)
	    lViewPtr += MoveViewPtr (lViewPtr, &sDeltaRows, Default.fDispASCII);

	 WinSendMsg (hwndVScroll, SBM_SETPOS,
		     MPFROM2SHORT ((SHORT)(lViewPtr/sVScrScale), 0), NULL);

	 if (sDeltaRows > 0) {
	    VioScrollUp (0, 0, sRowsVisible, AVIOCOLS, sDeltaRows,
			 Default.szAttribute, hvps);

	    sRow = sRowsVisible-sDeltaRows;

	    lTemp = lViewPtr;
	    lTemp += MoveViewPtr (lTemp, &sRow, Default.fDispASCII);

	    for (sRow = 0; sRow <= sDeltaRows; sRow++)
	       lTemp += DisplayLine (hvps, (sRowsVisible-sDeltaRows) + sRow,
				     lTemp, Default.fDispASCII);
	 }
	 if (sDeltaRows < 0) {
	    sDeltaRows = -sDeltaRows;
	    VioScrollDn (0, 0, sRowsVisible, AVIOCOLS, sDeltaRows,
			 Default.szAttribute, hvps);

	    lTemp = lViewPtr;
	    for (sRow = 0; sRow <= sDeltaRows; sRow++)
	       lTemp += DisplayLine (hvps, sRow, lTemp, Default.fDispASCII);
	 }
	 return 0;

      //-----------------------------------------------------------
      // Destroy the AVIO space nad the presentation space.
      //-----------------------------------------------------------
      case WM_DESTROY:

	 VioAssociate (NULL, hvps);
	 VioDestroyPS (hvps);
	 GpiDestroyPS (hps);
	 return 0;
   }
   return WinDefWindowProc (hwnd, msg, mp1, mp2);
}
//=====================================================================
// Dialog Procedures
//=====================================================================
//------------------------------------------------------------------
//  About Dialog Box
//------------------------------------------------------------------
MRESULT EXPENTRY AboutDlgProc (HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2)
   {
   switch (msg) {

      case WM_COMMAND:

	 switch (COMMANDMSG(&msg)->cmd) {

	    case DID_OK:
	    case DID_CANCEL:
		    WinDismissDlg (hwnd, TRUE);
		    return 0;
	 }
   }
   return WinDefDlgProc (hwnd, msg, mp1, mp2);
}
//------------------------------------------------------------------
//  Goto Dialog Box
//------------------------------------------------------------------
MRESULT EXPENTRY GotoDlgProc (HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2)
   {
   CHAR  szBuffer[12];

   switch (msg) {

      case WM_INITDLG:

	 WinSendDlgItemMsg (hwnd, IDD_NEWOFFSET, EM_CLEAR, NULL, NULL);

	 WinSendDlgItemMsg ( hwnd, IDD_NEWOFFSET, EM_SETTEXTLIMIT,
			     MPFROM2SHORT (11, 0), NULL);
	 return 0;

      case WM_COMMAND:

	 switch (COMMANDMSG(&msg)->cmd) {

	    case DID_OK:
	       if (WinQueryDlgItemText (hwnd, IDD_NEWOFFSET, sizeof szBuffer,
		    szBuffer)) {
		  lNewViewPtr = atol (szBuffer);
		  WinDismissDlg (hwnd, TRUE);
		  return 0;
	       }
	    case DID_CANCEL:
	       WinDismissDlg (hwnd, FALSE);
	       return 0;
	 }
   }
   return WinDefDlgProc (hwnd, msg, mp1, mp2);
}
//=====================================================================
// Search Dialog Procedures
//=====================================================================
//------------------------------------------------------------------
//  Search Dialog Box
//------------------------------------------------------------------
MRESULT EXPENTRY SearchDlgProc (HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2)
   {
   switch (msg) {

      case WM_INITDLG:

	 WinSetDlgItemText ( hwnd, IDD_SRCHSTR, szSearchStr);

	 WinSendDlgItemMsg (hwnd, IDD_CHKCASE, BM_SETCHECK,
			    MPFROM2SHORT (fCheckCase, 0), NULL);

	 WinSendDlgItemMsg (hwnd, IDD_TOPSEARCH, BM_SETCHECK,
			    MPFROM2SHORT (fTopSearch, 0), NULL);
	 return 0;

      case WM_CONTROL:

	 switch (SHORT1FROMMP (mp1)) {

	    case IDD_CHKCASE:
	       switch (SHORT2FROMMP (mp1)) {

		  case BN_CLICKED:
		     if (fCheckCase)
			fCheckCase = FALSE;
		     else
			fCheckCase = TRUE;
		     return 0;
	       }
	       break;

	    case IDD_TOPSEARCH:
	       switch (SHORT2FROMMP (mp1)) {

		  case BN_CLICKED:
		     if (fTopSearch)
			fTopSearch = FALSE;
		     else
			fTopSearch = TRUE;
		     return 0;
	       }
	       break;
	 }
	 break;

      case WM_COMMAND:

	 switch (COMMANDMSG(&msg)->cmd) {

	    case DID_OK:
	       WinQueryDlgItemText (hwnd, IDD_SRCHSTR, sizeof szSearchStr,
		    szSearchStr);
	       if (strlen (szSearchStr)) {
		  WinDismissDlg (hwnd, TRUE);
		  return 0;
	       }
	    case DID_CANCEL:
	       WinDismissDlg (hwnd, FALSE);
	       return 0;
	 }
   }
   return WinDefDlgProc (hwnd, msg, mp1, mp2);
}
//===================================================================
// Procedures for File open dialog box
//===================================================================
VOID FillDriveListBox (HWND hwnd, CHAR *pcCurrentPath) {
   static CHAR szDrive [] = "  :";
   SHORT  sDrive;
   ULONG  ulDriveMap;
   USHORT usDriveNum;

   DosQCurDisk (&usDriveNum, &ulDriveMap);

   pcCurrentPath[0] = (CHAR) usDriveNum + '@';
   pcCurrentPath[1] = ':';
   pcCurrentPath[2] = '\\';

   WinSendDlgItemMsg (hwnd, IDD_DRIVELIST, LM_DELETEALL, NULL, NULL);

   for (sDrive = 0; sDrive < 26 ; sDrive++ )
      if (ulDriveMap & 1L << sDrive) {
	 szDrive [1] = (CHAR) sDrive + 'A';

	 WinSendDlgItemMsg (hwnd, IDD_DRIVELIST, LM_INSERTITEM,
			      MPFROM2SHORT (LIT_END, 0),
			      MPFROMP (szDrive));
      }
      WinSendDlgItemMsg (hwnd, IDD_DRIVELIST, LM_SELECTITEM,
			   MPFROM2SHORT (usDriveNum - 1, 0),
			   MPFROM2SHORT (TRUE, 0));
}
//------------------------------------------------------------------
//  Fill Directory List Box Procedure
//------------------------------------------------------------------
VOID FillDirListBox (HWND hwnd, CHAR *pcCurPath, BOOL fIncHidden) {
   static      CHAR  szRootStr[] = "\\";
   FILEFINDBUF findbuf;
   HDIR        hDir = 1;
   USHORT      usCurPathLen = PATHLENGTH, usSearchCount = 1;
   USHORT      usAttr;

   if (fIncHidden)
      usAttr = 0x0017;
   else
      usAttr = 0x0015;

   WinSendDlgItemMsg (hwnd, IDD_DIRLIST, LM_DELETEALL, NULL, NULL);

   DosQCurDir (0, pcCurPath + 3 , &usCurPathLen);

   DosFindFirst ("*.*", &hDir, usAttr, &findbuf, sizeof findbuf,
		 &usSearchCount, 0L);

   while (usSearchCount) {
      if (findbuf.attrFile & 0x0010
	  && (findbuf.achName [0] != '.' || findbuf.achName [1] ))

	 if (pcCurPath [strlen(pcCurPath)-1] != '\\'
	     || findbuf.achName[1] != '.' || findbuf.achName [2])

	    WinSendDlgItemMsg (hwnd, IDD_DIRLIST, LM_INSERTITEM,
				 MPFROM2SHORT (LIT_SORTASCENDING, 0),
				 MPFROMP (findbuf.achName));

      DosFindNext (hDir, &findbuf, sizeof findbuf, &usSearchCount);
   }
   if (pcCurPath [strlen(pcCurPath)-1] != '\\')
      WinSendDlgItemMsg (hwnd, IDD_DIRLIST, LM_INSERTITEM,
			  MPFROM2SHORT (0, 0), MPFROMP (szRootStr));
}
//------------------------------------------------------------------
//  Fill File List Box Procedure
//------------------------------------------------------------------
VOID FillFileListBox (HWND hwnd, CHAR *pcSearchParam, BOOL fIncHidden) {
   FILEFINDBUF findbuf;
   HDIR        hDir = 1;
   USHORT      usSearchCount = 1;
   USHORT      usAttr;

   if (fIncHidden)
      usAttr = 0x0007;
   else
      usAttr = 0x0005;

   WinSendDlgItemMsg (hwnd, IDD_FILELIST, LM_DELETEALL, NULL, NULL);

   DosFindFirst (pcSearchParam, &hDir, usAttr, &findbuf, sizeof findbuf,
		 &usSearchCount, 0L);

   while (usSearchCount) {
      WinSendDlgItemMsg (hwnd, IDD_FILELIST, LM_INSERTITEM,
			   MPFROM2SHORT (LIT_SORTASCENDING, 0),
			   MPFROMP (findbuf.achName));
      DosFindNext (hDir, &findbuf, sizeof findbuf, &usSearchCount);
   }
}
//------------------------------------------------------------------
//  File Open Dialog Box Procedure
//------------------------------------------------------------------
MRESULT EXPENTRY FileDlgProc (HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2)
   {
   static      CHAR  szCurrentPath [PATHLENGTH], szFileName[PATHLENGTH] = "*.*";
   static      BOOL  fIncHidden;
   CHAR        szBuffer [PATHLENGTH];
   USHORT      usCurPathLen = PATHLENGTH - 3;
   SHORT       sSelect;
   FILEFINDBUF findbuf;
   HDIR        hDir = 1;
   USHORT      usSearchCount = 1, usAttr, i;

   switch (msg) {

      case WM_INITDLG:
	 fIncHidden = Default.fIncHiddenFiles;

	 strcpy (szFileName, "*.*");
	 FillDriveListBox (hwnd, szCurrentPath);
	 FillDirListBox (hwnd, szCurrentPath, fIncHidden);
	 FillFileListBox (hwnd, szFileName, fIncHidden);

	 WinSendDlgItemMsg (hwnd, IDD_INCHIDDEN, BM_SETCHECK,
			    MPFROM2SHORT (fIncHidden, 0), NULL);

	 DosQCurDir (0, szCurrentPath + 3, &usCurPathLen);

	 WinSendDlgItemMsg ( hwnd, IDD_FILEEDIT, EM_SETTEXTLIMIT,
			     MPFROM2SHORT (PATHLENGTH, 0), NULL);

	 WinSetDlgItemText ( hwnd, IDD_FILEEDIT, szCurrentPath);
	 return 0;

      case WM_CONTROL:

	 if (SHORT1FROMMP (mp1) == IDD_DRIVELIST ||
	     SHORT1FROMMP (mp1) == IDD_DIRLIST	 ||
	     SHORT1FROMMP (mp1) == IDD_FILELIST) {

	    sSelect = (USHORT) WinSendDlgItemMsg (hwnd, SHORT1FROMMP (mp1),
				    LM_QUERYSELECTION, 0L, 0L);

	    WinSendDlgItemMsg (hwnd, SHORT1FROMMP (mp1),
			      LM_QUERYITEMTEXT,
			      MPFROM2SHORT (sSelect, sizeof szBuffer),
			      MPFROMP (szBuffer));
	 }

	 switch (SHORT1FROMMP (mp1)) {

	    case IDD_DRIVELIST:
	       switch (SHORT2FROMMP (mp1)) {

		  case LN_ENTER:
		     szCurrentPath[0] = szBuffer[1];
		     DosSelectDisk (szBuffer [1] - '@');

		     FillDirListBox (hwnd, szCurrentPath, fIncHidden);
		     if (strcmp(szFileName, "*.*") != 0) {
			if (szCurrentPath [strlen (szCurrentPath)-1]
								!= '\\')
			   strcat (szCurrentPath, "\\");
			strcat (szCurrentPath, szFileName);
		     }
		     WinSetDlgItemText ( hwnd, IDD_FILEEDIT, szCurrentPath);
		     FillFileListBox (hwnd, szFileName, fIncHidden);

		     return 0;
	       }
	       break;

	    case IDD_DIRLIST:
	       switch (SHORT2FROMMP (mp1)) {

		  case LN_ENTER:

		     DosChDir (szBuffer, 0L);

		     FillDirListBox (hwnd, szCurrentPath, fIncHidden);
		     if (strcmp(szFileName, "*.*") != 0) {
			if (szCurrentPath [strlen (szCurrentPath)-1]
								!= '\\')
			   strcat (szCurrentPath, "\\");
			strcat (szCurrentPath, szFileName);
		     }
		     WinSetDlgItemText ( hwnd, IDD_FILEEDIT, szCurrentPath);
		     FillFileListBox (hwnd, szFileName, fIncHidden);
		     return 0;
	       }
	       break;

	    case IDD_FILELIST:
	       switch (SHORT2FROMMP (mp1)) {

		  case LN_SELECT:

		     DosQCurDir (0, szCurrentPath + 3, &usCurPathLen);
		     if (szCurrentPath [strlen(szCurrentPath)-1] != '\\')
			strcat (szCurrentPath, "\\");
		     strcat (szCurrentPath, szBuffer);

		     WinSetDlgItemText ( hwnd, IDD_FILEEDIT, szCurrentPath);
		     return 0;

		  case LN_ENTER:

		     WinQueryDlgItemText (hwnd, IDD_FILEEDIT,
					  sizeof szCurrentPath,
					  szCurrentPath);
		     if (DosQPathInfo (szCurrentPath, 5, szBuffer,
				   sizeof szBuffer, 0L)) {
			WinMessageBox ( HWND_DESKTOP, hwnd, szBuffer,
					     "Unrecognised path", 1,
					     MB_OK | MB_ERROR);
			return 0;
		     } else {
			strcpy (szNewFile, szBuffer);
			Default.fIncHiddenFiles = fIncHidden;
			WinDismissDlg (hwnd, TRUE);
			return 0;
		     }
	       }
	       break;

	    case IDD_INCHIDDEN:
	       switch (SHORT2FROMMP (mp1)) {

		  case BN_CLICKED:
		     if (fIncHidden)
			fIncHidden = FALSE;
		     else
			fIncHidden = TRUE;
		     FillFileListBox (hwnd, szFileName, fIncHidden);
		     FillDirListBox (hwnd, szCurrentPath, fIncHidden);
		     return 0;
	       }
	       break;
	 }
	 break;

      case WM_COMMAND:

	 switch (COMMANDMSG(&msg)->cmd) {

	    case DID_OK:
	       WinQueryDlgItemText (hwnd, IDD_FILEEDIT,
				    sizeof szCurrentPath,
				    szCurrentPath);
	       if (DosQPathInfo (szCurrentPath, 5, szBuffer,
			     sizeof szBuffer, 0L)) {
		  WinMessageBox ( HWND_DESKTOP, hwnd, szBuffer,
				       "Unrecognised path", 1,
				       MB_OK | MB_ERROR);
		  return 0;
	       } else
		  if (strchr (szBuffer, '*') == 0
		      && strchr (szBuffer, '?') == 0) {
		     if (fIncHidden)
			usAttr = 0x0007;
		     else
			usAttr = 0x0005;
		     usSearchCount = 1;

		     DosFindFirst (szBuffer, &hDir, usAttr, &findbuf,
				   sizeof findbuf, &usSearchCount, 0L);
		     if (usSearchCount	== 0) {
			WinMessageBox (HWND_DESKTOP, hwnd, szBuffer,
				       "File not Found", 1, MB_OK | MB_ERROR);
			return 0;
		     } else {
			strcpy (szNewFile, szBuffer);
			Default.fIncHiddenFiles = fIncHidden;
			WinDismissDlg (hwnd, TRUE);
			return 0;
		     }
		  } else {
		     for (i = strlen(szBuffer); i >= 0; i--)
			if (szBuffer[i] == '\\')
			   break;
		     strcpy (szFileName, szBuffer + i + 1);
		     FillFileListBox (hwnd, szFileName, fIncHidden);
		     return 0;
		  }
	       break;
	    case DID_CANCEL:
	       WinDismissDlg (hwnd, FALSE);
	       return 0;
	 }
	 break;
   }
   return WinDefDlgProc (hwnd, msg, mp1, mp2);
}
//===================================================================
// Procedures for Configure dialog box
//===================================================================
VOID FillColorBox (HWND hwnd, SHORT iListID) {

   static CHAR szColors [16][12] = { "Black", "Dk. Blue", "Dk. Green",
      "Dk. Cyan", "Dk. Red", "Dk. Magenta", "Brown", "Gray", "Dk. Gray",
      "Blue", "Green", "Cyan", "Red", "Magenta", "Yellow", "White"};

   SHORT  i;

   WinSendDlgItemMsg (hwnd, iListID, LM_DELETEALL, NULL, NULL);

   for (i = 0; i < 16 ; i++ )
	 WinSendDlgItemMsg (hwnd, iListID, LM_INSERTITEM,
			      MPFROM2SHORT (LIT_END, 0),
			      MPFROMP (szColors[i]));
   return;
}
//------------------------------------------------------------------
//  Fill Character cell size box
//------------------------------------------------------------------
VOID FillCellSizeBox (HWND hwnd, SHORT iListID) {

   static CHAR szCellSizes [10][8] = {"8 x 8", "8 x 12", "8 x 14", "7 x 15",
      "7 x 25", "8 x 17", "12 x 16", "12 x 20", "12 x 22", "12 x 30"};

   SHORT  i, sStart, sEnd;

   WinSendDlgItemMsg (hwnd, iListID, LM_DELETEALL, NULL, NULL);

   WinSendDlgItemMsg (hwnd, iListID, LM_INSERTITEM,
		    MPFROM2SHORT (LIT_END, 0), MPFROMP (szCellSizes[0]));

   switch (VideoData.usAdapterType) {
      case 0:
      case 1:			       //CGA, no other cell sizes.
	 sStart = 2;
	 sEnd = 1;
	 break;

      case 2:			       //EGA
	 sStart = 1;
	 sEnd = 1;
	 break;

      case 3:			       //VGA
	 sStart = 2;
	 sEnd = 2;
	 break;

      case 7:			       //8514
	 sStart = 3;
	 sEnd = 9;
	 break;
      }
   for (i = sStart; i <= sEnd; i++ )
      WinSendDlgItemMsg (hwnd, iListID, LM_INSERTITEM,
		      MPFROM2SHORT (LIT_END, 0), MPFROMP (szCellSizes[i]));
   return;
}
//------------------------------------------------------------------
// GetFontSize - Determine font size.
//------------------------------------------------------------------
SHORT GetCellSize (SHORT sHeight, SHORT sWidth) {

   SHORT  i, sIndex = 1, sStart = 1;

   if (sHeight == sCellSize[0][1] && sWidth == sCellSize[0][0])
      return 0;

   if (VideoData.usAdapterType == 3)
      sStart = 2;
   else if (VideoData.usAdapterType == 7)
      sStart = 3;
   for (i = sStart; i < 10; i++, sIndex++ )
      if (sHeight == sCellSize[i][1] && sWidth == sCellSize[i][0])
	 break;
   return sIndex;
}
//------------------------------------------------------------------
//  Configure Dialog Box
//------------------------------------------------------------------
MRESULT EXPENTRY ConfigDlgProc (HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2)
   {
   static   SHORT    sSelect;
   static   CHAR     szAttribute[2];
   static   SHORT    sCHeight;
   static   SHORT    sCWidth;
   static   SHORT    sTabStop;
   static   BOOL     fDispASCII;
   static   BOOL     fScrollBars;
   static   BOOL     fLocalSaveWinPos;

   switch (msg) {

      case WM_INITDLG:

	 szAttribute[1] = Default.szAttribute[1];
	 sCHeight = Default.sCellHeight;
	 sCWidth = Default.sCellWidth;
	 sTabStop = Default.sTabStop;
	 fDispASCII = Default.fDispASCII;
	 fScrollBars = Default.fScrollBars;
	 fLocalSaveWinPos = FALSE;

	 FillColorBox (hwnd, IDD_FCOLOR);
	 FillColorBox (hwnd, IDD_BCOLOR);
	 FillCellSizeBox (hwnd, IDD_CELLSIZE);

	 WinSendDlgItemMsg (hwnd, IDD_BCOLOR, LM_SELECTITEM,
		   MPFROM2SHORT (((szAttribute[1]>>4) & 0x0f), 0),
		   MPFROM2SHORT (TRUE,0));

	 WinSendDlgItemMsg (hwnd, IDD_FCOLOR, LM_SELECTITEM,
		   MPFROM2SHORT ((szAttribute[1] & 0x0f), 0),
		   MPFROM2SHORT (TRUE,0));

	 WinSendDlgItemMsg (hwnd, IDD_CELLSIZE, LM_SELECTITEM,
		   MPFROM2SHORT (GetCellSize (sCHeight,sCWidth), 0),
		   MPFROM2SHORT (TRUE,0));

	 WinSendDlgItemMsg (hwnd, IDD_DEFASCII, BM_SETCHECK,
			    MPFROM2SHORT (fDispASCII, 0), NULL);

	 WinSendDlgItemMsg (hwnd, IDD_DEFHEX, BM_SETCHECK,
			    MPFROM2SHORT (!fDispASCII, 0), NULL);

	 WinSendDlgItemMsg (hwnd, IDD_DEFSCROLL, BM_SETCHECK,
			    MPFROM2SHORT (fScrollBars, 0), NULL);

	 WinSendDlgItemMsg (hwnd, IDD_SAVEWINPOS, BM_SETCHECK,
			    MPFROM2SHORT (fLocalSaveWinPos, 0), NULL);

	 WinSendDlgItemMsg (hwnd, IDD_SAVE, BM_SETCHECK,
			    MPFROM2SHORT (fSaveToDisk, 0), NULL);

	 WinSetDlgItemShort ( hwnd, IDD_DEFTABS, sTabStop, FALSE);
	 return 0;

      case WM_CONTROL:

	 if (SHORT1FROMMP (mp1) == IDD_BCOLOR	 ||
	     SHORT1FROMMP (mp1) == IDD_FCOLOR	 ||
	     SHORT1FROMMP (mp1) == IDD_CELLSIZE) {
	    sSelect = (USHORT) WinSendDlgItemMsg (hwnd,
		      SHORT1FROMMP (mp1), LM_QUERYSELECTION, 0L, 0L);
	 }
	 switch (SHORT1FROMMP (mp1)) {

	    case IDD_BCOLOR:
	       switch (SHORT2FROMMP (mp1)) {

		  case LN_SELECT:

		     szAttribute[1] = (szAttribute[1] & 0x0f)
				      | ((sSelect << 4) & 0xf0);
		     return 0;
	       }
	       break;

	    case IDD_FCOLOR:
	       switch (SHORT2FROMMP (mp1)) {

		  case LN_SELECT:

		     szAttribute[1] = (szAttribute[1] & 0xf0)
				      | (sSelect & 0x0f);
		     return 0;
	       }
	       break;

	    case IDD_CELLSIZE:
	       switch (SHORT2FROMMP (mp1)) {

		  case LN_SELECT:

		     if (sSelect != 0)
			switch (VideoData.usAdapterType) {
			   case 3:			   //VGA
			      sSelect++;
			      break;

			   case 7:			   //8514
			      sSelect += 2;
			      break;
			}
		     sCHeight = sCellSize [sSelect][1];
		     sCWidth = sCellSize [sSelect][0];
		     return 0;
	       }
	       break;

	    case IDD_DEFASCII:
	       switch (SHORT2FROMMP (mp1)) {

		  case BN_CLICKED:
		     fDispASCII = TRUE;
		     return 0;
	       }
	       break;

	    case IDD_DEFHEX:
	       switch (SHORT2FROMMP (mp1)) {

		  case BN_CLICKED:
		     fDispASCII = FALSE;
		     return 0;
	       }
	       break;

	    case IDD_DEFSCROLL:
	       switch (SHORT2FROMMP (mp1)) {

		  case BN_CLICKED:
		     if (fScrollBars)
			fScrollBars = FALSE;
		     else
			fScrollBars = TRUE;
		     return 0;
	       }
	       break;

	    case IDD_SAVEWINPOS:
	       switch (SHORT2FROMMP (mp1)) {

		  case BN_CLICKED:
		     if (fLocalSaveWinPos)
			fLocalSaveWinPos = FALSE;
		     else
			fLocalSaveWinPos = TRUE;
		     return 0;
	       }
	       break;

	    case IDD_SAVE:
	       switch (SHORT2FROMMP (mp1)) {

		  case BN_CLICKED:
		     if (fSaveToDisk)
			fSaveToDisk = FALSE;
		     else
			fSaveToDisk = TRUE;
		     return 0;
	       }
	       break;
	 }
	 break;

      case WM_COMMAND:

	 switch (COMMANDMSG(&msg)->cmd) {

		  case DID_OK:
		     Default.szAttribute[1] = szAttribute[1];
		     Default.sCellHeight = sCHeight;
		     Default.sCellWidth = sCWidth;
		     Default.fDispASCII = fDispASCII;
		     Default.fScrollBars = fScrollBars;
		     WinQueryDlgItemShort (hwnd, IDD_DEFTABS, &sSelect,
					   FALSE);
		     sSelect = min (max (1, sSelect), AVIOCOLS - 1);
		     Default.sTabStop = sSelect;
		     fSaveWinPos = fLocalSaveWinPos;
		     WinDismissDlg (hwnd, TRUE);
		     return 1;

		  case DID_CANCEL:
			  WinDismissDlg (hwnd, FALSE);
			  return 0;
	 }
   }
   return WinDefDlgProc (hwnd, msg, mp1, mp2);
}
//------------------------------------------------------------------
//  Font Size Dialog Box
//------------------------------------------------------------------
MRESULT EXPENTRY FontDlgProc (HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2)
   {
   static   SHORT    sSelect;
   static   SHORT    sCHeight;
   static   SHORT    sCWidth;

   switch (msg) {

      case WM_INITDLG:

	 sCHeight = Default.sCellHeight;
	 sCWidth = Default.sCellWidth;

	 FillCellSizeBox (hwnd, IDD_FONTS);

	 WinSendDlgItemMsg (hwnd, IDD_FONTS, LM_SELECTITEM,
		   MPFROM2SHORT (GetCellSize (sCHeight,sCWidth), 0),
		   MPFROM2SHORT (TRUE,0));
	 return 0;

      case WM_CONTROL:

	 switch (SHORT1FROMMP (mp1)) {

	    case IDD_FONTS:
	       switch (SHORT2FROMMP (mp1)) {

		  case LN_SELECT:
		     sSelect = (USHORT) WinSendDlgItemMsg (hwnd,
			 IDD_FONTS, LM_QUERYSELECTION, 0L, 0L);

		     if (sSelect != 0)
			switch (VideoData.usAdapterType) {
			   case 3:			   //VGA
			      sSelect++;
			      break;

			   case 7:			   //8514
			      sSelect += 2;
			      break;
			}
		     sCHeight = sCellSize [sSelect][1];
		     sCWidth = sCellSize [sSelect][0];
		     return 0;

		  case LN_ENTER:
		     Default.sCellHeight = sCHeight;
		     Default.sCellWidth = sCWidth;
		     WinDismissDlg (hwnd, TRUE);
		     return 0;

	       }
	       break;
	 }
	 break;

      case WM_COMMAND:

	 switch (COMMANDMSG(&msg)->cmd) {

		  case DID_OK:
		     Default.sCellHeight = sCHeight;
		     Default.sCellWidth = sCWidth;
		     WinDismissDlg (hwnd, TRUE);
		     return 0;

		  case DID_CANCEL:
		     WinDismissDlg (hwnd, FALSE);
		     return 0;
	 }
   }
   return WinDefDlgProc (hwnd, msg, mp1, mp2);
}
