/*
 EPSHeader

   File: arglist.h
   Author: J. Kercheval
   Created: Thu, 09/05/1991  20:03:53
*/
/*
 EPSRevision History

   J. Kercheval  Thu, 09/05/1991  20:09:57  creation
   J. Kercheval  Thu, 09/05/1991  21:47:54  add DestroyArgList()
   J. Kercheval  Wed, 09/11/1991  00:19:12  add ArgCopy()
   J. Kercheval  Wed, 09/25/1991  13:11:35  add sorted support
*/

/*
 * This module implements a dynamically allocated argument list in the format
 * used by the startup code in C.  Since this is dynamically increased as new
 * arguments are registered, this is a good way to keep track of unsorted
 * lists and to generate list to use as inputs to modules originally intended
 * as standalone utilities.  This module is very naive about running out of
 * memory.  If an out of memory error occurs, processing is halted via
 * external_cleanup() (an external) and an exit(1).  Sorted list processing
 * is done if the sorted BOOLEAN is true in the arglist.  These routines
 * assume and maintain a list with unique elements if the list is sorted.
 * all elements are string elements which are assumed to be non case
 * sensitive.
 */

#ifndef ARGLIST_HEADER
#define ARGLIST_HEADER

#ifndef BOOLEAN
#define BOOLEAN int
#define TRUE 1
#define FALSE 0
#endif

#define ARGLIST_SORTED TRUE
#define ARGLIST_NORMAL FALSE


/* file argument list structure */
struct ArgListStruct {
    char **argv;                /* file list array */
    unsigned int size;          /* number of array entries */
    unsigned int num_files;     /* number of files in array */
    unsigned int num_args;      /* number of elements in the array */
    BOOLEAN sorted;             /* true if arglist is a sorted list */
};

/* file argument list typedef */
typedef struct ArgListStruct *ArgList;


/*----------------------------------------------------------------------------
 *
 * CreateArgList() will allocate the memory needed for the use of an ArgList
 * variable and will set the initial values of the list.  If sorted_list is
 * TRUE then the list is maintained in sorted order when additions are made
 * through the register arg routines.
 *
 ---------------------------------------------------------------------------*/

ArgList CreateArgList(BOOLEAN sorted_list);


/*----------------------------------------------------------------------------
 *
 * DestroyArgList() will deallocate the memory used by the ArgList variable.
 *
 ---------------------------------------------------------------------------*/

void DestroyArgList(ArgList arglist);


/*----------------------------------------------------------------------------
 *
 * ArgToOutputStream() will output all files in argv to stdout
 *
 ---------------------------------------------------------------------------*/

void ArgToOutputStream(FILE * output_file, ArgList arglist);


/*----------------------------------------------------------------------------
 *
 * ArgIsMember() returns true if s is a member of the string array arg
 *
 ---------------------------------------------------------------------------*/

BOOLEAN ArgIsMember(ArgList arglist, char *arg);


/*----------------------------------------------------------------------------
 *
 * ArgCopy() copies all the elements of argfrom to argto while adding
 * internal bookkeeping variables
 *
 ---------------------------------------------------------------------------*/

void ArgCopy(ArgList argto, ArgList argfrom);


/*----------------------------------------------------------------------------
 *
 * ArgRegisterArg() places an argument in the ArgList array and increments
 * arglist->num_args
 *
 ---------------------------------------------------------------------------*/

void ArgRegisterArg(ArgList arglist, char *arg);


/*----------------------------------------------------------------------------
 *
 * ArgRegister_name() place a file name into the ArgList array for later post
 * processing and increments arglist->num_files and arglist->num_args
 *
 ---------------------------------------------------------------------------*/

void ArgRegisterName(ArgList arglist, char *name);

#endif                          /* ARGLIST_HEADER */
