/*
 EPSHeader

   File: flags.c
   Author: J. Kercheval
   Created: Thu, 09/05/1991  20:15:26
*/
/*
 EPSRevision History

   J. Kercheval  Thu, 09/05/1991  20:19:46  creation
   J. Kercheval  Wed, 09/11/1991  01:43:31  add support for -cx
   J. Kercheval  Tue, 09/17/1991  19:39:31  add support for case_sensitive flag
   J. Kercheval  Fri, 10/04/1991  10:43:10  add ck flag
*/

#include <stdio.h>
#include <string.h>

#include "flags.h"
#include "log.h"


/*----------------------------------------------------------------------------
 *
 * init_ASM_flagsa() set the initial values of the ASM language option flags
 *
 ---------------------------------------------------------------------------*/

void init_ASM_flags(Flags * flags)
{
    flags->af = TRUE;           /* parse procedure labels by default */
    flags->al = TRUE;           /* parse local labels by default */
    flags->am = TRUE;           /* parse macro labels by default */
    flags->as = TRUE;           /* parse struc labels by default */
    flags->au = TRUE;           /* parse union labels by default */
    flags->ad = TRUE;           /* parse catstr labels by default */
}


/*----------------------------------------------------------------------------
 *
 * init_C_flags() set the initial values of the c language option flags
 *
 ---------------------------------------------------------------------------*/

void init_C_flags(Flags * flags)
{
    flags->cf = TRUE;           /* parse function calls by default */
    flags->cp = TRUE;           /* parse function prototypes by default */
    flags->cm = TRUE;           /* parse macros by default */
    flags->cs = TRUE;           /* parse global structs by default */
    flags->ct = TRUE;           /* parse global typedefs default */
    flags->ce = TRUE;           /* parse global enums by default */
    flags->ck = TRUE;           /* parse global enum constants by default */
    flags->cu = TRUE;           /* parse global unions by default */
    flags->cv = TRUE;           /* parse global variables by default */
    flags->cc = TRUE;           /* parse global classes by default */
    flags->cd = TRUE;           /* parse defines by default */
    flags->cx = TRUE;           /* parse externs by default */
    flags->ci = TRUE;           /* parse statics by default */
}


/*----------------------------------------------------------------------------
 *
 * init_output_flags() set the initial values of the output option flags
 *
 ---------------------------------------------------------------------------*/

void init_output_flags(Flags * flags)
{
    flags->oe = TRUE;           /* The Epsilon tags format is default */
    flags->og = FALSE;          /* GNU tags format used in GNU EMACS */
    flags->os = FALSE;          /* Space delimited format */
    flags->om = FALSE;          /* MicroSoft Error format */
}


/*----------------------------------------------------------------------------
 *
 * init_flags() set the initial values of the all option flags in FLAGS struct
 *
 ---------------------------------------------------------------------------*/

void init_flags(Flags * flags)
{
    /* tag_type is initially C type parsing */
    flags->tag_type = C;

    /* normally use the fully qualified pathname */
    flags->use_relative_pathnames = FALSE;

    /* normally we are noisy and verbose */
    flags->quiet = FALSE;

    /* normal logging defaults to overwrite */
    flags->log_overwrite = TRUE;

    /* normally we will sort the output to remove duplicate lines */
    flags->sort_tags = TRUE;

    /* normally a case insensitive sort is performed */
    flags->case_sensitive = FALSE;

    /* normally we will output to stdout */
    flags->output_file = FALSE;

    /* init the language specific flags */
    init_ASM_flags(flags);
    init_C_flags(flags);

    /* init the output flags */
    init_output_flags(flags);
}


/*----------------------------------------------------------------------------
 *
 * parse_ASM_flags() set the option flags for ASM command line switch
 *
 ---------------------------------------------------------------------------*/

void parse_ASM_flags(char *argv, Flags * flags)
{
    int i, end;
    char message[80];

    /* set tag_type */
    if (flags->tag_type != MERGE && flags->tag_type != SORT) {
        flags->tag_type = ASM;
        log_message("# Using ASM style tagging");
    }
    else {
        return;
    }

    /* determine the number of modifiers */
    end = strlen(argv);

    /* if customized parsing, all flags start off false */
    if (end > 2) {

        flags->af = FALSE;
        flags->am = FALSE;
        flags->al = FALSE;
        flags->as = FALSE;
        flags->au = FALSE;
        flags->ad = FALSE;
    }
    else {

        /* init flags */
        init_ASM_flags(flags);
    }

    /* parse the standard modifiers */
    for (i = 2; i < end; i++) {

        /* set flags as required */
        switch (argv[i]) {

            case 'f':           /* proc labels */
            case 'F':
                flags->af = TRUE;
                break;

            case 'l':           /* local labels */
            case 'L':
                flags->al = TRUE;
                break;

            case 'm':           /* macros */
            case 'M':
                flags->am = TRUE;
                break;

            case 's':           /* struc labels */
            case 'S':
                flags->as = TRUE;
                break;

            case 'u':           /* union labels */
            case 'U':
                flags->au = TRUE;
                break;

            case 'd':           /* data definition labels */
            case 'D':
                flags->ad = TRUE;
                break;

            default:
                sprintf(message,
                        "# Ignoring unknown ASM switch modifier ( %c )",
                        argv[i]);
                log_message(message);
                break;
        }
    }
}


/*----------------------------------------------------------------------------
 *
 * parse_C_flags() set the option flags for C command line switch
 *
 ---------------------------------------------------------------------------*/

void parse_C_flags(char *argv, Flags * flags)
{
    int i, end;
    char message[80];

    /* set tag_type */
    if (flags->tag_type != MERGE && flags->tag_type != SORT) {
        flags->tag_type = C;
        log_message("# Using C style tagging");
    }
    else {
        return;
    }

    /* determine the number of modifiers */
    end = strlen(argv);

    /* if customized parsing, all flags start off false */
    if (end > 2) {

        flags->cf = FALSE;
        flags->cp = FALSE;
        flags->cm = FALSE;
        flags->cs = FALSE;
        flags->ct = FALSE;
        flags->ce = FALSE;
        flags->ck = FALSE;
        flags->cu = FALSE;
        flags->cv = FALSE;
        flags->cc = FALSE;
        flags->cd = FALSE;
        flags->cx = FALSE;
        flags->ci = FALSE;
    }
    else {

        /* init flags */
        init_C_flags(flags);
    }

    /* parse the switch modifiers */
    for (i = 2; i < end; i++) {

        /* set flags as required */
        switch (argv[i]) {
            case 'f':           /* function calls (procedures) */
            case 'F':
                flags->cf = TRUE;
                break;

            case 'p':           /* function calls (procedures) */
            case 'P':
                flags->cp = TRUE;
                break;

            case 'm':           /* global macros */
            case 'M':
                flags->cm = TRUE;
                break;

            case 's':           /* global struct */
            case 'S':
                flags->cs = TRUE;
                break;

            case 't':           /* global typedef */
            case 'T':
                flags->ct = TRUE;
                break;

            case 'e':           /* global enum */
            case 'E':
                flags->ce = TRUE;
                break;

            case 'k':           /* global enum constants */
            case 'K':
                flags->ck = TRUE;
                break;

            case 'u':           /* global union */
            case 'U':
                flags->cu = TRUE;
                break;

            case 'v':           /* global variables */
            case 'V':
                flags->cv = TRUE;
                break;

            case 'c':           /* global classes */
            case 'C':
                flags->cc = TRUE;
                break;

            case 'd':           /* defines */
            case 'D':
                flags->cd = TRUE;
                break;

            case 'x':           /* extern defines */
            case 'X':
                flags->cx = TRUE;
                break;

            case 'i':           /* static declarations */
            case 'I':
                flags->ci = TRUE;
                break;

            default:
                sprintf(message,
                        "# Ignoring unknown C switch modifier ( %c )",
                        argv[i]);
                log_message(message);
                break;
        }
    }
}


/*----------------------------------------------------------------------------
 *
 * parse_output_flag() set the option flag for output format, These formats
 * are mutually exclusive from the point of view of this routine.
 *
 ---------------------------------------------------------------------------*/

void parse_output_flags(char *argv, Flags * flags)
{
    int i, end;

    /* init flags */
    init_output_flags(flags);

    end = strlen(argv);

    /* parse the switch modifiers */
    for (i = 2; i < end; i++) {

        /* set flags as required */
        switch (argv[i]) {
            case 'e':           /* Use Epsilon format */
            case 'E':
                flags->oe = TRUE;
                flags->og = FALSE;
                flags->os = FALSE;
                flags->om = FALSE;
                break;

            case 'g':           /* Use GNU format */
            case 'G':
                flags->oe = FALSE;
                flags->og = TRUE;
                flags->os = FALSE;
                flags->om = FALSE;
                break;

            case 's':           /* Use Space Delimite format */
            case 'S':
                flags->oe = FALSE;
                flags->og = FALSE;
                flags->os = TRUE;
                flags->om = FALSE;
                break;

            case 'm':           /* Use Space Delimite format */
            case 'M':
                flags->oe = FALSE;
                flags->og = FALSE;
                flags->os = FALSE;
                flags->om = TRUE;
                break;

            default:
                break;
        }
    }
}
