PROGRAM MkFntNdx;

{******************************************************************************
**
**  Author: Robert W. Bloom
**
**  Function:  This program reads a standard HP LaserJet-compatible font file
**             and develops a index to the characters in the file.  This index
**             is output to file to be used by the program 'SIGNS'.
**             See Signs.DOC for more info.
**
*****************************************************************************}

CONST
          Date = 'v5.0, 25 Sep 89';    {date of last revision of this prog}

TYPE
    CHAR_INDEX_RECORD = RECORD {points to char in soft font file}
    character : CHAR;          {the character}
     position : WORD;          {where found in font file?}
   top_offset : INTEGER;       {how far down does character start}
  left_offset : INTEGER;       {how far left does character start}
        width : INTEGER;       {how wide is it}
       height : INTEGER;       {how high}
      delta_x : INTEGER        {how far should 'cursor' move?}
    END; {record}

    IN_FILE_TYPE = FILE OF CHAR;
    OUT_FILE_TYPE = FILE OF CHAR_INDEX_RECORD;

VAR
       input_fn : IN_FILE_TYPE;
      output_fn : OUT_FILE_TYPE;
      ndx_array : ARRAY[0..255] OF CHAR_INDEX_RECORD;
      loop_ctrl : BYTE;

PROCEDURE init;                                      FORWARD;
PROCEDURE process;                                   FORWARD;
PROCEDURE findheader(VAR cnt : INTEGER);             FORWARD;
PROCEDURE findchar(VAR cnt : INTEGER);               FORWARD;
PROCEDURE outndxfile;                                FORWARD;


PROCEDURE init;
LABEL   restart;                   {for error recovery}
VAR
    ans,ifn,ofn : STRING[14];
         i,err : INTEGER;
BEGIN
restart:
    IF (paramcount = 0) OR (loop_ctrl > 0) THEN BEGIN
        WRITELN('A <return> without a filename will terminate program.');
        WRITELN;

        WRITELN('If not specified, an extension of .FNT will be assumed.');
        WRITE('Enter filename of input file ->');
        READLN(ans);
        IF ans = '' THEN BEGIN
           WRITELN;
           WRITELN('<<< MkFntNdx completed >>>');
           WRITELN;
           halt       {normal exit - not classic pascal!}
        END ELSE
           ifn := ans;
    END ELSE
       ifn := ParamStr(1);
   {end if a input filename was not given as a parameter}

    i := POS('.',ifn);
    IF i = 0 THEN BEGIN
       ofn := ifn + '.FNX';     {copy to the output file name}
       ifn := ifn + '.FNT'      {add extension if not given}
    END ELSE
       ofn := COPY(ifn,1,POS('.',ifn)-1) + '.FNX';

    ASSIGN(input_fn,ifn);
    {$I-} RESET(input_fn); {$I+}
    err := IORESULT;
    IF err <> 0 THEN BEGIN
        WRITELN('ERROR:',err,' Problem opening input file!'^G);
        GOTO restart
    END;

    ASSIGN(output_fn,ofn);
    {$I-} REWRITE(output_fn); {$I+}
    err := IORESULT;
    IF err <> 0 THEN BEGIN
        WRITELN('ERROR:',err,' Problem in opening output file!'^G);
        GOTO restart
    END;

    WRITELN;
    WRITELN('Initializing font index array');
    FOR i := 0 TO 255 DO BEGIN
        ndx_array[i].character   := CHR(i);
        ndx_array[i].position    := 0;
        ndx_array[i].top_offset  := 0;
        ndx_array[i].left_offset := 0;
        ndx_array[i].width       := 0;
        ndx_array[i].height      := 0;
        ndx_array[i].delta_x     := 0
    END  {for}
END; {procedure init}

PROCEDURE process;
VAR
      cnt : INTEGER; {count in the font file}
BEGIN
    cnt := 0;
    WRITELN;
    WRITELN('Font header info');
    findheader(cnt);
    WRITELN;
    WRITELN('Character processing:');
    WRITELN('Chr Position    Top_Offset  Left_Offset   Width      Height      Delta_X');
    WHILE not EOF(input_fn) DO findchar(cnt);
    ndx_array[32].delta_x := ndx_array[0].delta_x {default pitch for <sp> char}
END; {procedure process}

PROCEDURE findheader(VAR cnt:INTEGER);
VAR
    c,hc,lc : char;
    i : INTEGER;
lobyte,hibyte : INTEGER;
 found : BOOLEAN;
 pitch : REAL;
BEGIN
    found := FALSE;
    WHILE not EOF(input_fn) AND not found DO BEGIN
        READ(input_fn,c); cnt := cnt+1;
        IF ORD(c) = 27 THEN BEGIN                 {look for an <esc>}
            READ(input_fn,c); cnt := cnt+1;
            IF c = ')' THEN BEGIN                 {look for an )}
                READ(input_fn,c); cnt := cnt+1;
                IF c = 's' THEN BEGIN                    {followed by a 's'}
                    READ(input_fn,c); cnt := cnt+1;
                    WHILE (c >= '0') AND (c <= '9') DO BEGIN
                        READ(input_fn,c);
                        cnt := cnt+1
                    END; {skip over font header size numbers}
                    IF c = 'W' THEN BEGIN             {found it}
                        found := TRUE;
                        FOR i := 1 to 6 DO
                            READ(input_fn,c);  {discard next 6 chars}
                        cnt := cnt + 6;
                        READ(input_fn,hc);   {hi byte of baseline distance}
                        READ(input_fn,lc);   {lo}
                        ndx_array[0].top_offset := 256*ORD(hc)+ORD(lc);
                        WRITELN('   Baseline = ',ndx_array[0].top_offset);
                        READ(input_fn,hc);   {hi byte of max cell width}
                        READ(input_fn,lc);   {lo}
                        ndx_array[0].width := 256*ORD(hc)+ORD(lc);
                        WRITELN('   Maximum cell width = ',ndx_array[0].width);
                        READ(input_fn,hc);   {hi byte of max cell height}
                        READ(input_fn,lc);   {lo}
                        ndx_array[0].height := 256*ORD(hc)+ORD(lc);
                        WRITELN('   Maximum cell Height = ',ndx_array[0].height);
                        cnt := cnt + 6;
                        FOR i := 1 to 4 DO
                            READ(input_fn,c);  {discard next 4 chars}
                        cnt := cnt + 4;
                        READ(input_fn,hc);   {hi byte of default char spacing}
                        READ(input_fn,lc);   {lo}
                        cnt := cnt + 2;
                        pitch := (256*ORD(hc)+ORD(lc)) / 4;
                        ndx_array[0].delta_x := ROUND(pitch);
                        WRITELN('   Default Char spacing = ',ndx_array[0].delta_x)
                    END {end if c='W'}
                END {end if c='s'}
            END {end if c=')'}
        END {end if c=<esc>}
    END {while not found}
END; {procedure findheader}

PROCEDURE findchar(VAR cnt:INTEGER);
VAR
    c,hc,lc : char;
    i : INTEGER;
lobyte,hibyte,fnd_chr_num,errcode : INTEGER;
 found : BOOLEAN;
strnum : STRING[3];
pitch : REAL;
BEGIN
    found := FALSE;
    WHILE not EOF(input_fn) AND not found DO BEGIN
        READ(input_fn,c); cnt := cnt+1;
        IF ORD(c) = 27 THEN BEGIN                 {look for an <esc>}
            READ(input_fn,c); cnt := cnt+1;
            IF c = '*' THEN BEGIN                    {followed by a '*'}
                READ(input_fn,c); cnt := cnt+1;
                IF c = 'c' THEN BEGIN                    {followed by a 'c'}
                    READ(input_fn,c); cnt := cnt+1;
                    strnum := '';
                    WHILE (c >= '0') AND (c <= '9') DO BEGIN
                        strnum := strnum + c;
                        READ(input_fn,c); cnt := cnt+1
                    END;
                    val(strnum,fnd_chr_num,errcode);     {maybe this is it}
                    IF c = 'E' THEN BEGIN
                        found := TRUE;
                        WRITE(' ',CHR(fnd_chr_num));
                        READ(input_fn,c);
                        READ(input_fn,c);  {discard next 2 chars}
                        cnt:=cnt+2;
                        READ(input_fn,c); cnt := cnt+1;
                        WHILE c <> 'W' DO BEGIN  {find the 'W'}
                            READ(input_fn,c);
                            cnt := cnt+1
                        END; {skip over font header size numbers}
                        FOR i := 1 to 6 DO
                            READ(input_fn,c);  {discard next 6 chars}
                        cnt := cnt + 6;
                        READ(input_fn,hc);   {hi byte of left offset}
                        READ(input_fn,lc);   {lo}
                        ndx_array[fnd_chr_num].left_offset := 256*ORD(hc)+ORD(lc);
                        READ(input_fn,hc);   {hi byte of topoffset}
                        READ(input_fn,lc);   {lo}
                        ndx_array[fnd_chr_num].top_offset := 256*ORD(hc)+ORD(lc);
                        READ(input_fn,hc);   {hi byte of char width}
                        READ(input_fn,lc);   {lo}
                        ndx_array[fnd_chr_num].width := 256*ORD(hc)+ORD(lc);
                        READ(input_fn,hc);   {hi byte of char height}
                        READ(input_fn,lc);   {lo}
                        ndx_array[fnd_chr_num].height := 256*ORD(hc)+ORD(lc);
                        READ(input_fn,hc);   {hi byte of char delta x}
                        READ(input_fn,lc);   {lo}
                        pitch := (256*ORD(hc)+ORD(lc)) / 4;
                        ndx_array[fnd_chr_num].delta_x := ROUND(pitch);
                        cnt := cnt + 10;
                        ndx_array[fnd_chr_num].position := cnt;
                        WITH ndx_array[fnd_chr_num] DO
                            WRITELN(position:8,Top_Offset:12,left_Offset:12,Width:12,Height:12,Delta_X:12)
                    END {if c='E'}
                END {if c=the char}
            END {if c='c'}
        END {if c='*'}
    END {if c=<esc>}
END; {procedure findchar}

PROCEDURE outndxfile;
VAR
    i : INTEGER;
BEGIN
    WRITELN;
    WRITE('Writing output file ...');
    FOR i:=0 to 255 DO
        WRITE(output_fn,ndx_array[i]);
    CLOSE(input_fn);
    CLOSE(output_fn);
    WRITELN(' completed.');
    WRITELN; WRITELN;
    loop_ctrl := loop_ctrl + 1
END; {procedure outndxfile}

BEGIN
    WRITELN('<<< MkFntNdx ',Date,' >>>');
    WRITELN;
    WRITELN('This programs creates a ''index'' file to a HP LaserJet-compatible soft font');
    WRITELN('file to be used by ''Signs''.  Signs uses the fontfile and the associated');
    WRITELN('index to create signs and banners.  The index file will have the same name as');
    WRITELN('the font file but with the extension .FNX.');
    WRITELN;
    loop_ctrl := 0;
    WHILE loop_ctrl < 100 DO BEGIN
        init;                     {'halt' if no filename given}
        process;
        outndxfile
    END; {while}
    WRITELN;
    WRITELN('<<< MkFntNdx completed >>>')
END.
