-- ========================== begin comment ==================================
-- dlring.adb
--
--
--             *******    **  PACKAGE BODY  **       ******
--                        Doubly-Linked Ring
--
--     There would be some benefit in adding a storage management
--     capability, such as a free list handler, to this package.
--     Also, a storage_error exception would enhance the reliability
--     of the Insert procedure.
-- ========================== end comment ====================================
 
package body DLRing is
           --
        ZERO : constant INTEGER := 0;  -- named constant

        type CELL is
             record
                    ITEM        : DLRing_Type;
                    PREV_CELL   : POINTER;
                    NEXT_CELL   : POINTER;
             end record;

   function Is_Count (Finger : Finger_Type) return Natural is
   begin
        return Finger.Cntr;
   end Is_Count;

   procedure ROTATE   (DIRECTION : in DIRECTION_TYPE;
                       FINGER    : in out FINGER_TYPE) is
   begin  -- ROTATE
          --
        if (DIRECTION = FORWARD) then
           FINGER.PTR := FINGER.PTR.NEXT_CELL;
        else
           FINGER.PTR := FINGER.PTR.PREV_CELL;
        end if;
   exception
        when CONSTRAINT_ERROR =>
            Text_IO.Put_Line("Constraint Error during Rotate ");

   end ROTATE;


   procedure INSERT   (ITEM      : in     DLRing_Type;
                       FINGER    : in out FINGER_TYPE) is
             ForWard_Walker  : POINTER;
             BackWard_Walker : POINTER;
   begin  -- INSERT
          --
         if FINGER.CNTR = ZERO then
            FINGER.PTR  := new CELL'(ITEM,null, null);
            FINGER.PTR.NEXT_CELL := FINGER.PTR;
            FINGER.PTR.PREV_CELL := FINGER.PTR;
         else
            BackWard_Walker  := FINGER.PTR.PREV_CELL;
            ForWard_Walker   := FINGER.PTR;
            FINGER.PTR       := new CELL'(ITEM, BackWard_Walker,
                                                ForWard_Walker);
            BackWard_Walker.NEXT_CELL :=  FINGER.PTR;
            ForWard_Walker.PREV_CELL  :=  FINGER.PTR;
         end if;

         FINGER.CNTR := FINGER.CNTR + 1;  -- could be dangerous unless every
                                          -- subprogram handles the count in
                                          -- consistent manner.
 
   exception
         when others =>
            Text_IO.Put_Line("Some error during Insert ");
   end INSERT;

   procedure REPLACE  (ITEM      : in     DLRing_Type;
                       FINGER    : in out FINGER_TYPE) is
   begin --  REPLACE

             FINGER.PTR.ITEM  := ITEM;

   end REPLACE;

   procedure DELETE   (FINGER       : in out FINGER_TYPE;
                       EMPTY_ERROR  :    out BOOLEAN)   is

             BackWard_Walker, ForWard_Walker  : POINTER;

   begin

        EMPTY_ERROR := FALSE;
        if FINGER.CNTR  = ZERO  then
           EMPTY_ERROR  := TRUE;
        elsif
           FINGER.CNTR = 1 then
           FINGER.PTR  := null;
           FINGER.CNTR := ZERO;
        else
           BackWard_Walker           := FINGER.PTR.PREV_CELL;
           ForWard_Walker            := FINGER.PTR.NEXT_CELL;
           BackWard_Walker.NEXT_CELL := ForWard_Walker;
           ForWard_Walker.PREV_CELL  := BackWard_Walker;
           FINGER.CNTR               := FINGER.CNTR - 1;
           FINGER.PTR                := ForWard_Walker;
        end if;
   end DELETE;

   procedure PEEK     (ITEM         :    out DLRing_Type;
                       FINGER       : in out FINGER_TYPE;
                       EMPTY_ERROR  :    out BOOLEAN)   is
   begin

        if (FINGER.CNTR  /=  ZERO)  then
            ITEM        := FINGER.PTR.ITEM;
            EMPTY_ERROR := FALSE;
        else
            EMPTY_ERROR  := TRUE;
        end if;
   exception
        when Constraint_Error =>
           Text_IO.Put_Line("Constraint Error in Peek ");
   end PEEK;


   procedure SEARCH   (FINGER       : in out FINGER_TYPE;
                       ITEM         : in     DLRing_Type;
                       FOUND        :    out BOOLEAN)   is

             NUMBER : INTEGER := FINGER.CNTR;

   begin -- SEARCH

         for I in 1 .. NUMBER  loop
             if (EQUAL(FINGER.PTR.ITEM, ITEM)) then
                 FOUND := TRUE;
                 return;
             else
                 ROTATE(FORWARD, FINGER);
             end if;
         end loop;
         FOUND := FALSE;
         ROTATE(FORWARD, FINGER);
   exception
         when CONSTRAINT_ERROR =>
              Text_IO.Put_Line("Constraint error in Rotate ");
   end SEARCH;

end DLRing;
-- ++++++++++++++++++++ End of Package Body +++++++++++++++++++