/*
 *  Routines for sampling from Pro-Audio Spectrum 16bit soundcards
 *  These routines require the libraries from the MediaVision SDK.
 *  This SDK is available from ftp.uwp.edu:/pub/msdos/proaudio/passdk30.zip
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "freq.h"

#ifdef SC_PAS16

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <math.h>
#include <string.h>
#include <fcntl.h>
#include <io.h>
#include <graphics.h>

#include <pcmio.h>                   // Standard PCM record functions
#include <state.h>                   // Need this for Mixer code
#include <binary.h>                  // Hardware values
#include "extern.h"

/* Use default DMA and IRQ values, 4k DMA buffer divided into 4 parts */
#define DMA         -1
#define IRQ         -1
#define DMASize      4
#define DMADivisions 4
#define stereo       0
#define Compression  0
#define SampleSize  16
/* Variables for saving/restoring mixer values */
int LeftChannel;
int RightChannel;
int mv;           /* Handle for the mixer device */
double mixer_offset=0,mixer_slope=1;

/* Function prototypes */
void reset_pas16(void);
void halt_pas16(void);
void cleanup_pas16(void);
void recordblock_pas16(void far *buffer);
void set_mixer_pas16(int mix,int level);
int GetOutputMixer(char *name,int channel);
int GetInputMixer(char *name,int channel);
void SetOutputMixer(char *name,int level);
void SetInputMixer(char *name,int level);
void CalibrateMixer(void);

/* Turn off stack checking for this routine */
#pragma option -N-
void far callback1()
{
   /*
    *  Callback function.  This function is called every time a buffer has
    *  been filled.  It sets a flag so the main loop recognises and processes
    *  the buffer.
    */
   flag[record_buffer]=1;
   if(++record_buffer>=BUFFERS)
      record_buffer=0;
}
/* Restore stack checking to the command-line specified state */
#pragma option -N.

void init_pas16(void)
{
   reset_soundcard=reset_pas16;
   halt_soundcard=halt_pas16;
   cleanup_soundcard=cleanup_pas16;
   recordblock=recordblock_pas16;
   set_mixer=set_mixer_pas16;
   sample_size=16;

   /*
    *  Initialize link to Mixer control routines
    *  Then save current mixer settings and turn off the PCM output
    *  mixer to stop feedback and clicking noises.
    */
   DOUT("PAS: Initializing mixer connection");
   if ((mv = open ("MVPROAS",O_RDWR )) == -1)
   {
      mixers=0;
      puts("PAS: Unable to establish mixer connection.");
      puts("Press any key to continue ...");
      getch();
   }
   else
   {
      int in,out;
      long sumx=0,sumy=0,sumxy=0,sumxx=0,sumyy=0,count=0;

      mixers=1;
      DOUT("PAS: Getting mixer values");
      mic_level=GetInputMixer("MIC",1);
      ext_level=GetInputMixer("EXT",1);
      int_level=GetInputMixer("INT",1);
      if(mic_level>100) mic_level=100;
      if(ext_level>100) ext_level=100;
      if(int_level>100) int_level=100;
      LeftChannel=GetInputMixer("PCM",1);
      RightChannel=GetInputMixer("PCM",2);

      /* Get proportionality factor for mixer settings */
      for(in=1,out=0;(in<100) && (out<100);in++)
      {
         SetOutputMixer("PCM",in);
         out=GetOutputMixer("PCM",1);
         count++;
         sumy+=in;
         sumx+=out;
         sumxy+=in*out;
         sumyy+=in*in;
         sumxx+=out*out;
      }
      mixer_offset=(double)(sumy*sumxx-sumx*sumxy)/(double)(count*sumxx-sumx*sumx);
      mixer_slope=(double)(sumxy-mixer_offset*sumx)/(double)sumxx;
      SetOutputMixer("PCM",0);
   }
}

void reset_pas16(void)
{
   int i;

   DOUT("PAS: Enabling PCM buffering");
   if(OpenPCMBuffering(DMA,IRQ,DMASize,DMADivisions)!=0)
   {
      cleanup_pas16();
      closegraph();
      puts("Error trying to open PCM buffering.");
      exit(1);
   }
   DOUT("PAS: Setting PCM state");
   /* Round sampling rate to a valid value for the PAS card */
   /* And correct it using the fudgefactor */
   if((fudgefactor>0.5) && (fudgefactor<2))
      SampleRate*=fudgefactor;
   i=floor(1193180.0/SampleRate+0.5);
   if(i<1) i=1;
   SampleRate=floor(1193180.0/(double)i+0.5);
   if(PCMState(SampleRate,stereo,Compression,SampleSize)!=0)
   {
      cleanup_pas16();
      closegraph();
      puts("Error setting sample rate.");
      exit(1);
   }
   if((fudgefactor>0.5) && (fudgefactor<2))
      SampleRate/=fudgefactor;
   DOUT("PAS: Setting up the buffers");
   // Reset the buffer pointers
   queue_buffer=0;     // Pointer to next buffer to be queued
   record_buffer=0;    // Pointer to next buffer to be filled
   process_buffer=0;   // Pointer to next buffer to be FFTed
   for(i=0;i<BUFFERS;i++)
      flag[i]=0;
   /*
    *  Queue up all but the last buffer.  The last is queued
    *  up below, with the RecordThisBlock call.
    *  DMA transfer does not start until the Record Block call is made.
    */
   for(i=0;i<BUFFERS-1;i++)
   {
      if(QueueThisBlock((char far *)buffer[queue_buffer],fftlen*2*(stereo+1),callback1)!=0)
      {
         cleanup_pas16();
         closegraph();
         puts("Error queueing block.");
         exit(1);
      }
      if(++queue_buffer>=BUFFERS)
         queue_buffer=0;
   }
   /*
    *  This function starts the DMA process.
    */
   //recordblock_pas16((char far *)buffer[queue_buffer]);
   DOUT("Starting the recording process");
   if(RecordThisBlock((char far *)buffer[queue_buffer],fftlen*2*(stereo+1),callback1)==0)
   {
      cleanup_pas16();
      closegraph();
      puts("Error recording block.");
      exit(1);
   }
   if(++queue_buffer>=BUFFERS)
      queue_buffer=0;
}

void halt_pas16(void)
{
   /*
    *  Shut down the DMA system.
    */
   DOUT("PAS: Closing the PCM buffering");
   ClosePCMBuffering();
}

void cleanup_pas16(void)
{
   /*
    *  Shut down the DMA system.
    */
   DOUT("PAS: Closing the PCM buffering");
   ClosePCMBuffering();

   if(mixers)
   {
      /* Restore mixers to their original value. */
      DOUT("PAS: Restoring mixer levels");
      SetOutputMixer("LEFT PCM",LeftChannel);
      SetOutputMixer("RIGHT PCM",RightChannel);

      close(mv);
   }
}

void recordblock_pas16(void far *buffer)
{
   /*
    *  Place a buffer on the queue to be recorded,
    *  and start the DMA process if not currently active
    */
   if(QueueThisBlock((char far *)buffer,fftlen*2*(stereo+1),callback1)!=0)
   {
      cleanup_pas16();
      closegraph();
      puts("Error queueing block.");
      exit(1);
   }
}

void set_mixer_pas16(int mix,int level)
{
   /* Set a mixer level on the PAS16 card */
   if(mix==MIXER_EXT)
      SetInputMixer("EXT",level);
   else if(mix==MIXER_INT)
      SetInputMixer("INT",level);
   else if(mix==MIXER_MIC)
      SetInputMixer("MIC",level);
}

void SetInputMixer(char *name,int level)
{
   int i;
   char buffer[80];

   /* Scale the values to fit the actual readings */
   level=(int)floor(level*mixer_slope+mixer_offset+0.5);
   if(level<0) level=0;
   if(level>100) level=100;

   DOUT("Setting input mixer level");
   sprintf(buffer,"SET INPUT MIXER %s TO %d\n",name,level);
   write(mv,buffer,strlen(buffer));
   read (mv,buffer,80);
   for(i=0;(buffer[i]!='\n') && (i<80);i++);
   buffer[i]=0;
   DOUT(buffer);
}

void SetOutputMixer(char *name,int level)
{
   int i;
   char buffer[80];

   /* Scale the values to fit the actual readings */
   level=(int)floor(level*mixer_slope+mixer_offset+0.5);
   if(level<0) level=0;
   if(level>100) level=100;

   DOUT("Setting output mixer level");
   sprintf(buffer,"SET OUTPUT MIXER %s TO %d\n",name,level);
   write(mv,buffer,strlen(buffer));
   read (mv,buffer,80);
   for(i=0;(buffer[i]!='\n') && (i<80);i++);
   buffer[i]=0;
   DOUT(buffer);
}

int GetInputMixer(char *name,int channel)
{
   char buffer[80];
   int left=0,right=0,i,index;

   DOUT("Reading input mixer level");
   sprintf(buffer,"GET INPUT MIXER %s\n",name);
   write(mv,buffer,strlen(buffer));
   read (mv,buffer,80);
   for(i=0;(buffer[i]!='\n') && (i<80);i++);
   buffer[i]=0;
   DOUT(buffer);
   index=0;
   for(i=0;buffer[i] && buffer[i]!='\t';i++);
   if(buffer[i]=='\t')
   {
      buffer[i]=0;
      left=atoi(buffer+index);
      if(left<0) left=0;
      if(left>100) left=100;
      i++;
   }
   /*sw1=buffer[i];    +/- flag for mixer active */
   i+=2;
   index=i;
   for(;buffer[i] && buffer[i]!='\t';i++);
   if(buffer[i]=='\t')
   {
      buffer[i]=0;
      right=atoi(buffer+index);
      if(right<0) right=0;
      if(right>100) right=100;
      i++;
   }
   if(channel==1)
      return left;
   else
      return right;
   /*sw1=buffer[i];    +/- flag for mixer active */
}

int GetOutputMixer(char *name,int channel)
{
   char buffer[80];
   int left=0,right=0,i,index;

   DOUT("Reading output mixer level");
   sprintf(buffer,"GET OUTPUT MIXER %s\n",name);
   write(mv,buffer,strlen(buffer));
   read (mv,buffer,80);
   for(i=0;(buffer[i]!='\n') && (i<80);i++);
   buffer[i]=0;
   DOUT(buffer);
   index=0;
   for(i=0;buffer[i] && buffer[i]!='\t';i++);
   if(buffer[i]=='\t')
   {
      buffer[i]=0;
      left=atoi(buffer+index);
      if(left<0) left=0;
      if(left>100) left=100;
      i++;
   }
   /* sw1=buffer[i];    +/- flag for mixer active */
   i+=2;
   index=i;
   for(;buffer[i] && buffer[i]!='\t';i++);
   if(buffer[i]=='\t')
   {
      buffer[i]=0;
      right=atoi(buffer+index);
      if(right<0) right=0;
      if(right>100) right=100;
      i++;
   }
   /* sw2=buffer[i];    +/- flag for mixer active */
   if(channel==1)
      return left;
   else
      return right;
}


#endif
