/*
 *     Program: SPECGRAM.C
 *      Author: Philip VanBaren
 *        Date: 23 August 1994
 *
 * Description: This program samples data from the ProAudio Spectrum
 *              sound card, performs an FFT, and displays the result.
 *              Can handle up to 2048 points (actually any size is possible
 *              with a little fiddling of buffers to get around 64k limits).
 *              On a 486/33 this code can perform and plot 1024-point and
 *              below in nearly real-time at 44100kHz.  (1024 FFT=31ms)
 *
 *  The code requires the SDK for the ProAudio Spectrum (file SDK-V3.ZIP,
 *  found via Internet at /ftp.uwp.edu:pub/msdos/proaudio), and is written for
 *  a medium memory model.  The required PAS SDK files are included here.
 *
 *  The code was written for Borland C, but should work with Microsoft C if
 *  conio functions like kbhit() and getch() are changed.
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "specgram.h"
#include "fft.h"
#include "display.h"
#include "extern.h"

short *fftdata;         /* Array for FFT data */
short *fftdata2;        /* Array for FFT data for stereo mode */
short *olddata;         /* Storage for embossed mode */
short *olddata2;        /* Storage for stereo embossed mode */
short *wind;            /* Array storing windowing function */
void *buffer;           /* Pointer to the sample buffer */
void *offsetbuffer;     /* Pointer to halfway through the sample buffer */
int freeze=0;           /* Flag for frozen display */

char ini_file[100];     /* Filename for the INI file */

volatile int done=0;      /* Flag indicating exit is requested */
volatile int buffer_full=0; /* Flag indicating a sample buffer is ready */

int main(int argc,char *argv[],char *environ[])
{
   int i,x=STARTX,x2=STARTX+10;
   double curr,start;
   int second_buffer_full=0;
   int key=0;

   draw_init();

   /* Check if the first parameter is an ini file name */
   if(argc>1 && argv[1][0]!='-' && argv[1][0]!='/')
   {
      strncpy(ini_file,argv[1],sizeof(ini_file));
      i=2;
   }
   else
   {
      strncpy(ini_file,"specgram.ini",sizeof(ini_file));
      i=1;
   }

   /*
    *  Parse the ini file and command line
    */
   parse_ini_file();
   parse_command(argc-i,&argv[i],environ);

   /* Initialize the buffer info */
   setup_buffers(MAX_LEN);

   compute_window_function();

   /* Set up the required arrays in the FFT code. */
   InitializeFFT(fftlen);

   /*
    *  Attempt to initialize the DMA buffering,
    *  then set the sample rate/size
    */
   reset_soundcard();

   draw_setup_graphics();
   draw_cleardisplay();
   set_palette(palette);
   draw_fontcolor(TICK_COLOR);

   update_header();
   draw_axis();

   ftime(&tb);
   basetime=(long)tb.time;
   start=(double)tb.millitm/1000.0;
   starttime=start;

   /*
    *  Keep getting data and plotting it.
    *  A space will pause, a second space will continue.
    *  Any other key will quit.
    */
   while(!done)
   {
      key=draw_getkey();
      while(freeze && !key)
         key=draw_getkey();
      if(key)
         done |= process_input(key);

      if(!key && !freeze)
      {
         if(accel_factor>1)
         {
            if(buffer_offset>=fftlen)
            {
               memcpy(buffer,offsetbuffer,fftlen*(sample_size/8)*(stereo+1));
               buffer_offset=0;
               second_buffer_full=0;
            }
           /* The UNIX sound drivers don't let us know when buffer is full 
            * so we can't rely on that to tell us when to grab the next
            * buffer.  For DOS versions, this works fine.       */
           #ifdef DOS
            if(buffer_full)
            {
               if(second_buffer_full)
               {
                  /* If a third buffer is ready while we are still processing
                   * the first two, drop what we were doing, and start
                   * processing at the beginning of the second buffer.  */
                  memcpy(buffer,offsetbuffer,fftlen*(sample_size/8)*(stereo+1));
                  buffer_offset=0;
               }
               /* Get the new data into the second buffer area,
                * and flag it as full */
               recordblock(offsetbuffer);
               second_buffer_full=1;
            }
           #endif
            if((buffer_offset>0) && !second_buffer_full)
            {
               /* If we need the second buffer, and it is not yet full,
                * sit and wait for it to fill up.  */
               while(!done && !buffer_full)
               {
                  key=draw_getkey();
                  if(key)
                     done |= process_input(key);
               }
               if(buffer_full)
               {
                  recordblock(offsetbuffer);
                  second_buffer_full=1;
               }
            }
         }
         else
         {
            while(!done && !buffer_full)
            {
               key=draw_getkey();
               if(key)
                  done |= process_input(key);
            }
            if(buffer_full)
            {
               recordblock(buffer);
               second_buffer_full=0;
               buffer_offset=0;
            }
         }

         if(deriv==0)
            window_data_normal();
         else if(deriv==1)
            window_data_first_difference();
         else
            window_data_second_difference();

         buffer_offset+=buffer_step;

         /* The real meat of the code lies elsewhere! */
         RealFFT(fftdata);

         DisplayData(fftdata,olddata,x,MAXY);
         DrawBar(x2,MAXY);

         if(stereo)
         {
            RealFFT(fftdata2);
            DisplayData(fftdata2,olddata2,x,MAXY-repcount*fftlen/2-5);
            DrawBar(x2,MAXY-repcount*fftlen/2-5);
         }

         ftime(&tb);
         curr=(double)(tb.time-basetime)+(double)tb.millitm/1000.0;

         if((curr-start)>=1)
         {
            while(start<=(curr-1)) start+=1;
            draw_putpixel(x,MAXY-fftlen/2*repcount,TICK_COLOR);
            draw_putpixel(x,MAXY-fftlen/2*repcount-1,TICK_COLOR);
            draw_putpixel(x,MAXY-fftlen/2*repcount-2,TICK_COLOR);
            if(stereo)
            {
               draw_putpixel(x,MAXY-fftlen*repcount-5,TICK_COLOR);
               draw_putpixel(x,MAXY-fftlen*repcount-6,TICK_COLOR);
               draw_putpixel(x,MAXY-fftlen*repcount-7,TICK_COLOR);
            }
         }

         if(x++ == MAXX) x=STARTX;
         if(x2++ == MAXX) x2=STARTX;
         displayed++;   /* count the number of displayed FFTs */
         if(displayed==100)
         {
            char ach[10];

            double rate=curr-starttime;
            if(rate>0)
               rate=(double)displayed*fftlen/(rate*SampleRate);
            else
               rate=0;
            sprintf(ach,"(%4.1f)",rate);
            draw_text_left(AFX+80,AFY,ach);
            #ifdef DEBUG_OUTPUT
               printf("%g-%g, %g\n",curr,starttime,rate);
            #endif
         }
      }
   }

   cleanup_soundcard();   /* Clean up the sound card stuff */

   draw_cleanup_graphics();  /* Clean up the video stuff */
   printf("You have been using Spectrogram " VERSION);
#ifdef SC_PAS16
   if(Soundcard==SC_PAS16) printf(" in ProAudio Spectrum 16-bit mode.");
#endif
#ifdef SC_SB8
   if(Soundcard==SC_SB8) printf(" in Soundblaster 8-bit mode.");
#endif
#ifdef SC_VESA
   if(Soundcard==SC_VESA) printf(" in VESA audio interface %d-bit mode.",sample_size);
#endif
#ifdef SC_SB16
   if(Soundcard==SC_SB16) printf(" in Soundblaster 16-bit mode.");
#endif
#ifdef SC_LINUX
   if(Soundcard==SC_LINUX) printf(" in Linux audio device %d-bit mode.",sample_size);
#endif
#ifdef SC_MULAW
   if(Soundcard==SC_MULAW) printf(" in Sun mu-law audio device mode.");
#endif
   puts("\nCopyright (C) " COPYRIGHT " Philip VanBaren");
   return(0);
}

