;   /*\
;---|*|----====< VESA >====----
;---|*|
;---|*| high level support routines for the VESA AI interface
;---|*|
;---|*| Copyright (c) 1993,1994  V.E.S.A, Inc. All Rights Reserved.
;---|*|
;---|*| Portions Copyright (c) 1993 Jason Blochowiak, Argo Games. Used
;---|*| with permission.
;---|*|
;---|*| VBE/AI 1.0 Specification
;---|*|    February 2, 1994. 1.00 release
;---|*|
;   \*/

#include <stdio.h>
#include <stdlib.h>
#include <alloc.h>
#include <direct.h>
#include <string.h>
#include <dos.h>
#include <io.h>
#include <conio.h>

#include "vesa_aud.h"

#define TRUE    -1
#define FALSE    0
#define ON      TRUE
#define OFF     FALSE

#if DEBUG
#define BREAKPOINT _asm{int 3};
#else
#define BREAKPOINT _asm{nop};
#endif

;   /*\
;---|*| Global Variables
;   \*/

        // nada...

;   /*\
;---|*| Local Variables
;   \*/

#define CALLBACKS       16  // Maximum number of callbacks that can be installed
#define MAXNESTEDCBS    3   // Maximum nesting of timer interrupts

        typedef struct {
            VESAHANDLE  handle;                         // The device handle
            void        (far pascal *callBack)(void);   // The callback function
            short       semaphore;                      // Our lockout semaphore for this callback
            long        divisor;                        // The timer divisor this callback wants
            long        divCount;                       // Overflow counter for this callback
        } Timer_t;

    // timer callback control table

        static volatile Timer_t timers[CALLBACKS];

    // a table to store data from device info blocks

        static int  doreg[CALLBACKS][2]    =      // do give ticks to this dev
            {{0,0},{0,0},{0,0},{0,0},
             {0,0},{0,0},{0,0},{0,0},
             {0,0},{0,0},{0,0},{0,0},
             {0,0},{0,0},{0,0},{0,0}};

        static int  denom      = 18;            // decimation denominator
        static int  isopen     = 0;             // number of devices opened

        static int  timerhooked = FALSE;        // TRUE if timer is currently hooked
        static int  curnestedtimer = 0;         // Number of nested timer interrupts
        static long curtimerdiv = 0x10000L;     // Current hardware timer divisor
        static long curtimerslop = 0;           // Number of timer counts that we've missed, due to excessive nesting

    // general prototypes

        static int  cmpstr                      ( char far *, char far *, int );
        static int  readblock                   ( int, char huge *, int );

    // timer prototypes

        static int  findhandleindex ( VESAHANDLE han );
        static void setuptimer      ( void );
        static void programtimer    ( unsigned );
        static void interrupt ourintvect ( void );

;
;   /*\
;---|*|------------------==============================-------------------
;---|*|------------------====< Start of execution >====-------------------
;---|*|------------------==============================-------------------
;   \*/

;
;   /*\
;---|*|----====< AllocateBuffer >====----
;---|*|
;---|*| Allocate some huge memory from the DOS pool
;---|*|
;   \*/
void far * pascal AllocateBuffer(siz)
    long siz;
{
   unsigned seg,off;
   void far *f;
   long lin_addr;

   f=farmalloc(siz+32);
   if(f==NULL)
      return f;

   seg=FP_SEG(f);
   off=FP_OFF(f);
   lin_addr=((seg*16L)+off+19L)&0xFFFF0L;
   seg=lin_addr>>4;

   // Store the original pointer (for free function)
   *(unsigned far *)MK_FP((seg-1),0xC)=FP_OFF(f);
   *(unsigned far *)MK_FP((seg-1),0xE)=FP_SEG(f);
   return(MK_FP(seg,0));
}

;
;   /*\
;---|*|----====< FreeBuffer >====----
;---|*|
;---|*| Allocate some huge memory from the DOS pool
;---|*|
;   \*/
void pascal FreeBuffer(void far *p)
{
   unsigned seg,off;
   if(FP_OFF(p)==0)
   {
      off=*(unsigned far *)MK_FP(FP_SEG(p)-1,0xc);
      seg=*(unsigned far *)MK_FP(FP_SEG(p)-1,0xe);
      farfree(MK_FP(seg,off));
   }
   else
      farfree(p);
}

;
;   /*\
;---|*|----====< VESAFindADevice >====----
;---|*|
;---|*| This function returns the next available device handle
;---|*| from the driver list.
;---|*|
;   \*/

VESAHANDLE pascal VESAFindADevice ( class )
    int class;
{
static VESAHANDLE lh = 0;

    // if this is a reset call, do it...

        if (class == -1)
            return(lh = 0); // start over...

    // call the driver directly

        _asm {
            mov     ax,VESAFUNCID
            mov     bx,VESAFUNC1
            mov     cx,[lh]
            mov     dx,[class]
            int     INTHOOK
            mov     [lh],cx
        }

    // return the finding

        return (lh);

}

;
;   /*\
;---|*|----====< VESAQueryDevice >====----
;---|*|
;---|*| Query the driver for general or device specific data.
;---|*|
;   \*/
long pascal VESAQueryDevice ( han, query, ptr )
    VESAHANDLE han;
    int        query;
    void far  *ptr;
{
long cc;
int ticks,n;

    // ask the driver for some specific information

        cc = (long) ptr; // we have to do this due to MS bugs in inline code!

        _asm {
            push    si
            push    di

            mov     ax,VESAFUNCID
            mov     bx,VESAFUNC2
            mov     cx,[han]
            mov     dx,[query]

            mov     si,word ptr [cc+2]
            mov     di,word ptr [cc+0]

            int     INTHOOK

            sub     ax,004Fh        // considered a good return
            neg     ax              // null out SI:DI if AX != 0x004F
            sbb     ax,ax
            not     ax
            and     si,ax
            and     di,ax

            mov     word ptr [cc+2],si
            mov     word ptr [cc+0],di

            pop     di
            pop     si
        }

    // if just a length query, return the value now

        if (query == VESAQUERY1)
            return (cc);
        if (query == VESAQUERY3)
            return (cc);
        if (query == VESAQUERY5)
            return (cc);

    // if this is returning the info block, grab the timer tick count
    // since this is the only time we may see the INFO block.

        if (query == VESAQUERY2) {

            switch (((fpGDC)cc)->gdclassid) {

                case WAVDEVICE:
                    ticks = ((fpGDC)cc)->u.gdwi.witimerticks;
                    break;

                case MIDDEVICE:
                    ticks = ((fpGDC)cc)->u.gdmi.mitimerticks;
                    break;

                default:
                case VOLDEVICE:
                    ticks = 0;  // no timer ticks for these types
                    break;
            }

            // see if the handle is already registered, if so, we're done

            for (n=1;n<CALLBACKS;n++) {
                if (doreg[n][0] == han) {
                    ticks = 0;
                    break;
                }
            }

            // if timer ticks needed, save that fact for a later open

            if (ticks) {
                for (n=1;n<CALLBACKS;n++) {
                    if (!doreg[n][0]) {
                        doreg[n][0] = han;
                        doreg[n][1] = ticks;
                        break;
                    }
                }
            }
        }

    // return the completion code

        return (cc);

}

;
;   /*\
;---|*|----====< VESAOpenADevice >====----
;---|*|
;---|*| This function returns a pointer to the driver
;---|*| services table if the driver opens successfully.
;---|*|
;   \*/
void far *pascal VESAOpenADevice ( han, apiset, memptr )
    VESAHANDLE han;
    int apiset;
    void far *memptr;
{
void far *cc = 0;
int n;

    // attempt to open the device

        _asm {
            cmp     word ptr [memptr+0],0   // offset must be null
            jnz     voad_05

            mov     ax,VESAFUNCID           // open function
            mov     bx,VESAFUNC3
            mov     cx,[han]                // device handle is required
            mov     dx,[apiset]             // select 16/32 bit interface
            mov     si,word ptr [memptr+2]  // pass in the segment
            int     INTHOOK                 // do the interrupt

            mov     word ptr [cc+2],si
            mov     word ptr [cc+0],cx
        }
        voad_05:;

    // if successful, check to see if the device needs timer ticks

        if (cc) {

            for (n=1;n<CALLBACKS;n++) {

                // if this is it, setup for timer ticks

                if (doreg[n][0] == han) {

                    // register wave audio

                    if ( _fstrncmp(((fpWAVServ)cc)->wsname,"WAVS",4) == 0 )

                        VESARegisterTimer
                          (
                            han,                            // handle
                            ((fpWAVServ)cc)->wsTimerTick,   // address
                            VESARateToDivisor(doreg[n][1])  // tick count
                          );

                    // register midi audio

                    if ( _fstrncmp(((fpMIDServ)cc)->msname,"MIDS",4) == 0 )

                        VESARegisterTimer
                          (
                            han,                            // handle
                            ((fpMIDServ)cc)->msTimerTick,   // address
                            VESARateToDivisor(doreg[n][1])  // tick count
                          );

                    break;
                }
            }
        }

    // return the pointer to the services table

        return(cc);

}

;
;   /*\
;---|*|----====< VESACloseDevice >====----
;---|*|
;---|*| This function closes an opened device.
;---|*|
;   \*/
void  pascal VESACloseDevice ( han )
    VESAHANDLE han;
{
int n,gd;

    // remove the timer callbacks for this handle

        VESARegisterTimer( han, 0, 0 );

    // go close the device.

        _asm {
            mov     ax,VESAFUNCID
            mov     bx,VESAFUNC4
            mov     cx,[han]            // if the handle exists, the
            int     INTHOOK             // device will close
        }
}

;   /*\
;---|*|----====< VESARateToDivisor >====----
;---|*|
;---|*| Converts an integral rate (in Hz) to a timer divisor value, which
;---|*| can then be passed to VESARegisterTimer().
;---|*|
;---|*| Copyright (c) 1993 Jason Blochowiak, Argo Games. Used with permission.
;---|*|
;   \*/
long pascal VESARateToDivisor(rate)
    long rate;
{
long result;

    if (rate)
        result = 1193180L / rate;
    else
        result = 0x10000L;

    return(result);
}

;
;   /*\
;---|*|----====< VESARegisterTimer >====----
;---|*|
;---|*| Add/Remove a far call from the timer registration.
;---|*|
;---|*| Copyright (c) 1993 Jason Blochowiak, Argo Games. Used with permission.
;---|*|
;   \*/
int   pascal VESARegisterTimer( han, addr, divisor )
    VESAHANDLE han;
    void (far pascal *addr)();
    long divisor;
{
int n,i,max;
int retval = 0;
int needsProgram = FALSE;
int index;
Timer_t *t;

    // Puke if we have a zero handle

        if (!han)
            return(retval);

    // Make sure no interrupts come in while we're messing with the table

        _asm cli ;

    // Make sure that the special entry doesn't get used by something else

        timers[0].handle = -1;

    // If address in non-0, register it

        if (addr) {

        // See if an entry with this handle already exists

            index = findhandleindex(han);

        // If it returned -1, no entry with this handle exists, so find
        // an entry with no handle

            if (index == -1)
                index = findhandleindex(0);

        // If we found an entry with a matching handle, or an empty entry,
        // fill it in.

            if (index != -1) {

            // Get pointer to timer table, and cast to eliminate the volatile
            // modifier (interrupts are disabled, so there's no chance of the
            // structure being modified from an interrupt)

                t = (Timer_t *)(&timers[index]);

                t->handle = han;        // Store the handle
                t->divisor = divisor;   // Store our divisor
                t->divCount = 0;        // Clear the overflow counter
                t->callBack = addr;     // Store the callback address
                t->semaphore = 0;       // Initialize the semaphore

                retval = -1;            // Mark as successful
                needsProgram = TRUE;    // We'll need to mess with the hardware timer
            }
        }
        else {

        // We've got a 0 address - remove the timer entry

        // Find the entry with a matching handle

            index = findhandleindex(han);
            if (index != -1) {

            // We found the entry - kill it

                t = (Timer_t *)(&timers[index]);

                t->handle = 0;
                t->divisor = 0;
                t->divCount = 0;
                t->callBack = 0;

                retval = -1;            // Mark as successful
                needsProgram = TRUE;    // We'll need to mess with the hardware timer
            }
        }

    // If we need to, go mess with the hardware timer

        if (needsProgram)
            setuptimer();

        _asm sti ;

    // Return success or failure

        return(retval);
}

;
;   /*\
;---|*|------------------=============================-------------------
;---|*|------------------====< internal routines >====-------------------
;---|*|------------------=============================-------------------
;   \*/

;
;   /*\
;---|*|----====< cmpstr >====----
;---|*|
;---|*| compare strings at the end of far pointers
;---|*|
;   \*/

static int cmpstr(t,s,l)
    char far *t;
    char far *s;
    int l;
{
char c;

    while (l--)
        if ((c = *t++ - *s++) != 0)
            return(c);

    return(0);
}

;   /*\
;---|*|----====< ourintvect >====----
;---|*|
;---|*| Process timer interrupts.
;---|*|
;---|*| Copyright (c) 1993 Jason Blochowiak, Argo Games. Used with permission.
;---|*|
;   \*/

#if __BORLANDC__
static interrupt void ourintvect(void)
#else
static void interrupt ourintvect(void)
#endif

{
int     i;
int     didEOI = FALSE;
long    timercount;
Timer_t *t;

    // We want to process any previously unaccounted for timer divisor
    // counts, plus the number of counts that the hardware did before
    // triggering this interrupt

        curtimerslop += curtimerdiv;

    // If we don't have too many nested timer interrupts, process the
    // individual callbacks.

        if (++curnestedtimer < MAXNESTEDCBS) {

        // Clear any accumulated slop, and use for count to advance the
        // individual callbacks' overflow values

            timercount = curtimerslop;
            curtimerslop = 0;

            t = (Timer_t *)timers;
            for (i = 0;i < CALLBACKS;i++,t++) {

            // If this isn't a valid entry, skip it

                if (!(t->handle && t->callBack))
                    continue;

            // It's valid - advance its overflow value

                t->divCount += timercount;

            // If we're past the first (special) entry, and the we didn't
            // call the old timer ISR, do the EOI now.

                if (i && !didEOI) {
                    outp(0x20,0x20);
                    didEOI = TRUE;
                }

            // If we're not already in a call to this one, and its overflow
            // value equals or exceeds its timer count value, set its
            // semaphore and call it.

                if ((t->semaphore == 0) && (t->divCount >= t->divisor)) {

                    t->divCount -= t->divisor;
                    t->semaphore++;

                // If it's the special entry, push the flags for the old
                // timer ISR's IRET, and flag the EOI as having happened,
                // as the old ISR will have issued it. If it's not the
                // special entry, enable interrupts before the call.

                if (i) {
                    _asm { sti };
                }
                else {
                    didEOI = TRUE;
                    _asm { pushf };
                }

                t->callBack();

                // Re-disable interrupts, and flag entry as not currently
                // being processed.

                    _asm { cli };
                    t->semaphore--;

            }
        }
    }

    // Decrement nested interrupt call count

        curnestedtimer--;

    // If we didn't get to the EOI before, do it now

        if (!didEOI)
            outp(0x20,0x20);
}

;
;   /*\
;---|*|----====< programtimer >====----
;---|*|
;---|*| Program the timer chip for the appropriate interrupt rate. The
;---|*| parameter is the actual timer value, so it's 1193180/rate
;---|*|
;---|*| Copyright (c) 1993 Jason Blochowiak, Argo Games. Used with permission.
;---|*|
;   \*/

void programtimer (max)
    unsigned max;
{
        _asm    pushf
        _asm    cli

        _asm    mov al,0x36
        _asm    out 0x43,al
        _asm    jmp delay1

    delay1:;

        _asm    jmp delay2

    delay2:;

        _asm    mov ax,[max]
        _asm    out 0x40,al
        _asm    jmp delay3

    delay3:;

        _asm    jmp delay4

    delay4:;

        _asm    xchg ah,al
        _asm    out 0x40,al
        _asm    popf
}

;
;   /*\
;---|*|----====< readblock >====----
;---|*|
;---|*| read a chunk of the PCM file into the huge buffer
;---|*|
;   \*/
static int readblock (han,tptr,len)
    int han;
    char huge *tptr;
    int len;
{
int siz = 0;

    // go get it...

        _asm {
            push    ds

            mov     cx,[len]
            mov     ax,cx
            add     cx,word ptr [tptr]  // wrap?
            jnc     rdbl05              // no, go get the size

            sub     ax,cx               // ax holds the # of bytes to read
            mov     cx,ax
            sub     [len],ax

            mov     ah,03fh             // cx holds the length
            mov     bx,[han]
            lds     dx,[tptr]
            int     21h

            mov     [siz],ax            // we moved this much
            add     word ptr [tptr+2],0x1000

            cmp     ax,cx               // same size?
            jnz     rdbldone            // no, exit...

        } rdbl05: _asm {

            mov     ah,03fh
            mov     bx,[han]
            mov     cx,[len]

            jcxz    rdbldone

            lds     dx,[tptr]
            int     21h

            add     [siz],ax            // we moved this much

        } rdbldone: _asm {

            pop     ds

        }

    // return the amount read

        return(siz);
}


;   /*\
;---|*|----====< setuptimer >====----
;---|*|
;---|*| Something has changed the timer structures, so we need to figure
;---|*| out what to do with the hardware timer and the hardware timer ISR
;---|*|
;---|*| This is only called by VESARegisterTimer(), so we assume that
;---|*| interrupts are disabled.
;---|*|
;---|*| Copyright (c) 1993 Jason Blochowiak, Argo Games. Used with permission.
;---|*|
;   \*/
void
setuptimer(void)
{
int     i,j;
int     otherCount;
long    minDiv,minDivDelta,timerVal;
Timer_t *t,*t2;

    // If we haven't filled out the special slot 0 entry, do so now

        if (!timers[0].callBack) {
            timers[0].callBack = (void(far pascal *)(void))_dos_getvect(8);
            timers[0].divisor  = 0x10000L;
            timers[0].divCount = 0;
        }

    // Count the number of non-special entries, and find 1) The entry
    // with the smallest timer divisor, and 2) The smallest difference
    // between divisors.

        minDiv = 0x10000L;
        minDivDelta = 0x10000L;
        otherCount = 0;
        t = (Timer_t *)(&timers[1]);
        for (i = 1;i < CALLBACKS;i++,t++) {

        // Only pay attention to valid entries

            if (t->handle) {

            // Count this non-special entry

                otherCount++;

            // If this entry's divisor is smaller than our previous minimum,
            // use it for our new minimum.

                if (t->divisor && (t->divisor < minDiv))
                    minDiv = t->divisor;

            // Go through the table again, finding the smallest non-zero
            // difference between two divisors

                t2 = (Timer_t *)(&timers[0]);
                for (j = 0;j < CALLBACKS;j++,t2++) {

                    if ((j != i) && (t2->handle)) {

                        long    delta;

                    // Get absolute value of difference between the two
                    // entries' divisors

                        delta = t2->divisor - t->divisor;

                        if (delta < 0)
                            delta = -delta;

                    // If non-zero, and smaller than previous smallest
                    // difference, hold on to it

                        if (delta && (delta < minDivDelta))
                            minDivDelta = delta;
                    }
                }
            }
        }

    // If we no longer need to own the vector, and our ISR is installed,
    // remove it

        if ((otherCount == 0) && timerhooked) {

#if __BORLANDC__ //DSC
        _dos_setvect(8,(void interrupt (*)())(timers[0].callBack));
#else
        _dos_setvect(8,(void (interrupt far *)())(timers[0].callBack));
#endif
        timerhooked = FALSE;

        }

    // If we need to own the vector, and our ISR isn't installed,
    // install it

        if (otherCount && !timerhooked) {
            _dos_setvect(8,ourintvect);
            timerhooked = TRUE;
        }

    // Set timer divisor to minimum divisor

        timerVal = minDiv;

    //  This next bit is here to help out in situations where there are a
    // couple of fairly slow callbacks set up, and no fast callbacks. If the
    // only callbacks set up are at 40Hz and 50Hz, and this code weren't
    // here, the hardware would be set up to generate 50Hz interrupts.
    // Starting at a time of 0.0, the interrupts would be called at:
    // Time     40Hz    50Hz
    // ----     ----    ----
    // 0.02             x
    // 0.04     x       x
    // 0.06     x       x
    // 0.08     x       x
    // 0.10     x       x
    // 0.12             x
    // 0.14     x       x
    //
    // etc. In other words, four out of every five of the 40Hz callbacks
    // would be performed with 0.02 seconds in between, and then there
    // would be a 0.04 second delay before the next call to the 40Hz
    // callback. Although the 40Hz callback will get called 40 times a
    // second, the time between calls won't be regular.
    //  Although this isn't a problem under some circumstances, it can be a
    // problem under other circumstances, and a nuisance under still other
    // circumstances.
    //  This code can increase the timer resolution to about 145.6Hz,
    // which translates to about 6.87ms (0.00687 seconds). This means that
    // a particular callback will occur, at most, roughly 3.435ms before
    // or after it "should".
    //  Note that by changing the value 0x2000, you can alter the adjustment
    // frequency that will be used.
    //

        if (timerVal > minDivDelta) {

            if (minDivDelta >= 0x2000)
                timerVal = minDivDelta;
            else if (timerVal > 0x2000)
                timerVal = 0x2000;
        }

    // Set the hardware timer value. It's possible that timerVal will be
    // 0x10000L, more than will fit into a 16-bit word. This is ok, as
    // programming the timer for 0x0000 is treated by the hardware as
    // programming the timer for 0x10000.

        programtimer((unsigned)timerVal);
        curtimerdiv = timerVal;
}

;
;   /*\
;---|*|----====< findhandleindex >====----
;---|*|
;---|*| This finds an entry in the timers[] array with a handle value
;---|*| that matches the parameter. Returns -1 if a matching entry
;---|*| isn't found.
;---|*|
;---|*| Copyright (c) 1993 Jason Blochowiak, Argo Games. Used with permission.
;---|*|
;   \*/
static int findhandleindex(VESAHANDLE han)
{
    int i;

    for (i = 0;i < CALLBACKS;i++)
        if (timers[i].handle == han)
            return(i);
    return(-1);
}

;   /*\
;---|*| end of VESA.C
;   \*/




